/* help.c -- help functions for the bridge program.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */
 
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef VMS
#include <file.h>
#endif
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <sys/time.h>

#ifdef AIX
#include <time.h>
#endif

#ifndef SEEK_SET
#define SEEK_SET  0
#endif
 
#include "types.h"
#include "input.h"
#include "terminal.h"
#include "display.h"
 
/* char *help_file_name = "okbridge.help"; */
#include "helpfile.h"

extern strcasecmp (), fseek ();
 
extern int errno;
extern char *sys_errlist[];
extern char *strdup();
extern char *getenv ();
extern char *malloc ();
extern int  input_help_topic ();
extern int  exit_requested;

#ifdef VMS
extern int delete();
int unlink(foo)
char *foo;
{
    return(delete(foo));
}
#endif

typedef struct help_entry_struct {
	char	*topic;
	char	*description;
	long int file_offset;
	struct help_entry_struct *next;
} *help_entry;
 
static help_entry main_topic = NULL;
FILE *help_file = NULL;

long int current_page = -1;
  /* The offset into the help file of the current page being displayed,
     or -1 if we are not displaying any page. */

 
void display_topics (message)
	char *message;
/* Displays the list of topics, with the given message as the top line.
 . Does not display the main topic.
 */
{
	int i;
	help_entry e;
	char msg_buf[80];
 
	clear_screen ();
	print (1,1,message);
	if (main_topic == NULL) {
		print (3,1,"The help system is empty.");
	} else {
		i = 3;
		for (e = main_topic->next; e != NULL; e = e->next) {
		  if (strcasecmp(e->topic, "slam")) {
			sprintf (msg_buf, "%-10s -- %s",e->topic,
				e->description);
			print (i++, 1, msg_buf);
		      }
		}
	}
}
 
static int read_help_line (buf, buflen)
	char *buf; int buflen;
/* Reads a line from the help file.  Returns the number of characters
   read of -1 if EOF is encountered.  Skips lines beginning with a '#'. 
*/
{
	int i, ch;
 
	do {
		ch = getc(help_file);
		i = 0;
		while ((ch != '\n') && (ch != EOF)) {
			if (i < buflen-1) buf[i++] = ch;
			ch = getc(help_file);
		}
		buf[i] = '\0';
		if (ch == EOF) return (-1);
		while ((i > 0) && isspace(buf[i-1])) buf[--i] = '\0';
	} while (buf[0] == '#');
	return (i);
}

static void Display_help_region (beginning_of_region, no_lines)
     long beginning_of_region; int no_lines;
{
  char line_buf[81];
  int log, i;

  clear_screen ();
  fseek (help_file, beginning_of_region, SEEK_SET);
  for (i = 0; i < no_lines; i++) {
    log = read_help_line (line_buf, 81);
    if (log < 0) break;
    if (line_buf[0] == '^') line_buf[0] = ' ';
    print (i+1, 1, line_buf);
  }

  Press_Return_to_Continue ("");
}
 
static void display_help_entry (e)
	help_entry e;
/* Displays the help_entry e. */
{
  char line[81];
  int  log;

  /* The help display routine has been revised so that it determines for
     itself where to break the screens.  Any line which begins with a 
     caret '^' is interpreted as a potential screen break.  We try to
     fit as much as possible onto a single screen, only breaking at
     accepted screen breaks. */
  long screen_beginning;
    /* The position in the help file where the current screen begins. */
  int screen_length;
    /* The number of lines in the current screen. */
  int paragraph_length;
    /* The number of lines in the current paragraph. */
  int end_of_entry;
    /* Boolean flag that indicates that the end of this help entry was found */
  
  screen_length = 0;
  screen_beginning = e->file_offset;
  fseek (help_file, e->file_offset, SEEK_SET);

  end_of_entry = 0;
  exit_requested = 0;
  while (!exit_requested) {
    paragraph_length = 0;
    /* Read ahead until we reach the end of the current paragraph. */
    while (1) {
      log = read_help_line (line, 81);
      if ((log < 0) || !strcmp(line, "--")) {
	end_of_entry = 1;
	break;
      } else if (line[0] == '^') {
	paragraph_length++;
	break;
      }
      paragraph_length++;
    }

    if (paragraph_length + screen_length <= terminal_lines - 1) {
      /* The current paragraph fits on this page.  If this is the
         last paragraph, then display it and exit. */
      screen_length += paragraph_length;
      if (end_of_entry) {
	Display_help_region (screen_beginning, screen_length);
	break;
      }
    } else if (screen_length == 0) {
      /* The current paragraph is larger than a page, and we have
         displayed everything before the beginning of the paragraph.
         Consequently, we display as much as we can of the current
         paragraph. */
      Display_help_region (screen_beginning, terminal_lines - 1);
      screen_beginning = ftell (help_file);
    } else {
      /* The current page is partially filled, and the current paragraph
         does not fit on the current page. */
      Display_help_region (screen_beginning, screen_length);
      screen_beginning = ftell (help_file);
      screen_length = 0;
    }
  }
}

static FILE *open_helpfile ()
/* Tries to open the helpfile with given name from the help_directory.
 * If an error, prints an error message and returns NULL.  Otherwise,
 * returns a pointer to the opened file.
 */
{
	char msg_buf[80], *envhelpdir;
	FILE *fp;
 
	fp = fopen (help_file_name, "r");
	if (fp == NULL)
		fp = fopen ("okbridge.help", "r");

	if (fp == NULL) {
		sprintf (msg_buf, "Error opening helpfile %s", 
			 help_file_name);
		print (3, 1, msg_buf);
		sprintf (msg_buf, "System reports error: %s",
				sys_errlist[errno]);
		print (4, 1, msg_buf);
		Press_Return_to_Continue ("");
	}
	return (fp);
}
 
 
static help_entry find_help_topic (topic)
	char *topic;
/* Looks for the help entry with the associated topic.  If it is found,
 * returns a pointer to the corresponding record.  Otherwise, returns NULL.
 */
{
	help_entry e;
 
	e = main_topic;
	while (e != NULL) {
		if (!strcasecmp(e->topic, topic))
			return (e);
		e = e->next;
	}
	return (e);
}
 
static help_entry read_new_help_entry ()
/* Reads a help entry from the help_file.  Allocates a help_entry record
 . and records the pertinent information in that record.  Returns the
 . allocated record or NULL if the end of file is reached.
 */
{
	char line_buf[81];
	help_entry e;
	char *curpos, *keypos, *descpos;
	int log;

	log = read_help_line (line_buf, 81);
	if (log < 0)
		return (NULL);

	for (keypos = line_buf; isspace(*keypos); keypos++);
	for (curpos = keypos;(*curpos != '\0') && !isspace(*curpos);curpos++);
	for (descpos = curpos; isspace(*descpos); descpos++);

	e = (help_entry) malloc(sizeof(struct help_entry_struct));
	*curpos = '\0';
	e->topic = strdup (keypos);
	e->description = strdup (descpos);
	e->file_offset = ftell (help_file);
	e->next = NULL;

	do
	  log = read_help_line (line_buf, 81);
	while 
	  ((log >= 0) && strcmp(line_buf, "--"));

	return (e);

}

 
void initialize_help_system ()
/* Called once at the beginning of the program to read the file of help
 * topics.
 */
{
	help_entry e;
 
	help_file = open_helpfile ();
	if (help_file == NULL) return;
 
	e = main_topic = read_new_help_entry ();
	while (e != NULL) {
		e->next = read_new_help_entry ();
		e = e->next;
	}

}
 
void Browse_help_topics ();
 
void display_help (topic)
	char *topic;
/* Displays help on the given topic.  This consists of looking up the
 * help file associated to this topic and displaying the contents of this
 * file on the screen.  If the topic string is empty, then displays first
 * the contents of the main topic file, and then displays a list of the
 * topics.  If there is no help on the given topic, then displays a list
 * of topics.
 */
{
  help_entry he;
  char line_buf[81];
  
  if (main_topic == NULL)
    return;
  
  clear_screen ();
  if (strlen(topic) == 0) {
    display_help_entry (main_topic);
  } else if ((he = find_help_topic(topic)) != NULL)
    display_help_entry (he);
  else {
    sprintf (line_buf, "%s  %s",
	     "There is no help for this topic.",
	     "The available topics are");
    display_topics (line_buf);
    Press_Return_to_Continue ("");
  }
  
}

void browse_help (topic)
     char *topic;
/* Displays help on the given topic.  Afterwards, displays a list of
 * topics along with a request to enter the name of a new topic.
 */
{
  help_entry he;
  char line_buf[81];
  int no_topic = 0;
  
  if (main_topic == NULL)
    return;
  
  clear_screen ();
  if (strlen(topic) == 0) {
    display_help_entry (main_topic);
  } else if ((he = find_help_topic(topic)) != NULL)
    display_help_entry (he);
  else 
    no_topic = 1;
  
  display_topics ("LIST OF HELP TOPICS");
  print (terminal_lines-1, 1, "HELP");
  Refresh_Status_Display ();
  
  if (no_topic) {
    sprintf (line_buf, "THERE IS NO HELP FOR THE TOPIC %s", topic);
    print (terminal_lines-2, 1, line_buf);
    ring_bell ();
  }
}

void Refresh_Help_Display ()
/* Redisplays the current screen of help information. */
{
  char line[81];
  int lines_on_page, log;

  if (current_page < 0) {
    display_topics ("LIST OF HELP TOPICS");
    return;
  }
  
  lines_on_page = 0;
  fseek (help_file, current_page, SEEK_SET);
  log = read_help_line (line, 81);
  while ((log >= 0) && strcmp(line, "--")) {
    if ((lines_on_page > terminal_lines-2) || (line[0] == '^'))
      return;
    print (++lines_on_page, 1, line);
    log = read_help_line (line, 81);
  }
  
}

void Clear_Help_Display ()
/* Returns the help system to its initial state. */
{
  current_page = -1;
}

void display_news ()
/* Checks if there is any new news regarding OKbridge available on the
 * local system.  If yes, then displays the news file.
 *
 * Based on code contributed by Shih-Ping Spencer Sun.
 */
{
  struct stat news_st, last_st;
  char s[256], lastfn[256], *p;
  FILE *fp;
  int display_line = 4;

  p = getenv("HOME");
  sprintf(lastfn, "%s/.okbridgelast", (p == NULL) ? "." : p);

  /* if news file doesn't exist, then don't show of course */
  if (stat(news_file_name, &news_st) < 0)
    return;
  else if (stat(lastfn, &last_st) >= 0)
    if (news_st.st_mtime < last_st.st_mtime)
      return;

  unlink(lastfn);
  close (open(lastfn, O_WRONLY | O_CREAT | O_TRUNC, 0666));

  if ((fp = fopen(news_file_name, "r")) != NULL) {
    clear_screen ();
    print (2, 20, "LOCAL OKBRIDGE NEWS");
    while (fgets(s, terminal_cols-3, fp) != NULL) {
      if (s[strlen(s)-1] == '\n')
	s[strlen(s)-1] = '\0';
      if (display_line > terminal_lines - 3) {
	Press_Return_to_Continue("");
	clear_screen ();
	print (2, 20, "LOCAL OKBRIDGE NEWS");
	display_line = 4;
      }
      print (display_line++, 1, s);
    }
    Press_Return_to_Continue("");
    close(fp);
    Refresh_Display ();
  }      

}
