/* GPS.c -- global playing service

   Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved

   OKbridge is made available as a free service to the Internet.
   Accordingly, the following restrictions are placed on its use:

   1.  OKbridge may not be modified in any way without the explicit 
       permission of Matthew Clegg.  

   2.  OKbridge may not be used in any way for commercial advantage.
       It may not be placed on for-profit networks or on for-profit
       computer systems.  It may not be bundled as part of a package
       or service provided by a for-profit organization.

   If you have questions about restrictions on the use of OKbridge,
   write to mclegg@cs.ucsd.edu.

   DISCLAIMER:  The user of OKbridge accepts full responsibility for any
   damage which may be caused by OKbridge.


   The global playing service is a global database of player information.
   In the initial implementation, it will contain only information
   about currently playing tables.  At a later date, we will add
   capabilities for downloading sets of boards and playing competitively.

   Implementation Notes:

   1. The GPS is implemented on a strictly request/response basis,
      similar in principle to a remote procedure call.  However,
      rather than using RPC, we use TCP streams.  This is done because
      I don't know if a standard implementation of RPC is available 
      on all BSD systems.  Also, some of the data we communicate doesn't
      seem to fit easily into the RPC model.

   2. When we first start up, we will attempt to connect to the GPS
      and print out a message of the day along with a list of
      currently playing tables.  Thereafter, we will hold open the
      GPS connection.

   3. When a GPS request is made, we may not even have the connection
      open.  Therefore, the request/response loop will have the 
      following structure:

        A. Determine if the connection is still open.
           If not, attempt to connect.  If the attempt is unsuccessful,
           then exit with an error.

        B. Transfer our request to the GPS.

	C. Enter a wait-loop waiting for the response.  In this loop,
           we should monitor the other communication channels (e.g.,
           the keyboard and network) so as to allow a user interrupt
           if necessary.

	D. Process the GPS response.

   Unfortunately, this module is not re-entrant.  This can be a bit of a
   pain since the GPS_read_line procedure calls Wait_for_input in
   input.c, which in turn calls Wait_for_event in network.c.

*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>

#define GPS
#include "types.h"
#include "state.h"
#include "display.h"
#include "parser.h"
#include "terminal.h"
#include "conversation.h"
#include "input.h"
#include "gps_info.h"
#include "gps.h"

extern void close ();
extern int  getpid ();
extern void srand ();

extern int  No_connections ();
extern char *strdup ();

extern char *sys_errlist[];
extern int errno;
extern int exit_requested;

extern Sort_play_records_by_matchpoints ();
extern Sort_play_records_by_imps ();

static char GPS_buf [100];
  /* A general purpose buffer for reading and writing messages to the GPS. */

static char GPS_advise_buf [100];
  /* A general buffer used for sending GPS advisory requests. */

static char GPS_error_buf [100];
  /* An error output buffer. */

Non_blocking_buffer *GPS_inbuf  = NULL;
Non_blocking_buffer *GPS_outbuf = NULL;

/* Many of the messages transmitted by the GPS are "advisory" in nature,
   i.e., they do not require any response from the GPS.  These messages
   are frequently generated in response to network events (e.g., people
   joining/leaving a table or changing seats), and so they can occur
   at times when the GPS may already be processing a request.  To prevent
   these requests from being ignored, we queue them up in a request
   queue which is processed at the end of each GPS request.
*/

#define ADVISORY_MESSAGE_LENGTH 100

typedef struct Advisory_message_struct {
  char message [ADVISORY_MESSAGE_LENGTH];
  struct Advisory_message_struct *next;
} *Advisory_message;

static Advisory_message Advisory_queue_tail = NULL;
  /* A pointer to the last advisory message in the queue. */
static Advisory_message Advisory_queue_head = NULL;
  /* A pointer to the head of the advisory queue. */
static Advisory_message Advisory_free_list = NULL;
  /* A pointer to a list of unused message structures. */


#ifdef LOGFILE
extern FILE *net_log;
#endif

static void Enqueue_Advisory_Message (message)
     char *message;
{
  Advisory_message am;

  if (Advisory_free_list == NULL)
    am = (Advisory_message) malloc (sizeof(struct Advisory_message_struct));
  else {
    am = Advisory_free_list;
    Advisory_free_list = am->next;
  }

  strncpy (am->message, message, ADVISORY_MESSAGE_LENGTH);
  am->message[ADVISORY_MESSAGE_LENGTH-1] = '\0';

  if (Advisory_queue_tail == NULL) {
    Advisory_queue_tail = Advisory_queue_head = am;
    am->next = NULL;
  } else {
    Advisory_queue_tail->next = am;
    Advisory_queue_tail = am;
  }
}

static char *Dequeue_Advisory_Message ()
{
  Advisory_message am;

  if (Advisory_queue_head == NULL)
    return (NULL);

  am = Advisory_queue_head;
  Advisory_queue_head = am->next;
  if (Advisory_queue_head == NULL)
    Advisory_queue_tail = NULL;

  am->next = Advisory_free_list;
  Advisory_free_list = am;
  return (am->message);
}


static void GPS_Comment (msg)
     char *msg;
{
  Display_Player_Comment (COMMENT_PRIVATE, "GPS", msg);
}

static void GPS_Pause (msg)
     char *msg;
{
  suspend_gps_input = 1;
  Pause (msg);
  suspend_gps_input = 0;
}

static void Close_GPS_socket ()
/* If we are connected to the GPS, then closes the current connection. */
{
  if (GPS_socket > 0)
    close (GPS_socket);

  GPS_socket = 0;
  GPS_request_in_progress = 0;
  suspend_gps_input = 0;
  local_player_id = -1;
  Display_Player_Position ();
  if (client_mode)
    Send_name (Local_table, local_player_name, local_player_id);

  if (GPS_inbuf != NULL)
    nbb_clear (GPS_inbuf);
  if (GPS_outbuf != NULL)
    nbb_clear (GPS_outbuf);
  
}

static void GPS_transmit (buf)
     char *buf;
/* Transmits the buf to the GPS. */
{
  int status;

  if (!GPS_socket)
    return;

  status = nbb_writeln (GPS_socket, GPS_outbuf, buf);

  if (status < 0) {
    if (status == NBB_SYSERR) {
      sprintf (GPS_buf, "ERROR! %s", sys_errlist[errno]);
      GPS_Comment (GPS_buf);
    }
    Close_GPS_socket ();
  }

#ifdef LOGFILE
  fprintf (net_log, "GPS> %s\n", buf);
  fflush (net_log);
#endif
}

static void Flush_Advisory_Queue ()
{
  char *msg;

  if (GPS_request_in_progress)
    return;

  while ((msg = Dequeue_Advisory_Message()) != NULL)
    GPS_transmit (msg);
}

void GPS_Reset ()
{
  Close_GPS_socket ();
}

static int GPS_data_available_event ()
{
  GPS_request_in_progress = 1;

  if (!GPS_socket)
    return (1);
  else
    return(nbb_message_available(GPS_inbuf));
}

static int GPS_readline (buf, buflen)
     char *buf; int buflen;
/* Reads a line of data from the GPS without blocking.  Returns -1 if
   the "end-of-request" marker is received and -2 if an error occurs
   during the read.  Otherwise, returns 0.
*/
{
  int message_status;
    /* Return code from non blocking read. */

  if (GPS_unavailable || !Use_GPS || !GPS_socket)
    return (-2);

  do {
    message_status = nbb_readln (GPS_socket, GPS_inbuf, buf, buflen);
    if (message_status >= 0) {
#ifdef LOGFILE
  fprintf (net_log, "GPS< %s\n", buf);
  fflush (net_log);
#endif
      if (!strcmp(buf, GPS_RESPONSE_MARKER)) {
	GPS_request_in_progress = 0;
	return (-1);
      } else
	return (0);
    } else if (message_status == NBB_EOF) {
      Close_GPS_socket ();
      return (-2);
    } else if (message_status == NBB_SYSERR) {
      sprintf (GPS_buf, "ERROR! %s", sys_errlist[errno]);
      GPS_Comment (GPS_buf);
      Close_GPS_socket ();
      return (-2);
    } else if (message_status == NBB_NODATA) {
      Refresh_Input_Buffers ();
      Wait_for_event_at_game_level (GPS_data_available_event);
    }
  } while (1);
}

static void GPS_Ask_for_response ()
{
  GPS_transmit (GPS_REQUEST_MARKER);
}

void Conclude_GPS_request ()
/* Reads data from the GPS socket until the end of request is encountered. */
{
  int status;

  if (!GPS_request_in_progress) {
    Flush_Advisory_Queue ();
    return;
  }

  do
    status = GPS_readline (GPS_buf, 100);
  while
    ((status >= 0) && GPS_request_in_progress);

  GPS_request_in_progress = 0;
  Flush_Advisory_Queue ();
}

static void Check_GPS_connection ()
/* Checks if the connection to the GPS has been closed or if an
   interrupt occurred during the last transmit operation.  If so, then
   resets the connection to a state from which we can proceed. 
*/
{
  int status;

  if (GPS_request_in_progress)
    Conclude_GPS_request ();

  if (GPS_inbuf == NULL)
    return;

  if (GPS_socket > 0) {
    status = nbb_fill (GPS_socket, GPS_inbuf);
    if ((status < 0) && (status != NBB_NODATA))
      Close_GPS_socket ();
  }
}

static int Perform_GPS_handshake ()
/* Sends the handshake information to the GPS to test and also
   requests handshake information in return.  Verifies that we are
   running the same version as the server.  If the verification succeeds,
   then returns 0.  If we are running different versions, then returns 1.
   If some other error occurs, then returns -1.
*/
{
  char kbuf[100];
  int read_status;

  GPS_transmit (GPS_version);

  read_status = GPS_readline(kbuf, 100);
  if (read_status != 0) {
    sprintf (GPS_error_buf, "ERROR ACCESSING GPS SERVER.");
    return (1);
  }
  if (strcmp(GPS_version, kbuf)) {
    sprintf (GPS_error_buf, 
	     "GPS SERVER IS INCOMPATIBLE -- SERVER REPORTS VERSION %s", kbuf);
    return (1);
  }

  return (0);
}

static int Open_GPS_connection ()
/* Opens the connection to the GPS.  If an error occurs, then sets
   GPS_unavailable to 1 and exits with code -1.  If no error occurs,
   returns 0.
*/
{
  int status;
  char tmp_buf[80];

  Check_GPS_connection ();

  if (GPS_socket > 0)
    return (0);

  if (GPS_inbuf == NULL)
    GPS_inbuf = nbb_allocate (8192);
  if (GPS_outbuf == NULL)
    GPS_outbuf = nbb_allocate (4096);

  Status ("CONNECTING TO GLOBAL PLAYER SERVICE ...");
  restore_cursor ();
  status = client_init (GPS_IP, GPS_port, 1);
  if (status < 0) {
    Status ("CONNECTING TO GLOBAL PLAYER SERVICE ... UNABLE TO CONNECT");
    GPS_unavailable = 1;
    return (-1);
  }

  GPS_socket = status;
#ifdef F_SETFL
  fcntl (GPS_socket, F_SETFL, O_NDELAY);
#endif

  GPS_request_in_progress = 1;
  status = Perform_GPS_handshake ();
  if (status < 0) {
    Status ("CONNECTING TO GLOBAL PLAYER SERVICE ... ERROR IN HANDSHAKE");
    restore_cursor ();
    Close_GPS_socket ();
    GPS_unavailable = 1;
    return (-1);
  } else if (status) {
    GPS_Comment (GPS_error_buf);
    GPS_Comment ("DO YOU NEED TO UPDATE YOUR COPY OF OKBRIDGE?");
    restore_cursor ();
    GPS_unavailable = 1;
    return (-1);
  }

  Status ("CONNECTING TO GLOBAL PLAYER SERVICE ... CONNECTION ESTABLISHED");
  Refresh_Input_Buffers ();
  GPS_server_message[0] = '\0';

  if (local_player_pw != NULL) {
    GPS_request_in_progress = 0;
    sprintf (tmp_buf, "LOGGING IN AS %s...", local_player_name);
    Status (tmp_buf);
    local_player_id = GPS_Login (local_player_name, local_player_pw);
    Display_Player_Position ();
    GPS_request_in_progress = 1;
    if (client_mode)
      Send_name (Local_table, local_player_name, local_player_id);
  }
    
  return (0);
}


#define CHECK_GPS(x)  if(GPS_check_status()) x

int GPS_check_status ()
/* Checks the current status of our GPS connection.  If there is a
   reason that we cannot honor a request at the moment, we display
   an appropriate error message and return 1.  Otherwise, we return 0.
*/
{
  if (!Use_GPS) {
    Status ("WE ARE NOT USING THE GLOBAL PLAYER SERVICE NOW.");
    return (1);
  } else if (GPS_unavailable) {
    Status ("THE GLOBAL PLAYER SERVICE IS CURRENTLY UNAVAILABLE.");
    return (1);
  } else if (GPS_request_in_progress) {
#ifdef LOGFILE
    fprintf 
      (net_log, "GPS ** ERROR -- WE ARE ALREADY PROCESSING A GPS REQUEST.\n");
    fflush (net_log);
#endif
    Status ("ERROR -- WE ARE ALREADY PROCESSING A GPS REQUEST.");
    return (1);
  } else if (Open_GPS_connection()) {
    GPS_Comment ("UNABLE TO CONTACT GLOBAL PLAYER SERVICE.");
    return (1);
  }

  GPS_request_in_progress = 1;
  return (0);

}

static void GPS_advise (buf)
     char *buf;
{
  if (GPS_unavailable || !Use_GPS || (GPS_socket == 0))
    return;

  Enqueue_Advisory_Message (buf);
  Flush_Advisory_Queue ();
}

void GPS_Get_Message_of_the_Day ()
/* Contacts the GPS and prints out the "message of the day". */
{
  int status;
  int line_no = 4;
  int current_display_mode = display_mode;
  int current_input_mode = input_mode;
  char buf[80];

  CHECK_GPS(return);

  GPS_transmit ("MOTD");
  GPS_Ask_for_response ();

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  print (2, 30, "GPS NEWS");

  status = GPS_readline (buf, 80);
  while (status == 0) {
    print (line_no++, 1, buf);
    status = GPS_readline (buf, 80);
  }

  Press_Return_to_Continue ("");
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);

  Conclude_GPS_request ();
}

int GPS_Get_Server_IP (server_name, location, port)
     char *server_name, *location; int *port;
/* Searches the list of currently playing tables for a server whose name
   matches server_name.  If one is found, then copies the corresponding
   location and port to the buffers provided by the caller, and returns 0.
   If no match is found, returns 1.
*/
{
  int status;

  CHECK_GPS(return(1));

  sprintf (GPS_buf, "JOIN %s", (server_name == NULL)? "": server_name);
  GPS_transmit(GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return (-1);

  if (!strncmp(GPS_buf, "ERROR", 5)) {
    GPS_Comment (GPS_buf);
    Conclude_GPS_request ();
    return (-1);
  }

  sscanf (GPS_buf, "%s %d", location, port);
  Conclude_GPS_request ();
  return (0);
}

void GPS_Broadcast_Server ()
/* Sends a message to the global player service notifying it that we
   are serving a table.  Sends the names of each of the players at the
   table in the message.
*/
{
  sprintf (GPS_advise_buf, "*SERVE %d", network_port);
  GPS_advise (GPS_advise_buf);
}

void GPS_Broadcast_Server_Silently ()
/* Same as GPS_Broadcast_Server but does not print an error message if
   we are not using the GPS or if the GPS was found to be unavailable
   in a previous call.
*/
{
  if (!Use_GPS || GPS_unavailable)
    return;

  GPS_Broadcast_Server ();
}

void GPS_Advertise_Message (message)
     char *message;
/* If we are in server mode, then displays the given message along with
   our table announcement in the tables display. */
{

  if (message == NULL)
    GPS_server_message [0] = '\0';
  else {
    strncpy (GPS_server_message, message, 70);
    GPS_server_message [70] = '\0';
  }

  sprintf (GPS_advise_buf, "*NOTE %s", GPS_server_message);
  GPS_advise (GPS_advise_buf);
}

void GPS_List_Tables (clear_code)
     int clear_code;
/* Contacts the GPS for a list of the currently playing tables.
   Lists the tables at the terminal.
*/
{
  int status, no_tables;
  int current_display_mode = display_mode;
  int current_input_mode   = input_mode;
  int table_display_mode   = display_mode;
  int table_input_mode     = input_mode;
  int lines_left_on_page;

  CHECK_GPS(return);

  GPS_transmit("TABLES");
  GPS_Ask_for_response ();
  
  no_tables = 0;
  lines_left_on_page = 0;
  status = GPS_readline(GPS_buf, 100);

  exit_requested = 0;

  if ((terminal_lines < 35) && clear_code && !status) {
    table_display_mode = TALK_DISPLAY;
    table_input_mode   = TALK_INPUT;
    Clear_Comment_Display ();
    Set_Display_Mode (TALK_DISPLAY);
    Set_Input_Mode (TALK_INPUT);
  } else
    GPS_Comment (" ");

  while (status == 0) {
    if (lines_left_on_page <= 0) {
      if (no_tables == 0) {
	Set_Display_Mode (table_display_mode);
	Set_Input_Mode (table_input_mode);
      } else {
	GPS_Pause ("PRESS <ESC> TO SEE THE NEXT PAGE OF TABLES.");
	if (!exit_requested)
	  Clear_Comment_Display ();
      }
      if (exit_requested || !GPS_request_in_progress) break;

      GPS_Comment ("LIST OF CURRENTLY PLAYING TABLES");
      GPS_Comment (" ");
      GPS_Comment ("Name      Players  Level           Note");
      GPS_Comment ("----      -------  -----           ----");

      lines_left_on_page = terminal_lines - 10;
    }

    GPS_Comment (GPS_buf);
    status = GPS_readline(GPS_buf, 100);
    no_tables += 1;
    lines_left_on_page -= 1;
  }

  if (!exit_requested) {
    if (no_tables == 0) {
      GPS_Comment ("NO TABLES ARE CURRENTLY BEING SERVED.");
      if (server_mode)
	GPS_Comment 
	  ("TYPE '/PUBLISH' TO PUBLISH THIS TABLE IN THE GPS DATABASE.");
      else
	GPS_Comment 
	  ("TYPE '/SERVE' IF YOU WOULD LIKE TO SERVE A TABLE YOURSELF.");
    } else if (current_display_mode == TALK_DISPLAY) {
      if (lines_left_on_page > 1)
	GPS_Comment (" ");
      GPS_Comment ("TYPE '/JOIN name' TO JOIN THE TABLE SERVED BY name.");
    }
  }

  if ((current_display_mode != TALK_DISPLAY) && (no_tables > 0)
      && (current_display_mode != table_display_mode))
    GPS_Pause ("PRESS <ESC> TO CONTINUE ...");
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
  
  Conclude_GPS_request ();
}

void GPS_List_Players (server)
     char *server;
/* Lists the players at the given table. */
{
  int status, no_players;
  int current_display_mode = display_mode;
  int current_input_mode   = input_mode;
  int lines_left_on_page;
  char buf[80];

  CHECK_GPS(return);

  sprintf (GPS_buf, "PLAYERS %s", (server == NULL)? "": server);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  no_players = 0;
  exit_requested = 0;

  if (server != NULL) {
    status = GPS_readline(GPS_buf, 100);
    while (status == 0) {
      if (no_players == 0) {
	GPS_Comment (" ");
	sprintf (buf, "LIST OF PLAYERS AT %s's TABLE:", server);
	GPS_Comment (buf);
      }
      GPS_Comment (GPS_buf);
      status = GPS_readline(GPS_buf, 100);
      no_players += 1;
    }
    if (no_players == 0) {
      sprintf (buf, "%s IS NOT HOSTING A TABLE.", server);
      GPS_Comment (buf);
    }
  } else {
    status = GPS_readline(GPS_buf, 100);
    lines_left_on_page = 0;
    while (status == 0) {
      if (lines_left_on_page <= 0) {
	if (no_players == 0) {
	  Set_Display_Mode (TALK_DISPLAY);
	  Set_Input_Mode (TALK_INPUT);
	} else
	  GPS_Pause ("PRESS <ESC> TO SEE THE NEXT PAGE OF PLAYERS.");
	if (exit_requested || !GPS_request_in_progress) break;
	Clear_Comment_Display ();
	GPS_Comment (" ");
	sprintf (buf, "%-10s %-8s %s", "TABLE", "SEAT", "NAME");
	GPS_Comment (buf);
	sprintf (buf, "%-10s %-8s %s", "-----", "----", "----");
	GPS_Comment (buf);
	lines_left_on_page = terminal_lines - 9;
      }
      GPS_Comment (GPS_buf);
      status = GPS_readline(GPS_buf, 100);
      no_players += 1;
      lines_left_on_page -= 1;
    }
    if ((current_display_mode != TALK_DISPLAY) && (no_players > 0))
      GPS_Pause ("PRESS <ESC> TO CONTINUE ...");
    Set_Display_Mode (current_display_mode);
    Set_Input_Mode (current_input_mode);
    if (no_players == 0)
      GPS_Comment ("NO TABLES ARE CURRENTLY BEING SERVED.");
  }
  Conclude_GPS_request ();

}

void GPS_Find_Players (player_list)
     char *player_list;
/* Looks for the players given in the list at all the tables
   and reports on their location 

   This code generously contributed by Thomas Dressing (dressing@mcs.com)
*/
{
  int status, no_players, match;
  char buf[80];
  char original_list[100];
  char server[20], name[20], position[20], garbage[80];
  char lower_name[20];
  int i,j;
  
  CHECK_GPS(return);
  
  sprintf (GPS_buf, "PLAYERS" );
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();
  
  no_players = 0;
  exit_requested = 0;
  
  if (strlen( player_list ) > 80 ) {
    GPS_Comment ( "PLAYER LIST IS LONGER THAN 80 CHARACTERS." );
  } else {
    strcpy( original_list, player_list );
    /* Take the player list and make all lower case. */
    for( i=0; player_list[i]; i++ )
      player_list[i] = tolower( player_list[i] );
    /* For each player returned by GPS, check for a match */
    status = GPS_readline( GPS_buf, 100 );
    while( status==0 ) {
      sscanf( GPS_buf, "%s %s %s %s", server, position, name, garbage );
      /* take the player name and convert to lower case */
      for( i=0; name[i]; i++ )
	lower_name[i] = tolower( name[i] );
      lower_name[i] = 0;
      /* look for the substring of name in the player_list */
      for( i=0; player_list[i]; i++ ) {
	if( i==0 || player_list[i-1]==' ' ) {
	  for( j=0, match=1; lower_name[j] && match; j++ )
	    match = (player_list[i+j] == lower_name[j]);
	  if( match )
	    match = (player_list[i+j] == ' ' || player_list[i+j] == 0 );
	  if( match )
	    break;
	}
      }
      if( match ) {
	/* found this player at this table */
	switch( position[0] ) {
	case 'N' : /* playing North */
	  strcpy( position, "playing north" );
	  break;
	case 'S' : /* playing South */
	  strcpy( position, "playing south" );
	  break;
	case 'E' : /* playing East */
	  strcpy( position, "playing east" );
	  break;
	case 'W' : /* playing West */
	  strcpy( position, "playing west" );
	  break;
	  default : /* Must be otherwise observing */
	    strcpy( position, "watching" );
	  break;
	}
	sprintf( buf, "%s is %s at %s's table.", name, position, server);
	GPS_Comment( buf );
	/* remove the name from the player_list, match starts at i */ 
	for (j=0; name[j]; j++)
	  player_list[i+j] = ' ';
      }
      status = GPS_readline( GPS_buf, 100 );
    }
    /* for each player that was not found, it remains in the player_list */
    /* better print out a message that they were not found */
    for( i=0; player_list[i]; i++ ) {
      if( player_list[i] != ' ' ) {
	for( j=0; !(original_list[i+j]==' ' || original_list[i+j]==0); 
	    j++ )
	  name[j] = original_list[i+j];
	name[j] = 0;
	i += j-1;
	sprintf( buf, "%s is not playing now.", name );
	GPS_Comment( buf );
      }
    }
  }
  
  Conclude_GPS_request ();
}

void GPS_End_Server_Mode ()
{
  GPS_advise ("*CLOSE");
}

void GPS_Directory ()
/* Returns a listing of the email duplicate files which are available. */
{
  int status;

  CHECK_GPS(return);

  GPS_transmit ("DIR");
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  while (status == 0) {
    GPS_Comment (GPS_buf);
    status = GPS_readline (GPS_buf, 100);
  }
  Conclude_GPS_request ();
}

int GPS_Download (download_file)
     char *download_file;
/* Downloads the specified file from the global player service. 
   Returns 0 if successful and 1 if unsuccessful.
*/
{
  int status;
  FILE *fp;

  CHECK_GPS(return(1));

  fp = fopen (download_file, "w");
  if (fp == NULL) {
    sprintf (GPS_buf, "ERROR OPENING LOCAL FILE %s: %s", download_file,
	     sys_errlist[errno]);
    Moderator_Comment (GPS_buf);
    return (1);
  }

  sprintf (GPS_buf, "DOWNLOAD %s", download_file);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status) {
    Status ("ERROR IN ACCESSING DOWNLOAD FILE.");
    fclose (fp);
    Conclude_GPS_request ();
    return (1);
  }

  if ((GPS_buf[0] == 'E') && (GPS_buf[1] == 'R') && (GPS_buf[2] == 'R')) {
    sprintf (GPS_buf, "THERE IS NO SUCH DUPLICATE FILE %s.", download_file);
    Status (GPS_buf);
    fclose (fp);
    Conclude_GPS_request ();
    return (1);
  }

  Status ("DOWNLOADING FILE FROM GPS ...");
  restore_cursor ();
  while (status == 0) {
    fprintf (fp, "%s\n", GPS_buf);
    status = GPS_readline (GPS_buf, 100);
  }
  fclose (fp);
  Conclude_GPS_request ();
  sprintf (GPS_buf, "THE FILE %s HAS BEEN DOWNLOADED SUCCESSFULLY.",
	   download_file);
  Status (GPS_buf);
  return (0);
}

static Board *GPS_Download_Board ()
/* Downloads a board from the GPS.  Returns NULL if no board available or
   if an error occurred.
 */
{
  Board *b;
  Play_record *p;
  char buf1[100], buf2[100], buf3[100];
  int status;

  status = GPS_readline (buf1, 100);
  if (status)
    return (NULL);
  else if (!strncmp(buf1, "ERROR", 5)) {
    GPS_Comment (buf1);
    return (NULL);
  }

  GPS_readline (buf2, 100);
  b = Decode_board (buf1, buf2);
  if (b == NULL)
    return (NULL);

  status = GPS_readline (buf1, 100);
  while ((status == 0) && (buf1[0] != '-')) {
    GPS_readline (buf2, 100);
    GPS_readline (buf3, 100);
    p = Decode_play_record (buf1, buf2, buf3);
    if (p == NULL)
      return (NULL);
    if (p->bidding_completed)
      Compute_contract (b, p);
    Compute_Scores_for_Play_Record (b, p);
    p->next = b->play_records;
    b->play_records = p;
    status = GPS_readline (buf1, 100);
  }
  Compute_Scores_for_Board (b);

  return (b);
}

Board *GPS_Get_Next_Board (scoring_type)
     char *scoring_type;
/* Downloads the next duplicate board from the GPS. */
{
  Board *b;

  CHECK_GPS(return(NULL));

  sprintf (GPS_buf, "BOARD %s %s %d %s %d %s %d %s %d", scoring_type,
	   Local_table->Seats[PLAYER_NORTH].player_name,
	   Local_table->Seats[PLAYER_NORTH].connection->player_id,
	   Local_table->Seats[PLAYER_EAST].player_name,
	   Local_table->Seats[PLAYER_EAST].connection->player_id,
	   Local_table->Seats[PLAYER_SOUTH].player_name,
	   Local_table->Seats[PLAYER_SOUTH].connection->player_id,
	   Local_table->Seats[PLAYER_WEST].player_name,
	   Local_table->Seats[PLAYER_WEST].connection->player_id);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  Status ("DOWNLOADING NEXT BOARD FROM GPS ...");
  restore_cursor ();

  b = GPS_Download_Board ();
  Conclude_GPS_request ();
  Clear_Status ();
  return (b);
}

void GPS_Upload_Play_Record (b, p)
     Board *b; Play_record *p;
/* Uploads the play record p for the current email board to the GPS. */
{
  char buf1[100], buf2[100], buf3[100];

  CHECK_GPS(return);

  sprintf (GPS_buf, "PLAYREC %s %d %d", 
	   b->source,
	   (b->scoring_mode == MP_SCORING)? 0: 1,
	   b->serial_no);
  GPS_transmit (GPS_buf);

  Encode_play_record (p, buf1, buf2, buf3);
  GPS_transmit (buf1);
  GPS_transmit (buf2);
  GPS_transmit (buf3);
  GPS_Ask_for_response ();
  Conclude_GPS_request ();
}

Board *GPS_Download_Results (player_name)
     char *player_name;
/* Downloads the all of the recorded results in the GPS for the given 
 * player.  Returns a list of boards or NULL if no results are available.
 */
{
  Board *results_head, *results_tail;

  CHECK_GPS (return(NULL));

  sprintf (GPS_buf, "RESULTS %s", player_name);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  results_head = results_tail = GPS_Download_Board ();
  while (results_tail != NULL) {
    results_tail->next = GPS_Download_Board ();
    results_tail = results_tail->next;
  }
  Conclude_GPS_request ();
  return (results_head);
}

void GPS_Display_Scoreboard (refresh)
     int refresh;
/* Downloads the scoreboard from the GPS and displays it. */
{
  int line_no = 1;
  int status;
  int current_display_mode = display_mode;
  int current_input_mode = input_mode;
  char buf[80];

  CHECK_GPS (return);

  GPS_transmit ("SCOREBOARD");
/*  GPS_transmit (local_player_name); */
  GPS_Ask_for_response ();

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  clear_screen ();

  status = GPS_readline (buf, 80);
  while (status == 0) {
    print (line_no++, 1, buf);
    status = GPS_readline (buf, 80);
  }

  Press_Return_to_Continue ("");
  if (refresh) {
    Set_Display_Mode (current_display_mode);
    Set_Input_Mode (current_input_mode);
  }
  Conclude_GPS_request ();
}

void GPS_Display_YTD_Scoreboard ()
/* Downloads the scoreboard from the GPS and displays it. */
{
  int line_no = 1;
  int status;
  int current_display_mode = display_mode;
  int current_input_mode = input_mode;
  char buf[80];

  CHECK_GPS (return);

  GPS_transmit ("SCOREBOARD YTD");
/*  GPS_transmit (local_player_name); */
  GPS_Ask_for_response ();

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  clear_screen ();

  status = GPS_readline (buf, 80);
  while (status == 0) {
    print (line_no++, 1, buf);
    status = GPS_readline (buf, 80);
  }

  Press_Return_to_Continue ("");
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
  Conclude_GPS_request ();
}

void GPS_Add_Player (c)
     Connection c;
/* Advises the GPS that the player represented by connection c has joined
   the table. */
{
  if (c->local)
    return;

  sprintf (GPS_advise_buf, "*ADDPL %s %d %s", 
	   c->player_name, c->player_id, seat_names[c->seat]);
  GPS_advise (GPS_advise_buf);
}

void GPS_Delete_Player (c)
     Connection c;
/* Advises the GPS that the player represented by connection c has left
   the table. */
{
  if (c->local) return;

  sprintf (GPS_advise_buf, "*DELPL %s %d", c->player_name, c->player_id);
  GPS_advise (GPS_advise_buf);
}

void GPS_Move_Player (c)
     Connection c;
/* Advises the GPS of the current seat held by the player with connection c.*/
{
  sprintf (GPS_advise_buf, "*MOVEPL %s %d %s", 
	   c->player_name, c->player_id, seat_names[c->seat]);
  GPS_advise (GPS_advise_buf);
}

void GPS_Refresh_Players ()
/* Sends complete information about the seats held by all players at this
   table. */
{
  Connection c;

  GPS_advise ("*CLRPL");
  GPS_Move_Player (Local_Player_Connection);
  
  FOREACH_PLAYER (c, Local_table)
    if (!c->local && (c->state == CSTATE_PLAYING))
      GPS_Add_Player (c);
}


void GPS_Get_Scores (mp, imp)
     Score *mp, *imp;
/* Downloads the current scores for the local player. */
{
  int status;

  CHECK_GPS(return);
  GPS_transmit ("SCORES");
  GPS_Ask_for_response ();

  bzero (mp, sizeof(Score));
  bzero (imp, sizeof(Score));

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return;

  sscanf (GPS_buf, "%d %d %d %d",
	  &mp->duplicate.competitive.mp.mp_total,
	  &mp->duplicate.competitive.mp.mp_tables,
	  &imp->duplicate.competitive.imp.imp_total,
          &imp->duplicate.competitive.imp.imp_tables);

  Conclude_GPS_request ();
}

int GPS_Check_for_Login (login_name)
     char *login_name;
/* int GPS_Check_for_Login (char *login_name); */
/* Returns TRUE if there is an account assigned to login_name, and false
   otherwise. */
{
  int status;
  int response;

  CHECK_GPS(return(0));
  sprintf (GPS_buf, "LOGIN %s", login_name);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return (-1);

  if (strcasecmp(GPS_buf, "yes"))
    response = 0;
  else 
    response = 1;

  Conclude_GPS_request ();
  return (response);
}

int GPS_Login (login_name, pw)
     char *login_name; char *pw;
/* int GPS_Login (char *login_name, char *pw); */
/* Attempts to login the using the given login_name and pw.  If successful,
   returns a non-negative id assigned by the GPS.  Otherwise, returns -1
   and prints an error message. */
{
  int status;

  CHECK_GPS(return(-1));
  sprintf (GPS_buf, "LOGIN %s %s", login_name, pw);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return (-1);

  if (!strncmp(GPS_buf, "ERROR", 5)) {
    GPS_Comment (GPS_buf);
    GPS_Comment ("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
    Conclude_GPS_request ();
    return (-1);
  } else {
    sscanf (GPS_buf, "%d", &status);
    Conclude_GPS_request ();
    return (status);
  }

}

int GPS_Create_Account (login_name, pw)
     char *login_name; char *pw;
/* int GPS_Create_Account (char *login_name, char *pw); */
/* Creates an account with given login_name and pw.  If successful, returns
   a non-negative id assigned by the GPS.  Otherwise, returns -1 and prints
   an error message.
*/
{
  int status;

  CHECK_GPS(return(-1));
  sprintf (GPS_buf, "NEWUSER %s %s", login_name, pw);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return (-1);

  if (!strncmp(GPS_buf, "ERROR", 5)) {
    GPS_Comment (GPS_buf);
    Conclude_GPS_request ();
    return (-1);
  } else {
    sscanf (GPS_buf, "%d", &status);
    Conclude_GPS_request ();
    return (status);
  }
}

void GPS_Change_Password (oldpw, newpw)
     char *oldpw; char *newpw;
/* void GPS_Change_Password (char *oldpw, char *newpw); */
/* Attempts to change the password for the user. */
{
  int status;

  CHECK_GPS(return);
  sprintf (GPS_buf, "PASSWORD %s %s", oldpw, newpw);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (GPS_buf, 100);
  if (status < 0)
    return;

  if (!strncmp(GPS_buf, "ERROR", 5))
    GPS_Comment (GPS_buf);

  Conclude_GPS_request ();
}

void GPS_Display_Stats (player_name)
     char *player_name;
/* Displays the statistics for the given player. */
{
  int line_no = 1;
  int status;
  int current_display_mode = display_mode;
  int current_input_mode = input_mode;
  char buf[80];

  CHECK_GPS (return);

  sprintf (GPS_buf, "STATS %s", 
	   (player_name == NULL)? local_player_name: player_name);
  GPS_transmit (GPS_buf);
  GPS_Ask_for_response ();

  status = GPS_readline (buf, 80);

  if (status != 0) {
    Conclude_GPS_request ();
    return;
  } else if (!strncmp(buf, "ERROR", 5)) {
    GPS_Comment (buf);
    Conclude_GPS_request ();
    return;
  }

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  while (status == 0) {
    print (line_no++, 1, buf);
    status = GPS_readline (buf, 80);
  }

  Press_Return_to_Continue ("");
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
  Conclude_GPS_request ();
}

void GPS_Set_Field (field_name, field_value)
     char *field_name; char *field_value;
/* void GPS_Set_Field (char *field_name, char *field_value); */
/* Sets an informational field for the local player. */
{
  if (local_player_id < 0) {
    Status ("YOU MUST FIRST LOGIN TO MODIFY YOUR PLAYER INFORMATION.");
    return;
  }

  sprintf (GPS_advise_buf, "*FIELD %s %s", field_name, field_value);
  GPS_advise (GPS_advise_buf);
}

void GPS_Perish ()
/* void GPS_Perish (void); */
/* If we are serving, then removes the current table from the GPS. */
{
  GPS_advise ("*PERISH");
}

void GPS_Publish ()
/* void GPS_Publish (void); */
/* Restores the current table to the GPS listing. */
{
  GPS_advise ("*PUBLISH");
}

