/* display.h
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * The DISPLAY module for the bridge program is responsible for
 * presenting the information regarding various aspects of the
 * game to the screen.  The organization of the screen is embedded
 * within this module, and there are entry points for giving
 * updates to the screen at each step of the game.
 *
 * This module does not make any calls to the operating system
 * directly.  Instead, all of its output functions are channeled
 * through the TERMINAL module.
 *
 */

/*
  The display can be in one of four modes:
    TALK:  In this mode, almost the entire screen is used for displaying
           talk messages between the players.  
    BID:   This is the display mode used during bidding.  
    PLAY:  This is the display mode used during play.  
    SCORE: This display mode is used for displaying the history of scores
           or the results of competitive play.
    HELP:  This mode is used for displaying help screens.
    MANUAL:In this mode, the display is given only minimal support by
           the display module.  Refresh requests are not honored.
           The status line is at the bottom of the screen.

  In the bid, play and score three score modes, the lower 6 lines of the 
  display are reserved for talk messages between the players.  In the BID 
  and PLAY modes, an area in the upper right corner of the display is used for
  a scoring panel.  The scoring panel contains information such as the
  current vulnerabilities, the number of tricks which have been played
  form each side, and the total score of each side.
*/

#define TALK_DISPLAY      0
#define BIDDING_DISPLAY   1
#define PLAYING_DISPLAY   2
#define SCORING_DISPLAY   3
#define HELP_DISPLAY      4
#define MANUAL_DISPLAY    5

#ifdef _DISPLAY_
  int display_mode = TALK_DISPLAY; /* The current display mode. */
#else
  extern int display_mode;
#endif

extern void Initialize_Display ();
/* Should be called once when the program starts up. */
 
extern void Reset_Display ();
/* Redraws the main features of the screen.  Used in the process
   of doing a 'refresh'. */
 
extern void Refresh_Display ();
/* Updates the entire screen. */

extern void Refresh_Playing_Area ();
/* Refreshes the 'upper half' of the screen, where the bids and plays
   are displayed. */

extern void Refresh_Player ();
/* Refreshes the display of the holdings of a particular player. */

extern void Set_Display_Mode ();
/* void Set_Display_Mode (int mode); */
/* Sets the display mode.  If the mode is different than the current mode,
   then redraws the screen appropriately.
*/
 
extern void Display_Player_Position ();
/* Displays okbridge header and the seat in which the local player
   is sitting. */
 
/* The scoring display gives information about the following attributes
 * for each side:
 *
 * -- The number of tricks taken in the current hand.
 * -- The current 'above the line' score.
 * -- The current 'below the line' score.
 * -- The vulnerabilities of each side.
 *
 */
 
extern void Display_Tricks_Taken      ();
extern void Display_Above_Line_Points ();
extern void Display_Below_Line_Points ();
extern void Display_Vulnerabilities   ();
extern void Display_Total_Time ();
 
/* Updates to the bidding and playing displays are generally handled
 * by either calling Refresh_Playing_Display () or Refresh_Player ().
 *
 * Hands are revealed by calling the routines Display_Hand () and
 * Display_Partial_Hand ().
 *
 * The cards which are shown as being played on the table are 
 * controlled with the routines Clear_Plays () and Display_Play ().
 */
 
extern void Display_Partial_Hand ();
extern void Display_Hand ();
extern void Display_Player ();
extern void Display_Play ();
extern void Clear_Plays  ();


/* If a hand is been played by multiple tables, then we can display 
 * the records of play for each table to the user.  
 *
 * First, we setup the scoring display with a call to Setup_Scoring_Display.
 * This displays the first page of scores.  
 *
 * The function More_Scores_to_Display () returns TRUE when not all of
 * the scores have yet been displayed.
 *
 * And the function Display_More_Scores () displays the next page of scores.
 */

extern void Display_First_Page_of_Scores ();
/* void Display_First_Page_of_Scores (Board *b); */
/* Initiales the scoring display and displays the first page of results from
 * board b. 
 */

extern int More_Scores_to_Display ();
/* int More_Scores_to_Display (void); */
/* Returns true if not all of the results have been displayed from the
 * board which is currently being displayed. 
 */

extern void Display_More_Scores ();
/* void Display_More_Scores (void); */
/* Displays the next page of scores for the board b. */


 
/* One line of the display is reserved for displaying special messages about
 * the status of the game.  The following two procedures are given for
 * manipulating this display.
 */
 
extern void Status ();
/* void Status (char *status_message); */
/* Displays the given status message. */

extern void Clear_Status ();
/* void Clear_Status (void); */
/* Clears the status display. */
 
extern void Lock_Status ();
/* void Lock_Status (char *message); */
/* Locks the message into the status display.  If the status display is 
   already locked, pushes the current message onto a stack, to be redisplayed
   when Unlock_Status is called.
*/

extern void Unlock_Status ();
/* void Unlock_Status (void); */
/* Unlocks the status display.  Redisplays the previous message on the
   stack of status messages.  */
 
extern void Reset_Status ();
/* void Reset_Status (void); */
/* Clears all messages from the stack of status messages and resets the
   status display. */

extern void Refresh_Status_Display ();
/* void Refresh_Status_Display (void); */
/* Redisplays the contents of the status buffer. */


/* The bottom part of the screen is used for the exchange of comments
 * between the players.  The following procedures are used for managing
 * this part of the display.
 *
 * During startup, the entire screen is devoted to a comments display.
 */

/* A structure defining player comments which should be ignored. */
typedef struct ignore_struct {
  char *str; /* the string to be ignored. */
  struct ignore_struct *next;
} ignore;

extern void Initialize_Player_Comments ();
extern void Reinitialize_Player_Comments ();
extern void Display_Player_Comment ();
extern void Refresh_Player_Comments ();
extern void Clear_Comment_Display ();

extern void Moderator_Comment ();
 /* displays a comment from the MODERATOR. */
extern void Network_Comment ();
 /* displays a comment from the NETWORK. */

extern void Suspend_Comment_Display ();
extern void Continue_Comment_Display ();

