/* display.c -- Display functions for OKbridge program.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */
 
#include <stdio.h>
#include <string.h>

extern char *malloc ();

#define _DISPLAY_
 
#include "state.h"
#include "terminal.h"
#include "display.h"
#include "help.h"
#include "input.h"

#ifdef LOGFILE
extern FILE *net_log;
#endif

#ifdef VMS
#define status wstatus
#endif

extern int show_alerts;
extern int initializing;
extern int exit_requested;

typedef int window[4];
 
/* Conceptually, we think of the screen as being divided into a number
   of windows, where each window describes one particular type of activity
   or aspect of the game.  Therefore, we make the following definitions,
   although these are only guidelines for operation of the display.
 
Name of
Window          	 Ymin	Xmin	Height	Width
-------			 ----	----	----	----			*/
window title 	     = { 1,	1,	6,	25};
window top_hand      = { 2,	30,	4,	26};
window scores	     = { 1,	59,	6,	26};
window left_hand     = { 8,	6,	6,	18};
window playing_board = { 5,	30,	9,	20};
window bidding_board = { 5,	1,	9,	57};
window right_hand    = { 8,	56,	6,	23};
window input	     = { 13,	1,	6,	26};
window status	     = { 18,    1,	1,	79};
window bottom_hand   = { 13,	30,	4,	26};
window help	     = { 13,	53,	6,	26};
window converse	     = { 19,	1,	6,	78};
 
#define XMIN(w)		w[1]
#define YMIN(w)         w[0]
#define XMAX(w)		(w[1] + w[3] - 1)
#define YMAX(w)		(w[0] + w[2] - 1)
#define HEIGHT(w)	w[2]
#define WIDTH(w)	w[3]

typedef struct xy_pair_struct { int y; int x;} xy_pair;

int screen_location[4];
  /* This is an array which translates seat names to screen locations.  
     This array indexed by North, South, East, and West, where the
     indices are defined as follows:
       SOUTH:  Bottom of screen.
       WEST:   Left side of screen.
       NORTH:  Top of screen.
       EAST:   Right side of screen.
  */

/* The following list of points describes where the player's names will
   be printed.  The location of the name of the player at the bottom screen
   has to be modified slightly on the fly.  The ordering of this array
   is north, east, south, west (top, right, bottom, left). */
static xy_pair base_name_location [4] = 
  {{2, 30}, {7, 51}, {13, 30}, {7, 17}};

static xy_pair centered_name_location [4] =
  {{4, 40}, {9, 51}, {14, 40}, {9, 29}};

static xy_pair name_location[4];

/* The following arrays determine the positions where the hands will
   be displayed. */
static xy_pair top_hand_location [4] =
  {{5, 32}, {4, 32}, {3, 32}, {2, 32}};

static xy_pair right_hand_location [4] =
  {{11, 51}, {10, 51}, {9, 51}, {8, 51}};

static xy_pair bottom_hand_location [4] = 
  {{16, 32}, {15, 32}, {14, 32}, {13, 32}};

static xy_pair left_hand_location [4] =
  {{11, 17}, {10, 17}, {9, 17}, {8, 17}};

static xy_pair *base_hand_location [4] =
  {top_hand_location, right_hand_location, 
   bottom_hand_location, left_hand_location};

static xy_pair base_play_location [4] =
  {{7, 40}, {9, 45}, {11, 40}, {9, 33}};

static xy_pair top_bid_location [8] =
  {{2, 32}, {3, 32}, {4, 32}, {5, 32}, {2, 40}, {3, 40}, {4, 40}, {5, 40}};

static xy_pair top_revealed_bid_location [8] = 
  {{2, 44}, {3, 44}, {4, 44}, {5, 44}, {2, 52}, {3, 52}, {4, 52}, {5, 52}};

static xy_pair right_bid_location [8] =
  {{8, 51}, {9, 51}, {10, 51}, {11, 51},
   {8, 59}, {9, 59}, {10, 59}, {11, 59}};

static xy_pair right_revealed_bid_location [8] =
  {{8, 64}, {9, 64}, {10, 64}, {11, 64},
   {8, 72}, {9, 72}, {10, 72}, {11, 72}};

static xy_pair bottom_bid_location [8] =
  {{13, 32}, {14, 32}, {15, 32}, {16, 32}, 
   {13, 40}, {14, 40}, {15, 40}, {16, 40}};

static xy_pair bottom_revealed_bid_location [8] =
  {{13, 44}, {14, 44}, {15, 44}, {16, 44},
   {13, 52}, {14, 52}, {15, 52}, {16, 52}};

static xy_pair left_revealed_bid_location [8] =
  {{8, 2}, {9, 2}, {10, 2}, {11, 2}, 
   {8, 10}, {9, 10}, {10, 10}, {11, 10}};

static xy_pair left_bid_location [8] =
  {{8, 17}, {9, 17}, {10, 17}, {11, 17},
   {8, 24}, {9, 24}, {10, 24}, {11, 24}};

static xy_pair *base_bid_location [4] =
  {top_bid_location, right_bid_location, 
   bottom_bid_location, left_bid_location};

static xy_pair *base_revealed_bid_location [4] =
  {top_revealed_bid_location, right_revealed_bid_location,
   bottom_revealed_bid_location, left_revealed_bid_location};



static int comments_suspended = 0;
 
static char line_buf [81];
static int  status_loc;

#define COMMENT_LENGTH 78

typedef char comment_line [COMMENT_LENGTH];
typedef comment_line *comment_buf;

static int full_screen_talk_start = 3;
  /* The first line of the screen on which comments will be displayed,
     if the full screen is used for talking.  */
static int full_screen_talk_end  = 22;
  /* The last line on which comments will be displayed for full screen mode. */
static int full_screen_talk_size = 20;
  /* The total number of lines in the full screen talk display. */
static int full_screen_cursor = 0;
  /* The line in the full screen buffer where the next talk message will be
     displayed. */

static int playing_screen_talk_start = 20;
  /* The first line of the screen on which comments will be displayed,
     if we are in playing mode. */
static int playing_screen_talk_end = 24;
  /* last comment line for playing screen. */
static int playing_screen_talk_size = 5;
  /* The total number of lines in the playing screen talk display. */

static int talk_start, talk_end;
  /* The current first and last lines for display of player comments. */
static int talk_size;
  /* talk_end - talk_start + 1. */

static comment_line *full_screen_comments = NULL, 
                    *playing_screen_comments = NULL,
                    *player_comments;

static char Table_display [4][20];
  /* The most recent bids and plays are displayed "on the table." */
static char Table_play [4][20];
  /* The most recently played cards. */
static char Table_bid[4][20];
  /* The most recent bids. */

static int status_display_level = 0;
  /* The number of locks which have been placed on the status display. */
static char status_buf [10] [80];
  /* For each lock level, a record of the status message at that level. */

Board *Board_for_display;
  /* This is a pointer to the board which is being displayed in the
     SCORING_DISPLAY mode. */
Play_record *Current_page_of_scores = NULL;
  /* The first play record which is being displayed on the current page
     of scores. */
int Current_score_record = 0;
  /* The index of the first score record which is displayed on this page. */
Play_record *Next_page_of_scores = NULL;
  /* The first play record which will be displayed on the next page 
     of scores. */
int Next_score_record = 0;
  /* The index of the first score record which will be displayed on the
     next page. */

char total_time_buffer [20];
  /* A buffer used for displaying the total time playing the current hand. */
char local_time_buffer [20];
  /* A buffer used for displaying the time used by the local player. */
    
ignore *head_ignore = NULL;
  /* A list of strings that cause comments to be ignored. */

int result_mode = 0;
  /* Display result mode, 0 = all scores, 1 = summary of scores */

Board *result_board;
  /* Stores the board that was used for most recent /results command. */

void Select_Talk_Mode ();
void Select_Conversation_Mode ();
void Display_Page_of_Scores ();
void Refresh_Time_Display ();
static void Display_suit ();
void Display_Contract ();
void Display_Playing_Board ();


static void clear_screen_area (ymin, xmin, height, width)
       int ymin, xmin, height, width;
{
	int i;
 
	for (i = 0; i < width; i++) line_buf[i] = ' ';
	line_buf[width] = '\0';
	for (i = 0; i < height; i++) print (ymin+i, xmin, line_buf);
}

static void Display_Board_Number ()
{
  if ((Local_board != NULL) && (Local_board->source != NULL)) {
    sprintf (line_buf, "%s NO %d", Local_board->source,
	     Local_board->serial_no);
    print (YMIN(title)+1,  XMIN(title),  line_buf);
  } else {
    sprintf (line_buf, "%d", hands_played);
    print (YMIN(title)+1, XMIN(title)+6, line_buf);
  }

  sprintf (line_buf, "DEALER: %s", 
	   PLAYER_NAME(Local_table, Local_board->dealer));
  print (YMIN(title)+3, XMIN(title), line_buf);

}

static void Redraw_Scoring_Panel ()
/* Redraws the scoring panel in the upper corner of the screen, which
   displays the vulnerabilities, number of tricks taken, and scores.
*/
{
  /* Setup the scoreboard: */
  if (local_player >= 4) {
    print (YMIN(scores), XMIN(scores) + 8,   "  N-S");
    print (YMIN(scores), XMIN(scores) + 16,  "  E-W");
  } else if (side_of(local_player) == SIDE_NS) {
    print (YMIN(scores), XMIN(scores) + 8,   "   WE");
    print (YMIN(scores), XMIN(scores) + 16,  " THEY");
  } else {
    print (YMIN(scores), XMIN(scores) + 16,  "   WE");
    print (YMIN(scores), XMIN(scores) + 8,   " THEY");
  }
/*
  underline (YMIN(scores), XMIN(scores) + 8, 5);
  underline (YMIN(scores), XMIN(scores) + 16, 5);
*/
  print (YMIN(scores)+1, XMIN(scores)+2, "TRICKS");
  print (YMIN(scores)+2, XMIN(scores)+2, "VUL");
  
  switch (Local_board->scoring_mode) {
  case RUBBER_SCORING:
    print (YMIN(scores)+3, XMIN(scores)+2, "ABOVE");
    print (YMIN(scores)+4, XMIN(scores)+2, "BELOW");
    print (YMIN(title)+1,  XMIN(title),  "HAND");
    break;
  case DUPLICATE_SCORING:
    print (YMIN(scores)+3, XMIN(scores)+2, "TOTAL");
    print (YMIN(scores)+4, XMIN(scores)+2, "PREV");
    print (YMIN(title)+1,  XMIN(title),  "BOARD");
    break;
  case IMP_SCORING:
    print (YMIN(scores)+3, XMIN(scores)+2, "TOTAL");
    print (YMIN(scores)+4, XMIN(scores)+2, "IMP");
    print (YMIN(title)+1,  XMIN(title),  "BOARD");
    break;
  case MP_SCORING:
    print (YMIN(scores)+3, XMIN(scores)+2, "TOTAL");
    print (YMIN(scores)+4, XMIN(scores)+2, "MP");
    print (YMIN(title)+1,  XMIN(title),  "BOARD");
    break;
  default:
    break;
  }

  Display_Tricks_Taken ();
  Display_Above_Line_Points ();
  Display_Below_Line_Points ();
  Display_Vulnerabilities ();
  Display_Board_Number ();

}

static void Redraw_Conversation_Area ()
{
  int i;

  /* Setup the conversational area: */
  for (i = 0; i <= XMAX(converse); i++) line_buf[i] = '-';
  line_buf[29] = '+';
  line_buf[72] = '+';
  line_buf[XMAX(converse)+1] = '\0';
  print (YMIN(converse), XMIN(converse), line_buf);
  for (i = talk_start; i <= talk_end; i++) {
    print (i, XMIN(converse), "|");
    print (i, XMAX(converse)+1, "|");
  }

  /* Setup the input area: */
  print (TALK_ROW, TALK_COL, "TALK");
  status_loc = YMIN(status);

}

void Display_on_Table (p, play)
     int p; char *play;
{
  int x, y, n;
  char *s = "      ";
  
  y = base_play_location[screen_location[p]].y;
  x = base_play_location[screen_location[p]].x;
  n = (play == NULL)? 0: strlen(play);

  switch (screen_location[p]) {
  case PLAYER_NORTH:
  case PLAYER_SOUTH:
    print (y, x - 3, s);
    x -= n / 2;
    break;
  case PLAYER_EAST:
    print (y, x - 4, s);
    x -= (n - 2);
    break;
  case PLAYER_WEST:
    print (y, x, s);
    break;
  }

  if (play == NULL)
    Table_display[p][0] = '\0';
  else {
    if (n > 0)
      print (y, x, play);
    strcpy (Table_display[p], play);
  }
}
 
void Clear_Table  ()
{
  Display_on_Table (PLAYER_NORTH, "");
  Display_on_Table (PLAYER_EAST,  "");
  Display_on_Table (PLAYER_SOUTH, "");
  Display_on_Table (PLAYER_WEST,  "");
}

void Display_Bid_on_Table (p, b)
     int p; char *b;
{
  strcpy (Table_bid[p], b);
  Display_on_Table (p, b);
}

void Display_Play_on_Table (p, b)
     int p; char *b;
{
  strcpy (Table_play[p], b);
  Display_on_Table (p, b);
}

void Clear_Plays_from_Table ()
{
  Display_Play_on_Table (0, "");
  Display_Play_on_Table (1, "");
  Display_Play_on_Table (2, "");
  Display_Play_on_Table (3, "");
}


static void Clear_Player_Area (p)
     int p;
{
  switch (screen_location[p]) {
  case PLAYER_NORTH:
    clear_screen_area (2, 22, 4, 38);
    break;
  case PLAYER_EAST:
    clear_screen_area (7, 50, 5, 29);
    break;
  case PLAYER_SOUTH:
    clear_screen_area (13, 22, 4, 40);
    break;
  case PLAYER_WEST:  
    clear_screen_area (7, 2, 5, 29);
    break;
  }
}

void Recompute_Display_Positions ()
/* Recomputes a number of internal constants for the display. */
{
  int display_pos = local_player;
  int seat_pos    = PLAYER_SOUTH;
  int i;

  if (IS_OBSERVER(display_pos)) display_pos = PLAYER_SOUTH;

  for (i = 0; i < 4; i++) {
    screen_location[display_pos] = seat_pos;
    display_pos = player_next[display_pos];
    seat_pos = player_next[seat_pos];
  }
}

void Display_Player_Name (p)
     int p;
/* Displays the name of the player p in the appropriate position on the
   screen.  If p has not bid, or if the hand is being played and p's hand
   is not visible, then p's name is displayed near the center of the
   screen.  Otherwise, p's name is displayed near his bids and/or cards.
*/
{
  int display_near_center = 0;
    /* true if the players name should be displayed near the center. */

  int turn_number = 0;
    /* the position of the player p relative to the dealer. */

  int screen_pos = screen_location[p];
    /* The position on the screen where p's hand will be displayed. */

  int short_bidding_display = 0;
    /* True if we are in bidding mode and less than 4 rounds of bidding
       have occurred. */

  int dummy = (display_mode == PLAYING_DISPLAY) &&
              (p == player_partner[Local_play->declarer]);

  char *name = PLAYER_NAME(Local_table, p);
  int  len   = strlen(name);

  int i, y, x;

  for (i = Local_board->dealer; i != p; i = player_next[i])
    turn_number++;

  if (revealed_hands[p])
    display_near_center = 0;
  else if (turn_number >= Local_play->no_bids)
    display_near_center = 1;
  else if (display_mode == PLAYING_DISPLAY)
    display_near_center = 1;

  if (display_near_center) {
    y = centered_name_location[screen_pos].y;
    x = centered_name_location[screen_pos].x;
    switch (screen_pos) {
    case PLAYER_NORTH:
    case PLAYER_SOUTH:
      x -= (len+1)/2;
      break;
    case PLAYER_EAST:
      break;
    case PLAYER_WEST:
      x -= len;
      break;
    }
  } else {
    y = base_name_location[screen_pos].y;
    x = base_name_location[screen_pos].x;
    short_bidding_display = !revealed_hands[p] && (Local_play->no_bids < 17)
      && (display_mode == BIDDING_DISPLAY)
      && (screen_pos != PLAYER_EAST);
    switch (screen_pos) {
    case PLAYER_NORTH:
    case PLAYER_SOUTH:
      if (short_bidding_display)
	x += 8;
      if (dummy)
	print (y + 1, x - 7, "(DUMMY)");
      x -= len;
      break;
    case PLAYER_EAST:
      if (dummy)
	print (y, x + len + 1, "(DUMMY)");
      break;
    case PLAYER_WEST:
      if (short_bidding_display)
	x += 7;
      if (x + len > 29)
	x = 29 - len;
      if (dummy)
	print (y, x - 9, "(DUMMY)");
      break;
    }
  }

  print (y, x, name);
  name_location[p].x = x;
  name_location[p].y = y;
}

static int Display_Alert (i, b)
     int i, b;
{
  int alerted = 0;

  if (Local_play->alerts[i] & (1 << ALERT_BY_BIDDER)) {
    if (local_player != player_partner[b])
      alerted = 1;
/*    else if (!FORMAL(Local_table))
      alerted = 1;
*/
  }
  if (Local_play->alerts[i] & (1 << ALERT_BY_PARTNER)) {
    if (local_player != b)
      alerted = 1;
    else if (((Local_table->game_mode == PLAYING_MODE) 
		&& (side_of(local_player) == side_of(Local_play->declarer)))
  	     || ((show_alerts) && (!FORMAL(Local_table))))
      alerted = 1;
  }

  return (alerted);
}

void Display_Player_Bids (p)
     int p;
/* Displays the bids made so far by the player p. 
   Assumes that the area on the screen where the bids will be displayed
   is clean.
*/
{
  int i, b, bid, round;
  xy_pair *bid_location;
  char buf[20];

  if (revealed_hands[p])
    bid_location = base_revealed_bid_location [screen_location[p]];
  else
    bid_location = base_bid_location [screen_location[p]];

  if (Local_table->game_mode == BIDDING_MODE)
    Display_Bid_on_Table (p, "");

  if (Local_play->no_bids <= 32) {
    bid = 0;
    round = 1;
  } else {
    round = (Local_play->no_bids + 3) / 4 - 7;
    bid = (round - 1) * 4;
  }

  for (i = Local_board->dealer; i != p; i = player_next[i])
    bid += 1;

  if (Local_play->no_bids > 16)
    i = 0;
  else if (screen_location[p] == PLAYER_WEST)
    i = 4;
  else if (!revealed_hands[p] && (screen_location[p] != PLAYER_EAST))
    i = 4;
  else
    i = 0;

  if (bid < Local_play->no_bids) {
    while (bid < Local_play->no_bids) {
      print (bid_location[i].y, bid_location[i].x, "       ");
      sprintf (buf, "%d %s%c", round, 
	       (Local_play->bids[bid] == BID_PASS)? 
	       "--": bid_names[Local_play->bids[bid]],
	       Display_Alert (bid, p)? '!': ' ');
      print (bid_location[i].y, bid_location[i].x, buf);
      bid += 4;
      i += 1;
      round += 1;
    }

    if ((Local_table->game_mode == BIDDING_MODE) &&
	(Local_play->next_player != p)) {
      bid -= 4;
      b = Local_play->bids[bid];
      sprintf (buf, "%s%s", (b == BID_PASS)? "--": bid_names[b],
	       Display_Alert (bid, p)? "!": "");
      Display_Bid_on_Table (p, buf);
    }
  }
}

void Display_Final_Contract (b, p)
     Board *b; Play_record *p;
{
  char doubled_buf[40], buf[40];
  int i;

  for (i = 0; i < 4; i++)
    Display_Bid_on_Table (i, "");

  if (p->contract == BID_PASS)
    return;

  if (p->doubled == 2)
    strcpy (doubled_buf, "-XX");
  else if (p->doubled)
    strcpy (doubled_buf, "-X");
  else
    doubled_buf[0] = '\0';

  sprintf (buf, "%s%s", bid_names[p->contract], doubled_buf);
  Display_Bid_on_Table (p->declarer, buf);
}

static char *suit_letters[4] = {"C", "D", "H", "S"};

void Display_Player_Hand (p, no_plays)
     int p; int no_plays;
/* Displays the hand of player p, where no_plays is the number of plays
   from the play record which should be examined to determine the current
   holdings of the player p.  That is, no_plays == 0 to display the cards
   which the player picked up, and no_plays == Local_play->no_plays to
   display the current holdings of the player.  Assumes that the area of
   the screen where the hand will be displayed is 'clean'.
*/
{
  int i;
  hand h;
  xy_pair *hand_location;
    /* The locations on the screen where the hand will be displayed. */

  hand_location = base_hand_location[screen_location[p]];

  for (i = 0; i < 4; i++)
    print (hand_location[i].y, hand_location[i].x, "           ");
  
  Generate_holdings (Local_board, Local_play->play_list, no_plays, p, h);

  for (i = 0; i < 4; i++) {
    print (hand_location[i].y, hand_location[i].x, suit_letters[i]);
    Display_suit
      (hand_location[i].y, hand_location[i].x+2, h + 13 * i);
  }

  revealed_hands[p] = 1;
}

void Display_Partial_Hand (p, no_plays)
     int p; int no_plays;
{
  Clear_Player_Area (p);
  Display_Player_Hand (p, no_plays);
  Display_Player_Name (p);
}

void Display_Hand (p)
     int p;
{
  Display_Partial_Hand (p, Local_play->no_plays);
}

void Refresh_Player (p)
     int p;
/* Redraws the area of the screen for the player p. */
{
  char buf[40];

  if (IS_OBSERVER(p))
    return;

  Clear_Player_Area (p);
  strcpy (buf, Table_display[p]);
  Display_on_Table (p, buf);
  Display_Player_Name (p);

  if (display_mode == BIDDING_DISPLAY) {
    Display_Player_Bids (p);
    if (revealed_hands[p])
      Display_Player_Hand (p, 0);
    if ((p == Local_play->declarer) &&
	(Local_table->game_mode != BIDDING_MODE))
      Display_Final_Contract (Local_board, Local_play);
  } else if (Local_table->game_mode == PLAYING_MODE) { 
    if (VUGRAPH(Local_table))
      Display_Player_Bids (p);
    if (revealed_hands[p])
      Display_Player_Hand (p, Local_play->no_plays);
  } else {
    Display_Player_Hand (p, 0);
    Display_Player_Bids (p);
  }
}

static void Redraw_Bidding_Display ()
{
  Display_Player (Local_play->next_player);
  Refresh_Time_Display ();
  print (PLAY_ROW, PLAY_COL, "BID ");
}

void Redraw_Playing_Display ()
{
  int i;

  Display_Contract (); 
  for (i = 0; i < 4; i++)
    Display_on_Table (i, Table_play[i]);

  Display_Player (Local_play->next_player);
  Refresh_Time_Display ();
  print (PLAY_ROW, PLAY_COL, "PLAY");
}

void Refresh_Playing_Area ()
/* Redraws the bidding/playing area. */
{
  int i;

  if ((display_mode != BIDDING_DISPLAY) && 
      (display_mode != PLAYING_DISPLAY))
    return;

  clear_screen_area (1, 1, 16, 79);
  Display_Playing_Board ();
  Recompute_Display_Positions ();
  Display_Player_Position ();
  Redraw_Scoring_Panel ();

  for (i = 0; i < 4; i++)
    Refresh_Player (i);

  if (display_mode == BIDDING_DISPLAY)
    Redraw_Bidding_Display ();
  else if (display_mode == PLAYING_DISPLAY)
    Redraw_Playing_Display ();
}


static void Redraw_Scoring_Display ()
{
  Display_Page_of_Scores (Current_page_of_scores);
}

void Display_Player_Position ()
{
  char buf[80], *position_name;
  
  if (IS_PLAYER(local_player))
    position_name = seat_names[local_player];
  else if (spectator_mode)
    position_name = "SPECTATOR";
  else
    position_name = "OBSERVER";
  
  sprintf (buf, "OKBRIDGE %s%s  %s     ", 
	   major_revision_level, minor_revision_level, position_name);
  
  print (1, 1, buf);

  if (display_mode != TALK_DISPLAY)
    return;

  sprintf (buf, "%20s", " ");
  print (1, 79 - 20, buf);

  if (local_player_id == -1)
    sprintf (buf, "NOT LOGGED IN.");
  else
    sprintf (buf, "LOGGED IN AS %s", local_player_name);
  print (1, 79 - strlen(buf), buf);
} 

void Refresh_Status_Display ();
void Begin_full_screen_talk_mode ();
 
void Reset_Display ()
/* Redraws the main features of the screen.  Used in the process
 * of doing a 'refresh'.
 */
{
  if (display_mode == MANUAL_DISPLAY)
    return;

  clear_screen ();

  if (display_mode == HELP_DISPLAY) {
    Suspend_Comment_Display ();
    status_loc = terminal_lines;
    display_topics ("LIST OF HELP TOPICS");
    print (terminal_lines-1, 1, "HELP");
    Refresh_Status_Display ();
    return;
  }

  Display_Player_Position ();
  Continue_Comment_Display ();
  
  switch (display_mode) {
  case TALK_DISPLAY:
    Begin_full_screen_talk_mode ();
    break;
  case BIDDING_DISPLAY:
  case PLAYING_DISPLAY:
    Refresh_Playing_Area ();
    Redraw_Conversation_Area ();
    break;
  case SCORING_DISPLAY:
    Redraw_Scoring_Display ();
    Redraw_Conversation_Area ();
    break;
  }

  Refresh_Player_Comments ();
  Refresh_Status_Display ();
}

void Initialize_Display ()
/* Should be called once when the program starts up. */
{ 
  int i;

  Initialize_Player_Comments ();
  status_buf[0][0] = '\0';
  display_mode = MANUAL_DISPLAY;

  for (i = 0; i < 4; i++) {
    Table_display[i][0] = '\0';
    Table_play[i][0] = '\0';
    Table_bid[i][0] = '\0';
  }

  status_loc = terminal_lines;
  Reset_Display ();
  Recompute_Display_Positions ();
}

void Reinitialize_Display ()
{
  Reinitialize_Player_Comments ();
  Reset_Display ();

  Recompute_Display_Positions ();
}

void Refresh_Display ()
/* Resets the terminal display and redraws everything. */
{
  Reset_Display ();
}

void Refresh_Player_Names ()
/* Redraws the player names.  Useful in case one of the players has changed
   position. */
{
  Refresh_Playing_Area ();
}

void Set_Display_Mode (mode)
     int mode;
{
  int old_mode = display_mode;

  exit_requested = 0;

  if (display_mode == mode)
    return;

  display_mode = mode;
  if ((mode == BIDDING_DISPLAY) && (old_mode == PLAYING_DISPLAY)) {
    Refresh_Playing_Area ();
    return;
  } 

  if ((mode == PLAYING_DISPLAY) && (old_mode == BIDDING_DISPLAY)) {
    Refresh_Playing_Area ();
    return;
  }

  if ((mode == HELP_DISPLAY) || (mode == MANUAL_DISPLAY)) {
    clear_screen ();
    Suspend_Comment_Display ();
    status_loc = terminal_lines;
    return;
  }

  if (display_mode == TALK_DISPLAY)
    Select_Talk_Mode ();
  else
    Select_Conversation_Mode ();

  Refresh_Display ();
}
 
void Display_Tricks_Taken ()
{
  sprintf (line_buf,"%5d   %5d",
	   Local_play->tricks[SIDE_NS], Local_play->tricks[SIDE_EW]);
  print (YMIN(scores)+1,XMIN(scores)+8, line_buf);
}

void Format_IMP_string (score, buf)
     Score *score; char *buf;
{
  float truncated_avg;
  int tenths_of_imps = 0;

  if (score->duplicate.competitive.imp.imp_tables > 0) {
    truncated_avg = Average_Imps (score);
    tenths_of_imps = (int) (10.0 * truncated_avg);
  } else 
    truncated_avg = 0.0;

  if ((tenths_of_imps % 10) == 0)
    sprintf (buf, "%5.0f", truncated_avg);
  else
    sprintf (buf, "%5.1f", truncated_avg);
}

void Format_MP_string (score, buf)
     Score *score; char *buf;
{
  float percent;

  percent = Percent_Matchpoints (score);
  
  if (percent == 1.0)
    sprintf (buf, "%5.0f", 100.0);
  else
    sprintf (buf, "%5.2f", percent * 100.0);
}

void Display_Above_Line_Points ()
{
  char ew_buf[10], ns_buf [10], line_buf[20];
  Score *ns_score, *ew_score;

  ns_score = &Local_table->score[SIDE_NS];
  ew_score = &Local_table->score[SIDE_EW];

  switch (Local_board->scoring_mode) {
  case IMP_SCORING:
    if (IS_PLAYER(local_player)) {
      if (side_of(local_player) == SIDE_NS)
	ns_score = &local_player_imp_total;
      else
	ew_score = &local_player_imp_total;
    }
    Format_IMP_string (ns_score, ns_buf);
    Format_IMP_string (ew_score, ew_buf);
    break;

  case MP_SCORING:
    if (IS_PLAYER(local_player)) {
      if (side_of(local_player) == SIDE_NS)
	ns_score = &local_player_mp_total;
      else
	ew_score = &local_player_mp_total;
    }
    Format_MP_string (ns_score, ns_buf);
    Format_MP_string (ew_score, ew_buf);
    break;

  case DUPLICATE_SCORING:
    if (IS_PLAYER(local_player)) {
      if (side_of(local_player) == SIDE_NS)
	ns_score = &local_player_total_score;
      else
	ew_score = &local_player_total_score;
    }
    sprintf (ns_buf, "%d", ns_score->duplicate.total);
    sprintf (ew_buf, "%d", ew_score->duplicate.total);
    break;

  case RUBBER_SCORING:
/*  The following code appears to be wrong!  But why is it here?
    I'm taking it out until I figure out what was going on ... 7/11/93. mtc
    See also the corresponding section in Display_Below_Line_Points.

    if (IS_PLAYER(local_player)) {
      if (side_of(local_player) == SIDE_NS)
	ns_score = &local_player_total_score;
      else
	ew_score = &local_player_total_score;
    }
*/
    sprintf (ns_buf, "%d", ns_score->rubber.above_line);
    sprintf (ew_buf, "%d", ew_score->rubber.above_line);
    break;
  }

  sprintf (line_buf, "%5s   %5s", ns_buf, ew_buf);
  print (YMIN(scores)+3,XMIN(scores)+8, line_buf);
}
 
void Display_Below_Line_Points ()
{
  char ew_buf[10], ns_buf [10], line_buf[20];
  Score *ns_score, *ew_score;

  if (Local_play == NULL) {
    ns_score = &Local_table->score[SIDE_NS];
    ew_score = &Local_table->score[SIDE_EW];
  } else if ((Prev_board == NULL) || 
      (Prev_board->scoring_mode != Local_board->scoring_mode)) {
    ns_score = &Local_play->score[SIDE_NS];
    ew_score = &Local_play->score[SIDE_EW];
  } else {
    ns_score = &Prev_play->score[SIDE_NS];
    ew_score = &Prev_play->score[SIDE_EW];
  }

  if (IS_PLAYER(local_player)) {
    if (side_of(local_player) == SIDE_NS)
      ns_score = &local_player_score;
    else
      ew_score = &local_player_score;
  }

  switch (Local_board->scoring_mode) {
  case IMP_SCORING:
    Format_IMP_string (ns_score, ns_buf);
    Format_IMP_string (ew_score, ew_buf);
    break;

  case MP_SCORING:
    Format_MP_string (ns_score, ns_buf);
    Format_MP_string (ew_score, ew_buf);
    break;

  case DUPLICATE_SCORING:
    sprintf (ns_buf, "%d", ns_score->duplicate.total);
    sprintf (ew_buf, "%d", ew_score->duplicate.total);
    break;

  case RUBBER_SCORING:
    ns_score = &Local_table->score[SIDE_NS];
    ew_score = &Local_table->score[SIDE_EW];
/*  The following code appears to be wrong!  But why is it here?
    I'm taking it out until I figure out what was going on ... 7/11/93. mtc
    See also the corresponding section in Display_Below_Line_Points.

    if (IS_PLAYER(local_player)) {
      if (side_of(local_player) == SIDE_NS)
	ns_score = &local_player_total_score;
      else
	ew_score = &local_player_total_score;
    }
*/

    sprintf (ns_buf, "%d", ns_score->rubber.below_line);
    sprintf (ew_buf, "%d", ew_score->rubber.below_line);
  }

  sprintf (line_buf, "%5s   %5s", ns_buf, ew_buf);
  print (YMIN(scores)+4,XMIN(scores)+8, line_buf);
}
 
void Display_Vulnerabilities   ()
{
  char *nsv, *ewv;
 
  if (Local_board->vulnerable[SIDE_NS]) 
    nsv = "  YES";
  else
    nsv = "   NO";

  if (Local_board->vulnerable[SIDE_EW]) 
    ewv = "  YES";
  else
    ewv = "   NO";

  sprintf (line_buf, "%s   %s",nsv,ewv);
  print (YMIN(scores)+2, XMIN(scores)+8, line_buf);
}
 
void Refresh_Time_Display ()
{
  int tn = strlen(total_time_buffer);
  int ln = strlen(local_time_buffer);

  if ((display_mode != PLAYING_DISPLAY) && (display_mode != BIDDING_DISPLAY))
    return;

/* efg - shouldn't this be changed so that when it is passed "","" */
/* the current display will be cleared? */
  print (PLAY_ROW,   80 - (tn? tn: 6), total_time_buffer);
  print (PLAY_ROW+1, 80 - (ln? ln: 6), local_time_buffer);
}

void Display_Total_Time (total_buf, local_buf)
     char *total_buf, *local_buf;
{
  sprintf (total_time_buffer, "%s", total_buf);
  sprintf (local_time_buffer, "%s", local_buf);
  Refresh_Time_Display ();
}

static void Display_suit (y, x, cards)
     int y, x; suit_type cards;
/* Displays the cards in a given suit held by a player.  As input,
 * cards[] is an array of 13 elements, where cards[i] is TRUE if the
 * player holds the given card.  Displays the cards as a string on
 * the terminal, beginning at coordinates <y,x>.
 */
{
  int i;
 
  for (i = 12; i >= 0; i--)
    if (cards[i])
      print (y, x++, rank_names[i]);
}
 
void Display_Contract ()
{
  char double_buf[40], contract_buf[60];
  int trump_suit = trumpsuit_of (Local_play->contract);
  int contract   = level_of (Local_play->contract);
 
  if (Local_play->contract == BID_PASS) {
    print (YMIN(title)+3, XMIN(title), "PASSED HAND     ");
    return;
  }

  clear_screen_area (YMIN(title)+4, XMIN(title), 1, 20);
  if (Local_play->doubled == 2)
    sprintf (double_buf,"-XX");
  else if (Local_play->doubled == 1)
    sprintf (double_buf, "-X");
  else
    double_buf[0] = '\0';
  sprintf (contract_buf, "%1d%s%s BY %s",contract,suit_names[trump_suit],
	   double_buf, PLAYER_NAME(Local_table, Local_play->declarer));
 
  print (YMIN(title)+4, XMIN(title), contract_buf);
}


void Clear_Playing_Board ()
{
  clear_screen_area (YMIN(playing_board)+2, XMIN(playing_board)+2,
		     HEIGHT(playing_board)-4, WIDTH(playing_board)-4);

  Clear_Plays_from_Table ();
}
 
void Display_Playing_Board ()
{
  int i;
  
  for (i = 0; i < WIDTH(playing_board)-1; i++) line_buf[i] = '-';
  line_buf[WIDTH(playing_board)-2] = '\0';
  print (YMIN(playing_board)+1,XMIN(playing_board)+1,line_buf);
  print (YMAX(playing_board)-1,XMIN(playing_board)+1,line_buf);
  
  for (i = YMIN(playing_board)+2; i < YMAX(playing_board)-1; i++) {
    print (i, XMIN(playing_board)+1, "|");
    print (i, XMAX(playing_board)-1, "|");
  }
}
 
void Display_Player (player)
     int player;
{
  int i;

  for (i = 0; i < 4; i++)
    print (name_location[i].y, name_location[i].x - 1, " ");

  if (IS_PLAYER(player))
    print (name_location[player].y, name_location[player].x - 1, "*");
}
 


void Clear_Hand (p)
     int p;
{
  Clear_Player_Area (p);
  revealed_hands[p] = 0;
  Display_Player_Name (p);
}

static char *seat_letters [4] = {"N", "E", "S", "W"};
static char *double_names [3] = {"", "-X", "-XX"};

static void Display_Score_Record (b, p, line)
     Board *b;
     Play_record *p;
     int line;
{
  char contract_buf[10];
  char buf[100];
  char mark;

  if (p->contract == BID_PASS)
    sprintf (contract_buf, "PASSED");
  else
    sprintf (contract_buf, "%s%s", bid_names[p->contract], 
	     double_names[p->doubled]);

  mark = ' ';
  if (p == Local_play || p == Prev_play) {
    if (IS_PLAYER(local_player) && (side_of(local_player) == SIDE_NS))
      mark = '>';
    else if (!IS_PLAYER(local_player))
      mark = '>';
  } else if (!strcasecmp(p->player_names[PLAYER_NORTH], local_player_name) ||
      !strcasecmp(p->player_names[PLAYER_SOUTH], local_player_name))
    mark = '*';

  sprintf (buf,
	   "%c %3d N %-8s S %-8s %-8s %-2s %-4s %+6d %6d %6.1f %6.2f",
	   mark, ++Next_score_record,
	   p->player_names[PLAYER_NORTH], p->player_names[PLAYER_SOUTH],
	   contract_buf, seat_letters[p->declarer], 
	   (p->no_plays == 0)? " ": card_names[p->play_list[0]],
	   p->result,
	   p->imatch_points[SIDE_NS].duplicate.total,
	   Average_Imps(&p->imatch_points[SIDE_NS]),
	   100.0 * Percent_Matchpoints(&p->match_points[SIDE_NS]));

  print (line, 1, buf);

  mark = ' ';
  if (p == Local_play || p == Prev_play) {
    if (IS_PLAYER(local_player) && (side_of(local_player) == SIDE_EW))
      mark = '>';
  } else if (!strcasecmp(p->player_names[PLAYER_EAST], local_player_name) ||
	   !strcasecmp(p->player_names[PLAYER_WEST], local_player_name))
    mark = '*';

  sprintf (buf,
	   "%c     E %-8s W %-8s %-8s %2s %4s %6s %6d %6.1f %6.2f", mark,
	   p->player_names[PLAYER_EAST], p->player_names[PLAYER_WEST],
	   " ", " ", " ", " ", 
	   p->imatch_points[SIDE_EW].duplicate.total,
	   Average_Imps(&p->imatch_points[SIDE_EW]),
	   100.0 * Percent_Matchpoints(&p->match_points[SIDE_EW]));

  print (line+1, 1, buf);
}

static void Display_Score_Summary (b, pl, line)
     Board *b;
     Play_record **pl;
     int line;
{
  char contract_buf[10];
  char tables_buf[20];
  char buf[100];
  char mark;
  int side,score;
  int ntables = 0;
  int first_table, last_table;
  Play_record *p = *pl;
  Play_record *pn = *pl;

  mark = ' ';
    
  if (p->contract == BID_PASS)
    sprintf (contract_buf, "PASSED");
  else
    sprintf (contract_buf, "%s%s", bid_names[p->contract], 
	     double_names[p->doubled]);

  while ((pn != NULL) && (p->contract == pn->contract) &&
	   (p->result == pn->result) && (p->doubled == pn->doubled)) {
    ntables++;
    Next_score_record++;
    if (pn == Local_play || pn == Prev_play)
      mark = '>';
    else if (mark == ' ') {
      if (!strcasecmp(pn->player_names[PLAYER_NORTH], local_player_name)) mark ='N';
      if (!strcasecmp(pn->player_names[PLAYER_SOUTH], local_player_name)) mark ='S';
      if (!strcasecmp(pn->player_names[PLAYER_EAST], local_player_name)) mark ='E';
      if (!strcasecmp(pn->player_names[PLAYER_WEST], local_player_name)) mark ='W';
    }
    *pl = pn;
    pn = pn->next;
    if ((pn != NULL) && !pn->hand_completed)
      pn = pn->next;
  }

  first_table = Next_score_record - ntables + 1;
  last_table = Next_score_record;
  if (first_table == last_table)
    sprintf (tables_buf, "(%d)", first_table);
  else
    sprintf (tables_buf, "(%d-%d)", first_table, last_table);

  sprintf (buf,"   %c   %6d  %-9s   %-8s  %-2s  %+6d ",
	   mark, ntables, tables_buf,
	   contract_buf, seat_letters[p->declarer], 
	   p->result);

  side = (IS_PLAYER(local_player) && (side_of(local_player) == SIDE_EW)) ?
         SIDE_EW : SIDE_NS;
  score = (p->imatch_points[side].duplicate.total > 0) ?
	  p->imatch_points[side].duplicate.total :
	  -p->imatch_points[1-side].duplicate.total;
  if (b->scoring_mode == MP_SCORING)
    sprintf (buf + strlen(buf),"%6d   %6.2f",score,
      	     100.0 * Percent_Matchpoints(&p->match_points[side]));
  else
    sprintf (buf + strlen(buf),"%6d    %6.1f",score,
	     Average_Imps(&p->imatch_points[side]));

  print (line, 1, buf);
}

void Display_Page_of_Scores (p)
     Play_record *p;
/* Displays a page of scores, starting with the play record p.  Sets the
   variable Current_page_of_scores equal to p, and the variable 
   Next_page_of_scores equal to the first undisplayed record.
*/
{
  char buf[100];
  char *vul_string;
  int line = 5;
  Board *b = Board_for_display;

  clear_screen_area (1, 1, 16, 80);

  if (p == Next_page_of_scores)
    Current_score_record = Next_score_record;
  else
    Next_score_record = Current_score_record;

  Current_page_of_scores = Next_page_of_scores = p;

  if (b->vulnerable[SIDE_NS] && b->vulnerable[SIDE_EW])
    vul_string = "BOTH";
  else if (b->vulnerable[SIDE_NS])
    vul_string = "N-S";
  else if (b->vulnerable[SIDE_EW])
    vul_string = "E-W";
  else
    vul_string = "NONE";

  sprintf (buf, "%s BOARD %d", b->source, b->serial_no);
  print (1, 78 - strlen(buf), buf);

  switch (b->scoring_mode) {
  case RUBBER_SCORING:
    print (1, 1, "RUBBER BRIDGE");
    break;
  case DUPLICATE_SCORING:
    print (1, 1, "DUPLICATE BRIDGE");
    break;
  case MP_SCORING:
    print (1, 1, "MATCH POINTS");
    break;
  case IMP_SCORING:
    print (1, 1, "INTL MATCH POINTS");
    break;
  }

  sprintf (buf, "DEALER %s, VUL %s", seat_names[b->dealer], vul_string);
  print (2, 1, buf);

  if (!result_mode)
    sprintf (buf, "%5s %21s %-8s %-2s %4s %6s %6s %6s %6s", " ", " ",
	   "CONTRACT", "BY", "LEAD", "RESULT", "SCORE", "IMPS", "MPS");
  else {
    sprintf (buf, "%-11s %-13s  %-8s  %-2s  %6s  %5s  ", " ",
	   "TABLES","CONTRACT", "BY", "RESULT", "SCORE");
    if (IS_PLAYER(local_player) && (side_of(local_player) == SIDE_EW))
      strcat(buf,"E-W ");
    else
      strcat(buf,"N-S ");
    if (b->scoring_mode == MP_SCORING)
      strcat(buf,"MPS ");
    else
      strcat(buf,"IMPS ");
  }
  print (4, 1, buf);

  while ((line < ((result_mode) ? 16 : 15)) && (Next_page_of_scores != NULL)) {
    if (!result_mode) {
      Display_Score_Record (b, Next_page_of_scores, line);
      line += 3;
    } else {
      Display_Score_Summary (b, &Next_page_of_scores, line);
      line += 1;
    }
    Next_page_of_scores = Next_page_of_scores->next;
    if ((Next_page_of_scores != NULL) && !Next_page_of_scores->hand_completed)
      Next_page_of_scores = Next_page_of_scores->next;
  }

}

void Display_First_Page_of_Scores (b)
     Board *b;
/* Initiales the scoring display and displays the first page of results from
 * board b. 
 */
{
  Play_record *Current_page_of_scores_save;

  result_board = b;

  if ((b->play_records == NULL) || (b->play_records->next == NULL)) {
    Next_page_of_scores = Current_page_of_scores = NULL;
    return;
  }

  Next_page_of_scores = b->play_records;

  if (!Next_page_of_scores->hand_completed)
    Next_page_of_scores = Next_page_of_scores->next;
  if (!Next_page_of_scores) return;

  Current_page_of_scores_save = Next_page_of_scores;
  Current_page_of_scores = 0;
  Board_for_display = b;
  Set_Display_Mode (SCORING_DISPLAY);
  Set_Input_Mode (TALK_INPUT);
  Current_page_of_scores = Next_page_of_scores = Current_page_of_scores_save;

  Current_score_record = Next_score_record = 0;
  Display_Page_of_Scores (Current_page_of_scores);
}

int More_Scores_to_Display ()
/* int More_Scores_to_Display (void); */
/* Returns true if not all of the results have been displayed from the
 * board which is currently being displayed. 
 */
{
  return (Next_page_of_scores != NULL);
}

void Init_Scores_to_Display ()
/* int Init_Scores_to_Display (void); */
/* Initializes scoring page listing.
 */
{
  Next_page_of_scores = NULL;
}

void Display_More_Scores ()
/* void Display_More_Scores (void); */
/* Displays the next page of scores for the board b. */
{
  Display_Page_of_Scores (Next_page_of_scores);
}

static void Save_Status_Message (message)
     char *message;
/* Saves the given message into the status display buffer at the
   current level of locking. */
{
  int i;

  for (i = 0; (i < WIDTH(status)) && (message[i] != '\0'); i++)
    status_buf[status_display_level][i] = message[i];
  status_buf[status_display_level][i] = '\0';
}

void Status (message)
     char *message;
{
if (!comments_suspended) {
  if (status_display_level > 0) {
    Moderator_Comment (message);
    return;
  }

  Clear_Status ();
  Save_Status_Message (message);
  Refresh_Status_Display ();
}
}
 
void Clear_Status ()
{
  if (status_display_level > 0)
    return;

  clear_screen_area (status_loc, XMIN(status), 1, WIDTH(status));
  status_buf [0][0] = '\0';
}

void Lock_Status (message)
     char *message;
/* Locks the status display.  This prevents the current message from being
   erased from the display.  If a new request to Status is made, that
   message is displayed as a moderator comment instead.
*/
{
  status_display_level += 1;
  Save_Status_Message (message);
  Refresh_Status_Display ();
}

void Unlock_Status ()
/* Unlocks the status display. */
{
  if (status_display_level > 0)
    status_display_level -= 1;
  Refresh_Status_Display ();
}

void Reset_Status ()
/* Resets the status display. */
{
  status_display_level = 0;
  Clear_Status ();
}

void Refresh_Status_Display ()
{
    clear_screen_area (status_loc, XMIN(status), 1, WIDTH(status));
    print (status_loc, XMIN(status), status_buf[status_display_level]);
    set_cursor 
      (status_loc, XMIN(status) + strlen(status_buf[status_display_level]) + 1);
}

/* The bottom part of the screen is used for the exchange of comments
   between the players.  The following procedures are used for managing
   this part of the display. */
 
 
static void blank_out_comment (c)
     comment_line c;
{
  int i;
 
  for (i = 0; i < COMMENT_LENGTH-1; i++) c[i] = ' ';
  c[COMMENT_LENGTH-1] = '\0';
}
 
static void copy_string_to_comment (c, s)
     comment_line c; char *s;
{
  int i;
 
  blank_out_comment(c);
  i = 0;
  while ((s[i] != '\0') && (i < COMMENT_LENGTH-1)) {
    c[i] = s[i]; 
    i++; 
  }
 
}
 
static void scroll_player_comments ()
{
  int i;

  for (i = 0; i < playing_screen_talk_size-1; i++)
    bcopy (playing_screen_comments[i+1], playing_screen_comments[i], 
	   sizeof(comment_line));
  blank_out_comment (playing_screen_comments[playing_screen_talk_size-1]);
}

void Select_Conversation_Mode ()
{
  talk_start = playing_screen_talk_start;
  talk_end   = playing_screen_talk_end;
  talk_size  = playing_screen_talk_size;
  player_comments = playing_screen_comments;
}

void Select_Talk_Mode ()
{
  talk_start = full_screen_talk_start;
  talk_end   = full_screen_talk_end;
  talk_size  = full_screen_talk_size;
  player_comments = full_screen_comments;
}

void Refresh_Player_Comments ()
{
  int i;

  if (comments_suspended)
    return;

  for (i = 0; i < talk_size; i++)
    print (talk_start+i, 2, player_comments[i]);
}

void Clear_Comment_Display ()
{
  int i;

  for (i = 0; i < full_screen_talk_size; i++)
    blank_out_comment (full_screen_comments + i);
  for (i = 0; i < playing_screen_talk_size; i++)
    blank_out_comment (playing_screen_comments + i);

  full_screen_cursor = 0;
  Refresh_Player_Comments ();
}
 
void Initialize_Player_Comments ()
{
  int i, m, n;
  
  full_screen_talk_end = terminal_lines - 3;
  playing_screen_talk_end = terminal_lines;
  
  n = full_screen_talk_end - full_screen_talk_start + 1;
  m = playing_screen_talk_end - playing_screen_talk_start + 1;
  if (n <= 0)
    n = 1;
  else if (m <= 0)
    m = 1;
  
  full_screen_comments = (comment_line *) 
    malloc (n * sizeof(comment_line));
  
  playing_screen_comments = (comment_line *)
    malloc (m * sizeof(comment_line));
  
  for (i = 0; i < n; i++)
    blank_out_comment (full_screen_comments[i]);
  for (i = 0; i < m; i++)
    blank_out_comment (playing_screen_comments[i]);
  
  full_screen_cursor = 0;
  full_screen_talk_size = n;
  playing_screen_talk_size = m;

  if (display_mode == TALK_DISPLAY)
    Select_Talk_Mode ();
  else
    Select_Conversation_Mode ();
}

void Reinitialize_Player_Comments ()
{
  Initialize_Player_Comments ();
}

static char *mystrstr (fullstring, substring)
     char *fullstring; char *substring;
{
  int k = strlen (substring);
  int n = strlen (fullstring);
  int i;

  for (i = 0; i < n-k+1; i++)
    if (!strncasecmp(substring, fullstring + i, k))
      return (fullstring + i);

  return (NULL);
}

void Display_Player_Comment (comment_level, player_name, comment)
     int comment_level; char *player_name, *comment;
{
  char message_buf [200];
  char level_id;
  ignore *ign;

  if (initializing) return;

  switch (comment_level) {
  case COMMENT_PRIVATE: level_id = '-'; break;
  case COMMENT_FORMAL:  level_id = '='; break;
  case COMMENT_PUBLIC:  level_id = ':'; break;
  case COMMENT_SPEC:    level_id = '%'; break;
  default:              level_id = '*';
  }
  
  sprintf (message_buf, "%s%c %s", player_name, level_id, comment);
  
  for (ign = head_ignore; ign != NULL; ign = ign -> next)
    if (mystrstr(message_buf,ign->str) != NULL)
      return;

  copy_string_to_comment (full_screen_comments[full_screen_cursor], 
			  message_buf);
  if (full_screen_cursor < full_screen_talk_size - 1)
    full_screen_cursor++;
  else
    full_screen_cursor = 0;
  blank_out_comment (full_screen_comments[full_screen_cursor]);

  scroll_player_comments ();
  copy_string_to_comment 
    (playing_screen_comments[playing_screen_talk_size-1], message_buf);

  if (comments_suspended)
    return;

  Refresh_Player_Comments ();
}

void Moderator_Comment (msg)
     char *msg;
{
  Display_Player_Comment (COMMENT_PUBLIC, "MODERATOR", msg);
}

void Network_Comment (msg)
     char *msg;
{
  Display_Player_Comment (COMMENT_PRIVATE, "NETWORK", msg);
}

void Suspend_Comment_Display ()
{
  comments_suspended = 1;
}

void Continue_Comment_Display ()
{
  comments_suspended = 0;
}

 
void Begin_full_screen_talk_mode ()
{
  int i;

  Select_Talk_Mode ();
  
  for (i = 0; i <= XMAX(converse); i++) line_buf[i] = '-';
  line_buf[XMAX(converse)+1] = '\0';
  print (full_screen_talk_start-1, 1, line_buf);
  line_buf[72] = '+';
  print (full_screen_talk_end+1, 1, line_buf);
  for (i = 1; i < XMAX(converse); i++) line_buf[i] = ' ';
  line_buf[0] = line_buf[XMAX(converse)] = '|';
  for (i = full_screen_talk_start; i <= full_screen_talk_end; i++)
    print (i, 1, line_buf);
  
  Refresh_Player_Comments ();
  
  /* setup the talk buffer: */
  
  print (terminal_lines-1, 1, "TALK");
  status_loc = terminal_lines;
}

void End_full_screen_talk_mode ()
{
  Select_Conversation_Mode ();
  status_loc = YMIN(status);
  Refresh_Display ();
}



