/* conversation.c -- processor for conversation level messages.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This module is responsible for processing messages at the conversation
 * level of the okbridge protocol.
 */

#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include "network.h"
#include "state.h"
#include "display.h"
#include "terminal.h"
#include "cs.h"
#include "input.h"
#include "conversation.h"

#ifdef GCC
extern gettimeofday (), strcasecmp ();
extern fprintf ();
#endif

extern int pause_mode;
extern int claim_in_progress;

extern char *strdup ();
extern void free ();

extern void Generate_reset ();
extern void Clear_Hand ();


char msg_buf[100];  /* A buffer used for formatting messages that will
		       be displayed or sent to other players. */

struct timeval ping_start;  /* time at which last ping command
			       was issued. */

#ifdef LOGFILE
extern FILE *net_log;
#endif

static void Respond_to_Echo (msg)
     Message msg;
/* Responds to an echo message. */
{
  int mils;	/* milliseconds elapsed between ping and echo */
  struct timeval ping_end;

  if (!strcmp(msg->p.data.echo.ping_source, local_player_name)) {
    gettimeofday (&ping_end, NULL);
    mils = (ping_end.tv_sec - ping_start.tv_sec)
      * 1000;
    mils += ping_end.tv_usec / 1000;
    mils -= ping_start.tv_usec / 1000;
    sprintf (msg_buf, "ECHO RECEIVED FROM (%c) %-10s IN %7.2f SECONDS",
	     *("NESWO" + msg->p.player_no), msg->p.player_name, 
	     ((float) mils) * 0.001);
    Moderator_Comment (msg_buf);
  }
}

void Handle_Conversation_Message (msg)
     Message msg;
/* Processes the message m.  If the message is a protocol message, then
   takes appropriate action based on the message.  Appends the message
   to the appropriate conversation queue if further action is warranted.
*/
{
  char message_buf [100];
  int i, s;
  int  pass_upwards;
    /* A boolean flag indicating that the message should be passed upwards
       to the game queue. */
  Table t = msg->source->table;
  int additional_tricks_claimed;

  pass_upwards = 1;
  switch (msg->p.command) {
  case CMD_ERROR  :
    sprintf (message_buf, "ERROR IN MESSAGE FROM %s (%s):",
	     msg->source->player_name, 
	     (msg->source->seat < 4)? seat_names[msg->source->seat]: 
	     "OBSERVER");
    Network_Comment (message_buf);
    Network_Comment (msg->p.data.error.message);
    Network_Comment (msg->p.command_text);
    pass_upwards = 0;
    break;

  case CMD_ACK:
    if (!quiet_mode) {
      sprintf (message_buf, "%s HAS JOINED THE GAME AS %s.",
	       msg->p.data.ack.player_name,
	       (msg->p.data.ack.position == PLAYER_OBS)? "AN OBSERVER":
	       seat_names[msg->p.data.ack.position]);
      Network_Comment (message_buf);
    }
    break;

  case CMD_CC:
    if (conventions[side_of(msg->p.player_no)] != NULL)
      free (conventions[side_of(msg->p.player_no)]);
    conventions[side_of(msg->p.player_no)] = strdup(msg->p.data.cc);
    Display_Player_Comment 
      (COMMENT_PUBLIC, side_of(msg->p.player_no)? "E-W CC": "N-S CC",
       msg->p.data.cc);
    break;

  case CMD_CLAIMREQ:
    claim_responses = 0;
    claim_accepted  = 1;
    if (IS_PLAYER(local_player)  && 
	(side_of(local_player) != side_of(Local_play->declarer))) {
      Set_Display_Mode (PLAYING_DISPLAY);
      Display_Hand (Local_play->declarer);
      Display_Hand (player_partner[local_player]);
      Display_Hand (player_partner[Local_play->declarer]);
      additional_tricks_claimed = msg->p.data.claimreq.no_tricks -
	Local_play->tricks[side_of(Local_play->declarer)];
      if (msg->p.data.claimreq.no_tricks >= level_of(Local_play->contract) + 6)
	sprintf 
	  (message_buf,
	   "DECLARER CLAIMS %d MORE TRICK%s, MAKING %d.  DO YOU ACCEPT? ",
	   additional_tricks_claimed,
	   (additional_tricks_claimed == 1)? "": "S",
	   msg->p.data.claimreq.no_tricks - 6);
      else
	sprintf
	  (message_buf,
	   "DECLARER CLAIMS %d MORE TRICK%s, FOR DOWN %d.  DO YOU ACCEPT? ",
	   additional_tricks_claimed,
	   (additional_tricks_claimed == 1)? "": "S",
	   level_of(Local_play->contract)+6 - msg->p.data.claimreq.no_tricks);
      Send_claimresp (Local_table, Ask(message_buf));
      Clear_Hand (Local_play->declarer);
      Clear_Hand (player_partner[local_player]);
    } else {
      sprintf (message_buf, "DECLARER IS CLAIMING %d TRICKS TOTAL...",
	       msg->p.data.claimreq.no_tricks);
      Moderator_Comment (message_buf);
    }
    break;

  case CMD_CLAIMRESP:
    claim_responses += 1;
    claim_accepted  = claim_accepted && msg->p.data.claimresp;
    if ((claim_responses == 2) && !claim_accepted)
      Status ("THE CLAIM OFFER HAS BEEN DECLINED.");
    break;

  case CMD_COMMENT:
    Moderator_Comment (msg->p.data.comment);
    break;

  case CMD_ECHO:
    Respond_to_Echo (msg);
    break;
    
  case CMD_HELLO  :
    if (strcmp(major_revision_level, msg->p.data.hello.version)) {
      sprintf (message_buf, "%s HAS ATTEMPTED TO CONNECT USING VERSION %s",
	       msg->p.data.hello.player_name, msg->p.data.hello.version);
      Network_Comment (message_buf);
    } else {
      sprintf (message_buf, "%s HAS JOINED THE GAME.", 
	       msg->p.data.hello.player_name);
      if (!VUGRAPH(Local_table) && !quiet_mode) {
	Network_Comment (message_buf);
	if (Local_table->game_mode == STARTUP_MODE)
	  ring_bell ();
      }
    }
    break;

  case CMD_MODE:
    if (msg->p.data.mode != t->playing_mode) {
      switch (msg->p.data.mode) {
      case CLUB_PLAYING_MODE:
	Moderator_Comment ("WE ARE PLAYING CLUB STYLE BRIDGE.");
	break;
      case PRACTICE_PLAYING_MODE:
	Moderator_Comment ("WE ARE PLAYING PRACTICE BRIDGE.");
	break;
      case FORMAL_PLAYING_MODE:
	Moderator_Comment ("WE ARE PLAYING FORMAL BRIDGE.");
	break;
      case VUGRAPH_PLAYING_MODE:
	Moderator_Comment ("WE ARE WATCHING A VUGRAPH EXHIBITION.");
	break;
      }
    }
    break;

  case CMD_NAME:
    if (strcasecmp(msg->p.player_name, msg->p.data.name.new_name)) {
      sprintf (message_buf, "%s IS NOW LOGGED IN AS %s.", msg->p.player_name,
	       msg->p.data.name.new_name);
      Network_Comment (message_buf);
      if (IS_PLAYER(msg->p.player_no)) {
	sprintf (PLAYER_NAME(t, msg->p.player_no), "%s",
		 msg->p.data.name.new_name);
	Refresh_Playing_Area ();
      }
    }
    break;

  case CMD_QUIT:
    if (!VUGRAPH(Local_table) && !quiet_mode) {
      sprintf (message_buf, "%s HAS QUIT.", msg->p.player_name);
      Network_Comment (message_buf);
    }
    if (IS_PLAYER(msg->p.player_no)) {
      if (claim_in_progress &&
	  (msg->p.player_no != player_partner[Local_play->declarer])) {
	claim_accepted = 0;
	claim_responses = 2;
	Status ("THE CLAIM HAS BEEN ABORTED.");
      }
      Refresh_Playing_Area ();
      Refresh_Input_Buffers ();
    }
    pass_upwards = 0;
    break;

  case CMD_PING:
    if (!msg->loopback)
      Send_echo (msg->source->table, msg->p.player_name);
    break;

  case CMD_RESET:
    Moderator_Comment ("RESET RECEIVED!");
    Generate_reset (RESET_FULL);
    break;

  case CMD_SKIP:
    sprintf (message_buf, "THE HAND HAS BEEN SKIPPED BY %s ...", 
	     msg->p.player_name);
    Moderator_Comment (message_buf);
    pause_mode = 0;
    break;

  case CMD_SEAT:
    if (strcasecmp(msg->p.data.seat.player_name, local_player_name)) {
      s = msg->p.data.seat.new_pos;
      sprintf (message_buf, "%s IS SITTING AS %s.", 
	       msg->p.data.seat.player_name,
	       IS_PLAYER(s)? seat_names[s]: "AN OBSERVER");
      Network_Comment (message_buf);
      if (IS_PLAYER(msg->p.data.seat.new_pos) ||
	  IS_PLAYER(msg->p.data.seat.old_pos)) {
	Refresh_Playing_Area ();
	Refresh_Input_Buffers ();
	if (claim_in_progress &&
	    (msg->p.data.seat.old_pos != 
	     player_partner[Local_play->declarer])) {
	  claim_accepted = 0;
	  claim_responses = 2;
	  Status ("THE CLAIM HAS BEEN ABORTED.");
	}
      }
    } else
      Display_Player_Position ();
    break;

  case CMD_SEATERR:
    Network_Comment ("THE SEAT WHICH YOU REQUESTED IS OCCUPIED.");
    if (msg->p.data.seaterr.free_seats[0] >= 4)
      Network_Comment ("THERE ARE NO FREE SEATS CURRENTLY.");
    else {
      sprintf (message_buf, "THE CURRENTLY FREE SEATS ARE:");
      for (i = 0; i < 3; i++)
	if (msg->p.data.seaterr.free_seats[i] < 4)
	  sprintf (message_buf + strlen(message_buf), " %s,",
		   seat_names[msg->p.data.seaterr.free_seats[i]]);
      message_buf[strlen(message_buf)-1] = '.';
      Network_Comment (message_buf);
    }
    break;

  case CMD_SEATPOS:
    Assign_seat (Local_table, Local_Player_Connection, msg->p.data.seatpos);
    local_player = msg->p.data.seatpos;
    if (Local_table->game_mode == BIDDING_MODE) {
      if (VUGRAPH(Local_table) && server_mode)
	Set_Input_Mode (BID_INPUT);
      else if (IS_OBSERVER(msg->p.data.seatpos))
	Set_Input_Mode (TALK_INPUT);
      else
	Set_Input_Mode (BID_INPUT);
      if (IS_PLAYER(local_player))
	Display_Hand (local_player);
      Refresh_Input_Buffers ();
    } else if (Local_table->game_mode == PLAYING_MODE) {
      if (VUGRAPH(Local_table) && server_mode)
	Set_Input_Mode (PLAY_INPUT);
      else if (PRACTICE(Local_table) && IS_PLAYER(msg->p.data.seatpos))
	Set_Input_Mode (PLAY_INPUT);
      else if (IS_OBSERVER(msg->p.data.seatpos) || 
	  (msg->p.data.seatpos == player_partner[Local_play->declarer]))
	Set_Input_Mode (TALK_INPUT);
      else
	Set_Input_Mode (PLAY_INPUT);
      Refresh_Input_Buffers ();
      if (IS_PLAYER(local_player))
	Display_Hand (local_player);
    }
    Display_Player_Position ();
    Refresh_Playing_Area ();
    sprintf (message_buf, "YOU ARE NOW SITTING AS %s.",
	     IS_PLAYER(local_player)? seat_names[local_player]: 
	     "AN OBSERVER");
    Network_Comment (message_buf);
    break;

  case CMD_SPEC:
    if (!VUGRAPH(Local_table) && !quiet_mode) {
      sprintf (message_buf, "%s HAS ENTERED SPECTATOR MODE.", 
	       msg->p.player_name);
      Moderator_Comment (message_buf);
    }
    msg->source->spectator = 1;
    break;

  case CMD_TABLE:
    if (t->table_no == msg->p.data.table) {
      sprintf (message_buf, "YOU HAVE JOINED TABLE %d.", t->table_no);
      Moderator_Comment (message_buf);
    } else {
      sprintf (message_buf, "ERROR IN ASSIGNMENT TO TABLE %d", 
	       msg->p.data.table);
      Moderator_Comment (message_buf);
    }
    break;

  case CMD_TALK:
    switch (msg->p.data.talk.recipients) {
    case TALK_RCPT_LHO:
      if (player_next[msg->p.player_no] == local_player)
	Display_Player_Comment
	  (COMMENT_PRIVATE, msg->p.player_name, msg->p.data.talk.message);
      break;
    case TALK_RCPT_RHO:
      if (player_prev[msg->p.player_no] == local_player)
	Display_Player_Comment
	  (COMMENT_PRIVATE, msg->p.player_name, msg->p.data.talk.message);
      break;
    case TALK_RCPT_OPPS:
      if ((player_partner[msg->p.player_no] != local_player) &&
	  IS_PLAYER(local_player))
	Display_Player_Comment
	  (COMMENT_FORMAL, msg->p.player_name, msg->p.data.talk.message);
      break;
    case TALK_RCPT_SPEC:
      if (spectator_mode)
	Display_Player_Comment
	  (COMMENT_SPEC, msg->p.player_name, msg->p.data.talk.message);
      break;
    case TALK_RCPT_ALL:
    default:
      Display_Player_Comment
	(COMMENT_PUBLIC, msg->p.player_name, msg->p.data.talk.message);
      break;
    }
    break;
    
  case CMD_WAKEUP:
    i = bell_is_on;
    bell_is_on = 1;
    if (!strcasecmp(msg->p.data.wakeup.recipient, "ALL")) {
      Display_Player_Comment (COMMENT_PUBLIC, msg->p.player_name, "WAKE UP!");
      ring_bell ();
    } else if (!strcasecmp(msg->p.data.wakeup.recipient, local_player_name)) {
      Display_Player_Comment (COMMENT_PRIVATE, msg->p.player_name, "WAKE UP!");
      ring_bell ();
    }
    bell_is_on = i;
    break;
    
  case CMD_WHORESP:
    sprintf (message_buf, "WHOIS %s", msg->p.data.whoresp.recipient);
    Display_Player_Comment (COMMENT_PUBLIC, message_buf,
			    msg->p.data.whoresp.message);
    break;
    
  default:
    break;
  }

  if (pass_upwards)
    enqueue_message(t->game_queue, msg);
  else
    deallocate_message (msg);
}

static int Handle_Protocol_Messages ()
/* Handles all messages which are on protocol queues.  If any messages
   were found, returns TRUE.
*/
{
  Table t;
  Message m;
  int message_found = 0;
  
  for (t = Table_List; t != NULL; t = t->next) {
    while (message_available(t->protocol_queue)) {
      m = dequeue_message (t->protocol_queue);
      message_found = 1;
      if (server_mode)
	Handle_Protocol_Message_for_Server (m);
      else
	Handle_Protocol_Message_for_Client (m);
    }
  }
  
  return (message_found);
}

static int Handle_Conversation_Messages ()
/* Handles all messages which are on conversation queues.  If any messages
   were found, returns TRUE.
*/
{
  Table t;
  Message m;
  int message_found = 0;

  for (t = Table_List; t != NULL; t = t->next) {
    while (message_available(t->conversation_queue)) {
      message_found = 1;
      m = dequeue_message (t->conversation_queue);
      Handle_Conversation_Message (m);
    }
  }
  
  return (message_found);
}

void Wait_for_event_at_conversation_level (e)
     event_signal e;
/* Receives incoming messages at all tables.  Handles keyboard characters
 * and messages which arrive on the protocol queues.  Each time a keyboard
 * character is received or a new message is processed, the event_signal
 * routine is called.  If it returns true, then the procedure exits.
 */
{
  do {
    Handle_Protocol_Messages ();
    if ((*e)()) return;
    restore_cursor ();
    Wait_for_network_event ();
    if ((*e)()) return;
    Accept_Keyboard_Characters ();
  } while (1);
    
}

void Wait_for_event_at_game_level (e)
     event_signal e;
/* Receives incoming messages at all tables.  Handles keyboard characters
 * and messages which arrive on the protocol and conversation queues.  Each
 * time a keyboard character is received or a new message is processed, the
 * event_signal e routine is called.  If it returns true, then the procedure
 * exits.  
 */
{
  int protocol_message_handled, 
      conversation_message_handled;
  
  do {
    do {
      protocol_message_handled = Handle_Protocol_Messages ();
      conversation_message_handled = Handle_Conversation_Messages ();
     } while (protocol_message_handled || conversation_message_handled);

    if ((*e)()) return;
    restore_cursor ();
    Wait_for_network_event ();
    if ((*e)()) return;
    Accept_Keyboard_Characters ();
  } while (1);
}

Table Wait_for_conversation_message ()
/* Receives incoming message at all tables.  Handles keyboard characters
 * and messages which arrive on the protocol queues.  When a message is
 * placed onto the conversation queue of a table, returns a pointer to
 * the table. */
{
  Table t;

  do {
    Handle_Protocol_Messages ();
    for (t = Table_List; t != NULL; t = t->next)
      if (message_available(t->conversation_queue))
	return (t);
    restore_cursor ();
    Wait_for_network_event ();
    Accept_Keyboard_Characters ();
  } while (1);
}

Table Wait_for_game_message ()
/* Receives incoming messages at all tables.  Handles keyboard characters
 * and messages which arrive on the protocol and game queues.  When a message
 * is placed onto the game queue of a table, returns a pointer to the table.
 */
{
  Table t;
  int protocol_message_handled, 
      conversation_message_handled;
  
  do {
    do {
      protocol_message_handled = Handle_Protocol_Messages ();
      conversation_message_handled = Handle_Conversation_Messages ();
     } while (protocol_message_handled || conversation_message_handled);

    for (t = Table_List; t != NULL; t = t->next)
      if (message_available(t->game_queue))
	return (t);

    restore_cursor ();
    Wait_for_network_event ();
    Accept_Keyboard_Characters ();
  } while (1);
}
