/* commands.c
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This module implements the commands which are available through the
 * input processor.  For instructions on adding or modifying commands
 * in OKbridge, see the documentation preceding the Input_Commands table
 * near the end of this file.
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <ctype.h>
#include <errno.h>

#include "types.h"
#include "parser.h"
#include "state.h"
#include "terminal.h"
#include "display.h"
#include "input.h"
#include "gps.h"
#include "help.h"
#include "conversation.h"
#include "log.h"
#include "gps_info.h"

/* extern fclose (), strcasecmp (), gettimeofday (); */

extern int errno;
extern char *sys_errlist[];
extern char *strdup();
extern char *malloc ();
extern void free ();

extern int  query_complete;
extern int  pause_mode;
extern int  save_defaults;
extern int  display_scoreboard_at_end;
extern ignore *head_ignore;
extern char *help_file_name;
extern int  never_prompt;


extern void Assign_seat ();
extern void Broadcast_seat_assignment ();
extern void Generate_reset ();
extern void Broadcast_Comment ();
extern void Quit_program ();
extern void Disconnect_Player ();
extern void Vacate_seat ();
extern int  String_is_alphameric ();
extern void GPS_Display_Scoreboard ();
extern void Display_timer ();
extern void Write_Initialization_File ();
extern void Request_Seat ();
extern Command_Descriptor *Search_for_Command ();


/* The following structures define the set of commands which are available
   to the local player.
 */


char *autoload_file = NULL;     /* The name of the file from which we
				   will initially try to load a sequence
				   of boards, if we are north in email mode. */
char *autosave_file = NULL;     /* The name of the file to which we will
				   automatically save the boards that we
				   have played. */

int claim_in_progress = 0;      /* TRUE if we are currently processing a
				   claim request.  This blocks us from making
				   another claim. */

int exit_requested = 0;         /* TRUE if the user has requested an exit
				   from an action currently in progress. */

int initializing = 0;	        /* TRUE if starting up, means that we should
					not write things to the screen */

int never_prompt_save;          /* The saved version of never_prompt. */

int new_score_mode = -1;	/* if non-negative, scoring a new scoring
					mode has been requested */

static char *logfile_name = NULL;
static char *recfile_name = NULL;

extern FILE *logfile;
extern FILE *logfile_email;

extern FILE *recfile;
extern FILE *recfile_email;

extern int  record_this_hand;
extern int  show_alerts;
extern int  show_results;

extern FILE *zhang_logfile;

extern int server_mode;

extern char *Get_CC_Value();                                          /* JAS */
extern int Define_CC();                                               /* JAS */
extern void Display_All_CCs();                                        /* JAS */

extern int No_connections ();

extern int timer_is_on;
extern struct timeval ping_start;
extern int sort_direction;
extern int result_mode;
extern Board *result_board;

static char msg_buf[100];         /* A buffer for error messages. */
static int Command_Error();

/* the Parse_..._input routines that essentially just set or
 . toggle a variable use this routine to set the variable.
 */
static void Toggle (var, val)
int *var, *val;
{
  if (val != NULL) {
    switch (*val) {
	case OFF: *var = 0; break;
	case ON:  *var = 1; break;
	case TOGGLE: *var = !(*var); break;
    }
  }
}

void Command_Initializing(flag)
int flag;
{
  initializing = flag;
  /* efg The routines ought not to send comments, ring bell, etc during */
  /* initialization.  It seems simplest to turn off comments and    */
  /* messages, but maybe each command should test for initializing ? */
  /* Perhaps there should be a disable_screen command.  This routine */
  /* is only called from the Read_Initialization_File routine. */
  if (flag) {
    Suspend_Comment_Display();
    Disable_Send_Message();
  }
  else {
    Continue_Comment_Display();
    Enable_Send_Message();
  }
}

static void Error (error_string)
     char *error_string;
/* Displays an error message in response to a command.  This routine
   should be used for any command which may be used both at
   initialization time and during normal operation.
*/
{
  if (initializing) {
    Field_Error (error_string);
  } else
    Status (error_string);
}

static void load_email_file (filename)
     char *filename;
/* Attempts to read a sequence of deals from the email duplicate file
 . with name filename.
 */
{
  FILE *f = fopen (filename, "r");
  int status;

  if (f == NULL) {
    sprintf (msg_buf, "COULD NOT OPEN FILE %s: %s", filename, 
	     sys_errlist[errno]);
    Status (msg_buf);
    return;
  }

  Clear_All_Boards ();
  status = Load_Email_Duplicate_File (f);
  fclose (f);
  if (status == 1) {
    sprintf (msg_buf, "%s IS NOT AN EMAIL DUPLICATE FILE.", filename);
    Status (msg_buf);
    replay_mode = 0;
  } else if (status) {
    sprintf (msg_buf, "ERROR READING %s.", filename);
    Status (msg_buf);
    replay_mode = 0;
  } else {
    Send_skip (Local_table);
    sprintf (msg_buf, "LOADED EMAIL BOARDS FROM %s.", filename);
    Moderator_Comment (msg_buf);
    email_filename = strdup (filename);
  }

}

static void save_email_file (filename)
     char *filename;
{
  FILE *f = fopen (filename, "w");

  if (f == NULL) {
    sprintf (msg_buf, "COULD NOT OPEN FILE %s: %s", filename, 
	     sys_errlist[errno]);
    Status (msg_buf);
    return;
  }

  Write_Email_Duplicate_File (f);
  fclose (f);

  sprintf (msg_buf, "SAVED EMAIL BOARDS TO %s.", filename);
  Moderator_Comment (msg_buf);
}

static void Parse_bids_input()
/* Generates a review of the bidding. */
{
  int mode = Local_table->game_mode;
  int current_input_mode = input_mode;
  int current_display_mode = display_mode;

  if (FORMAL(Local_table))
    Status ("THE BIDDING CANNOT BE REVIEW DURING FORMAL PLAY.");
  else if (display_mode == BIDDING_DISPLAY)
    return;
  else if ((mode == PLAYING_MODE) || (mode == SCORING_MODE)) {
    Set_Display_Mode (BIDDING_DISPLAY);
    Set_Input_Mode (TALK_INPUT);
    Pause ("PRESS <ESC> TO RETURN TO PLAY ...");
    Set_Display_Mode (current_display_mode);
    Set_Input_Mode (current_input_mode);
  } else 
    Status ("THE BIDDING CANNOT BE REVIEWED NOW.");
  
  Refresh_Input_Buffers ();
}

static void Parse_look_input (num)
     int *num;
{
  int current_display_mode = display_mode;
  int current_input_mode   = input_mode;
  int i, picture_length;
  Board *b = (result_board != Local_board) ? Prev_board : Local_board;
  struct Play_record_struct *p = b ? b->play_records : 0;

  if (p == NULL) {
    Status ("THERE ARE NO TABLES TO REVIEW.");
    return;
  }

  for (i=1;i<*num;i++) {
    p = p->next;
    if (p != NULL && (!Percent_Matchpoints(&p->match_points[SIDE_NS]) &&
        !Percent_Matchpoints(&p->match_points[SIDE_EW])))
      p = p->next;  /* Skip local result if not complete. */
    if (p == NULL) {
      Status ("THE SPECIFIED TABLE NUMBER WAS NOT FOUND.");
      return;
    }
  }

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  picture_length = Create_Picture_of_Hand (b,p);

  if (picture_length > 23)
    picture_length = 23;

  for (i = 0; i < picture_length; i++)
    print (i+1, 1, Play_picture[i]);
  Press_Return_to_Continue ("");
  
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
}

static void Parse_review_input (h)
     int *h;
{
  int current_display_mode = display_mode;
  int current_input_mode   = input_mode;
  int i, picture_length;
  int hand_completed;

  if (h) {
    if (*h == 0 && !spectator_mode) {
      Status ("ONLY SPECTATORS CAN REVIEW CURRENT BOARD.");
      return;
    } else if (Local_board == NULL) {
      Status ("THERE IS NO BOARD TO REVIEW.");
      return;
    } else if (!Local_play->bidding_completed) {
      Status ("PLAY HAS NOT YET BEGUN.");
      return;
    }
    if (*h == 1 && !spectator_mode
	&& IS_PLAYER(local_player) && !PRACTICE(Local_table)
	&& local_player != player_partner[Local_play->declarer]) {
      if (!Ask
 ("PLAYERS WILL BE NOTIFIED THAT YOU ARE REVIEWING PLAY.  CONTINUE TO DO SO? "))
        return;
      Send_talk (Local_table, TALK_RCPT_ALL, "I AM REVIEWING PLAY OF HAND!!!");
    }
  } else if (Prev_board == NULL) {
    Status ("THERE IS NO PREVIOUS BOARD TO REVIEW.");
    return;
  }

  Set_Display_Mode (MANUAL_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  if (h) {
    hand_completed = Local_play->hand_completed;
    Local_play->hand_completed = 1;
    picture_length = Create_Picture_of_Hand (Local_board, Local_play);
    Local_play->hand_completed = hand_completed;
  } else
    picture_length = Create_Picture_of_Hand (Prev_board, Prev_play);
  if (picture_length > 23)
    picture_length = 23;

  for (i = 0; i < picture_length; i++)
    print (i+1, 1, Play_picture[i]);
  Press_Return_to_Continue ("");
  
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
}

static int Partner_has_bid (t, b, p, player)
     Table t;
     Board *b;
     Play_record *p;
     int player;
/* Returns true if the partner of p has bid. */
{
  int partner = player_partner[player];
  int i, bidder;

  if (p->no_bids > 3)
    return (1);

  bidder = b->dealer;
  for (i = 0; i < p->no_bids; i++)
    if (bidder == partner)
      return (1);
    else
      bidder = player_next [bidder];

  return (0);
}

static void Parse_addcc_input (addition)
    char * addition;
/* ADDCC <conventions>
 . Adds the conventions to the current cc
 */
{
  if (addition && local_player < 4) {
    char *oldcc = conventions[side_of(local_player)];
    if (oldcc == NULL)
	Status ("YOU HAVE NOT SPECIFIED YOUR CONVENTION CARD YET.");
    else {
      char *newcc = malloc(strlen(oldcc) + strlen(addition) + 2);
      strcpy(newcc, oldcc);
      if (oldcc[strlen(oldcc)-1] != ' ')
	strcat(newcc, " ");
      strcat(newcc, addition);
      /* XXX Should check to make sure new cc is not too long ?? */
      free(oldcc);
      conventions[side_of(local_player)] = newcc;
      Send_cc (Local_table, newcc);
    }
  }
}
static void Parse_address_input (addr)
     char *addr;
/* ADDRESS <player-address>
 . Sets the player address.
 */
{
  GPS_Set_Field ("MAIL", addr);
  sprintf (msg_buf, "NEW ADDRESS: %s", addr);
  Status (msg_buf);
}

static void Parse_alert_input ()
/* ALERT
 . Sends an alert message showing that partner's most recent bid is 
 . unconventional.
 */
{
  int seq_no;

  if (IS_OBSERVER(local_player))
    Status ("OBSERVERS MAY NOT USE THE ALERT COMMAND.");
  else if (Local_table->game_mode != BIDDING_MODE)
    Status ("YOU MAY ONLY ALERT DURING BIDDING MODE.");
  else if (!Partner_has_bid (Local_table,Local_board,Local_play,local_player))
    Status ("YOUR PARTNER HAS NOT BID YET.");
  else if ((Local_play->next_player == player_next[local_player]) ||
	   (Local_play->next_player == player_partner[local_player]))
    Status ("YOU CANNOT ALERT AFTER YOU HAVE ALREADY BID.");
  else {
    seq_no = Index_of_Last_Bid (Local_board, Local_play, 
				player_partner[local_player]);
    Send_alert (Local_table, ALERT_BY_PARTNER, seq_no);
/* efg After sending the alert, alerter should be prompted for
	a one-line description of the alert. That message should
	then be sent as a /opp message. */
  }
}

static void Parse_assign_input (player, seat)
     char *player; int *seat;
{
  Connection c;

  if (!server_mode) {
    Status ("ONLY THE SERVER CAN USE THE /ASSIGN COMMAND.");
    return;
  }

  FOREACH_PLAYER(c, Local_table) {
    if (!strcasecmp(player, c->player_name))
      break;
  }

  if (c == NULL) {
    sprintf (msg_buf, "THERE IS NO PLAYER NAMED %s", player);
    Status (msg_buf);
    return;
  }

  if (IS_PLAYER(*seat) && OCCUPIED(Local_table, *seat)) {
    sprintf (msg_buf, "THE %s SEAT IS ALREADY OCCUPIED.", seat_names[*seat]);
    Status (msg_buf);
    return;
  }

  sprintf (msg_buf, "%s %s SEATREQ %s", seat_names[c->seat], c->player_name,
	   seat_names[*seat]);
  loopback_message_unformatted (c->table, c, msg_buf);
}

static void Parse_autopass_input (a)
     int *a;
/* AUTOPASS [ON|OFF|TOGGLE]
 . The AUTOPASS command controls whether or not the moderator will 
 . automatically provide "PASS" bids for absent players during practice mode.
 */
{
  if (client_mode) {
    Status ("ONLY THE SERVER MAY SPECIFY THE AUTOPASS MODE.");
    return;
  }

  Toggle(&autopass_mode, a);
  sprintf (msg_buf, "AUTOPASS MODE IS %s.", autopass_mode? "ON": "OFF");
  Status (msg_buf);
}

static void Parse_autosave_input (a)
     int *a;
/* AUTOSAVE [ON|OFF|TOGGLE]
 . Determines whether or not we will save the default settings of the
 . program upon exiting.
 */
{
  Toggle (&save_defaults, a);
  sprintf (msg_buf, "THE DEFAULTS %s WHEN THE PROGRAM TERMINATES.",
	   save_defaults? "WILL": "WILL NOT");
  Status (msg_buf);
}

static void Parse_bell_input (b)
     int *b;
/* BELL [ON|OFF|TOGGLE]
 . The BELL command updates the state of the bell and/or displays its
 . current state.  
 */
{
  Toggle(&bell_is_on, b);
  sprintf (msg_buf, "THE BELL IS %s.", bell_is_on? "ON": "OFF");
  Status (msg_buf);
  if (!initializing && bell_is_on)
    ring_bell ();
}


static void Parse_cc_input (whose)
     int *whose;
{
  int they;

  if (whose == NULL) {
    if (local_player < 4) {
      they = side_of(player_next[local_player]);
      if (conventions[they] == NULL)
	Status ("THEY HAVE NOT SPECIFIED THEIR CONVENTION CARD YET.");
      else
	Status (conventions[they]);
    } else {
      if ((conventions[0] == NULL) && (conventions[1] == NULL))
	Status
	  ("NEITHER SIDE HAS SPECIFIED A CONVENTION CARD YET.");
      else {
	if (conventions[0] != NULL)
	  Display_Player_Comment (COMMENT_PUBLIC, "N-S", conventions[0]);
	if (conventions[1] != NULL)
	  Display_Player_Comment (COMMENT_PUBLIC, "E-W", conventions[1]);
      }
    }
  } else {
    if (*whose < 2) {
      if (conventions[*whose] == NULL) {
	sprintf (msg_buf, "%s %s", *whose? "E-W": "N-S",
		 "HAVE NOT SPECIFIED THEIR CONVENTION CARD YET.");
	Status (msg_buf);
      } else
	Status (conventions[*whose]);
    } else if (*whose == 2) {
      if (local_cc == NULL)
	Status ("YOU HAVE NOT SPECIFIED YOUR CONVENTION CARD YET.");
      else
	Status (local_cc);
    } else {
      if ((conventions[0] == NULL) && (conventions[1] == NULL))
	Status
	  ("NEITHER SIDE HAS SPECIFIED A CONVENTION CARD YET.");
      else {
	if (conventions[0] != NULL)
	  Display_Player_Comment (COMMENT_PUBLIC, "N-S", conventions[0]);
	if (conventions[1] != NULL)
	  Display_Player_Comment (COMMENT_PUBLIC, "E-W", conventions[1]);
      }
    }
  }

}


static void Parse_ccdef_input(cc)                                 /* JAS */
     char *cc;
{
  if (!cc)
  {
    Display_All_CCs(Moderator_Comment);
  }
  else
  {
    if (Define_CC(cc, Status))
    {
      /* returns 1 on error */
      sprintf (msg_buf, "ERROR DEFINING CC: %s", cc);
      Status (msg_buf);
    }
  }
}

static int  Claim_responses_received_event ()
{
  return ((claim_responses >= 2) || exit_requested);
}

static void Handle_claim (tricks_claimed)
     int tricks_claimed;
/* Checks to see if the local player is the declarer and is able to 
   make a claim of n more tricks.  If so, then presents the claim to
   the defenders and waits for a response.  Otherwise, then prints an 
   error message and returns.
*/
{
  int tricks_played = Local_play->tricks[0] + Local_play->tricks[1];
  int tricks_made = Local_play->tricks[side_of(Local_play->declarer)];
  int tricks_lost = tricks_played - tricks_made;

  if (Local_table->game_mode != PLAYING_MODE) {
    Status ("YOU MAY ONLY CLAIM DURING THE PLAY OF THE HAND.");
    return;
  } else if (claim_in_progress) {
    Status ("A CLAIM IS ALREADY IN PROGRESS.  YOU MUST WAIT TO CLAIM AGAIN.");
    return;
  } 

  switch (Local_table->playing_mode) {
  case CLUB_PLAYING_MODE:
  case FORMAL_PLAYING_MODE:
    if (Local_play->declarer != local_player) {
      Status ("ONLY THE DECLARER MAY MAKE A CLAIM.");
      return;
    }
    break;

  case PRACTICE_PLAYING_MODE:
    if (PRACTICE(Local_table) && (Local_play->declarer != local_player)) {
      Status ("ONLY THE DECLARER MAY MAKE A CLAIM.");
      return;
    }
    break;

  case VUGRAPH_PLAYING_MODE:
    if (!server_mode) {
      Status ("ONLY THE SERVER MAY CLAIM DURING AN EXHIBITION.");
      return;
    }
    break;
  }

  if ((tricks_claimed < tricks_made) || (tricks_claimed  + tricks_lost > 13)) {
    sprintf (msg_buf, 
	     "ILLEGAL CLAIM -- YOU MAY CLAIM BETWEEN %d AND %d TRICKS.",
	     tricks_made, 13 - tricks_lost);
    Status (msg_buf);
    return;
  }

  if (PRACTICE(Local_table) || VUGRAPH(Local_table)) {
    Send_claim (Local_table, tricks_claimed);
    return;
  }

  if (tricks_claimed > tricks_made) {
    claim_in_progress = 1;
    exit_requested = 0;
    Send_claimreq (Local_table, tricks_claimed);
    Status ("WAITING FOR A REPLY TO YOUR CLAIM ...");
    Set_Input_Mode (TALK_INPUT);

    Refresh_Input_Buffers ();
    Wait_for_event_at_game_level (Claim_responses_received_event);

    Set_Input_Mode (PLAY_INPUT);
  } else
    claim_accepted = 1;

  if (claim_accepted) {
    Clear_Status ();
    Send_claim (Local_table, tricks_claimed);
  }

  claim_in_progress = 0;
}


static void Parse_claim_input (n)
     int *n;
/* CLAIM [n]
 . The CLAIM command is used by declarer to finish the hand early,
 . claiming an additional number of tricks.  If n is omitted, then
 . all remaining tricks are claimed.
 */
{
  int tricks_claimed;
  int tricks_played;
  int tricks_made;

  if (Local_play == NULL) {
    Status ("YOU MAY NOT CLAIM AT THIS TIME.");
    return;
  }

  tricks_played = Local_play->tricks[0] + Local_play->tricks[1];
  tricks_made   = Local_play->tricks[side_of(Local_play->declarer)];
  if (n == NULL)
    tricks_claimed = 13 - tricks_played + tricks_made;
  else if (*n >= 0)
    tricks_claimed = *n + tricks_made;
  else
    tricks_claimed = 13 - tricks_played + *n + tricks_made;

  Handle_claim (tricks_claimed);
}

static void Parse_down_input (n)
     int *n;
/* DOWN n
 .   Claims a total number of tricks which is n less than the level of the
 .   contract.
 */
{
  Handle_claim (level_of(Local_play->contract) + 6 - *n);
}

static void Parse_make_input (n)
     int *n;
/* MAKE [n]
 .   Claims a total of 6+n tricks, or if n is omitted, then claims a
 .   number of tricks equal to the level of the contract.
 */
{
  int tricks_claimed;

  if (n == NULL)
    tricks_claimed = level_of (Local_play->contract) + 6;
  else if (*n < 0)
    tricks_claimed = level_of(Local_play->contract) + 6 + *n;
  else
    tricks_claimed = *n + 6;
    
  Handle_claim (tricks_claimed);
}

static void Parse_cls_input ()
{
  Clear_Comment_Display ();
}

static void Parse_connect_input (s, p)
     char *s; int *p;
/* CONNECT [server] [ip]
 . The CONNECT command is used by a client to establish a connection
 . with a server. 
 */
{

  if ((s == NULL) && (server_name == NULL)) {
    Status ("YOU MUST SPECIFY THE SERVER TO WHICH TO CONNECT.");
    return;
  }

  if (server_mode && (No_connections(Local_table) > 1)) {
    Moderator_Comment 
      ("WARNING! THIS WILL DISCONNECT THE PLAYERS AT THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
	return;
  } else if (client_mode && (Local_table->game_mode != STARTUP_MODE) &&
	     IS_PLAYER(local_player)) {
    Moderator_Comment
      ("WARNING! THIS WILL DISCONNECT YOU FROM THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
      return;
  }

  if (client_mode)
    Send_quit (Local_table);

  if (p != NULL)
    network_port = *p;
  else
    network_port = DEFAULT_PORT;

  if (s != NULL)
    server_name = strdup (s);

  Generate_reset (RESET_CONNECT);
}

static void Parse_default_input (d)
     int *d;
/* DEFAULT [ON|OFF|TOGGLE]
 . The DEFAULT command controls whether or not default inputs will be
 . automatically provided for the user.  
 */
{
  Toggle (&default_plays, d);
  sprintf (msg_buf, "DEFAULT MODE IS %s", default_plays? "ON": "OFF");
  Status (msg_buf);
}

static void Parse_download_input (d)
     char *d;
/* DOWNLOAD [<email-filename>]
 . Downloads an email duplicate file from the GPS or requests a list of
 . the available email duplicate files.
 */
{
  if (d == NULL)
    GPS_Directory ();
  else
    GPS_Download (d);
}

static void Parse_disconnect_input (player)
     char *player;
{
  Connection p;

  if (!server_mode) {
    Status ("THE DISCONNECT COMMAND CAN ONLY BE USED BY THE SERVER.");
    return;
  }

  if (!strcasecmp(player, "NORTH"))
    p = Local_table->Seats[PLAYER_NORTH].connection;
  else if (!strcasecmp(player, "EAST"))
    p = Local_table->Seats[PLAYER_EAST].connection;
  else if (!strcasecmp(player, "SOUTH"))
    p = Local_table->Seats[PLAYER_SOUTH].connection;
  else if (!strcasecmp(player, "WEST"))
    p = Local_table->Seats[PLAYER_WEST].connection;
  else {
    FOREACH_CONNECTION (p)
      if (!strcasecmp(player, p->player_name))
	break;
    if (p == NULL) {
      sprintf (msg_buf, "THERE IS NO PLAYER NAMED %s", player);
      Status (msg_buf);
      return;
    }
  }

  if (p == NULL) {
    sprintf (msg_buf, "THE %s SEAT IS NOT CURRENTLY OCCUPIED.", player);
    Status (msg_buf);
    return;
  }

  if (p->local) {
    Status ("YOU CANNOT DISCONNECT YOURSELF!");
    return;
  }

  sprintf (msg_buf, "THE CONNECTION WITH %s HAS BEEN CLOSED.",p->player_name);
  Network_Comment (msg_buf);
  Disconnect_Player (p);
}

static void Parse_email_input (a)
     char *a;
/* EMAIL <email-address>
 * Specifies the local player's email address.
 */
{
  if (local_player_email != NULL)
    free (local_player_email);

  local_player_email = strdup (a);
  if (initializing) return;

  sprintf (msg_buf, "NEW EMAIL: %s", a);
  Status (msg_buf);
  Send_email (Local_table, local_player_email);
  if (Use_GPS && (local_player_id >= 0))
    GPS_Set_Field ("EMAIL", local_player_email);
}

static void Parse_exit_input ()
/* EXIT
 * Attempts to exit from whatever we might be doing at the moment.
 * Currently, this means one of two things:
 *  1. If we have made a claim, then aborts the claim.
 *  2. If we are displaying results, then aborts the display.
 */
{
  exit_requested = 1;
  query_complete = 1;
  pause_mode = 0;

  Init_Scores_to_Display ();

  if (claim_in_progress) {
    claim_in_progress = 0;
    claim_accepted = 0;
    Broadcast_Comment ("THE CLAIM HAS BEEN ABORTED.");
  }
}

static void Set_playing_mode (m)
     int m;
/* If m is -1, then displays the current playing mode.  Otherwise,
 * if we are the server, then sets the current playing mode to *m.
 */
{
  int i;

  if ((m == -1) || (m == Local_table->playing_mode)) {
    switch (Local_table->playing_mode) {
    case CLUB_PLAYING_MODE:
      Status ("WE ARE PLAYING CLUB STYLE BRIDGE.");
      return;
    case PRACTICE_PLAYING_MODE:
      Status ("WE ARE PLAYING PRACTICE HANDS.");
      return;
    case FORMAL_PLAYING_MODE:
      Status ("WE ARE PLAYING FORMAL BRIDGE.");
      return;
    case VUGRAPH_PLAYING_MODE:
      Status ("WE ARE PROVIDING A VUGRAPH EXHIBITION.");
      return;
    }
  }

  if (client_mode) {
    Status ("ONLY THE SERVER MAY CHANGE THE PLAYING MODE.");
    return;
  }

  if ((Local_table->game_mode == BIDDING_MODE) || 
      (Local_table->game_mode == PLAYING_MODE))
    if (!Ask("THIS WILL END THE CURRENT HAND.  DO YOU WISH TO DO THIS? "))
      return;

  if (VUGRAPH(Local_table)) {
    for (i = 0; i < 4; i++)
      Vacate_seat (Local_table, i);
  }

  Local_table->playing_mode = m;
  Send_skip (Local_table);
}

static void Parse_find_input( s )
     char *s;
/* FIND <player-list>
 . Find any of the given players in the list and report on their location
  */
{
  GPS_Find_Players (s);
}

static void Parse_formal_input (f)
     int *f;
/* FORMAL [ON|OFF]
 . The formal command controls whether talk messages are sent to all of
 . the other players or just to the opponents.
 */
{

  if (f == NULL)
    Set_playing_mode (FORMAL_PLAYING_MODE);
  else if (*f == TOGGLE) {
    if (Local_table -> playing_mode == FORMAL_PLAYING_MODE)
      Set_playing_mode (CLUB_PLAYING_MODE);
    else
      Set_playing_mode (FORMAL_PLAYING_MODE);
  } else
    Set_playing_mode (*f? FORMAL_PLAYING_MODE: CLUB_PLAYING_MODE);
}

static void Parse_fullname_input (f)
     char *f;
/* FULLNAME <player-full-name>
 * Specifies the player's full name. 
 */
{
  if (!strcmp(f, "(null)"))  /* a kludge to make up for a previous bug. */
    return;

  if (local_player_full_name != NULL)
    free (local_player_full_name);

  local_player_full_name = strdup (f);
  sprintf (User_fullname, "%s", f);
  
  if (initializing) return;

  sprintf (msg_buf, "NEW FULLNAME: %s", f);
  Status (msg_buf);
  Send_fullname (Local_table, User_fullname);
  if (Use_GPS && (local_player_id >= 0))
    GPS_Set_Field ("FULLNAME", User_fullname);
}

static int Exit_help_mode_event ()
{
  return (!pause_mode);
}

static void Parse_gps_input (g)
     int *g;
/* GPS [ON|OFF|TOGGLE]
 . The GPS command controls whether or not we will use the GPS to download
 . boards and locate players.
 */
{
  if (g == NULL) {
    if (!Use_GPS)
      Status ("WE ARE NOT USING THE GPS.");
    else if (GPS_unavailable)
      Status 
	("THE GPS APPEARS TO BE UNAVAILABLE.  USE '/GPS ON' TO RECONNECT.");
    else
      Status ("WE ARE USING THE GPS.");
    return;
  }

  Toggle(&Use_GPS, g);

  if (initializing) 
    return;

  sprintf (msg_buf, "WE %s USING THE GPS.", Use_GPS? "ARE": "ARE NOT");
  Status (msg_buf);

  if (Use_GPS) {
    GPS_unavailable = 0;
    GPS_Get_Message_of_the_Day ();
    if (server_mode) {
      GPS_Broadcast_Server ();
      GPS_Refresh_Players ();
    }
  } else
    GPS_Reset ();
}

static void Parse_GPS_IP_input (ip, port)
     char *ip; int *port;
{
  if (!initializing)
    GPS_Reset ();

  GPS_IP = strdup (ip);
  if (port != NULL)
    GPS_port = *port;

  if (initializing)
    return;

  Use_GPS = 1;
  GPS_unavailable = 0;
  GPS_Get_Message_of_the_Day ();
  if (server_mode) {
    GPS_Broadcast_Server ();
    GPS_Refresh_Players ();
  }

}

static void Parse_help_input (t)
     char *t;
/* HELP [<help-topic-name>]
 . Gives help on the named topic, or a general description of the program
 . if no topic is specified.
 */
{
  int input_save = input_mode;
  int display_save = display_mode;

  Set_Display_Mode (HELP_DISPLAY);
  Set_Input_Mode (TALK_INPUT);
  if (t != NULL)
    display_help (t);
  else {
    Lock_Status ("TYPE THE NAME OF A TOPIC OR PRESS <ESC> TO EXIT HELP.");
    browse_help ("");
    Refresh_Input_Buffers ();
    pause_mode = 1;
    Wait_for_event_at_game_level (Exit_help_mode_event);
    Unlock_Status ();
  }
  Set_Display_Mode (display_save);
  Set_Input_Mode (input_save);

}

static void Parse_helpfile_input (s)
	char *s;
{
  help_file_name = strdup (s);

  if (!initializing)
    initialize_help_system ();
}


static void Parse_ignore_input (s)
char *s;
/* IGNORE <string> causes all messages matching the string to not 
 . be displayed in the comment area.  Perhaps this should be improved
 . to work on wildcards and the like, but for now a comment is not
 . printed if it contains s as a substring.
 */
{
ignore *new;

  new = (ignore *) malloc (sizeof(ignore));
  new -> str = strdup(s);
  new -> next = head_ignore;
  sprintf (msg_buf, "THIS IS THE LAST COMMENT YOU WILL SEE CONTAINING: %s", s);
  Moderator_Comment (msg_buf);
  head_ignore = new;
}

static void Parse_attend_input ()
/* ATTEND cancels all ignores.  
 */
{
ignore *curr, *next;

  for (curr = head_ignore; curr != NULL; curr = next) {
    next = curr -> next;
    free (curr);
  }
  head_ignore = NULL;

  Moderator_Comment ("PREVIOUS IGNORE MESSAGES ARE NO LONGER IN EFFECT.");
}


static void Parse_key_input (code, s)
     char *code; char *s;
{
  int key, nitems;
  char describe_buf[80];

  nitems = sscanf (code, "%d", &key);
  if ((nitems == 0) && (strlen(code) == 1))
    key = *code;
  else if (nitems != 1) {
    sprintf (msg_buf, "ERRONEOUS KEY CODE: %s", code);
    Error (msg_buf);
  }
  
  if (s == NULL) {
    if (initializing) {
      Error ("A KEY MAPPING MUST BE SPECIFIED.");
      return;
    }
    Describe_Key (key, describe_buf);
    sprintf (msg_buf, "KEY %s = %s", code, describe_buf);
  } else {
    Install_Key (key, s);
    sprintf (msg_buf, "KEY %s = %s", code, s);
  }

  Status (msg_buf);
}

static void Parse_control_input (code, s)
     char *code; char *s;
{
  int key, nitems;
  char describe_buf[80];

  nitems = sscanf (code, "%d", &key);
  if ((nitems == 0) && (strlen(code) == 1) && isalpha(*code))
    key = toupper(*code) - 'A' + 1;
  else if (nitems != 1) {
    sprintf (msg_buf, "ERRONEOUS KEY CODE: %s", code);
    Error (msg_buf);
  }

  if (s == NULL) {
    if (initializing) {
      Error ("A KEY MAPPING MUST BE SPECIFIED.");
      return;
    }
    Describe_Key (key, describe_buf);
    sprintf (msg_buf, "CONTROL %s = %s", code, describe_buf);
  } else {
    Install_Key (key, s);
    sprintf (msg_buf, "CONTROL %s = %s", code, s);
  }

  Status (msg_buf);
}

static void Parse_meta_input (code, s)
     char *code; char *s;
{
  int key, nitems;
  char describe_buf[80];

  nitems = sscanf (code, "%d", &key);
  if ((nitems == 0) && (strlen(code) == 1))
    key = *code;
  else if (nitems != 1) {
    sprintf (msg_buf, "ERRONEOUS KEY CODE: %s", code);
    Error (msg_buf);
  }

  if (s == NULL) {
    if (initializing) {
      Error ("A KEY MAPPING MUST BE SPECIFIED.");
      return;
    }
    Describe_Meta_Key (key, describe_buf);
    sprintf (msg_buf, "META %s = %s", code, describe_buf);
  } else {
    Install_Meta_Key (key, s);
    sprintf (msg_buf, "META %s = %s", code, s);
  }

  Status (msg_buf);
}

static void Parse_join_input (s)
     char *s;
/* JOIN [<server-name>]
 . Joins the table being served by the named person.  If the server-name
 . is omitted, then joins the most recently opened table.
 */
{
  char server_ip_buf[60];
  int server_port, status;

  Conclude_GPS_request ();

  suspend_gps_input = 0;
  status = GPS_Get_Server_IP (s, server_ip_buf, &server_port);
  exit_requested = 1;
  pause_mode = 0;

  if (status < 0) {
    if (s == NULL)
      Status ("THERE ARE NO TABLES CURRENTLY BEING SERVED.");
    else
      Status ("NO TABLE IS BEING HOSTED BY THAT PERSON.");
    return;
  } else if (status > 0)
    return;

  if (server_mode && (No_connections(Local_table) > 1)) {
    Moderator_Comment 
      ("WARNING! THIS WILL DISCONNECT THE PLAYERS AT THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
	return;
  } else if (client_mode && (Local_table->game_mode != STARTUP_MODE) &&
	     IS_PLAYER(local_player)) {
    Moderator_Comment
      ("WARNING! THIS WILL DISCONNECT YOU FROM THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
      return;
  }

  if (client_mode)
    Send_quit (Local_table);

  network_port = server_port;
  server_name = strdup (server_ip_buf);

  Generate_reset (RESET_CONNECT);
}

static void Parse_level_input (lvl)
     char *lvl;
/* LEVEL <player-level>
 . Records the player level with the GPS.
 */
{
  GPS_Set_Field ("LEVEL", lvl);
  sprintf (msg_buf, "NEW LEVEL: %s", lvl);
  Status (msg_buf);
}

static void Parse_lho_input (m)
     char *m;
/* LHO <message>
 . Sends a message to the left-hand opponent.
 */
{
  if (IS_OBSERVER(local_player))
    Status ("PRIVATE MESSAGES CANNOT BE SENT BY AN OBSERVER.");
  else {
    Display_Player_Comment (COMMENT_PRIVATE, local_player_name, m);
    Send_talk (Local_table, TALK_RCPT_LHO, m);
  }
}

static void Parse_load_input (f)
     char *f;
/* LOAD <filename>
 . The LOAD command is used by the server to LOAD a sequence of email
 . boards from a file.  
 */
{
  if (client_mode) {
    Status ("CLIENTS CANNOT LOAD EMAIL BOARDS.");
    return;
  }

  replay_mode = 0;
  load_email_file (f);
}

static void Parse_lurker_input ()
/* LURKER
 . Makes the person a lurker, disconnecting from table or serving.
 */
{

  if (server_mode && (No_connections(Local_table) > 1)) {
    Moderator_Comment 
      ("WARNING! THIS WILL DISCONNECT THE PLAYERS AT THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
	return;
  } else if (client_mode && (Local_table->game_mode != STARTUP_MODE) &&
	     IS_PLAYER(local_player)) {
    Moderator_Comment
      ("WARNING! THIS WILL DISCONNECT YOU FROM THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
      return;
  } else if (!client_mode && !server_mode) {
    Moderator_Comment("YOU ARE ALREADY A LURKER.");
    return;
  }

  if (client_mode)
    Send_quit (Local_table);

  Generate_reset (RESET_LURKER);
}

static FILE *Open_Log_File (filename, success_message, saved_filename)
     char *filename; char *success_message; char **saved_filename;
{
  char error_buf[80];
  FILE *fp;

  fp = fopen (filename, "a");
  if (fp == NULL) {
    sprintf (error_buf, "ERROR OPENING %s: %s", filename, 
	     sys_errlist[errno]);
    Error (error_buf);
  } else if (success_message != NULL) {
    Status (success_message);
  }
  
  if (saved_filename != NULL)
    *saved_filename = (fp == NULL)? NULL: strdup(filename);
  return (fp);
}

static FILE *Open_Email_Log_File (filename, success_message, saved_filename)
     char *filename;  char *success_message; char **saved_filename;
{
  char error_buf [80];
  FILE *fp;
  Cipher c;

  fp = fopen (filename, "r");
  if (fp == NULL) {
    fp = fopen (filename, "w");
    if (fp == NULL) {
      sprintf (error_buf, "ERROR CREATING EMAIL LOG FILE %s: %s", filename,
	       sys_errlist[errno]);
      Status (error_buf);
      if (saved_filename != NULL)
	*saved_filename = NULL;
      return (NULL);
    }
    Write_Email_Header (fp);
    Create_Cipher_Descriptor (fp, &c);
    Write_Cipher_Descriptor (fp, &c);
    fflush (fp);
    Status (success_message);
    if (saved_filename != NULL)
      *saved_filename = (fp == NULL)? NULL: strdup(filename);
    return (fp);
  }

  fclose (fp);
  fp = fopen (filename, "a");
  if (fp == NULL) {
    sprintf (error_buf, "ERROR OPENING EMAIL LOG FILE %s: %s", filename,
	     sys_errlist[errno]);
    Error (error_buf);
    return (NULL);
  }

  if (saved_filename != NULL)
    *saved_filename = (fp == NULL)? NULL: strdup(filename);
  Status (success_message);
  return (fp);
  
}

static void Parse_log_input (f, options)
     char *f; int *options;
/* LOG [<filename>  [TEXT|EMAIL|BOTH]]
 . The LOG command opens a file to which the results of play will be
 . recorded, or it closes a currently open logfile if no filename is
 . specified.  
 */
{
  char email_filename[80];

  if ((logfile == NULL) && (logfile_email == NULL)) {
    if (f == NULL)
      Moderator_Comment ("THERE IS NO OPEN LOGFILE.");
  } else {
    if (logfile != NULL)
      fclose (logfile);
    if (logfile_email != NULL)
      fclose (logfile_email);
    logfile = NULL;
    logfile_email = NULL;
    if (f == NULL)
      Moderator_Comment ("THE LOG FILE HAS BEEN CLOSED.");
  }

  if (f == NULL)
    return;

  if ((options == NULL) || (*options == 0)) {
    sprintf (msg_buf, "NOW LOGGING TO %s", f);
    logfile = Open_Log_File (f, msg_buf, &logfile_name);
  } else if (*options == 1) {
    sprintf (msg_buf, "NOW LOGGING EMAIL BOARDS TO %s", f);
    logfile_email = Open_Email_Log_File (f, msg_buf, &logfile_name);
  } else {
    sprintf (email_filename, "%s.em", f);
    sprintf (msg_buf, "NOW LOGGING TO %s AND %s.", f, email_filename);
    logfile = Open_Log_File (f, msg_buf, &logfile_name);
    if (logfile != NULL)
      logfile_email = Open_Email_Log_File (email_filename, msg_buf, NULL);
  }
}

int Get_password_and_login (login)
     char *login;
{
  char buf[100], pw_buf1[80], pw_buf2[80];
  int login_exists;

  local_player_id = -1;
  if (client_mode)
    Send_name (Local_table, local_player_name, local_player_id);

  if (local_player_pw != NULL)
    free (local_player_pw);
  local_player_pw = NULL;
  Display_Player_Position ();

  if (!String_is_alphameric(login) || (strlen(login) > 8)) {
    sprintf (buf, "ILLEGAL LOGIN NAME: %s", login);
    Status (buf);
    Moderator_Comment ("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
    return (0);
  }

  login_exists = GPS_Check_for_Login (login);

  if (GPS_unavailable) {
    Status ("THE GPS IS CURRENTLY UNAVAILABLE.");
    return (0);
  }
  
  if (login_exists) {
    sprintf (buf, "ENTER PASSWORD FOR %s: ", login);
    Query (buf, NULL, 0, pw_buf1, 80);
    local_player_id = GPS_Login (login, pw_buf1);
    if (local_player_id < 0) {
      return (0);
    }
  } else {
    Clear_Comment_Display ();
    Moderator_Comment ("NO ACCOUNT EXISTS WITH THIS NAME!");
    sprintf (buf, "DO YOU WISH TO CREATE AN ACCOUNT FOR %s?", login);
    if (!Ask(buf)) {
      Moderator_Comment
	("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
      return (0);
    }
    sprintf (buf, "ENTER PASSWORD FOR %s: ", login);
    Query (buf, NULL, 0, pw_buf1, 80);
    if (!String_is_alphameric(pw_buf1)) {
      Pause ("ILLEGAL PASSWORD ... PRESS <ESC> TO CONTINUE.");
      Moderator_Comment
	("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
      return (0);
    }
    
    if (strlen(pw_buf1) > 8) {
      Pause 
	("PASSWORDS MAY NOT BE LONGER THAN 8 CHARACTERS ... PRESS <ESC> TO CONTINUE.");
      Moderator_Comment
	("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
      return (0);
    }

    if (strlen (pw_buf1) > 0) {
      Query ("ENTER PASSWORD AGAIN: ", NULL, 0, pw_buf2, 80);
      if (strcmp(pw_buf1, pw_buf2)) {
	Pause
	  ("PASSWORDS DO NOT MATCH!");
	Moderator_Comment 
	  ("USE THE COMMAND '/LOGIN <player-name>' TO LOGIN.");
	return (0);
      }
      local_player_id = GPS_Create_Account (login, pw_buf1);
      if (local_player_id < 0)
	return (0);
      if (local_player_email != NULL)
	GPS_Set_Field ("EMAIL", local_player_email);
      if (local_player_full_name != NULL)
	GPS_Set_Field ("FULLNAME", local_player_full_name);
    } else {
      Moderator_Comment ("NO ACCOUNT CREATED.");
      return (0);
    }
  }

  if (local_player_name != NULL)
    free (local_player_name);
  local_player_name = strdup (login);
  strcpy (Local_Player_Connection->player_name, local_player_name);

  if (local_player_pw != NULL)
    free (local_player_pw);
  local_player_pw = strdup (pw_buf1);

  Local_Player_Connection->player_id = local_player_id;
  if (client_mode)
    Send_name (Local_table, local_player_name, local_player_id);
  GPS_Get_Scores (&local_player_mp_total, &local_player_imp_total);

  sprintf (buf, "LOGGED IN AS %s.", local_player_name);
  Moderator_Comment (buf);
  
  Display_Player_Position ();
  return (1);
}

int Initiate_Login (id)
     char *id;
/* Performs a single login attempt.  If successful, returns 1.  If not,
   prints an error message and returns 0. */
{
  char login_buf[80];
  int acceptable_login;

  Query ("ENTER YOUR LOGIN NAME: ", id, 1, login_buf, 80);
  acceptable_login = String_is_alphameric (login_buf);
  if (!acceptable_login) {
    Moderator_Comment ("THAT IS NOT AN ACCEPTABLE LOGIN NAME!");
    return (0);
  }

  return (Get_password_and_login (login_buf));
}

static void Parse_login_input (id)
     char *id;
{
  char id_buf[100];
  char *user_id = (id == NULL)? local_player_name: id;
  int Was_using_GPS = Use_GPS;

  if (initializing) {
    if (!String_is_alphameric(id))
      Error ("PLAYER NAMES MUST NOT CONTAIN SPECIAL CHARACTERS");
    else
      local_player_name = strdup (id);
    return;
  }

  strcpy (id_buf, user_id);
  Use_GPS = 1;
  if (Get_password_and_login (id_buf) && !Was_using_GPS) {
    GPS_unavailable = 0;
    GPS_Get_Message_of_the_Day ();
    if (server_mode) {
      GPS_Broadcast_Server ();
      GPS_Refresh_Players ();
    }
  }
}

static void Parse_lrec_input ()
/* LREC
 * Writes the last hand played to the record file.
 */
{
  if ((Prev_board == NULL) || (Prev_play == NULL)) {
    Status ("THERE IS NO PREVIOUS BOARD TO RECORD.");
    return;
  }

  if ((recfile == NULL) && (recfile_email == NULL)) {
    Status ("THERE IS NO RECORD FILE CURRENTLY OPEN.");
    return;
  }

  if (recfile != NULL)
    Write_hand (recfile, Prev_board, Prev_play, 1);
  if (recfile_email != NULL) {
    Write_Email_Board (recfile_email, NULL, Prev_board);
    fflush (recfile_email);
  }

  Status ("THE LAST BOARD HAS BEEN WRITTEN TO THE RECORD FILE.");
}

static void Parse_message_input (m)
     char *m;
/* MESSAGE [<GPS-message>]
 *  If we are in server mode, sends a message to the GPS which will be
 *  displayed with our table announcement.
 */
{
  if (!server_mode) {
    Status 
      ("YOU CANNOT SEND A GPS MESSAGE UNLESS YOU ARE IN SERVER MODE.");
    return;
  }

  if (GPS_unavailable || !Use_GPS) {
    Status
      ("THE GLOBAL PLAYER SERVICE IS CURRENTLY UNAVAILABLE.");
    return;
  }

  GPS_Advertise_Message (m);
}

static void Parse_opp_input (m)
     char *m;
/* OPP <message>
 . Sends a message to (both of) the opponents.
 */
{
  if (IS_OBSERVER(local_player))
    Status ("PRIVATE MESSAGES CANNOT BE SENT BY AN OBSERVER.");
  else {
/*    Display_Player_Comment (COMMENT_FORMAL, local_player_name, m); */
    Send_talk (Local_table, TALK_RCPT_OPPS, m);
  }
}

static void Parse_password_input ()
/* PASSWORD
 . Requests a password change.
 */
{
  char old_pw_buf[80], new_pw_buf[80], new_pw_buf2[80];

  if (local_player_id < 0) {
    Status ("YOU MUST FIRST BE LOGGED IN TO CHANGE YOUR PASSWORD.");
    return;
  }

  Query ("ENTER OLD PASSWORD: ", NULL, 0, old_pw_buf, 80);

  Query ("ENTER NEW PASSWORD: ", NULL, 0, new_pw_buf, 80);
  if (!String_is_alphameric(new_pw_buf)) {
    Status ("ILLEGAL PASSWORD.");
    return;
  }

  if (strlen(new_pw_buf) > 8) {
    Status ("PASSWORDS MAY NOT BE LONGER THAN 8 CHARACTERS.");
    return;
  }

  Query ("ENTER NEW PASSWORD AGAIN: ", NULL, 0, new_pw_buf2, 80);
  if (strcmp(new_pw_buf, new_pw_buf2)) {
    Status ("PASSWORDS DON'T MATCH!");
    return;
  }

  GPS_Change_Password (old_pw_buf, new_pw_buf);
}

static void Parse_pause_input ()
/* PAUSE
 . Returns to full screen talk mode if we are not already there.
 */
{
  int d = display_mode;
  int i = input_mode;
  
  if (display_mode != TALK_DISPLAY) {
    Set_Display_Mode (TALK_DISPLAY);
    Set_Input_Mode (TALK_INPUT);
    Pause ("");
    Set_Display_Mode (d);
    Set_Input_Mode (i);
  }
  Clear_Status ();
}

static void Parse_perish_input ()
/* PERISH
 . If we are a server, removes our table from the GPS database.
 */
{
  if (!server_mode) {
    Status ("ONLY THE SERVER CAN USE THE /PERISH COMMAND.");
    return;
  }

  if (!Use_GPS) {
    Status ("THE /PERISH COMMAND ONLY APPLIES IF YOU ARE USING THE GPS.");
    return;
  }

  GPS_Perish ();
}

static void Parse_ping_input ()
/* PING
 . Sends a message to each of the other players, which is automatically
 . echoed.  The time delay until the echo's are received is reported.
 */
{
  gettimeofday (&ping_start, NULL);
  Send_ping (Local_table);
}

static void Parse_players_input (s)
     char *s;
/* PLAYERS <table-name>
 . Lists the players at the given table.
 */
{
  GPS_List_Players (s);
}

static void Parse_practice_input (p)
     int *p;
/* PRACTICE [ON|OFF]
 . Practice mode is used for partnerships to practice bidding and planning
 . play.
 */
{
  if (p == NULL)
    Set_playing_mode (PRACTICE_PLAYING_MODE);
  else if (*p == TOGGLE) {
    if (Local_table -> playing_mode == PRACTICE_PLAYING_MODE)
      Set_playing_mode (CLUB_PLAYING_MODE);
    else
      Set_playing_mode (PRACTICE_PLAYING_MODE);
  } else
    Set_playing_mode (*p? PRACTICE_PLAYING_MODE: CLUB_PLAYING_MODE);
}

static void Parse_prompt_input (p)
     int *p;
/* PROMPT [ON|OFF|TOGGLE|NEVER]
 . Sets or toggles whether the player is always prompted after each
 . trick has been played.  For some reason, var is called prompt_dummy.
 */
{

  if ((p != NULL) && (*p == 3)) {
    /* special case: /PROMPT NEVER */
    never_prompt_save = 1;
    Status ("YOU WILL NEVER BE PROMPTED.");
    return;
  }

  if (never_prompt_save)
    never_prompt_save = 0;
  else
    Toggle (&prompt_dummy, p);

  sprintf (msg_buf,  "YOU %s BE PROMPTED AFTER EACH TRICK",
	   prompt_dummy? "WILL": "WILL NOT");
  Status (msg_buf);
}

static void Parse_publish_input ()
/* PUBLISH
 . If we are a server, restores our table to the GPS database.
 . (This is the inverse of the /perish command.)
 */
{
  if (!server_mode) {
    Status ("ONLY THE SERVER CAN USE THE /PERISH COMMAND.");
    return;
  }

  if (!Use_GPS) {
    Status ("THE /PERISH COMMAND ONLY APPLIES IF YOU ARE USING THE GPS.");
    return;
  }

  GPS_Publish ();
  GPS_Refresh_Players ();
}

static void Parse_quiet_input (q)
     int *q;
/* QUIET [ON|OFF|TOGGLE]
 . The quiet command toggles the state of the quiet mode flag, which
 . controls whether JOIN, SPEC and QUIT messages are displayed.
 */
{
  Toggle(&quiet_mode, q);
  sprintf (msg_buf, "QUIET MODE IS %s.", quiet_mode? "ON": "OFF");
  Status (msg_buf);
}


static void Parse_quit_input ()
/* QUIT
 . Terminates the program.
 */
{
  Quit_program ();
}

static void Parse_rec_input (f, options)
     char *f; int *options;
/* REC [<filename>  [TEXT|EMAIL|BOTH]]
 . The REC command is similar to the log command, except that hands
 . are only recorded on demand.
 */
{
  char email_filename[80];

  if (f == NULL) {
    if ((recfile == NULL) && (recfile_email == NULL))
      Moderator_Comment ("NO FILE IS OPEN FOR RECORDING.");
    else
      record_this_hand = 1;
    return;
  }

  sprintf 
    (msg_buf, 
     "TO RECORD A HAND OF INTEREST, YOU MAY NOW USE THE '/REC' COMMAND.");
	   
  if ((options == NULL) || (*options == 0)) {
    recfile = Open_Log_File (f, msg_buf, &recfile_name);
  } else if (*options == 1) {
    recfile_email = Open_Email_Log_File (f, msg_buf, &recfile_name);
  } else {
    sprintf (email_filename, "%s.em", f);
    recfile = Open_Log_File (f, msg_buf, &recfile_name);
    if (recfile != NULL)
      recfile_email = Open_Email_Log_File (email_filename, msg_buf, NULL);
  }
  
}

static void Parse_remcc_input (items)
    char *items ;
/* REMCC <conventions>
 . Removes the conventions from the current cc, if they are there
 */
{
  if (items && local_player < 4) {
    char *oldcc = conventions[side_of(local_player)];
    if (oldcc == NULL)
	Status ("YOU HAVE NOT SPECIFIED YOUR CONVENTION CARD YET.");
    else {
      enum { _looking, _found } state;
      char *src, *dst, *tok;
      int all, found_some = 0;

      for (tok = strtok(items, " "); tok; tok = strtok(NULL, " ")) {
	src = dst = oldcc;
	all = strlen(tok);
	state = _looking;
	while (*src) {
	  if (state == _looking && strncasecmp(src, tok, all) == 0) {
	    state = _found;
	    if (src > oldcc && src[-1] == ' ' && src[all] == ' ')
	      src++;
	    src += all;
	    continue;
	  }
	  *dst++ = *src++;
	}
	*dst = 0;
	if (state == _found)
	  found_some = 1;
      }
      if (found_some)
	Send_cc (Local_table, oldcc);
      else
	Status("NO SUCH ITEM ON YOUR CONVENTION CARD.");
    }
  }
}

static void Parse_replay_input (f)
     char *f;
/* REPLAY <filename>
 . Loads a sequence of email boards from the file f.  After they have been
 . played, writes the results back to f.
 */
{
  if (client_mode) {
    Status ("CLIENTS CANNOT LOAD EMAIL BOARDS.");
    return;
  }

  replay_mode = 1;
  load_email_file (f);
}

static void Parse_reset_input ()
/* RESET
 . Resets the state of the program.
 */
{
  Connection c;
  Table t;

  if (client_mode) {
    Status ("CLIENTS MAY NOT USE THE /RESET COMMAND.");
    return;
  }

  for (t = Table_List; t != NULL; t = t->next)
    Send_reset (t);

  FOREACH_CONNECTION (c)
    if (!c->local)
      Assign_seat (c->table, c, PLAYER_OBS);

}

static void Parse_results_input (h)
    int *h;
/* RESULTS
 . Lists the results of the previous hand.
 */
{
  int current_display_mode = display_mode;
  int current_input_mode   = input_mode;
  struct Play_record_struct *current_Local_play = Local_play;
  struct Board_struct *save_board = Prev_board;
  struct Play_record_struct *save_play = Prev_play;
  static struct Board_struct *current_Local_board = 0;
  Play_record *p,*last_record;
  static showing_results = 0;

  if (result_mode == 2)
    result_mode = 1;
  else
    result_mode = 0;

  if (h && *h == 0) {
    if (!spectator_mode) {
      Status ("ONLY SPECTATORS CAN DISPLAY RESULTS OF CURRENT BOARD.");
      return;
    }
    if (current_Local_board != Local_board) {
      if ((p = Local_board->play_records->next) == NULL ||
          p->next == NULL) {
        Status ("THE BOARD DOES NOT HAVE RESULTS TO REVIEW.");
        return;
      }
      for (p = Local_board->play_records; p->next->next != NULL; p = p->next) {
        last_record = p->next->next;
      }
      p->next = NULL;
      Compute_Scores_for_Board (Local_board);
      p->next = last_record;
      if (IS_OBSERVER(local_player))
        sort_direction = 1;
      else if (side_of(local_player) == SIDE_NS)
        sort_direction = 1;
      else
        sort_direction = -1;
      switch (Local_board->scoring_mode) {
      case MP_SCORING:
        Sort_play_records_by_matchpoints (Local_board);
        break;
      case IMP_SCORING:
        Sort_play_records_by_imps (Local_board);
      }
      current_Local_board = Local_board;
    }
    Prev_board = Local_board;
  }

  if (Prev_board == NULL) {
    Status ("THERE IS NO BOARD TO REVIEW.");
    Prev_board = save_board;
    return;
  } else if (Prev_board->play_records->next == NULL) {
    Status ("THE BOARD DOES NOT HAVE RESULTS TO REVIEW.");
    Prev_board = save_board;
    return;
  } else if (showing_results || (More_Scores_to_Display()
      && Local_table->game_mode == SCORING_MODE)) {
    Status ("ERROR: WE ARE ALREADY PROCESSING A RESULTS REQUEST.");
    Prev_board = save_board;
    return;
  }

  showing_results = 1;
  if (h && *h == 0)  Prev_play = Local_play;
   
  Local_play = Prev_play;
  exit_requested = 0;
  Display_First_Page_of_Scores (Prev_board);
  while (More_Scores_to_Display() && !exit_requested) {
    Local_play = current_Local_play;
    Pause ("PRESS <ESC> TO SEE MORE RESULTS OR /EXIT TO QUIT...");
    if (!exit_requested) {
      Local_play = Prev_play;
      Display_More_Scores ();
    }
  }
  Local_play = current_Local_play;
  if (!exit_requested && (current_display_mode != SCORING_DISPLAY))
    Pause ("PRESS <ESC> TO CONTINUE ...");
  Prev_play = save_play;
  Prev_board = save_board;
  Set_Display_Mode (current_display_mode);
  Set_Input_Mode (current_input_mode);
  Init_Scores_to_Display ();
  showing_results = 0;
}

static void Parse_summary_input (h)
    int *h;
/* SUMMARY
 . Lists the summary of the previous hand.
 */
{
  result_mode = 2;
  Parse_results_input (h);
}

static void Reveal_hand (position)
     int position;
/* Reveals to the observer the contents of the hand held by 'position' */
{
  if (!spectator_mode) {
    spectator_mode = 1;
    Send_spec (Local_table);
    if (!PRACTICE(Local_table))
      Status ("YOU MAY TALK ONLY TO OTHER SPECTATORS NOW.");
    if (!pause_mode)
      Display_Player_Position ();
  }
    
  revealed_bidder = position;
  revealed_hands[position] = 1;
  
  if (pause_mode) return;

  Display_Hand (position);
  Refresh_Playing_Area ();
}

static void Parse_reveal_input (player)
     char *player;
{
  int i, known;
  int game_mode = Local_table->game_mode;

  if (VUGRAPH(Local_table)) {
    Status ("THERE IS NO SPECTATOR MODE DURING VUGRAPH EXHIBITIONS.");
    return;
  }

  if (IS_PLAYER(local_player) && !PRACTICE(Local_table)) {
    if (game_mode != PLAYING_MODE) {
      Status ("YOU MAY NOT ENTER SPECTATOR MODE IF YOU ARE PLAYING.");
      return;
    } else if (local_player != player_partner[Local_play->declarer]) {
      Status 
	("ONLY THE OBSERVER AND THE DUMMY MAY SEE OTHER PLAYER'S HANDS!");
      return;
    }
  }

  if ((game_mode != BIDDING_MODE) && (game_mode != PLAYING_MODE)) {
    Status ("YOU WILL NOT BE ABLE TO SEE ANYTHING UNTIL NEXT HAND.");
    return;
  }

  if (FORMAL(Local_table)) {
    Status ("YOU MAY NOT SEE THE PLAYER'S HANDS DURING FORMAL PLAY.");
    return;
  }

  if (player == NULL) {
    Reveal_hand (PLAYER_WEST);
    Reveal_hand (PLAYER_SOUTH);
    Reveal_hand (PLAYER_EAST);
    Reveal_hand (PLAYER_NORTH);
  } else if (!strcasecmp(player, "NORTH") || !strcasecmp(player, "N"))
    Reveal_hand (PLAYER_NORTH);
  else if (!strcasecmp(player, "EAST") || !strcasecmp(player, "E"))
    Reveal_hand (PLAYER_EAST);
  else if (!strcasecmp(player, "SOUTH") || !strcasecmp(player, "S"))
    Reveal_hand (PLAYER_SOUTH);
  else if (!strcasecmp(player, "WEST") || !strcasecmp(player, "W"))
    Reveal_hand (PLAYER_WEST);
  else if (!strcasecmp(player, "NE") || !strcasecmp(player, "EN")) {
    Reveal_hand (PLAYER_EAST);
    Reveal_hand (PLAYER_NORTH);
  } else if (!strcasecmp(player, "NS") || !strcasecmp(player, "SN")) {
    Reveal_hand (PLAYER_SOUTH);
    Reveal_hand (PLAYER_NORTH);
  } else if (!strcasecmp(player, "NW") || !strcasecmp(player, "WN")) {
    Reveal_hand (PLAYER_WEST);
    Reveal_hand (PLAYER_NORTH);
  } else if (!strcasecmp(player, "SE") || !strcasecmp(player, "ES")) {
    Reveal_hand (PLAYER_SOUTH);
    Reveal_hand (PLAYER_EAST);
  } else if (!strcasecmp(player, "SW") || !strcasecmp(player, "WS")) {
    Reveal_hand (PLAYER_WEST);
    Reveal_hand (PLAYER_SOUTH);
  } else if (!strcasecmp(player, "EW") || !strcasecmp(player, "WE")) {
    Reveal_hand (PLAYER_WEST);
    Reveal_hand (PLAYER_EAST);
  } else {
    known = 0;
    for (i = 0; i < 4; i++)
      if (OCCUPIED(Local_table, i) && 
	  !strcasecmp(player, PLAYER_NAME(Local_table, i))) {
	Reveal_hand (i);
	known = 1;
      }
    if (!known) {
      sprintf (msg_buf, "THERE IS NO PLAYER NAMED %s", player);
      Status (msg_buf);
      return;
    }
  }

}

static void Parse_rho_input (m)
     char *m;
/* RHO <message>
 . Sends a message to the right-hand opponent.
 */
{
  if (IS_OBSERVER(local_player))
    Status ("PRIVATE MESSAGES CANNOT BE SENT BY AN OBSERVER.");
  else {
    Display_Player_Comment (COMMENT_PRIVATE, local_player_name, m);
    Send_talk (Local_table, TALK_RCPT_RHO, m);
  }
}
     
static void Parse_save_input (f)
     char *f;
/* SAVE <filename>
 . Saves the sequence of boards that have been played to the file f.
 */
{
  if ((f == NULL) && (email_filename == NULL)) {
    Status ("YOU MUST SPECIFY THE NAME OF THE EMAIL SAVE FILE.");
    return;
  } else if (f == NULL)
    save_email_file (email_filename);
  else
    save_email_file (f);
}

static void Parse_score_input (s)
     int *s;
/* SCORE RUBBER|DUPLICATE|IMP|MP|CLEAR
 . Chooses a scoring mode.  Can be used by the server.
 */
{
  if (initializing) {
    scoring_mode = *s;
    return;
  }

  if (client_mode) {
    Status ("ONLY THE SERVER MAY CHANGE THE SCORING MODE.");
    return;
  }

  if (*s == Local_table->scoring_mode) return;

  if ((Local_table->game_mode == PLAYING_MODE) || 
      (Local_table->game_mode == BIDDING_MODE)) {
    Moderator_Comment 
      ("CHANGE WILL TAKE EFFECT AT NEXT HAND.");
    new_score_mode = *s;
    return;
  }

  Local_table->scoring_mode = *s;

  bzero(&Local_table->score[SIDE_NS], sizeof(Score));
  bzero(&Local_table->score[SIDE_EW], sizeof(Score));
  bzero(&local_player_score, sizeof(Score));

  if ((Local_table->game_mode != STARTUP_MODE) &&
      (Local_table->game_mode != SCORING_MODE))
    Send_skip (Local_table);
  else {
    Moderator_Comment 
      ("YOU MAY NEED TO /SKIP FOR THE SCORING CHANGE TO TAKE EFFECT.");
  }

  scoring_mode = *s;

  switch (scoring_mode) {
  case RUBBER_SCORING:
    Moderator_Comment ("WE ARE PLAYING RUBBER BRIDGE.");
    break;
  case DUPLICATE_SCORING:
    Moderator_Comment ("WE ARE PLAYING DUPLICATE BRIDGE.");
    break;
  case IMP_SCORING:
    Moderator_Comment ("WE ARE PLAYING IMP BRIDGE.");
    break;
  case MP_SCORING:
    Moderator_Comment ("WE ARE PLAYING MATCH POINT BRIDGE.");
    break;
  }

}

static void Parse_scoreboard_input (sc)
/* SCOREBOARD [ON|OFF|TOGGLE]
 . Displays the GPS scoreboard.
 */
int *sc;
{
  if ((sc == NULL) && (!initializing)) {
    GPS_Display_Scoreboard (1);
  } else {
    Toggle (&display_scoreboard_at_end, sc);
    sprintf (msg_buf, "THE SCOREBOARD %s BE DISPLAYED AT THE END.",
	     show_alerts ? "WILL": "WILL NOT");
    Status (msg_buf);
  }
}

static void Parse_seat_input (s)
     int *s;
/* SEAT NORTH|EAST|SOUTH|WEST|OBS
 . Requests the given seat for the local player.
 */
{
/*  CHECK_ENABLED (INPUT_SEAT); */

  if (spectator_mode) {
    Status ("YOU CANNOT CHANGE YOUR SEAT IN SPECTATOR MODE.");
    return;
  }

  Send_seatreq (Local_table, *s);
}

static void Handle_seat_command (seat, cmd)
     int seat; char *cmd;
{
  if (cmd == NULL)
    Parse_seat_input (&seat);
  else if (!server_mode
#ifndef cleggy
	   || !VUGRAPH(Local_table)
#endif
	   ) {
    sprintf (msg_buf, "THE /%s COMMAND DOES NOT HAVE ANY PARAMETERS.",
	     seat_names[seat]);
    Status (msg_buf);
  } 
  else {
    sprintf (msg_buf, "%s %s %s", seat_names[seat],
	     PLAYER_NAME(Local_table, seat), cmd);
    loopback_message_unformatted 
      (Local_table, Shadow_Connections[seat], msg_buf);
  }
}

static void Parse_east_input (cmd)
     char *cmd;
{
  Handle_seat_command (PLAYER_EAST, cmd);
}

static void Parse_north_input (cmd)
     char *cmd;
{
  Handle_seat_command (PLAYER_NORTH, cmd);
}

static void Parse_observe_input ()
{
  int i = PLAYER_OBS;

  if (local_player == PLAYER_OBS) {
    Status ("YOU ARE ALREADY AN OBSERVER.");
    return;
  }

  Parse_seat_input (&i);
}

static void Parse_south_input (cmd)
     char *cmd;
{
  Handle_seat_command (PLAYER_SOUTH, cmd);
}

static void Parse_west_input (cmd)
     char *cmd;
{
  Handle_seat_command (PLAYER_WEST, cmd);
}

static void Parse_serve_input (p)
     int *p;
/* SERVE [<port-number>]
 . Enters Server mode.
 */
{
  Conclude_GPS_request ();
  exit_requested = 1;
  pause_mode = 0;

  if (client_mode && (Local_table->game_mode != STARTUP_MODE) &&
      IS_PLAYER(local_player)) {
    Moderator_Comment
      ("WARNING! THIS WILL DISCONNECT YOU FROM THIS TABLE!");
    if (!Ask("ARE YOU SURE YOU WISH TO DO THIS? "))
      return;
  } else if (server_mode) {
    Status ("YOU ARE ALREADY SERVING A TABLE.");
    GPS_Refresh_Players ();
    return;
  }

  if (p != NULL)
    network_port = *p;
  else
    network_port = DEFAULT_PORT;

  if (client_mode)
    Send_quit (Local_table);

  Generate_reset (RESET_SERVE);
}

static void Parse_setcc_input (cc)
     char *cc;
{
  if (spectator_mode) {
    Status ("YOU MAY NOT MODIFY YOUR CC WHILE YOU ARE A SPECTATOR.");
    return;
  }

  if (cc != NULL) {
    if (local_cc != NULL)
      free (local_cc);
    local_cc = strdup (Get_CC_Value(cc));                             /* JAS */
  }

  if ((local_player < 4) && (local_cc != NULL)) {
    Send_cc (Local_table, local_cc);
    conventions[side_of(local_player)] = strdup(local_cc);
    sprintf (msg_buf, "%s: %s", (side_of(local_player) == SIDE_NS)? 
	     "N-S": "E-W", local_cc);
  } else if (local_cc == NULL)
    Status ("YOUR CONVENTION CARD IS EMPTY.");
}

static void Parse_showalerts_input (sr)
/* SHOWALERTS [ON|OFF|TOGGLE]
 . The SHOWALERTS command controls whether or not a player sees the
 . alerts made by hir partner.
 */
int *sr;
{
  Toggle (&show_alerts, sr);
  sprintf (msg_buf, "PARTNER'S ALERTS %s BE DISPLAYED.",
	   show_alerts ? "WILL": "WILL NOT");
  Status (msg_buf);
}

static void Parse_showresults_input (sr)
     int *sr;
/* SHOWRESULTS ["OFF|SUMMARY|FULL|ON]
 . The SHOWRESULTS command controls whether or not the results are
 . displayed at the end of each duplicate hand.
 */
{
  show_results = *sr;
  sprintf (msg_buf, "DUPLICATE RESULTS %s BE DISPLAYED",
	   show_results? "WILL": "WILL NOT");
  switch (show_results) {
    case 0:
      sprintf (&msg_buf[strlen(msg_buf)], ".");
      break;
    case 1:
      sprintf (&msg_buf[strlen(msg_buf)], " IN SUMMARY MODE.");
      break;
    case 2:
      sprintf (&msg_buf[strlen(msg_buf)], " IN FULL MODE.");
      break;
    case 3:
      sprintf (&msg_buf[strlen(msg_buf)], " IN SUMMARY AND FULL MODE.");
      break;
  }
  Status (msg_buf);
}

static void Parse_skip_input (n)
     int *n;
/* SKIP [n]
 . Ends the current hand prematurely, and skips the next n-1 hands.
 */
{
  int i;
  Board *b;

  if ((n != NULL) && !server_mode) {
    Status ("ONLY THE SERVER MAY SPECIFY A SKIP COUNT.");
    return;
  } else if (VUGRAPH(Local_table)) {
    if (!server_mode) {
      Status ("ONLY THE SERVER MAY SKIP DURING AN EXHIBITION.");
      return;
    }
  } else if (IS_OBSERVER(local_player) && !server_mode) {
    Status ("OBSERVERS MAY NOT USE THE /SKIP COMMAND.");
    return;
  }

  Send_skip (Local_table);

  if (n != NULL) {
    sprintf (msg_buf, "SKIPPED %d HANDS.", *n);
    for (i = 1; i < *n; i++) {
      b = Next_Unplayed_Board ();
      if (b == NULL) {
	sprintf (msg_buf, "ALL UNPLAYED BOARDS HAVE BEEN SKIPPED.");
	break;
      }
      Record_Played_Board (b);
    }
    Moderator_Comment (msg_buf);
  }

}

static void Parse_stats_input (player_name)
     char *player_name;
{
  GPS_Display_Stats (player_name);
}

static void Parse_subcc_input (request)
    char *request ;
/* SUBCC <old> <new>
 . Substitutes <new> for <old> in current cc.
 */
{
  if (request && local_player < 4) {
    char *oldcc = conventions[side_of(local_player)];
    if (oldcc == NULL)
	Status ("YOU HAVE NOT SPECIFIED YOUR CONVENTION CARD YET.");
    else {
      enum { _looking, _found } state = _looking;
      char *old = 0, *new = 0, *src = oldcc, *dst, *newcc, *tok;
      int ntok, all;

      /*
      ** First split the request up into tokens, and make sure
      ** we have exactly two.
      */
      for (tok = strtok(request, " "), ntok = 0; tok; tok = strtok(NULL, " ")) {
	if (++ntok == 1)
	  old = strdup(tok);
	else if (ntok == 2)
	  new = strdup(tok);
	else
	  break;
      }
      if (ntok != 2)
	Status ("USAGE: /SUBCC <OLD> <NEW>");
      else {

	/*
	** New look for the old and replace it with the new
	*/
	dst = newcc = malloc(strlen(oldcc) - strlen(old) + strlen(new) + 1);
	all = strlen(old);

	while (*src) {
	  if (state == _looking && strncasecmp(src, old, all) == 0) {
	    state = _found;
	    src += all;
	    strncpy(dst, new, strlen(new));
	    dst += strlen(new);
	    continue;
	  }
	  *dst++ = *src++;
	}
	*dst = 0;
	if (state == _looking)
	  Status("NO SUCH ITEM ON YOUR CONVENTION CARD.");
	else {
	  free(oldcc);
	  conventions[side_of(local_player)] = newcc;
	  Send_cc (Local_table, newcc);
	}
      }
      if (old) free(old);
      if (new) free(new);
    }
  }
}

/*
static void Parse_table_input (table_no)
     int *table_no;
 * TABLE table-no
 . Requests to switch to the given table.
 *
{
  Send_tablereq (Local_table, *table_no);
}
*/

static void Parse_tables_input ()
/* TABLES
 . Lists the tables which are currently playing.
 */
{
  GPS_List_Tables (1);
}

static void Parse_timer_input (tm)
/* TIMER[ON|OFF|TOGGLE]
 . Sets whether to turn on the timer.  If it is on, it updates it.
 */
     int *tm;
{
  Toggle (&timer_is_on, tm);
  if (timer_is_on)
    Display_timer ();
  else {
    Display_Total_Time ("", "");
    Status ("THE TIMER IS OFF.");
  }
}


static void Parse_update_input (flags)
     char *flags;
{
  char command_buf[100];

  Moderator_Comment ("WARNING!! THIS WILL TERMINATE THE PROGRAM!");
  if (!Ask("ARE YOU SURE YOU WISH TO DO THIS?"))
    return;

  sprintf (command_buf, "csh compile.script %s",
	   (flags == NULL)? "": flags);
  if (GPS_Download("compile.script"))
    return;

  if (server_mode) {
    GPS_End_Server_Mode ();
    server_mode = 0;
  }

  if (save_defaults)
    Write_Initialization_File (".okdefaults");

  clear_screen ();
  Reset_Terminal ();
  Send_quit (Local_table);

  system (command_buf);
  exit (0);
  
}

static void Parse_vugraph_input (v)
     int *v;
{
  int old_mode = Local_table->playing_mode;
  int i;
  Connection c;

  if (v == NULL)
    Set_playing_mode (VUGRAPH_PLAYING_MODE);
  else
    Set_playing_mode (*v? VUGRAPH_PLAYING_MODE: CLUB_PLAYING_MODE);

  if ((Local_table->playing_mode != VUGRAPH_PLAYING_MODE) ||
      (old_mode == VUGRAPH_PLAYING_MODE))
    return;

  FOREACH_CONNECTION (c)
    if (IS_PLAYER(c->seat))
      Request_Seat (c, PLAYER_OBS);

  for (i = 0; i < 4; i++)
    Assign_seat (Local_table, Shadow_Connections[i], i);
}

static void Parse_wakeup_input (player)
     char *player;
{

  if (IS_OBSERVER(local_player) && !server_mode) {
    Status ("THE WAKEUP COMMAND CANNOT BE USED BY AN OBSERVER.");
    return;
  }

  if (FORMAL(Local_table))
    player = "ALL";
  else if (player == NULL) {
    if (!OCCUPIED(Local_table, player_partner[local_player])) {
      Status ("YOU HAVE NO PARTNER TO WAKE UP!");
      return;
    }
    player = PLAYER_NAME(Local_table, player_partner[local_player]);
  }
  
  Send_wakeup (Local_table, player);
  sprintf (msg_buf, "SENDING WAKEUP SIGNAL TO %s ...", player);
  Status (msg_buf);

}

static void Parse_who_input ()
/* WHO
 . Generates a short listing of the players who are currently connected.
 */
{
  Send_who (Local_table);
}

static void Parse_whois_input (p)
     char *p;
/* WHOIS [<player-name>]
 . Asks for the identity of the given player.  If we are sitting at the
 . table, then omitting <player-name> causes both of the opponents to
 . be identified.
 */
{
  int opp1, opp2;

  if ((p == NULL) && (local_player >= 4)) {
    Send_whois (Local_table, "*ALL*");
/*    Status ("ERROR: YOU MUST SPECIFY THE NAME OF A PLAYER"); */
  } else if (p == NULL) {
    opp1 = side_of(player_next[local_player]);
    opp2 = player_partner[opp1];
    if (OCCUPIED(Local_table, opp1))
      Send_whois (Local_table, PLAYER_NAME(Local_table, opp1));
    if (OCCUPIED(Local_table, opp2))
      Send_whois (Local_table, PLAYER_NAME(Local_table, opp2));
  } else
    Send_whois (Local_table, p);
}

static void Parse_ytd_input ()
/* YTD
 . Displays the year-to-date scoreboard.
 */
{
  GPS_Display_YTD_Scoreboard ();
}

static void Parse_zlog_input (f)
     char *f;
/* ZLOG <filename>
 . The ZLOG command opens or closes a file for recording the play in
 . Zhang's format. 
 */
{
  char *filename;

  if (f == NULL) {
    if (zhang_logfile == NULL)
      Moderator_Comment ("THERE IS NO OPEN ZLOGFILE.");
    else {
      fclose (zhang_logfile);
      zhang_logfile = NULL;
      Moderator_Comment ("THE ZLOGFILE HAS BEEN CLOSED.");
    }
    return;
  } else {
    if (zhang_logfile != NULL) fclose (zhang_logfile);
    if (f[0] == '+') {
      filename = f + 1;
      zhang_logfile = fopen (filename, "a");
    } else {
      filename = f;
      zhang_logfile = fopen (filename, "w");
    }
    if (zhang_logfile == NULL) {
      sprintf (msg_buf, "%s ERROR OPENING %s",
	       sys_errlist[errno], filename);
      Moderator_Comment (msg_buf);
    } else {
      sprintf (msg_buf, "NOW ZLOGGING TO %s", filename);
      Moderator_Comment (msg_buf);
    }
  }
}

/***********************************************************************/
/*                                                                     */
/* INSTRUCTIONS FOR MAINTAINING THE COMMAND LIST                       */
/*                                                                     */
/***********************************************************************/

/*

The following table defines the commands which are available in OKbridge.
During normal operation of the program, these commands are entered
by typing 
  /command-name [parameters]

A subset of the commands may also occur in the .okbridgerc and
.okdefaults files.  In this case, they appear without the leading 
virgule '/'.

The entries in the following table must occur in alphabetical order
by command name.  Each entry consists of four parts:
  - the command name, which should be an upper case string
  - a description of the command parameters
  - the command function, a routine to be invoked when this command 
    is parsed
  - a descriptor indicating what information should be stored in the
    .okdefaults file in conjunction with this command.

Each command may have up to three parameters.  Some of the parameters
may be made optional, with the restriction that if one the parameters is
omitted, then all subsequent parameters must be omitted too.
The following types of parameters may be present:
  keyword lists: A list of uppercase keywords separated by vertical
    bars, e.g., "OFF|ON|TOGGLE".  Each keyword is indexed implicitly
    by its order in the list.
  bids and plays: A string naming a bid or play (e.g., "1H, SK")
  integers:   A string of decimal digits.
  names:      A string of up to 8 alphameric characters.
  filenames:  An arbitrary string of characters terminated by whitespace.
  strings:    Everything up to the end of the command line.

For more information on formatting parameter descriptions, see the
file parser.h.  The parser reads a command and its parameters
from an input string.  It then calls the associated command function,
passing the parsed parameters as input.  The command function
should have one parameter in its header for each parameter defined
in the command table.  Each function parameter should be a pointer.
For keywords and integers, a pointer to an integer is passed to
the command function, while in other cases, a string pointer is passed.
Optional parameters that are omitted are passed as NULL pointers.

The fourth field in the command table entry describes information
that may be stored in the .okbridgerc and .okdefaults files.  Some of
the commands may occur as initialization fields in these files.
If the fourth field is a pointer to a variable, then the state of
this variable is recorded in the .okdefaults file when the program
terminates as being the initial setting for that command.  For example,
if the variable bell_is_on is set to 1, then the command "BELL ON"
will be written to the .okdefaults file.  The Write_Command_Variables ()
routine figures out the appropriate keywords and strings to write
to the file based on the rest of the command entry.  Some of the
commands do not represent simple state variables but may nonetheless
be allowed in .okbridgerc and .okdefaults.  For these commands, the
constant RCOK is specified in place of a pointer to a state variable.
This prevents the initialization routines from flagging an error.
For those commands which may not occur as part of the initialization
file, the constant NOS is specified.

The procedure for adding a new command to okbridge consists then of
two steps:

  1. Add a new entry to the command descriptor table describing the
     command and its parameters.  For examples of table entries,
     see below.

  2. Write a command function which will be invoked when the
     command is parsed.  For examples of command functions, see
     the first part of this file.
*/

/* The commands in this table are organized into groups of five to make
   it easier on the eyes. */

static Command_Descriptor Input_Commands [] = {
  {"ADDCC",  C_PARAM(F_STRING, "<new-convention>"), Parse_addcc_input, NOS},
  {"ADDRESS", C_PARAM(F_STRING, "<player-address>"), Parse_address_input, NOS},
  {"ALERT",  C_EMPTY, Parse_alert_input, NOS},
  {"ASSIGN", {{F_NAME, "<player-name>"}, A_SEAT, NOP}, Parse_assign_input, NOS},
  {"ATTEND", C_EMPTY, Parse_attend_input, NOS},

  {"AUTOPASS", C_STATE, Parse_autopass_input, &autopass_mode},
  {"AUTOSAVE",  C_STATE, Parse_autosave_input, RCOK},
  {"BELL",   C_STATE, Parse_bell_input, &bell_is_on},
  {"BIDS", C_EMPTY, Parse_bids_input, NOS},
  {"CC",     C_PARAM(F_KEYWORD|F_OPTIONAL, "NS|EW|MY|BOTH"), 
	Parse_cc_input, NOS},

  {"CCDEF", C_PARAM(F_STRING | F_OPTIONAL, "<cc-def>"), 
     Parse_ccdef_input,RCOK},
  {"CLAIM",  C_PARAM(F_INT | F_OPTIONAL, "<no-tricks>"),
     Parse_claim_input, NOS},
  {"CLS",    C_EMPTY, Parse_cls_input, NOS},
  {"CNTRL",  {{F_FILENAME, "<decimal-keycode>"}, 
    {F_STRING | F_OPTIONAL, "<key-mapping>"}, NOP},  Parse_control_input, RCOK},
  {"CONNECT",{{F_FILENAME | F_OPTIONAL, "<server-name>"},
    {F_INT | F_OPTIONAL, "<port-number>"}, NOP}, Parse_connect_input, NOS},

  {"DEFAULT", C_STATE, Parse_default_input, &default_plays},
  {"DISCONNECT", C_PARAM(F_NAME, "<player-name>"), Parse_disconnect_input, NOS},
  {"DL",     C_PARAM(F_FILENAME | F_OPTIONAL, "<email-file>"), 
     Parse_download_input, NOS},
  {"DOWN", C_PARAM(F_INT, "<no-tricks>"), Parse_down_input, NOS},
  {"DOWNLOAD",C_PARAM(F_FILENAME | F_OPTIONAL, "<email-file>"), 
     Parse_download_input, NOS},

  {"EAST",   C_PARAM(F_STRING | F_OPTIONAL, "<east-command>"), 
     Parse_east_input, NOS},
  {"EMAIL",  C_PARAM (F_STRING, "<email-address>"), Parse_email_input, 
		((int *) &local_player_email)},
  {"EXIT",   C_EMPTY, Parse_exit_input, NOS},
  {"FIND",   C_PARAM(F_STRING, "<player-list>"),Parse_find_input,NOS},
  {"FORMAL", C_STATE, Parse_formal_input, NOS},
  {"FULLNAME", C_PARAM (F_STRING, "<full-name>"), Parse_fullname_input, 
		((int *) &local_player_full_name)},

  {"GPS",    C_STATE, Parse_gps_input, RCOK},
  {"GPS_IP",    {{F_FILENAME, "<GPS-server-IP>"}, 
     {F_INT | F_OPTIONAL, "<GPS-port>"}, NOP}, Parse_GPS_IP_input, NOS},
  {"HELP",   C_PARAM(F_NAME | F_OPTIONAL, "<help-topic-name>"), 
     Parse_help_input, NOS},
  {"HELPFILE",  C_PARAM(F_FILENAME, "<helpfile-name>"), Parse_helpfile_input,
   RCOK},
  {"IGNORE", C_PARAM(F_STRING, "<ignore-string>"), Parse_ignore_input, RCOK},

  {"JOIN",   C_PARAM(F_NAME | F_OPTIONAL, "<name-of-server>"), 
     Parse_join_input, NOS},
  {"KEY", {{F_FILENAME, "<decimal-keycode>"}, 
     {F_STRING | F_OPTIONAL, "<key-mapping>"}, NOP}, Parse_key_input, RCOK},
  {"LEVEL",  C_PARAM(F_STRING, "<playing-level>"), Parse_level_input, NOS},
  {"LHO",    C_PARAM(F_STRING, "<message>"), Parse_lho_input, NOS},
  {"LOAD",   C_PARAM(F_FILENAME, "<board-file>"), Parse_load_input, NOS},

  {"LOG",    {{F_FILENAME | F_OPTIONAL, "<logfile>"}, 
    {F_KEYWORD | F_OPTIONAL, PARSER_LOG}, NOP}, Parse_log_input, RCOK},
  {"LOGIN",  C_PARAM(F_NAME | F_OPTIONAL, "<login-name>"), 
    Parse_login_input, NOS},
  {"LOOK", C_PARAM(F_INT, "<table-number>"), Parse_look_input, NOS},
  {"LREC",   C_EMPTY, Parse_lrec_input, NOS},
  {"LURKER",   C_EMPTY, Parse_lurker_input, NOS},

  {"MAKE",   C_PARAM(F_INT | F_OPTIONAL, "<no-tricks>"), Parse_make_input, NOS},
  {"META", {{F_FILENAME, "<decimal-keycode>"}, 
    {F_STRING | F_OPTIONAL, "<key-mapping>"}, NOP}, Parse_meta_input, RCOK},
  {"NAME", C_PARAM (F_NAME, "<player-name>"), Parse_login_input, 
     ((int *) &local_player_name)},
  {"NORTH",  C_PARAM(F_STRING | F_OPTIONAL, "<north-command>"), 
     Parse_north_input, NOS},
  {"NOTE",   C_PARAM(F_STRING | F_OPTIONAL, "<GPS-message>"), 
     Parse_message_input, NOS},

  {"OBSERVE",C_EMPTY, Parse_observe_input, NOS},
  {"OPP",    C_PARAM(F_STRING, "<message>"), Parse_opp_input, NOS},
  {"PASSWORD", C_EMPTY, Parse_password_input, NOS},
  {"PAUSE",  C_EMPTY, Parse_pause_input, NOS},
  {"PERISH", C_EMPTY, Parse_perish_input, NOS},

  {"PING",   C_EMPTY, Parse_ping_input, NOS},
  {"PLAYERS",C_PARAM(F_NAME | F_OPTIONAL, "<name-of-server>"), 
     Parse_players_input, NOS},
  {"PRACTICE", C_STATE, Parse_practice_input, NOS},
  {"PROMPT", C_PARAM(F_KEYWORD|F_OPTIONAL, "OFF|ON|TOGGLE|NEVER"), 
     Parse_prompt_input, &prompt_dummy},
  {"PUBLISH",C_EMPTY, Parse_publish_input, NOS},

  {"QUIET",  C_STATE, Parse_quiet_input, &quiet_mode},
  {"QUIT",   C_EMPTY, Parse_quit_input, NOS},
  {"REC",    {{F_FILENAME | F_OPTIONAL, "<logfile>"}, 
    {F_KEYWORD | F_OPTIONAL, PARSER_LOG}, NOP}, Parse_rec_input, RCOK},
  {"REMCC", C_PARAM(F_STRING, "<convention>"), Parse_remcc_input, NOS},
  {"REPLAY", C_PARAM(F_FILENAME, "<board-file>"), Parse_replay_input, NOS},

  {"RESET",  C_EMPTY, Parse_reset_input, NOS},
  {"RESULTS", C_PARAM(F_KEYWORD | F_OPTIONAL, "CURRENT"), 
     Parse_results_input, NOS},
  {"REVEAL", C_PARAM(F_NAME | F_OPTIONAL, "<player-name>"),
     Parse_reveal_input, NOS},
  {"REVIEW", C_PARAM(F_KEYWORD | F_OPTIONAL, "CURRENT|PLAY"), 
     Parse_review_input, NOS},
  {"RHO",    C_PARAM(F_STRING, "<message>"), Parse_rho_input, NOS},

  {"SAVE",   C_PARAM(F_FILENAME | F_OPTIONAL, "<board-file>"), 
     Parse_save_input, NOS},
  {"SCORE",  C_PARAM(F_KEYWORD, PARSER_SCORE), Parse_score_input, 
		&scoring_mode},
  {"SCOREBOARD", C_STATE, Parse_scoreboard_input, &display_scoreboard_at_end},
  {"SEAT",   C_SEAT, Parse_seat_input, &local_player},
  {"SERVE",  C_PARAM(F_INT | F_OPTIONAL, "<port-number>"), Parse_serve_input, 
     NOS},

  {"SETCC",  C_PARAM(F_STRING | F_OPTIONAL, "<convention-card>"), 
     Parse_setcc_input, NOS},
  {"SHOWALERTS", C_STATE, Parse_showalerts_input, &show_alerts},
  {"SHOWRESULTS", C_PARAM(F_KEYWORD, "OFF|SUMMARY|FULL|ON"), 
     Parse_showresults_input, &show_results},
  {"SKIP",   C_PARAM(F_INT | F_OPTIONAL, "<skip-count>"), Parse_skip_input, 
     NOS},
  {"SOUTH",  C_PARAM(F_STRING | F_OPTIONAL, "<south-command>"),
     Parse_south_input, NOS},

  {"SPEC",   C_PARAM(F_NAME | F_OPTIONAL, "<player-name>"),
     Parse_reveal_input, NOS},
  {"STATS",  C_PARAM(F_NAME | F_OPTIONAL, "<player-name>"), Parse_stats_input, 
     NOS},
  {"SUBCC",  C_PARAM(F_STRING, "<old-convention> <new-convention>"), 
		Parse_subcc_input, NOS},
  {"SUMMARY", C_PARAM(F_KEYWORD | F_OPTIONAL, "CURRENT"), 
     Parse_summary_input, NOS},
  {"TABLES", C_EMPTY, Parse_tables_input, NOS},

  {"TIMER",  C_STATE, Parse_timer_input, &timer_is_on},
  {"UPDATE", C_PARAM(F_STRING | F_OPTIONAL, "<compile-flags>"), 
     Parse_update_input, NOS},
  {"VUGRAPH",C_STATE, Parse_vugraph_input, NOS},
  {"WAKEUP", C_PARAM(F_NAME | F_OPTIONAL, "<player-name> | ALL"), 
     Parse_wakeup_input, NOS},
  {"WEST",   C_PARAM(F_STRING | F_OPTIONAL, "<west-command>"),
     Parse_west_input, NOS},

  {"WHO",    C_EMPTY, Parse_who_input, NOS},
  {"WHOIS",  C_PARAM(F_NAME | F_OPTIONAL, "<player-name>"), Parse_whois_input, NOS},
  {"YTD",    C_EMPTY, Parse_ytd_input, NOS},
  {"ZLOG",   C_PARAM(F_FILENAME | F_OPTIONAL, "<logfile>"), Parse_zlog_input, RCOK},
  END_PARSE_TABLE
};
#define COMMAND_TABLE_SIZE ((sizeof(Input_Commands)/sizeof(Command_Descriptor))-1)


int Parse_Server_Command (command_buf)
     char *command_buf;
/* On entry, command_buf should contain a special request to the server
   made by a client.  Parses the command and executes it.  Returns 0 if
   the command was parsed correctly, and 1 if an error occurs.  In the
   latter case, the error message is placed in the Parser_Error_Buf.
*/
{
/*  return (Parse_Command (Input_Commands, INPUT_MAX-1, command_buf)); */
  return (0);
}

int Command_Parse (line)
char *line;
{
  int status;

  never_prompt_save = never_prompt;
  never_prompt = 0;
  status = Parse_Command(Input_Commands, COMMAND_TABLE_SIZE, line);
  never_prompt = never_prompt_save;
  return status;
}

int Parse_Initialization_Command (line)
     char *line;
{
  Command_Descriptor *c = 
    Search_for_Command (Input_Commands, COMMAND_TABLE_SIZE, line);

  if ((c != NULL) && (c->save_field == NULL)) {
    sprintf (Parser_Error_Buf, 
	     "THIS COMMAND MAY NOT BE USED IN THE INITIALIZATION FILE.");
    return (1);
  } else
    return (Parse_Command(Input_Commands, COMMAND_TABLE_SIZE, line));
}

void Parse_Input_Command (command_buf)
     char *command_buf;
/* Parses the command contained in the buffer command_buf and executes
   the corresponding command.  If an error occurs, then displays an
   appropriate message in the status line. 
*/
{
  char copied_command_buf [100];
  
  strncpy (copied_command_buf, command_buf + 1, 100);
  copied_command_buf[99] = '\0';

  Clear_Focus_Buffer ();

  if (Command_Parse(copied_command_buf))
    Status (Parser_Error_Buf);
}


void Write_Command_Variables(fp)
FILE *fp;
{
  int i;
  int word;
  int pos;
  int ftype;

  for (i = 0; Input_Commands[i].command_name != NULL; i++) {
    if ((Input_Commands[i].save_field != NULL) &&
	(Input_Commands[i].save_field != RCOK)) {
      ftype = Input_Commands[i].fields[0].field_type | F_OPTIONAL;
      ftype = ftype ^ F_OPTIONAL;
      switch (ftype) {
        case F_STRING:
        case F_NAME:
        case F_FILENAME:
          if (Input_Commands[i].save_field != NULL)
            fprintf (fp, "%s %s\n", Input_Commands[i].command_name, 
		    (char *) (*(Input_Commands[i].save_field)));
          break;
        case F_KEYWORD:
          fprintf (fp, "%s ", Input_Commands[i].command_name);
          pos = 0;
          for (word = 0; word < *(Input_Commands[i].save_field); word++) {
	    while (Input_Commands[i].fields[0].field_descriptor[pos] != '|')
	      pos++;
	    pos++;
          }
          /* now pos points to the first character to be written */
          while ((Input_Commands[i].fields[0].field_descriptor[pos] != '|') &&
      	         (Input_Commands[i].fields[0].field_descriptor[pos] != '\0')) {
	    fprintf(fp,"%c", Input_Commands[i].fields[0].field_descriptor[pos]);
	    pos++;
          }
          fprintf (fp, "\n");
          break;
      }
    }
  }

  if ((logfile != NULL) || (logfile_email != NULL))
    fprintf (fp, "LOG %s %s\n", logfile_name,
	     (logfile == NULL)? "EMAIL":
	     (logfile_email == NULL)? "": "BOTH");

  if ((recfile != NULL) || (recfile_email != NULL))
    fprintf (fp, "REC %s %s\n", recfile_name,
	     (recfile == NULL)? "EMAIL":
	     (recfile_email == NULL)? "": "BOTH");
}

static int Command_Error (s)
char *s;
{
  return (Parse_Command_Dont_Execute(Input_Commands, COMMAND_TABLE_SIZE, s));
}






