/*
 * ccfile.c  Routines to support the file of convention cards for use with
 * okbridge.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This module written by Jonathan Segal, and is donated to the okbridge
 * elder (i.e. Matt Clegg) to do with as he will.
 *
 *
 * This is quite simple, actually- it is a simple name-value pair table.
 *
 * Since this is fused onto an existing system, it uses global variables.
 *  
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
/* #include <malloc.h> */

extern char *strdup ();
extern void free ();
extern char *realloc (), *malloc ();

#define	MAX_LENGTH	256

typedef struct cc_el
{
  char *ccname;
  char *ccvalue;
} cc_el;

#define NEW_CC_BLOCK_SIZE  5

/*
  The convention card table is a NULL-terminated array of cc_els
  */

static cc_el *cc = 0;                                  /* base table pointer */
static int max_num_ccs = 0;                /* referent for memory allocation */
static int num_ccs = 0;                        /* current number of used ccs */


/*
  forward declarations */
void Free_CCs(/*_ void _*/);
static int Get_Some_CCs(/*_ void _*/);
static cc_el *Find_CC(/*_ char *name _*/);
static int Add_CC_Definition(/*_ char *n, char *v _*/);
static void Display_CC(/*_ char *ccname, void (*display_cc)() _*/);

/*------------------------------Get_CC_Value---------------------------------*/
char *Get_CC_Value(/*_ char *c _*/);
/*
  This functions is passed a cc candidate.  If this string only contains one
  word, and this word is a name in the cc table, the value of the cc will be
  returned (note the value is from the table, so may need to be strduped.
  Otherwise, it will simply return its parameter.

*/

char *Get_CC_Value( c )
char *c;
{
  char *n;
  cc_el *cp;

  for (n = c; *n && !isspace(*n); ++n);                 /* check if one word */
  if (*n)                                              /* more than one word */
    return (c);
  
  cp = Find_CC(c);

  if (!cp)                                             /* named CC not found */
  {
    return (c);
  }
  
  else
  {
    return (cp->ccvalue);
  }
}

/*---------------------------------Define_CC---------------------------------*/
int Define_CC(/*_ char *ccdef, void (*display_cc)() _*/);
/*
  this will parse the ccdef string -- if the
*/

int Define_CC( ccdef, display_cc )
char *ccdef;
void (*display_cc)();
{
  char *n, *v;
  char buf[MAX_LENGTH];
  
  for (n = ccdef; *n && isspace(*n); ++n)
    ;                                             /* skip to first non-white */
  if (!*n)
  {
    return (1);
  }

  for(v = n; *v && !isspace(*v) ; ++v)
    ;                                               /* skip to first white */
  if (!*v)
  {
    Display_CC(n, display_cc);
    return(0);
  }
  
  *v = '\0';                                        /* null terminate name */
  for (++v; *v && isspace(*v); ++v)
    ;
  if (!*v)
  {
    Display_CC(n, display_cc);
    return (0);
  }
  if (Add_CC_Definition(n,v))
  {
    return (1);
  }
  else
  {
    if (display_cc)
    {
      sprintf(buf,"CC defined: %s",n);
      (*display_cc)(buf);
    }
    return (0);
  }
}

/*--------------------------------Display_CC---------------------------------*/
/*
  Display the CC with the specified name
*/

static void Display_CC( n, display_cc )
char  *n;
void (*display_cc)();
{
  cc_el *cp;
  char linebuf[MAX_LENGTH];

  if (!display_cc)
    return;                           /* do nothing with no display function */
  /* display the cc */
  if (cp = Find_CC(n))
  {
    sprintf(linebuf,"%s = %s",cp->ccname, cp->ccvalue);
    (*display_cc)(linebuf);
  }
  else
  {
    sprintf(linebuf, "%s IS UNDEFINED AS A CONVENTION CARD", n);
    (*display_cc)(linebuf);
  }
}    

/*-----------------------------Add_CC_Definition-----------------------------*/
/*
  Add a new cc definition to the CC table
*/

static int Add_CC_Definition( n, v )
char *n;
char *v;
{
  cc_el *cp;
  /* first see if we have a redefinition */

  if (cp = Find_CC(n))
  {
    char *old_val = cp->ccvalue;
    if (!(cp->ccvalue = strdup(v)))
    {
      cp->ccvalue = old_val;
      return (1);
    }
    free(old_val);
    return (0);
  }

  /* we're adding a new one */
  if (num_ccs >= max_num_ccs &&  Get_Some_CCs())
  {
    return (1);                                /* we're out of room for more */
  }
    
  /* now n is pointing to name, v is pointing to value */
  
  if (!(cc[num_ccs].ccname = strdup(n)))
    return (1);                                             /* no more room! */
  if (!(cc[num_ccs].ccvalue = strdup(v)))
  {
    free(cc[num_ccs].ccname);                               /* no more room! */
    cc[num_ccs].ccname = (char *) NULL;
    return (1);
  }
  ++num_ccs;
  return (0);
}

/*------------------------------Display_All_CCs------------------------------*/
void Display_All_CCs(/*_ void (*display)() _*/);
/*
Enter function description here
*/

void Display_All_CCs( display )
void (*display)();
{
  cc_el *cp;
  char   linebuf[MAX_LENGTH];

  if (!cc)
  {
    (*display) ("NO NAMED CONVENTION CARDS HAVE BEEN DEFINED.");
    return;
  }
  for (cp = cc; cp->ccname; ++cp)
  {
    sprintf(linebuf,"%s = %s",cp->ccname, cp->ccvalue);
    (*display)(linebuf);
  }
}



/*---------------------------------Free_CCs----------------------------------*/
/*
  Empty the CC table, freeing the memory each CC.
*/

void Free_CCs()
{
  cc_el *cp = cc;

  if (!cc)
    return;
  while (cp->ccname)
  {
    free(cp->ccname);
    free(cp->ccvalue);
  }

  cc->ccname = cc->ccvalue = (char *) NULL;
  num_ccs = 0;
  
}

/*-------------------------------Get_Some_CCs--------------------------------*/
/*
  Increase the CC table size.  Returns 1 on success, 0 on failure.
*/

static int Get_Some_CCs()
{
  cc_el *newcc;

  max_num_ccs += NEW_CC_BLOCK_SIZE;

  if (cc)
  {
    newcc = (cc_el *) realloc(cc, sizeof(cc_el) * (max_num_ccs + 1));
                                         /* add one for the null termination */
  }
  else
  {
    newcc = (cc_el *) malloc(sizeof(cc_el) * (max_num_ccs + 1));
    bzero (newcc, sizeof(cc_el) * (max_num_ccs + 1));
  }
  if (!newcc)
  {
    max_num_ccs -= NEW_CC_BLOCK_SIZE;
    return (1);
  }
  else
  {
    cc = newcc;
    return (0);
  }
}

/*----------------------------------Find_CC----------------------------------*/
/*
  Search the cc table for a CC named name

  return the element found, NULL if none.
*/

static cc_el *Find_CC( name )
char *name;
{
  cc_el *cp;
  if (!cc)
  {
    return (NULL);
  }
  
  for (cp = cc; cp->ccname; ++cp)
  {
    if (!strcasecmp(cp->ccname, name))
    {
      /* a match! */
      return (cp);
    }
  }
  /* no match */
  return (NULL);
  
}

