/* bridge.c
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This file contains an implementation of a computer-moderated
 * bridge program.  The program serves as a mediator allowing
 * four people on separate systems to cooperate in a bridge game.
 * The game is screen oriented and interactive, although the display
 * is very simple.  The program handles the shuffling and dealing
 * of the cards, and it prompts each player for input when it is
 * his turn.  The lower area of the screen is a dialog box, where
 * players may enter comments that will be seen by the others.
 *
 *
 */
 
#define _BRIDGE_
 
#include <sys/time.h>
#ifndef VMS
#include <sys/resource.h>
#endif
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#ifdef AIX
#include <time.h>
#endif
#include <string.h>

#include "state.h"
#include "parser.h"
#include "display.h"
#include "input.h"
#include "terminal.h"
#include "conversation.h"

#include "cs.h"
#include "help.h"
#include "gps.h"
#include "rc.h"
#include "log.h"
#include "ccdef.h"

#define  VISIBLE(table) 1
extern int errno;
extern char *sys_errlist[];

extern int  atoi ();
extern void exit ();
extern char *getenv ();
extern getpid ();
extern void srand ();


#ifdef DEBUG
extern getrlimit (), setrlimit ();
#endif

extern Begin_timer_for_hand ();
extern Begin_timer_for_play ();
extern Clear_Plays_from_Table ();
extern Conclude_GPS_request ();
extern Define_CC ();
extern Display_Bid_on_Table ();
extern Display_Final_Contract ();
extern Display_Play_on_Table ();
extern Display_timer ();
extern End_timer_for_hand ();
extern End_timer_for_play ();
extern GPS_Display_Scoreboard ();


extern int network_port;
extern int server_mode;
extern int claim_in_progress;
extern int pause_mode;
extern int exit_requested;
extern int sort_direction;
extern int new_score_mode;
extern char *help_file_name;
extern int result_mode;

typedef  void (*signal_handler) ();

int server_mode_requested = 0; /* TRUE if user has requested server mode. */
int client_mode_requested = 0; /* TRUE if user has requested client mode. */
 
int save_defaults = 1;         /* TRUE if we should save some of the 
				  flags controlling the program in a 
				  defaults file. */

long startup_time;             /* The time in seconds at which the program
				  was first started. */

int  display_scoreboard_at_end = 1;
                               /* TRUE if we should display the GPS scoreboard
				  when the program ends. */

int  duplicate_board_in_use = 0;
                               /* TRUE if the board which we are currently
				  playing is a GPS duplicate board. */

int show_alerts = 1;           /* TRUE indicates that the player will see the
				  alerts made by partner. */

int show_results = 3;          /* 1 indicates that a summary of the results
				  of the other boards should be displayed
				  after each hand, 2 full results, 3 both */

#ifdef LOGFILE
  extern FILE *net_log;
#endif

extern char *strdup ();
extern void Broadcast_Comment ();

extern char *autoload_file, *autosave_file;

FILE *logfile = NULL;  /* The file where we log the hands that have
			. been played. */
FILE *logfile_email = NULL;
                       /* This is a log file for recording hands in
			  email duplicate format. */

FILE *recfile = NULL;  /* This is a log file where hands are only
			  recorded on request. */
FILE *recfile_email = NULL;
                       /* A recfile where hands are recorded in email
			  duplicate format. */

int record_this_hand = 0;
                       /* A boolean flag indicating that this hand
			  should be recorded after it is finished. */

FILE *zhang_logfile = NULL; 
                       /* The file where we log hands that are written
			  in Zhang's format. */
int table_no = -1;     /* Table number printed in Szhang format log files. */


hand saved_hands [4];  /* The cards as they were dealt. */
hand saved_plays [4];  /* The cards as they were played. */
int  leading_play [14];/* The leading player for each trick. */

static int prompt_before_hand = 1;
        /* A flag which indicates that the server should be given
           the opportunity to delay the beginning of the hand. */

static jmp_buf error_return;
static jmp_buf reset_buf;

static int     abort_flag;
 
#ifndef VMS
void core_dump ()
{
  kill (getpid(), SIGTRAP);
}
#endif

int String_is_alphameric (s)
     char *s;
/* Examines the string s to see if every character is alphameric, i.e.,
   taken from the set A-Za-z0-9_.-
*/
{
  for (; *s; s++)
    if (!isalnum(*s) && (*s != '.') && (*s != '-') && (*s != '_'))
      return (0);

  return (1);
}

int bidding_code (bid_type, level, suit)
    int bid_type, level, suit;
/* Given the bidding type, as above, and the level and suit if it is
 * a contract bid, returns a unique integer in the range 0-37 associated
 * with the bid.  Contract bids are ordered in increasing order, so that
 * 1C is the smallest and 7NT is the largest.
 */
{
  switch (bid_type) {
  case BID_PASS:     return (BID_PASS);
  case BID_DOUBLE:   return (BID_DOUBLE);
  case BID_REDOUBLE: return (BID_REDOUBLE);
  case BID_SUIT:     return (5 * (level - 1) + suit + 3);
  default:	   ;
  }
  return (-1);
}

static void Initiate_Skip_Mode (t)
     Table t;
{
  int i;

  Send_skipack (t);
  Clear_Spectators (t);

  if (!server_mode) {
    t->game_mode = STARTUP_MODE;
    return;
  }

  for (i = 0; i < 4; i++)
    if (OCCUPIED(t, i))
      t->Seats[i].skipping = 1;

}

static int Handle_event_in_Skip_Mode (t, c)
     Table t;
     player_command c;
{
  int i;

  switch (c->command) {
  case CMD_SKIPACK:
    t->Seats[c->player_no].skipping = 0;
    for (i = 0; i < 4; i++)
      if (OCCUPIED(t, i) && t->Seats[i].skipping)
	return (0);
    t->game_mode = STARTUP_MODE;
    return (1);
    break;

  default:
    break;
  }
  return (0);
}

static void Initiate_Deal_Mode_if_Appropriate (t)
     Table t;
{
  int i, n;
  int ready_to_deal = 0;

  if (!server_mode)
    return;

  for (i = n = 0; i < 4; i++)
    if (t->Seats[i].occupied)
      n++;

  if (VUGRAPH(t))
    ready_to_deal = 1;
  else if ((t->playing_mode == PRACTICE_PLAYING_MODE) && (n > 0))
    ready_to_deal = 1;
  else if (n == 4)
    ready_to_deal = 1;
  else 
    ready_to_deal = 0;

  if (ready_to_deal && VISIBLE(t)) {
    if (prompt_before_hand) {
      Pause
	( (hands_played == 0)?
	 "PRESS <ESC> TO DEAL THE FIRST HAND ..." :
	 "PRESS <ESC> FOR THE NEXT HAND ..."   );

      for (i = n = 0; i < 4; i++)
	if (t->Seats[i].occupied)
	  n++;
    }

    if (VUGRAPH(t))
      ready_to_deal = 1;
    else if ((t->playing_mode == PRACTICE_PLAYING_MODE) && (n > 0))
      ready_to_deal = 1;
    else if (n == 4)
      ready_to_deal = 1;
    else 
      ready_to_deal = 0;

    prompt_before_hand = prompt_before_hand && !ready_to_deal &&
      IS_PLAYER(local_player) && prompt_dummy;

  } else if (!ready_to_deal)
    prompt_before_hand = IS_PLAYER(local_player) && prompt_dummy;

  if (ready_to_deal)
    Send_deal (t);
}

static void Sequencing_Error (s)
     char *s;
{
  char buf[100];

  sprintf (buf, "SEQUENCING ERROR: %s", s);
  Moderator_Comment (buf);
}

static void Initiate_Startup_Mode (t)
     Table t;
/* Perform setup processing when we first enter startup mode. */
{
  if (new_score_mode != -1) {
    scoring_mode = t->scoring_mode = new_score_mode;
    bzero(&t->score[SIDE_NS], sizeof(Score));
    bzero(&t->score[SIDE_EW], sizeof(Score));
    bzero(&local_player_score, sizeof(Score));
    new_score_mode = -1;
    Send_skip (t);
  }
  t->game_mode = STARTUP_MODE;
  Local_Player_Connection->spectator = 0;
  spectator_mode = 0;
  Clear_Spectators (t);
  Initiate_Deal_Mode_if_Appropriate (t);
}

static int Handle_event_in_Startup_Mode (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
/* Handles an event while we are in startup mode.  Returns 0 if the event
   causes us to remain in startup mode, and 1 if we advance to bidding mode.
*/
{
  switch (c->command) {

  case CMD_HELLO:
  case CMD_SEAT:
  case CMD_ACK:
  case CMD_SEATPOS:
    Initiate_Deal_Mode_if_Appropriate (t);
    return (t->game_mode != STARTUP_MODE);
    break;

  case CMD_DEAL:
    t->game_mode = DEALING_MODE;
    return (1);
    break;

  default:
    break;
  }
  return (0);
}

static void Initiate_Deal_Mode (t)
     Table t;
/* Perform setup processing when we enter deal mode. */
{
  Board *b;
  Play_record *p;
  int gps_duplicate_mode;


  if (!server_mode) {
    t->board = NULL;
    t->play_record = NULL;
    return;
  }

  if ((Prev_board != NULL) && (Prev_board->scoring_mode != t->scoring_mode)) {
    bzero(&t->score[SIDE_NS], sizeof(Score));
    bzero(&t->score[SIDE_EW], sizeof(Score));
    bzero(&local_player_score, sizeof(Score));
    bzero(&local_player_total_score, sizeof(Score));
  }

  /* If we are the server, then it is our responsibility to shuffle
     and deal the cards. */
  b = NULL;
  duplicate_board_in_use = 0;
  gps_duplicate_mode = !(PRACTICE(t) || VUGRAPH(t) 
			 || !Use_GPS || GPS_unavailable
			 || Board_is_Available ());

  if (gps_duplicate_mode) {
    switch (t->scoring_mode) {
    case RUBBER_SCORING:
    case DUPLICATE_SCORING:
      break;

    case IMP_SCORING:
      b = GPS_Get_Next_Board ("IMP");
      if (b != NULL)
	Broadcast_Comment ("WE ARE PLAYING IMP DUPLICATE BRIDGE.");
      break;

    case MP_SCORING:
      b = GPS_Get_Next_Board ("MP");
      if (b != NULL)
	Broadcast_Comment ("WE ARE PLAYING MATCH POINT DUPLICATE BRIDGE.");
      break;
    }
    duplicate_board_in_use = (b != NULL);
  }
  
  if (b == NULL) {
    if (Board_is_Available())
      b = Next_Unplayed_Board();
    else if (scoring_mode == RUBBER_SCORING) {
      if ((t->score[0].rubber.below_line >= 100) || 
	  (t->score[1].rubber.below_line >= 100)) {
	t->score[0].rubber.above_line += t->score[0].rubber.below_line;
	t->score[1].rubber.above_line += t->score[1].rubber.below_line;
	t->score[0].rubber.below_line = t->score[1].rubber.below_line = 0;
      }
      if ((t->board != NULL) && (t->board->scoring_mode == RUBBER_SCORING))
	b = Generate_Random_Rubber_Board (t->board, t->play_record);
      else
	b = Generate_Random_Rubber_Board (NULL, NULL);
    } else {
      b = Generate_Random_Match_Board (scoring_mode, t->board);
    }
  }

  Send_board (t, b);
  for (p = b->play_records; p != NULL; p = p->next)
    Send_record (t, b, p);

  t->play_record = Locate_play_record_by_foursome
    (b, PLAYER_NAME(t,PLAYER_NORTH), PLAYER_NAME(t, PLAYER_EAST),
     PLAYER_NAME(t,PLAYER_SOUTH),  PLAYER_NAME(t, PLAYER_WEST));
  if ((p = t->play_record) != NULL) {
    Send_userec (t, 
		 p->player_names[PLAYER_NORTH], p->player_names[PLAYER_EAST], 
		 p->player_names[PLAYER_SOUTH], p->player_names[PLAYER_WEST]);
  }

  Send_score (t, &t->score[SIDE_NS], &t->score[SIDE_EW]);
  Send_mode  (t, t->playing_mode);
  Send_begin (t);
  t->play_request_sequence_number = 0;
}

static int Handle_event_in_Deal_Mode (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
/* Handles an event in deal mode.  Returns 0 if the event causes us to
   remain in deal mode, and 1 otherwise.  If we exit deal mode, then
   the board and play record to be used are stored in the corresponding
   fields of the table record t.
*/
{
  switch (c->command) {
  case CMD_BEGIN:
    t->game_mode = BIDDING_MODE;
    if (t->play_record == NULL) {
      t->play_record = Allocate_play_record 
	(PLAYER_NAME(t,PLAYER_NORTH), PLAYER_NAME(t, PLAYER_EAST),
	 PLAYER_NAME(t,PLAYER_SOUTH),  PLAYER_NAME(t, PLAYER_WEST));
      Append_play_record (b, t->play_record);
    }

    t->play_record->next_player = b->dealer;
    Compute_contract (t->board, t->play_record);
    if (t->play_record->hand_completed)
      t->game_mode = SCORING_MODE;
    else if (t->play_record->bidding_completed)
      t->game_mode = PLAYING_MODE;
    return (1);
    break;

  case CMD_BOARD:
    t->board = c->data.board.record;
    t->play_record = NULL;
    if ((Prev_board != NULL) && 
	(Prev_board->scoring_mode != t->board->scoring_mode)) {
      bzero(&Local_table->score[SIDE_NS], sizeof(Score));
      bzero(&Local_table->score[SIDE_EW], sizeof(Score));
      bzero(&local_player_score, sizeof(Score));
      bzero(&local_player_total_score, sizeof(Score));
    }
    break;

  case CMD_MODE:
    t->playing_mode = c->data.mode;
    break;

  case CMD_RECORD:
    p = c->data.record.play;
    if (p->bidding_completed)
      Compute_contract (t->board, p);
    if (p->hand_completed)
      Compute_Scores_for_Play_Record (t->board, p);
    if (!server_mode)
      Append_play_record (t->board, c->data.record.play);
    break;

  case CMD_SCORE:
    bcopy (&c->data.score.ns, &t->score[SIDE_NS], sizeof(Score));
    bcopy (&c->data.score.ew, &t->score[SIDE_EW], sizeof(Score));
    t->scoring_mode =  c->data.score.score_type;
    break;

  case CMD_USEREC:
    t->play_record = Locate_play_record_by_foursome 
      (t->board, c->data.userec.north, c->data.userec.east,
       c->data.userec.south, c->data.userec.west);
    break;

  default:
    return (0);
    break;
  }
  return (0);
}

static int Handle_event_in_Bidding_Mode (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
/* Handles an event while we are in bidding mode.  Returns 0 if the event
   causes us to remain in bidding mode, and 1 otherwise.  If we exit
   bidding mode, then the final contract is recorded in the play record.
   If the contract is a game contract, then the game mode is advanced to
   playing mode.  Otherwise, the mode is advanced to scoring mode.
*/
{
  int n = p->no_bids;
  char message_buf[80];

  if (c->command == CMD_ALERT) {
    if (p->no_bids <= c->data.alert.seq_no) {
      Sequencing_Error ("Illegal Alert Received");
      return (0);
    }

    p->alerts [c->data.alert.seq_no] |= 1 << c->data.alert.who;

    if ((p->no_bids != c->data.alert.seq_no + 1) && !PRACTICE(t)) {
      Status ("NOTE:  THE BIDDING HAS BEEN UNWOUND.");
      while (p->no_bids != c->data.alert.seq_no + 1) {
	p->no_bids--;
	p->next_player = player_prev[p->next_player];
      }
    }

    Compute_contract (b, p);
    Refresh_Playing_Area ();
    return (0);
  }

  if (c->command != CMD_BID) {
    return (0);
  } else if (!VUGRAPH(t) && (c->data.bid.seq_no != p->no_bids)) {
    sprintf (message_buf, "BID %s received from %s (%d).",
	     bid_names[c->data.bid.level], c->player_name, c->player_no);
    Sequencing_Error (message_buf);
    return (0);
  }

  p->bids[p->no_bids] = c->data.bid.level;
  if (c->data.bid.alert) 
    p->alerts[p->no_bids] = 1 << ALERT_BY_BIDDER;
  else
    p->alerts[p->no_bids] = 0;
  p->no_bids++;
  Compute_contract (b, p);
  p->next_player = player_next[p->next_player];
  if (p->no_bids < 4)
    return (0);
  else if ((p->bids[n-2] != BID_PASS) || (p->bids[n-1] != BID_PASS) ||
	   (p->bids[n] != BID_PASS))
    return (0);

  p->bidding_completed = 1;
  if (p->contract == BID_PASS) {
    t->game_mode = SCORING_MODE;
    p->hand_completed = 1;
  } else
    t->game_mode = PLAYING_MODE;
  p->next_player = player_next[p->declarer];
  return (1);
}

static void Record_Player_Names (t, b, p)
     Table t; Board *b; Play_record *p;
/* Records the names of the players at the local table into the play
   record p.  
 */
{
  int i;

  for (i = 0; i < 4; i++) {
    if (p->player_names[i] != NULL)
      free (p->player_names[i]);
    p->player_names[i] = strdup(PLAYER_NAME(t,i));
  }
}

static void Compute_Result (p)
     Play_record *p;
/* Uses the tricks[] and contract fields of the play record to compute the
   result of play.
*/
{
  p->hand_completed = 1;
  if (p->tricks[side_of(p->declarer)] >= 6 + p->contract)
    p->result = p->tricks[side_of(p->declarer)] - 6;
  else
    p->result = p->tricks[side_of(p->declarer)] - p->contract - 6;
}

static int Handle_event_in_Playing_Mode (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
/* Handles an event while we are in playing mode.  Returns 0 if the event
   causes us to remain in playing mode, and 1 otherwise.  If we exit
   playing mode, then the result of play is recorded in the play record.
*/
{
  int n, w, card;
  char buf[80];

  if (c->command == CMD_CLAIM) {
    if (c->data.claim.no_tricks == p->tricks[side_of(p->declarer)])
      Moderator_Comment ("THE DECLARER HAS CONCEDED THE REMAINING TRICKS.");
    p->tricks[side_of(p->declarer)] = c->data.claim.no_tricks;
    p->tricks[1 - side_of(p->declarer)] = 
      13 - p->tricks[side_of(p->declarer)];
    Compute_Result (p);
    t->game_mode = SCORING_MODE;
    Display_Tricks_Taken ();
    return (1);
  }

  if (c->command != CMD_PLAY)
    return (0);

  if (c->data.play.seq_no != p->no_plays) {
    if (c->data.play.seq_no < p->no_plays)
      sprintf (buf, "WARNING! DUPLICATE (%d, %d) PLAY (%s) RECEIVED FROM %s.",
	       c->data.play.seq_no, p->no_plays,
	       card_names[c->data.play.card], c->player_name);
    else
      sprintf (buf, 
	       "WARNING! OUT OF ORDER (%d, %d) PLAY (%s) RECEIVED FROM %s.",
	       c->data.play.seq_no, p->no_plays,
	       card_names[c->data.play.card], c->player_name);
    Moderator_Comment (buf);
    return (0);
  }

  card = c->data.play.card;
  pause_mode = 0;
  p->play_list[p->no_plays++] = card;
  p->next_player = player_next[p->next_player];
  if (p->no_plays % 4 == 0) {
    n = p->no_plays - 4;
    w = Winning_card (trumpsuit_of(p->contract), p->next_player,
		      p->play_list[n], p->play_list[n+1], 
		      p->play_list[n+2], p->play_list[n+3]);
    p->tricks[side_of(w)] += 1;
    p->next_player = w;
    Display_Tricks_Taken ();
  }

  if (p->no_plays == 52) {
    Compute_Result (p);
    t->game_mode = SCORING_MODE;
    return (1);
  }

  return (0);
}

static void Initiate_Scoring_Mode (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
/* Initiates scoring mode for the current board.  This consists of computing
   the score for the board based upon scoring mode and result.
*/
{
  spectator_mode = 0;
  if (server_mode)
    Clear_Spectators (t);

  if (Local_table->playing_mode != PRACTICE_PLAYING_MODE)
    hands_played += 1;

  Compute_Scores_for_Play_Record (b, p);
  Compute_Scores_for_Board (b);

  Record_Played_Board (b);
  Prev_board = Local_board;
  Prev_play  = Local_play;

  if (IS_PLAYER(local_player)) {
    bcopy (&p->score[side_of(local_player)], &local_player_score, 
	   sizeof(Score));
    if (b->scoring_mode == MP_SCORING)
      Add_scores (MP_SCORING, &local_player_mp_total, &local_player_score);
    else if (b->scoring_mode == IMP_SCORING) 
      Add_scores (IMP_SCORING, &local_player_imp_total, &local_player_score);
    else if (b->scoring_mode == RUBBER_SCORING) {
      Add_scores (RUBBER_SCORING, &local_player_total_score, 
		  &local_player_score);
      if (local_player_total_score.rubber.below_line >= 100) {
	local_player_total_score.rubber.above_line +=
	  local_player_total_score.rubber.below_line;
	local_player_total_score.rubber.below_line = 0;
      }
    } else if (b->scoring_mode == DUPLICATE_SCORING)
      Add_scores (DUPLICATE_SCORING, &local_player_total_score, 
		  &local_player_score);
  }

  Add_scores (b->scoring_mode, &t->score[0], &p->score[0]);
  Add_scores (b->scoring_mode, &t->score[1], &p->score[1]);
  
  if (!server_mode)
    return;

  Send_score (t, &t->score[SIDE_NS], &t->score[SIDE_EW]);
  Send_end (t);
  
}

static int Handle_event_in_Scoring_Mode (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
/* Handles an event while we are in scoring mode.  Returns 0 if the
   event causes us to remain in scoring mode, and 1 otherwise.
*/
{
  switch (c->command) {
  case CMD_END:
    t->game_mode = STARTUP_MODE;
    return (1);

  case CMD_SCORE:
    bcopy (&c->data.score.ns, &t->score[SIDE_NS], sizeof(Score));
    bcopy (&c->data.score.ew, &t->score[SIDE_EW], sizeof(Score));
    t->scoring_mode =  c->data.score.score_type;
    break;

  case CMD_RECORD:
    if (!server_mode) {
      Compute_Scores_for_Play_Record (t->board, c->data.record.play);
      Append_play_record (t->board, c->data.record.play);
    }
    break;

  default:
    break;
  }
  return (0);
}

static void Setup_Startup_Display ()
{
/*
  Set_Display_Mode (TALK_DISPLAY);
  Set_Input_Mode (TALK_INPUT);
*/
  Begin_timer_for_hand ();
}

static void Display_event_in_Startup_Mode 
  (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
{
  ;
}

static void Setup_Deal_Display ()
{
  ;
}

static void Display_event_in_Deal_Mode
  (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
{
  int i;

  switch (c->command) {

  case CMD_BEGIN:
    Local_table = t;
    Local_board = t->board;
    Local_play  = t->play_record;

    spectator_mode = 0;
    for (i = 0; i < 4; i++)
      revealed_hands[i] = 0;

    if (Local_play->bidding_completed) {
      Set_Display_Mode (BIDDING_DISPLAY);
      Set_Input_Mode (TALK_INPUT);
      if (!server_mode || prompt_dummy || IS_PLAYER(local_player))
	Pause ("");
    }
    break;

  default:
    break;
  }
}

static void Prepare_for_input_during_bidding (t)
     Table t;
{

  Board *b;
  Play_record *p;
  char buf[80];

  b = t->board;
  p = t->play_record;

  Display_Player (p->next_player);
  if (p->next_player == local_player)
    Begin_timer_for_play ();

  if ((p->next_player == local_player) && !p->bidding_completed) {
    ring_bell ();
    if (Talking())
      Status ("PRESS <TAB> TO ENTER YOUR BID.");
  }

  if ((t->playing_mode == PRACTICE_PLAYING_MODE) && server_mode)
    if (VACANT(t, p->next_player) && autopass_mode) {
      sprintf (buf, "%s %s BID PASS %d", seat_names[p->next_player],
	       seat_names[p->next_player], Local_play->no_bids);
      loopback_message_unformatted (Local_table, Local_Player_Connection, buf);
    }

  Display_Bid_on_Table (p->next_player, "");
}

static void Setup_Bidding_Display (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
/* Initializes the bidding display and global variables which are
   pertinent to the bidding.
*/
{
  int i;

  Begin_timer_for_hand ();
  Set_Display_Mode (BIDDING_DISPLAY);
  if (VUGRAPH(Local_table)) {
    if (server_mode)
      Set_Input_Mode (BID_INPUT);
    else
      Set_Input_Mode (TALK_INPUT);
    for (i = 0; i < 4; i++)
      revealed_hands[i] = 1;
    Refresh_Playing_Area ();
  } else if (IS_PLAYER(local_player)) {
    Set_Input_Mode (BID_INPUT);
    Display_Hand (local_player);
  } else
    Set_Input_Mode (TALK_INPUT);

  switch (p->no_bids % 4) {
  case 0: p->next_player = b->dealer; break;
  case 1: p->next_player = player_next[b->dealer]; break;
  case 2: p->next_player = player_partner[b->dealer]; break;
  case 3: p->next_player = player_prev[b->dealer]; break;
  }

  Refresh_Playing_Area ();
  for (i = 0; i < 4; i++)
    Display_Player_Bids (i);

  pause_mode = 0;
  Prepare_for_input_during_bidding (t);

}

static void Display_event_in_Bidding_Mode
  (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
{
  int i;

  if (c->command == CMD_ALERT) {
    if ((local_player == player_partner[c->player_no]) && 
	((!show_alerts) || FORMAL(Local_table)))
      return;

    i = Index_of_Last_Bid (b, p, player_partner[c->player_no]);
    if (i >= 0) {
      Refresh_Player (player_partner[c->player_no]);
      Moderator_Comment ("ALERT!");
      ring_bell ();
    }
  } else if (c->command == CMD_BID) {
    if (c->player_no == local_player)
      End_timer_for_play ();
    Display_timer ();
    Refresh_Player (c->player_no);
    Prepare_for_input_during_bidding (t);
  }
  if (VUGRAPH(Local_table))
    Refresh_Playing_Area ();
}

static void Setup_Playing_Display (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
{
  int i, j, player;

  Set_Display_Mode (PLAYING_DISPLAY);
  if (PRACTICE(t)) {
    if (IS_PLAYER(local_player))
      Set_Input_Mode (PLAY_INPUT);
    else
      Set_Input_Mode (TALK_INPUT);
    for (i = 0; i < 4; i++)
      revealed_hands[i] = 1;
  } else if (VUGRAPH(t)) {
    for (i = 0; i < 4; i++)
      revealed_hands[i] = 1;
    if (server_mode)
      Set_Input_Mode (PLAY_INPUT);
    else
      Set_Input_Mode (TALK_INPUT);
  } else if (IS_PLAYER(local_player) && 
	     (local_player != player_partner[p->declarer]))
    Set_Input_Mode (PLAY_INPUT);
  else
    Set_Input_Mode (TALK_INPUT);
  Clear_Plays_from_Table ();

  if (IS_PLAYER(local_player))
    revealed_hands[local_player] = 1;

  if (local_player == p->declarer)
    revealed_hands[player_partner[local_player]] = 1;
  else if (local_player == player_partner[p->declarer])
    revealed_hands[p->declarer] = 1;
  Refresh_Playing_Area ();

  /* If we are replaying a hand which has already been (partially) played,
     then display the plays up to this point:
  */
  player = player_next[p->declarer];
  for (i = 0; i < p->no_plays; i++) {
    if (i % 4 == 0)
      Clear_Plays_from_Table ();
    Display_Play_on_Table (player, card_names[p->play_list[i]]);
    player = player_next[player];
    if (!revealed_hands[player_partner[p->declarer]])
      Display_Hand (player_partner[p->declarer]);

    if (i % 4 == 3) {
      for (j = 0; j < 4; j++)
	if (revealed_hands[j])
	  Display_Partial_Hand (j, i);

      if (prompt_dummy && !exit_requested)
	Press_Return_to_Continue ("");

      player = Winning_card(trumpsuit_of(p->contract), player,
			    p->play_list[i-3], p->play_list[i-2],
			    p->play_list[i-1], p->play_list[i]);
    }
  }
  p->next_player = player;

  Refresh_Playing_Area ();

  Display_Player (p->next_player);

  if ((Next_Player (p) == local_player) || PRACTICE(t) || VUGRAPH(t))
    Compute_Default_Play ();
  else
    Clear_Default_Play ();

  if (Next_Player (p) == local_player) {
    Begin_timer_for_play ();
    ring_bell ();
    if (Talking())
      Status ("PRESS <TAB> TO ENTER YOUR PLAY.");
  }
}

static void Display_event_in_Playing_Mode
  (t, b, p, c, player)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
     int player;
{
  if (c->command != CMD_PLAY)
    return;

  if (p->no_plays % 4 == 1)
    Clear_Plays_from_Table ();

  Display_Play_on_Table (player, card_names[c->data.play.card]);
  End_timer_for_play ();
  Display_timer ();

  Clear_Default_Play ();

  if ((p->no_plays % 4 == 0) && prompt_dummy) {
    Refresh_Playing_Area ();
    Pause ("");
    Clear_Plays_from_Table ();
  }

  if (p->no_plays == 1)
    Display_Hand (player_partner[p->declarer]);

  Display_Player (p->next_player);

  if ((Next_Player (p) == local_player) || PRACTICE(t) || VUGRAPH(t))
    Compute_Default_Play ();

  if (Next_Player (p) == local_player) {
    Begin_timer_for_play ();
    ring_bell ();
    if (Talking())
      Status ("PRESS <TAB> TO ENTER YOUR PLAY.");
  }

  Refresh_Playing_Area ();
}

static void Setup_Scoring_Display (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
{
  char message_buf[100];

  int contractor = side_of (p->declarer);
  int honors;
  int i, below[2], hcp;

  End_timer_for_play ();
  End_timer_for_hand ();
  Display_timer ();

  if (IS_PLAYER(local_player) && 
      (side_of(local_player) == contractor) &&
      (level_of (p->contract) >= 6) &&
      (p->result > 0)) {
    Set_Display_Mode (HELP_DISPLAY);
    Set_Input_Mode (TALK_INPUT);
    display_help ("slam");
  }

  if (display_mode != PLAYING_DISPLAY)
    Set_Display_Mode (PLAYING_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  for (i = 0; i < 4; i++)
    Display_Hand (i);
  Refresh_Playing_Area ();

  switch (b->scoring_mode) {
  case RUBBER_SCORING:
    if (p->result == 0)
      break;

    honors = Honors_Point_Bonus (b, p);
    if (honors == 100)
      Moderator_Comment 
	("100 POINT BONUS FOR 4 HONORS IN ONE HAND");
    else if ((honors == 150) && (trumpsuit_of(p->contract) != SUIT_NOTRUMP))
      Moderator_Comment
	("150 POINT BONUS FOR ALL 5 HONORS IN ONE HAND");
    else if (honors == 150)
      Moderator_Comment
	("150 POINT BONUS FOR ALL 4 ACES IN ONE HAND");

    below[0] = b->part_score[0] + p->score[0].rubber.below_line;
    below[1] = b->part_score[1] + p->score[1].rubber.below_line;
    if (below[contractor] >= 100) {
      if (b->vulnerable[contractor]) {
	if (contractor == SIDE_NS)
	  sprintf (message_buf, "%s AND %s HAVE WON THE RUBBER.",
		   PLAYER_NAME(Local_table, PLAYER_NORTH), 
		   PLAYER_NAME(Local_table, PLAYER_SOUTH));
	else
	  sprintf (message_buf, "%s AND %s HAVE WON THE RUBBER.",
		   PLAYER_NAME(Local_table, PLAYER_EAST), 
		   PLAYER_NAME(Local_table, PLAYER_WEST));
	Moderator_Comment (message_buf);
      }
    }
    break;

  case DUPLICATE_SCORING:
  case MP_SCORING:
  case IMP_SCORING:
    hcp = Highcard_points (b, contractor);
    sprintf 
      (message_buf,
       "HCP %d, EXPECTED SCORE: %d, ACTUAL SCORE: %d, MONO-IMPS: %5.1f",
       hcp,
       b->vulnerable[contractor] ?
         MIMP_scoring_vuln[hcp]: MIMP_scoring_nonvuln[hcp],
       (p->result >= 0)?
         Duplicate_score_made 
           (b->vulnerable[contractor], level_of (p->contract),
	    trumpsuit_of (p->contract), p->doubled, p->result):
         -Duplicate_score_set
           (b->vulnerable[contractor], level_of (p->contract),
	    trumpsuit_of (p->contract), p->doubled, p->result),
       ((float) p->mimp_points[contractor]) * 0.5);
    Display_Player_Comment (COMMENT_PUBLIC, "SCORE", message_buf);
    break;
  }

  Display_Above_Line_Points ();
  Display_Below_Line_Points ();
  Display_Vulnerabilities ();
}

static void Display_event_in_Scoring_Mode
  (t, b, p, c)
     Table t;
     Board *b;
     Play_record *p;
     player_command c;
{
  switch (c->command) {
  case CMD_SCORE:
    t->scoring_mode =  c->data.score.score_type;
    bcopy (&c->data.score.ns, &t->score[SIDE_NS], sizeof(Score));
    bcopy (&c->data.score.ew, &t->score[SIDE_EW], sizeof(Score));
    Display_Above_Line_Points ();
    Display_Below_Line_Points ();
    break;

  default:
    break;
  }
}

static void Record_Board (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
{
  if (logfile != NULL)
    Write_hand (logfile, b, p, 1);
  if (logfile_email != NULL) {
    Write_Email_Board (logfile_email, NULL, b);
    fflush (logfile_email);
  }

  if (record_this_hand) {
    if (recfile != NULL)
      Write_hand (recfile, b, p, 1);
    if (recfile_email != NULL) {
      Write_Email_Board (recfile_email, NULL, b);
      fflush (recfile_email);
    }
    record_this_hand = 0;
  }

  if (zhang_logfile != NULL)
    Write_hand_compactly (zhang_logfile, t->table_no, b, p);
}

static void Display_Summary_of_Hand (t, b, p)
     Table t;
     Board *b;
     Play_record *p;
{
  FILE *fp;
  char buf[80];

  Compute_Matchpoints (b);
  Compute_Intl_Matchpoints (b);

  Clear_Plays_from_Table ();

  Record_Board (t, b, p);

  if (server_mode && duplicate_board_in_use) {
    GPS_Upload_Play_Record (Local_board, Local_play);
    if (b->play_records->next == NULL)
      Broadcast_Comment 
	("THIS BOARD HAS NOT BEEN PLAYED AT ANY OTHER TABLES.");
  }

  if (replay_mode && !Board_is_Available()) {
    replay_mode = 0;
    fp = fopen (email_filename, "w");
    if (fp == NULL) {
      sprintf (buf, "ERROR RE-OPENING EMAIL FILE %s: %s", email_filename,
	       sys_errlist[errno]);
      Status (buf);
    } else {
      Write_Email_Duplicate_File (fp);
      sprintf (buf, "WROTE EMAIL BOARDS TO %s.", email_filename);
      Moderator_Comment (buf);
      fclose (fp);
    }
  }

  if (b->play_records->next != NULL) {
    if (IS_OBSERVER(local_player))
      sort_direction = 1;
    else if (side_of(local_player) == SIDE_NS)
      sort_direction = 1;
    else
      sort_direction = -1;

    switch (b->scoring_mode) {
    case RUBBER_SCORING:
    case DUPLICATE_SCORING:
    case MP_SCORING:
      Sort_play_records_by_matchpoints (b);
      break;
    case IMP_SCORING:
      Sort_play_records_by_imps (b);
      break;
    }

    if (!server_mode || prompt_dummy || IS_PLAYER(local_player)) {
      if (show_results) {
	Pause ("PRESS <ESC> TO SEE THE RESULTS OF PLAY BY OTHER TABLES ...");
	if (show_results & 1) {
          result_mode = 1;
          Display_First_Page_of_Scores (b);
	  while (More_Scores_to_Display() && !exit_requested) {
	    Pause ("PRESS <ESC> TO SEE MORE RESULTS OR /EXIT TO QUIT...");
	    if (!exit_requested)
	      Display_More_Scores ();
	  }
	  Init_Scores_to_Display ();
          if (show_results >= 2) Pause ("PRESS <ESC> FOR FULL RESULTS ...");
        }
	if (!exit_requested && (show_results >= 2)) {
          result_mode = 0;
          Display_First_Page_of_Scores (b);
	  while (More_Scores_to_Display() && !exit_requested) {
	    Pause ("PRESS <ESC> TO SEE MORE RESULTS OR /EXIT TO QUIT...");
	    if (!exit_requested)
	      Display_More_Scores ();
	  }
	  Init_Scores_to_Display ();
        }
      }
    }
  }
  
  if (!server_mode || prompt_dummy || IS_PLAYER(local_player) || 
      VUGRAPH (Local_table))
    Pause ("PRESS <ESC> FOR THE NEXT HAND ...");
  Init_Scores_to_Display ();
}

static void Transmit_Game_State_Information (t, c)
     Table t; Connection c;
{
  char buf1[100], buf2[100], buf3[100];
  Play_record *p;

  c->state = CSTATE_PLAYING;

  if (t->game_mode == STARTUP_MODE)
    return;

  send_private_message (c, "DEAL");

  Format_Score_Message (buf1, t, &t->score[SIDE_NS], &t->score[SIDE_EW]);
  send_private_message (c, buf1);
  
  if (t->board == NULL)
    return;

  sprintf (buf1, "BOARD %s %d", t->board->source, t->board->serial_no);
  send_private_message (c, buf1);

  Encode_board (t->board, buf1, buf2);
  server_send_unformatted (c, buf1);
  server_send_unformatted (c, buf2);

  for (p = t->board->play_records; p != NULL; p = p->next) {
    sprintf (buf1, "RECORD %s %d", t->board->source, t->board->serial_no);
    send_private_message (c, buf1);
    Encode_play_record (p, buf1, buf2, buf3);
    server_send_unformatted (c, buf1);
    server_send_unformatted (c, buf2);
    server_send_unformatted (c, buf3);
  }

  if ((p = t->play_record) != NULL) {
    sprintf (buf1, "USEREC %s %s %s %s", 
	     p->player_names[PLAYER_NORTH], p->player_names[PLAYER_EAST],
	     p->player_names[PLAYER_SOUTH], p->player_names[PLAYER_WEST]);
    send_private_message (c, buf1);
  }

  send_private_message (c, "BEGIN");

  if (t->game_mode != SCORING_MODE)
    return;

  send_private_message (c, "END");
}

static void main_event_loop ()
{
  Table t;
  Board *b;
  Play_record *p;
  Message m;
  player_command c;
  int transition, player;
  int prev_mode;

  do {
    Refresh_Input_Buffers ();
    t = Wait_for_game_message ();
    b = t->board;
    p = t->play_record;
    m = dequeue_message (t->game_queue);
    c = &(m->p);
/*    Network_Comment (m->p.command_text); */

    if (server_mode && (c->command == CMD_HELLO))
      /* If this person has just joined the game, then we need to transmit
         state information. */
      Transmit_Game_State_Information (t, m->source);

    if (c->command == CMD_SKIP) {
      if (VUGRAPH(Local_table)) {
	t->game_mode = STARTUP_MODE;
	Set_Display_Mode (TALK_DISPLAY);
	Set_Input_Mode (TALK_INPUT);
      } else
	t->game_mode = SKIP_MODE;
      pause_mode = claim_in_progress = 0;
      transition = 1;
    } else {
      switch (t->game_mode) {
      case SKIP_MODE:
	transition = Handle_event_in_Skip_Mode (t, c);
	break;

      case STARTUP_MODE:
	transition = Handle_event_in_Startup_Mode (t, b, p, c);
	if (VISIBLE(t))
	  Display_event_in_Startup_Mode (t, b, p, c);
	break;
	
      case DEALING_MODE:
	transition = Handle_event_in_Deal_Mode (t, b, p, c);
	if (VISIBLE(t))
	  Display_event_in_Deal_Mode (t, b, p, c);
	break;
	
      case BIDDING_MODE:
	transition = Handle_event_in_Bidding_Mode (t, b, p, c);
	if (VISIBLE(c))
	  Display_event_in_Bidding_Mode (t, b, p, c);
	if (transition && VISIBLE (c)) {
	  /* If player is on declaring side, previously invisible alerts */
	  /* may now be visible, so must refresh. */
	  Refresh_Player(local_player); 
	  Clear_Default_Play ();
	  Display_Final_Contract (b, p);
	  if (!server_mode || prompt_dummy || IS_PLAYER(local_player)) {
	    if (t->game_mode == PLAYING_MODE)
	      Pause ("ALL PASS -- PRESS <ESC> TO BEGIN PLAY ...");
	    else if (t->game_mode == SCORING_MODE)
	      Pause ("THROWING IN THE HAND -- PRESS <ESC> TO CONTINUE ...");
	  }
	}
	break;
	
      case PLAYING_MODE:
	player = p->next_player;
	transition = Handle_event_in_Playing_Mode (t, b, p, c);
	if (VISIBLE(t))
	  Display_event_in_Playing_Mode (t, b, p, c, player);
	break;
	
      case SCORING_MODE:
	transition = Handle_event_in_Scoring_Mode (t, b, p, c);
	if (VISIBLE(t))
	  Display_event_in_Scoring_Mode (t, b, p, c);
	if (transition && VISIBLE(t)) {
	  prev_mode = Local_table->game_mode;
	  Local_table->game_mode = SCORING_MODE;
	  Display_Summary_of_Hand (t, b, p);
	  Local_table->game_mode = prev_mode;
	}
	break;
      }
    }

    if (transition) {
      Clear_Status ();
      switch (t->game_mode) {
      case SKIP_MODE:
	Initiate_Skip_Mode (t);
	if (VISIBLE(t)) {
	  prompt_before_hand = IS_PLAYER(local_player) || prompt_dummy;
	  Set_Display_Mode (TALK_DISPLAY);
	  Set_Input_Mode (TALK_INPUT);
	}
	break;
	
      case STARTUP_MODE:
	Initiate_Startup_Mode (t);
	if (VISIBLE(t))
	  Setup_Startup_Display ();
	break;
	
      case DEALING_MODE:
	Initiate_Deal_Mode (t);
	if (VISIBLE(t))
	  Setup_Deal_Display (b, p);
	break;
	
      case BIDDING_MODE:
	if (VISIBLE(t))
	  Setup_Bidding_Display (t, t->board, t->play_record);
	break;
	
      case PLAYING_MODE:
	Record_Player_Names (t, t->board, t->play_record);
	if (VISIBLE(t))
	  Setup_Playing_Display (t, t->board, t->play_record);
	break;
	
      case SCORING_MODE:
	Initiate_Scoring_Mode (t, t->board, t->play_record);
	if (VISIBLE(t))
	  Setup_Scoring_Display (t, t->board, t->play_record);
	break;
      }
      deallocate_message (m);
    }
  } while (1);
}

long Seconds ()
/* Returns current time in seconds. */
{
  struct timeval tp;

  gettimeofday (&tp, NULL);
  return (tp.tv_sec);
}

int Terminate_Program (termination_message)
     char *termination_message;
{
  if (!abort_flag) {
    abort_flag = 1;
    if ((Local_table->game_mode == PLAYING_MODE) || 
	(Local_table->game_mode == BIDDING_MODE)) {
      Record_Board (Local_table, Local_board, Local_play);
    }
    if (save_defaults)
      Write_Initialization_File (".okdefaults");
    if (GPS_socket && !GPS_request_in_progress) {
      if (server_mode) {
	GPS_End_Server_Mode ();
	server_mode = 0;
      }
    } else
      GPS_socket = Use_GPS = 0;
    Refresh_Display ();
    Send_quit (Local_table);
    Press_Return_to_Continue (termination_message);
  }
  clear_screen ();
  Reset_Terminal ();
  exit (0);
  return (0);
}

int recovery_routine (parm)
	int parm;
{
  Terminate_Program ("PROGRAM ABORT");
  return (0);
}

int Hangup_recovery_routine ()
{
  FILE *fp;

  if (!abort_flag) {
    abort_flag = 1;
    if ((Local_table->game_mode == PLAYING_MODE) || 
	(Local_table->game_mode == BIDDING_MODE)) {
      Record_Board (Local_table, Local_board, Local_play);
    }
    if (save_defaults)
      Write_Initialization_File (".okdefaults");
    if (Local_table->board != NULL)
      Record_Played_Board (Local_table->board);
    fp = fopen ("okbridge.hands", "w");
    if (fp != NULL) {
      Write_Email_Duplicate_File (fp);
      fclose (fp);
    }
  }
  exit (0);
  return (0);
}

int Quit_program ()
{
  Clear_Status ();
  Clear_Focus_Buffer ();
  Continue_Comment_Display ();
  if (Ask("ARE YOU SURE YOU WISH TO QUIT? ")) {
    abort_flag = 1;
    Reset_Status ();
    if (GPS_socket && !GPS_request_in_progress) {
      if (Use_GPS && display_scoreboard_at_end)
	GPS_Display_Scoreboard (0);
      if (server_mode) {
	GPS_End_Server_Mode ();
	server_mode = 0;
      }
    }
    if (save_defaults)
      Write_Initialization_File (".okdefaults");
    clear_screen ();
    Reset_Terminal ();
    Send_quit (Local_table);
    exit (0);
  }
#ifdef HPUX
  signal (SIGINT,  (signal_handler) Quit_program);
#endif
  return (0);
}

int quit_error_routine ()
{
  if (!abort_flag) {
    abort_flag = 1;
#ifdef LOGFILE
    fflush (net_log);
#endif
    if (server_mode) {
      GPS_End_Server_Mode ();
      server_mode = 0;
    }
    clear_screen ();
    Reset_Terminal ();
  }
  kill (getpid(), SIGTRAP);
  exit (0);
  return (0);
}

void Core_dump ()
{
  quit_error_routine ();
}

int socket_recovery_routine (parm)
     int parm;
{
  Network_Comment ("BROKEN NETWORK CONNECTION!");

#ifdef HPUX
	signal (SIGPIPE, SIG_IGN);
#ifdef SIGURG
	signal (SIGURG,  SIG_IGN);
#endif
#endif

  return 0;
}

int system_call_error_routine (parm)
     int parm;
{
  if (!abort_flag) {
    abort_flag = 1;
#ifdef LOGFILE
    fflush (net_log);
#endif
    if (server_mode) {
      GPS_End_Server_Mode ();
      server_mode = 0;
    }
    Moderator_Comment ( "INTERRUPTED SYSTEM CALL");
  }
  Terminate_Program ("PROGRAM TERMINATING");
  return (0);
}

int access_error_routine (parm)
     int parm;
{
  if (!abort_flag) {
    abort_flag = 1;
    if (server_mode) {
      GPS_End_Server_Mode ();
      server_mode = 0;
    }
    Send_quit (Local_table);
#ifdef LOGFILE
    fflush (net_log);
#endif
    Moderator_Comment
      ("ACCESS VIOLATION!!  YOU HAVE FOUND A BUG IN THE PROGRAM.");
    Moderator_Comment ( "PROGRAM TERMINATING");
/*    input_acknowledgment (-1); */
  }
  clear_screen ();
  Reset_Terminal ();
  kill (getpid(), SIGTRAP);
  return (0);
}


void Generate_reset (reset_type)
     int reset_type;
{
  longjmp (reset_buf, reset_type);
}

void Handle_reset (reset_status)
     int reset_status;
{
  int seat;
  Table t;
  Connection c;

  if (reset_status == 0)
    return;

  spectator_mode = 0;
  for (t = Table_List; t != NULL; t = t->next) {
    clear_all_message_queues (t);
    t->game_mode = STARTUP_MODE;
  }

  FOREACH_CONNECTION (c)
    c->spectator = 0;
  for (seat = 0; seat < 4; seat++)
    Local_table->Seats[seat].skipping = 0;

  prompt_before_hand = 1;
  claim_in_progress = 0;

  bzero (&Local_table->score[0], sizeof(Score));
  bzero (&Local_table->score[1], sizeof(Score));
  bzero (&local_player_score, sizeof(Score));
  bzero(&local_player_total_score, sizeof(Score));

  Local_table->playing_mode = CLUB_PLAYING_MODE;
  Reset_Status ();
  Continue_Comment_Display ();
  Reinitialize_Input ();
  Set_Display_Mode (TALK_DISPLAY);
  Set_Input_Mode (TALK_INPUT);

  GPS_server_message[0] = '\0';
  duplicate_board_in_use = 0;

  if (conventions[SIDE_EW] != NULL) free (conventions[SIDE_EW]);
  if (conventions[SIDE_NS] != NULL) free (conventions[SIDE_NS]);
  conventions[SIDE_EW] = conventions[SIDE_NS] = NULL;

  Prev_board = NULL;
  Prev_play = NULL;

  switch (reset_status) {
  case RESET_FULL:
/*    GPS_Reset (); */
    Set_Display_Mode (TALK_DISPLAY);
    Set_Input_Mode (TALK_INPUT);
    Clear_All_Boards ();
    replay_mode = 0;
    if (server_mode) {
      FOREACH_CONNECTION (c)
	c->state = CSTATE_CONNECTED;
      GPS_Broadcast_Server ();
      GPS_Refresh_Players ();
    }
    if (client_mode) {
      seat = local_player;
      local_player = PLAYER_OBS;
      Send_hello (Local_table, major_revision_level, local_player_name, seat);
    }
    break;

  case RESET_CONNECT:
    Conclude_GPS_request ();
    Close_all_connections ();
/*    Attempt_to_connect (local_player); */
    Attempt_to_connect (PLAYER_OBS);
    break;

  case RESET_LURKER:
    Conclude_GPS_request ();
    Close_all_connections ();
    Network_Comment ("YOU ARE NOW A LURKER.");
    break;

  case RESET_SERVE:
    Conclude_GPS_request ();
    Close_all_connections ();
    Setup_server ();
    hands_played = 0;
    break;

  case RESET_DISCONNECT:
    Conclude_GPS_request ();
    Close_all_connections ();
    break;
  }

}

void parameter_error (error_msg)
     char *error_msg;
{
  printf ("Okbridge %s%s:: Error: %s\n", 
	  major_revision_level, minor_revision_level, error_msg);
  printf ("First player starts the program on his machine with\n");
  printf ("   okbridge [-c] [-d] [-e] [-i] [-R] <seat> <name> \n%s\n%s\n",
	  "        [-P portno] [-r replay-file] [-l load-file] [-L log-file]",
	  "        [-z# [zlog-file]]");
  printf ("Others connect with\n");
  printf ("   okbridge [-c] [-d] [-e] [-i] [-R] %s\n%s\n",
	  "<seat> <name> <host>","        [ ... additional parameters ... ]");
  printf ("<seat> is one of the characters n, s, e, w\n");
  printf ("<name> is your identifying name (ex. 'steve')\n");
  printf ("<host> is the internet name of the first player's machine\n\n");
  printf ("-c     is for Chicago scoring\n");
  printf ("-d     is for Duplicate scoring\n");
  printf ("-e     is for Email duplicate scoring\n");
  printf ("-i     is for simulated IMP scoring\n");
  printf ("-m     is for Modern IMP scoring\n");
  printf ("  NOTE: Only the server can choose the scoring option.\n\n");
  printf ("-P portno specifies the internet port number for communications\n");
  printf ("-r replay-file specifies an email duplicate file for replay\n");
  printf ("-l load-file specifies an email duplicate file for loading\n");
  printf ("-L log-file specifies a logfile file saving the results of play\n");
  printf ("-z# zlog-file specifies a table number & Zhang-format log file\n");
  exit (1);
}

static void parse_position_parameter (position_string)
     char *position_string;
{
  char ch, error_buf[80];

  if (strlen(position_string) == 1) {
    ch = *position_string;
    switch (ch) {
    case 'n': 
    case 'N': 
      local_player = PLAYER_NORTH;
      return;
    case 'e': 
    case 'E': 
      local_player = PLAYER_EAST;
      return;
    case 's': 
    case 'S': 
      local_player = PLAYER_SOUTH;
      return;
    case 'w': 
    case 'W': 
      local_player = PLAYER_WEST;
      return;
    }
  }
  sprintf (error_buf, "Error -- unrecognized <seat> name: %s\n", 
	   position_string);
  parameter_error (error_buf);
}


#ifdef SUNOS
void Change_display_size (parm, code, scp, addr)
     int parm, code; struct sigcontext *scp; char *addr;
/* Computes the current dimensions of the display and updates the size of
 . the talk window appropriately.
 */
{
  char msg_buf[80];

  Reinitialize_Terminal ();
  Reinitialize_Display ();
  Reinitialize_Input ();
  Refresh_Display ();

  sprintf (msg_buf, "THE CURRENT DIMENSIONS APPEAR TO BE: %d x %d\n",
	   terminal_lines, terminal_cols);
  Display_Player_Comment (COMMENT_PRIVATE, "SIGWINCH", msg_buf);
/*
  restore_cursor ();
*/
}
#endif /* SUNOS */


#ifndef VMS
void main (argc, argv)
#else
main (argc, argv)
#endif
     int argc; char **argv;
{
        int error_value;   /* error return from setjmp/signal */
	int argi;          /* index of current argument being processed */
	int posi;          /* index of next positional parameter to process */
	char *filename;    /* filename specified for log file on command line*/
	char *helpenv;     /* filename specified in OKBRIDGE_HELPFILE
			      environment variable. */
        char filename_buf[80], error_buf[80];
        FILE *fp;          /* used only for checking the existence of files */
	int need_to_save_context;
	int reset_in_progress;
	int status;
	int ccardi;

	/* Various signal handlers: */
	int recovery_routine (), Quit_program (), socket_recovery_routine (),
		quit_error_routine (), access_error_routine (),
		system_call_error_routine (), Hangup_recovery_routine();

#define current_arg  (argv[argi])
#define next_arg     argi++
#define more_args    (argi < argc)

#ifdef DEBUG
#ifndef HPUX
	struct rlimit rlp;

	getrlimit (RLIMIT_CORE, &rlp);
	rlp.rlim_cur = rlp.rlim_max;
	setrlimit (RLIMIT_CORE, &rlp);
#endif
#endif

#ifdef MDEBUG
	malloc_debug (2);
#endif

	local_player = PLAYER_OBS;
	server_name = NULL;
	client_mode_requested = 0;
	server_mode_requested = 0;
	bzero (&local_player_score, sizeof(Score));
	bzero (&local_player_mp_total, sizeof(Score));
	bzero (&local_player_imp_total, sizeof(Score));
	bzero(&local_player_total_score, sizeof(Score));

 	for (ccardi = 0; ccardi < built_in_cc_count; ccardi++)
 	  Define_CC 
 	    (built_in_ccs[ccardi], NULL);
  
	if ((helpenv = getenv("OKBRIDGE_HELPFILE")) != NULL)
	  help_file_name = strdup (helpenv);

	Initialize_Key_Mappings ();

	Read_Initialization_File (".okdefaults");
	Read_Initialization_File (".okbridgerc");

	if (server_name != NULL) {
	  client_mode_requested = 1;
	  server_mode_requested = 0;
	}
	  
	argi = 1;
	posi = 1;
	while (more_args) {
	  if(*current_arg == '-') {
	    /* process a position-independent parameter. */
	    if (!strcmp(current_arg, "-d")) {
	      scoring_mode = DUPLICATE_SCORING;
	    } else if (!strcmp(current_arg, "-i")) {
              scoring_mode = IMP_SCORING;
	    } else if (!strcmp(current_arg, "-m")) {
              scoring_mode = MP_SCORING;
	    } else if (!strcmp(current_arg, "-R")) {
	      scoring_mode = RUBBER_SCORING;
	    } else if ((*current_arg == '-') && (*(current_arg+1) == 'p')) {
	      parse_position_parameter (current_arg+2);
	      if (posi == 1) posi++;
	    } else if (!strcmp(current_arg, "-n")) {
	      next_arg;
	      if (!more_args)
		parameter_error 
		  ("-n parameter requires player name to follow");
	      if (!String_is_alphameric(current_arg))
		parameter_error 
		  ("player names must not contain special characters");
	      local_player_name = strdup (current_arg);
	      if (posi == 2) posi++;
	    } else if (!strcmp(current_arg, "-s")) {
	      next_arg;
	      if (!more_args)
		parameter_error 
		  ("-s parameter requires server name or 'ME' to follow");
	      if (!strcasecmp(current_arg, "me")) {
		client_mode_requested = 0;
		server_mode_requested = 1;
	      } else {
		server_name = strdup (current_arg);
		client_mode_requested = 1;
		server_mode_requested = 0;
	      }
	      if (posi == 3) posi++;
	    } else if (!strcmp(current_arg, "-P")) {
	      next_arg;
	      if (!more_args)
		parameter_error 
			("Error -- port number must follow -P parameter");
	      network_port = atoi (current_arg);
	      if (!network_port) {
		sprintf (error_buf, "Error: %s\n", 
			"-P parameter requires a positive port number");
		parameter_error (error_buf);
	      }
	    } else if (!strcmp(current_arg, "-G")) {
	      Use_GPS = 0;
	    } else if (!strcmp(current_arg, "-b")) {
	      next_arg;
	      if (!more_args)
		parameter_error
		  ("Error -- boards filename must follow -b parameter");
	      if ((fp = fopen(current_arg,"r")) == NULL) {
		sprintf (error_buf, 
			 "Error -- could not open boards file %s", 
			 current_arg);
		parameter_error (error_buf);
	      }
	      email_filename = strdup(current_arg);
	      status = Load_Email_Duplicate_File (fp);
	      if (status == -1) {
		sprintf (error_buf, "%s is not an email duplicate file.",
			 email_filename);
		parameter_error (error_buf);
	      } else if (status) {
		sprintf (error_buf, "Error accessing email duplicate file %s",
			 email_filename);
		parameter_error (error_buf);
	      }
	      fclose (fp);
	    } else if (!strcmp(current_arg, "-L")) {
	      if (logfile != NULL) fclose (logfile);
	      next_arg;
	      if (!more_args)
		parameter_error
		  ("Error -- logfile filename must follow -L parameter");
	      if (current_arg[0] == '+') {
		filename = current_arg + 1;
		logfile = fopen (filename, "a");
	      } else {
		filename = current_arg;
		logfile = fopen (filename, "w");
	      }
	      if (logfile == NULL) {
                sprintf (error_buf,
			 "Error -- Could not open log file %s: %s\n", 
			 filename, sys_errlist[errno]);
		parameter_error (error_buf);
	      }
	    } else if ((current_arg[0] == '-') && (current_arg[1] == 'z')) {
	      table_no = atoi (current_arg+2);
	      if (table_no == 0) {
		sprintf (error_buf, 
			"Error in table number for Zhang mode: %s",
			current_arg+2);
		parameter_error (error_buf);
	      }
	      if ((argi < argc-1) && (argv[argi+1][0] != '-')) {
		next_arg;
		if (current_arg[0] == '+') {
		  filename = current_arg+1;
		  zhang_logfile = fopen (filename, "a");
		} else {
		  filename = current_arg;
		  zhang_logfile = fopen(filename, "w");
		}
	      } else {
		sprintf (filename_buf, "okb_%s_rec", current_arg+2);
		filename = filename_buf;
		zhang_logfile = fopen (filename, "a");
	      }
	      if (zhang_logfile == NULL) {
                sprintf (error_buf,
			 "Error -- Could not open zhang log file %s: %s", 
			filename, sys_errlist[errno]);
		parameter_error (error_buf);
	      }
            } else {
	      sprintf (error_buf,
		      "Error -- unrecognized parameter: %s", current_arg);
	      parameter_error (error_buf);
	    }
	  } else {
	    /* Process a positional parameter. */
	    switch (posi) {
	    case 1: 
	      parse_position_parameter (current_arg);
	      break;
	    case 2:
	      if (!String_is_alphameric (current_arg))
		parameter_error 
		  ("player names must not contain special characters");
	      local_player_name = strdup(current_arg);
	      break;
	    case 3:
	      if (!strcasecmp(current_arg, "me")) {
		client_mode_requested = 0;
		server_mode_requested = 1;
	      } else {
		server_name = strdup (current_arg);
		client_mode_requested = 1;
		server_mode_requested = 0;
	      }
	      break;
	    default:
	      parameter_error("Error -- too many parameters given.");
	    }
	    posi++;
	  }
	  next_arg;
	}
        if (local_player_name == NULL) {
	  filename = getenv("USER");
	  if (filename != NULL)
	    local_player_name = strdup (filename);
	  else
	    parameter_error
	      ("Error -- you must specify your name to play.");
	}

	if (strlen(local_player_name) > 8)
	  local_player_name[8] = '\0';
 
	/* Initialize the state of the email duplicate variables: */

	abort_flag = 0;
#ifdef SIGABRT
	signal (SIGABRT, (signal_handler) recovery_routine);
#endif
	signal (SIGHUP,  (signal_handler) Hangup_recovery_routine);
#ifndef VMS
	/* VMS handles this at AST level, and Quit_program does read */
	/* with timer, and timer can't be handled at AST level! */
	signal (SIGINT,  (signal_handler) Quit_program);
#endif
	signal (SIGPIPE, (signal_handler) socket_recovery_routine);
#ifdef SIGURG
	signal (SIGURG,  (signal_handler) socket_recovery_routine);
#endif

	signal (SIGQUIT, (signal_handler) quit_error_routine);
	signal (SIGFPE,  (signal_handler) access_error_routine);
	signal (SIGILL,  (signal_handler) access_error_routine);
	signal (SIGBUS,  (signal_handler) access_error_routine);
	signal (SIGSEGV, (signal_handler) access_error_routine);
	signal (SIGSYS,  (signal_handler) system_call_error_routine);

/*
#ifdef SUNOS
	signal (SIGWINCH, (signal_handler) Change_display_size);
#endif SUNOS
*/

	startup_time = Seconds ();

	error_value = setjmp (error_return);
	if (error_value != 0)
	        Terminate_Program ("SIGNAL RECEIVED");
 
	/* The order of the following initialization calls should not
           be changed. */
	Initialize_Network ();
	Initialize_Terminal ();
	Initialize_Display ();
	Initialize_Input ();
	initialize_help_system ();

	Set_Display_Mode (MANUAL_DISPLAY);
	Set_Input_Mode (TALK_INPUT);
	
	display_news ();
	clear_screen ();

	Set_Display_Mode (TALK_DISPLAY);
	Set_Input_Mode (TALK_INPUT);
	
	Moderator_Comment
	  ("WELCOME TO OKBRIDGE.");
	Moderator_Comment
	  ("COPYRIGHT (C) 1990-1993 BY MATTHEW CLEGG.  ALL RIGHTS RESERVED.");

	if (Use_GPS) {
	  if(Initiate_Login (local_player_name))
	    GPS_Get_Message_of_the_Day ();
	}

	Moderator_Comment
	  ("TYPE /HELP FOR INSTRUCTIONS ABOUT THIS PROGRAM");

	setjmp (reset_buf);

#ifdef DEBUG	
        Moderator_Comment
	  ("WARNING!! THIS VERSION OF OKBRIDGE IS UNSTABLE AND MAY CRASH!!");
#endif

	if (server_mode_requested)
	  Setup_server ();
	else if (client_mode_requested)
	  Attempt_to_connect (local_player);
	else if (Use_GPS) {
	  if ((local_player_id > 0) && !GPS_unavailable) {
	    Clear_Status ();
	    Pause ("PRESS <ESC> TO SEE A LIST OF CURRENTLY PLAYING TABLES ...");
	    GPS_List_Tables (1);
	  }
	}

	hands_played = 0;
	need_to_save_context = 1;

	Initiate_Startup_Mode (Local_table);
	Begin_timer_for_hand ();

	while (1) {
	  do {
	    reset_in_progress = setjmp (reset_buf);
	    if (reset_in_progress) {
	      if (abort_flag)
		Terminate_Program ("RESET RECEIVED DURING QUIT!");
	      else
		Handle_reset (reset_in_progress);
	    }
	  } while (reset_in_progress);
	  main_event_loop ();
	}
}
 



