/* boards.h -- routines for manipulating boards.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

#ifndef CIPHER_INCLUDED
#include "cipher.h"
#endif

#ifndef SCORING_INCLUDED
#include "scoring.h"
#endif

#define BOARDS_INCLUDED

/* This module provides routines for manipulating the set of boards.
   Conceptually, the interface to this module is a local database for
   accessing and updating boards.  However, this module also contains
   routines for reading and writing boards from files and also
   routines for transmitting and receiving boards over the network.
*/

/* The following definitions record the information which we will store
   for each board and for the play of each board.  These structures are
   meant to be robust enough so that the entire state of the game can
   be reconstructed from them.
*/

typedef struct Board_struct {
  /* First, we must provide some information describing where this
     board came from: */
  char *source;           /* A character string identifying the source
			     of this board -- perhaps a filename or an
			     internet name. */
  int  serial_no;         /* The number of the board within the file. */
  char *startup_comment;  /* a comment which will be displayed when the
			     hand is first displayed. */

  /* The following information describes the board itself: */
  int  dealer;             /* the dealer for this hand. */
  int  vulnerable [2];     /* boolean flags for each side. */
  int  scoring_mode;       /* the scoring mode used when this hand was
			      generated (and played). */
  int  part_score [2];     /* A part score carried over from a previous
			      hand(s), if relevant. */
  hand deal;               /* the cards. */

  int  mps_computed;       /* true if the match point scores have been
			      computed for this board. */
  int  imps_computed;      /* true if the imp scores have been computed
			      for this board. */

  /* Then we need info about the people who have played this board: */
  struct Play_record_struct *play_records;

  struct Board_struct *next;
} Board, *Board_ptr;

typedef struct Play_record_struct {
  /* The following provides the minimal amount of information needed to
     represent a hand. */
  char *player_names[4];   /* The names of each of the players. */
  void *player_ids[4];     /* Possible pointers to structures which give
			      further info about the players. */
  int  table;              /* The table number where this board was played,
			      or -1 not played locally. */
  int  no_bids;            /* The number of actual bids that were made. */
  bid_list bids;           /* The list of bids. */
  bid_list alerts;         /* For each bid, a nonzero value if the bid was
			      alerted. */
  int  no_plays;           /* The number of plays. */
  deal play_list;          /* A list of the cards played, in order.
			      The person who played a card play_list[i]
			      can be found as deal[play_list[i]].  In
			      particular, the leader for the k-th trick
			      is deal[play_list[4*k]]. */
  int  bidding_completed;  /* A boolean flag indicating that the bidding
			      was completed. (not strictly necessary) */
  int  hand_completed;     /* A boolean flag indicating that the play
			      was completed. */

  int  next_player;        /* The index of the next person who should make
			      a bid or play. */
  int  result;             /* If the contract was made, the number of
			      tricks-6 taken by the declaring side. OW,
			      the negative of the number of tricks down. */
  int  tricks[2];          /* Number of tricks taken by each side. */
  Score score[2];          /* The score for this hand. */

  /* The following information is additional info. which is implied
     by the above: */
  int  contract;           /* The contract, but without doubling info. */
  int  doubled;            /* 0=not doubled, 1=doubled, 2=redoubled. */
  int  declarer;
  int  mimp_points [2];    /* 2 * The number of MIMP points awarded. */
  Score match_points[2];   /* The matchpoint score for this hand.  */
  Score imatch_points[2];  /* The IMP score for this hand. */
  
  struct Play_record_struct *next;
} Play_record, *Play_record_ptr;

  
#ifdef _BOARDS_

  Board *Played_boards = NULL;
    /* The list of boards which have already been played. */
  Board *Played_boards_tail = NULL;
    /* The last element on the list of played boards. */

  Board *Unplayed_boards = NULL;
    /* The list of boards which are yet to be played. */

#else

  extern Board *Played_boards, *Played_boards_tail, *Unplayed_boards;

#endif

/* The following primitives are for manipulating the boards and play
   records. */

Board *Allocate_board ();
/* Returns a pointer to a newly allocated board.  Does not fill in any
   of the fields. */

void Deallocate_board ();
/* Deallocates the board b.  This includes freeing up strings which are
   part of the structure and de-allocating the play records associated to
   this board. */

Play_record *Allocate_play_record ();
/* Returns a pointer to a newly allocated play record.  Does not fill in any
   of the fields. */

void Deallocate_play_record ();
/* Deallocates the play record p.  This includes freeing up strings
   which are part of the structure. */

void Erase_board_list ();
/* Deallocates all of the boards in the list. */

void Append_board ();
/* Appends the board b to the list of unplayed boards. */

void Append_play_record ();
/* Appends the play record p to the list of play records associated to the
   board b. */

void Encode_board ();
/* Encodes the board in the string array buf1, buf2.  This encoding does not 
   include the play records. */

Board *Decode_board ();
/* Decodes the string array buf and returns a corresponding board structure. */

void Encode_play_record ();
/* Encodes the play_record p into buffers buf1 .. buf3. */

Play_record *Decode_play_record ();
/* Decodes the strings buf1 .. buf3, returning a corresponding play record
   structure.  Returns NULL if an error occurs in the decoding.  */


/* Primitives for accessing email duplicate files: */

int Read_Email_Header ();
/* Searches the file f until an email header is found.  If the header
   is found, then returns 0.  Otherwise, returns 1. */

void Write_Email_Header ();
/* Writes an email header to the file f. */


int Read_Email_Board ();
/* Reads a board from the file f using the cipher descriptor c.
   Stores in b a pointer to an allocated board record containing the
   data which has been read.  If successful, returns 0.  If the end of 
   file is reached, returns 1.  If some other error occurs, returns -1.
*/

void Write_Email_Board ();
/* Writes the board b to the file f using the cipher descriptor c. */

int Load_Email_Duplicate_File ();
/* First erases all of the current boards.
   Then, reads an entire email duplicate file, recording each of the hands 
   read from the file into the current list of unplayed boards. 
   If successful, returns 0.  If the header could not be found,
   returns 1.  If some other error occurs, returns -1.
*/

int Merge_Email_Duplicate_File ();
/* Reads an email duplicate file.  For each board found in the file,
   if the same board occurs among the list of boards, then updates
   the data associated with that board using the data read from the file.
   Discards boards which do not match those stored in memory.
   If successful, returns 0.  If the header could not be found, returns 1.
   If some other error occurs, returns -1.
*/


void Write_Email_Duplicate_File ();
/* Writes the list of played and unplayed boards to the file f. */


/* Routines for accessing and updating the current set of email 
   duplicate boards. */

Board *Generate_Random_Match_Board ();
/* Generates a random match board for the given scoring mode,
   and returns a pointer to it.  If b is non-null, then uses the
   dealer and vulnerability information in b to determine who the
   dealer and vulnerabilities in the generated board.  This is
   done by looking up the dealer and vulnerabilities in a table
   (defined in types.h) and then picking the cyclically next
   entry in the table.
*/

Board *Generate_Random_Rubber_Board ();
/* Generates a random rubber board and returns a pointer to it,
   based upon the part score and result recorded in the previously
   played board b and play record p.  If b or p is NULL, then
   generates a board with neither side vulnerable and a 0 part score.
*/

Play_record *Search_for_play_record ();
/* Searches the list of play records associated with the board b for a
   record of play by the same players as mentioned in p.  If such a
   record is found, then returns a pointer to it.  Otherwise, returns
   NULL. */

Play_record *Locate_play_record_by_foursome 
  ();
/* Searches the list of play records associated with the board b for a record
   where the names of the players exactly match n, e, s, and w, in order.
   Returns the play record if found, or NULL if no matching record is found. 
*/


void Clear_All_Boards ();
/* Clears the list of played and unplayed boards. */

int Board_is_Available ();
/* Returns true if there is a board available in the list of unplayed
   boards. */

Board *Next_Unplayed_Board ();
/* If there is a board on the list of unplayed boards, then unlinks it and
   returns it.  Otherwise, returns NULL. */

void Record_Played_Board ();
/* Appends the board to the list of played boards. */


/* Routines for scoring play records: */

int Highcard_points ();
/* Returns the number of highcard points held by the partnership 'side'. */

extern int Honors_Point_Bonus ();
/* Returns the number of bonus points awarded in a rubber contract
   for holding many honors in one hand. */

void Compute_Matchpoints ();
/* Computes the match point totals for the current board. */

void Compute_Intl_Matchpoints ();
/* Computes the internation match point totals for the current board. */

void Compute_MIMP_points ();
/* Computes the number of MIMP points for the play record p, based upon
   the scoring information found in the play record.  In addition,
   we compute simulated match_point and imp values for the play record
   which are based upon the number of MIMP's.
*/

void Compute_Scores_for_Play_Record ();
/* Computes the scores for the play record p.  This includes setting
   the score, mimp_points, match_points, and imatch_points fields
   appropriately.
*/

void Compute_Scores_for_Board ();
/* Computes the matchpoint and imp scores for each of the play records
   in the board b. 
*/

void Compute_contract ();
/* Computes the contract for the play record p based upon the bids present
   in the bidding list. */

int Index_of_Last_Bid ();
/* Returns the index in the bid list p of the last bid made by the given
   player, or -1 if the player has not made any bids.
*/

void Generate_valid_bids ();
/* Determines the valid bids for the player, based on the information
   contained in the play record p.  Assumes the declarer, contract
   and doubled fields of p have been correctly computed from the bid
   list.
*/

void Generate_holdings ();
/* Records in the hand h the cards held by the given player, based on the
   deal contained in the board b and on the past no_plays recorded plays. 
   The holdings of the player are recorded as a boolean string in h, so that
     h[c] == true iff the player holds the card c.
*/

void Generate_valid_leads ();
/* Records in the hand h the set of cards which the player may lead,
   based upon the plays which have already been recorded in p.
   The valid plays are stored as a boolean string in h, so that
     h[c] == true iff it is legal to play card c.
*/

void Generate_valid_follows 
  ();
/* Records in the hand h the set of cards which the player may play to
   follow the given lead, based upon the plays which have already been
   recorded in p.

   The valid plays are stored as a boolean string in h, so that
     h[c] == true iff it is legal to play card c.
*/

int Next_Player ();
/* Returns the index of the next person who should play, or -1 if the
   hand has been completed.  If it is the dummy's play, then the
   index of the declarer is returned.
*/

int Winning_card ();
/* Returns the index of the player who played the winning card from
 * the cards c_l, c_lho, c_p, c_rho, based on the given trump suit
 * and that c_l was the card lead.
 */
