/* boards.c -- routines for manipulating boards.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

/* This module provides routines for manipulating the set of boards.
   Conceptually, the interface to this module is a local database for
   accessing and updating boards.  However, this module also contains
   routines for reading and writing boards from files and also
   routines for transmitting and receiving boards over the network.
*/

#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <string.h>

#define _BOARDS_

#include "types.h"
#include "cipher.h"
#include "boards.h"

#ifdef GCC
extern sscanf (), strcasecmp (), fgetc (), fprintf (), bcopy ();
extern time_t time ();
#endif

extern char *month_names[];
extern char *malloc ();
extern char *strdup ();
extern void free ();
extern void qsort ();

#ifdef HPUX
#define RANDOM rand
#define SRANDOM srand
#endif

#ifdef VMS
#define RANDOM rand
#define SRANDOM srand
#endif

#ifdef SCO
#define RANDOM rand
#define SRANDOM srand
#endif

#ifdef XENIX
#define RANDOM rand
#define SRANDOM srand
#endif

#ifndef RANDOM
#define RANDOM random
#define SRANDOM srandom
#endif

extern long RANDOM ();

#define random_mod(n) ((RANDOM()/64) % (n))

int Header_has_been_generated = 0;
char Email_header_buffer[80];

char *local_source = NULL;
  /* The source to which locally generated boards will be attributed. */
int   local_seq_no = 0;
  /* The sequence number which will be assigned to the next locally
     generated board. */

int random_seed_generated = 0;
  /* TRUE if a pseudo-random seed has been generated for shuffling the 
     boards. */

static Board *board_freelist = NULL;
static Play_record *play_record_freelist = NULL;

/* The following primitives are for manipulating the boards and play
   records. */

Board *Allocate_board ()
/* Returns a pointer to a newly allocated board.  Does not fill in any
   of the fields. */
{
  Board *b;

  if (board_freelist == NULL)
    b = (Board *) malloc(sizeof(Board));
  else {
    b = board_freelist;
    board_freelist = b->next;
  }

  b->source = NULL;
  b->serial_no = 0;
  b->part_score[0] = b->part_score[1] = 0;
  b->mps_computed = b->imps_computed = 0;
  b->startup_comment = NULL;
  b->play_records = NULL;

  return (b);
}


void Deallocate_board (b)
     Board *b;
/* Deallocates the board b.  This includes freeing up strings which are
   part of the structure and de-allocating the play records associated to
   this board. */
{
  Play_record *p;

  if (b->source != NULL) free(b->source);
  if (b->startup_comment != NULL) free (b->startup_comment);

  while (b->play_records != NULL) {
    p = b->play_records;
    b->play_records = p->next;
    Deallocate_play_record (p);
  }

  b->next = board_freelist;
  board_freelist = b;

}


Play_record *Allocate_play_record (n, e, s, w)
     char *n;
     char *e;
     char *s;
     char *w;
/* Returns a pointer to a newly allocated play record.  Does not fill in any
   of the fields. */
{
  Play_record *p;

  if (play_record_freelist == NULL)
    p = (Play_record *) malloc(sizeof(Play_record));
  else {
    p = play_record_freelist;
    play_record_freelist = p->next;
  }

  p->player_names[0] = NULL;
  p->player_names[1] = NULL;
  p->player_names[2] = NULL;
  p->player_names[3] = NULL;

  p->player_ids[0] = NULL;
  p->player_ids[1] = NULL;
  p->player_ids[2] = NULL;
  p->player_ids[3] = NULL;

  if (n != NULL)
    p->player_names[PLAYER_NORTH] = strdup (n);
  if (e != NULL)
    p->player_names[PLAYER_EAST]  = strdup (e);
  if (s != NULL)
    p->player_names[PLAYER_SOUTH] = strdup (s);
  if (w != NULL)
    p->player_names[PLAYER_WEST]  = strdup (w);

  p->table = 0;
  p->no_bids = 0;
  p->no_plays = 0;
  p->bidding_completed = 0;
  p->hand_completed = 0;
  p->contract = p->doubled = p->declarer = 0;
  p->result = 0;
  p->tricks[0] = p->tricks[1] = 0;
  bzero (&p->score[0], sizeof(Score));
  bzero (&p->score[1], sizeof(Score));
  bzero (&p->match_points[0], sizeof(Score));
  bzero (&p->match_points[1], sizeof(Score));
  bzero (&p->imatch_points[0], sizeof(Score));
  bzero (&p->imatch_points[1], sizeof(Score));
  p->mimp_points[0] = p->mimp_points[1] = 0;
  p->next = 0;

  return (p);
}

void Deallocate_play_record (p)
     Play_record *p;
/* Deallocates the play record p.  This includes freeing up strings
   which are part of the structure. */
{
  if (p->player_names[0] != NULL) free (p->player_names[0]);
  if (p->player_names[1] != NULL) free (p->player_names[1]);
  if (p->player_names[2] != NULL) free (p->player_names[2]);
  if (p->player_names[3] != NULL) free (p->player_names[3]);

  p->next = play_record_freelist;
  play_record_freelist = p;

}


void Erase_board_list (list)
     Board **list;
/* Deallocates all of the boards in the Board_list. */
{
  Board *b;

  while (*list != NULL) {
    b = *list;
    *list = b->next;
    Deallocate_board (b);
  }
}

void Append_board (list, b)
     Board **list;
     Board *b;
/* Appends the board b to the list of unplayed boards. */
{
  Board *l;

  if (*list == NULL)
    *list = b;
  else {
    for (l = *list; l->next != NULL; l = l->next);
    l->next = b;
  }
  b->next = NULL;
}

void Append_play_record (b, p)
     Board *b;
     Play_record *p;
/* Appends the play record p to the list of play records associated to the
   board b. */
{
  Play_record *q;

  if (b->play_records == NULL)
    b->play_records = p;
  else {
    for (q = b->play_records; q->next != NULL; q = q->next);
    q->next = p;
  }
  p->next = NULL;
}

static int  Locate_field (buf, field_no)
     char *buf;
     int field_no;
/* Returns the index in buf of the first character of field_no. */
{
  int i, n;

  for (i = 0; (buf[i] == ' ') || (buf[i] == '\t'); i++);
  n = 1;
  while ((buf[i] != '\0') && (n < field_no)) {
    while ((buf[i] != ' ') && (buf[i] != '\t') && (buf[i] != '\0')) i++;
    while ((buf[i] == ' ') || (buf[i] == '\t')) i++;
    n++;
  }
  return (i);
}

void Encode_board (b, buf1, buf2)
     Board *b;
     char *buf1;
     char *buf2;
/* Encodes the board in the string array buf1, buf2.  This encoding does not 
   include the play records. */
{
  int i;

  sprintf (buf1, "%d %d %d %d %s %s", b->serial_no, 
	   b->scoring_mode, b->part_score[0], b->part_score[1],
	   b->source, (b->startup_comment == NULL)? " ": b->startup_comment);
  buf2[0] = b->dealer + '0';
  buf2[1] = b->vulnerable[0] + '0';
  buf2[2] = b->vulnerable[1] + '0';
  for (i = 0; i < 52; i++)
    buf2[i+3] = b->deal[i] + '0';
  buf2[52+3] = '\0';
}


Board *Decode_board (buf1, buf2)
     char *buf1;
     char *buf2;
/* Decodes the string array buf and returns a corresponding board structure. */
{
  Board *b;
  char source_buf[80];
  int i, comment_index;

  b = Allocate_board ();
  sscanf (buf1, "%d %d %d %d %s", &(b->serial_no), &(b->scoring_mode),
	  &(b->part_score[0]), &(b->part_score[1]),
	  source_buf);
  comment_index = Locate_field (buf1, 6);
  b->source = strdup (source_buf);
  b->startup_comment = strdup (buf1 + comment_index);
  
  b->dealer = buf2[0] - '0';
  b->vulnerable[0] = buf2[1] - '0';
  b->vulnerable[1] = buf2[2] - '0';
  for (i = 0; i < 52; i++)
    b->deal[i] = buf2[i+3] - '0';

  return (b);
}


void Encode_play_record (p, buf1, buf2, buf3)
     Play_record *p;
     char *buf1;
     char *buf2;
     char *buf3;
/* Encodes the play_record p into buffers buf1 .. buf3. */
{
  int i, j;

  sprintf (buf1, "%d %d %d %d %d %d %d %d %d %s %s %s %s",
	   p->bidding_completed, p->hand_completed, p->result,
	   p->tricks[0], p->tricks[1],
	   p->score[0].rubber.above_line,
	   p->score[1].rubber.above_line,
	   p->score[0].rubber.below_line,
	   p->score[1].rubber.below_line,
/*	   p->above_line[0], p->above_line[1], 
	   p->below_line[0], p->below_line[1], */
	   p->player_names[0], p->player_names[1],
	   p->player_names[2], p->player_names[3]);

  for (i = j = 0; i < p->no_bids; i++) {
    buf2[j++] = p->bids[i] + 'A';
    if (p->alerts[i] & (1 << ALERT_BY_PARTNER)) buf2[j++] = '!';
    if (p->alerts[i] & (1 << ALERT_BY_BIDDER)) buf2[j++] = '*';
  }
  buf2[j++] = '\0';

  for (i = 0; i < p->no_plays; i++)
    buf3[i] = (p->play_list[i] < 26)? 
      'a' + p->play_list[i]: 'A' + p->play_list[i]-26;
  buf3[p->no_plays] = '\0';
  
}

Play_record *Decode_play_record (buf1, buf2, buf3)
     char *buf1;
     char *buf2;
     char *buf3;
/* Decodes the strings buf1 .. buf3, returning a corresponding play record
   structure.  Returns NULL if an error occurs in the decoding.  */
{
  Play_record *p;
  int i, n;
  char p0[15], p1[15], p2[15], p3[15];

  p = Allocate_play_record (NULL, NULL, NULL, NULL);
  n = sscanf (buf1, "%d %d %d %d %d %d %d %d %d %s %s %s %s",
	      &(p->bidding_completed), &(p->hand_completed), &(p->result),
	      &(p->tricks[0]), &(p->tricks[1]), 
	      &p->score[0].rubber.above_line,
	      &p->score[1].rubber.above_line,
	      &p->score[0].rubber.below_line,
	      &p->score[1].rubber.below_line,
/*	      &(p->above_line[0]), &(p->above_line[1]), 
	      &(p->below_line[0]), &(p->below_line[1]),
*/
	      p0, p1, p2, p3);

  if (n < 9) {
    Deallocate_play_record (p);
    return (NULL);
  }

  p->player_names[0] = strdup(p0);
  p->player_names[1] = strdup(p1);
  p->player_names[2] = strdup(p2);
  p->player_names[3] = strdup(p3);

  for (i = p->no_bids = 0; buf2[i] != '\0';) {
    p->bids[p->no_bids] = buf2[i++] - 'A';
    p->alerts[p->no_bids] = 0;
    if ((p->bids[p->no_bids] < 0) || (37 < p->bids[p->no_bids])) {
      Deallocate_play_record (p);
      return (NULL);
    }
    if (buf2[i] == '!') {
      p->alerts[p->no_bids] |= 1 << ALERT_BY_PARTNER;
      i++;
    } 
    if (buf2[i] == '*') {
      p->alerts[p->no_bids] |= 1 << ALERT_BY_BIDDER;
      i++;
    }
    p->no_bids++;
  }
  
  p->no_plays = strlen(buf3);
  for (i = 0; i < p->no_plays; i++)
    if (('a' <= buf3[i]) && (buf3[i] <= 'z'))
      p->play_list[i] = buf3[i] - 'a';
    else if (('A' <= buf3[i]) && (buf3[i] <= 'Z'))
      p->play_list[i] = buf3[i] - 'A' + 26;
    else {
      Deallocate_play_record (p);
      return (NULL);
    }

  return (p);
}


/* Routines for searching boards and play records. */

static Board *Search_for_board (list, b)
     Board *list;
     Board *b;
/* Searches the current set of email duplicate boards for a board which
   matches b.  Returns a pointer to the matching board if it is found,
   or NULL otherwise. */
{
  int j, matches;
  Board *c;

  for (c = list; c != NULL; c = c->next) {
    matches = 1;
    for (j = 0; (j < 52) && matches; j++)
      if (b->deal[j] != c->deal[j])
	matches = 0;
    if (matches)
      return (c);
  }
  return (NULL);
}

Play_record *Search_for_play_record (b, p)
     Board *b;
     Play_record *p;
/* Searches the list of play records associated with the board b for a
   record of play by the same players as mentioned in p.  If such a
   record is found, then returns a pointer to it.  Otherwise, returns
   NULL. */
{
  Play_record *q;
  int i, match;

  for (q = b->play_records; q != NULL; q = q->next) {
    match = 1;
    for (i = 0; (i < 4) && match; i++)
      if (strcasecmp(p->player_names[i], q->player_names[i]))
	match = 0;
    if (match)
      return (q);
  }
  return (NULL);
}

Play_record *Locate_play_record_by_foursome 
  (b, n, e, s, w)
     Board *b;
     char *n;
     char *e;
     char *s;
     char *w;
/* Searches the list of play records associated with the board b for a record
   where the names of the players exactly match n, e, s, and w, in order.
   Returns the play record if found, or NULL if no matching record is found. 
*/
{
  Play_record *p;

  for (p = b->play_records; p != NULL; p = p->next) {
    if (!strcmp(p->player_names[PLAYER_NORTH], n) &&
	!strcmp(p->player_names[PLAYER_EAST], e)  &&
	!strcmp(p->player_names[PLAYER_SOUTH], s) &&
	!strcmp(p->player_names[PLAYER_WEST], w))
      return (p);
  }
  return (NULL);

}

/* Primitives for accessing email duplicate files: */

static void Generate_email_header ()
{
  if (Header_has_been_generated)
    return;

  sprintf (Email_header_buffer,"%s %s",
	   "This is an email duplicate file for OKBridge version",
	   major_revision_level);
  Header_has_been_generated = 1;
}

static int Get_line (f, buf, buflen)
     FILE *f;
     char *buf;
     int buflen;
/* Reads a line from file f, placing up to buflen-1 characters into buf,
   terminated by a '\0'.  Returns the number of characters actually read
   or -1 if EOF.
*/
{
	int n, ch;

	if (fgets(buf, buflen, f) == NULL)
	  return (-1);

	n = strlen(buf);
	if ((n >= buflen) && (buf[n-1] != '\n'))
	  do
	    { ch = fgetc (f); }
	  while ((ch != '\n') && (ch != EOF));
	else
	  buf[--n] = '\0';

	return (n);
}

int Read_Email_Header (f)
     FILE *f;
/* Searches the file f until an email header is found.  If the header
   is found, then returns 0.  Otherwise, returns 1. */
{
  char buf[100];

  Generate_email_header ();
  while (Get_line (f, buf, 100) >= 0) {
    if (!strcmp(buf, Email_header_buffer))
      return (0);
  }

  return (1);
}

void Write_Email_Header (f)
     FILE *f;
/* Writes an email header to the file f. */
{
  Generate_email_header ();
  fprintf (f, "%s\n", Email_header_buffer);
}


int Read_Email_Board (f, c, b)
     FILE *f;
     Cipher *c;
     Board **b;
/* Reads a board from the file f using the cipher descriptor c.
   Stores in b a pointer to an allocated board record containing the
   data which has been read.  If successful, returns 0.  If the end of 
   file is reached, returns 1.  If some other error occurs, returns -1.
*/
{
  char buf1[100], buf2[100], buf3[100];
  Play_record *p;

  if (Read_Ciphered_Line (f, c, buf1, 100) < 0)
    return (1);

  if (Read_Ciphered_Line (f, c, buf2, 100) != 55)
    return (-1);

  *b = Decode_board (buf1, buf2);
  (*b)->play_records = NULL;

  if (Read_Ciphered_Line (f, c, buf1, 100) < 0) return (-1);
  while (strcmp(buf1, "--")) {
    if (Read_Ciphered_Line (f, c, buf2, 100) < 0) return (-1);
    if (Read_Ciphered_Line (f, c, buf3, 100) < 0) return (-1);
    if (!strcmp(buf2, "--") || !strcmp(buf3, "--")) return (-1);

    p = Decode_play_record (buf1, buf2, buf3);
    if (p == NULL)
      return (-1);

    Compute_contract (*b, p);
    if (p->hand_completed)
      Compute_Scores_for_Play_Record (*b, p);

    p->next = (*b)->play_records;
    (*b)->play_records = p;
    if (Read_Ciphered_Line (f, c, buf1, 100) < 0) return (-1);
  }

  Compute_Scores_for_Board (*b);
  return (0);
}

void Write_Email_Board (f, c, b)
     FILE *f;
     Cipher *c;
     Board *b;
/* Writes the board b to the file f using the cipher descriptor c. */
{
  char buf1[100], buf2[100], buf3[100];
  Play_record *p;

  Encode_board (b, buf1, buf2);
  Write_Ciphered_Line (f, c, buf1);
  Write_Ciphered_Line (f, c, buf2);

  for (p = b->play_records; p != NULL; p = p->next) {
    Encode_play_record (p, buf1, buf2, buf3);
    Write_Ciphered_Line (f, c, buf1);
    Write_Ciphered_Line (f, c, buf2);
    Write_Ciphered_Line (f, c, buf3);
  }
  fprintf (f, "--\n");
}

static void Copy_play_record (p, q)
     Play_record *p;
     Play_record *q;
/* Copies the contents of the play record p to q, without destroying
   q's next field. */
{
  Play_record *r = q->next;
  int i;

  for (i = 0; i < 4; i++)
    if (q->player_names[i] != NULL)
      free (q->player_names[i]);

  bcopy (p, q, sizeof(Play_record));
  q->next = r;
  free (p);
}

void Merge_play_record (b, p)
     Board *b;
     Play_record *p;
/* Links p into the list of play records for the board b.  If b already
   has a record of play for the players listed in p, then the older record
   is deleted.
*/
{
  Play_record *q;
  int i, match;

  for (q = b->play_records; q->next != NULL; q = q->next) {
    match = 1;
    for (i = 0; (i < 4) && match; i++)
      if (strcasecmp(p->player_names[i], q->player_names[i]))
	match = 0;
    if (match) {
      if (p->hand_completed)
	Copy_play_record (p, q);
      else if (p->bidding_completed && !q->hand_completed)
	Copy_play_record (p, q);
      else if (!q->bidding_completed)
	Copy_play_record (p, q);
      else
	Deallocate_play_record (p);
      return;
    }
  }

}

static int Merge_board (list, b)
     Board *list;
     Board *b;
/* Searches the board list for a board matching b.  If one is found,
   then destructively merges the play records in b with those in the 
   matching record.  If no matching board is found, then does nothing.
   Returns 0 if successful, and 1 if no match is found.
*/
{
  Board *c;
  Play_record *p, *q;

  c = Search_for_board (list, b);
  if (c == NULL)
    return (1);

  p = b->play_records;
  while (p != NULL) {
    q = p->next;
    p->next = NULL;
    Merge_play_record (c, p);
    p = q;
  }
  b->play_records = NULL;
  Deallocate_board (b);

  return (0);
}

int Load_Email_Duplicate_File (f)
     FILE *f;
/* Reads an entire email duplicate file, recording each of the hands 
   read from the file into the current list of boards. */
{
  Board *b, *tail;
  Cipher c;
  int status;

  if (Read_Email_Header (f))
    return (1);

  if (Read_Cipher_Descriptor(f, &c))
    return (-1);

  tail = Unplayed_boards;
  if (tail != NULL)
    while (tail->next != NULL) tail = tail->next;

  status = Read_Email_Board (f, &c, &b);
  while (status == 0) {
    if ((Unplayed_boards == NULL) || Merge_board (Unplayed_boards, b)) {
      if (tail == NULL) {
	Unplayed_boards = tail = b;
      } else {
	tail->next = b;
	tail = b;
      }
      b->next = NULL;
    }
    status = Read_Email_Board (f, &c, &b);
  }

  if (status == 1)
    return (0);
  else
    return (1);
}

int Merge_Email_Duplicate_File (f, list)
     FILE *f;
     Board *list;
/* Reads an email duplicate file.  For each board found in the file,
   if the same board occurs among the list of boards, then updates
   the data associated with that board using the data read from the file.
   Discards boards which do not match those stored in memory.
   If successful, returns 0.  If the header could not be found, returns 1.
   If some other error occurs, returns -1.
*/
{
  Board *b;
  Cipher c;
  int status;

  if (Read_Email_Header (f))
    return (1);

  if (Read_Cipher_Descriptor(f, &c))
    return (-1);

  status = Read_Email_Board (f, &c, &b);
  while (status == 0) {
    if (!Merge_board (list, b))
      Deallocate_board (b);
    status = Read_Email_Board (f, &c, &b);
  }

  if (status == 1)
    return (0);
  else
    return (1);
}

void Write_Email_Duplicate_File (f)
     FILE *f;
/* Writes the list of played and unplayed boards to the file f. */
{
  Board *b;
  Cipher c;

  Write_Email_Header (f);
  Create_Cipher_Descriptor (f, &c);
  Write_Cipher_Descriptor (f, &c);

  for (b = Played_boards; b != NULL; b = b->next)
    Write_Email_Board (f, &c, b);

  for (b = Unplayed_boards; b != NULL; b = b->next)
    Write_Email_Board (f, &c, b);
  
}


/* Routines for accessing and updating the current set of email 
   duplicate boards. */


void Generate_local_source_name ()
{
  char buf[20];
  time_t current_time;
  struct tm *decoded_time;

  time (&current_time);
  decoded_time = localtime (&current_time);
  sprintf (buf, "%s%0d", month_names[decoded_time->tm_mon],
	   decoded_time->tm_mday);
  local_source = strdup(buf);

}

static void shuffle_the_deck (cards)
	deal cards;
/* Using the algorithm suggested by Douglas Foxvog.  Thanks, Doug! */
{
	int i, t, c;
	deal shuffle;

	if (!random_seed_generated) {
	  SRANDOM(time(NULL));
	  random_seed_generated = 1;
	}

	for (i = 0; i < 52; i++) 
		shuffle [i] = i;
	for (i = 0; i < 51; i++) {
		c = random_mod (52 - i);
		t = shuffle[i+c]; 
		shuffle[i+c] = shuffle[i];
		shuffle[i] = t;
	};
	for (i = 0; i < 52; i++)
		cards[shuffle[i]] = (i % 4);
		
}

Board *Generate_Random_Match_Board (scoring, prevboard)
     int scoring;
     Board *prevboard;
/* Generates a random match board and appends it to the Board_list. */
{
  int r;
  Board *b;

  if (local_source == NULL)
    Generate_local_source_name ();

  b = Allocate_board ();

  b->source = strdup (local_source);
  b->serial_no = ++local_seq_no;

  if (prevboard == NULL)
    r = local_seq_no + ROTATION_LENGTH - 1;
  else {
    for (r = 0; r < ROTATION_LENGTH; r++)
      if ((prevboard->dealer == dealer_list[r]) &&
	  (prevboard->vulnerable[SIDE_NS] == ns_vulnerability_list[r]) &&
	  (prevboard->vulnerable[SIDE_EW] == ew_vulnerability_list[r]))
	break;
    r += 1;
  }
  r %= ROTATION_LENGTH;

  b->dealer = dealer_list[r];
  b->vulnerable[SIDE_NS] = ns_vulnerability_list[r];
  b->vulnerable[SIDE_EW] = ew_vulnerability_list[r];
  b->scoring_mode = scoring;
  shuffle_the_deck (b->deal);
  
  return (b);
}

Board *Generate_Random_Rubber_Board (prevboard, prevplay)
     Board *prevboard;
     Play_record *prevplay;
/* Generates a random rubber board and returns a pointer to it,
   based upon the part score and result recorded in the previously
   played board b and play record p.  If b or p is NULL, then
   generates a board with neither side vulnerable and a 0 part score.
*/
{
  int r;
  Board *b;

  if (local_source == NULL)
    Generate_local_source_name ();

  b = Allocate_board ();

  b->source = strdup (local_source);
  b->serial_no = local_seq_no + 1;

  if ((prevboard == NULL) || (prevplay == NULL) ||
      (prevboard->scoring_mode != RUBBER_SCORING)) {
    r = local_seq_no++ % ROTATION_LENGTH;
    b->dealer = dealer_list [r];
    b->vulnerable[SIDE_NS] = b->vulnerable[SIDE_EW] = 0;
  } else {
    b->dealer = player_next [prevboard->dealer];
    b->vulnerable[SIDE_NS] = prevboard->vulnerable[SIDE_NS];
    b->vulnerable[SIDE_EW] = prevboard->vulnerable[SIDE_EW];
    b->part_score[SIDE_NS] = prevboard->part_score[SIDE_NS] +
      prevplay->score[SIDE_NS].rubber.below_line;
    b->part_score[SIDE_EW] = prevboard->part_score[SIDE_EW] +
      prevplay->score[SIDE_EW].rubber.below_line;
    if (b->part_score[SIDE_NS] >= 100) {
      if (prevboard->vulnerable[SIDE_NS])
	b->vulnerable[SIDE_NS] = b->vulnerable[SIDE_EW] = 0;
      else
	b->vulnerable[SIDE_NS] = 1;
      b->part_score[SIDE_NS] = b->part_score[SIDE_EW] = 0;
    } else if (b->part_score[SIDE_EW] >= 100) {
      if (prevboard->vulnerable[SIDE_EW])
	b->vulnerable[SIDE_EW] = b->vulnerable[SIDE_NS] = 0;
      else
	b->vulnerable[SIDE_EW] = 1;
      b->part_score[SIDE_NS] = b->part_score[SIDE_EW] = 0;
    }
  }

  b->scoring_mode = RUBBER_SCORING;
  shuffle_the_deck (b->deal);
  
  return (b);
}



void Clear_All_Boards ()
/* Clears the list of played and unplayed boards. */
{
  Erase_board_list (&Unplayed_boards);
  Erase_board_list (&Played_boards);

  Unplayed_boards = NULL;
  Played_boards = NULL;
  Played_boards_tail = NULL;
}

int Board_is_Available ()
/* Returns true if there is a board available in the list of unplayed
   boards. */
{
  return (Unplayed_boards != NULL);
}

Board *Next_Unplayed_Board ()
/* If there is a board on the list of unplayed boards, then unlinks it and
   returns it.  Otherwise, returns NULL. */
{
  Board *b;

  if (Unplayed_boards != NULL) {
    b = Unplayed_boards;
    Unplayed_boards = b->next;
    b->next = NULL;
  } else
    b = NULL;

  return (b);
}

void Record_Played_Board (b)
     Board *b;
/* Appends the board to the list of played boards. */
{
  if (Played_boards == NULL) {
    Played_boards = Played_boards_tail = b;
  } else {
    Played_boards_tail->next = b;
    Played_boards_tail = b;
  }
  Played_boards_tail->next = NULL;
}

/* Scoring routines: */


int Highcard_points (b, side)
     Board *b;
     int side;
/* Returns the number of highcard points held by the partnership 'side'. */
{
  int partner = side + 2;
  int i, hcp = 0;

  for (i = 0; i < 52; i++)
    if ((b->deal[i] == side) || (b->deal[i] == partner)) {
      if (rank_of(i) > 8)
	hcp += rank_of(i) - 8;
    }
  return (hcp);

}

static int NS_score (p)
     Play_record *p;
{
  return (p->score[SIDE_NS].duplicate.total - 
	  p->score[SIDE_EW].duplicate.total);
}

int score_compare (p1, p2)
     Play_record **p1;
     Play_record **p2;
{
  if (NS_score(*p1) < NS_score(*p2))
    return (-1);
  else if (NS_score(*p1) == NS_score(*p2))
    return (0);
  else
    return (1);
}

void Compute_Matchpoints (b)
     Board *b;
/* Computes the match point totals for the current board. */
{
  Play_record *Play_array[100], *p;
  int i, j, n, t, score_compare();

  b->mps_computed = 1;
  if ((b->play_records == NULL) || (b->play_records->next == NULL))
    return;

  n = 0;
  for (p = b->play_records; p != NULL; p = p->next) {
/*    p->match_points[0] = p->match_points[1] = 0.0; */
    if (p->hand_completed)
      Play_array[n++] = p;
  }

  qsort (Play_array, n, sizeof(Play_record *), score_compare);
  
  i = 0;
  while (i < n) {
    for (t = i; (t < n) && 
	 (NS_score(Play_array[i]) == NS_score(Play_array[t])); t++);
    for (j = i; j < t; j++) {
      Play_array[j]->match_points[0].duplicate.competitive.mp.mp_total 
	= t + i - 1;
      Play_array[j]->match_points[1].duplicate.competitive.mp.mp_total 
	= n + n - t - i - 1;
      Play_array[j]->match_points[0].duplicate.competitive.mp.mp_tables
	= Play_array[j]->match_points[1].duplicate.competitive.mp.mp_tables 
	  = n - 1;
    }
    i = t;
  }

}

void Compute_Intl_Matchpoints (b)
     Board *b;
/* Computes the internation match point totals for the current board. */
{
  Play_record *p, *q;
  int pdiff, qdiff;
  int contractor, defender;
  int no_tables = -1;

  b->imps_computed = 1;
  if (b->play_records == NULL)
    return;
  else if (b->play_records->next == NULL) {
    p = b->play_records;
    Compute_MIMP_points (b, p);
    return;
  }

  no_tables = 0;
  for (p = b->play_records; p != NULL; p = p->next)
    no_tables += 1;

  for (p = b->play_records; p != NULL; p = p->next) {
    p->imatch_points[0].duplicate.competitive.imp.imp_total 
      = p->imatch_points[1].duplicate.competitive.imp.imp_total = 0;
    p->imatch_points[0].duplicate.competitive.imp.imp_tables
      = p->imatch_points[1].duplicate.competitive.imp.imp_tables
	= no_tables - 1;

    contractor = side_of (p->declarer);
    defender   = 1 - contractor;
    if (!p->hand_completed)
      continue;
    pdiff = p->imatch_points[contractor].duplicate.total -
      p->imatch_points[defender].duplicate.total;
    for (q = b->play_records; q != NULL; q = q->next) {
      if (!q->hand_completed)
	continue;
      qdiff = q->imatch_points[contractor].duplicate.total -
	q->imatch_points[defender].duplicate.total;
      if (pdiff > qdiff)
	p->imatch_points[contractor].duplicate.competitive.imp.imp_total += 
	  IMP_rating (pdiff - qdiff);
      else if (pdiff < qdiff)
	p->imatch_points[contractor].duplicate.competitive.imp.imp_total -= 
	  IMP_rating (qdiff - pdiff);
    }
    p->imatch_points[defender].duplicate.competitive.imp.imp_total 
      = -p->imatch_points[contractor].duplicate.competitive.imp.imp_total;
  }
}

void Compute_contract (b, p)
     Board *b;
     Play_record *p;
/* Computes the contract for the play record p based upon the bids present
   in the bidding list. */
{
  int i, bidder, contractor;

  p->contract = BID_PASS;
  p->doubled  = 0;
  p->declarer = PLAYER_NORTH;

  if (p->no_bids == 0)
    return;

  i = p->no_bids - 1;
  switch (i%4) {
  case 0: bidder = b->dealer; break;
  case 1: bidder = player_next[b->dealer]; break;
  case 2: bidder = player_partner[b->dealer]; break;
  case 3: bidder = player_prev[b->dealer]; break;
  }

  do {
    if (p->bids[i] == BID_REDOUBLE)
      p->doubled = 2;
    else if (p->bids[i] == BID_DOUBLE) {
      if (!p->doubled)
	p->doubled = 1;
    } else if (p->bids[i] != BID_PASS) {
      p->contract = p->bids[i];
      contractor  = side_of (bidder);
      break;
    }
    bidder = player_prev[bidder];
    i = i - 1;
  } while (i >= 0);

  if (p->contract != BID_PASS) {
    bidder = b->dealer;
    for (i = 0; i < p->no_bids; i++) {
      if ((side_of(bidder) == contractor) && 
	  (trumpsuit_of(p->bids[i]) == trumpsuit_of(p->contract))) {
	p->declarer = bidder;
	break;
      } else
	bidder = player_next[bidder];
    }
  }
}



void Compute_MIMP_points (b, p)
     Board *b;
     Play_record *p;
/* Computes the number of MIMP points for the play record p, based upon
   the scoring information found in the play record.  In addition,
   we compute simulated match_point and imp values for the play record
   which are based upon the number of MIMP's.
*/
{
  int hcp, diff;
  int contractor = side_of (p->declarer);
  int level      = level_of (p->contract);
  int trumpsuit  = trumpsuit_of (p->contract);

  p->mimp_points[0] = p->mimp_points[1] = 0;
  if (!p->hand_completed)
    return;

  hcp = Highcard_points (b, side_of(p->declarer));
  if (p->result >= 0)
    p->mimp_points[side_of(p->declarer)] = 
      MIMP_score_made 
	(b->vulnerable[contractor], level, trumpsuit, 
	 p->doubled, p->result, hcp);
  else
    p->mimp_points[side_of(p->declarer)] = 
      MIMP_score_set 
	(b->vulnerable[contractor], level, trumpsuit, 
	 p->doubled, p->result, hcp);

  p->imatch_points[0].duplicate.competitive.imp.imp_total = 
    p->mimp_points[0];
  p->imatch_points[1].duplicate.competitive.imp.imp_total = 
    p->mimp_points[1];
  p->imatch_points[0].duplicate.competitive.imp.imp_tables =
    p->imatch_points[1].duplicate.competitive.imp.imp_tables = 1;

  p->match_points[0].duplicate.competitive.mp.mp_total = 
    p->match_points[1].duplicate.competitive.mp.mp_total = 0;
  p->match_points[0].duplicate.competitive.mp.mp_tables =
    p->match_points[1].duplicate.competitive.mp.mp_tables = 1;
    
  diff = p->mimp_points[0] - p->mimp_points[1];
  if (diff >= 1)
    p->match_points[0].duplicate.competitive.mp.mp_total = 2;
  else if (diff <= -1)
    p->match_points[1].duplicate.competitive.mp.mp_total = 2;
  else
    p->match_points[0].duplicate.competitive.mp.mp_total = 
      p->match_points[1].duplicate.competitive.mp.mp_total = 1;

}

int Honors_Point_Bonus (b, p)
     Board *b;
     Play_record *p;
/* Returns the number of bonus points awarded in a rubber contract
   for holding many honors in one hand. */
{
  int i;
  int dummy = player_partner [p->declarer];
  int trumps = trumpsuit_of (p->contract);
  int decl_honors = 0;   /* no of honors held by declarer. */
  int dummy_honors = 0;  /* no of honors held by dummy. */

  if (trumps != SUIT_NOTRUMP) {

    for (i = RANK_TEN; i <= RANK_ACE; i++)
      if (b->deal[card_code(trumps, i)] == p->declarer) 
	decl_honors++;
      else if (b->deal[card_code(trumps, i)] == dummy)
	dummy_honors++;

    if ((decl_honors == 5) || (dummy_honors == 5))
      return (150);
    else if ((decl_honors == 4) || (dummy_honors == 4))
      return (100);

  } else {
    /* In the case of a no trump contract, we count the total number of aces */
    for (i = 0; i < 4; i++)
      if (b->deal[card_code(i, RANK_ACE)] == p->declarer)
	decl_honors++;
      else if (b->deal[card_code(i, RANK_ACE)] == dummy)
	dummy_honors++;

    if ((decl_honors == 4) || (dummy_honors == 4))
      return (150);
  }
  return (0);
  
}

void Compute_Scores_for_Play_Record (b, p)
     Board *b; Play_record *p;
/* Computes the scores for the play record p.  This includes setting
   the score, mimp_points, match_points, and imatch_points fields
   appropriately.
*/
{
  int contractor = side_of (p->declarer);
  int defender   = 1 - contractor;
  int vul        = b->vulnerable[contractor];
  int level      = level_of (p->contract);
  int trumpsuit  = trumpsuit_of (p->contract);

  if (p->contract == BID_PASS)
    p->result = 0;
  else if (p->tricks[contractor] >= level + 6)
    p->result = p->tricks[contractor] - 6;
  else
    p->result = p->tricks[contractor] - level - 6;

  Compute_base_scores
    (b->scoring_mode, vul, level, trumpsuit, p->doubled, p->result,
     &p->score[contractor], &p->score[defender]);

  Compute_base_scores
    (MP_SCORING, vul, level, trumpsuit, p->doubled, p->result,
     &p->match_points[contractor], &p->match_points[defender]);

  Compute_base_scores
    (IMP_SCORING, vul, level, trumpsuit, p->doubled, p->result,
     &p->imatch_points[contractor], &p->imatch_points[defender]);

  Compute_MIMP_points (b, p);

  if (b->scoring_mode == RUBBER_SCORING) {
    if (p->result != 0)
      p->score[contractor].rubber.above_line +=  Honors_Point_Bonus (b, p);
    if (p->result > 0) {
      if (p->score[contractor].rubber.below_line 
	  + b->part_score[contractor] >= 100) {
	/* If we have won a game, we check to see if we have also won
	   the rubber. */
	if (b->vulnerable[contractor])
	  p->score[contractor].rubber.above_line += 
	    b->vulnerable[defender]? 500: 700;
      }
    }
  }


}

void Compute_Scores_for_Board (b)
     Board *b;
/* Computes the matchpoint and imp scores for each of the play records
   in the board b. 
*/
{
  Play_record *p;

  Compute_Intl_Matchpoints (b);
  Compute_Matchpoints (b);
  
  if (b->scoring_mode == MP_SCORING)
    for (p = b->play_records; p != NULL; p = p->next) {
      bcopy (&p->match_points[0], &p->score[0], sizeof(Score));
      bcopy (&p->match_points[1], &p->score[1], sizeof(Score));
    }
  else if (b->scoring_mode == IMP_SCORING)
    for (p = b->play_records; p != NULL; p = p->next) {
      bcopy (&p->imatch_points[0], &p->score[0], sizeof(Score));
      bcopy (&p->imatch_points[1], &p->score[1], sizeof(Score));
    }
}

int Index_of_Last_Bid (b, p, player)
     Board *b;
     Play_record *p;
     int player;
/* Returns the index in the bid list p of the last bid made by the given
   player, or -1 if the player has not made any bids.
*/
{
  int index, bidder;

  index = p->no_bids - 1;

  switch (p->no_bids % 4) {
  case 1: bidder = b->dealer; break;
  case 2: bidder = player_next[b->dealer]; break;
  case 3: bidder = player_partner[b->dealer]; break;
  case 0: bidder = player_prev[b->dealer];
  }

  while ((index >= 0) && (bidder != player)) {
    index -= 1;
    bidder = player_prev[bidder];
  }
  
  return (index);
}


void Generate_valid_bids (p, player, minimum_bid, double_ok, redouble_ok)
     Play_record *p;
     int player;
     int *minimum_bid;
     int *double_ok;
     int *redouble_ok;
/* Determines the valid bids for the player, based on the information
   contained in the play record p.  Assumes the declarer, contract
   and doubled fields of p have been correctly computed from the bid
   list.
*/
{
  if (p->contract == BID_PASS)
    *minimum_bid = 3;   /* The code for the 1C bid. */
  else
    *minimum_bid = p->contract + 1;

  if (side_of(player) == side_of(p->declarer)) {
    *double_ok = 0;
    if (p->doubled == 1)
      *redouble_ok = 1;
    else
      *redouble_ok = 0;
  } else {
    *redouble_ok = 0;
    *double_ok = 0;
    if (!p->doubled && (p->contract != BID_PASS))
      *double_ok = 1;
  }
}

void Generate_holdings (b, plays, no_plays, player, h)
     Board *b;
     card_type *plays;
     int no_plays;
     int player;
     card_type *h;
/* Records in the hand h the cards held by the given player, based on the
   deal contained in the board b and on the past no_plays recorded plays. 
   The holdings of the player are recorded as a boolean string in h, so that
     h[c] == true iff the player holds the card c.
*/
{
  int i;

  for (i = 0; i < 52; i++)
    if (b->deal[i] == player)
      h[i] = 1;
    else
      h[i] = 0;

  for (i = 0; i < no_plays; i++)
    h[plays[i]] = 0;
}

void Generate_valid_leads (b, p, player, h)
     Board *b;
     Play_record *p;
     int player;
     card_type *h;
/* Records in the hand h the set of cards which the player may lead,
   based upon the plays which have already been recorded in p.
   The valid plays are stored as a boolean string in h, so that
     h[c] == true iff it is legal to play card c.
*/
{
  Generate_holdings (b, p->play_list, p->no_plays, player, h);
}

void Generate_valid_follows 
  (b, p, player, lead, h)
     Board *b;
     Play_record *p;
     int player;
     int lead;
     card_type *h;
/* Records in the hand h the set of cards which the player may play to
   follow the given lead, based upon the plays which have already been
   recorded in p.

   The valid plays are stored as a boolean string in h, so that
     h[c] == true iff it is legal to play card c.
*/
{
  int i;
  int lead_suit  = suit_of (lead);
  int has_lead_suit = 0;

  Generate_holdings (b, p->play_list, p->no_plays, player, h);

  for (i = 0; (i < 13) && !has_lead_suit; i++)
    if (h[card_code(lead_suit, i)])
      has_lead_suit = 1;

  if (has_lead_suit)
    for (i = 0; i < 52; i++)
      if (suit_of(i) != lead_suit)
	h[i] = 0;
}

int Next_Player (p)
     Play_record *p;
/* Returns the index of the next person who should play, or -1 if the
   hand has been completed. */
{
  if (p->hand_completed)
    return (-1);
  else if (p->next_player == player_partner[p->declarer])
    return (p->declarer);
  else
    return (p->next_player);
     
}

static int outranks (c1, c2, t)
    int c1, c2, t;
/* Returns true if c1 outranks c2, according to the current trump suit t.
 * If c1 and c2 are of different suits, neither of which is the trump suit t,
 * then returns false.
 */
{
	if (suit_of (c1) == suit_of (c2))
		return (rank_of(c1) > rank_of(c2));
	else
		return (suit_of (c1) == t);
}

int Winning_card (trump_suit, leader, c_l, c_lho, c_p, c_rho)
     int trump_suit;
     int leader;
     int c_l;
     int c_lho;
     int c_p;
     int c_rho;
/* Returns the index of the player who played the winning card from
   the cards c_l, c_lho, c_p, c_rho, based on the given trump suit
   and that c_l was the card lead.
*/
{
  int w = leader;
  int c = c_l;

  if (outranks (c_lho, c, trump_suit)) {
    c = c_lho;
    w = player_next [leader];
  }

  if (outranks (c_p, c, trump_suit)) {
    c = c_p;
    w = player_partner [leader];
  }

  if (outranks (c_rho, c, trump_suit))
    w = player_prev [leader];

  return (w);
}

