/*************************************************************************
 *  TinyFugue - programmable mud client
 *  Copyright (C) 1993, 1994 Ken Keys
 *
 *  TinyFugue (aka "tf") is protected under the terms of the GNU
 *  General Public License.  See the file "COPYING" for details.
 ************************************************************************/
/* $Id: util.c,v 34000.9 1994/08/26 04:08:47 hawkeye Exp $ */


/*
 * Fugue utilities.
 *
 * Uppercase/lowercase table
 * String handling routines
 * Mail checker
 * Cleanup routine
 */

#include "config.h"
#include <errno.h>
extern int errno;
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "tfio.h"
#include "output.h"	/* fix_screen() */
#include "tty.h"	/* reset_tty() */
#include "signals.h"	/* core() */

static TIME_T mail_mtime;	/* mail file modification time */
static long mail_size;		/* mail file size */

static char *FDECL(cmatch,(CONST char *pat, int ch));

int mail_flag = 0;
char lowercase_values[128], uppercase_values[128];    /* for lcase(), ucase() */
char tf_ctype[128];

struct {
    regexp *re;
    CONST char *str;
    short temp, ok;
} reginfo;

void init_util()
{
    int i;

    /* Some non-standard compilers don't handle tolower() or toupper()
     * on a character that wouldn't ordinarily be converted.  So we
     * create our own conversion table and use macros in port.h.
     */
    for (i = 0; i < 128; i++) {
        lowercase_values[i] = isupper(i) ? tolower(i) : i;
        uppercase_values[i] = islower(i) ? toupper(i) : i;
        tf_ctype[i] = 0;
    }
    tf_ctype['"']  |= IS_QUOTE;
    tf_ctype['`']  |= IS_QUOTE;
    tf_ctype['\''] |= IS_QUOTE;
    tf_ctype['/']  |= IS_STATMETA;
    tf_ctype['%']  |= IS_STATMETA;
    tf_ctype['$']  |= IS_STATMETA;
    tf_ctype[')']  |= IS_STATMETA;
    tf_ctype['\\'] |= IS_STATMETA;
    tf_ctype[';']  |= IS_STATEND;
    tf_ctype['\\'] |= IS_STATEND;
}

#undef CTRL
/* convert to or from ctrl character */
#define CTRL(c)  ((ucase(c) + '@') & 0x7F)

/* Convert ascii string to printable string with "^X" forms. */
/* Returns pointer to static area; copy if needed. */
char *ascii_to_print(str)
    CONST char *str;
{
    STATIC_BUFFER(buffer);

    for (Stringterm(buffer, 0); *str; str++) {
        if (*str == '^' || *str == '\\') {
            Stringadd(Stringadd(buffer, '\\'), *str);
        } else if (!isprint(*str) || iscntrl(*str)) {
            Stringadd(Stringadd(buffer, '^'), CTRL(*str));
        } else Stringadd(buffer, *str);
    }
    return buffer->s;
}

/* Convert a printable string containing '^X' and '\nnn' to real ascii. */
/* Returns pointer to static area; copy if needed. */
char *print_to_ascii(src)
    CONST char *src;
{
    STATIC_BUFFER(dest);

    Stringterm(dest, 0);
    while (*src) {
        if (*src == '^') {
            Stringadd(dest, *++src ? CTRL(*src++) : '^');
        } else if (*src == '\\' && isdigit(*++src)) {
            Stringadd(dest, strtochr((char**)&src));
        } else Stringadd(dest, *src++);
    }
    return dest->s;
}

/* String handlers
 * Some of these are already present in most C libraries, but go by
 * different names or are not always there.  If autoconfig couldn't
 * find them, we use our own.
 * These are heavily used functions, so speed is favored over simplicity.
 */

#ifndef HAVE_STRTOL
char strtochr(sp)
    char **sp;
{
    int c;

    if (**sp != '0') {
        c = atoi(*sp);
        while (isdigit(*++(*sp)));
    } else if (lcase(*++(*sp)) == 'x') {
        for ((*sp)++, c = 0; isxdigit(**sp); (*sp)++)
            c = c * 0x10 + lcase(**sp) - (isdigit(**sp) ? '0' : ('a' - 10));
    } else {
        for (c = 0; isdigit(**sp); (*sp)++)
            c = c * 010 + **sp - '0';
    }
    return (char)(c % 128);
}

int strtoi(sp)
    char **sp;
{
    int i;
    while (isspace(**sp)) ++*sp;
    i = atoi(*sp);
    while (isdigit(**sp)) ++*sp;
    return i;
}
#endif

/* case-insensitive strchr() */
char *cstrchr(s, c)
    register CONST char *s;
    register int c;
{
    for (c = lcase(c); *s; s++) if (lcase(*s) == c) return (char *)s;
    return (c) ? NULL : (char *)s;
}

/* c may be escaped by preceeding it with e */
char *estrchr(s, c, e)
    register CONST char *s;
    register int c, e;
{
    while (*s) {
        if (*s == c) return (char *)s;
        if (*s == e) {
            if (*++s) s++;
        } else s++;
    }
    return NULL;
}

#ifdef sun
# ifndef HAVE_INDEX
/* Workaround for some buggy Solaris 2.x systems, where libtermcap calls index()
 * and rindex(), but they're not defined in libc.
 */
#undef index
char *index(s, c)
    CONST char *s;
    char c;
{
    return strchr(s, c);
}

#undef rindex
char *rindex(s, c)
    CONST char *s;
    char c;
{
    return strrchr(s, c);
}
# endif /* HAVE_INDEX */
#endif /* sun */

#ifndef HAVE_STRSTR
char *STRSTR(s1, s2) 
    register CONST char *s1, *s2;
{
    int len;

    if ((len = strlen(s2) - 1) < 0) return (char*)s1;
    while ((s1 = strchr(s1, *s2))) {
        if (strncmp(s1 + 1, s2 + 1, len) == 0) return (char*)s1;
    }
    return NULL;
}
#endif

#ifndef cstrcmp
/* case-insensitive strcmp() */
int cstrcmp(s, t)
    register CONST char *s, *t;
{
    register int diff;
    while ((*s || *t) && !(diff = lcase(*s) - lcase(*t))) s++, t++;
    return diff;
}
#endif

#if 0  /* not used */
/* case-insensitive strncmp() */
int cstrncmp(s, t, n)
    register CONST char *s, *t;
    int n;
{
    register int diff;
    while (n && *s && !(diff = lcase(*s) - lcase(*t))) s++, t++, n--;
    return n ? diff : 0;
}
#endif

/* numarg
 * Converts argument to a nonnegative integer.  Returns -1 for failure.
 * The *str pointer will be advanced to beginning of next word.
 */
int numarg(str)
    char **str;
{
    int result;
    if (isdigit(**str)) {
        result = strtoi(str);
    } else {
        eprintf("invalid or missing numeric argument");
        result = -1;
        while (**str && !isspace(**str)) ++*str;
    }
    while (isspace(**str)) ++*str;
    return result;
}

/* stringarg
 * Returns pointer to first space-delimited word in *str.  *str is advanced
 * to next word.  If end != NULL, *end will get pointer to end of first word;
 * otherwise, word will be null terminated.
 */
char *stringarg(str, end)
    char **str, **end;
{
    char *start;
    while (isspace(**str)) ++*str;
    for (start = *str; (**str && !isspace(**str)); ++*str) ;
    if (end) *end = *str;
    else if (**str) *((*str)++) = '\0';
    if (**str)
        while (isspace(**str)) ++*str;
    return start;
}

int regexec_and_hold(re, str, temp)
    regexp *re;
    CONST char *str;
    int temp;
{
    reghold(re, str, temp);
    return (int)(reginfo.ok = regexec(reginfo.re, (char*)reginfo.str));
}

/* reghold
 * Hold onto a regexp and string for future reference.  If temp is true,
 * we are responsible for [de]allocation; otherwise, caller will handle it.
 */
void reghold(re, str, temp)
    regexp *re;
    CONST char *str;
    int temp;
{
    regrelease();
    reginfo.re = re;
    reginfo.str = temp ? STRDUP(str) : str;
    reginfo.temp = temp;
    reginfo.ok = 1;
}

/* regrelease
 * Deallocate previously held regexp and string.
 */
void regrelease()
{
    if (reginfo.temp) {
        if (reginfo.re)  free(reginfo.re);
        if (reginfo.str) FREE(reginfo.str);
    }
    reginfo.re = NULL;
    reginfo.str = NULL;
    reginfo.ok = 0;
}

/* returns length of substituted string, or -1 if no substitution */
int regsubstr(dest, n)
    String *dest;
    int n;
{
    CONST regexp *re = reginfo.re;
    if (!(reginfo.ok && n < NSUBEXP && re && re->startp[n])) return -1;
    Stringncat(dest, re->startp[n], re->endp[n] - re->startp[n]);
    return re->endp[n] - re->startp[n];
}

void regerror(msg)
    char *msg;
{
    eprintf("regexp error: %s", msg);
}

/* call with (pat, NULL, 0) to zero-out pat.
 * call with (pat, str, 0) to init pat with some outside string (ie, strdup).
 * call with (pat, str, mflag) to init pat with some outside string.
 */
int init_pattern(pat, str, mflag)
    Pattern *pat;
    CONST char *str;
    int mflag;
{
    pat->re = NULL;
    pat->str = NULL;
    if (!str) return 1;
    if (mflag == 2) {
        if (!(pat->re = regcomp((char*)str))) return 0;
    } else if (mflag == 1) {
        if (!smatch_check(str)) return 0;
    }
    pat->str = STRDUP(str);
    return 1;
}

void free_pattern(pat)
    Pattern *pat;
{
    if (pat->str) FREE(pat->str);
    if (pat->re) free(pat->re);    /* was allocated by malloc() in regcomp() */
    pat->str = NULL;
    pat->re  = NULL;
}

int patmatch(pat, str, mflag, temp)
    Pattern *pat;
    CONST char *str;
    int mflag, temp;
{
    if (!pat->str || !*pat->str) return 1;
    else if (mflag >= 2) return regexec_and_hold(pat->re, str, temp);
    else if (mflag == 1) return !smatch(pat->str, str);
    else return !strcmp(pat->str, str);
}

/* pat is a pointer to a string of the form "[...]..."
 * ch is compared against the character class described by pat.
 * If ch matches, cmatch() returns a pointer to the char after ']' in pat;
 * otherwise, cmatch() returns NULL.
 */
static char *cmatch(pat, ch)
    CONST char *pat;
    int ch;
{
    int not;

    ch = lcase(ch);
    if ((not = (*++pat == '^'))) ++pat;

    while (1) {
        if (*pat == ']') return (char*)(not ? pat + 1 : NULL);
        if (*pat == '\\') ++pat;
        if (pat[1] == '-') {
            char lo = *pat;
            pat += 2;
            if (*pat == '\\') ++pat;
            if (ch >= lcase(lo) && ch <= lcase(*pat)) break;
        } else if (lcase(*pat) == ch) break;
        ++pat;
    }
    return not ? 0 : (estrchr(++pat, ']', '\\') + 1);
}

/* smatch_check() should be used on pat to check pattern syntax before
 * calling smatch().
 */
/* Based on code by Leo Plotkin. */

int smatch(pat, str)
    CONST char *pat, *str;
{
    CONST char *start = str;
    static int inword = FALSE;

    while (*pat) {
        switch (*pat) {

        case '\\':
            pat++;
            if (lcase(*pat++) != lcase(*str++)) return 1;
            break;

        case '?':
            if (!*str || (inword && isspace(*str))) return 1;
            str++;
            pat++;
            break;

        case '*':
            while (*pat == '*' || *pat == '?') {
                if (*pat == '?') {
                    if (!*str || (inword && isspace(*str))) return 1;
                    str++;
                }
                pat++;
            }
            if (inword) {
                while (*str && *str != ' ')
                    if (!smatch(pat, str++)) return 0;
                return smatch(pat, str);
            } else if (*pat == '{') {
                if (str == start || *(str - 1) == ' ')
                    if (!smatch(pat, str)) return 0;
                while ((str = strchr(str, ' ')) != NULL)
                    if (!smatch(pat, ++str)) return 0;
                return 1;
            } else if (*pat == '[') {
                while (*str) if (!smatch(pat, str++)) return 0;
                return 1;
            } else {
                if (pat[0] == '\\' && pat[1]) pat++;
                /* cstrchr() quickly finds the next possible match */
                while ((str = cstrchr(str, *pat)) != NULL)
                    if (!smatch(pat, str++)) return 0;
                return 1;
            }

        case '[':
            if (inword && *str == ' ') return 1;
            if (!(pat = cmatch(pat, *str++))) return 1;
            break;

        case '{':
            if (str != start && *(str - 1) != ' ') return 1;
            {
                CONST char *end;
                int result = 1;

                if (!(end = estrchr(pat, '}', '\\'))) {     /* can't happen if*/
                    tfputs("% smatch: unmatched '{'", tferr); /* smatch_check */
                    return 1;                                /* is used first */
                }

                inword = TRUE;
                for (pat++; pat <= end; pat++) {
                    if ((result = smatch(pat, str)) == 0) break;
                    if (!(pat = estrchr(pat, '|', '\\'))) break;
                }
                inword = FALSE;
                if (result) return result;
                pat = end + 1;
                while (*str && *str != ' ') str++;
            }
            break;

        case '}': case '|':
            if (inword) return (*str && *str != ' ');
            /* else FALL THROUGH to default case */

        default:
            if (lcase(*pat++) != lcase(*str++)) return 1;
            break;
        }
    }
    return lcase(*pat) - lcase(*str);
}

/* verify syntax of smatch pattern */
int smatch_check(pat)
    CONST char *pat;
{
    int inword = FALSE;

    while (*pat) {
        switch (*pat) {
        case '\\':
            if (*++pat) pat++;
            break;
        case '[':
            if (!(pat = estrchr(pat, ']', '\\'))) {
                eprintf("glob error: unmatched '['");
                return 0;
            }
            pat++;
            break;
        case '{':
            if (inword) {
                eprintf("glob error: nested '{'");
                return 0;
            }
            inword = TRUE;
            pat++;
            break;
        case '}':
            inword = FALSE;
            pat++;
            break;
        case '?':
        case '*':
        default:
            pat++;
            break;
        }
    }
    if (inword) eprintf("glob error: unmatched '{'");
    return !inword;
}

/* remove leading and trailing spaces */
char *stripstr(s)
    char *s;
{
    char *end;

    while (isspace(*s)) s++;
    if (*s) {
        for (end = s + strlen(s) - 1; isspace(*end); end--);
        *++end = '\0';
    }
    return s;
}


/* General command option parser

   startopt should be called before nextopt.  args is the argument list
   to be parsed, opts is a string containing valid options.  Options which
   take string arguments should be followed by a ':'; options which take
   numeric arguments should be followed by a '#'.  String arguments may be
   omitted.  The special character '0' expects an integer option of the
   form "-nn".  The special charcter '@' expects a time option of the
   form "-hh:mm:ss", "-hh:mm", or "-ss".

   nextopt returns the next option character.  If option takes a string
   argument, a pointer to it is returned in *arg; an integer argument
   is returned in *num.  If end of options is reached, nextopt returns
   '\0', and *arg points to remainder of argument list.  End of options
   is marked by "\0", "=", "--", or a word not beggining with
   '-'.  If an invalid option is encountered, an error message is
   printed and '?' is returned.

   Option Syntax Rules:
      All options must be preceded by '-'.
      Options may be grouped after a single '-'.
      There must be no space between an option and its argument.
      String option-arguments may be quoted.  Quotes in the arg must be escaped.
      All options must precede operands.
      A '--' or '-' with no option may be used to mark the end of the options.
*/

static char *argp;
static CONST char *options;
static int inword;

void startopt(args, opts)
    char *args;
    CONST char *opts;
{
    argp = args;
    options = opts;
    inword = 0;
}

char nextopt(arg, num)
    char **arg;
    int *num;
{
    char *q, opt, quote;
    STATIC_BUFFER(buffer);

    if (!inword) {
        while (isspace(*argp)) argp++;
        if (*argp != '-') {
            *arg = argp;
            return '\0';
        } else {
            if (*++argp == '-') argp++;
            if (isspace(*argp) || !*argp) {
                for (*arg = argp; isspace(**arg); ++*arg);
                return '\0';
            }
        }
    } else if (*argp == '=') {        /* '=' marks end, & is part of parms */
        *arg = argp;                  /*... for stuff like  /def -t"foo"=bar */
        return '\0';
    }

    opt = *argp;

    /* time option */
    if ((isdigit(opt) || opt == ':') && strchr(options, '@')) {
        *num = parsetime(&argp, NULL);
        return '@';

    /* numeric option */
    } else if (isdigit(opt) && strchr(options, '0')) {
        *num = strtoi(&argp);
        return '0';
    }

    /* other options */
    if (opt == '@' || opt == ':' || opt == '#' || !(q = strchr(options, opt))) {
        eprintf("invalid option: %c", opt);
        return '?';
    }

    /* option takes a string argument */
    if (*++q == ':') {
        Stringterm(buffer, 0);
        ++argp;
        if (is_quote(*argp)) {
            quote = *argp;
            for (argp++; *argp && *argp != quote; Stringadd(buffer, *argp++))
                if (*argp == '\\' && (argp[1] == quote || argp[1] == '\\'))
                    argp++;
            if (!*argp) {
                eprintf("unmatched %c in %c option", quote, opt);
                return '?';
            } else argp++;
        } else while (*argp && !isspace(*argp)) Stringadd(buffer, *argp++);
        *arg = buffer->s;

    /* option takes a numeric argument */
    } else if (*q == '#') {
        argp++;
        if (!isdigit(*argp)) {
            eprintf("%c option requires numeric argument", opt);
            return '?';
        }
        *num = strtoi(&argp);

    /* option takes no argument */
    } else {
        argp++;
    }

    inword = (*argp && !isspace(*argp));
    return opt;
}

Aline *dnew_aline(str, attrs, file, line)
    CONST char *str;
    CONST char *file;
    int attrs, line;
{
    Aline *aline;
    int len;
    void *memory;

    len = strlen(str);
    /* Optimization: allocating aline and aline->str in one chunk is faster,
     * and helps increase locality of reference.  Chars have size 1, so
     * alignment and arithmetic for the offset of str is not a problem.
     */
    memory = dmalloc(sizeof(Aline) + len + 1, file, line);
    aline = (Aline *)memory;
    aline->str = strcpy((char*)memory + sizeof(Aline), str);
    aline->len = len;
    aline->attrs = attrs;
    aline->partials = NULL;
    aline->links = 0;
    aline->time = time(NULL);
    return aline;
}

void dfree_aline(aline, file, line)
    Aline *aline;
    CONST char *file;
    int line;
{
    if (aline->links <= 0)
        core("dfree_aline: links == %ld", file, line, (long)aline->links);
    if (!--aline->links) {
        if (aline->partials) FREE(aline->partials);
        FREE(aline);  /* struct and string */
    }
}

void ch_maildelay()
{
    extern TIME_T mail_update;
    mail_update = 0;
}

void ch_mailfile()
{
    if (MAIL) {
        mail_mtime = -1;
        mail_size = -1;
        ch_maildelay();
    }
}

void init_mail()
{
    Stringp path;
    CONST char *name;

    if (MAIL) {  /* was imported from environment */
        ch_mailfile();
    } else if ((name = getvar("LOGNAME")) || (name = getvar("USER"))) {
        Sprintf(Stringinit(path), 0, "%s/%s", MAILDIR, name);
        setvar("MAIL", path->s, FALSE);
        Stringfree(path);
    } else {
        tfputs("% Warning:  Can't figure out name of mail file.", tferr);
    }
}

void check_mail()
{
    struct stat buf;

    if (!MAIL || !*MAIL || maildelay <= 0) return;

    if (stat(expand_filename(MAIL), &buf) < 0) {
        /* There is no mail. */
        if (mail_flag) { mail_flag = 0; status_bar(STAT_MAIL); }
        mail_mtime = 0;
        mail_size = 0;
        return;
    }

    /* Any of mtime==atime, or change in size, mtime, or ctime, could be due
     * to a mail reader.
     * If file didn't exist when mail arrived, comparing atime and mtime is
     * useless; we have to see if the old size was 0.
     */
    if (buf.st_size == 0) {
        /* There is no mail. */
        if (mail_flag) { mail_flag = 0; status_bar(STAT_MAIL); }
    } else if (mail_size == 0 || buf.st_mtime > buf.st_atime) {
        /* Mail is unread.  (Actually, if mail_size==0, we can't tell if mail
         * has been read, we just know that it's new since our last check.)
         */
        if (!mail_flag) { mail_flag = 1; status_bar(STAT_MAIL); }
        if (mail_mtime >= 0 && buf.st_mtime > mail_mtime) {
            /* There is new mail since the last time we checked. */
            do_hook(H_MAIL, "%% You have new mail in %s", "%s", MAIL);
        }
    } else {
        /* Mail exists, but has been read. */
        if (mail_flag) { mail_flag = 0; status_bar(STAT_MAIL); }
    }
    mail_mtime = buf.st_mtime;
    mail_size = buf.st_size;
}

/* Converts a string of the form "hh:mm:ss", "hh:mm", or "ss" to seconds.
 * Return value in <istime> is true string must be a time; otherwise,
 * the string could be interpreted as a plain integer.  <strp> is
 * advanced to the first character past the time string.  Return
 * value is -1 for an invalid string, a nonnegative integer otherwise.
 */
int parsetime(strp, istime)
    char **strp;
    int *istime;     /* return true if ':' found (i.e., can't be an int) */
{
    static int t;

    if (!isdigit(**strp) && **strp != ':') {
        eprintf("invalid or missing integer or time value");
        return -1;
    }
    t = strtoi(strp);
    if (**strp == ':') {
        if (istime) *istime = TRUE;
        t *= 3600;
        ++*strp;
        if (isdigit(**strp)) {
            t += strtoi(strp) * 60;
            if (**strp == ':') {
                ++*strp;
                if (isdigit(**strp))
                    t += strtoi(strp);
            }
        }
    } else {
        if (istime) *istime = FALSE;
    }
    return t;
}

/* Converts an hms value (from parsetime()) to an absolute clock time
 * within the last 24h.
 * Ideally, we'd use mktime() to find midnight, but not all systems
 * have it.  So we use ctime() to get <now> in a string, convert the
 * HH:MM:SS fields to an integer, and subtract from <now> to get the
 * time_t for midnight.  This function isn't heavily used anyway.
 * BUG: doesn't handle switches to/from daylight savings time.
 */
TIME_T abstime(hms)
    int hms;
{
    TIME_T result, now;
    char *ptr;

    now = time(NULL);
    ptr = ctime(&now) + 11;                  /* find HH:MM:SS part of string */
    result = now - parsetime(&ptr,NULL);     /* convert, subtract -> midnight */
    if ((result += hms) > now) result -= 24 * 60 * 60;
    return result;
}

/* returns a pointer to a formatted time string */
char *tftime(fmt, t)
    CONST char *fmt;
    TIME_T t;
{
    static smallstr buf;

    if (!*fmt) fmt = (time_format && *time_format) ? time_format : "%c";
    if (strcmp(fmt, "@") == 0) {
        sprintf(buf, "%ld", t);
        return buf;
    } else {
#ifdef HAVE_STRFTIME
        return strftime(buf, sizeof(buf) - 1, fmt, localtime(&t)) ? buf : NULL;
#else
        char *str = ctime(&t);
        str[strlen(str) - 1] = '\0';    /* remove ctime()'s '\n' */
        return str;
#endif
    }
}

void die(why, err)
    CONST char *why;
    int err;
{
    fix_screen();
    reset_tty();
    if (err) perror(why);
    else {
        fputs(why, stderr);
        fputc('\n', stderr);
    }
    exit(1);
}

