/*************************************************************************
 *  TinyFugue - programmable mud client
 *  Copyright (C) 1993, 1994 Ken Keys
 *
 *  TinyFugue (aka "tf") is protected under the terms of the GNU
 *  General Public License.  See the file "COPYING" for details.
 ************************************************************************/
/* $Id: output.c,v 34000.11 1994/09/25 06:33:15 hawkeye Exp $ */


/*****************************************************************
 * Fugue output handling
 *
 * Written by Ken Keys (Hawkeye).
 * Handles all screen-related phenomena.
 *****************************************************************/

#include "config.h"
#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "tfio.h"
#include "world.h"
#include "socket.h"	/* fgprompt(), fworld() */
#include "output.h"
#include "macro.h"	/* add_ibind() */
#include "search.h"
#include "tty.h"	/* init_tty(), get_window_wize() */


/* Terminal codes and capabilities.
 * The codes below are for vt100.  Some additional vt220 and ANSI codes not
 *   shared by vt100 are also given in #if 0 blocks.
 * To hardcode for a different terminal, edit the codes below.
 * Visual mode requires at least clear_screen and cursor_address.  The other
 *   codes are good to have, but are not strictly necessary.
 * Scrolling in visual mode requires change_scroll_region (preferred), or
 *   insert_line and delete_line (may appear jumpy).
 * Fast insert in visual mode requires insert_start and insert_end (preferred),
 *   or insert_char.
 * Fast delete in visual mode requires delete_char.
 * Attributes require the appropriate attribute code and attr_off; but if
 *   attr_off is not defined, underline and standout (replacement for bold)
 *   can still be used if underline_off and standout_off are defined.
 */

#define DEFAULT_LINES   24
#define DEFAULT_COLUMNS 80

#ifdef HARDCODE
# define origin 1      /* top left corner is (1,1) */
# define TERMCODE(id, code)   static CONST char *(id) = (code);
#else
# define origin 0      /* top left corner is (0,0) */
# define TERMCODE(id, code)   static CONST char *(id) = NULL;
#endif

TERMCODE (clear_screen,		"\033[H\033[J")
TERMCODE (clear_to_eos,		"\033[J")
TERMCODE (clear_to_eol,		"\033[K")
TERMCODE (cursor_address,	"\033[%d;%dH")  /* (in printf format) */
TERMCODE (enter_ca_mode,	NULL)           /* enable cursor motion mode */
TERMCODE (exit_ca_mode,		NULL)           /* disable cursor motion mode */
TERMCODE (change_scroll_region,	"\033[%d;%dr")  /* (in printf format) */
#if 0
TERMCODE (insert_line,		"\033[L")	/* vt220, ansi */
TERMCODE (delete_line,		"\033[M")	/* vt220, ansi */
TERMCODE (delete_char,		"\033[P")	/* vt220, ansi */
TERMCODE (insert_char,		"\033[@")	/* vt220, ansi */
TERMCODE (insert_start,		"\033[4h")	/* vt220 */
TERMCODE (insert_end,		"\033[4l")	/* vt220 */
#else
TERMCODE (insert_line,		NULL)		/* vt100 */
TERMCODE (delete_line,		NULL)		/* vt100 */
TERMCODE (delete_char,		NULL)		/* vt100 */
TERMCODE (insert_char,		NULL)		/* vt100 */
TERMCODE (insert_start,		NULL)		/* vt100, ansi */
TERMCODE (insert_end,		NULL)		/* vt100, ansi */
#endif
TERMCODE (underline,		"\033[4m")
TERMCODE (reverse,		"\033[7m")
TERMCODE (flash,		"\033[5m")
TERMCODE (dim,			NULL)
TERMCODE (bold,			"\033[1m")
TERMCODE (attr_off,		"\033[m")  /* turn off all attrs */
TERMCODE (underline_off,	NULL)      /* only used if !attr_off */
TERMCODE (standout,		NULL)      /* only used if !bold */
TERMCODE (standout_off,		NULL)      /* only used if !bold */

/* end HARDCODE section */


static void  NDECL(init_term);
static int   FDECL(fbufputc,(int c));
static void  NDECL(bufflush);
static void  FDECL(tp,(CONST char *str));
static void  FDECL(xy,(int x, int y));
static void  NDECL(clr);
static void  NDECL(clear_line);
static void  NDECL(clear_input_line);
static void  FDECL(clear_lines,(int start, int end));
static void  NDECL(clear_input_window);
static void  FDECL(setscroll,(int start, int end));
static void  FDECL(scroll_input,(int n));
static int   FDECL(ioutputs,(CONST char *str, int len));
static void  FDECL(ioutall,(int kpos));
static void  FDECL(attributes_off,(int attrs));
static void  FDECL(attributes_on,(int attrs));
static void  FDECL(hwrite,(CONST char *s, unsigned int len, int attrs,
                   short *partials));
static int   NDECL(check_more);
static int   FDECL(clear_more,(int new));
static Aline *NDECL(wrapline);
static void  NDECL(output_novisual);
#ifdef SCREEN
static void  NDECL(output_noscroll);
static void  NDECL(output_scroll);
#endif

#ifdef TERMCAP
#define tpgoto(seq,x,y)  tp(tgoto(seq, (x)-1+origin, (y)-1+origin))
#else
#define tpgoto(seq,x,y)  Sprintf(outbuf,SP_APPEND,seq,(y)-1+origin,(x)-1+origin)
#endif

#define ipos()		xy(ix, iy)

/* Buffered output */

#define bufputs(s)		Stringcat(outbuf, s)
#define bufputns(s, n)		Stringncat(outbuf, s, n)
#define bufputc(c)		Stringadd(outbuf, c)
#define bufputnc(c, n)		Stringnadd(outbuf, c, n)
#define bufprintf1(f, p)	Sprintf(outbuf, SP_APPEND, f, p)

#ifdef USE_SGTTY
# define crnl() bufputs("\r\n")    /* CRMOD is off (tty.c) */
#else
# define crnl() bufputc('\n')      /* ONLCR is on (tty.c) */
#endif

/* Others */

#define Wrap (wrapsize ? wrapsize : columns)
#define more_attrs (F_BOLD | F_REVERSE)
#define morewait 20

extern int keyboard_pos;            /* position of logical cursor in keybuf */
extern Stringp keybuf;              /* input buffer */

STATIC_BUFFER(outbuf);              /* output buffer */
static int cx = -1, cy = -1;        /* Real cursor ((-1,-1)==unknown) */
static int ox = 1, oy;              /* Output cursor */
static int ix, iy;                  /* Input cursor */
static int ystatus;                 /* line # of status bar */
static int istarty, iendy, iendx;   /* start/end of current input line */
STATIC_BUFFER(moreprompt);          /* pager prompt */
static String *prompt;              /* current prompt */
static int outcount;                /* lines remaining until more prompt */
static int paused = FALSE;          /* output paused */
static short have_attr = 0;         /* available attributes */
static Aline *currentline = NULL;   /* current logical line for printing */
static int wrap_offset;             /* physical offset into currentline */
static int screen_setup = FALSE;    /* is *screen* in visual mode? */
static int can_have_visual = FALSE;

int lines   = 0;
int columns = 0;
int need_refresh = 0;               /* does input need refresh? */
int sockecho = TRUE;                /* echo input? */
#define echoflag (sockecho || always_echo)
unsigned int tfscreen_size;         /* # of lines in tfscreen */
TFILE *tfscreen;                    /* text waiting to be displayed */
TFILE *tfout;                       /* pointer to current output queue */
TFILE *tferr;                       /* pointer to current error queue */

#ifdef TERMCAP
extern int   FDECL(tgetent,(char *buf, CONST char *name));
extern int   FDECL(tgetnum,(CONST char *id));
extern char *FDECL(tgetstr,(CONST char *id, char **area));
extern char *FDECL(tgoto,(CONST char *code, int destcol, int destline));
extern int   FDECL(tputs,(CONST char *cp, int affcnt, int (*outc)(int)));
#endif

/****************************
 * BUFFERED OUTPUT ROUTINES *
 ****************************/

static void bufflush()
{
    if (outbuf->len) {
        write(1, outbuf->s, outbuf->len);
        Stringterm(outbuf, 0);
    }
}

static int fbufputc(c)
    int c;
{
    Stringadd(outbuf, c);
    return c;
}

void bell(n)
    int n;
{
    if (beep) {
        bufputnc('\007', n);
        bufflush();
    }
}

/********************
 * TERMCAP ROUTINES *
 ********************/

void change_term()
{
    int old = visual;
    setvar("visual", "0", FALSE);
    init_term();
    if (old) setvar("visual", "1", FALSE);
}

/* Initialize basic output data. */
void init_output()
{
    CONST char *str;

    tfout = tferr = tfscreen = tfopen(NULL, "q");
    tfscreen_size = 0;
    Stringcpy(moreprompt, "--More--");
    prompt = fgprompt();

    init_tty();

    /* window size: try environment, ioctl TIOCGWINSZ, termcap, defaults. */
    if ((str = getvar("LINES"))) lines = atoi(str);
    if ((str = getvar("COLUMNS"))) columns = atoi(str);
    if (lines <= 0 || columns <= 0) get_window_size();

    init_term();
    ch_hilite();
    ch_visual();
}

static void init_term()
{
    CONST char *errmsg = NULL;
#ifdef TERMCAP
    CONST char *str;
    /* Termcap entries are supposed to fit in 1024 bytes.  But, if a 'tc'
     * field is present, some termcap libraries will just append the second
     * entry to the original.  Also, some overzealous versions of tset will
     * also expand 'tc', so there might be *2* entries appended to the
     * original.  And, linux termcap adds 'li' and 'co' fields, so it could
     * get even longer.  To top it all off, some termcap libraries don't
     * do any length checking in tgetent().  We should be safe with 4096.
     */
    char termcap_entry[4096];
    /* termcap_buffer will hold at most 1 copy of any field; 1024 is enough. */
    static char termcap_buffer[1024];
    char *area = termcap_buffer;

    have_attr = 0;
    can_have_visual = FALSE;
    clear_screen = clear_to_eos = clear_to_eol = NULL;
    change_scroll_region = insert_line = delete_line = NULL;
    delete_char = insert_char = insert_start = insert_end = NULL;
    enter_ca_mode = exit_ca_mode = cursor_address = NULL;
    standout = underline = reverse = flash = dim = bold = NULL;
    standout_off = underline_off = attr_off = NULL;

    if (!TERM || !*TERM) {
        errmsg = "%% Warning: TERM undefined.";
        /* Can't print this message until all screen info is initialized. */
    } else if (tgetent(termcap_entry, TERM) <= 0) {
        errmsg = "%% Warning: \"%s\" terminal unsupported.";
        /* Can't print this message until all screen info is initialized. */
    } else {
        if (columns <= 0) columns = tgetnum("co");
        if (lines   <= 0) lines   = tgetnum("li");

        clear_screen         = tgetstr("cl", &area);
        clear_to_eol         = tgetstr("ce", &area);
        clear_to_eos         = tgetstr("cd", &area);
        enter_ca_mode        = tgetstr("ti", &area);
        exit_ca_mode         = tgetstr("te", &area);
        cursor_address       = tgetstr("cm", &area);
        change_scroll_region = tgetstr("cs", &area);
        delete_char          = tgetstr("dc", &area);
        insert_char          = tgetstr("ic", &area);
        insert_start         = tgetstr("im", &area);
        insert_end           = tgetstr("ei", &area);
        insert_line          = tgetstr("al", &area);
        delete_line          = tgetstr("dl", &area);

        if (!insert_end) insert_start = NULL;

        underline	= tgetstr("us", &area);
        reverse		= tgetstr("mr", &area);
        flash		= tgetstr("mb", &area);
        dim		= tgetstr("mh", &area);
        bold		= tgetstr("md", &area);
        standout	= tgetstr("so", &area);
        underline_off	= tgetstr("ue", &area);
        standout_off	= tgetstr("se", &area);
        attr_off	= tgetstr("me", &area);

        if (!attr_off) {
            /* can't exit all attrs, but maybe can exit underline/standout */
            reverse = flash = dim = bold = NULL;
            if (!underline_off) underline = NULL;
            if (!standout_off) standout = NULL;
        }

        if ((str = tgetstr("ku", &area))) add_ibind(str, "/DOKEY UP");
        if ((str = tgetstr("kd", &area))) add_ibind(str, "/DOKEY DOWN");
        if ((str = tgetstr("kr", &area))) add_ibind(str, "/DOKEY RIGHT");
        if ((str = tgetstr("kl", &area))) add_ibind(str, "/DOKEY LEFT");

        /* Many old xterm termcaps mistakenly omit "cs". */
        if (!change_scroll_region && strcmp(TERM, "xterm") == 0)
            change_scroll_region = "\033[%i%d;%dr";
    }
#endif

    if (columns <= 0) columns = DEFAULT_COLUMNS;
    if (lines   <= 0) lines   = DEFAULT_LINES;
    setivar("wrapsize", columns - 1, FALSE);
    outcount = lines;
    ix = 1;
    can_have_visual = clear_screen && cursor_address;
    setivar("scroll", change_scroll_region||(insert_line&&delete_line), FALSE);
    have_attr = F_BELL;
    if (underline) have_attr |= F_UNDERLINE;
    if (reverse)   have_attr |= F_REVERSE;
    if (flash)     have_attr |= F_FLASH;
    if (dim)       have_attr |= F_DIM;
    if (bold)      have_attr |= F_BOLD;
    if (standout)  have_attr |= F_BOLD;

    if (errmsg) tfprintf(tferr, errmsg, TERM);
}

static void setscroll(start,end)
    int start, end;
{
    if (!change_scroll_region) return;  /* shouldn't happen */
    tpgoto(change_scroll_region, (end), (start));
    cx = cy = -1;   /* cursor position is undefined after termcap "cs" */
}

static void xy(x,y)
    int x, y;
{
    if (x == cx && y == cy) return;                        /* already there */
    if (cy < 0 || cx < 0 || cx > columns) {                /* direct movement */
        tpgoto(cursor_address, x, y);
    } else if (x == 1 && y >= cy && y < cy + 5) {          /* optimization */
        /* note: '\n' is not safe outside scroll region */
        if (cx != 1) bufputc('\r');
        bufputnc('\n', y - cy);
    } else if (y == cy && x < cx && x > cx - 7) {          /* optimization */
        bufputnc('\010', cx - x);
    } else {                                               /* direct movement */
        tpgoto(cursor_address, x, y);
    }
    cx = x;  cy = y;
}

static void clr()
{
    tp(clear_screen);
    cx = 1;  cy = 1;
}

static void clear_line()
{
    if (cx != 1) bufputc('\r');
    cx = 1;
    if (clear_to_eol) {
        tp(clear_to_eol);
    } else {
        bufputnc(' ', Wrap);
        bufputc('\r');
    }
}

static void tp(str)
    CONST char *str;
{
    if (str)
#ifdef TERMCAP
        tputs(str, 1, fbufputc);
#else
        bufputs(str);
#endif
}

/*******************
 * WINDOW HANDLING *
 *******************/

void setup_screen()
{
    if (!visual) {
        if (paused) prompt = moreprompt;

#ifdef SCREEN
    } else {
        screen_setup = TRUE;
        prompt = fgprompt();
        if (isize > lines - 3) setivar("isize", lines - 3, FALSE);
        ystatus = lines - isize;
        outcount = ystatus - 1;
        if (enter_ca_mode) tp(enter_ca_mode);
    
        if (scroll && (change_scroll_region || (insert_line && delete_line))) {
            xy(1, lines);
            bufputnc('\n', isize);  /* At bottom.  Don't increment cy. */
        } else {
            clr();
            if (scroll) setivar("scroll", 0, FALSE);
        }
        status_bar(STAT_ALL);
        ix = iendx = oy = 1;
        iy = iendy = istarty = ystatus + 1;
        ipos();
#endif
    }

    set_refresh_pending(REF_LOGICAL);
}

int redraw()
{
    if (!visual) {
        bufputnc('\n', lines);
        logical_refresh();
    } else if (screen_setup) {
        clr();
        setup_screen();
    }
    return 1;
}

void status_bar(seg)
    int seg;
{
    extern int mail_flag;           /* should mail flag on status bar be lit? */
    extern int active_count;        /* number of active sockets */
#ifndef NO_HISTORY
    extern int log_count;           /* number of open log files */
#else
    #define log_count 0
#endif

    if (!screen_setup) return;
    if (!(seg & STAT_MORE))
        oflush();  /* make sure output window is up to date first */

    if (seg & STAT_MORE) {
        xy(1, ystatus);
        if (paused) {
            attributes_on(more_attrs);
            /* tfscreen_size + 1 accounts for tfscreen and currentline */
            if (tfscreen_size + 1 < morewait) bufputs("--More--");
            else if (tfscreen_size + 1 >= 1000) bufputs("MuchMore");
            else bufprintf1("More:%3u", tfscreen_size + 1);
            attributes_off(more_attrs);
        } else bufputs("________");
        bufputc('_');  cx += 9;
    }
    if (seg & STAT_WORLD) {
        World *world;
        int hole, max;
        xy(10, ystatus);
        hole = max = columns - 40 - 10 - 1;
        if ((world = fworld())) {
            bufputns(world->name, max);
            if ((hole -= strlen(world->name)) < 0) hole = 0;
        }
        bufputnc('_', hole + 1);
        cx += max + 1;
    }
    if (seg & STAT_ACTIVE) {
        xy(columns - 40, ystatus);
        if (active_count) bufprintf1("(Active:%2d)_", active_count);
        else bufputs("____________");
        cx += 12;
    }
    if (seg & STAT_LOGGING) {
        xy(columns - 28, ystatus);
        bufputs(log_count ? "(Log)_" : "______");  cx += 6;
    }
    if (seg & STAT_MAIL) {
        xy(columns - 22, ystatus);
        bufputs(mail_flag ? "(Mail)_" : "_______");  cx += 7;
    }
    if (seg & STAT_INSERT) {
        xy(columns - 15, ystatus);
        bufputs(insert ? "___________" : "(Typeover)_");  cx += 11;
    }
    if (seg & STAT_CLOCK) {
        xy(columns - 4, ystatus);
#ifdef HAVE_STRFTIME
        if (clock_flag) {
            extern TIME_T clock_update;
            bufputs(tftime("%H:%M", time(&clock_update)));
            clock_update += 60 - (localtime(&clock_update))->tm_sec;
        } else
#endif
            bufputs("_____");
        cx += 5;
    }

    bufflush();
    set_refresh_pending(REF_PHYSICAL);
}

void tog_insert()
{
    status_bar(STAT_INSERT);
}

/* used by %{visual} and %{isize} */
void ch_visual()
{
    if (visual && (!can_have_visual || columns < 55 || lines < 5)) {
        setvar("visual", "0", FALSE);
        tfputs("% Visual mode not supported.", tferr);
    } else {
        fix_screen();
        setup_screen();
    }
}

void fix_screen()
{
    oflush();
    if (!screen_setup) {
        clear_line();
    } else {
        clear_input_window();
        outcount = lines - 1;
#ifdef SCREEN
        if (change_scroll_region) setscroll(1, lines);
        xy(1, ystatus);
        clear_line();
        if (exit_ca_mode) tp(exit_ca_mode);
#endif
    }
    cx = cy = -1;
    bufflush();
    screen_setup = 0;
}

/* This doesn't use outbuf or other global structures, so it's safe to use
 * even if one of those structures is corrupted.
 */
void panic_fix_screen()
{
    int i;
    puts("\r\n");
    if (screen_setup) for (i = lines - cy; i; i--) putchar('\n');
    fflush(stdout);
}

static void clear_lines(start, end)
    int start, end;
{
    if (start > end) return;
    xy(1, start);
    if (end == lines && clear_to_eos) {
        tp(clear_to_eos);
    } else {
        clear_line();
        while (start++ < end) {
             bufputc('\n');  cy++;
             clear_line();
        }
    }
}

/* clear entire input window */
static void clear_input_window()
{
    /* only called if screen_setup */
    clear_lines(ystatus + 1, lines);
    ix = iendx = 1;
    iy = iendy = istarty = ystatus + 1;
    ipos();
}

/* clear logical input line */
static void clear_input_line()
{
    if (!screen_setup) clear_line();
    else clear_lines(istarty, iendy);
    ix = iendx = 1;
    iy = iendy = istarty;
    if (visual) ipos();
}

/* affects iendx, iendy, istarty.  No effect on ix, iy. */
static void scroll_input(n)
    int n;
{
    if (delete_line) {
        xy(1, ystatus + 1);
        for (iendy = lines + 1; iendy > lines - n + 1; iendy--)
            tp(delete_line);
    } else if (change_scroll_region) {
        setscroll(ystatus + 1, lines);
        xy(1, lines);
        bufputnc('\n', n);  /* cy += n; */
        setscroll(1, lines);
        iendy = lines - n + 1;
    }
    xy(iendx = 1, iendy);
}


/***********************************************************************
 *                                                                     *
 *                        INPUT WINDOW HANDLING                        *
 *                                                                     *
 ***********************************************************************/

/* ioutputs
 * Print string within bounds of input window.  len is the number of
 * characters to print; return value is the number actually printed,
 * which may be less than len if string doesn't fit in the input window.
 * precondition: iendx,iendy and real cursor are at the output position.
 */
static int ioutputs(str, len)
    CONST char *str;
    int len;
{
    int space, written;

    for (written = 0; len > 0; len -= space) {
        if ((space = Wrap - iendx + 1) <= 0) {
            if (!visual || iendy == lines) break;   /* at end of window */
            if (visual) xy(iendx = 1, ++iendy);
            space = Wrap;
        }
        if (space > len) space = len;
        bufputns(str, space);  cx += space;
        str += space;
        written += space;
        iendx += space;
    }
    return written;
}

/* ioutall
 * Performs ioutputs() for input buffer starting at kpos.
 * A negative kpos means to display that much of the end of the prompt.
 */
static void ioutall(kpos)
    int kpos;
{
    int ppos;

    if (kpos < 0) {                  /* posible only if there's a prompt */
        ppos = prompt->len + kpos;
        if (ppos < 0) ppos = 0;      /* possible if prompt->len > wrapsize */
        if (prompt == moreprompt) attributes_on(more_attrs);
        ioutputs(prompt->s + ppos, prompt->len - ppos);
        if (prompt == moreprompt) attributes_off(more_attrs);
        kpos = 0;
    }
    if (echoflag)
        ioutputs(keybuf->s + kpos, keybuf->len - kpos);
}

void iput(s, len)
    CONST char *s;
    int len;
{
    int count, scrolled = 0, oiex = iendx, oiey = iendy;

    if (!echoflag) return;
    if (visual) ipos();

    if (keybuf->len - keyboard_pos > 8 &&     /* faster than redisplay? */
        visual && insert && clear_to_eol &&
        (insert_char || insert_start) &&      /* can insert */
        cy + (cx + len) / Wrap <= lines &&    /* new text will fit in window */
        Wrap - len > 8)                       /* faster than redisplay? */
    {
        /* fast insert */
        iy = iy + (ix - 1 + len) / Wrap;
        ix = (ix - 1 + len) % Wrap + 1;
        iendy = iendy + (iendx - 1 + len) / Wrap;
        iendx = (iendx - 1 + len) % Wrap + 1;
        if (iendy > lines) { iendy = lines; iendx = Wrap + 1; }
        if (cx + len <= Wrap) {
            if (insert_start) tp(insert_start);
            else for (count = len; count; count--) tp(insert_char);
            bufputns(s, len);
            s += Wrap - (cx - 1);
            cx += len;
        } else {
            bufputns(s, Wrap - (cx - 1));
            s += Wrap - (cx - 1);
            cx = Wrap + 1;
            if (insert_start) tp(insert_start);
        }
        while (s < keybuf->s + keybuf->len) {
            if (Wrap < columns && cx <= Wrap) {
                xy(Wrap + 1, cy);
                tp(clear_to_eol);
            }
            if (cy == lines) break;
            xy(1, cy + 1);
            if (!insert_start)
                for (count = len; count; count--) tp(insert_char);
            bufputns(s, len);  cx += len;
            s += Wrap;
        }
        if (insert_start) tp(insert_end);
        ipos();
        bufflush();
        return;
    }

    iendx = ix;
    iendy = iy;

    /* Display as much as possible.  If it doesn't fit, scroll and repeat
     * until the whole string has been displayed.
     */
    while (count = ioutputs(s, len), s += count, (len -= count) > 0) {
        scrolled++;
        if (!visual) {
            crnl();  cx = 1;
            iendx = ix = 1;
        } else if (scroll && !clearfull) {
            scroll_input(1);
            if (istarty > ystatus + 1) istarty--;
        } else {
            clear_input_window();
        }
    }
    ix = iendx;
    iy = iendy;

    if (insert || scrolled) {
        /* we must (re)display tail */
        ioutputs(keybuf->s + keyboard_pos, keybuf->len - keyboard_pos);
        if (visual) ipos();
        else { bufputnc('\010', iendx - ix);  cx -= (iendx - ix); }

    } else if ((iendy - oiey) * Wrap + iendx - oiex < 0) {
        /* if we didn't write past old iendx/iendy, restore them */
        iendx = oiex;
        iendy = oiey;
    }

    bufflush();
}

void inewline()
{
    ix = iendx = 1;
    if (!visual) {
        crnl();  cx = 1; cy++;
        if (prompt && prompt->len > 0) set_refresh_pending(REF_PHYSICAL);

    } else {
        if (cleardone) {
            clear_input_window();
        } else {
            iy = iendy + 1;
            if (iy > lines) {
                if (scroll && !clearfull) {
                    scroll_input(1);
                    iy--;
                } else {
                    clear_input_window();
                }
            }
        }
        istarty = iendy = iy;
        set_refresh_pending((prompt && prompt->len) ?
            REF_LOGICAL : REF_PHYSICAL);
    }

    bufflush();
}

/* idel() assumes place is in bounds and != keyboard_pos. */
void idel(place)
    int place;
{
    int len;
    int oiey = iendy;

    if ((len = place - keyboard_pos) < 0) keyboard_pos = place;
    if (!echoflag) return;
    if (len < 0) ix += len;
    
    if (!visual) {
        if (ix < 1 || need_refresh) {
            physical_refresh();
            return;
        }
        if (len < 0) { bufputnc('\010', -len);  cx += len; }

    } else {
        /* visual */
        if (ix < 1) {
            iy -= ((-ix) / Wrap) + 1;
            ix = Wrap - ((-ix) % Wrap);
        }
        if (iy <= ystatus) {
            logical_refresh();
            return;
        }
        ipos();
    }

    if (len < 0) len = -len;

    if (visual && delete_char &&
        keybuf->len - keyboard_pos > 3 && len < Wrap/3)
    {
        /* hardware method */
        int i, space, pos;

        iendy = iy;
        if (ix + len <= Wrap) {
            for (i = len; i; i--) tp(delete_char);
            iendx = Wrap + 1 - len;
        } else {
            iendx = ix;
        }
        pos = keyboard_pos - ix + iendx;

        while (pos < keybuf->len) {
            if ((space = Wrap - iendx + 1) <= 0) {
                if (iendy == lines) break;   /* at end of window */
                xy(iendx = 1, ++iendy);
                for (i = len; i; i--) tp(delete_char);
                space = Wrap - len;
                if (space > keybuf->len - pos) space = keybuf->len - pos;
            } else {
                xy(iendx, iendy);
                if (space > keybuf->len - pos) space = keybuf->len - pos;
                bufputns(keybuf->s + pos, space);  cx += space;
            }
            iendx += space;
            pos += space;
        }

        /* erase tail */
        if (iendy < oiey) {
            crnl(); cx = 1; cy++;
            clear_line();
        }

    } else {
        /* redisplay method */
        iendx = ix;
        iendy = iy;
        ioutputs(keybuf->s + keyboard_pos, keybuf->len - keyboard_pos);

        /* erase tail */
        if (len > Wrap - cx + 1) len = Wrap - cx + 1;
        if (visual && clear_to_eos && (len > 2 || cy < oiey)) {
            tp(clear_to_eos);
        } else if (clear_to_eol && len > 2) {
            tp(clear_to_eol);
            if (visual && cy < oiey) clear_lines(cy + 1, oiey);
        } else {
            bufputnc(' ', len);  cx += len;
            if (visual && cy < oiey) clear_lines(cy + 1, oiey);
        }
    }
    
    /* restore cursor */
    if (visual) ipos();
    else { bufputnc('\010', cx - ix);  cx = ix; }

    bufflush();
}

int igoto(place)
    int place;
{
    int diff, new;

    if (place < 0) place = 0;
    if (place > keybuf->len) place = keybuf->len;
    diff = place - keyboard_pos;
    keyboard_pos = place;

    if (!diff) {
        bell(1);

    } else if (!echoflag) {
        /* no physical change */

    } else if (!visual) {
        ix += diff;
        if (ix < 1) {
            physical_refresh();
        } else if (ix > Wrap) {
            crnl();  cx = 1;  /* old text scrolls up, for continutity */
            physical_refresh();
        } else {
            cx += diff;
            if (diff < 0)
                bufputnc('\010', -diff);
            else for ( ; diff; diff--)
                bufputc(keybuf->s[place - diff]);
        }

    /* visual */
    } else {
        new = (ix - 1) + diff;
        iy += ndiv(new, Wrap);
        ix = nmod(new, Wrap) + 1;

        if ((iy > lines) && scroll) {
            scroll_input(iy - lines);
            ioutall(place - (ix - 1) - (iy - lines - 1) * Wrap);
            iy = lines;
            ipos();
        } else if ((iy < ystatus + 1) || (iy > lines)) {
            logical_refresh();
        } else {
            ipos();
        }
    }

    bufflush();
    return keyboard_pos;
}

void do_refresh()
{
    if (need_refresh == REF_LOGICAL) logical_refresh();
    else if (need_refresh == REF_PHYSICAL) physical_refresh();
}

void physical_refresh()
{
    if (visual) {
        ipos();
    } else {
        clear_input_line();
        ix = ((prompt?prompt->len:0) + (echoflag?keyboard_pos:0)) % Wrap + 1;
        ioutall((echoflag?keyboard_pos:0) - (ix - 1));
        bufputnc('\010', iendx - ix);  cx -= (iendx - ix);
    }
    bufflush();
    if (need_refresh <= REF_PHYSICAL) need_refresh = 0;
}

void logical_refresh()
{
    int kpos, nix, niy;

    if (!visual)
        oflush();  /* no sense refreshing if there's going to be output after */

    kpos = prompt ? -prompt->len : 0;
    nix = ((echoflag ? keyboard_pos : 0) - kpos) % Wrap + 1;

    if (visual) {
        niy = istarty + (keyboard_pos - kpos) / Wrap;
        if (niy <= lines) {
            clear_input_line();
        } else {
            clear_input_window();
            kpos += (niy - lines) * Wrap;
            niy = lines;
        }
        ioutall(kpos);
        ix = nix;
        iy = niy;
        ipos();
    } else {
        clear_input_line();
        ioutall(kpos);
        kpos += Wrap;
        while ((echoflag && kpos <= keyboard_pos) || kpos < 0) {
            crnl();  cx = 1;
            iendx = 1;
            ioutall(kpos);
            kpos += Wrap;
        }
        ix = nix;
        bufputnc('\010', iendx - nix);  cx -= (iendx - nix);
    }
    bufflush();
    if (need_refresh <= REF_LOGICAL) need_refresh = 0;
}

void update_prompt(newprompt)
    String *newprompt;
{
    String *oldprompt = prompt;

    if (oldprompt == moreprompt) return;
    prompt = (newprompt && newprompt->len) ? newprompt : NULL;
    if ((oldprompt && oldprompt->len) || prompt)
        set_refresh_pending(REF_LOGICAL);
}


/*****************************************************
 *                                                   *
 *                  OUTPUT HANDLING                  *
 *                                                   *
 *****************************************************/

/*************
 * Utilities *
 *************/

static void attributes_off(attrs)
    int attrs;
{
    CONST char *colorcode;

    if (attrs & F_HILITE) attrs |= hiliteattr;
    if (have_attr & attrs & F_SIMPLE) {
        if (attr_off) tp(attr_off);
        else {
            if (have_attr & attrs & F_UNDERLINE) tp(underline_off);
            if (have_attr & attrs & F_BOLD     ) tp(standout_off);
        }
    }
    if ((attrs & F_COLOR) && (colorcode = getvar("end_color"))) {
        bufputs(print_to_ascii(colorcode));
    }
}

static void attributes_on(attrs)
    int attrs;
{
    if (attrs & F_HILITE) attrs |= hiliteattr;
    if (have_attr & attrs & F_BELL)      bell(1);
    if (have_attr & attrs & F_UNDERLINE) tp(underline);
    if (have_attr & attrs & F_REVERSE)   tp(reverse);
    if (have_attr & attrs & F_FLASH)     tp(flash);
    if (have_attr & attrs & F_DIM)       tp(dim);
    if (have_attr & attrs & F_BOLD)      tp(bold ? bold : standout);
    if (attrs & F_COLOR) {
        CONST char *colorcode;
        extern CONST char *enum_color[];
        char buf[32];
        sprintf(buf, "start_color_%s", enum_color[attrs & F_COLORMASK]);
        if ((colorcode = getvar(buf))) {
            bufputs(print_to_ascii(colorcode));
        } else {
            sprintf(buf, "start_color_%d", attrs & F_COLORMASK);
            if ((colorcode = getvar(buf))) {
                bufputs(print_to_ascii(colorcode));
            }
        }
    }
}

static void hwrite(s, len, attrs, partials)
    CONST char *s;
    unsigned int len;
    int attrs;
    short *partials;
{
    cx += len;

    if (attrs & F_INDENT) {
        bufputnc(' ', wrapspace);  cx += wrapspace;
    }
    attrs &= F_HWRITE;

    if (!partials || !hilite) {
        if (hilite && attrs) attributes_on(attrs);
        bufputns(s, len);
        if (hilite && attrs) attributes_off(attrs);
    } else {
        short current = 0;
        int i;
        for (i = 0; i < len; ++i) {
            if ((partials[i] | attrs) != current) {
                if (current) attributes_off(current);
                current = (partials[i] | attrs);
                if (current) attributes_on(current);
            }
            bufputc(s[i]);
        }
        if (current) attributes_off(current);
    }
}

void reset_outcount()
{
    outcount = visual ? (scroll ? (ystatus - 1) : outcount) : lines - 1;
}

/* return TRUE if okay to print */
static int check_more()
{
    if (paused) return FALSE;
    if (!more) return TRUE;
    if (outcount-- > 0) return TRUE;

    /* status bar will be updated in oflush() to avoid scroll region problems */
    paused = 1;
    do_hook(H_MORE, NULL, "");
    return FALSE;
}

static int clear_more(new)
    int new;
{
    if (!paused) return 0;
    paused = 0;
    outcount = new;
    if (visual) {
        status_bar(STAT_MORE);
        if (!scroll) outcount = ystatus - 1;
    } else {
        prompt = fgprompt();
        clear_input_line();
    }
    set_refresh_pending(REF_PHYSICAL);
    return 1;
}

void tog_more()
{
    if (!more) clear_more(outcount);
    else reset_outcount();
}

int dokey_page()
{
    return clear_more(visual ? ystatus - 1 : lines - 1);
}

int dokey_hpage() 
{
    return clear_more((visual ? ystatus - 1 : lines - 1) / 2);
}

int dokey_line()
{
    return clear_more(1);
}

int screen_flush(selective)
    int selective;
{
    ListEntry *node, *next;

#define interesting(alin)  ((alin)->attrs & ~(F_NORM|F_GAG) || (alin)->partials)

    if (!paused) return 0;
    outcount = visual ? ystatus - 1 : lines - 1;
    for (node = tfscreen->u.queue->head; node; node = next) {
        next = node->next;
        if (!selective || !interesting((Aline*)node->datum)) {
            free_aline((Aline*)unlist(node, tfscreen->u.queue));
            tfscreen_size--;
        }
    }
    if (currentline && (!selective || !interesting(currentline))) {
        free_aline(currentline);
        currentline = NULL;
        wrap_offset = 0;
    }
    clear_more(outcount);
    screenout(new_aline("--- Output discarded ---", 0));
    return 1;
}

/* wrapline
 * Return a pointer to a static Aline containing the next physical line
 * to be printed.
 */
static Aline *wrapline()
{
    int remaining;
    static Aline *dead = NULL;
    static Aline result;
    /* result is not a "normal" Aline: it's static, its fields are actually
     * pointers into the fields of another Aline, and result.str[result.len]
     * is not neccessarily '\0'.
     */

    if (dead) { free_aline(dead); dead = NULL; }

    while (!currentline || ((currentline->attrs & F_GAG) && gag)) {
        if (currentline) free_aline(currentline);
        if (!(currentline = dequeue(tfscreen->u.queue))) return NULL;
        tfscreen_size--;
    }
    if (!check_more()) return NULL;

    remaining = currentline->len - wrap_offset;
    result.str = currentline->str + wrap_offset;
    result.attrs = currentline->attrs;
    result.partials = currentline->partials;
    if (result.partials) result.partials += wrap_offset;
    if (wrap_offset != 0) result.attrs &= ~F_BELL;
    if (wrap_offset != 0 && wrapflag && wrapspace < Wrap) {
        result.attrs |= F_INDENT;
    }
    result.len = wraplen(result.str, remaining, wrap_offset != 0);
    wrap_offset += result.len;

    if (wrap_offset == currentline->len) {
        dead = currentline;   /* remember so we can free it next time */
        currentline = NULL;
        wrap_offset = 0;
    }
    return &result;
}

/* returns length of prefix of str that will fit in {wrapsize} */
int wraplen(str, len, indent)
    CONST char *str;
    int len;
    int indent; /* flag */
{
    char incode = FALSE;
    int visible, total, max;

    if (emulation == EMUL_RAW) return len;

    /* Don't count nonprinting chars or "ansi" display codes (anything
     * starting with ESC and ending with a letter, for our purposes).
     */
    max = (indent && wrapspace < Wrap) ? (Wrap - wrapspace) : Wrap;
    for (visible = total = 0; total < len && visible < max; total++) {
        if (incode) {
            if (isalpha(str[total])) incode = FALSE;
        } else {
            if (isprint(str[total])) visible++;
            incode = (str[total] == '\33');
        }
    }
    if (total == len) return len;
    len = total;
    if (wrapflag)
        while (len && !isspace(str[len-1])) --len;
    return len ? len : total;
}


/****************
 * Main drivers *
 ****************/

/* write to tfscreen (no history) */
void screenout(aline)
    Aline *aline;
{
    if (!currentline && !tfscreen->u.queue->head) {
        /* shortcut if screen queue is empty */
        (currentline = aline)->links++;
    } else {
        aline->links++;
        enqueue(tfscreen->u.queue, aline);
        tfscreen_size++;
#define SCREENSPAM 20
        /* If this is one of several lines, it would be ineffecient to
         * flush each one individually; but waiting until the return to
         * main_loop() to flush might take too long (e.g., in an infinitely
         * looping macro, we'd never flush).  So we compromise, and flush
         * when we get a good sized chunk.
         */
        if (!paused && tfscreen_size > SCREENSPAM) oflush();
    }
}

void oflush()
{
    static int lastsize;
    int waspaused;

    if (!(waspaused = paused)) lastsize = 0;

    if (!screen_setup) output_novisual();
#ifdef SCREEN
    else if (scroll) output_scroll();
    else output_noscroll();
#endif

    if (paused) {
        if (visual) {
            if ((lastsize < 1000 &&
              (tfscreen_size+1) / morewait > lastsize / morewait) ||
                !waspaused)
                    status_bar(STAT_MORE);
        } else if (!waspaused) {
            prompt = moreprompt;
            set_refresh_pending(REF_LOGICAL);
        }
        lastsize = tfscreen_size + 1;
    }
}

static void output_novisual()
{
    Aline *line;
    int count = 0;

    while ((line = wrapline()) != NULL) {
        if (count == 0 && (keybuf->len || ix > 1)) {
            clear_input_line();
            set_refresh_pending(REF_PHYSICAL);
        }
        count++;
        hwrite(line->str, line->len, line->attrs, line->partials);
        crnl();  cx = 1; cy++;
        bufflush();
    }
}

#ifdef SCREEN
static void output_noscroll()
{
    Aline *line;

    while ((line = wrapline()) != NULL) {
        xy(1, (oy + 1) % (ystatus - 1) + 1);
        clear_line();
        xy(ox, oy);
        set_refresh_pending(REF_PHYSICAL);
        hwrite(line->str, line->len, line->attrs, line->partials);
        oy = oy % (ystatus - 1) + 1;
        bufflush();
    }
}

static void output_scroll()
{
    Aline *line;
    int count = 0;

    while ((line = wrapline()) != NULL) {
        if (change_scroll_region) {
            if (!count) {
                setscroll(1, ystatus - 1);
                xy(1, ystatus - 1);
                bufputc('\n');
                /* Some brain damaged emulators lose attributes under cursor
                 * when that '\n' is printed.  Too bad. */
            } else {
                crnl();  cx = 1;
            }
        } else {
            xy(1, 1);
            tp(delete_line);
            xy(1, ystatus - 1);
            tp(insert_line);
        }
        hwrite(line->str, line->len, line->attrs, line->partials);
        set_refresh_pending(REF_PHYSICAL);
        bufflush();
        count++;
    }
    if (count && change_scroll_region) setscroll(1, lines);
}
#endif

/***********************************
 * Interfaces with rest of program *
 ***********************************/

void tog_clock()
{
#ifdef HAVE_STRFTIME
    status_bar(STAT_CLOCK);
#else
    if (clock_flag) {
        tfputs("% clock not supported.", tferr);
        setvar("clock", "off", FALSE);
    }
#endif
}

void ch_hilite()
{
    CONST char *str;

    if (!(str = getvar("hiliteattr"))) return;
    if ((special_var[VAR_HILITEATTR].ival = parse_attrs((char **)&str)) < 0) {
        special_var[VAR_HILITEATTR].ival = 0;
    }
}

#define ANSI_CSI        (char)0233    /* ANSI terminal Command Sequence Intro */

/* Interpret embedded codes from a subset of ansi codes:
 * ansi attribute/color codes are converted to tf partial or line attrs;
 * tabs and backspaces were handled in handle_socket_input();
 * all other codes are ignored.
 */
void handle_ansi_attr(aline)
    Aline *aline;
{
    char *s, *t;
    short attrs = 0;
    int i;

    for (s = t = aline->str; *s; s++) {
        if (*s == ANSI_CSI || (*s == '\033' && *++s == '[')) {
            short new = attrs;
            s++;
            do {
                i = strtoi(&s);
                if (!i) new = 0;
                else if (i >= 30 && i <= 37)
                    new = (new & ~F_COLORMASK) | F_COLOR | (i - 30);
                else switch (i) {
                    case 1:   new |= F_BOLD;        break;
                    case 4:   new |= F_UNDERLINE;   break;
                    case 5:   new |= F_FLASH;       break;
                    case 7:   new |= F_REVERSE;     break;
                    case 22:  new &= ~F_BOLD;       break;
                    case 24:  new &= ~F_UNDERLINE;  break;
                    case 25:  new &= ~F_FLASH;      break;
                    case 27:  new &= ~F_REVERSE;    break;
                    default:  /* ignore it */       break;
                }
            } while (*s == ';' && *++s);

            if (*s == 'm') {
                if (!aline->partials) {
                    aline->partials = (short*)MALLOC(sizeof(short)*aline->len);
                    for (i = 0; i < aline->len; ++i)
                        aline->partials[i] = attrs;
                }
                attrs = new;
            } /* ignore any other CSI codes */

        } else if (isprint(*s)) {
            if (aline->partials)
                aline->partials[t - aline->str] = attrs;
            *t++ = *s;

        } else if (*s == '\07') {
            aline->attrs |= F_BELL;
        }
    }

    *t = '\0';
    aline->len = t - aline->str;
    /* optimization: replace array of char attrs with line attrs if possible. */
    if (aline->partials) {
        for (i = 1; i < aline->len; i++) {
            if (aline->partials[i] != aline->partials[0]) break;
        }
        if (i >= aline->len) {
            aline->attrs = aline->partials[0];
            FREE(aline->partials);
            aline->partials = NULL;
        }
    }
}

#ifdef DMALLOC
void free_term()
{
    tfclose(tfscreen);
    Stringfree(outbuf);
    Stringfree(moreprompt);
}
#endif
