/*************************************************************************
 *  TinyFugue - programmable mud client
 *  Copyright (C) 1993, 1994 Ken Keys
 *
 *  TinyFugue (aka "tf") is protected under the terms of the GNU
 *  General Public License.  See the file "COPYING" for details.
 ************************************************************************/
/* $Id: expand.c,v 34000.16 1994/10/09 05:28:14 hawkeye Exp $ */


/********************************************************************
 * Fugue macro text interpreter
 *
 * Written by Ken Keys
 * Interprets expressions and macro statements.
 * Performs substitutions for positional parameters, variables, macro
 * bodies, and expressions.
 ********************************************************************/

#include "config.h"
#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "tfio.h"
#include "macro.h"
#include "signals.h"
#include "socket.h"
#include "search.h"
#include "output.h"	/* igoto() */
#include "keyboard.h"	/* kb*() */
#include "expand.h"
#include "commands.h"
#include "command.h"

/* keywords: must be sorted, and numbered sequentially */
#define BREAK    '\200'
#define DO       '\201'
#define DONE     '\202'
#define ELSE     '\203'
#define ELSEIF   '\204'
#define ENDIF    '\205'
#define IF       '\206'
#define THEN     '\207'
#define WHILE    '\210'

/* note: all 2-char operators must have high bit set */
#define OP_EQUAL    '\300'
#define OP_NOTEQ    '\301'
#define OP_GTE      '\302'
#define OP_LTE      '\303'
#define OP_STREQ    '\304'
#define OP_STRNEQ   '\305'
#define OP_MATCH    '\306'
#define OP_NMATCH   '\307'
#define OP_ASSIGN   '\310'
#define OP_FUNC     '\311'

#define TYPE_ID     1
#define TYPE_STR    2
#define TYPE_INT    3

typedef struct Arg {
    char *start, *end;
} Arg;

typedef struct Value {
    int type;
    union {
        int ival;
        CONST char *sval;
        struct Value *next;		/* for valpool */
    } u;
} Value;

#define STACKSIZE 128

int user_result = 0;			/* result of last user command */

static Arg *argv;			/* shifted argument vector */
static int argc;			/* shifted argument count */
static int recur_count = 0;		/* expansion nesting count */
static int cmdsub_count = 0;		/* cmdsub nesting count */
static char *ip;			/* instruction pointer */
static int condition = 1;		/* checked by /if and /while */
static int evalflag = 1;		/* flag: should we evaluate? */
static int block = 0;			/* type of current block */
static int breaking = 0;		/* flag: are we /break'ing? */
static int stacktop = 0;
static Value *stack[STACKSIZE];
static Value *valpool = NULL;		/* freelist */

static int    NDECL(keyword);
static int    FDECL(list,(Stringp dest, int subs));
static int    FDECL(statement,(Stringp dest, int subs));
static Value *FDECL(newint,(int i));
static Value *FDECL(newstrid,(CONST char *str, int len, int type));
static void   FDECL(freeval,(Value *val));
static int    NDECL(top_int);
static int    FDECL(valint,(Value *val));
static CONST char  *FDECL(valstr,(Value *val));
static int    FDECL(slashsub,(Stringp dest));
static int    FDECL(macsub,(Stringp dest));
static int    FDECL(varsub,(Stringp dest));
static int    FDECL(backsub,(Stringp dest));
static int    FDECL(exprsub,(Stringp dest));
static int    FDECL(cmdsub,(Stringp dest));
static int    NDECL(expr);
static int    NDECL(comma_expr);
static int    NDECL(assignment_expr);
static int    NDECL(conditional_expr);
static int    NDECL(or_expr);
static int    NDECL(and_expr);
static int    NDECL(relational_expr);
static int    NDECL(additive_expr);
static int    NDECL(multiplicative_expr);
static int    NDECL(unary_expr);
static int    NDECL(function_expr);
static int    NDECL(primary_expr);
static int    FDECL(pushval,(Value *val));
static int    FDECL(reduce,(int op, int n));
static Value *FDECL(do_function,(int n));
static void   FDECL(parse_error,(CONST char *expect));

#define newstr(s,l)  (newstrid(s,l,TYPE_STR))
#define newid(s,l)   (newstrid(s,l,TYPE_ID))

#define end_of_statement(p) ((p)[0] == '%' && is_statend((p)[1]))
#define end_of_cmdsub(p) (cmdsub_count && *(p) == ')')

/* get Nth operand from stack (counting backwards from top) */
#define opd(N)     (stack[stacktop-(N)])
#define opdint(N)  valint(opd(N))
#define opdstr(N)  valstr(opd(N))

typedef struct ExprFunc {
    CONST char *name;		/* name invoked by user */
    unsigned min, max;		/* allowable argument counts */
} ExprFunc;

/* These must be in sorted order. */
static ExprFunc functab[] = {
    { "filename",	1,	1 },
    { "getpid",		0,	0 },
    { "idle",		0,	0 },
    { "kbdel",		1,	1 },
    { "kbgoto",		1,	1 },
    { "kbhead",		0,	0 },
    { "kblen",		0,	0 },
    { "kbpoint",	0,	0 },
    { "kbtail",		0,	0 },
    { "kbwordleft",	0,	0 },
    { "kbwordright",	0,	0 },
    { "mod",		2,	2 },
    { "rand",		0,	2 },
    { "regmatch",	2,	2 },
    { "strcat",		0,	(unsigned)-1 },
    { "strchr",		2,	2 },
    { "strcmp",		2,	2 },
    { "strlen",		1,	1 },
    { "strncmp",	3,	3 },
    { "strrchr",	2,	2 },
    { "strrep",		2,	2 },
    { "strstr",		2,	2 },
    { "substr",		3,	3 },
    { "tolower",	1,	1 },
    { "toupper",	1,	1 }
};

/* These must be in same order as functab[]. */
enum func_id {
    FN_FILENAME,
    FN_GETPID,
    FN_IDLE,
    FN_KBDEL,
    FN_KBGOTO,
    FN_KBHEAD,
    FN_KBLEN,
    FN_KBPOINT,
    FN_KBTAIL,
    FN_KBWORDLEFT,
    FN_KBWORDRIGHT,
    FN_MOD,
    FN_RAND,
    FN_REGMATCH,
    FN_STRCAT,
    FN_STRCHR,
    FN_STRCMP,
    FN_STRLEN,
    FN_STRNCMP,
    FN_STRRCHR,
    FN_STRREP,
    FN_STRSTR,
    FN_SUBSTR,
    FN_TOLOWER,
    FN_TOUPPER
};



static int expr()
{
    int ok;
    int stackbot = stacktop;
    int old_eflag = evalflag;
    int old_condition = condition;
    int old_breaking = breaking;

    ok = comma_expr();
    if (ok && stacktop < stackbot + 1) {
        tfputs("% internal error: expression stack underflow", tferr);
        ok = 0;
    } else if (ok && stacktop > stackbot + 1) {
        tfputs("% internal error: dirty expression stack", tferr);
        ok = 0;
    }
    while (stacktop > stackbot + ok) freeval(stack[--stacktop]);

    /* in case some short-circuit code left these in a weird state */
    evalflag = old_eflag;
    condition = old_condition;
    breaking = old_breaking;

    return ok;
}

int handle_test_command(args)
    char *args;
{
    char *saved_ip = ip;
    int result = 0;

    ip = args;
    if (expr()) {
        if (*ip) parse_error("operand");
        else result = valint(stack[stacktop-1]);
        freeval(stack[--stacktop]);
    }
    ip = saved_ip;
    return result;
}

int process_macro(body, args, subs)
    char *body;
    CONST char *args;
    int subs;
{
    Stringp buffer;
    Arg *true_argv;			/* unshifted argument vector */
    int true_argc;			/* unshifted argument count */
    int vecsize = 20, error = 0;
    int saved_cmdsub, saved_argc, saved_breaking;
    Arg *saved_argv;
    char *saved_ip;
    List scope[1];

    if (++recur_count > max_recur && max_recur) {
        tfputs("% Too many recursions.", tferr);
        recur_count--;
        return 0;
    }
    saved_cmdsub = cmdsub_count;
    saved_ip = ip;
    saved_argc = argc;
    saved_argv = argv;
    saved_breaking = breaking;

    ip = body;
    cmdsub_count = 0;

    newvarscope(scope);

    argc = 0;
    if (args) {
        argv = (Arg *)MALLOC(vecsize * sizeof(Arg));
        while (*args) {
            if (argc == vecsize)
                argv = (Arg*)REALLOC((char*)argv, sizeof(Arg)*(vecsize+=10));
            argv[argc].start = stringarg((char **)&args, &argv[argc].end);
            argc++;
        }
    } else {
        argv = NULL;
    }

    true_argv = argv;
    true_argc = argc;

    if (!error) {
        Stringinit(buffer);
        if (!list(buffer, subs)) user_result = 0;
        Stringfree(buffer);
    }

    if (argv) FREE(true_argv);
    nukevarscope();

    cmdsub_count = saved_cmdsub;
    ip = saved_ip;
    argc = saved_argc;
    argv = saved_argv;
    breaking = saved_breaking;
    recur_count--;
    return user_result;
}

static int list(dest, subs)
    Stringp dest;
    int subs;
{
    int oneslash, oldcondition, oldevalflag, oldblock;
    int iterations = 0, failed = 0;
    char *start = NULL;
    STATIC_BUFFER(mprefix_deep);

    /* Do NOT strip leading space here.  This allows user to type and send
     * lines with leading spaces (but completely empty lines are handled
     * by handle_input_line()).  During expansion, spaces AFTER a "%;"
     * or keyword will be skipped.
     */

    if (!*ip || end_of_cmdsub(ip)) user_result = 1;  /* empty list returns 1 */

    if (block == WHILE) start = ip;

    do /* while (*ip) */ {
        if (interrupted()) {
            tfputs("% macro evaluation interrupted.", tferr);
            return 0;
        }
        Stringterm(dest, 0);
        /* Lines begining with one "/" are tf commands.  Lines beginning
         * with multiple "/"s have the first removed, and are sent to server.
         */
        oneslash = (*ip == '/') && (*++ip != '/');
        if (oneslash) {
            oldblock = block;
            block = keyword();
            if (block && subs < SUB_NEWLINE) subs = SUB_NEWLINE;
            switch(block) {
            case WHILE:
                oldevalflag = evalflag;
                oldcondition = condition;
                if (!list(dest, subs)) failed = 1;
                else if (block == WHILE) {
                    eprintf("missing /do");
                    failed = 1;
                } else if (block == DO) {
                    eprintf("missing /done");
                    failed = 1;
                }
                evalflag = oldevalflag;
                condition = oldcondition;
                block = oldblock;
                if (failed) return 0;
                continue;
            case DO:
                if (oldblock != WHILE) {
                    eprintf("unexpected /do");
                    block = oldblock;
                    return 0;
                }
                evalflag = evalflag && condition;
                condition = user_result;
                if (breaking) breaking++;
                continue;
            case BREAK:
                if (!breaking && evalflag && condition) {
                    if ((breaking = atoi(ip)) <= 0) breaking = 1;
                }
                block = oldblock;
                continue;
            case DONE:
                if (oldblock != DO) {
                    eprintf("unexpected /done");
                    block = oldblock;
                    return 0;
                }
                if (breaking || !condition || !evalflag) {
                    if (breaking) breaking--;
                    evalflag = 0;  /* don't eval any trailing garbage */
                    return statement(dest, subs);  /* parse end of statement */
                } else if (++iterations > max_iter && max_iter) {
                    eprintf("too many iterations");
                    block = oldblock;
                    return 0;
                } else {
                    ip = start;
                    block = WHILE;
                    continue;
                }
            case IF:
                oldevalflag = evalflag;
                oldcondition = condition;
                if (!list(dest, subs)) failed = 1;
                else if (block == IF || block == ELSEIF) {
                    eprintf("missing /then");
                    failed = 1;
                } else if (block == THEN || block == ELSE) {
                    eprintf("missing /endif");
                    failed = 1;
                }
                evalflag = oldevalflag;
                condition = oldcondition;
                block = oldblock;
                if (failed) return 0;
                continue;
            case THEN:
                if (oldblock != IF && oldblock != ELSEIF) {
                    eprintf("unexpected /then");
                    block = oldblock;
                    return 0;
                }
                evalflag = evalflag && condition;
                condition = user_result;
                continue;
            case ELSEIF:
                if (oldblock != THEN) {
                    eprintf("unexpected /elseif");
                    block = oldblock;
                    return 0;
                }
                condition = !condition;
                continue;
            case ELSE:
                if (oldblock != THEN) {
                    eprintf("unexpected /else");
                    block = oldblock;
                    return 0;
                }
                condition = !condition;
                continue;
            case ENDIF:
                if (oldblock != THEN && oldblock != ELSE) {
                    eprintf("unexpected /endif");
                    block = oldblock;
                    return 0;
                }
                evalflag = 0; /* don't eval any trailing garbage */
                return statement(dest, subs);  /* parse end of statement */
            default:
                /* not a control statement */
                ip--;
                block = oldblock;
                break;
            }
        }

        if (!statement(dest, subs)) return 0;

        if (!breaking && evalflag && condition && (dest->len || !snarf)) {
            extern int invis_flag;
            if (subs == SUB_MACRO && (mecho + !invis_flag) >= 2) {
                int i;
                Stringterm(mprefix_deep, 0);
                for (i = 0; i < recur_count + cmdsub_count; i++)
                    Stringcat(mprefix_deep, mprefix);
                tfprintf(tferr, "%S %S", mprefix_deep, dest);
            }
            if (oneslash) {
                user_result = handle_command(dest);
            } else {
                extern int send_hook_level;
                if (send_hook_level || !do_hook(H_SEND, NULL, "%S", dest)) {
                    Stringadd(dest, '\n');
                    user_result = send_line(dest->s, dest->len);
                }
            }
        }

        if (end_of_cmdsub(ip)) break;
    } while (*ip);
    return 1;
}

static int keyword()
{
    char *end, save;
    CONST char **result;
    static CONST char *keyword_table[] = {
        "break", "do", "done", "else", "elseif",
        "endif", "if", "then", "while"
    };

    for (end = ip; *end && !isspace(*end) && *end != '%' && *end != ')'; end++);
    save = *end;
    *end = '\0';
    result = (CONST char **)binsearch((GENERIC*)&ip, (GENERIC*)keyword_table,
        sizeof(keyword_table)/sizeof(char*), sizeof(char*), gencstrcmp);
    *end = save;
    if (!result) return 0;
    for (ip = end; isspace(*ip); ip++);
    return BREAK + (result - keyword_table);
}

static int statement(dest, subs)
    Stringp dest;
    int subs;
{
    char *start;

    while (*ip) {
        if (*ip == '\\' && subs >= SUB_NEWLINE) {
            ++ip;
            if (!backsub(dest)) return 0;
        } else if (*ip == '/' && subs >= SUB_FULL) {
            ++ip;
            if (!slashsub(dest)) return 0;
        } else if (*ip == '%' && subs >= SUB_NEWLINE) {
            ++ip;
            if (end_of_statement(ip-1)) {
                while (dest->len && isspace(dest->s[dest->len-1]))
                    Stringterm(dest, dest->len-1);  /* nuke spaces before %; */
                ++ip;
                while (isspace(*ip)) ip++; /* skip space after %; */
                break;
            } else if (*ip == '%') {
                while (*ip == '%') Stringadd(dest, *ip++);
            } else if (subs >= SUB_FULL) {
                if (!varsub(dest)) return 0;
            } else {
                Stringadd(dest, '%');
            }
        } else if (*ip == '$' && subs >= SUB_FULL) {
            ++ip;
            if (*ip == '[') {
                ++ip;
                if (!exprsub(dest)) return 0;
            } else if (*ip == '(') {
                ++ip;
                if (!cmdsub(dest)) return 0;
            } else {
                if (!macsub(dest)) return 0;
            }
        } else if (subs >= SUB_FULL && end_of_cmdsub(ip)) {
            break;
        } else {
            /* is_statmeta() is much faster than all those if statements. */
            for (start = ip++; *ip && !is_statmeta(*ip); ip++);
            Stringncat(dest, start, ip - start);
        }
    }

    return 1;
}

static int slashsub(dest)
    Stringp dest;
{
    if (breaking || !evalflag || !condition)
        while (*ip == '/') ++ip;
    else if (*ip == '/' && oldslash)
        while (*ip == '/') Stringadd(dest, *ip++);
    else
        Stringadd(dest, '/');
    return 1;
}

static Value *newint(i)
    int i;
{
    Value *val;

    if (breaking || !evalflag || !condition) return NULL;
    palloc(val, Value, valpool, u.next);
    val->type = TYPE_INT;
    val->u.ival = i;
    return val;
}

static Value *newstrid(str, len, type)
    CONST char *str;
    int len, type;
{
    Value *val;
    char *new;

    if (breaking || !evalflag || !condition) return NULL;
    palloc(val, Value, valpool, u.next);
    val->type = type;
    new = strncpy((char *)MALLOC(len + 1), str, len);
    new[len] = '\0';
    val->u.sval = new;
    return val;
}

static void freeval(val)
    Value *val;
{
    if (!val) return;   /* val may have been placeholder for short-circuit */
    if (val->type == TYPE_STR || val->type == TYPE_ID) FREE(val->u.sval);
    pfree(val, valpool, u.next);
}

/* convert the top item on the stack to an int */
static int top_int()
{
    int ival;
    if (!opd(1)) return 0;
    if (opd(1)->type != TYPE_INT) {
        ival = valint(opd(1));
        FREE(opd(1)->u.sval);
        opd(1)->u.ival = ival;
        opd(1)->type = TYPE_INT;
    }
    return opd(1)->u.ival;
}

/* return integer value of item */
static int valint(val)
    Value *val;
{
    CONST char *str;
    int result;

    if (val->type == TYPE_INT) return val->u.ival;
    str = val->u.sval;
    if (val->type == TYPE_ID) {
        str = getnearestvar(str, &result);
        if (result != 0 || !str) return result;
    }
    while (isspace(*str)) ++str;
    if (*str == '-' || *str == '+') return atoi(str);
    return (isdigit(*str)) ? parsetime((char **)&str, NULL) : 0;
}

/* return string value of item */
static CONST char *valstr(val)
    Value *val;
{
    CONST char *str;
    STATIC_BUFFER(buffer);

    switch (val->type) {
        case TYPE_INT:  Sprintf(buffer, 0, "%d", val->u.ival); return buffer->s;
        case TYPE_STR:  return val->u.sval;
        case TYPE_ID:   return (str=getnearestvar(val->u.sval,NULL)) ? str :
                            (Stringterm(buffer, 0), buffer->s);
    }
    return NULL; /* impossible */
}

static int pushval(val)
    Value *val;
{
    if (stacktop == STACKSIZE) {
        eprintf("expression stack overflow");
        return 0;
    }
    stack[stacktop++] = val;
    return 1;
}

static void parse_error(expect)
    CONST char *expect;
{
    if (*ip) {
        char *end = ip + 1, save;
        if (isalnum(*ip) || is_quote(*ip)) {
            while (isalnum(*end)) end++;
        }
        save = *end;
        *end = '\0';
        eprintf("expression parse error: expected %s, found '%s'", expect, ip);
        *end = save;
    } else {
        eprintf("expression parse error: expected %s, found end of expression.",
            expect);
    }
}

/* Pop n operands, apply op to them, and push result */
static int reduce(op, n)
    int op;   /* operator */
    int n;    /* number of operands */
{
    Value *val = NULL;
    CONST char *str;
    STATIC_BUFFER(buf);
    int i; /* scratch */

    if (stacktop < n) {
        tfputs("% internal error:  stack underflow in reduce()", tferr);
        return 0;
    }

    if (!evalflag || !condition || breaking) {
        /* Just maintain the depth of the stack, for parsing purposes;
         * the (nonexistant) value will never be used.
         */
        stacktop -= n - 1;
        return 1;
    }

    Stringterm(buf, 0);

    switch (op) {
    case OP_ASSIGN: if (opd(2)->type != TYPE_ID) {
                        tfputs("% illegal left side of assignment.", tferr);
                    } else {
                        str = setnearestvar(opd(2)->u.sval, opdstr(1));
                        val = newstr(str, strlen(str));
                    }
                    break;
    case '?':       val = opdint(3) ? opd(2) : opd(1);
                    /* Only the condition should be freed:  the selected opd
                     * is reused, and the other is NULL because it was never
                     * evaluated.  So we fool the cleanup into doing this.
                     */
                    stacktop -= 2;
                    n -= 2;
                    break;
    case '|':       val = newint(opdint(2) ? opdint(2) : opdint(1));    break;
    case '&':       val = newint(opdint(2) ? opdint(1) : 0);            break;
    case '>':       val = newint(opdint(2) > opdint(1));                break;
    case '<':       val = newint(opdint(2) < opdint(1));                break;
    case '=':       /* fall thru to OP_EQUAL */
    case OP_EQUAL:  val = newint(opdint(2) == opdint(1));               break;
    case OP_NOTEQ:  val = newint(opdint(2) != opdint(1));               break;
    case OP_GTE:    val = newint(opdint(2) >= opdint(1));               break;
    case OP_LTE:    val = newint(opdint(2) <= opdint(1));               break;
    case OP_STREQ:  val = newint(strcmp(opdstr(2), opdstr(1)) == 0);    break;
    case OP_STRNEQ: val = newint(strcmp(opdstr(2), opdstr(1)) != 0);    break;
    case OP_MATCH:  val = newint(smatch_check(opdstr(1)) &&
                        smatch(opdstr(1),opdstr(2))==0);
                    break;
    case OP_NMATCH: val = newint(smatch_check(opdstr(1)) &&
                        smatch(opdstr(1),opdstr(2))!=0);
                    break;
    case '+':       val = newint(((n>1) ? opdint(2) : 0) + opdint(1));  break;
    case '-':       val = newint(((n>1) ? opdint(2) : 0) - opdint(1));  break;
    case '*':       val = newint(opdint(2) * opdint(1));                break;
    case '/':       if (block == IF && opd(1)->type == TYPE_ID)
                        /* common error: "/if /test <expr> /then ..." */
                        tfprintf(tferr,
                            "%% warning: possibly missing %%; before /%s",
                            opd(1)->u.sval);
                        if ((i = opdint(1)) == 0)
                            eprintf("division by zero");
                        else
                            val = newint(opdint(2) / i);
                    break;
    case '!':       val = newint(!opdint(1));                           break;
    case OP_FUNC:   val = do_function(n);                               break;
    default:        tfprintf(tferr,
                        "%% internal error: reduce: unknown op %c", op);
                    break;
    }

    stacktop -= n;
    while (n) freeval(stack[stacktop + --n]);
    if (val) pushval(val);
    return !!val;
}

static Value *do_function(n)
    int n;    /* number of operands (including function id) */
{
    Handler *handler;
    ExprFunc *funcrec;
    Macro *macro;
    int i, j, len;
    CONST char *id, *str, *ptr;
    extern Stringp keybuf;
    extern int keyboard_pos;
    extern TIME_T keyboard_time;
    regexp *re;
    STATIC_BUFFER(scratch);

    if (opd(n)->type != TYPE_ID) {
        tfputs("% function name must be an identifier.", tferr);
        return NULL;
    }
    id = opd(n--)->u.sval;

    funcrec = (ExprFunc *)binsearch((GENERIC*)&id, (GENERIC*)functab,
        sizeof(functab)/sizeof(ExprFunc), sizeof(ExprFunc), genstrcmp);

    if (funcrec) {
        if (n < funcrec->min || n > funcrec->max) {
            tfprintf(tferr, "%% %s: incorrect number of arguments", id);
            return NULL;
        }
        switch (funcrec - functab) {

        case FN_MOD:
            return newint(opdint(2) % opdint(1));

        case FN_RAND:
            if (n == 0) return newint(RAND());
            i = (n==1) ? 0 : opdint(2);
            if (i < 0) i = 0;
            j = opdint(1) - (n==1);
            return newint((j > i) ? RRAND(i, j) : i);

        case FN_IDLE:
            return newint((int)(time(NULL) - keyboard_time));

        case FN_FILENAME:
            str = expand_filename(opdstr(1));
            return newstr(str, strlen(str));

        case FN_GETPID:
            return newint((int)getpid());

        case FN_REGMATCH:
            if (!(re = regcomp((char*)opdstr(2)))) return newint(0);
            return newint(regexec_and_hold(re, opdstr(1), TRUE));

        case FN_STRCAT:
            for (Stringterm(scratch, 0); n; n--)
                Stringcat(scratch, opdstr(n));
            return newstr(scratch->s, scratch->len);

        case FN_STRREP:
            i = opdint(1);
            str = opdstr(2);
            for (Stringterm(scratch, 0); i > 0; i--)
                Stringcat(scratch, str);
            return newstr(scratch->s, scratch->len);

        case FN_STRCMP:
            return newint(strcmp(opdstr(2), opdstr(1)));

        case FN_STRNCMP:
            return newint(strncmp(opdstr(3), opdstr(2), opdint(1)));

        case FN_STRLEN:
            return newint(strlen(opdstr(1)));

        case FN_SUBSTR:
            len = strlen(str = opdstr(3));
            if ((i = opdint(2)) < 0) i = 0;
            if (i > len) i = len;
            len -= i;
            if ((j = opdint(1)) < 0) j = 0;
            if (j > len) j = len;
            return newstr(str + i, j);

        case FN_STRSTR:
            str = opdstr(2);
            ptr = STRSTR(str, opdstr(1));
            return newint(ptr ? (ptr - str) : -1);

        case FN_STRCHR:
            str = opdstr(2);
            ptr = opdstr(1);
            for (i = 0; str[i]; i++)
                for (j = 0; ptr[j]; j++)
                    if (str[i] == ptr[j]) return newint(i);
            return newint(-1);

        case FN_STRRCHR:
            str = opdstr(2);
            ptr = opdstr(1);
            for (i = strlen(str) - 1; i >= 0; i--)
                for (j = 0; ptr[j]; j++)
                    if (str[i] == ptr[j]) return newint(i);
            return newint(-1);

        case FN_TOLOWER:
            Stringterm(scratch, 0);
            for (str = opdstr(1); *str; str++)
                Stringadd(scratch, lcase(*str));
            return newstr(scratch->s, scratch->len);

        case FN_TOUPPER:
            Stringterm(scratch, 0);
            for (str = opdstr(1); *str; str++)
                Stringadd(scratch, ucase(*str));
            return newstr(scratch->s, scratch->len);

        case FN_KBHEAD:
            return newstr(keybuf->s, keyboard_pos);

        case FN_KBTAIL:
            return newstr(keybuf->s + keyboard_pos, keybuf->len - keyboard_pos);

        case FN_KBPOINT:
            return newint(keyboard_pos);

        case FN_KBGOTO:
            return newint(igoto(opdint(1)));

        case FN_KBDEL:
            return (newint(do_kbdel(opdint(1))));

        case FN_KBWORDLEFT:
            return newint(do_kbwordleft());

        case FN_KBWORDRIGHT:
            return newint(do_kbwordright());

        case FN_KBLEN:
            return newint(keybuf->len);

        default:
            /* impossible */
            return NULL;

        }

    } else if ((macro = find_macro(id)) || (handler = find_command(id))) {
        extern CONST char *current_command;
        CONST char *old_command;
        if (n > 1) {
            tfprintf(tferr, "%% %s:  command or macro called as function must have 0 or 1 argument", id);
            return NULL;
        }
        old_command = current_command;
        current_command = id;
        j = (macro) ?
            do_macro(macro, opdstr(1)) :
            (*handler)(Stringcpy(scratch, n ? opdstr(1) : "")->s);
        current_command = old_command;
        return newint(j);
    }

    tfprintf(tferr, "%% %s: no such function", id);
    return NULL;
}

static int comma_expr()
{
    if (!assignment_expr()) return 0;
    while (*ip == ',') {
        ip++;
        freeval(stack[--stacktop]);
        if (!assignment_expr()) return 0;
    }
    return 1;
}

static int assignment_expr()
{
    if (!conditional_expr()) return 0;
    if (ip[0] == ':' && ip[1] == '=') {
        ip += 2;
        if (!assignment_expr()) return 0;
        if (!reduce(OP_ASSIGN, 2)) return 0;
    }
    return 1;
}

static int conditional_expr()
{
    int old_eflag = evalflag;
    if (!or_expr()) return 0;
    if (*ip == '?') {
        ip++;
        top_int();
	evalflag = old_eflag && opd(1)->u.ival;  /* short-circuit */
        if (!comma_expr()) return 0;
        if (*ip != ':') {
            parse_error("':'");
            return 0;
        }
        ip++;
	evalflag = old_eflag && !opd(2)->u.ival;  /* short-circuit */
        if (!conditional_expr()) return 0;
	evalflag = old_eflag;
        if (!reduce('?', 3)) return 0;
    }
    return 1;
}

static int or_expr()
{
    int old_eflag = evalflag;
    if (!and_expr()) return 0;
    while (*ip == '|') {
        ip++;
	if (top_int()) evalflag = 0;  /* short-circuit */
        if (!and_expr()) return 0;
        if (!reduce('|', 2)) return 0;
    }
    evalflag = old_eflag;
    return 1;
}

static int and_expr()
{
    int old_eflag = evalflag;
    if (!relational_expr()) return 0;
    while (*ip == '&') {
        ip++;
	if (!top_int()) evalflag = 0;  /* short-circuit */
        if (!relational_expr()) return 0;
        if (!reduce('&', 2)) return 0;
    }
    evalflag = old_eflag;
    return 1;
}

static int relational_expr()
{
    char op;
    if (!additive_expr()) return 0;
    while (*ip) {
        if      (ip[0] == '=' && ip[1] == '~') op = OP_STREQ;
        else if (ip[0] == '!' && ip[1] == '~') op = OP_STRNEQ;
        else if (ip[0] == '=' && ip[1] == '/') op = OP_MATCH;
        else if (ip[0] == '!' && ip[1] == '/') op = OP_NMATCH;
        else if (ip[0] == '>' && ip[1] == '=') op = OP_GTE;
        else if (ip[0] == '<' && ip[1] == '=') op = OP_LTE;
        else if (ip[0] == '=' && ip[1] == '=') op = OP_EQUAL;
        else if (ip[0] == '!' && ip[1] == '=') op = OP_NOTEQ;
        else if (*ip == '<' || *ip == '>' || *ip == '=') op = *ip;
        else break;
        ip += 1 + !!(op & 0x80);     /* high bit means it's a 2-char op */
        if (!additive_expr()) return 0;
        if (!reduce(op, 2)) return 0;
    }
    return 1;
}

static int additive_expr()
{
    char op;
    if (!multiplicative_expr()) return 0;
    while (*ip == '+' || *ip == '-') {
        op = *ip++;
        if (!multiplicative_expr()) return 0;
        if (!reduce(op, 2)) return 0;
    }
    return 1;
}

static int multiplicative_expr()
{
    char op;

    if (!unary_expr()) return 0;
    while (*ip == '*' || *ip == '/') {
        op = *ip++;
        if (!unary_expr()) return 0;
        if (!reduce(op, 2)) return 0;
    }
    return 1;
}

static int unary_expr()
{
    char op;

    while (isspace(*ip)) ip++;
    op = *ip;
    if (op == '!' || op == '+' || op == '-') {
        ++ip;
        if (!unary_expr()) return 0;
        if (!reduce(op, 1)) return 0;
        return 1;
    } else return function_expr();
}

static int function_expr()
{
    int n = 0;

    if (!primary_expr()) return 0;
    if (*ip == '(') {
        for (++ip; isspace(*ip); ip++);
        if (*ip != ')') {
            while (1) {
                if (!assignment_expr()) return 0;
                n++;
                if (*ip == ')') break;
                if (*ip != ',') { parse_error("',' or ')'"); return 0; }
                ++ip;
            }
        }
        for (++ip; isspace(*ip); ip++);
        if (!reduce(OP_FUNC, n + 1)) return 0;
    }
    return 1;
}

static int primary_expr()
{
    char *end, quote;
    STATIC_BUFFER(buffer);

    while (isspace(*ip)) ip++;
    if (*ip == '(') {
        ++ip;
        if (!comma_expr()) return 0;
        if (*ip != ')') {
            tfputs("% missing )", tferr);
            return 0;
        }
        ++ip;
    } else if (isdigit(*ip)) {
        if (!pushval(newint(parsetime(&ip, NULL)))) return 0;
    } else if (is_quote(*ip)) {
        Stringterm(buffer, 0);
        quote = *ip;
        for (ip++; *ip && *ip != quote; Stringadd(buffer, *ip++))
            if (*ip == '\\' && (ip[1] == quote || ip[1] == '\\')) ip++;
        if (!*ip) {
            eprintf("unmatched %c in expression string", quote);
            return 0;
        }
        ip++;
        if (!pushval(newstr(buffer->s, buffer->len))) return 0;
    } else if (isalpha(*ip) || *ip == '_') {
        for (end = ip + 1; isalnum(*end) || *end == '_'; end++);
        if (!pushval(newid(ip, end - ip))) return 0;
        ip = end;
    } else if (*ip == '%') {
        ++ip;
        Stringterm(buffer, 0);
        if (!varsub(buffer)) return 0;
        if (!pushval(newstr(buffer->s, buffer->len))) return 0;
    } else {
        parse_error("operand");
        return 0;
    }
    
    while (isspace(*ip)) ip++;
    return 1;
}

static int exprsub(dest)
    Stringp dest;
{
    int result = 0;
    Value *val;

    while (isspace(*ip)) ip++;
    if (!expr()) return 0;
    val = stack[--stacktop];
    if (!*ip || end_of_statement(ip)) {
        eprintf("unmatched $[");
    } else if (*ip != ']') {
        parse_error("operator");
    } else {
        if (val) Stringcat(dest, valstr(val));
        ++ip;
        result = 1;
    }
    freeval(val);
    return result;
}

static int cmdsub(dest)
    Stringp dest;
{
    TFILE *oldout, *olderr, *file;
    extern TFILE *tfout, *tferr;
    Stringp buffer;
    int result, first = 1;
    Aline *aline;

    file = tfopen(NULL, "q");
    cmdsub_count++;
    oldout = tfout;
    olderr = tferr;
    tfout = file;
    /* tferr = file; */

    Stringinit(buffer);
    result = list(buffer, SUB_MACRO);
    Stringfree(buffer);

    tferr = olderr;
    tfout = oldout;
    cmdsub_count--;

    if (*ip != ')') {
        eprintf("unmatched (");
        tfclose(file);
        return 0;
    }

    while ((aline = dequeue(file->u.queue))) {
        if (!((aline->attrs & F_GAG) && gag)) {
            if (!first) Stringadd(dest, ' ');
            first = 0;
            Stringncat(dest, aline->str, aline->len);
        }
        free_aline(aline);
    }

    tfclose(file);
    ip++;
    return result;
}

static int macsub(dest)
    Stringp dest;
{
    STATIC_BUFFER(buffer);
    CONST char *body, *start;
    int bracket;

    if (*ip == '$') {
        while (*ip == '$') Stringadd(dest, *ip++);
        return 1;
    }

    Stringterm(buffer, 0);
    if ((bracket = (*ip == '{'))) ip++;
    while (*ip) {
        if (*ip == '\\') {
            ++ip;
            if (!backsub(dest)) return 0;
        } else if (end_of_statement(ip) || end_of_cmdsub(ip)) {
            break;
        } else if (*ip == '/') {
            ++ip;
            if (!slashsub(buffer)) return 0;
        } else if (*ip == '}') {
            /* note: in case of "%{var-$mac}", we break even if !bracket. */
            break;
        } else if (!bracket && isspace(*ip)) {
            break;
        } else if (*ip == '$') {
            if (ip[1] == '$') {
                while(*++ip == '$') Stringadd(buffer, *ip);
            } else {
                if (!bracket) break;
                else Stringadd(buffer, *ip++);
            }
        } else if (*ip == '%') {
            ++ip;
            if (!varsub(buffer)) return 0;
        } else {
            for (start = ip++; *ip && !ispunct(*ip) && !isspace(*ip); ip++);
            Stringncat(buffer, start, ip - start);
        }
    }
    if (bracket) {
        if (*ip != '}') {
            eprintf("unmatched ${");
            return 0;
        } else ip++;
    } else if (*ip == '$') {
        ip++;
    }

    if (breaking || !evalflag || !condition) return 1;

    if ((body = macro_body(buffer->s))) Stringcat(dest, body);
    else tfprintf(tferr, "%% macro not defined: %S", buffer);
    if (mecho) tfprintf(tferr, "%s $%S --> %s", mprefix, buffer, body);
    return 1;
}

static int backsub(dest)
    Stringp dest;
{
    if (isdigit(*ip)) {
        Stringadd(dest, strtochr(&ip));
    } else if (!backslash) {
        Stringadd(dest, '\\');
    } else if (*ip) {
        Stringadd(dest, *ip++);
    }
    return 1;
}

static int varsub(dest)
    Stringp dest;
{
    CONST char *value, *start;
    int bracket, except, ell = FALSE, pee = FALSE, n = -1;
    int first, last, empty = 0;
    STATIC_BUFFER(selector);
    char *p;

    if (*ip == '%') {
        while (*ip == '%') Stringadd(dest, *ip++);
        return 1;
    }
    if (!*ip || isspace(*ip)) {
        Stringadd(dest, '%');
        return 1;
    }

    Stringterm(selector, 0);
    if ((bracket = (*ip == '{'))) ip++;

    if (*ip && *ip != '}')
        do {
            Stringadd(selector, *ip++);
        } while (*ip && (isalnum(*ip) || *ip == '_'));

    if (selector->s[0] == '#' && selector->len == 1) {
        if (!breaking && evalflag && condition)
            Sprintf(dest, SP_APPEND, "%d", argc);
        empty = FALSE;
    } else if (selector->s[0] == '?' && selector->len == 1) {
        if (!breaking && evalflag && condition)
            Sprintf(dest, SP_APPEND, "%d", user_result);
        empty = FALSE;
    } else {
        p = selector->s;
        if ((except = (p[0] == '-' && (isdigit(p[1]) || ucase(p[1])=='L'))))
            p++;
        if ((ell = (ucase(*p) == 'L')) || (pee = (ucase(*p) == 'P'))) {
            ++p;
            n = 1;
        }
        if (isdigit(*p)) {
            n = strtoi(&p);
        } else if (*p == '*') {
            ++p;
            n = 0;
        }
        if (*p) n = -1, ell = pee = FALSE;

        if (!breaking && evalflag && condition) {

            if (pee) {
                empty = (regsubstr(dest, n) <= 0);
                n = -1;
            } else if (ell) {
                if (except) first = 0, last = argc - n - 1;
                else first = last = argc - n;
            } else if (n == 0) {
                first = 0, last = argc - 1;
            } else if (n > 0) {
                if (except) first = n, last = argc - 1;
                else first = last = n - 1;
            } else if (cstrcmp(selector->s, "R") == 0) {
                if (argc > 0) {
                    n = 1;
                    first = last = RRAND(0, argc-1);
                } else empty = TRUE;
            } else {
                value = getnearestvar(selector->s, NULL);
                if (!(empty = !value || !*value))
                    Stringcat(dest, value);
            }

            if (n >= 0) {
                empty = (first > last || first < 0 || last >= argc);
                if (!empty) {
                    Stringncat(dest, argv[first].start,
                        argv[last].end - argv[first].start);
                }
            }

        } /* eval */
    }

    if (*ip == '-') {
        int oldevalflag = evalflag;
        evalflag = empty;
        ++ip;
        while (*ip) {
            if (end_of_statement(ip) || end_of_cmdsub(ip)) {
                break;
            } else if (bracket && *ip == '}') {
                break;
            } else if (!bracket && isspace(*ip)) {
                break;
            } else if (*ip == '%') {
                ++ip;
                if (!varsub(dest)) return 0;
            } else if (*ip == '$') {
                ++ip;
                if (!macsub(dest)) return 0;
            } else if (*ip == '/') {
                ++ip;
                if (!slashsub(dest)) return 0;
            } else {
                for (start = ip++; *ip && isalnum(*ip); ip++);
                if (!breaking && evalflag && condition)
                    Stringncat(dest, start, ip - start);
            }
        }

        evalflag = oldevalflag;
    }

    if (bracket) {
        if (*ip != '}') {
            eprintf("unmatched %{");
            return 0;
        } else ip++;
    }

    return 1;
}

int handle_shift_command(args)
    char *args;
{
    int count;
    int error;

    if (!argv) return 1;
    count = (*args) ? atoi(args) : 1;
    if (count < 0) return 0;
    if ((error = (count > argc))) count = argc;
    argc -= count;
    argv += count;
    return !error;
}

#ifdef DMALLOC
void free_expand()
{
    Value *val;
    while (valpool) {
       val = valpool;
       valpool = valpool->u.next;
       FREE(val);
    }
}
#endif
