/*	game.c		8/3/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"

/*----------------------------------------------------------------------
 *	startgame -- start a game
 *
 * This function is called in response to the -start command line
 * option to start a game with another user.  It allocates a game
 * record and fills it in, then sends a START packet to the opponent.
 * The arguments are:
 *	The email address of the opponent
 *	The direction I want to play
 *	The color I want to play
 *	The color I want the opponent to play
 *	The F_JACOBY/F_CRAWFORD/F_EUROPE/F_INVERT/F_PERM flags, if desired.
 *	The match value (number of points to play to)
 *	The time to use as the start time of the game (0 = current time)
 *----------------------------------------------------------------------
 */

struct game *startgame(addr,d,mc,oc,flgs,mch,stime)
char *addr;			/* path to opponent */
int d;				/* my direction */
char mc, oc;			/* my color, opponent's color */
int flgs;			/* flags (F_*) */
int mch;			/* match score */
long stime;			/* start time (0 = current time) */
{
struct game *g;
char c1, c2, *newid;
struct people *ppl;

newid = makeid();		/* give it a unique id */
g = addgame();			/* allocate new game */
g->gameid = newid;		/* store new id */
if (rc.debug & DB_GSTART) {
	message("DB-startgame:\tstarted game %s\n",newid);
	message("\t\twith %s flags=%04x match=%d\n",addr,flgs,mch);
	}
if ( (ppl = findppl(addr,P_ADDR|P_ALIAS)) != NULL) {	/* we know this guy */
	g->opaddr = save(ppl->addr);	/* copy out people info */
	g->opname = save(ppl->name);
	g->myaddr = save(ppl->myaddr);
	g->ppl = ppl;
	}
else {					/* new opponent */
	g->opaddr = save(addr);		/* save his address */
	g->opname = NULL;		/* don't know his name yet */
	g->myaddr = save(rc.myaddr);	/* store my return address */
	newppl(g);			/* make up a people record */
	}
g->mycolor = mc;		/* set starting colors */
g->opcolor = oc;
g->mydir = d;			/* set starting directions */
g->opdir = REV(d);
g->gameval = 1;			/* no doubles yet */
g->adcnt = 0;			/* no autodoubles yet */
g->admax = rc.autodouble;	/* max allowed autodoubles */
g->flags = flgs & (F_JACOBY|F_CRAWFORD|F_PERM|F_EUROPE|F_INVERT);
g->state = ST_OPSTART;		/* need to send first roll */
g->seq = 1;			/* start with sequence number = 1 */
g->notify = notify;		/* copy notify address (if any) */
if (d > 0) {
	c1 = mc;	/* upbound color is mine */
	c2 = oc;	/* downbound color is opponent's */
	}
else {
	c1 = oc;	/* upbound color is opponent's */
	c2 = mc;	/* downbound color is mine */
	}
clearmvs(g->mvs);
clearmvs(g->opmvs);
newboard(g->opbd,c1,c2);	/* set up boards for new game */
newboard(g->mybd,c1,c2);
newboard(g->board,c1,c2);
g->mtotal = mch;
g->mvs[0].roll = Rolldie();	/* roll an initial die */
if (stime == 0L)
	g->starttime = time( (long *) 0);
else
	g->starttime = stime;	/* hack to detect duplicate remotestart pkts */
g->lastacc = g->starttime;
sendpkt(g,START);		/* send the start message */
return(g);			/* and return pointer to new game */
}


/*----------------------------------------------------------------------
 *	makeid -- create a unique game identifier.
 *
 * This function creates a string that is guaranteed unique among all
 * ldb games worldwide, provided that email addresses are unique.
 * This should be a good assumption, since if there is a duplicate,
 * the users with the duplicate id's will have a great deal of difficulty
 * getting mail delivered, and therefore won't be able to play ldb anyway.
 * To make id's created by the same user unique, the time is
 * appended to the mail address; to make sure the time is unique when
 * the user creates more than 1 game per second, the games list is searched
 * for a new id before it is returned and, if it is found, we sleep for
 * 1 second and try again.
 *----------------------------------------------------------------------
 */

char *makeid()
{
char *n;

if ( (n = calloc(strlen(rc.myaddr)+10,1)) == NULL)
	fatal("ERROR: Out of memory!");
do {
	sprintf(n,"%s|%08x",rc.myaddr,time((long *)0));
	if (findgame(n) == NULL)
		return(n);
	sleep(1);
	} while (1);
}


/*---------------------------------------------------------------------------
 *	addgame -- allocate a game struct and link it into the game list
 *
 * This function allocates a game structure and links it into the
 * doubly-linked game list.  The head of this list is ghead, and the
 * tail is gtail.
 *
 * NOTE: the memory-zeroing feature of calloc is depended on to
 *	 initialize the allocated game struct.
 *---------------------------------------------------------------------------
 */

struct game *addgame()
{
struct game *g;

if ( (g = (struct game *)calloc(sizeof(struct game),1)) == NULL)
	fatal("Out of memory!");
g->next = NULL;
if (gtail == NULL) {		/* this is the first game in the list */
	ghead = g;
	gtail = g;
	g->prev = NULL;
	}
else {
	g->prev = gtail;	/* link onto end of list */
	gtail->next = g;
	gtail = g;
	}
return(g);
}


/*----------------------------------------------------------------------
 *	deletegame -- delete a game from the game list
 *
 * This function removes a game from the game list by linking around
 * it, then frees the memory associated with the game structure.
 *----------------------------------------------------------------------
 */

deletegame(g)
struct game *g;
{

if (g == ghead) {		/* deleting first game in list */
	ghead = g->next;	/* move head pointer to next game */
	if (ghead == NULL)	/* we just deleted the last game */
		gtail = NULL;	/* set both ptrs to NULL */
	else
		ghead->prev = NULL;	/* first in list has no prev */
	}
else if (g == gtail) {		/* deleting last game in list */
	gtail = g->prev;	/* move tail pointer back */
	gtail->next = NULL;	/* last game has no next */
	}
else {
	g->next->prev = g->prev;	/* link back link around g */
	g->prev->next = g->next;	/* and forward link too */
	}
if (g->gameid != NULL)
	free(g->gameid);		/* free string space */
if (g->opname != NULL)
	free(g->opname);
if (g->opaddr != NULL)
	free(g->opaddr);
if (g->mycmt != NULL)
	free(g->mycmt);
if (g->mycmt2 != NULL)
	free(g->mycmt2);
if (g->opcmt != NULL)
	free(g->opcmt);
if (g->opcmt2 != NULL)
	free(g->opcmt2);
if (g->dispmsg != NULL)
	free(g->dispmsg);
free(g);			/* free the memory */
}


/*----------------------------------------------------------------------
 *	findgame -- find a game based on its game id
 *
 * This function performs a linear search through the game list
 * for a game id.  It returns a pointer to the game, or NULL if
 * the game does not exist.
 *----------------------------------------------------------------------
 */

struct game *findgame(gid)
char *gid;
{
struct game *g;

for (g = ghead; g != NULL; g = g->next)
	if (strcmp(gid,g->gameid) == 0)		/* is this it? */
		return(g);			/* return it */
return(NULL);					/* no such game */
}


/*---------------------------------------------------------------------------
 *	addppl -- allocate a people struct and link it into the list
 *
 * This function allocates a people structure and links it into the
 * people list.  The head of this list is phead.
 *
 * NOTE: the memory-zeroing feature of calloc is depended on to
 *	 initialize the allocated people struct.
 *---------------------------------------------------------------------------
 */

struct people *addppl()
{
struct people *p, *t;

for (t = NULL, p = phead; p != NULL; t = p, p = p->next); /* t = end of list */
if ( (p = (struct people *)calloc(sizeof(struct people),1)) == NULL)
	fatal("Out of memory!");
if (t == NULL)
	phead = p;
else
	t->next = p;
p->next = NULL;
return(p);
}


/*----------------------------------------------------------------------
 *	findppl -- find a people struct by address or alias
 *
 * This function performs a linear search through the people list
 * searching for an address (if flag & P_ADDR) or an alias (if flag & P_ALIAS).
 * It returns a pointer to the struct, or NULL if the address does not exist.
 *----------------------------------------------------------------------
 */

struct people *findppl(a,flag)
char *a;
int flag;
{
int i;
struct people *p;

if (a == NULL)
	fatal("NULL address in findppl");
rescan:
for (p = phead; p != NULL; p = p->next) {
	if ( (flag & P_ADDR) && (strcmp(a,p->addr) == 0) ) {   /* check addr */
		if (p->equiv != NULL) {		/* if equiv record, */
			a = p->equiv;		/* go look for base record */
			goto rescan;
			}
		return(p);			/* return it */
		}
	if (p->equiv != NULL)		/* equiv records don't have aliases */
		continue;
	if ( (flag & P_ALIAS) && (strcmp(a,p->alias) == 0) ) /* check alias */
		return(p);
	}
return(NULL);					/* no such record */
}


/*----------------------------------------------------------------------
 *	newppl -- create a new people struct for a game
 *
 * This function creates a new people record for a new opponent.  It takes
 * a game structure, extracts the necessary information, and inserts
 * the new record into the people list, as well as storing a pointer
 * to the new people struct into the game's ppl pointer.
 *
 * The alias field is initialized to the first word of g->opname
 * with all upper case converted to lower.
 *
 * To handle people with more than one e-mail address, the people list
 * is scanned for identical name fields.  If one is found, the
 * user is asked if these people are the same.  If he answers yes,
 * the "equiv" field is set to the name of the existing people
 * record, and the rest of the record is zeroed out.
 *----------------------------------------------------------------------
 */

struct people *newppl(g)
struct game *g;
{
char *a;
register struct people *p, *q;
char buf[80];

if (g->opname == NULL) {	/* return a struct, but don't put in list */
	if ( (p = (struct people *) calloc(sizeof(struct people),1)) == NULL)
		fatal("Out of memory!");
	p->addr = g->opaddr;	/* this will keep everyone happy until */
	p->name = "UNKNOWN";	/* we find out what his name is */
	p->alias = "NONE";
	g->ppl = p;
	return(p);
	}
p = addppl();				/* create new people struct */
p->addr = save(g->opaddr);		/* copy opponent's address */
for (q = phead; q != NULL; q = q->next) {
	if ( (q->name != NULL) && (strcmp(q->name,g->opname) == 0) ) {
		printf("The following e-mail addresses:\n\n\t%s\n\t%s\n\n",
			q->addr,g->opaddr);
		printf("have the same name. [%s]\n",g->opname);
		printf("Do they refer to the same person? [default=yes]  ");
		fgets(buf,sizeof(buf),stdin);
		if ( (*buf == 'n') || (*buf == 'N') )
			break;
		p->equiv = save(q->addr);
		g->ppl = q;
		return(q);
		}
	}
p->equiv = NULL;
p->name = save(g->opname);		/* copy opponent's name */
if ( (a = strchr(g->opname,' ')) != NULL)	/* create default alias */
	*a = '\0';
p->alias = save(g->opname);		/* first word of opponent's name */
if (a != NULL)
	*a = ' ';
for (a = p->alias; *a; a++)		/* converted to lower case */
	if (isupper(*a))
		*a = tolower(*a);
p->myaddr = save(g->myaddr);		/* copy out my address */
p->fence = 0L;				/* no fence time yet */
g->ppl = p;				/* side pointer from game struct */
return(p);
}


/*----------------------------------------------------------------------
 *	printscore -- print the cumulative score for each opponent
 *
 * This function scans the people file printing the contents of the
 * score field for each opponent.  It also prints the total over
 * all opponents at the end.  For the following:
 *	points
 *	games
 *	matches
 *	gammons
 *	backgammons
 * the number won/lost/net is printed.  Games in progress are not counted,
 * but games that have been completed are, even if they are part of
 * a match that has not completed.
 *----------------------------------------------------------------------
 */

printscore()
{
register struct people *p;
register int i;
int total[10];			/* to store the total for all opponent's */

if (phead == NULL)		/* nothing to print */
	return;
printf("opponent                   points     games      gammons  backgammons  matches\n");
printf("------------------------------------------------------------------------------\n");
for (i = 0; i < 10; i++)
	total[i] = 0;
for (p = phead; p != NULL; p = p->next) {
	if (p->equiv != NULL)
		continue;		/* skip equiv records */
	pscore(p->name,p->score);	/* print this opponent */
	for (i = 0; i < 10; i++)	/* keep running total */
		total[i] += p->score[i];
	}
pscore("total",total);			/* print the total */
}


/*----------------------------------------------------------------------
 *	pscore -- print the score for one opponent
 *
 * This function is called by printscore to print each opponent.
 * The opponent name is in "name", and the score table is in "score".
 *----------------------------------------------------------------------
 */

pscore(name,score)
char *name;
int score[10];
{
static int sc_idx[] = { 2, 0, 4, 6, 8 };	/* order to print p->score[] */
register int i, diff;
char buf[30];
char c;

if (strlen(name) > 25) {	/* truncate name to 25 chars */
	c = name[25];
	name[25] = '\0';
	}
else
	c = '\0';
printf("%-25s",name);
if (c != '\0')
	name[25] = c;		/* restore name */
for (i = 0; i < 5; i++) {
	sprintf(buf,"%d/%d",score[sc_idx[i]],score[sc_idx[i]+1]);
	buf[9] = '\0';		/* truncate to 9 chars */
	if (strcmp(buf,"0/0") == 0)	/* none won or lost */
		*buf = '\0';		/* just leave it blank */
	printf("  %-9s",buf);	/* print with field width of 9 */
	}
printf("\n%21s","");
for (i = 0; i < 5; i++) {
	diff = score[sc_idx[i]] - score[sc_idx[i]+1];
	if (diff == 0) {
		if ( (score[sc_idx[i]] == 0) && (score[sc_idx[i]+1] == 0) )
			printf("           ");
		else
			printf("      even ");
		}
	else
		printf("      %+-5d", diff);
	}
printf("\n\n");
}


ilose(g,term,rsflag)
struct game *g;
int term;			/* T_* */
int rsflag;			/* 1 = restart game if necessary */
{
int bg, gv;

g->state = ST_GAMEOVER;
g->term = term;
bg = gvalue(g,&gv);
g->mcurrent[WHO_OPP] += gv;		/* bump match count */
g->ppl->score[SC_GAMESLOST]++;		/* inc games lost */
g->ppl->score[SC_PTSLOST] += gv;
if (bg == 1)			/* gammon */
	g->ppl->score[SC_GMNLOST]++;
else if (bg == 2)
	g->ppl->score[SC_BGLOST]++;
endgame(g,rsflag);
}


iwin(g,term,rsflag)
struct game *g;
int term;			/* T_* */
int rsflag;			/* 1 = restart game if necessary */
{
int bg, gv;

g->state = ST_GAMEOVER;
g->term = term;
bg = gvalue(g,&gv);
g->mcurrent[WHO_ME] += gv;		/* bump match count */
g->ppl->score[SC_GAMESWON]++;		/* inc games lost */
g->ppl->score[SC_PTSWON] += gv;
if (bg == 1)			/* gammon */
	g->ppl->score[SC_GMNWON]++;
else if (bg == 2)
	g->ppl->score[SC_BGWON]++;
endgame(g,rsflag);
}


endgame(g,rsflag)
struct game *g;
int rsflag;				/* 1 = restart game if necessary */
{

if (g->ppl->fence < g->starttime)   /* if newer than fence */
	g->ppl->fence = g->starttime;	/* move up fence */
if (rsflag && (g->flags & F_PERM) && (g->mcurrent[WHO_OPP] >= g->mtotal) ) {
	message("Restarted game with %s (%s).\n",g->opname,g->opaddr);
	notify = NULL;
	startgame(g->opaddr,g->mydir,g->mycolor,g->opcolor,
	   g->flags & (F_JACOBY|F_CRAWFORD|F_PERM|F_EUROPE|F_INVERT),
	   g->mtotal,0);
	}
if ( (g->notify != NULL) && (g->mcurrent[WHO_OPP] >= g->mtotal) )
	sendpkt(g,NOTIFY);
}
