/*	save.c		8/4/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"

/*===========================================================================
 * This file contains miscellaneous functions that save and load things.
 *===========================================================================
 */

/*---------------------------------------------------------------------------
 *	save -- make a copy of a string
 *
 * This function makes a copy of a string in malloc memory, and returns
 * a pointer to the copy.
 *---------------------------------------------------------------------------
 */

char *save(s)
char *s;
{
char *n;

if (s == NULL) {
	return(NULL);
	}
else {
	if ( (n = calloc(strlen(s)+1,1)) == NULL)
		fatal("Out of memory!");
	strcpy(n,s);
	return(n);
	}
}


/*---------------------------------------------------------------------------
 *	readldbrc -- read in the .ldbrc file
 *
 * This function reads the .ldbrc file, which contains the setup info
 * for this user.  If the HOME environment variable is set, we chdir to it.
 * If the LDBRC environment variable is set, it is used as the file to
 * read instead of .ldbrc.
 *---------------------------------------------------------------------------
 */

readldbrc()
{
FILE *fp;
char *s, *n, *getenv();
char buf[80];

if ( (s = getenv("HOME")) != NULL)	/* if we have a home */
	chdir(s);			/* go there */
if ( (s = getenv("LDBRC")) == NULL)	/* LDBRC not set */
#ifdef VMS
	s = "ldb.rc";			/* use default file name */
#else
	s = ".ldbrc";			/* use default file name */
#endif
if ( (fp = fopen(s,"r")) == NULL) {
	printf("'%s' does not exist.  Do you want to create it?",s);
	if ( (fgets(buf,sizeof(buf),stdin) == NULL)
	     || ( (*buf != 'y') && (*buf != 'Y') ) )
		fatal("ldb aborted.");
	if ( (fp = fopen(s,"w")) == NULL) {
		printf("Sorry, could not create %s.\n",s);
		ldbexit(STAT_ABORT);
		}
	printf("Please enter your personal name: ");
	if (fgets(buf,sizeof(buf),stdin) == NULL) {
		fclose(fp);
		unlink(s);
		printf("ldb aborted.\n");
		ldbexit(STAT_ABORT);
		}
	buf[strlen(buf)-1] = '\0';	/* clobber the newline char */
	fprintf(fp,"myname=%s\n",buf);
	printf("Please enter your e-mail address: ");
	if (fgets(buf,sizeof(buf),stdin) == NULL) {
		fclose(fp);
		unlink(s);
		printf("ldb aborted.\n");
		ldbexit(STAT_ABORT);
		}
	buf[strlen(buf)-1] = '\0';	/* clobber the newline char */
	fprintf(fp,"myaddr=%s\n",buf);
	fprintf(fp,"mailfile=ldb.rcv\n");
#ifdef VMS
	fprintf(fp,"sendcmd=mail/subject=\"$s\" $f IN%%\"\"\"$a\"\"\"\n");
	fprintf(fp,"gamefile=ldb.data\n");
	fprintf(fp,"backupfile=ldb.olddata\n");
	fprintf(fp,"peoplefile=ldb.people\n");
	fprintf(fp,"lockfile=ldb.lock\n");
	fprintf(fp,"supercmd=\n");
#else
	fprintf(fp,"sendcmd=mail -s '$s' $a < $f\n");
	fprintf(fp,"gamefile=.ldbdata\n");
	fprintf(fp,"backupfile=.oldldbdata\n");
	fprintf(fp,"peoplefile=.ldbpeople\n");
	fprintf(fp,"lockfile=.ldb_lock\n");
	fprintf(fp,"supercmd=sh\n");
#endif
	fprintf(fp,"delmail=no\n");
	fprintf(fp,"tempfile=ldb.tmp\n");
	fprintf(fp,"colors=rw\n");
	fprintf(fp,"direction=up\n");
	fprintf(fp,"initialboard=current\n");
	fprintf(fp,"autoroll=yes\n");
	fprintf(fp,"automove=no\n");
	fprintf(fp,"autodouble=0\n");
	fprintf(fp,"superkey=0\n");
	fprintf(fp,"checkpoint=yes\n");
	fprintf(fp,"timeout=7\n");
	fprintf(fp,"keepold=7\n");
	fprintf(fp,"debug=0\n");
	fclose(fp);
	printf("\nYour %s file was created.  You may want to read the\n",s);
	printf("manual for additional options available in this file.\n\n");
#ifdef VMS
	printf("VMS users should edit the sendcmd command in %s to use\n",s);
	printf("the appropriate mailer.  The IN%% mailer is assumed.\n\n");
#endif
	if ( (fp = fopen(s,"r")) == NULL) {
		printf("I can't re-open your %s file!\n",s);
		ldbexit(STAT_ABORT);
		}
	}
rc.myname = NULL;		/* these fields are required */
rc.myaddr = NULL;
#ifdef VMS
rc.gfile = "ldb.data";	/* default game storage file */
rc.pfile = "ldb.people";	/* default opponent file */
rc.gbackup = "ldb.olddata";	/* game backup file */
rc.sendcmd = "mail/subject=\"$s\" $f IN%%\"\"\"$a\"\"\""; /* dflt mail cmd */
rc.lockfile = "ldb.lock";	/* default lock file */
rc.supercmd = "";		/* command to run when we have company */
#else
rc.gfile = ".ldbdata";	/* default game storage file */
rc.pfile = ".ldbpeople";	/* default opponent file */
rc.gbackup = ".oldldbdata";	/* game backup file */
rc.sendcmd = "mail -s '$s' $a < $f";	/* default mail command */
rc.lockfile = ".ldb_lock";	/* default lock file */
rc.supercmd = "sh";		/* command to run when we have company */
#endif
rc.mfile = "ldb.rcv";	/* default file for received mail */
rc.delmail = "no";			/* don't delete mail by default */
rc.tempfile = "ldb.tmp";		/* default temp file */
rc.defclrs = "rw";			/* "default" default colors */
rc.defdir = "u";			/* and direction */
rc.initboard = "c";			/* show current board by default */
rc.autoroll = "y";			/* enable autoroll by default */
rc.automove = "n";		/* disabled by default (it's really annoying */
rc.autodouble = 0;		/* disable autodouble by default */
rc.superkey = 0;		/* key to activate supercmd (dflt=disabled) */
rc.chkpt = "y";			/* checkpoint is enabled by default */
rc.acctime = 7;			/* access timeout in 7 days by default */
rc.keepold = 7;			/* keep dead games 7 days by default */
rc.debug = 0;			/* default to no debug */
if ( (n = nvscan(fp,nv_rcfile,&rc)) != NULL) {
	fprintf(stderr,"Invalid line in .ldbrc: %s\n",n);
	ldbexit(STAT_ABORT);
	}
fclose(fp);
if (rc.myname == NULL) {
	fprintf(stderr,"ERROR: missing 'myname' line in %s\n",s);
	ldbexit(STAT_ABORT);
	}
if (rc.myaddr == NULL) {
	fprintf(stderr,"ERROR: missing 'myaddr' line in %s\n",s);
	ldbexit(STAT_ABORT);
	}
if ( (strlen(rc.defclrs) != 2) || (! isalpha(rc.defclrs[0])) ||
     (! isalpha(rc.defclrs[1])) || (rc.defclrs[0] == rc.defclrs[1]) ) {
	fprintf(stderr,"ERROR: invalid color string in %s: %s",s,rc.defclrs);
	ldbexit(STAT_ABORT);
	}
if (strchr("ud",*rc.defdir) == NULL) {
	fprintf(stderr,"ERROR: direction must be 'up' or 'down' in %s\n",s);
	ldbexit(STAT_ABORT);
	}
if (strchr("bac",*rc.initboard) == NULL) {
	fprintf(stderr,
	 "ERROR: initialboard must be 'before', 'after', or 'current' in %s\n"
	 ,s);
	ldbexit(STAT_ABORT);
	}
if (strchr("yn",*rc.autoroll) == NULL) {
	fprintf(stderr,"ERROR: autoroll must be 'yes' or 'no' in %s\n",s);
	ldbexit(STAT_ABORT);
	}
if (strchr("yn",*rc.automove) == NULL) {
	fprintf(stderr,"ERROR: automove must be 'yes' or 'no' in %s\n",s);
	ldbexit(STAT_ABORT);
	}
if (strchr("yn",*rc.chkpt) == NULL) {
	fprintf(stderr,"ERROR: checkpoint must be 'yes' or 'no' in %s\n",s);
	ldbexit(STAT_ABORT);
	}
}


/*---------------------------------------------------------------------------
 *	readgames -- read in games in progress
 *
 * This function reads the games file specified in .ldbrc and loads
 * the games into the games list (ghead/gtail).
 *---------------------------------------------------------------------------
 */

readgames()
{
FILE *fp;
char c, *s;
struct game *g;
long old;

readpeople();
if ( (fp = fopen(rc.gfile,"r")) == NULL)
	return;				/* no games */
old = time( (long *) 0);
if (rc.keepold > 0)		/* if keeping old games is enabled */
	old -= (rc.keepold * 86400);	/* how old is too old? */
while ( (c = getc(fp)) != EOF) {
	ungetc(c,fp);	/* put char back */
	g = addgame();		/* insert a new game */
	g->opaddr = NULL;
	g->opname = NULL;
	g->myaddr = NULL;
	g->mycmt = NULL;
	g->mycmt2 = NULL;
	g->opcmt = NULL;
	g->opcmt2 = NULL;
	g->dispmsg = NULL;
	g->hiused = 0;
	g->maxused = 0;
	g->opver = 100;		/* default to version 1.0 */
	g->starttime = 0L;
	clearmvs(g->opmvs);
	clearmvs(g->mvs);
	if ( (s = nvscan(fp,nv_gfile,g)) != NULL) {	/* read game */
		FeFinishSession();	/* close down front-end */
		TFinishSession();	/* close down transport */
		fprintf(stderr,"ERROR: invalid line in %s: %s\n", rc.gfile, s);
		ldbexit(STAT_ABORT);
		}
	if (g->gameid == NULL) {		/* empty game (prob. EOF) */
		deletegame(g);
		continue;
		}
	if (g->myaddr == NULL)		/* no myaddr line */
		g->myaddr = save(rc.myaddr);	/* dflt to ldbrc setting */
	if ( (g->ppl = findppl(g->opaddr,P_ADDR)) == NULL)
		newppl(g);
	if (
	   (g->state == ST_GAMEOVER)	&&
	   (g->flags & F_DISPLAYED)	&&
	   (g->lastacc < old) )
		g->flags |= F_DELETE;	/* game is old, mark it for deletion */
	if (rc.debug & DB_RWGAMES)
		message("DB-readgames: read game %s\n",g->gameid);
	}
fclose(fp);
}


/*----------------------------------------------------------------------
 *	writegames -- save the game list to a file
 *
 * This function writes each game in the game list to the specified
 * file.  Games with F_DELETE set in their flags are skipped, and
 * are thus effectively deleted.  If the bkup arg is not NULL and
 * is not an empty string, the old file is renamed to the string
 * specified in bkup before the new file is created.
 *----------------------------------------------------------------------
 */

writegames(file,bkup,pfile)
char *file, *bkup, *pfile;
{
FILE *fp;
struct game *g;

writepeople(pfile);			/* save the people file */
if ( (bkup != NULL) && (*bkup != '\0') ) {
	unlink(bkup);		/* prevent multiple versions on VMS */
	rename(file,bkup);	/* save old game file */
	}
else
	unlink(file);		/* prevent multiple versions on VMS */
if ( (fp = fopen(file,"w")) == NULL) {
	message("ERROR: can't save games in %s!\n",file);
	return;			/* should do something to save games... */
	}
for (g = ghead; g != NULL; g = g->next) {
	if (g->flags & F_DELETE)
		continue;			/* game is over, delete it */
	if (g->state == ST_GAMEOVER)
		continue;			/* skip complete games 4 now */
	nvwrite(fp,nv_gfile,g);	/* write the struct */
	if (rc.debug & DB_RWGAMES)
		message("DB-writegames: write game %s\n",g->gameid);
	}
for (g = ghead; g != NULL; g = g->next) {
	if (g->flags & F_DELETE)
		continue;		/* game is over, delete it */
	if (g->state != ST_GAMEOVER)
		continue;		/* put complete games @ end */
	nvwrite(fp,nv_gfile,g);	/* write the struct */
	if (rc.debug & DB_RWGAMES)
		message("DB-writegames: write game %s\n",g->gameid);
	}
fclose(fp);
}


/*---------------------------------------------------------------------------
 *	readpeople -- read in the people file
 *
 * This function reads the "people" file, which is the list of all of the
 * opponents we have ever played.  The people file serves three purposes:
 *	1. It keeps a record of games/gammons/backgammons/matches won/lost
 *	   to each person.
 *	2. It stores the mail address so you don't have to remember,
 *	   and allows you to assign an alias to each person.  This
 *	   alias can be used in the -start argument instead of the
 *	   mail address.
 *	3. It stores the "fence" time, which is the time that the
 *	   (newest game played with that person that has finished)
 *	   was started.  (Read it again, it makes sense).
 *	   This information allows us to discard START packets that
 *	   have remained in our mail (that we should have deleted,
 *	   but didn't) by discarding START packets that are not newer
 *	   than the fence time.
 *---------------------------------------------------------------------------
 */

readpeople()
{
FILE *fp;
char c, *s;
struct people *p;

if ( (fp = fopen(rc.pfile,"r")) == NULL)
	return;				/* no people file */
while ( (c = getc(fp)) != EOF) {
	ungetc(c,fp);	/* put char back */
	p = addppl();		/* insert a new record */
	p->name = NULL;		
	p->addr = NULL;
	p->alias = NULL;
	p->myaddr = NULL;
	p->equiv = NULL;
	p->fence = 0L;
	if ( (s = nvscan(fp,nv_pfile,p)) != NULL) {	/* read record */
		FeFinishSession();	/* close down front-end */
		TFinishSession();	/* close down transport */
		fprintf(stderr,"ERROR: invalid line in %s: %s\n", rc.pfile, s);
		ldbexit(STAT_ABORT);
		}
	if (p->equiv != NULL)		/* an equiv record */
		continue;		/* no other fields */
	if (p->myaddr == NULL)			/* if no myaddr line */
		p->myaddr = save(rc.myaddr);	/* dflt to ldbrc setting */
	if (rc.debug & DB_RWGAMES)
		message("DB-readpeople: read '%s'\n",p->name);
	}
fclose(fp);
}


/*----------------------------------------------------------------------
 *	writepeople -- save the people list to a file
 *
 * This function writes the people list to rc.pfile.
 *----------------------------------------------------------------------
 */

writepeople(file)
char *file;
{
FILE *fp;
struct people *p;

unlink(file);		/* prevent multiple versions on VMS */
if ( (fp = fopen(file,"w")) == NULL) {
	message("ERROR: can't save people in %s!\n",file);
	return;			/* should do something to save people list */
	}
for (p = phead; p != NULL; p = p->next) {
	if (p->equiv == NULL) {
		nvwrite(fp,nv_pfile,p);	/* write the struct */
		if (rc.debug & DB_RWGAMES)
			message("DB-writepeople: write '%s'\n",p->name);
		}
	else
		nvwrite(fp,nv_pequiv,p);	/* write an equiv record */
	}
fclose(fp);
}



/*----------------------------------------------------------------------
 *	boardstr -- generate an ascii representation of a board
 *
 * This function produces a visible representation of a board.  Each point
 * on the board takes two characters; the quantity is offset by 65,
 * putting it in the range [A-P], and the color is unchanged since it
 * is already a printable character.  This results in a string of
 * 28 character pairs, one for each point on the board.  These are
 * in the order:
 *	0:	BAR point for upbound player
 *	1-24:	board points
 *	25:	BAR point for downbound player
 *	26:	OFF point for upbound player
 *	27:	OFF point for downbound player
 *----------------------------------------------------------------------
 */

char *boardstr(b)
board b;
{
static char buf[BOARDSIZE*2+1];
char *s, c;
int i;

s = buf;
for (i = 0; i < BOARDSIZE; i++) {
	*s++ = b[i].qty + 'A';		/* offset qty into u.c. letters */
	c = b[i].color;
	if (! isalpha(c))
		c = '-';		/* use printing chars */
	*s++ = c;
	}
*s = '\0';
return(buf);
}



/*----------------------------------------------------------------------
 *	nvscan -- read name/value pairs from a file
 *
 * This function provides a generalized method for reading name/value
 * pairs.  The names and value types are specified in an array of
 * struct namevalue's, along with an offset which is used to store
 * the value.  The offset is added to a base pointer, passed as the
 * "st" argument, to form a pointer, which is then converted to the
 * type indicated by the "type" field of the namevalue table using
 * the "nvtypes" union, and used to store the value.  The legal
 * value types are defined in ldb.h as FT_*.  Name/value pairs are
 * expected to be in the form "name=value\n", with no spaces before
 * or after name, and with any spaces after the = or before the
 * newline being part of the value string.  Comments are indicated by
 * a # in column 1, which comments to the end of the line.
 *----------------------------------------------------------------------
 */

char *nvscan(fp,t,st)
FILE *fp;		/* file to scan */
struct namevalue *t;	/* table of name/value pairs */
char *st;		/* really a pointer to a structure */
{
static char buf[128];
char *s, **p;
int i, j;
union nvtypes u;
long atol();

while (fgets(buf,sizeof(buf),fp) != NULL) {
	if (*buf == '#')			/* comment character */
		continue;
	buf[strlen(buf)-1] = '\0';		/* clobber the newline */
	if ( (s = strchr(buf,'=')) == NULL)
		return(buf);			/* bad line, return it */
	*s++ = '\0';
	for (i = 0; t[i].name != NULL; i++)
		if (strcmp(t[i].name,buf) == 0)
			break;
	if (t[i].name == NULL)		/* got a name we don't recognize */
		continue;		/* ignore it */
	u.nvchar = st + t[i].offset;	/* put pointer into union */
	switch (t[i].type) {
	case FT_CHAR:			/* just store a single char */
		*u.nvchar = atoi(s);	/* chars stored as ints in the file */
		break;
	case FT_INT:			/* store an int */
		*u.nvint = atoi(s);
		break;
	case FT_STRING:			/* store a copy of a string */
		*u.nvstring = save(s);
		break;
	case FT_MOVE:			/* store a struct mv */
		str2mv(s,u.nvmove);
		break;
	case FT_BOARD:			/* store an entire board */
		for (j = 0; j < BOARDSIZE; j++) {
			u.nvboard[j].qty = *s++ - 'A';
			u.nvboard[j].color = *s++;
			}
		break;
	case FT_STRLKUP:		/* look up string & store index */
		p = (char **) t[i].dflt;	/* unions are such a pain */
		if (p == NULL)		/* choke... */
			fatal("ERROR: NULL string table in nvscan.");
		for (j = 0; *p; j++, p++)
			if (strcmp(s,*p) == 0)
				break;
		if (*p == NULL) {
			FeFinishSession();	/* close down front-end */
			TFinishSession();	/* close down transport */
			fprintf(stderr,"ERROR: unknown string: %s\n",s);
			ldbexit(STAT_ABORT);	/* shouldn't do this... */
			}
		*u.nvint = j;	/* store integer opcode */
		break;
	case FT_TIME:			/* read in a timestamp */
		*u.nvtime = atol(s);
		break;
	case FT_INTARRAY:		/* array of integers */
		for (j = 0; j < t[i].dflt; j++) {
			while (isspace(*s))
				s++;
			(u.nvint)[j] = atoi(s);		/* store an int */
			while (isdigit(*s))	/* skip to end of int */
				s++;
			}
		break;
	case FT_END:			/* we hit the end marker */
		return(NULL);		/* return success */
	default:			/* we have a bad nv table */
		*--s = '=';		/* restore the buffer */
		return(buf);		/* return bad line */
		}
	}
return(NULL);
}



/*----------------------------------------------------------------------
 *	nvwrite -- write name/value pairs into a file
 *
 * This function writes name/value pairs to a file in the same format
 * used by nvscan.  Nvwrite is merely the inverse of nvscan, taking values
 * out of the structure in the same manner nvscan used to store them
 * there, and generating "name=value" lines.  One line is generated for
 * each element in the namevalue table, except that elements of type
 * FT_MOVE whose "roll" field is <= 0 are skipped, as are elements of
 * type FT_STRING that are equal to NULL.
 *----------------------------------------------------------------------
 */

nvwrite(fp,t,st)
FILE *fp;			/* file to write to */
struct namevalue *t;		/* table of name/value pairs */
char *st;			/* really a pointer to a structure */
{
struct namevalue *n;
static char buf[128];
int nstr;
union nvtypes u;
char c, **stbl;
int j;

for (n = t; n->name != NULL; n++) {
	u.nvchar = st + n->offset;
	switch (n->type) {
	case FT_CHAR:			/* just store a single char */
		fprintf(fp,"%s=%d\n",n->name,*u.nvchar);
		break;
	case FT_INT:			/* store an int */
		fprintf(fp,"%s=%d\n",n->name,*u.nvint);
		break;
	case FT_STRING:			/* store a copy of a string */
		if (*u.nvstring != NULL)/* skip NULL strings */
			fprintf(fp,"%s=%s\n",n->name,*u.nvstring);
		break;
	case FT_MOVE:			/* store a struct mv */
		if (u.nvmove->roll > 0) {
			mv2str(u.nvmove,buf);
			fprintf(fp,"%s=%s\n",n->name,buf);
			}
		break;
	case FT_BOARD:			/* store an entire board */
		fprintf(fp,"%s=%s\n",n->name,boardstr(u.nvboard));
		break;
	case FT_STRLKUP:		/* look up string & store index */
		stbl = (char **) n->dflt;	/* hope char** fits in int */
		if (stbl == NULL)
			fatal("ERROR: NULL string table in nvwrite.");
		for (nstr = 0; stbl[nstr]; nstr++); /* # strings */
		if ( (*u.nvint < 0) || (*u.nvint >= nstr) ) {
			FeFinishSession();	/* close down front-end */
			TFinishSession();	/* close down transport */
			fprintf(stderr,"ERROR: invalid index: %s=%d\n",
				n->name,*u.nvint);
			ldbexit(STAT_ABORT);	/* shouldn't do this... */
			}
		fprintf(fp,"%s=%s\n",n->name,stbl[*u.nvint]);
		break;
	case FT_TIME:			/* generate a timestamp */
		fprintf(fp,"%s=%lu\n",n->name,*u.nvtime);
		break;
	case FT_INTARRAY:		/* generate an integer array */
		fprintf(fp,"%s",n->name);
		c = '=';
		for (j = 0; j < n->dflt; j++) {
			fprintf(fp,"%c%d",c,(u.nvint)[j]);
			c = ' ';
			}
		fprintf(fp,"\n");
		break;
	case FT_END:			/* we hit the end marker */
		fprintf(fp,"%s=end\n",n->name);
		break;
	default:			/* we have a bad nv table */
		FeFinishSession();	/* close down front-end */
		TFinishSession();	/* close down transport */
		fprintf(stderr,"ERROR: bad namevalue type: %s (%d)\n",
			n->name,n->type);
		ldbexit(STAT_ABORT);		/* should have saved games? */
		}
	}
}


/*---------------------------------------------------------------------------
 *	check_timeout -- see if a game has been accessed recently
 *
 * This function checks the access time of a game.  If it is less than
 * rc.acctime (default 7 days), and the game is waiting for the opponent,
 * an automatic timeout is performed.  If rc.acctime is <= 0,
 * automatic timeouts are disabled.
 *---------------------------------------------------------------------------
 */

check_timeout(g)
struct game *g;
{
long old;

if (rc.acctime <= 0)
	return;
old = time((long *) 0) - ((long) (rc.acctime*86400)); /* look for idle games */
if ( (g->lastacc < old) && (g->state < OPSTATES) ) {
	message("Access timeout (%d days) with %s -- resending...\n",
		rc.acctime,g->opaddr);
	g->lastacc = time( (long *) 0);
	resendpkt(g);	/* auto resend */
	}
}
