/*	rcvop.c		8/6/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"

/*===========================================================================
 * This file contains the functions which make up the receive state
 * machine.  These are called through the func[][] matrix, which takes
 * the current state and a received operation and calls the appropriate
 * handler function.  These functions will typically change the game
 * state into one requiring user input at this host (ST_MY*), and return.
 *===========================================================================
 */


/*---------------------------------------------------------------------------
 *	start -- initiate a game
 *
 * This function is called when a remote user starts a game with us.
 * We store his personal information (opaddr & opname), roll 1 die,
 * and compare it to the one he sent.  If we won the roll, the roll
 * is stored in mvs[] and state is set to MYMOVE.  If we lost the roll,
 * both dice are sent back in a USTART line.  If the roll was a tie,
 * a TIE packet is sent back .  The originator will re-roll and send
 * us a RESTART packet, which will repeat the opening roll.
 *---------------------------------------------------------------------------
 */

start(g)
struct game *g;
{
int mydie;
char c1, c2;

g->opaddr = P.addr;	/* save mail address of opponent */
g->gameid = P.gameid;		/* copy game id */
g->opver = P.version;		/* save opponent's ldb version */
g->mycolor = P.colors[1];	/* copy out colors */
g->opcolor = P.colors[0];
if (isupper(*P.dir))
	*P.dir = tolower(*P.dir);
g->mydir = (*P.dir == 'u') ? 1 : -1;	/* copy out directions */
g->opdir = REV(g->mydir);
g->gameval = 1;			/* no doubles yet */
g->flags = 0;
g->seq = 2;			/* we rcvd 1 pkt already, init to 2 */
if ( (g->ppl = findppl(g->opaddr,P_ADDR)) == NULL) { /* know this guy? */
	g->myaddr = save(rc.myaddr);	/* nope, create a new ppl record */
	newppl(g);
	}
else
	g->myaddr = save(g->ppl->myaddr);
g->starttime = P.timestamp;	/* store timestamp from start packet */
g->lastacc = P.timestamp;	/* set last access time to start time */
if (P.autodbl == NULL)		/* set admax to MIN(my count, op's count) */
	g->admax = 0;
else
	g->admax = atoi(P.autodbl);
if (rc.autodouble < g->admax)
	g->admax = rc.autodouble;
g->adcnt = 0;			/* no autodoubles yet */
clearmvs(g->mvs);
clearmvs(g->opmvs);
if (g->mydir > 0) {
	c1 = g->mycolor;	/* upbound color is mine */
	c2 = g->opcolor;	/* downbound color is opponent's */
	}
else {
	c1 = g->opcolor;	/* upbound color is opponent's */
	c2 = g->mycolor;	/* downbound color is mine */
	}
newboard(g->opbd,c1,c2);		/* set up boards for new game */
newboard(g->mybd,c1,c2);
newboard(g->board,c1,c2);
if (P.jacoby != NULL)
	g->flags |= F_JACOBY;
if (P.crawford != NULL)
	g->flags |= F_CRAWFORD;
if (P.european != NULL)
	g->flags |= F_EUROPE;
if (P.perm != NULL)
	g->flags |= F_PERM;
if (P.match != NULL)
	g->mtotal = atoi(P.match);
g->curbd = boardnums[*rc.initboard - 'a'];	/* display initial board */
mydie = Rolldie();
if (P.mvs[0].roll == mydie) {		/* a !#$%&@ tie */
	if (g->adcnt < g->admax)	/* do an autodouble */
		g->gameval = 1 << ++(g->adcnt);
	sendpkt(g,TIE);
	message("Tie on initial roll with %s (%s).\n",g->opname,g->opaddr);
	return;			/* opponent will send RESTART */
	}
if (mydie > (int) P.mvs[0].roll) {		/* we won the initial roll */
	g->mvs[0].roll = P.mvs[0].roll;	/* copy initial roll */
	g->mvs[1].roll = mydie;
	g->mvs[0].pt = -1;	/* mark both rolls unused */
	g->mvs[1].pt = -1;
	g->state = ST_MYMOVE;	/* set state so we make a move */
	legalmoves(g);		/* calculate legal moves for these rolls */
	g->rolls[g->mvs[0].roll - 1]++;	/* count the rolls we got */
	g->rolls[g->mvs[1].roll - 1]++;
	}
else {				/* we lost, tell the opponent to start */
	g->mvs[0].roll = P.mvs[0].roll;	/* copy initial roll */
	g->mvs[1].roll = mydie;	/* store so sendpkt can find it */
	g->state = ST_OPTURN;
	sendpkt(g,USTART);
	message("Started game with %s (%s).\n",g->opname,g->opaddr);
	}
}


/*---------------------------------------------------------------------------
 *	istart -- I won the opening toss
 *
 * This function is called when a USTART packet is received.  Both rolls
 * are copied into the game structure and the state is set to MYMOVE,
 * allowing us to use the roll but not to double.
 *---------------------------------------------------------------------------
 */

istart(g)
struct game *g;
{

g->mvs[0].roll = P.mvs[0].roll;	/* copy rolls from packet */
g->mvs[1].roll = P.mvs[1].roll;
g->mvs[0].pt = -1;	/* mark both rolls unused */
g->mvs[1].pt = -1;
g->state = ST_MYMOVE;	/* set state so we make a move */
legalmoves(g);		/* calculate legal moves */
g->rolls[g->mvs[0].roll - 1]++;	/* count the rolls we got */
g->rolls[g->mvs[1].roll - 1]++;
}


/*---------------------------------------------------------------------------
 *	tie -- The opening toss was a tie, try again
 *
 * This function is called when we receive a TIE packet.  We reroll
 * one die and send a RESTART packet.  If the autodbl field in
 * the received packet is > 0, the game value is set to 2 ** autodbl.
 *---------------------------------------------------------------------------
 */

tie(g)
struct game *g;
{

clearmvs(g->mvs);
g->mvs[0].roll = Rolldie();
if (P.autodbl != NULL)
	g->gameval = 1 << (g->adcnt = atoi(P.autodbl));
sendpkt(g,RESTART);
message("Tie on initial roll with %s (%s).\n",g->opname,g->opaddr);
}


/*---------------------------------------------------------------------------
 *	restart -- restart after opening tie
 *
 * This function is called when we receive a RESTART packet.  It is
 * mostly the same as start().
 *---------------------------------------------------------------------------
 */

restart(g)
struct game *g;
{
int mydie;

clearmvs(g->mvs);
clearmvs(g->opmvs);
mydie = Rolldie();
if (P.mvs[0].roll == mydie) {		/* a !#$%&@ tie */
	if (g->adcnt < g->admax)	/* do an autodouble */
		g->gameval = 1 << ++(g->adcnt);
	sendpkt(g,TIE);
	message("Tie on initial roll with %s (%s).\n",g->opname,g->opaddr);
	return;			/* opponent will send RESTART */
	}
g->mvs[0].roll = P.mvs[0].roll;	/* copy initial roll */
g->mvs[1].roll = mydie;	/* store so sendpkt can find it */
if (mydie > (int) P.mvs[0].roll) {		/* we won the initial roll */
	g->state = ST_MYMOVE;	/* set state so we make a move */
	legalmoves(g);		/* calculate legal moves for these rolls */
	g->rolls[g->mvs[0].roll - 1]++;	/* count the rolls we got */
	g->rolls[g->mvs[1].roll - 1]++;
	}
else {				/* we lost, tell the opponent to start */
	g->state = ST_OPTURN;
	sendpkt(g,USTART);
	message("Started game with %s (%s).\n",g->opname,g->opaddr);
	}
}


/*---------------------------------------------------------------------------
 *	mstart -- start next game of match
 *
 * This function is called when we receive an MSTART packet.  It is
 * similar to restart, except that it also reinitializes the game
 * structure for the next game of the match.  All games of a match
 * use the same gameid and game structure.  Mstart also checks to
 * see if this is a Crawford rule game, and if so, sets F_CRGAME in g->flags.
 *---------------------------------------------------------------------------
 */

mstart(g)
struct game *g;
{
int mydie;
int i;
char c1, c2;

g->state = ST_OPSTART;
g->gameval = 1;		/* reset for next game */
g->adcnt = 0;
g->flags &= ~F_IDOUBLED;
g->term = 0;
if (g->mydir > 0) {
	c1 = g->mycolor;	/* upbound color is mine */
	c2 = g->opcolor;	/* downbound color is opponent's */
	}
else {
	c1 = g->opcolor;	/* upbound color is opponent's */
	c2 = g->mycolor;	/* downbound color is mine */
	}
newboard(g->opbd,c1,c2);
newboard(g->mybd,c1,c2);
newboard(g->board,c1,c2);
for (i = 0; i < 6; i++) {
	g->rolls[i] = 0;
	g->doubles[i] = 0;
	g->oprolls[i] = 0;
	g->opdoubles[i] = 0;
	}
clearmvs(g->mvs);
clearmvs(g->opmvs);
crawford_check(g);			/* is this the Crawford rule game? */
mydie = Rolldie();
if (P.mvs[0].roll == mydie) {		/* a !#$%&@ tie */
	if (g->adcnt < g->admax)	/* do an autodouble */
		g->gameval = 1 << ++(g->adcnt);
	sendpkt(g,TIE);
	message("Tie on initial roll with %s (%s).\n",g->opname,g->opaddr);
	return;			/* opponent will send RESTART */
	}
g->mvs[0].roll = P.mvs[0].roll;	/* copy initial roll */
g->mvs[1].roll = mydie;	/* store so sendpkt can find it */
if (mydie > (int) P.mvs[0].roll) {		/* we won the initial roll */
	g->state = ST_MYMOVE;	/* set state so we make a move */
	legalmoves(g);		/* calculate legal moves for these rolls */
	g->rolls[g->mvs[0].roll - 1]++;	/* count the rolls we got */
	g->rolls[g->mvs[1].roll - 1]++;
	}
else {				/* we lost, tell the opponent to start */
	g->state = ST_OPTURN;
	sendpkt(g,USTART);
	message("Started game with %s (%s).\n",g->opname,g->opaddr);
	}
}


/*---------------------------------------------------------------------------
 *	opmove -- opponent moved
 *
 * This function is called when we receive a MOVE packet.  The move is
 * copied into the opmvs field of the game structure and applied to the
 * board.  A copy of the board before the moves are applied is stored
 * in the opbd field, and a copy of the board after the moves are applied
 * is stored in the mybd field.  These two boards, along with the
 * current board in the board field, make up the three boards that can
 * be displayed with the "Board" command in the user menus in process.c
 *---------------------------------------------------------------------------
 */

opmove(g)
struct game *g;
{
int i, n, r;
static char buf[] = "Opponent move dated DDD MMM NN HH:MM:SS YYYY";
struct game *pg;

copyboard(g->board,g->opbd);	/* save board before opponent moved */
g->curbd = boardnums[*rc.initboard - 'a'];	/* display initial board */
clearmvs(g->opmvs);		/* clear old moves */
g->opmvs[0] = P.mvs[0];	/* extract opponent's moves */
g->opmvs[1] = P.mvs[1];
g->oprolls[g->opmvs[0].roll - 1]++;		/* count rolls opponent got */
g->oprolls[g->opmvs[1].roll - 1]++;
if (g->opmvs[0].roll == g->opmvs[1].roll) {
	g->opmvs[2] = P.mvs[2];	/* he got doubles */
	g->opmvs[3] = P.mvs[3];	/* extract 2 more moves */
	g->opdoubles[g->opmvs[0].roll - 1]++;
	n = 4;
	}
else
	n = 2;
for (i = 0; i < 4; i++)
	g->blot[i] = 0;		/* clear all blot locations */
for (i = 0; i < n; i++) {
	if ( (r = apply(g,WHO_OPP,i,0,NULL)) < 0) {	/* err w/ op move */
		copyboard(g->opbd,g->board);	/* restore board */
		message("ERROR: Opponent move rejected, id=%s\n",
			P.gameid);
		message("       %s\n",rejmsg[-r]);
		return;
		}
	else		/* if opponent hit our blot */
		g->blot[i] = r;		/* save location */
	}
copyboard(g->board,g->mybd);		/* save board after op move */
if (g->board[OFFPT(g->opdir)].qty == 15)	/* opponent won */
	ilose(g,T_ILOSE,1);
else
	g->state = ST_MYTURN;		/* opponent has moved, it's our turn */
clearmvs(g->mvs);		/* erase our previous move */
if (P.timestamp > 0) {		/* if we got a timestamp */
	strncpy(&buf[20],ctime(&P.timestamp),24);	/* gen message */
	g->dispmsg = save(buf);		/* save copy in game */
	}
}


/*---------------------------------------------------------------------------
 *	opofr -- opponent offered to double
 *
 * This function is called when we receive an OFRDBL packet, indicating
 * the opponent wishes to double.  The game moves to state MYACCEPT,
 * where the user will decide to accept or decline the double.
 *---------------------------------------------------------------------------
 */

opofr(g)
struct game *g;
{

if (g->flags & F_CRGAME) {
	message("ERROR: Opponent move rejected, id=%s\n",P.gameid);
	message("       Invalid double (Crawford rule)\n");
	return;
	}
g->state = ST_MYACCEPT;			/* send us to an accept screen */
g->flags &= ~F_IDOUBLED;		/* I didn't double last */
copyboard(g->board,g->opbd);
copyboard(g->board,g->mybd);
g->curbd = BD_CUR;		/* display current board */
clearmvs(g->opmvs);		/* clear old moves */
clearmvs(g->mvs);		/* erase our previous move */
}


/*---------------------------------------------------------------------------
 *	opconc -- opponent conceded
 *
 * This function is called when we receive a CONCEDE packet, indicating
 * the opponent has given up.
 *---------------------------------------------------------------------------
 */

opconc(g)
struct game *g;
{
int gv;

iwin(g,T_OPCONCEDE,1);			/* wimp */
}


/*---------------------------------------------------------------------------
 *	opacpt -- opponent accepted double
 *
 * This function is called when we receive an ACPTDBL packet, indicating
 * the opponent has accepted our double.  The IDOUBLED flag is set to
 * prevent us from doubling again until the opponent doubles.  Since it
 * is now our turn, we go ahead and roll the dice and proceed to the
 * MYMOVE state.
 *---------------------------------------------------------------------------
 */

opacpt(g)
struct game *g;
{

g->gameval *= 2;		/* double game value */
g->flags |= F_IDOUBLED;		/* I doubled last */
g->state = ST_MYMOVE;		/* It is my move */
copyboard(g->board,g->opbd);
copyboard(g->board,g->mybd);
g->curbd = BD_CUR;		/* display current board */
clearmvs(g->opmvs);		/* clear old moves */
clearmvs(g->mvs);		/* erase our previous move */
rolldice(g);			/* go ahead and roll, I can't double */
g->dispmsg = save("Opponent has accepted your double.");/* notify user */
}


/*---------------------------------------------------------------------------
 *	opdec -- opponent declined double
 *
 * This function is called when a DECDBL packet is received.  This
 * indicates that the opponent has declined our double, and the game is over.
 *---------------------------------------------------------------------------
 */

opdec(g)
struct game *g;
{
int gv;

iwin(g,T_OPDECLINE,1);
copyboard(g->board,g->opbd);
copyboard(g->board,g->mybd);
g->curbd = BD_CUR;		/* display current board */
clearmvs(g->opmvs);		/* clear old moves */
clearmvs(g->mvs);		/* erase our previous move */
}


/*---------------------------------------------------------------------------
 *	smerr -- an illegal operation was received for this state
 *
 * This function is called when a packet is received that is invalid
 * for the game state.  An example of this would be receiving an
 * ACPTDBL packet when we did not send an OFRDBL.
 *---------------------------------------------------------------------------
 */

smerr(g)
struct game *g;
{

message("ERROR: invalid operation (%d) for state (%d), id=%s\n",
	P.opcode, g->state, g->gameid);
}
