/*	misc.c		8/8/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"


/*----------------------------------------------------------------------
 *	rolldice -- roll two dice
 *
 * This function calls Rolldie twice and fills in the game structure
 * with the resulting values.  If the two calls to Rolldie return the
 * same number, the mvs field is filled in so that the user has 4 rolls
 * to use, otherwise the rolls are stored in the first two elements
 * of the mvs field and the last two are marked unused.
 *----------------------------------------------------------------------
 */

rolldice(g)
struct game *g;
{

clearmvs(g->mvs);			/* clear old stuff */
g->mvs[0].roll = Rolldie();             /* roll the dice */
g->mvs[1].roll = Rolldie();
g->rolls[g->mvs[0].roll - 1]++;		/* keep count of what rolls we got */
g->rolls[g->mvs[1].roll - 1]++;
if (g->mvs[0].roll == g->mvs[1].roll) {	/* hot damn, we got doubles */
	g->mvs[2].roll = g->mvs[0].roll;	/* copy roll into two */
	g->mvs[3].roll = g->mvs[0].roll;	/* more moves */
	g->doubles[g->mvs[1].roll - 1]++;	/* keep track of doubles */
	}
legalmoves(g);			/* calculate the # of moves & hi roll */
}



/*----------------------------------------------------------------------
 *	sendpkt -- send a packet to the opponent
 *
 * This function fills in the fields of a packet and passes that
 * packet to the transport using TSendPacket.  It also stores the
 * opcode sent in the lastop field of the game, so the packet
 * can be regenerated if necessary.  Sendpkt returns 1 if the
 * packet was sent, and 0 if an error occurred.
 *----------------------------------------------------------------------
 */

sendpkt(g,op)
struct game *g;
char op;
{
static char colors[4], adbl[10], mch[10];
char cmt[60], cmt2[60], *sendaddr;
int i, status;

if (FeIsActive)
	FeMessage("Sending...");
sendaddr = g->opaddr;		/* this is overridden for NOTIFY */
if ( (op != RESEND) && (op != NOTIFY) )
	g->lastop = op;			/* save last op for resend */
for (i = 0; i < 4; i++)
	g->blot[i] = 0;			/* clear blots hit */
if (*rc.chkpt == 'y') {
	writegames(rc.gfile,rc.gbackup,rc.pfile);
	rc.gbackup = NULL;	/* only backup old file once */
	}
P.version = LDB_VER;			/* these fields go in all packets */
P.gameid = g->gameid;
P.opcode = op;
P.seq = g->seq;
P.jacoby = NULL;
P.crawford = NULL;
P.european = NULL;
P.perm = NULL;
P.match = NULL;
if (g->opver > 100) {	/* versions after 1.0 rot13 comments */
	if (g->mycmt != NULL) {
		strncpy(cmt,g->mycmt,sizeof(cmt));	/* make copy */
		cmt[sizeof(cmt)-1] = '\0';		/* null term */
		P.comment = cmt;			/* save pointer */
		rotate(P.comment);			/* rot13 the copy */
		}
	else
		P.comment = NULL;
	if (g->mycmt2 != NULL) {
		strncpy(cmt2,g->mycmt2,sizeof(cmt2));	/* make copy */
		cmt2[sizeof(cmt2)-1] = '\0';		/* null term */
		P.comment2 = cmt2;			/* save pointer */
		rotate(P.comment2);			/* rot13 the copy */
		}
	else
		P.comment2 = NULL;
	}
else {				/* version 1.0 sends comments as cleartext */
	P.comment = g->mycmt;
	P.comment2 = g->mycmt2;
	}
if (g->flags & F_SENTNAME) {
	P.name = NULL;
	P.notify = NULL;
	}
else {
	P.name = rc.myname;
	P.notify = g->notify;
	g->flags |= F_SENTNAME;
	}
P.addr = NULL;				/* these fields only used by START */
P.colors = NULL;
P.dir = NULL;
P.autodbl = NULL;			/* used by START and TIE */
P.timestamp = time((long *) 0);		/* attach timestamp */
if (g->lastacc < P.timestamp)		/* update last access time */
	g->lastacc = P.timestamp;	/* but don't let time go backwards */
clearmvs(P.mvs);
switch (op) {				/* now do operation-specific stuff */
case START:
	P.addr = g->myaddr;		/* send opponent my email address */
	P.mvs[0].roll = g->mvs[0].roll;	/* send initial die roll */
	sprintf(colors,"%c%c",g->mycolor,g->opcolor);
	P.colors = colors;
	P.dir = (g->mydir > 0) ? "down" : "up";
	sprintf(adbl,"%d",rc.autodouble);
	P.autodbl = adbl;
	if (g->flags & F_JACOBY)	/* enable jacoby */
		P.jacoby = "yes";
	if (g->flags & F_CRAWFORD)	/* enable crawford */
		P.crawford = "yes";
	if (g->flags & F_EUROPE)	/* enable european rule */
		P.european = "yes";
	if (g->flags & F_PERM)		/* game is permanent */
		P.perm = "yes";
	if (g->mtotal > 0) {		/* enable match play */
		sprintf(mch,"%d",g->mtotal);
		P.match = mch;
		}
	break;
case USTART:
	P.mvs[0].roll = g->mvs[0].roll;	/* send both initial dice */
	P.mvs[1].roll = g->mvs[1].roll;
	break;
case MSTART:
case RESTART:				/* retry initial roll */
	P.mvs[0].roll = g->mvs[0].roll;	/* send new roll */
	break;
case TIE:
	if (g->adcnt > 0) {		/* send current autodouble count */
		sprintf(adbl,"%d",g->adcnt);
		P.autodbl = adbl;
		}
	break;
case MOVE:
	for (i = 0; i < 4; i++)
		P.mvs[i] = g->mvs[i];
	break;
case NOTIFY:
	P.addr = g->myaddr;
	P.seq = 0;			/* no sequences for notify packets */
	P.notify = NULL;
	P.name = rc.myname;
	i = gvalue(g,&status);
	sprintf(cmt,"%d %d %d",g->term,status,i);
	P.comment = cmt;	/* send term code, game val, & bg flag */
	P.comment2 = g->opaddr;	/* send opponent address */
	if ( (sendaddr = g->notify) == NULL)
		return;		/* shouldn't happen */
	}
status = TSendPacket(&P,sendaddr);		/* send the packet */
if (FeIsActive)				/* clear "Sending..." from mesg line */
	FeMessage(NULL);
return(status);
}



/*----------------------------------------------------------------------
 *	resendpkt -- resend the last packet
 *
 * This function takes a game structure and causes the last packet sent
 * in that game to be resent.
 *
 * The F_SENTNAME flag is cleared before resending, causing our personal
 * name to be resent.  This is because the packet being resent might have
 * been the one that sent the personal name (and set the F_SENTNAME flag),
 * and since the F_SENTNAME flag is now set, the personal name would not
 * be included in the resent packet.
 *----------------------------------------------------------------------
 */

resendpkt(g)
struct game *g;
{

g->flags &= ~F_SENTNAME;
sendpkt(g,g->lastop);
}



/*----------------------------------------------------------------------
 *	str2mv -- decode move string to struct mv
 *
 * This function takes a string representation of a move, decodes it,
 * and places the information into a mv structure.  This format is:
 *
 *	roll/move
 *
 * where roll is the die value used in the move, and is a single
 * digit in [1..6], and move is one of the following:
 *
 *	a 1 or 2 digit number in [1..24]
 *		This designates the point the move originates from.
 *		This number is stored in the "pt" field of the move
 *		structure without modification.
 *	the string "BAR"
 *		This means the piece is coming off the bar.
 *		Zero is stored in the "pt" field of the move structure,
 *		regardless of whether the player's bar point is 0 or 25.
 *		Apply() and FeDrawMove understand this and convert 0 to
 *		the appropriate bar point before using it.
 *	the string "UNUSED"
 *		This means the roll is unused.  -1 is stored in the
 *		"pt" field of the mv structure.
 *----------------------------------------------------------------------
 */

str2mv(s,m)
char *s;
struct mv *m;
{
char *p, *strchr();

if ( (p = strchr(s,'/')) == NULL) {
	message("ERROR: malformed move: %s\n",s);
	return;
	}
if ( ( (m->roll = atoi(s)) < 0) || (m->roll > 6) ) {
	message("ERROR: invalid roll: %d\n",m->roll);
	return;
	}
p++;
if ( (m->roll == 0) || (*p == 'U') || (*p == 'u') )
	m->pt = -1;		/* this roll is unused */
else if ( (*p == 'B') || (*p == 'b') )
	m->pt = 0;		/* move from bar */
else if ( ( (m->pt = atoi(p)) < 0) || (m->pt > 25) ) {
	message("ERROR: invalid point: %d\n",m->pt);
	return;
	}
}


/*----------------------------------------------------------------------
 *	mv2str -- encode move string from struct mv
 *
 * This function forms a string representation of a move based on
 * the information in a mv structure.  This format is:
 *
 *	roll/move
 *
 * where roll is the die value stored in mv->roll, and move is:
 *
 *	mv->pt if mv->pt is in [1..24]
 *	the string "BAR", if mv->pt is 0 or 25
 *	the string "UNUSED", if mv->pt is < 0
 *----------------------------------------------------------------------
 */

mv2str(m,s)
struct mv *m;
char *s;
{

if (m->roll <= 0) {		/* non-existant roll */
	strcpy(s,"0/0");	/* should be skipped by nvwrite */
	return;			/* so we should never get here */
	}
if (m->pt < 0)
	sprintf(s,"%d/UNUSED",m->roll);
else if ( (m->pt == DOWNBAR) || (m->pt == UPBAR) )
	sprintf(s,"%d/BAR",m->roll);
else
	sprintf(s,"%d/%d",m->roll,m->pt);
}


/*----------------------------------------------------------------------
 *	clearmvs -- mark all entries in a mv array empty
 *
 * This function marks all elements of a mv array as being unused.
 *----------------------------------------------------------------------
 */


clearmvs(m)
struct mv *m;
{
int i;

for (i = 0; i < 4; i++) {
	m[i].roll = -1;
	m[i].pt = -1;
	}
}



/*----------------------------------------------------------------------
 *	canmove -- see if a roll is usable in a range of points
 *
 * This function trys to use a roll over a range of points.  If it
 * finds a point where a roll could be used, it returns 1, otherwise
 * it returns 0.  The board is not changed.
 *----------------------------------------------------------------------
 */

canmove(g,r,p1,p2)
struct game *g;		/* the game structure */
int r;			/* which element of g->mvs */
int p1, p2;		/* the range of points to try */
{
int i, op;

if (p1 > p2) {
	i = p1;
	p1 = p2;
	p2 = i;
	}
op = g->mvs[r].pt;		/* so we can restore it */
for (i = p1; i <= p2; i++) {
	g->mvs[r].pt = i;		/* try from this point */
	if (apply(g,WHO_ME,r,A_CHKONLY,NULL) >= 0) {
		g->mvs[r].pt = op;
		return(1);
		}
	}
g->mvs[r].pt = op;
return(0);
}


/*----------------------------------------------------------------------
 *	rotate -- rot13 a buffer
 *
 * This function performs the popular "rot13" conversion to a buffer.
 * The buffer is modified in place.  This conversion makes a
 * string unreadable by adding 13 to letters in [A-Ma-m] and subtracting
 * 13 from letters in [N-Zn-z].  All other characters are unchanged.
 * Applying the rot13 transformation again returns the buffer to normal.
 *----------------------------------------------------------------------
 */

rotate(buf)
char *buf;
{
register char *s;

for (s = buf; *s; s++) {
	if (! isalpha(*s))
		continue;
	if ( ((*s >= 'A') && (*s <= 'M')) || ((*s >= 'a') && (*s <= 'm')) )
		*s += (char) 13;
	else
		*s -= (char) 13;
	}
}


/*----------------------------------------------------------------------
 *	message -- print a message in proper way
 *
 * This function checks to see if the front end is active.  If
 * so, it calls FeMessage, otherwise it prints on stderr and
 * sets the FeWaitInit flag so that when FeInitialize is called,
 * it will wait for return to be pressed before clearing the screen.
 *----------------------------------------------------------------------
 */

message(s,a1,a2,a3)
char *s;
int a1, a2, a3;
{
char buf[80];

if (FeIsActive) {
	sprintf(buf, s, a1, a2, a3);
	FeMessage(buf);
	}
else {
	fprintf(stderr,s,a1,a2,a3);
	FeWaitInit++;
	if (FeWaitInit > 22) {
		fprintf(stderr,"Press <return> to continue...");
		fgets(buf,sizeof(buf),stdin);
		FeWaitInit = 0;
		}
	}
}


/*----------------------------------------------------------------------
 *	fatal -- terminate program with error message
 *
 * This function prints a message to stderr and exits.
 *----------------------------------------------------------------------
 */

fatal(msg)
char *msg;
{

FeFinishSession();
TFinishSession();
fprintf(stderr,"%s\n",msg);
ldbexit(STAT_ABORT);
}


/*----------------------------------------------------------------------
 *	ldbexit -- terminate program in an orderly fashion
 *
 * This function can be called at any time, and ensures that the
 * front end and the transport are closed down cleanly before
 * exiting.
 *----------------------------------------------------------------------
 */

ldbexit(code)
int code;
{

FeFinishSession();
TFinishSession();
release_lock(rc.lockfile);
fflush(stdout);
fflush(stderr);
exit(code);
}


/*----------------------------------------------------------------------
 *	pipcount -- calculate a pip count for both players
 *
 * This function takes a board and a pointer to a game structure
 * and calculates a pip count for both players.  These are returned
 * in *mp and *op.
 *----------------------------------------------------------------------
 */

pipcount(b,g,mp,op)
board b;			/* the board */
struct game *g;			/* the game */
int *mp;			/* where to store my pip count */
int *op;			/* where to store opponent's pip count */
{
int direction[2];
int player;			/* me = 0; op = 1 */
int PIP[2];			/* mypip = 0; oppip = 1 */
int i;				/* counter */

PIP[WHO_ME] = 0;			/* initialize my PIP count */
PIP[WHO_OPP] = 0;			/* initialize op PIP count */
direction[WHO_ME] = g->mydir;		/* my direction of travel */
direction[WHO_OPP] = g->opdir;		/* op direction of travel */

for (i = 0; i <= DOWNBAR; i++) {	/* for each point on the board */

	if (b[i].qty > 0) {
		player = (b[i].color == g->mycolor) ? WHO_ME : WHO_OPP;
		if ((i == UPBAR) || (i == DOWNBAR))
			PIP[player] += b[i].qty * 25;
		else {
			if (direction[player] == 1)
				PIP[player] += b[i].qty * (25 - i);
			else
				PIP[player] += b[i].qty * i;
			}
		}
	}
*mp = PIP[WHO_ME];		/* return my pip count */
*op = PIP[WHO_OPP];		/* return opponent's pip count */
}


/*----------------------------------------------------------------------
 *	gvalue -- calculate the game value
 *
 * This function takes a completed game and calculates its value,
 * taking into account gammons, backgammons, and the Jacoby and
 * European rules.  These are:
 *	- Gammon (when the loser does not bear off any pieces)
 *	  counts as double the game value shown on the cube.
 *	- Backgammon (when the loser does not bear off any pieces
 *	  and has pieces in the winner's home board) counts as
 *	  triple the game value shown on the cube.
 *	- Jacoby rule (if enabled) states that Gammons and Backgammons
 *	  count as single games if neither player doubled during the game.
 *	- European rule (if enabled) states the Backgammons count double,
 *	  not triple, the game value shown on the cube.
 * The game value is returned in *vp; the return value is 1 for a gammon,
 * 2 for a backgammon, and 0 for neither.
 *
 * If the game was conceded, gvalue() scores a gammon if the loser has
 * not borne off any pieces, and a backgammon if the loser has any pieces
 * in the winners inner table.  This prevents players from conceding to
 * avoid a gammon/backgammon.
 *----------------------------------------------------------------------
 */

gvalue(g,vp)
struct game *g;
int *vp;
{
int bf;
int p1, p2;

bf = 0;				/* init to no gammon/backgammon */
*vp = g->gameval;		/* init to game value on cube */
if ( (g->term == T_ILOSE) || (g->term == T_ICONCEDE) ) {
	if (g->board[OFFPT(g->mydir)].qty == 0) {
		p1 = (g->opdir > 0) ? 19 : 0;/* check op's inner tbl*/
		p2 = (g->opdir > 0) ? 25 : 6;	/* for my pieces */
		if (addpcs(g->board,g->mycolor,p1,p2) > 0)
			bf = 2;			/* flag a backgammon */
		else
			bf = 1;			/* flag a gammon */
		}
	}
else if ( (g->term == T_IWIN) || (g->term == T_OPCONCEDE) ) {
	if (g->board[OFFPT(g->opdir)].qty == 0) {
		p1 = (g->mydir > 0) ? 19 : 0;	/* check my inner tbl*/
		p2 = (g->mydir > 0) ? 25 : 6;	/* for op's pieces */
		if (addpcs(g->board,g->opcolor,p1,p2) > 0)
			bf = 2;			/* flag a backgammon */
		else
			bf = 1;			/* flag a gammon */
		}
	}
if ( (g->flags & F_JACOBY) && (g->gameval == (1 << g->adcnt)) )
	return(bf);	/* jacoby enabled & no doubles, don't mult game val */
if ( (g->flags & F_EUROPE) && (bf == 2) ) {
	*vp *= 2;	/* european rule enabled, bg = 2 multiplier */
	return(bf);
	}
if (bf == 2)
	*vp *= 3;	/* backgammon = 3 multiplier */
else if (bf == 1)
	*vp *= 2;	/* gammon = 2 multipler */
return(bf);		/* return gammon/backgammon/neither flag */
}


/*----------------------------------------------------------------------
 *	iscontact -- determine if contact is possible for a game
 *
 * This function returns 1 if it is possible for any piece to hit
 * any other piece, 0 if both players have moved past each other
 * and the game is a race.
 *----------------------------------------------------------------------
 */

iscontact(g)
struct game *g;
{
int dc, dn, i;

if (g->mydir > 0)		/* which color is down-bound? */
	dc = g->opcolor;
else
	dc = g->mycolor;
dn = g->board[DOWNOFF].qty;		/* start with # pcs borne off */
for (i = 0; (i <= 24) && (dn < 15); i++) {
	if (g->board[i].qty == 0)
		continue;
	if (g->board[i].color != dc)	/* found some upbound pieces */
		return(1);		/* that are in hitting range */
	dn += g->board[i].qty;		/* keep count of down pcs found */
	}
if (dn >= 15)				/* found all down pcs */
	return(0);			/* no more contact for this game */
return(1);
}


/*----------------------------------------------------------------------
 *	crawford_check -- is this the crawford rule game for a match?
 *
 * This function sets the F_CRGAME and F_CRDONE flags for a match.
 * F_CRGAME is set if this is the crawford rule game for the match.
 * F_CRDONE is set after the crawford rule game has been played.
 *
 * F_CRGAME is set if:
 *	F_CRAWFORD is set	AND
 *	F_CRDONE is not set	AND
 *	a player is within one point of winning the match.
 * otherwise F_CRGAME is cleared.
 *
 * F_CRDONE is set if:
 *	F_CRAWFORD is set	AND
 *	F_CRGAME was set (before the above)
 * otherwise F_CRDONE is not changed.
 *----------------------------------------------------------------------
 */

crawford_check(g)
struct game *g;
{
int old_crgame;

old_crgame = g->flags & F_CRGAME;		/* save F_CRGAME */
if ( ((g->flags & (F_CRAWFORD|F_CRDONE)) == F_CRAWFORD) &&
     ( (g->mcurrent[WHO_ME] == g->mtotal-1) ||
       (g->mcurrent[WHO_OPP] == g->mtotal-1) ) )
	g->flags |= F_CRGAME;		/* this is the crawford rule game */
else
	g->flags &= ~F_CRGAME;		/* not the crawford game, clear flag */
if ( (g->flags & F_CRAWFORD) && old_crgame)
	g->flags |= F_CRDONE;		/* crawford rule game is over */
}
