/*	main.c		8/3/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"

/*============================================================================
 *	ldb --	Long Distance Backgammon
 *
 * The following arguments are recognized:
 *	-read		Mail is read, and the games are updated and saved.
 *			The user is not prompted for his moves.
 *	-play		Any games that are waiting for local input are
 *			displayed for the user to process.  No mail is read.
 *	-color xy	The colors for any games started are set to x and y.
 *			The first color is played by the local user, and the
 *			other is played by the opponent.  Legal color
 *			characters are upper and lower case letters.
 *			The default is "-color rw".
 *	-direction up/down
 *			The direction of play for the local user is set
 *			to the specified value.  The default is
 *			"-direction up".
 *	-myaddr addr	Override the "myaddr" field of .ldbrc for any
 *			games started by this invocation of ldb.  This
 *			is effective only for games started by -start
 *			and only for -start's that appear after the -myaddr
 *			on the command line.
 *	-start user	A game is started with the specified user.  User may
 *			be either an e-mail address or an alias for an
 *			opponent that has been played previously.
 *	-remotestart user1 user2
 *			A game is started between user1 and user2.  The local
 *			host sends a remote start message to user1 instructing
 *			it to start a game with user2.  The local host
 *			does not participate thereafter in the game.
 *			For the purposes of the -color and -direction options,
 *			user1 is considered the local user.
 *	-broadcast file
 *			A file is mailed to all opponents.  This is useful
 *			for announcing vacation absences, etc.
 *	-help		A summary of available options is printed.  This
 *			list may be more up to date than these comments.
 *	-jacoby		Enables the jacoby rule for any games subsequently
 *			started.  This rule states that gammons or backgammons
 *			only count as 1 when neither player doubled during
 *			the game.
 *	-crawford	Enables the crawford rule for any games subsequently
 *			started.  This rule prevents doubling when either
 *			player is within 1 point of winning the match.
 *	-european	Enables european scoring for any subsequently
 *			created games.  This rule makes backgammons count
 *			as double games, rather than triple.
 *	-permanent	Marks any subsequently created games as permanent.
 *			Permanent games will be restarted whenever they
 *			end.  This is for people you play all the time.
 *			Permanent games must be deleted via -control.
 *	-match points	Marks any subsequently created games as matches
 *			that play to the specified number of points.
 *			Games will be restarted until one player wins
 *			this number of points.
 *	-control	Enters control mode.  No games may be started
 *			or played.  Each game is displayed, along with
 *			a menu that allows various administrative
 *			functions to be performed.
 *	-reconstruct file
 *			This option allows you to recover your .ldbdata
 *			file even if you lose all copies.  Simply have
 *			all your opponents send you their .ldbdata files
 *			and, one by one, feed them into ldb with the
 *			-reconstruct option.  Ldb will scan the file,
 *			find any games that have your name as the
 *			opponent, and rebuild your game data using the
 *			data in that game.
 *	-score		Print the cumulative score for all your opponents.
 *
 * If neither -read or -play is given, the default is to do both; incoming mail
 * is read, then any games requiring the user's attention are displayed.
 *
 * Note that the -start and -remotestart options use the color and direction
 * options that have been set *at the time the argument is processed*.
 * Thus,
 *
 *	ldb -start joe@momma -color rb
 *
 * will NOT use the colors r and b for the game with joe, but
 *
 *	ldb -color rb -start joe@momma
 *
 * will.  The color and direction arguments may be changed between -start
 * and -remotestart arguments.  For example,
 *
 *	ldb -color wr -direction down -start user1 -direction up -start user2
 *
 * will start a game with user1 with your direction being down, and one with
 * user2 with your direction being up.
 *
 *----------------------------------------------------------------------------
 *				REVISION HISTORY
 *** Version 01   Rev 0   Patch 0
 * 1.	Initial release.  Countless helpful suggestions from Karen Ward.
 *
 *** Version 01   Rev 1   Patch 0
 * 1.	The personal name and e-mail address printed on the top line
 *	were switched.  It makes more sense to be playing a person rather
 *	than an address.  If the e-mail address is too long to fit on
 *	the top line, it is truncated and an ellipsis is added.
 * 2.	Pressing ESC while entering a comment now returns you to the
 *	previous menu and discards the comment.  This allows you to
 *	change your mind after hitting "Send", for example.  Suggested
 *	by Franc,ois Pinard.
 * 3.	The checkpoint code was moved to sendpkt(), where the games
 *	are saved before the packet is sent.  This removes a window
 *	where a packet could be sent but ldb could croak before saving
 *	the game.  Suggested by Earle Ake.
 * 4.	Ldb will not ask for the second digit of a point number when:
 *		The first digit is 3 - 9,
 *		The first digit is 1 and there is no point in 10 - 19
 *			that could use the selected roll, or
 *		The first digit is 2 and there is no point in 20 - 24
 *			that could use the selected roll.
 *	Suggested by Earle Ake and Franc,ois Pinard.
 * 5.	When a roll is selected and a piece is on the bar, no point is
 *	prompted for, since moving off the bar is the only legal move.
 *	Suggested by Franc,ois Pinard.
 * 6.	The Makefile now used $(CC) instead of cc.  Suggested by
 *	Franc,ois Pinard.
 * 7.	A number of patches to make ldb run under SCO Xenix (and probably
 *	SCO UNIX as well) were sent by Dinesh Vichare.
 * 8.	A bare bones rename function is included for those systems that
 *	don't have one.  Suggested by Franc,ois Pinard.
 * 9.	Comments are now rot13'd before they are sent.  This keeps
 *	them from being read while mail is extracted.  They are
 *	rotated back when ldb displays them.  This is disabled if the
 *	opponent is using the old (1.0) ldb version, so 1.1 and 1.0
 *	ldb's will still work together.  The only catch is when you switch
 *	from 1.0 to 1.1 with games in progress, your opponents won't know
 *	you've switched until you send them a move.  As a result, the
 *	comment from their next message will be garbled.  To avoid this,
 *	do "ldb -read" under 1.0, then switch to 1.1 to play.  This is
 *	only necessary once, after that your opponent's will know you
 *	are using 1.1 and the messages will work normally.
 *	Suggested by Franc,ois Pinard.
 * 10.	Ldb now clears the screen before exiting.  Keeps wandering
 *	manager types from seeing the remnants of a game.
 * 11.	The sequence number warning was removed for leftover messages
 *	from dead games.
 * 12.	There is now a "Press <return> to continue" before the screen
 *	is drawn if any messages were printed.  This lets you read
 *	the messages before the screen is cleared.  Suggested by
 *	practically everybody.
 * 13.	Default file names for vms are now ldb.rc, ldb.data,  and ldb.olddata.
 *	This keeps the ldb files together and out of the way, since vms
 *	doesn't treat filenames with a leading '.' like UNIX does.
 *	Suggested by Earle Ake.
 * 14.	There is now a debug setting in .ldbrc.  It is a mask, with each
 *	bit enabling debug info from a different piece of code.  The
 *	bits are defined as DB_* in ldb.h Suggested by Earle Ake.
 * 15.	A sum of all rolls is kept in the "rolls" and "doubles" fields
 *	of the game structure.  This counts the number of each type of
 *	roll you get, and the number of doubles of each type.
 *	The same is true for your opponent's rolls and doubles in
 *	the oprolls and opdoubles fields.  Suggested by Earle Ake.
 * 16.	The crawford and jacoby rules are now supported.  They are activated
 *	by the -crawford and -jacoby command line options, and can only
 *	be enabled by whoever starts the game.  If they are used to start
 *	a game with a version of ldb that does not support them, a message
 *	is printed and they are disabled.
 * 17.	Ldb now allows a game to be declared "permanent", meaning that
 *	it is automatically restarted when it is over.  This is activated
 *	the the -permanent command line option, and can only be enabled
 *	by whoever starts the game.
 * 18.	Ldb now supports match play.  This is activated by the -match
 *	command line options, which takes a numeric argument specifying
 *	how many points the match plays to.  When a game that is part of
 *	a match ends, and the specified number of points has not been reached,
 *	the game is restarted.
 * 19.	The "buttons" in the menu in fe_curses now light up to show the
 *	item that was chosen.
 * 20.	Ldb now supports the "european rule", which states that backgammons
 *	only count double, not triple, the game value.
 * 21.	Ldb now creates a lock file when it starts in such a way as to
 *	prevent multiple ldb processes from running in the same account at the
 *	same time.  The lock code does not depend on any operating system
 *	locking features, other than O_EXCL, and so should be fairly portable.
 * 22.	There is now an option in .ldbrc to automatically delete mail
 *	files after they have been read.  It will refuse to delete
 *	files that begin with '/' (on UNIX only) to help prevent
 *	catastrophic mistakes.  This option is off by default,
 *	and should only be used with great care.
 *	You should also read the disclaimer at the top of this (and all
 *	other) files, where it says I'm not responsible.  Caveat User.
 * 23.	A "timeout" setting has been added to .ldbrc.  Games that have
 *	not been accessed for more than "timeout" days (default 7),
 *	and are waiting on remote input, have the last packet automatically
 *	resent.  If timeout is set to 0, automatic resends are disabled.
 * 24.	A "keepold" setting has been added to .ldbrc.  Games that are over
 *	are retained in .ldbdata for "keepold" days (default 7), allowing
 *	time for the last move to be resent if necessary.  If "keepold" is
 *	set to 0, games are deleted as soon as they are over.
 * 25.	Old start packets are now rejected.  To do this, a new file
 *	called the "people" file is maintained.  This file is organised
 *	by opponent address, and stores the start time of the newest completed
 *	game, called the "fence".  Any start packets not newer than the
 *	fence are discarded.  The only way a legitimate start packet would be
 *	rejected is if it were not fed to ldb until after another game with
 *	the same opponent had been started and finished.  This should never
 *	happen, but if it did, all that would be required to fix it would
 *	be to have the opponent resend the start packet via -control.
 * 26.	The people file also stores the opponent's name and
 *	an alias, or nickname, for the opponent.  The alias is initialized
 *	to the first word of the opponent's name with all capitals changed
 *	to lower case.  To change an opponent's alias, a text editor may
 *	be used on the people file.  The -start command will take an
 *	alias in place of an e-mail address.  There is no check for
 *	duplicate aliases, and if one is used in a -start command, the
 *	game will be started with the first record found in the people
 *	file with that alias.
 * 27.	The people file also stores a record of all games won/lost to
 *	that opponent.  The data stored includes games won/lost, points
 *	won/lost, gammons won/lost, backgammons won/lost, and matches won/lost.
 *	There is currently no utility to print these out, but they can
 *	be extracted from the people file manually.  See the definitions
 *	for SC_* in ldb.h to see which number is which.
 * 28.	There is now a way to reconstruct the .ldbdata file from your
 *	opponents' .ldbdata files.  Just have them mail their ldbdata
 *	files to you, remove the mail header, and run:
 *		ldb -reconstruct file
 *	where file is the file containing the opponent's ldbdata file.
 *	Each game in that file showing you as the opponent is extracted
 *	and inserted into your game list.  You will be prompted for the
 *	address of your opponent (and name, if they are not in the
 *	people file) for each game found.
 * 29.	The people file has a method for handling people with multiple
 *	e-mail addresses.  When a new opponent is found with the
 *	same name as an existing one, but a different address, the
 *	user is asked if these addresses refer to the same person.
 *	If they do, an "equiv" record is added to the people file,
 *	which merely records that the second address refers to the
 *	same person as the first.  If they don't (e.g. John Smith),
 *	two separate people records are created for the different people.
 * 30.	There is now a way to request a resend from your opponent under
 *	-control.  The menu item is "Get Resend", since "Request Resend"
 *	would have used the same first letter as "Resend". (oh well)
 *	A special packet is sent to your opponent; when his ldb receives it,
 *	it will perform a resend automatically.  This feature is disabled
 *	if the opversion field of the game structure indicates your
 *	opponent is using a pre-1.1 version of ldb.  Sending the resend
 *	packet to an ldb older than 1.1 would cause it to abort.  Oops.
 * 31.	Any upper or lower case letters may now be used to draw the
 *	pieces on the board.  Previously only r, w, and b were used.
 *	As r, w, and b stood for red, white, and black, the option
 *	to set them was called -color.  While the letters no longer
 *	(necessarily) represent colors, the option is still -color.
 * 32.	A screen dump of a game may be obtained via the -control menu.
 *	The dump goes to ldb_screen.dmp.
 *
 * ---  The following modifications made by Earle Ake.
 *
 * 1.	A context-sensitive help function was added that is activated by
 *	pressing 'h' or '?' at any time other than while composing a message.
 * 2.	The redraw key now works under VMS.  Clearok() needed to be called
 *	to force a redraw.
 * 3.	The superkey function now works under VMS.  It was rewritten to
 *	spawn a subprocess and execute the supercmd if specified.
 *	The user must logout of the subprocess to resume the game.
 * 4.	The .ldbrc file now contains reasonable defaults for VMS when
 *	the VMS-compiled ldb creates it.  The sendcmd setting calls
 *	the IN% mailer by default.
 * 5.	Old versions of .oldldbdata are now purged.
 * 6.	setattr() and clrattr() are used instead of standout() and
 *	standend() to get reverse video on VMS.  standout() on VMS
 *	used bold instead of reverse video, which made the cube
 *	look non-cubical.
 * 7.	The current pip count for you and your opponent are displayed
 *	above the board.  These change depending on which board is
 *	displayed, and are updated as you move.
 * 8.	An extensive display of statistics regarding the number and
 *	frequency of rolls and doubles for both users can be displayed
 *	by pressing the % or # keys.  This data can also be displayed
 *	as a histogram.
 *
 *** Version 01   Rev 2   Patch 0
 *** Note Rev 2 was never officially released.  A number of beta versions
 *** of 1.2 were distributed, and this led to so much confusion that the
 *** official release was called 1.3.
 * 1.	The "Get Resend" option in -control is disabled for games
 *	that are over.  Patch by Earle Ake.
 * 2.	All occurrences of "gets" have been replaced by "fgets" to
 *	avoid buffer overrun problems.
 * 3.	Ldb now detects games where no further contact is possible and
 *	changes the "BAR" indicator to "---".
 * 4.	The "Resend" option in -control is disabled for games that are in
 *	a "local" state (i.e. it is the local player's turn).
 * 5.	The cumulative score from the people file can now be displayed
 *	via the -score command line argument.  This is similar to a
 *	system implemented by Doug Parisek, but uses the info from the
 *	people file rather than from a separate standings file.
 * 6.	An option has been added to make ldb notify the game starter
 *	when a game started by -remotestart has ended.  If a
 *	-notify <addr> option is on the command line before the
 *	-remotestart, the ldb from both players will send a message
 *	to <addr> when the game ends.  This message will look like
 *	a normal ldb packet, but the opcode will be NOTIFY, and
 *	the comment field will contain three numbers; these are
 *	the termination code (T_* in ldb.h), the game value, and
 *	the backgammon flag (1 = gammon, 2 = backgammon, 0 = neither).
 *	Note that the integer values of the termination codes changed
 *	between rev 1.0 and rev 1.1 -- the one in the comment field
 *	corresponds to the new codes (the ones found in ldb.h from
 *	rev 1.1 or higher).  The comment2 field contains the address
 *	of the opponent played.  Ldb itself has no provisions to read
 *	notify packets, and would not be able to do anything useful
 *	with them if it could.  They are purely for the use of a
 *	game starter, if anyone feels like writing one.
 * 7.	Remote start packets left in the mail file no longer start
 *	a new game every time ldb is run.  This is done by scanning
 *	the game list for a game where both opaddr and starttime match
 *	those shown in the remotestart packet, and discarding it if
 *	one is found.  If it is not found, the address is looked up in
 *	the people list and the fence time, if any, is checked.
 *	This catches the case where the game started by that remotestart
 *	packet has already finished.  Bug found by Mark Rubin.
 * 8.	The -remotestart option now supports the crawford, jacoby, europe,
 *	and permanent options, as well as match play.
 * 9.	The crawford rule code has been rewritten.  It now (correctly)
 *	only disallows doubling for the first game after either player
 *	reaches match score - 1.  Subsequent games allow doubling.
 *	During the crawford rule game, the C indicator above the board
 *	is drawn in reverse video.
 * 10.	Concede now scores a gammon if the loser has not borne off any
 *	pieces, and a backgammon if the loser has any pieces in the winners
 *	inner table.  This fixes a bug where Concede could be used to
 *	avoid a gammon/backgammon.
 * 11.	Ldb now scans the mail file(s) before checking for access timeouts.
 *	Previously, it would perform automatic resends for games when
 *	a message for that game was waiting in the mail, resulting in an
 *	unnecessary resend.  This was common when returning from vacation,
 *	for example, when all games would time out.
 * 12.	The error messages displayed when a move is rejected have been
 *	made more understandable.  Previously the same set of messages
 *	were used for the local player and for received moves, and the
 *	wording was somewhat awkward to allow this double usage.  Local
 *	errors now use the messages in rejlcl[], while received moves still
 *	use the messages in rejmsg[].
 * 13.	A serious bug was fixed in rcvop.c.  restart() and mstart() did not
 *	call legalmoves() to regenerate the maxused and hiused fields of
 *	the game structure.  The consequence of this for restart() is that
 *	move checking will not work for the first move of any game that had
 *	a tie on an opening roll.  For mstart(), which starts the next
 *	game of a match, maxused and hiused will still be whatever they
 *	were for the last move of the previous game, which could result in
 *	either allowing an illegal move or disallowing a legal one.
 *	Bug found by Earle Ake.
 * 14.	Cbreak is now turned off during the supercmd.
 * 15.	Any printing character (other than space) is allowed to be used
 *	to draw game pieces.  These can be upper or lower case letters,
 *	numbers, or punctuation.  The only restriction is that the characters
 *	used may not be the same for both players.  This includes using
 *	the upper and lower case of the same letter.
 * 16.	Command line options may now be abbreviated by the shortest
 *	unique string.
 * 17.	If the -broadcast option is given without a file argument, it
 *	reads from stdin.  -broadcast also reads the people file instead
 *	of the game file, so duplicate messages are not sent to opponents
 *	with more than one game in progress.
 * 18.	The -start and -remotestart options are deferred until after all
 *	options have been scanned.  This removes order dependencies in
 *	command line options, so that:
 *		ldb -match 7 -start joe
 *	and	ldb -start joe -match 7
 *	have identical effect.  Because of this change, only one -start and
 *	one -remotestart may be used per run of ldb.  Suggested by Earle Ake.
 *** Version 01   Rev 2   Patch 0
 * 1.	A warning is printed if the crawford rule is used with a pre-1.3
 *	version of ldb.  The Crawford rule was fixed during 1.2, but many
 *	beta versions were distributed before this fix was included.
 *============================================================================
 */

main(argc,argv)
int argc;
char *argv[];
{
struct game *g;
struct people *p;
FILE *fp;
char subj[128];
char *bcfile;
int i, j;
char c, c2;
int done;
int ldbsignal();
int flags, match;
char *start_addr;
char *rst1, *rst2;

ghead = NULL;			/* init game list to empty */
gtail = NULL;
signal(SIGINT,ldbsignal);	/* set up interrupt trap to save games */
Rflag = 1;		/* should we try to extract incoming mail? */
Pflag = 1;		/* should we process waiting games? */
RandomInit(time((long *)0));	/* seed the random number generator */

readldbrc();		/* read startup file */

get_lock(rc.lockfile);	/* is another ldb already running? */

cr_mycolor = rc.defclrs[0];	/* default color when creating games */
cr_opcolor = rc.defclrs[1];
cr_mydir = (*rc.defdir == 'u') ? 1 : -1;	/* default direction */

readgames();		/* load games in progress */

match = 0;		/* default to no match play */
flags = 0;		/* default to no special rules or perm games */
notify = NULL;		/* default to no notify address */
start_addr = NULL;	/* default to no game started */
rst1 = NULL;		/* default to no remote start game */
rst2 = NULL;
for (i = 1; (i < argc) && (argv[i][0] == '-'); i++) {
	j = optlookup(&argv[i][1]);
	if (j == -2) {
		printf("%s:\tambiguous option: %s\n\n",*argv,argv[i]);
		usage(0);		/* print short help */
		ldbexit(STAT_ABORT);
		}
	if (j < 0) {
		printf("%s:\tunrecognized option: %s\n\n",*argv,argv[i]);
		usage(0);		/* print short help */
		ldbexit(STAT_ABORT);
		}
	switch (options[j].index) {
	case OPT_START:			/* start a game */
		i++;
		if (argv[i] == NULL) {
			printf("%s: -start needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		if (start_addr != NULL) {
			printf("%s: only one -start allowed.\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		start_addr = argv[i];
		break;
	case OPT_RSTART:		/* remote start */
		i++;
		if ( (argv[i] == NULL) || (argv[i+1] == NULL) ) {
			printf("%s: -remotestart needs two arguments\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		if (rst1 != NULL) {
			printf("%s: only one -remotestart allowed.\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		rst1 = argv[i];
		rst2 = argv[i+1];
		i++;
		break;
	case OPT_READ:
		Pflag = 0;		/* just read, no processing */
		break;
	case OPT_PLAY:
		Rflag = 0;		/* just process, no read */
		break;
	case OPT_MYADDR:		/* set my e-mail address */
		i++;
		if (argv[i] == NULL) {
			printf("%s: -myaddr needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		strcpy(rc.myaddr,argv[i]);	/* copy in new address */
		break;
	case OPT_HELP:			/* print long help */
		usage(1);
		ldbexit(STAT_NORM);
	case OPT_NOTIFY:
		i++;
		if (argv[i] == NULL) {
			printf("%s: -notify needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		notify = argv[i];
		break;
	case OPT_COLOR:			/* set colors */
		if (argv[++i] == NULL) {
			printf("%s: -color option needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		cr_mycolor = argv[i][0];	/* first char is my color */
		cr_opcolor = argv[i][1]; /* second char is opponent's color */
		if ( (! isprint(cr_mycolor)) || (cr_mycolor == ' ') ) {
			printf("%s: invalid color: %d\n",*argv,cr_mycolor);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		if ( (! isprint(cr_opcolor)) || (cr_opcolor == ' ') ) {
			printf("%s: invalid color: %d\n",*argv,cr_opcolor);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		c = cr_mycolor;
		if (isupper(c))
			c = tolower(c);
		c2 = cr_opcolor;
		if (isupper(c2))
			c2 = tolower(c2);
		if (c == c2) {
			printf("%s: duplicate color: %c\n",*argv,cr_mycolor);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		break;
	case OPT_DIRECTION:		/* set direction */
		if (argv[++i] == NULL) {
			printf("%s: -direction option needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		c = argv[i][0];
		if (isupper(c))
			c = tolower(c);
		if (c == 'u')
			cr_mydir = 1;		/* I play up */
		else if (c == 'd')
			cr_mydir = -1;		/* I play down */
		else {
			printf("%s: invalid direction: %s\n",*argv,argv[i]);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		break;
	case OPT_JACOBY:
		flags |= F_JACOBY;
		break;
	case OPT_CRAWFORD:
		flags |= F_CRAWFORD;
		break;
	case OPT_EUROPE:
		flags |= F_EUROPE;
		break;
	case OPT_PERM:
		flags |= F_PERM;
		break;
	case OPT_MATCH:
		if (argv[++i] == NULL) {
			printf("%s: -match option needs argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		if (! isdigit(*argv[i])) {
			printf("%s: -match needs numeric argument\n",*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		match = atoi(argv[i]);
		break;
	case OPT_SCORE:
		if (start_addr != NULL)
			printf("warning: -start not processed.\n");
		if (rst1 != NULL)
			printf("warning: -remotestart not processed.\n");
		printscore();
		ldbexit(STAT_NORM);
	case OPT_BCAST:				/* broadcast a message */
		if (argv[++i] == NULL) {	/* no arg, read stdin */
			if ( (fp = fopen(rc.tempfile,"w")) == NULL) {
				printf("%s: can't write temp file %s\n",
					*argv,rc.tempfile);
				ldbexit(STAT_ABORT);
				}
			while (fgets(subj,sizeof(subj),stdin) != NULL)
				fputs(subj,fp);
			fclose(fp);
			bcfile = rc.tempfile;
			}
		else
			bcfile = argv[i];	/* just read named file */
		sprintf(subj,"LDB Broadcast Message from %s",rc.myname);
		for (p = phead; p != NULL; p = p->next)	{ /* for all people */
			if (p->equiv != NULL)	/* it's an equiv record */
				continue;	/* skip it */
			for (g = ghead; g != NULL; g = g->next)	/* cur opp? */
				if (strcmp(g->opaddr,p->addr) == 0)
					break;
			if (g == NULL)	/* not a current opponent */
				continue;	/* skip it */
			TSendFile(p->addr,bcfile,subj); /* send msg */
			}
		break;
	case OPT_CONTROL:			/* control my games */
		control();
		if (start_addr != NULL)
			printf("warning: -start not processed.\n");
		if (rst1 != NULL)
			printf("warning: -remotestart not processed.\n");
		ldbexit(STAT_NORM);
	case OPT_RECONS:			/* reconstruct a game */
		if (argv[++i] == NULL) {
			printf("%s: -reconstruct option needs argument\n",
				*argv);
			usage(0);
			ldbexit(STAT_ABORT);
			}
		recons(argv[i]);
		if (start_addr != NULL)
			printf("warning: -start not processed.\n");
		if (rst1 != NULL)
			printf("warning: -remotestart not processed.\n");
		ldbexit(STAT_NORM);
	default:
		fprintf(stderr,
		   "Sorry, the %s option is not implemented yet.\n",
			options[j].name);
		ldbexit(STAT_ABORT);
		}
	}

if (start_addr != NULL)
	startgame(start_addr,cr_mydir,cr_mycolor,cr_opcolor,flags,match,0);
if (rst1 != NULL)
	remotestart(rst1,rst2,flags,match);

if ( (Pflag == 0) && (Rflag == 0) ) {	/* user gave both -play and -read */
	Pflag = 1;			/* turn both back on */
	Rflag = 1;
	}
while (i < argc)		/* if files given on command line, read them */
	readmail(argv[i++]);
if (Rflag)			/* if we are supposed to read default file */
	readmail(rc.mfile);	/* do that too */
i = 0;
for (g = ghead; g != NULL; g = g->next)	{ /* does any game need our input? */
	check_timeout(g);	/* check for access timeouts */
	if ( (g->state >= OPSTATES) &&
	     ! ( (g->state == ST_GAMEOVER) && (g->flags & F_DISPLAYED) ) )
		i++;
	}
if ( (i == 0) || (Pflag == 0) ) {		/* if not, exit */
	writegames(rc.gfile,rc.gbackup,rc.pfile);	/* save games */
	ldbexit(STAT_NORM);
	}
TInitialize();					/* fire up the transport */
FeInitialize();					/* fire up the front end */
FeDrawScreen();				/* draw the screen outline */
for (g = ghead, done = 0; (g != NULL) && (done >= 0); g = g->next)
	while ( (done = process(g)) > 0);	/* process game til done */
FeFinishSession();				/* close down the front end */
TFinishSession();				/* close down the transport */
writegames(rc.gfile,rc.gbackup,rc.pfile);	/* save the games in a file */
ldbexit(STAT_NORM);
}


/*----------------------------------------------------------------------
 *	ldbsignal -- signal handler
 *
 * This function is called when the user hits the interrupt character.
 * It is currently a very simple function; it saves the games in the
 * INTGFILE file, closes down the front end and the transport, and exits.
 *----------------------------------------------------------------------
 */

ldbsignal()
{

writegames(INTGFILE,NULL,INTPFILE);
FeFinishSession();	/* let front-end close down gracefully */
TFinishSession();	/* let transport close down gracefully */
fprintf(stderr,"WARNING: games saved in %s and %s\n",INTGFILE,INTPFILE);
ldbexit(STAT_ABORT);
}


/*----------------------------------------------------------------------
 *	usage -- print command line options.
 *
 * This function prints a help message.  This can be either in the
 * short or long format.  The short format merely lists all options
 * in a very dense format.  The long format prints each option on
 * a separate line, along with a short explanation of its purpose.
 *----------------------------------------------------------------------
 */

usage(help)
int help;		/* 0 = short message, 1 = long message */
{
struct opt *o;
int l;

printf("options:\n");
if (help) {		/* print out the whole shootin' match */
	for (o = options; o->name != NULL; o++)
		printf("\t-%s%s%s\n",o->name,o->args,o->help);
#if PATCHLEVEL == 0
	printf("\nLdb version %d.%d by Perry R. Ross.  Mail comments\n",
		VERSION,REVISION);
#else
	printf(
	"\nLdb version %d.%d (patch %d) by Perry R. Ross.  Mail\ncomments",
	VERSION,REVISION,PATCHLEVEL);
#endif
	printf("or suggestions to \"%s\".\n",AUTHOR_EMAIL);
	}
else {
	l = 0;
	printf("\t");
	for (o = options; o->name != NULL; o++) {
		if ( (l += (strlen(o->name)+strlen(o->args)+3)) > 55) {
			printf("\n\t");
			l = 0;
			}
		printf("[-%s%s] ",o->name,o->args);
		}
	printf("\n\n");
	}
}


/*----------------------------------------------------------------------
 *	remotestart -- start a game between two other people
 *
 * This function tells a user to start a game with another user.
 * Neither user needs to be the one running remotestart; although
 * this would work, -start is a more efficient way to do that.
 * Remotestart could be used to start games between opponents in
 * a tournament, or to set up a pickup game facility, where people
 * wanting to play would mail to a central machine, which would
 * pair players by some criteria (such as ability) and start a
 * game between them.
 *----------------------------------------------------------------------
 */

remotestart(u1,u2,flags,match)
char *u1, *u2;
int flags;
int match;
{
struct packet p;
char colors[4];
char mbuf[8];

p.version = LDB_VER;		/* fill in a packet */
p.timestamp = time( (long *)0);	/* give it a timestamp */
p.gameid = "REMOTESTART";	/* give it a phony gameid */
p.opcode = RSTART;		/* remote start opcode */
p.name = NULL;			/* we don't need to send a name */
p.addr = u2;			/* put opponent's address in packet */
p.comment = NULL;		/* don't have a comment */
p.comment2 = NULL;
p.seq = 1;			/* start with sequence number 1 */
p.notify = notify;		/* send notify address if any */
clearmvs(p.mvs);		/* no moves to send */
sprintf(colors,"%c%c",cr_mycolor,cr_opcolor);
p.colors = colors;
p.dir = (cr_mydir > 0) ? "up" : "down";
p.autodbl = NULL;
p.jacoby = (flags & F_JACOBY) ? "yes" : NULL;		/* jacoby rule? */
p.crawford = (flags & F_CRAWFORD) ? "yes" : NULL;	/* crawford rule? */
p.european = (flags & F_EUROPE) ? "yes" : NULL;		/* european scoring? */
p.perm = (flags & F_PERM) ? "yes" : NULL;		/* perm game? */
if (match > 0) {		/* match play */
	sprintf(mbuf,"%d",match);	/* make it a string */
	p.match = mbuf;			/* and put it in the packet */
	}
else
	p.match = NULL;			/* not a match, omit this field */
p.gameptr = NULL;		/* just in case */
TSendPacket(&p,u1);		/* send the remote start command to u1 */
}


/*----------------------------------------------------------------------
 *	optlookup -- find command line option in options table
 *
 * This function looks up a command line switch in the options table,
 * returning the index into options[].  It returns -1 if the option
 * was not found, and -2 if the option was ambiguous.  Options may be
 * abbreviated to the shortest unique substring of the option.
 *----------------------------------------------------------------------
 */

optlookup(optstr)
char *optstr;
{
int i, j, l, n;

n = 0;
if ( (l = strlen(optstr)) == 0)			/* empty option string */
	return(-1);				/* that's no good */
for (i = 0; options[i].name != NULL; i++)	/* look for arg */
	if (strncmp(options[i].name,optstr,l) == 0) {	/* found it */
		j = i;				/* remember index */
		n++;				/* count how many */
		}
if (n == 0)					/* didn't find it */
	return(-1);				/* return error */
if (n > 1)					/* found more than 1 match */
	return(-2);				/* return error */
return(j);					/* return the index */
}
