/*	control.c		9/12/91
 *
 * Copyright 1991  Perry R. Ross
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee is hereby granted, subject to the restrictions
 * detailed in the README file, which is included here by reference.
 * Any other use requires written permission from the author.  This software
 * is distributed "as is" without any warranty, including any implied
 * warranties of merchantability or fitness for a particular purpose.
 * The author shall not be liable for any damages resulting from the
 * use of this software.  By using this software, the user agrees
 * to these terms.
 */

#include "ldb.h"

/*----------------------------------------------------------------------
 *	control -- control a game
 *
 * This function is called to process the -control command line option.
 * It currently allows the following:
 *    -	The board may be inverted.  The points are still numbered the
 *	same, the board is merely drawn upside down.  This can help users
 *	who are playing a number of games simultaneously by allowing him
 *	to make all of his games move in the same direction on the screen.
 *    - The last packet sent may be resent.  This is useful when moves
 *	are lost in the mail, or if players forget whose turn it is.
 *	Ldb rejects packets that have already been received, so both
 *	players may simply resend their last packet, and ldb will display
 *	the board to the player who is supposed to move.
 *----------------------------------------------------------------------
 */

control()
{
register struct game *g;
static char *m[] = {"Invert","Resend","Get Resend","Screen Dump","Delete Game",
			"Next Game","Quit",NULL};
char buf[60], c, done;
int mod;
char pm = '\0';
static char oldmsg[] = "Your opponent's ldb does not support resend requests.";
static char gmover[] = "This game is over -- you cannot request a resend.";

if (ghead == NULL) {
	printf("You don't have any games in progress.\n");
	printf("Use the -start option to start one.\n");
	usage(0);
	ldbexit(STAT_ABORT);
	}
mod = 0;				/* init to no mods */
rc.chkpt = "no";			/* disable checkpoint in sendpkt */
FeInitialize();				/* initialize front end */
FeDrawScreen();				/* draw board outline */
done = 0;
for (g = ghead; (done < 2) && (g != NULL); g = g->next) {/* for all games */
	g->curbd = BD_CUR;		/* make sure we draw the current bd */
	FeDrawGame(g);
	FeDrawMenu(m);
	sprintf(buf,"Current state: %s",states[g->state]);
	FeMessage(buf);
	done = 0;
	GameState = STATE_CONTROL;
	while (! done) {
		c = FeMenu(m,0,0," \n\r",pm);
		pm = c;
		switch (c) {
		case 'I':			/* invert board */
			g->flags ^= F_INVERT;	/* toggle invert bit */
			FeDrawGame(g);		/* redraw the screen */
			mod++;			/* games have been modified */
			break;
		case 'R':			/* resend last packet */
			if ((g->state>=OPSTATES)&&(g->state!=ST_GAMEOVER)) {
				FeMessage("Can't resend -- it is your move.");
				break;
				}
			resendpkt(g);
			FeMessage("Last packet re-sent.");
			break;
		case 'G':
			if (g->state == ST_GAMEOVER) {
				FeMessage(gmover);
				break;
				}
			if (g->opver < 110) {
				FeMessage(oldmsg);
				break;
				}
			sendpkt(g,RESEND);
			FeMessage("Resend requested.");
			break;
		case 'S':			/* dump screen */
			FeMessage(buf);		/* restore state msg */
			FeDumpScreen("ldb_screen.dmp");	/* do the dump */
			FeMessage("Screen dumped to ldb_screen.dmp");
			break;
		case 'D':			/* delete game */
			mod++;			/* games have been modified */
			if (g->flags & F_DELETE) {	/* undelete */
				g->flags &= ~F_DELETE;
				FeMessage("Game undeleted.");
				}
			else {
				g->flags |= F_DELETE;	/* delete */
				FeMessage(
				 "Game deleted -- press D again to undelete.");
				}
			break;
		case 'Q':			/* exit ldb */
			done = 2;		/* 2 means Really done */
			break;
		case ' ':
		case '\n':
		case '\r':
			FeOnMenuItem(m,'N');	/* highlight Next Game item */
			pm = 'N';		/* remember to unhighlight */
			/* Fall through */
		case 'N':
			done = 1;	/* 1 means just done with this game */
			break;		/* go to next game */
			}
		}
	}
FeFinishSession();
while (mod) {
	printf("Save changes? [yn]: ");
	fflush(stdout);
	if ( ( (c = getchar()) == 'y') || (c == 'Y') ) {
		writegames(rc.gfile,rc.gbackup,rc.pfile);
		break;
		}
	if ( (c == 'n') || (c == 'N') ) {
		printf("Changes discarded.\n");
		break;
		}
	if ( (c != '\n') && (c != EOF) )
		while ( ( (c = getchar()) != '\n') && (c != EOF) );
	printf("Please respond with y or n.\n");
	}
}


/*----------------------------------------------------------------------
 *	recons -- reconstruct games from opponent's .ldbdata
 *
 * This function allows a game to be reconstructed when the .ldbdata
 * file has been irretrievably lost.  The opponent merely mails
 * his .ldbdata file, and we scan it for all the important information.
 * Games that are in the OPSTART or GAMEOVER states cannot be reconstructed
 * due to the lack of adequate information, but these games either have
 * not started or are already over.  If the game is in state OPSTART
 * because the next game of a match is being started, the match can
 * be rescued by starting a new match that plays up to the number of
 * points remaining in the old match.
 *
 * Before sending the .ldbdata file, your opponent should incorporate
 * any moves you sent before you lost your file.  If your move is lost
 * as well, it may be necessary for you to replay your last move, with
 * a new roll.  This should not be considered cheating, as it is unavoidable.
 * Alternatively, if you remember your last roll, you could plug the rolls
 * into your .ldbdata after the reconstruct.
 *
 * You will have to edit the mail header off the file before feeding it
 * to -reconstruct.  The first line of the file MUST be the first line
 * of the data file.
 *
 * Because data fields are frequently added to data files between
 * revisions, -reconstruct is only guaranteed to work when you
 * are using the same ldb revision as your opponent.
 *
 * If you started the game being reconstructed, and you used the
 * -myaddr argument when you started it, you will need to use
 * the -myaddr argument when you reconstruct it.
 *----------------------------------------------------------------------
 */

recons(file)
char *file;
{
FILE *fp;
char c;
int i, j;
struct mv m;
struct game *g;
struct people *p;
char *s;
char buf[80];

if ( (fp = fopen(file,"r")) == NULL) {
	fprintf(stderr,"ERROR: cannot open %s for reconstruct\n",file);
	ldbexit(STAT_ABORT);
	}
g = addgame();			/* get a game structure */
while ( (c = getc(fp)) != EOF) {
	ungetc(c,fp);			/* put char back */
	nvscan(fp,nv_gfile,g);
	if (strcmp(g->opaddr,rc.myaddr))	/* this game wasn't with me */
		continue;
	if ( (g->state == ST_OPSTART) || (g->state == ST_GAMEOVER) )
		continue;		/* can't handle these */
	printf("\n--- Reconstruct found game %s ---\n",g->gameid);
	s = g->gameid;			/* copy out gameid */
	g->gameid = "";			/* temporarily delete gameid field */
	if (findgame(s) != NULL) {
		printf("ERROR: This game already exists -- ignoring.\n");
		continue;
		}
	g->gameid = s;			/* restore gameid field */
	printf("Enter opponent address or alias: ");
	if (fgets(buf,sizeof(buf),stdin) == NULL) {
		printf("\n--- Reconstruct deleting game %s ---\n",g->gameid);
		continue;
		}
	buf[strlen(buf)-1] = '\0';		/* clobber newline */
	if ( (p = findppl(buf,P_ADDR|P_ALIAS)) == NULL) {/* not in ppl file */
		g->opaddr = save(buf);		/* save address in game */
		printf("Enter opponent name: ");
		if (fgets(buf,sizeof(buf),stdin) == NULL) {
			printf("\n--- Reconstruct deleting game %s ---\n",
				g->gameid);
			continue;
			}
		buf[strlen(buf)-1] = '\0';	/* clobber newline */
		g->opname = save(buf);		/* save opponent's name */
		newppl(g);		/* generate a new people rec */
		}
	else {				/* it was an alias */
		g->opname = save(p->name);	/* copy name from people rec */
		g->opaddr = save(p->addr);	/* address too */
		g->ppl = p;			/* store people pointer */
		}
	g->dispmsg = NULL;
	g->myaddr = save(rc.myaddr);
	for (i = 0; i < 4; i++) {
		g->blot[i] = 0;		/* clear blot array */
		m = g->mvs[i];		/* switch mvs and opmvs fields */
		g->mvs[i] = g->opmvs[i];
		g->opmvs[i] = m;
		}
	for (i = 0; i < 6; i++) {
		j = g->rolls[i];	/* switch rolls and oprolls */
		g->rolls[i] = g->oprolls[i];
		g->oprolls[i] = j;
		j = g->doubles[i];	/* switch doubles and opdoubles */
		g->doubles[i] = g->opdoubles[i];
		g->opdoubles[i] = j;
		}
	i = g->mydir;			/* switch mydir and opdir */
	g->mydir = g->opdir;
	g->opdir = i;
	i = g->mycolor;			/* switch mycolor and opcolor */
	g->mycolor = g->opcolor;
	g->opcolor = i;
	s = g->mycmt;
	g->mycmt = g->opcmt;
	g->opcmt = s;
	s = g->mycmt2;
	g->mycmt2 = g->opcmt2;
	g->opcmt2 = s;
	i = g->mcurrent[WHO_ME];		/* switch match scores */
	g->mcurrent[WHO_ME] = g->mcurrent[WHO_OPP];
	g->mcurrent[WHO_OPP] = i;
	copyboard(g->mybd,g->opbd);	/* my before bd == op's after bd */
	if (g->state < OPSTATES) {	/* I'm sending next */
		g->seq++;	/* set seq# to what opp expects in next pkt */
		copyboard(g->board,g->mybd);	/* my after bd == cur board */
		}
	else {				/* he's sending next */
		g->seq--;	/* set seq# to what we're expecting */
		copyboard(g->mybd,g->board);	/* erase his moves */
		}
	g->lastop = START;		/* has to be set to something */
	switch (g->state) {		/* invert state */
	case ST_OPTURN:
		g->state = ST_MYTURN;	/* it's my turn */
		break;
	case ST_OPACCEPT:
		g->state = ST_MYACCEPT;	/* waiting for me to accept double */
		break;
	case ST_MYTURN:
	case ST_MYMOVE:
		g->state = ST_OPTURN;	/* waiting for op to move */
		break;
	case ST_MYACCEPT:
		g->state = ST_OPACCEPT;	/* waiting for op to accept double */
		break;
		}
	printf("--- Reconstructed game %s ---\n",g->gameid);
	g = addgame();		/* get new buffer to read into */
	}
deletegame(g);			/* get rid of buffer */
writegames(rc.gfile,rc.gbackup,rc.pfile);	/* save new games */
}
