; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; File:         x-misc.el
; RCS:          $Header: $
; Description:	Miscellaneous routines for X-windows mice.  Includes
;		compatibility routines to allow the use of suntool GNU
;		Emacs mouse routines in X-windows.
; Author:       Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
; Created:      Sun Sep 16 16:06:52 1990
; Modified:     Wed Dec  5 12:32:32 1990 (Darryl Okahata) darrylo@hpsrdmo
; Language:     Emacs-Lisp
; Package:      N/A
; Status:       Experimental
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; ***** IMPORTANT *****
;;
;; UNLESS YOU TAKE STEPS TO INSURE THE FOLLOWING, EMACS V18.55 WILL
;; HANG, DUE TO A BUG.  THERE ARE NO "IFs", "ANDs", OR "BUTs"
;;
;; In order for the functions in this file to work, you must have a
;; version of GNU Emacs that:
;;
;; 1. Is compiled with X11 support.
;;
;; 2. Is compiled with the patch attached at the end of this file.  This
;;    patch fixes a small bug in V18.55, but please note that it does
;;    not fix the "hang".
;;
;; 3. Is compiled with INTERRUPT_INPUT defined (AND YOU MUST TAKE ANY
;;    OTHER STEPS NECESSARY TO INSURE THAT SIGNAL-DRIVEN INPUT WORKS).
;;    One way to check if signal-driven input works is to do a `dired'
;;    on a fairly large directory; if, while the "Reading directory ..."
;;    is displayed in the minibuffer status area, the inverse-video
;;    character cursor changes from a solid block to a rectangular
;;    outline and vice-versa as you move the mouse in and out of the
;;    Emacs X11 window, then signal-driven input works in your copy of
;;    Emacs.  If the character cursor remains a solid inverse-video
;;    block and does not change into a rectangular outline as you move
;;    the mouse in and out of the window (while Emacs is busy reading
;;    the directory), then signal-driven input does not work in your
;;    copy of Emacs, the functions contained in this file will not work,
;;    and Emacs will eventually hang if you try to use them.  The
;;    problem here is that the `x-get-mouse-event' routine assumes that
;;    signal-driven input works, and, if it waits for a mouse event,
;;    will go into an infinite loop waiting for a SIGIO signal that will
;;    never occur if signal-driven input does not work.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; ***** NOTES *****
;;
;; The routines in this file fake buffer-local mousemaps by hiding them
;; in local keymaps.
;;
;; Much of this was stolen from sun-mouse.el
;;
;; The suntool mouse bindings to the "scrollbar", "modeline", and
;; "minibuffer" do not work; they are silently mapped to the "meta" key.
;; If you want an error to be raised, change the associated number "32"
;; in `x-mouse-keyword-alist' to a large number like "9999".  Perhaps
;; this should be made into an option?  Doubleclicks are also mapped to
;; the "meta" key.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defconst x-mouse-key-prefix "\C-x\C-@"
  "The X-windows mouse key prefix used by GNU Emacs.  Note that this prefix is
compiled into Emacs; you cannot change it without recompiling Emacs."
)


(defconst x-mouse-secondary-key-prefix "\C-c\C-@"
  "The key prefix under which global and local mouse bindings are stored."
)


(defconst x-mouse-keyword-alist
  '(
    (right . 0) (middle . 1) (left . 2)
    (control . 64) (meta . 32) (shift . 16) (up . 4) (double . 32)
    (text . 0) (scrollbar . 32) (modeline . 32) (minibuffer . 32)
   )
  ""
)


(defvar x-mouse-recurse-level 0
  ""
)


(defvar x-mouse-message ""
  "Last message displayed by `message'."
)


(defvar x-mouse-error-occurred nil
  "If non-nil, an error occurred, and so the mouse up-button action
should not redisplay any messages (which would overwrite any
error messages)."
)



(defun x-get-mouse-key (keywords)
  "Map mouse keywords to a mouse-key (also called a mouse-code)"
  (let (key)
    (setq key (apply 'logior
		     (mapcar (function
			       (lambda (x)
				 (cdr (assq x x-mouse-keyword-alist))))
			     keywords)
	      )
    )
    (if (> key 127)
      (error "Error: mouse keyword sequence `%s' not supported." keywords)
    )
    key
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun x-get-mouse-bindings (keymap)
  ""
  (let (result i)
    (if (listp keymap)
      (progn
	(cdr keymap)				;; drop 'keymap atom
      )
      (progn
	(setq i 0)
	(while (< i 128)
	  (if (aref keymap i)
	    (progn
	      (setq result (cons i (aref keymap i)))
	    )
	  )
	)
	result
      )
    )
  )
)


(defun x-get-global-mousemap ()
  ""
  (let ( (keymap (current-global-map)) mousemap)
    (setq mousemap (condition-case nil
		     (lookup-key keymap x-mouse-secondary-key-prefix)
		     nil
		   )
    )
    (if (integerp mousemap)
      nil
      mousemap
    )
  )
)


(defun x-get-local-mousemap ()
  ""
  (let ( (keymap (current-local-map)) mousemap)
    (setq mousemap (condition-case nil
		     (lookup-key keymap x-mouse-secondary-key-prefix)
		     nil
		   )
    )
    (if (integerp mousemap)
      nil
      mousemap
    )
  )
)


(defun x-mousemap-to-alist (map)
  ""
  (let (i result func)
    (if (vectorp map)
      (progn
	(setq i 0)
	(while (< i 128)
	  (if (setq func (aref map i))
	    (setq result (cons (cons i func) result))
	  )
	  (setq i (1+ i))
	)
	result
      )
      (progn
	(cdr map)
      )
    )
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-region-to-code (region)
  "Returns partial mouse-code for specified REGION."
  (if (equal region 'text)
    (cdr (assq region x-mouse-keyword-alist))
    0
  )
)


(defun logtest (x y)
  "True if any bits set in X are also set in Y.
Just like the Common Lisp function of the same name."
  (not (zerop (logand x y))))


;;;
;;; I (jpeck) don't understand the utility of the next four functions
;;; ask Steven Greenbaum <froud@kestrel>
;;;
(defun mouse-mask-lookup (mask list)
  "Args MASK (a bit mask) and LIST (a list of (code . form) pairs).
Returns a list of elements of LIST whose code or'ed with MASK is non-zero."
  (let ((result nil))
    (while list
      (if (logtest mask (car (car list)))
	  (setq result (cons (car list) result)))
      (setq list (cdr list)))
    result))

(defun mouse-union (l l-unique)
  "Return the union of list of mouse (code . form) pairs L and L-UNIQUE,
where L-UNIQUE is considered to be union'ized already."
  (let ((result l-unique))
    (while l
      (let ((code-form-pair (car l)))
	(if (not (assq (car code-form-pair) result))
	    (setq result (cons code-form-pair result))))
      (setq l (cdr l)))
    result))

(defun mouse-union-first-prefered (l1 l2)
  "Return the union of lists of mouse (code . form) pairs L1 and L2,
based on the code's, with preference going to elements in L1."
  (mouse-union l2 (mouse-union l1 nil)))

(defun mouse-code-function-pairs-of-region (region)
  "Return a list of (code . function) pairs, where each code is
currently set in the REGION."
  (let ((mask (mouse-region-to-code region)))
    (mouse-union-first-prefered
     (mouse-mask-lookup mask (x-mousemap-to-alist (x-get-local-mousemap)))
     (mouse-mask-lookup mask (x-mousemap-to-alist (x-get-global-mousemap)))
     )))

(defun sm::event-bindings (region)
  "Returns an alist of (function . (mouse-list1 ... mouse-listN)) for REGION,
where each mouse-list is bound to the function in REGION."
  (let ((mouse-bindings (mouse-code-function-pairs-of-region region))
	(result nil))
    (while mouse-bindings
      (let* ((code-function-pair (car mouse-bindings))
	     (current-entry (assoc (cdr code-function-pair) result)))
	(if current-entry
	    (setcdr current-entry
		    (cons (mouse-code-to-mouse-list (car code-function-pair))
			  (cdr current-entry)))
	  (setq result (cons (cons (cdr code-function-pair)
				   (list (mouse-code-to-mouse-list
					  (car code-function-pair))))
			     result))))
      (setq mouse-bindings (cdr mouse-bindings))
      )
    result))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun make-mousemap ()
  "Returns a new mousemap."
  (make-sparse-keymap)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun copy-mousemap (mousemap)
  "Return a copy of mousemap."
  (copy-keymap mousemap)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun define-mouse (mouse-map mouse-list def)
  "Compatibility routine to allow sunwindows mouse calls to work in
X-windows:

Args MOUSE-MAP, MOUSE-LIST, DEF.  Define MOUSE-LIST in MOUSE-MAP as DEF.
MOUSE-LIST is a list of atoms specifing a mouse hit according to these rules:
  * One of these atoms specifies the active region of the definition:
	text
  * One or two or these atoms specify the button or button combination:
        left, middle, right
  * Any combination of these atoms specify the active shift keys:
        control, shift, meta
  * With a single unshifted button, you can add:
	up
    to indicate an up-click.

The following atoms are not supported:
	scrollbar
	modeline
	minibuffer
	double
"
  (let ( (mouse-key (x-get-mouse-key mouse-list)) keystring)
    (setq keystring (char-to-string mouse-key))
    (define-key mouse-map keystring def)
    (if (not (memq 'up mouse-list))
      (progn
	(setq mouse-key (x-get-mouse-key (cons 'up mouse-list)))
	(if (not (lookup-key mouse-map (char-to-string mouse-key)))
	  (progn
	    (setq keystring (char-to-string mouse-key))
	    (define-key mouse-map keystring 'x-mouse-redisplay-message)
	  )
	)
      )
    )
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun global-set-mouse (mouse-list def)
  "Give MOUSE-LIST a global definition of DEF.
See define-mouse for a description of MOUSE-EVENT-LIST and DEF.
Note that if MOUSE-LIST has a local definition in the current buffer,
that local definition will continue to shadow any global definition."
  (interactive "xMouse event: \nxDefinition: ")
  (let (
	(mouse-key (x-get-mouse-key mouse-list))
	keystring
	binding
       )
    (setq keystring (concat x-mouse-secondary-key-prefix
			    (char-to-string mouse-key)))
    (global-set-key keystring def)
    (if (not (memq 'up mouse-list))
      (progn
	(setq mouse-key (x-get-mouse-key (cons 'up mouse-list)))
	(setq keystring (concat x-mouse-secondary-key-prefix
				(char-to-string mouse-key)))
	(setq binding (key-binding keystring))
	(if (or (not binding) (integerp binding))
	  (progn
	    (global-set-key keystring 'x-mouse-redisplay-message)
	  )
	)
      )
    )
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun local-set-mouse (mouse-list def)
  "Give MOUSE-LIST a local definition of DEF.
See define-mouse for a description of MOUSE-EVENT-LIST and DEF.
Note that if MOUSE-LIST has a local definition in the current buffer,
that local definition will continue to shadow any global definition."
  (interactive "xMouse event: \nxDefinition: ")
  (let (
	(mouse-key (x-get-mouse-key mouse-list))
	keystring
	binding
       )
    (setq keystring (concat x-mouse-secondary-key-prefix
			    (char-to-string mouse-key)))
    (local-set-key keystring def)
    (if (not (memq 'up mouse-list))
      (progn
	(setq mouse-key (x-get-mouse-key (cons 'up mouse-list)))
	(setq keystring (concat x-mouse-secondary-key-prefix
				(char-to-string mouse-key)))
	(setq binding (key-binding keystring))
	(if (or (not binding) (integerp binding))
	  (progn
	    (local-set-key keystring 'x-mouse-redisplay-message)
	  )
	)
      )
    )
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun use-global-mousemap (mousemap)
  "Selects MOUSEMAP as the global mousemap."
  (global-set-key x-mouse-secondary-key-prefix mousemap)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun use-local-mousemap (mousemap)
  "Selects MOUSEMAP as the local mousemap.
nil for MOUSEMAP means no local mousemap."
  (local-set-key x-mouse-secondary-key-prefix mousemap)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun move-to-loc (x y)
  "Move cursor to window location X, Y.
Handles wrapped and horizontally scrolled lines correctly."
  (move-to-window-line y)
  ;; window-line-end expects this to return the window column it moved to.
  (let ((cc (current-column))
	(nc (move-to-column
	     (if (zerop (window-hscroll))
		 (+ (current-column)
		    (min (- (window-width) 2)	; To stay on the line.
			 x))
	       (+ (window-hscroll) -1
		  (min (1- (window-width))	; To stay on the line.
		       x))))))
    (- nc cc)
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun x-get-mouse-window-info (xy)
  "Find Emacs window the mouse is in."
  (let (
	(start-w (selected-window))
	(done nil)
	(w (selected-window))
	(rel-coordinate nil)
       )
    (while (and (not done)
		(null (setq rel-coordinate
			    (coordinates-in-window-p xy w))))
      (setq w (next-window w))
      (if (eq w start-w)
	  (setq done t)
      )
    )
    (list rel-coordinate w)	;; Return '((x y) window)
  )
)


(defun x-process-mouse-event ()
  ""
  (let (event char command window-info window xy
	      (old-window (selected-window))
       )
    (setq event (x-get-mouse-event t))
    (setq char (car event))
    (setq xy (car (cdr event)))
    (setq window-info (x-get-mouse-window-info xy))
    (setq window (car (cdr window-info)))
    (select-window window)
    (setq command (key-binding (concat x-mouse-secondary-key-prefix
				       (char-to-string char))))
    (if (and command (symbolp command))
      (progn
	;; found a function to call using the sun-mouse semantics
	(select-window old-window)
	(funcall command window
		 (car (car window-info))
		 (car (cdr (car window-info))))
      )
      (progn
	;; look in the old-style X11 mouse map for a function
	(setq command (lookup-key mouse-map
				  (char-to-string char)))
	(select-window old-window)
	(if (and command (symbolp command))
	  (progn		;;; found one
	    ;; call it using the old style semantics
	    (funcall command xy)
	  )
	  (progn		;; did not find one
	    (if (= (logand char 4) 0)
	      (ding)		;; ding on not buttonrelease
	    )
	  )
	)
      )
    )
    (if (and (not (eq (logand char (cdr (assq 'up x-mouse-keyword-alist))) 0))
	     (not (eq command 'x-mouse-redisplay-message))
	)
      (progn
	;;
	;; This is an up-event
	;;
	(x-mouse-redisplay-message window
				   (car (car window-info))
				   (car (cdr (car window-info))))
      )
    )
  )
)


(defun x-process-mouse () 
  "Process all queued mouse events.
This code was stolen from x-flush-mouse-queue.
"
  ;; A mouse event causes a special character sequence to be given
  ;; as keyboard input.  That runs this function, which process all
  ;; queued mouse events and returns.
  (interactive)
  (let ()
    (setq x-mouse-recurse-level (1+ x-mouse-recurse-level))
    (unwind-protect
      (while (> (x-mouse-events) 0)
	(x-process-mouse-event)
	(and (boundp 'x-process-mouse-hook)
	     (symbol-value 'x-process-mouse-hook)
	     (funcall x-process-mouse-hook x-mouse-pos x-mouse-item))
      )
      (setq x-mouse-recurse-level (1- x-mouse-recurse-level))
    )
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun x-mouse-message (mess &rest args)
  ""
  (let (msg)
    (setq msg (apply 'format mess args))
    (setq x-mouse-message msg)
    (funcall x-mouse-message-old "%s" msg)
    (if (> x-mouse-recurse-level 0)
      (progn
	(discard-input)
      )
    )
    msg
  )
)


(defun x-mouse-redisplay-message (win x y)
  ""
  (let ()
    (if (not x-mouse-error-occurred)
      (progn
	(funcall x-mouse-message-old "%s" x-mouse-message)
      )
    )
    (setq x-mouse-error-occurred nil)
  )
)

;;
;; We replace the existing definition of `message' because we need to
;; "refresh" the minibuffer status line after a mouse up-button event,
;; which gets cleared by that event (it would be nice to see any
;; messages produced by functions activated by the down-button event).
;;

(if (or (not (boundp 'x-mouse-message-old))
	(not x-mouse-message-old))
  (progn
    (setq x-mouse-message-old (symbol-function 'message))
    (fset 'message 'x-mouse-message)
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun x-mouse-error (mess &rest args)
  ""
  (let (msg)
    (setq msg (apply 'format mess args))
    (setq x-mouse-message msg)
    (funcall x-mouse-error-old "%s" msg)
    (setq x-mouse-error-occurred t)
    (discard-input)
  )
)


;;
;; We need to keep track of any errors that occur, as the mouse
;; up-button event will clear any error message in the minibuffer.  We
;; then redisplay the error message after the up-button event.
;; Unfortunately, any error mechanism that bypasses `error' will not
;; have its error message redisplayed.
;;

(if (or (not (boundp 'x-mouse-error-old))
	(not x-mouse-error-old))
  (progn
    (setq x-mouse-error-old (symbol-function 'error))
    (fset 'error 'x-mouse-error)
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(global-set-key x-mouse-key-prefix 'x-process-mouse)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  This patch needs to be applied to the GNU Emacs V18.55 sources to
;;  fix a bug during mouse event handling.  If this patch is not
;;  applied, the functions contained in this file will not work.
;;
;;*** ../../dist-18.55/src/x11fns.c	Sun Aug 13 12:54:21 1989
;;--- x11fns.c	Fri Sep 21 22:20:12 1990
;;***************
;;*** 620,625
;;  		XXm_queue_num--;
;;  		com_letter = 3-(event.xbutton.button & 3);
;;  		key_mask = (event.xbutton.state & 15) << 4;
;;  		com_letter |= key_mask;
;;  		if (event.type == ButtonRelease)
;;  			com_letter |= 0x04;
;;
;;--- 683,694 -----
;;  		XXm_queue_num--;
;;  		com_letter = 3-(event.xbutton.button & 3);
;;  		key_mask = (event.xbutton.state & 15) << 4;
;;+ 		/* Report meta in 2 bit, not in 8 bit.  */
;;+ 		if (key_mask & 0x80)
;;+ 		  {
;;+ 		    key_mask |= 0x20;
;;+ 		    key_mask &= ~0x80;
;;+ 		  }
;;  		com_letter |= key_mask;
;;  		if (event.type == ButtonRelease)
;;  			com_letter |= 0x04;
;;***************
;;*** 634,640
;;  		Vx_mouse_pos = Fcons (tempx, Fcons (tempy, Qnil));
;;  		XSET (tempx, Lisp_Int, event.xbutton.x_root);
;;  		XSET (tempy, Lisp_Int, event.xbutton.y_root);
;;! 		Vx_mouse_abs_pos = Fcond (tempx, Fcons (tempy, Qnil));
;;  		return Fcons (com_letter, Fcons (Vx_mouse_pos, Qnil));
;;  	}
;;  	return Qnil;
;;
;;--- 703,709 -----
;;  		Vx_mouse_pos = Fcons (tempx, Fcons (tempy, Qnil));
;;  		XSET (tempx, Lisp_Int, event.xbutton.x_root);
;;  		XSET (tempy, Lisp_Int, event.xbutton.y_root);
;;! 		Vx_mouse_abs_pos = Fcons (tempx, Fcons (tempy, Qnil));
;;  		return Fcons (com_letter, Fcons (Vx_mouse_pos, Qnil));
;;  	}
;;  	return Qnil;
;;

(provide 'x-misc)

;; rnotes-> Here is Darryl's second pass at x-misc. "It has some bug fixes."
;; rnotes-> 
;; rnotes-> I have not run this myself -- please let me know how robust it turns out to
;; rnotes-> be. 
;; rnotes-> 
;; rnotes-> I would still like to see someone tackle a more conventional x binding --
;; rnotes-> something along the line of the example below taken from non-empire code
;; rnotes-> when I ran on X. 
;; rnotes-> 
;; rnotes-> ;;->(defun end-quick-browse ()
;; rnotes-> ;;->  "De-Activates the quick browse key mappings."
;; rnotes-> ;;->  (interactive)
;; rnotes-> ;;->  (cond ((eq window-system 'x)
;; rnotes-> ;;->	 (define-key mouse-map x-button-right 'x-mouse-select)
;; rnotes-> ;;->	 (define-key mouse-map x-button-left 'x-mouse-set-mark)
;; rnotes-> ;;->	 (define-key mouse-map x-button-middle 'x-mouse-set-point)
;; rnotes-> ;;->	 )
;; rnotes-> ;;->	((null window-system)
;; rnotes-> ;;->	 (global-set-mouse '(text        left)	'mouse-drag-move-point)
;; rnotes-> ;;->	 (global-set-mouse '(text	middle)	'mouse-set-mark-and-stuff)
;; rnotes-> ;;->	 (global-set-mouse '(text	right)	'emacs-menu-eval)
;; rnotes-> ;;->	 )
;; rnotes-> ;;->	(t (error "Unrecognized window system for quick browsal"))))
;; rnotes-> ;;->
;; rnotes-> ;;->(defun x-scroll-window (arg)
;; rnotes-> ;;->  (if (< (car arg) (/ (window-width) 2))
;; rnotes-> ;;->      (scroll-down (/ (window-height) 2))
;; rnotes-> ;;->    (scroll-up (/ (window-height) 2))))
;; rnotes-> ;;->
;; rnotes-> ;;->(defun x-mouse-select (arg)
;; rnotes-> ;;->  "Select Emacs window the mouse is on."
;; rnotes-> ;;->  (let ((start-w (selected-window))
;; rnotes-> ;;->	(done nil)
;; rnotes-> ;;->	(w (selected-window))
;; rnotes-> ;;->	(rel-coordinate nil)
;; rnotes-> ;;->	(arg2 (list (car arg) (- (car (cdr arg)) 1)))
;; rnotes-> ;;->	)
;; rnotes-> ;;->	;;(message "looking for select %s" arg) (sit-for 1)
;; rnotes-> ;;->    (while (and (not done)
;; rnotes-> ;;->		(null (setq rel-coordinate
;; rnotes-> ;;->			    (coordinates-in-window-p arg w))))
;; rnotes-> ;;->      (setq w (next-window w))
;; rnotes-> ;;->      (if (eq w start-w)
;; rnotes-> ;;->	  (setq done t)))
;; rnotes-> ;;->    (if rel-coordinate
;; rnotes-> ;;->	(select-window w)
;; rnotes-> ;;->      (progn;; scroll instead
;; rnotes-> ;;->	;;(message "looking for scroll %s" arg2) (sit-for 1)
;; rnotes-> ;;->	(setq w (selected-window))
;; rnotes-> ;;->	(setq done ())
;; rnotes-> ;;->	(while (and (not done)
;; rnotes-> ;;->		    (null (setq rel-coordinate
;; rnotes-> ;;->				(coordinates-in-window-p arg2 w))))
;; rnotes-> ;;->	  (setq w (next-window w))
;; rnotes-> ;;->	  (if (eq w start-w)
;; rnotes-> ;;->	      (setq done t)))
;; rnotes-> ;;->	;;(message "found rel %s" rel-coordinate) (sit-for 2)
;; rnotes-> ;;->	(if rel-coordinate
;; rnotes-> ;;->	    (progn
;; rnotes-> ;;->	      (select-window w)
;; rnotes-> ;;->	      (if (< (car rel-coordinate) (/ (window-width) 2))
;; rnotes-> ;;->		  (scroll-down (/ (window-height) 2))
;; rnotes-> ;;->		(scroll-up (/ (window-height) 2)))))
;; rnotes-> ;;->	(setq rel-coordinate ())
;; rnotes-> ;;->	))
;; rnotes-> ;;->    rel-coordinate))
;; rnotes-> ;;->
;; rnotes-> ;;->(defun x-mouse-find-more (arg)
;; rnotes-> ;;->  ""
;; rnotes-> ;;->  (find-tag last-tag t))
