;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-survey.el -- experimental survey command for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright Robert Forsman, et.al. (GNU General Public License)
;;
;; Author          : Robert Forsman <thoth@manatee.cis.ufl.edu>
;; Created On      : Mon Jan  7 19:46:48 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:38 1991
;; Update Count    : 10
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; 9-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Fri Feb  8 09:45:48 1991 #8 (Lynn Slater x2048)
;;    added a mode map
;; PURPOSE
;;      I missed the surveying capability of ve, so I wrote emp-survey.el.
;; It allows you to temporarily replace designations on the map with numeric
;; values corresponding to commodity levels in the sectors.  Resource levels
;; are not currently stored so its usefulness is limited.
;;
;; BUGS
;;	This code is NOT integrated with the rest of GEET.  You will
;;  get incorrect map designations when using some functions (I don't
;;  know which ones yet) if you have your survey commodity set to
;;  anything other than "des".  Otherwise it should be transparent.
;;  Use M-X map-empire to refresh the map when this happens.
;;  Also tell us approximately what you were doing when this happened
;;  so we can try to nail the offender.
;;      empire-set-survey-commodity should do completion and
;;  empire-set-survey-scale should do range checking.
;; 
;; TABLE OF CONTENTS
;;   map-empire -- Shows all that is known about the world map.  The map has special
;;   empire-reset-survey-all -- This function resets the survey designations for the entire known
;;   empire-calc-survey-all -- This command performs an empire-calc-survey-region on the entire known
;;   empire-set-survey-commodity -- Set the survey commodity to the offset corresponding to the string COMM.
;;   empire-set-survey-scale -- Set the survey commodity scale to SCALE.  The optional SMART-REDRAW
;;   empire-toggle-survey -- Makes the survey active or inactive
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-survey)
(require 'emp-db)

;;;
;;; constants
;;;

(defvar empire-survey-on nil
  "Is the survey view activated?" )

(defvar map-surveyed-commodity (position-of 'des)
  "Which commodity are we surveying now.  Defaults to the designation")

(defconst empire-pos-survey-offset (position-of 'survey)
  "")

(defconst map-survey-fallback-commodity (position-of 'des)
  "Which commodity to use when we don't know what the primary commodity
 level is.  Defaults to the designation.  I would recommend against
 changing this to anything else")

(defvar map-survey-scale 10
  "how much of the current surveyed commodity is considered `1'")

(defconst map-nonscalable-commodities (mapcar '(lambda (a)
						 (dynamic-position-of a))
					      '( des sdes ))
  "Which commodities is it unreasonable to scale (all the non-numeric ones).")

(defconst survey-digits "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
  "The digits for the scaled survey commodity")

(defconst survey-digits-length (length survey-digits)
  "is this really a good optimization?")


;;;
;;; functions
;;;

(autoload 'empire-parse-condition "emp-nsc" t)

(defun map-representation (x y comm scale)
  (let ( ( val (recall x y comm) ) )
    (if val
	(if (memq comm map-nonscalable-commodities)
	    ;; if the commodity is non-scalable (non-numeric)
	    ;; then display it as is (hope it's a character)
	    val
	  ;; otherwise
	  (let (scaled)
	    ;; divide by the scale
	    (setq scaled  (/ val scale) )
	    (if (> survey-digits-length scaled)
		;; and if it's "on the scale" print the
		;; appropriate digit, ve style
		(substring survey-digits scaled (1+ scaled))
	      ;; otherwise it's out of range
	      "$")
	    )
	  ;; we had a nil commodity value.  Fallback.
	  )
      (recall x y map-survey-fallback-commodity)
      )
    )
  )

;;  Well, since this is a copy of a function in another file I'll feel
;; free to incorporate any non-destructive mostly-backward-compatible
;; changes I feel like.  If you have any problems with the behavior,
;; report it, and reload emp-db.el to disable survey and get the old
;; function back.
(defun map-empire ()
  "Shows all that is known about the world map.  The map has special
keybindings and can also support mouse clicks.  Do a mode-help from the map
window to learn what the map can do for you."
  (interactive)
  (setq pending-map-redesignations nil)
  (let* (
	 (cb (current-buffer))
	 (x empire-x-start)
	 (y empire-y-start)
	 (bw (if empire-map-buffer (get-buffer-window empire-map-buffer)))
	 (old-start (if bw (window-start bw)))
	 (old-hscroll (if bw (window-hscroll bw) ))
	 row
	 data
	 xo)
    (with-output-to-temp-buffer "*Map*"
      (set-buffer "*Map*")
      (setq empire-map-buffer (current-buffer))
      (setq highlighted-map nil)	; it's not highlighted any more!
      (if overlay-arrow-position
	  (set-marker overlay-arrow-position 0 (current-buffer)))
      (empire-map-mode)

      (princ-map-empire-x-line)
      ;;(sit-for 0)

      (while (< y empire-y-stop)
	(if empire-interaction-verbosity
	    (message "Mapping %s ..." y))
	(if (= (% y 2) 0) 
	    ;; make x be even
	    (setq x  (+ empire-x-start (abs (% empire-x-start 2))))
	  ;; else make x be odd
	  (setq x (+ empire-x-start (- 1 (abs (% empire-x-start 2)))))
	  )
	(insert (if (= x empire-x-start)
		    (format "%s\t" y);; no extra space
		  (format "%s\t " y);; extra space
		  ))
	(if (setq row (aref empire-sects (+ y empire-height)))
	    (progn
	      (setq xo (/ (+ x empire-width) 2))
	      (while (<= x empire-x-stop)
		;;(insert (or (recall-macro x y (position-of 'des)) " ") " ")
		(setq data (aref row xo))
		(insert (if data (if (and empire-survey-on
					  (< empire-pos-survey-offset
					     (length data)))
				     (progn
				       (if (not (setq val (aref data empire-pos-survey-offset)))
					   (aref data map-survey-fallback-commodity)
					 val)
				       )
				   (aref data map-survey-fallback-commodity)
				   ) " ") " ")
		(setq x (+ 2 x))
		(setq xo (1+ xo)))))
	(insert "\n")
	(setq y (1+ y))
	;;(sit-for 0)
	)

      (princ-map-empire-x-line)

      (if empire-development-mode	; explain this?
	  (empire-display-dev-sectors-if-necessary))
      )

    ;; If the map is visible somewhere, get back to it
    (if bw
	;; try to restore map view to what it was before
	(progn
	  (set-window-start bw old-start)
	  (set-window-hscroll bw old-hscroll)
	  ))

    ;;
    ;; Move point to the last map location, or to the center of the map,
    ;; if there is no "last map location".  
    ;;
    (setq x empire-current-map-x
	  y empire-current-map-y)
  
    (if (or (not x) (not y))		; handle center of map case
	(progn
	  (setq x (+ (/ (- empire-x-stop empire-x-start) 2) empire-x-start))
	  (setq y (+ (/ (- empire-y-stop empire-y-start) 2) empire-y-start))
	  (if (= 0 (% y 2))
	      (setq x  (+ x (abs (% x 2))));; make x be even
	    (setq x (+ x (- 1 (abs (% x 2)))));; else make x be odd
	    )
	  (move-to-in-empire-map x y)
	  (set-marker overlay-arrow-position (point) (current-buffer))
	  (show-map (point))
	  (point-wysiwyg)
	  )
      )

    ;;If the last map location is not visible, move as close to it as possible.
    (move-to-in-empire-map x y)		; this takes care of "closest" visible
					; sector positioning

    (set-buffer cb)
    )
  (message ""))


(defun empire-reset-survey-region (start-x end-x start-y end-y
				   &optional fcn)
  "reset the survey designation of every sector in the range
START-X:END-X,START-Y:END-Y to the 'des only if (FCN x y) returns
true for that sector.  If FCN is nil then it is assumed true.
  This will clear the results of a previous survey and the use of
FCN can make it quite selective.  See (empire-parse-condition) to
easily build FCNs."
  (setq start-x (normalize-x start-x)
	end-x   (normalize-x end-x)
	start-y (normalize-y start-y)
	end-y   (normalize-y end-y) )
  (let ( (y start-y)
	 x
	 )
    (while (<= y end-y)
      (setq x start-x)
      (if (not (zerop (- (% x 2) (% y 2) ) ) )
	  (setq x (1+ x))
	)
      (while (<= x end-x)
	(if (recall x y (position-of 'mob))
	    (if (or (not fcn)
		    (funcall fcn x y)
		    )
		(record x y empire-pos-survey-offset
			(recall x y (position-of 'des))
			)
	      )
	  )
	(setq x (normalize-x (+ 2 x)) )
	)
      (setq y (1+ y))
      )
    )
  )

(defun empire-reset-survey-all ()
  "This function resets the survey designations for the entire known
map.  Use it when you want to indiscriminately clear the survey."
  (interactive)
  (empire-reset-survey-region empire-x-start empire-x-stop
			      empire-y-start empire-y-stop)
  )

(defun empire-calc-survey-region (start-x end-x start-y end-y comm scale
				   &optional fcn)
  "Set the survey designation of every sector in the range
START-X:END-X,START-Y:END-Y according to COMM only if (FCN x y) returns
true for that sector.  If FCN is nil then it is assumed true.
  This is the basic survey function.  empire-calc-survey-all is less
general but more useful."
  (setq start-x (normalize-x start-x)
	end-x   (normalize-x end-x)
	start-y (normalize-y start-y)
	end-y   (normalize-y end-y) )
  (let ( (y start-y)
	 x
	 )
    (while (<= y end-y)
      (setq x start-x)
      (if (not (zerop (- (% x 2) (% y 2) ) ) )
	  (setq x (1+ x))
	)
      (while (<= x end-x)
	(if (recall x y (position-of 'mob))
	    (record x y empire-pos-survey-offset
		    (if (or (not fcn)
			    (funcall fcn x y)
			    )
			(map-representation x y comm scale )
		      (recall x y (position-of 'des))
		      )
		    )
	  )
	(setq x (normalize-x (+ 2 x)) )
	)
      (setq y (1+ y))
      )
    )
  )

(defun empire-calc-survey-all ( comm scale &optional fcn)
  "This command performs an empire-calc-survey-region on the entire known
map using FCN as a selector.  One of my favorites is

  (empire-calc-surve-region-all (position-of 'mob) 13
		(empire-parse-condition \"?des=+&eff>99\") ) "
  (interactive
   (list ( dynamic-position-of
	   (read (prompt-for-item empire-survey-commodity-list)))
	 (empire-read-number-from-minibuffer "scale? " "10")
	 (empire-parse-condition (read-string "restrictions: " "?"))
	 )
   )
  (empire-calc-survey-region empire-x-start empire-x-stop
			      empire-y-start empire-y-stop
			      comm scale fcn)
  )
      

(defun empire-set-survey-commodity (comm &optional smart-redraw)
  "Set the survey commodity to the offset corresponding to the string COMM.
 The optional SMART-REDRAW parameter controls whether it supresses redraw
 when the commodity is the same as the old one.

 This does not check to see if the user has entered an invalid commodity"
  (interactive "scommodity: ")
  ;; dammit, I need to copy the completion code from some other function
  (setq empire-survey-on t)
  (let ( (new-comm (dynamic-position-of (read comm)))
	 changed)
    (setq changed (not (eq new-comm map-surveyed-commodity)))
    (if (and (< new-comm (1- empire-pos-count))
	     new-comm)
	(progn
	  (setq map-surveyed-commodity new-comm)
	  (if (or (not smart-redraw) changed)
	      (progn
		( empire-calc-survey-all
		  map-surveyed-commodity map-survey-scale)
		(map-empire)
		)
	    )
	  )
      (message "nonsense")
      nil
      )
    )
  map-surveyed-commodity
  )

(defun empire-set-survey-scale (scale &optional smart-redraw)
  "Set the survey commodity scale to SCALE.  The optional SMART-REDRAW
 parameter controls whether it supresses redraw when the scale is the
 same as the old one."
  (interactive "nscale: ")

  (let (changed)
    (setq changed (not (eq scale map-survey-scale)))
    (setq map-survey-scale scale)
    (if (or (not smart-redraw) changed)
	(progn
	  (empire-calc-survey-all map-surveyed-commodity map-survey-scale)
	  (map-empire)
	  )
      )
    )
  map-survey-scale
  )


(defun empire-toggle-survey ()
  "Makes the survey active or inactive"
  (interactive)
  (setq empire-survey-on (not empire-survey-on))
  (map-empire)
  )

