;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-ship.el -- Ship basic routines for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright Robert Forsman, et.al. (GNU General Public License)
;; 
;; Author          : Robert Forsman <thoth@cis.ufl.edu>
;; Created On      : Sat Jan  5 12:05:20 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:17:01 1991
;; Update Count    : 27
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	Ship storage fundemental calls.
;;   See the check-ship-food in emp-anal.el
;;   Note: A lot of the code was stolen from emp-db.el and the plane code
;;         therein
;; HISTORY
;; 25-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Thu Jan 24 19:34:22 1991 #15 (Lynn Slater x2048)
;;    split out into emp-const
;; 24-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Wed Jan  9 13:19:34 1991 #13 (Lynn Slater x2048)
;;    replaced whip access macros with onew from ken Stevens
;; 8-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Tue Jan  8 19:38:49 1991 #11 (Lynn Slater x2048)
;;    made sail-wait-for-condition-all bind values
;; 5-Jan-1991		Lynn Slater x2048	
;;    Cleaned up/edited original file from author.
;;    added ship food check and the mapship routines
;; TABLE OF CONTENTS
;;   empire-read-ships -- Reads an empire ship report.  If given an arg, will forget ALL ships
;;   empire-read-cargos -- Reads an empire cargo report.  If given an arg, will forget ALL ships
;;   empire-map-forget-ships -- Takes all ships off of the empire map and thus forgets obsolete data.
;; 
;;
;; TODO: update much existing code to grok ships as first-class entities.
;; Other ideas as people bitch to me.
;;
;; BUGS: none known.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-ship)

(require 'emp-db)

(defvar empire-ships nil
  "An associative array for ships. Its key is shipnum and its single value 
is a vector containing the relevent ship info. The data in the vector
corresponds to the positions named in empire-ship-pos.")
(put 'empire-ships 'empire-system t)	; Cause ships to be dumped

(put 'empire-ship-pos 'empire-system t)	; Causes var to be dumped

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ship data access 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro ship-pos-of (name)		; lookup at compile time
  (if (and (listp name) (eq (car name) 'quote))
      (setq name (car (cdr name))))
  (if (not (symbolp name))
      (error "Compile time ship-pos-of used for run time value")
    (let ((offset (cdr (assq name empire-ship-pos))))
      (if (not offset)
	  (error "ship-pos-of used for value not in empire-ship-pos: %s" name)
	offset))))

(defun dynamic-ship-pos-of (name)	; lookup at run time
  (cdr (assq name empire-ship-pos)))

;;; handy constant to have around
(defconst empire-ship-pos-count
  (let ((max 0))
    (mapcar '(lambda (pair)
	       (if (> (or (cdr pair) 0) max)
		   (setq max (cdr pair))))
	    empire-ship-pos)
    (+ max 2))
  "Size of vector to allocate to store the ship info array")

(mapcar 
 '(lambda (pair)
    ;; scan list and replace nils with a dummy bit-bucket element at the 
    ;; end of each vector. This will make 'record faster at the cost of some
    ;; memory. 
    (if (not (cdr pair)) (setcdr pair (1- empire-ship-pos-count)))
    pair)
 empire-ship-pos
)

(defmacro empire-ship-pos-var-let (&rest let-body)
  "Introduces a let where each var in empire-ship-pos is bound and has nil
as its value."
  (cons 'let (cons
	(mapcar '(lambda (pair) (list (car pair)))
		empire-ship-pos)
	let-body)))

(defmacro empire-ship-pos-var-let-set (ship-values &rest let-body)
  "Introduces a let where each var in empire-ship-pos is bound and has its value
taken from values."
  (cons 'let (cons
	      (mapcar '(lambda (pair) (list (car pair)
					    (list 'aref ship-values (cdr pair))))
		      empire-ship-pos)
	      let-body)))

(defmacro empire-ship-pos-var-let-set-from-ship (number &rest let-body)
  "Introduces a let where each var in empire-ship-pos is bound and has its value
taken from values."
  (cons 'let* (cons
	       (cons
		(` (ship-values (cdr (assq (, number)  empire-ships))))
		(mapcar '(lambda (pair) (list (car pair)
					      (list 'aref 'ship-values (cdr pair))))
			empire-ship-pos))
	       let-body)))

;;(empire-ship-pos-var-let-set (cdar empire-ships) (message "type is %s" type))
;;(empire-ship-pos-var-let-set-from-ship 127 (message "type is %s" type))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ship data access macros curtosy of Ken Stevens
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro ship-lookup (num ship-list)
  "Returns ship number NUM in list SHIP-LIST"
  (list 'assq num ship-list))

(defmacro shipdata-recall (fact ship-data)
  "if you already have a ship, find data from that one!"
  (` (aref (cdr (, ship-data)) (ship-pos-of (, fact)))))

;;; be sure ship-lookup and shipdata-recall are defined!
(defmacro ship-recall (number fact)
  (` (let ((ship (ship-lookup (, number) empire-ships)))
       (when ship (shipdata-recall (, fact) ship)))))

(defun ship-record (number fact value)
  (let* ((offset (if (numberp fact)
		     fact
		   (dynamic-ship-pos-of fact)))
	 (data (assq number empire-ships)))
    (aset (cdr data) offset value)
    ))

(defmacro dynamic-shipdata-recall (fact ship-data)
  (` (aref (cdr (, ship-data)) (dynamic-ship-pos-of (, fact)))))

(defmacro dynamic-ship-recall (number fact)
  (` (let ((ship (ship-lookup (, number) empire-ships)))
       (when ship (dynamic-shipdata-recall (, fact) ship)))))

(defun ship-storedp (number)
  (assq number empire-ships))

(defun mapships (fcn &optional label)
  "Calls FCN on each ships, ours or others, and passes the ship number"
  (let ((ships empire-ships))
    (while ships
      (funcall fcn (caar ships))
      (setq ships (cdr ships)))))

(defun mapships-with-vars-bounded (fcn &optional label)
  "Calls FCN on each ships, ours or others, and passes the ship number
   However, each stored ship data variable is bounded to a variable of the
same name. This is not particularly fast, but it can make the ship access
function very easy to code."
  (let ((ships empire-ships))
    (while ships
      (empire-ship-pos-var-let-set (cdar ships)
				   (funcall fcn (caar ships))
				   )
      (setq ships (cdr ships)))))

(defun mapships-ours (fcn &optional label)
  "Calls FCN on each ship owned by us and passes the ship number"
  (let ((ships empire-ships))
    (while ships
      (if (equal empire-nation-number (aref (cdar ships) (ship-pos-of 'owner)))
	  (funcall fcn (caar ships)))
      (setq ships (cdr ships)))))

(defun mapships-ours-with-vars-bounded (fcn &optional label)
  "Calls FCN on each ship owned by us and passes the ship number
   However, each stored ship data variable is bounded to a variable of the
same name. This is not particularly fast, but it can make the ship access
function very easy to code."
  (let ((ships empire-ships))
    (while ships
      (if (equal empire-nation-number (aref (cdar ships) (ship-pos-of 'owner)))
	  (empire-ship-pos-var-let-set (cdar ships)
				       (funcall fcn (caar ships))
				       ))
      (setq ships (cdr ships)))))

;;(mapships '(lambda (num) (message "Ship %s" num)))
;;(mapships-ours-with-vars-bounded '(lambda (num) (message "Ship %s is %s" num type)))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Utilities
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun duplicate-string (str num)
  "This function makes a new string with STR copies NUM times."
  (let ( (rval "") )
    (while (> num 0)
      (setq rval (concat rval str))
      (setq num (1- num))
      )
    rval
))

(defun read-buffer-regexp-match (func num)
  " extracts the string matched by the regexp in \(\) block number NUM
 using function FUNC.
 Example: (read-buffer-regexp-match 'buffer-substring 2)"
  (funcall func (match-beginning num) (match-end num))
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Extraction constants
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Key fields for empire-ship-pos when scanning ship and cargo.
;; It must be the index in the regexp of the ship number
(defconst empire-ship-key-field 1)
(defconst empire-cargo-key-field 1)


;;; I need some regexps to extract the ship and cargo data
;;; and some structures to tell me which pieces go where.
(defconst empire-ship-regexp-1 " *\\([0-9]+\\) \\(................\\) +\\(-?[0-9]+\\),\\(....\\)  \\(.\\) \\(....\\)%")
(defconst empire-ship-regexp-pos-1
  (list
   ;; regexp   info vector      extraction
   ;; index	index		 function
   (list 2 (ship-pos-of 'type)	'buffer-substring)
   (list 3 (ship-pos-of 'x)	'read-buffer-num)
   (list 4 (ship-pos-of 'y)	'read-buffer-num)
   (list 5 (ship-pos-of 'flt)	'buffer-substring)
   (list 6 (ship-pos-of 'eff)	'read-buffer-num)
))
(defconst empire-ship-regexp-2 (duplicate-string " +\\(-?[0-9]+\\)" 7))
(defconst empire-ship-regexp-pos-2
  (list
   (list 1 (ship-pos-of 'civ) 'read-buffer-num)
   (list 2 (ship-pos-of 'mil) 'read-buffer-num)
   (list 3 (ship-pos-of 'uw) 'read-buffer-num)
   (list 4 (ship-pos-of 'food) 'read-buffer-num)
   (list 5 (ship-pos-of 'pln) 'read-buffer-num)
   (list 6 (ship-pos-of 'mob) 'read-buffer-num)
   (list 7 (ship-pos-of 'tech) 'read-buffer-num)
   )
)

(defconst empire-cargo-regexp-1 " *\\([0-9]+\\) \\(................\\) +\\(-?[0-9]+\\),\\(....\\)  \\(.\\)\\(....\\)% +\\([0-9]+\\)")
(defconst empire-cargo-regexp-pos-1
  (append empire-ship-regexp-pos-1
	  (list (list 7 ( ship-pos-of 'shell) 'read-buffer-num ))))
(defconst empire-cargo-regexp-2 (duplicate-string " +\\(-?[0-9]+\\)" 9))
(defconst empire-cargo-regexp-pos-2
  (list
   (list 1 (ship-pos-of 'gun) 'read-buffer-num)
   (list 2 (ship-pos-of 'pet) 'read-buffer-num)
   (list 3 (ship-pos-of 'iron) 'read-buffer-num)
   (list 4 (ship-pos-of 'dust) 'read-buffer-num)
   (list 5 (ship-pos-of 'bar) 'read-buffer-num)
   (list 6 (ship-pos-of 'oil) 'read-buffer-num)
   (list 7 (ship-pos-of 'lcm) 'read-buffer-num)
   (list 8 (ship-pos-of 'hcm) 'read-buffer-num)
   (list 9 (ship-pos-of 'rad) 'read-buffer-num)
   )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ship data extraction
;;   These procedures are really strange and they go to the metal on the
;;   data structures. 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-read-ships (&optional arg)
  "Reads an empire ship report.  If given an arg, will forget ALL ships
   first."
  (interactive "P")
  (if arg
      (setq empire-ships nil))
  ;; my right eye teeth for a perl regexp
  (if (re-search-forward "shp# *ship type *x,y *flt *eff *civ *mil *uw *food *pln *mob *tech" nil t)
      (let ( key
	     data
	     hit )
	(forward-line 1)
	(while (looking-at empire-ship-regexp-1)
	  (setq key (read-buffer-regexp-match 'read-buffer-num 1))

	  (if (setq hit (assoc key empire-ships))
	      (setq data (cdr hit))
	    (setq empire-ships (cons (setq hit (cons key (setq data (make-vector empire-ship-pos-count nil)))) empire-ships)))
	  (do ( (fields empire-ship-regexp-pos-1 (cdr fields)) )
	      ( (null fields) nil )
	    (aset data (cadar fields)
		  (read-buffer-regexp-match (caddar fields) (caar fields)))
	    )
	  (aset data (ship-pos-of 'owner) empire-nation-number)
	  ;;(message "%s %d %d" (read-buffer-regexp-match 2) (match-beginning 2) (match-end 2))

	  (goto-char (match-end 0))

	  (looking-at empire-ship-regexp-2)
	  (do ( (fields empire-ship-regexp-pos-2 (cdr fields)) )
	      ( (null fields) nil)
	    (aset data (cadar fields)
		  (read-buffer-regexp-match (caddar fields) (caar fields)))
	    )
	  (setcdr hit data)
	  (sit-for 0)
	  (forward-line 1)
	  )
	)
    )
  )

(defun empire-read-cargos (&optional arg)
  "Reads an empire cargo report.  If given an arg, will forget ALL ships
   first."
  (interactive "P")
  (if arg
      (setq empire-ships nil))
  ;; my right eye teeth for a perl regexp
  (if (re-search-forward "shp# *ship type *x,y *flt *eff *sh *gun *pet *irn *dst *bar *oil *lcm *hcm *rad" nil t)
      (let ( key
	     data
	     hit )
	(forward-line 1)
	(while (looking-at empire-cargo-regexp-1)
	  (setq key (read-buffer-regexp-match 'read-buffer-num 1))

	  (if (setq hit (assoc key empire-ships))
	      (setq data (cdr hit))
	    (setq empire-ships (cons (cons key (setq data (make-vector empire-ship-pos-count nil))) empire-ships)))
	  (do ( (fields empire-cargo-regexp-pos-1 (cdr fields)) )
	      ( (null fields) nil )
	    (aset data (cadar fields)
		  (read-buffer-regexp-match (caddar fields) (caar fields)))
	    )
	  ;;(message "%s %d %d" (read-buffer-regexp-match 2) (match-beginning 2) (match-end 2))

	  (goto-char (match-end 0))

	  (looking-at empire-cargo-regexp-2)
	  (do ( (fields empire-cargo-regexp-pos-2 (cdr fields)) )
	      ( (null fields) nil)
	    (aset data (cadar fields)
		  (read-buffer-regexp-match (caddar fields) (caar fields)))
	    )
	  (sit-for 0)
	  (forward-line 1)
	  )
	)
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; forget old ship-pos
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-map-forget-ships ()
  "Takes all ships off of the empire map and thus forgets obsolete data.
   Places current ships back on the map."
  (interactive)
  (walksects-inactive
   (if (shipp x y)
       (progn
	 (message "Forgetting ship at %s,%s" x y)
	 (show-pt (cons x y) t)
	 (sit-for 3)
	 (record-des x y "." t t)))
   "Forgetting ships")
  ;; Now, put current ships back on the map
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Things To Do
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;   extract ships from looks, sonar, and bombing and store in empire-ships
;;   Actually use the labels in the mapship calls -- will need something
;;     like empire-flash-coordinates
;;   Use this ship data in maps and display sectors
