;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-plane.el -- Plane analysis routines for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright Ken Stevens, et.al. (GNU General Public License)
;; 
;; Author          : Ken Stevens <stevens@hplabs.hp.com>
;; Created On      : 
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:48 1991
;; Update Count    : 38
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;;    Provide plane analysis.
;;    Simplify the use of planes in assaults and transports.
;;
;; HISTORY
;; 30-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Fri Jan 25 19:00:47 1991 #25 (Lynn Slater x2048)
;;    made drop incriment/decriment commodity dropped
;; 25-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Fri Jan 25 09:20:21 1991 #13 (Lynn Slater x2048)
;;    moved in sat launch code, cleaned up check messages
;;    added commands to plane map
;;    mu is now mob
;; 24-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Wed Jan 23 09:43:27 1991 #9 (Lynn Slater x2048)
;;    changed prompts to remember some state
;; 23-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Tue Jan 22 22:06:38 1991 #8 (Ken Stevens)
;;    added Ken's changes
;; 21-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan 21 20:08:36 1991 #2 (Lynn Slater x2048)
;;    added command map, make hooks be cleaner
;;    made 'mob be 'mu
;;
;; TABLE OF CONTENTS
;;   air-path-to-target -- Finds a path to the target sector.
;;   print-avail-bombers -- Returns a list of bombers that can bomb sector X,Y
;;   print-avail-cargo-planes -- Returns a list of cargo planes that can be used for missions
;;   drop-paradrop-sector -- Drop cargo on sector X,Y with at least NUM cargo.
;;   bomb-sector -- Bomb sector X,Y with at least MIN-SHELLS.
;;   empire-read-flight-conflict -- Reduces plane efficiency and mobility as reported by flight conflicts.
;;   empire-target-plane -- Makes the closest plane of type TYPE have x y as the dest
;;   empire-target-grid -- Targets a type of satellite in a grid around a sector.
;;   empire-launch-targeted-sats -- Launches all targeted sats that are ready
;;   empire-edit-plane-check-hooks -- Displays a buffer showing each hook, its description, and whether it is
;;   check-empire-planes -- Runs all the empire-plane-check-hooks.  These work from the emacs database
;;
;; TODO:
;;   There is no support for launching missiles yet.
;;   There is no support for flying nuclear bombing missions.
;;   It would be nice to add and automated assault command that would
;;     use these functions to bomb, paradrop, and hold a set of sectors
;;     from one command.
;;
;; BUGS:
;;   It would be nice to automatically call empire-read-flight-conflict
;;     on all fly/bomb/drop/paradrop/recon missions to update local data.
;;   bomb-sector automatically determines from the sector type if it will
;;     be strategic or tactical.  Maybe I should take that out as some
;;     people will get unexpected results?
;;   Missions are only taken from 1 sector.  It's possible to have planes
;;     come from multiple close sectors (within 8 sectors...)
;;     same with escorts -- they only come from the same sectors as the
;;     bombers or cargo planes.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-plane)
(require 'emp-db)
(require 'emp-anal)
(require 'emp-help)
(require 'emp-ship)				; for planes on carriers.
(require 'emp-sector)


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Global variables
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-verbose-plane-checks t
  "*If not nil, planes in sectors will be displayed when a check notices something")

;;; The original idea was to have fully automated assaults.  With what we
;;; have in this file so far, using this global may be a bogus idea.
(defconst strategic-bomb-sectors '("e" "f" "n" "#" ")")
  "Set this regular expression to match any sectors you want to strategic bomb.
   All other sectors will use tactical bombing.  See \"bomb-sector\".

   Initialized with '(\"e\" \"f\" \"n\" \"#\" \")\").  May want to add \"*\" \"z\" \"h\"")



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User-accessible functions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; attack functions (bomb-sector, drop-paradrop-sector) only check
;;; availability per individual plane.
;;; 
;;; The check funtions (empire-plane-analysis) check to assure there are
;;; adequate supplies for all planes in the sector.
;;; 
;;; If you're planning a serious assault to use up a lot of mobility, etc,
;;; The attack functions won't inform you if you run out of commodities.
;;; Move all planes you plan on using into airports and then run the plane
;;; check.  That will tell you how many resources each sector needs to run
;;; a full mobility attack.
;;;
;;; The database needs to be kept up-to-date between missions for these
;;; functions to correctly prioritize which plane to choose.  For now you need
;;; do it manually.
;;; 
;;; These automatic functions always leave 2 flights on each plane.
;;; To lower them to below 0 mobility, you have to do it by manual override!
;;; That leaves some protection in there for unforseen needs.

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Randomly useful plane functions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun plane-string (plane-list)
  "returns a \"slashed\" string of plane numbers for PLANE-LIST."
  (let (str-result plane)
    (while plane-list
      (setq plane (car plane-list)
	    plane-list (cdr plane-list))
      (setq str-result (concat str-result
			       (when str-result "/")
			       (planedata-recall '\# plane))))
    (or str-result "-1")))


(defun get-planes-xy (x y plane-list)
  "return planes at sector X Y from PLANE-LIST"
  (let (plane good-ones)
    (while plane-list
      (setq plane (car plane-list)
	    plane-list (cdr plane-list))
      (when (and (= x (planedata-recall 'x plane))
		 (= y (planedata-recall 'y plane)))
	(setq good-ones (cons plane good-ones))))
    good-ones))


(defun plane-biggerp (plane1 plane2)
  (< (+ (* empire-height (planedata-recall 'y plane1))
	(planedata-recall 'x plane1))
     (+ (* empire-height (planedata-recall 'y plane2))
	(planedata-recall 'x plane2))))

(defun sort-plane-by-sector (planes)
  "destructively sorts a plane list by sector."
;;; Be sure to reassign this to the plane list or you'll destroy
;;; the list passed.
  (sort planes 'plane-biggerp))


(defun plane-to-string (plane)
  (format "%d  %s  %d,%d  %s  %d%%  %d  %d/%d  %d  %d  %d  %d  %s"
	  (planedata-recall '\# plane) (planedata-recall 'type plane)
	  (planedata-recall 'x plane) (planedata-recall 'y plane)
	  (planedata-recall 'w plane) (planedata-recall 'eff plane)
	  (planedata-recall 'mob plane) (planedata-recall 'att plane)
	  (planedata-recall 'def plane) (planedata-recall 'tech plane)
	  (planedata-recall 'ran plane) (planedata-recall 'hard plane)
	  (planedata-recall 'ship plane)
	  (if (planedata-recall 'nuke plane) (planedata-recall 'nuke plane) "")))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Display macro
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; I suppose I'm doing this wrong if this macro hasn't been written before...

; (defmacro display-before-empire-command (&rest body)
;   "places info above command string (current location) in empire-shell."
;   (` (progn
;        (pop-to-buffer empire-shell-buffer)
;        (push-mark)
;        (beginning-of-line)
;        (open-line 1)
;        (progn 
; 	 (,@ body))
;        (goto-char (mark))
;        (pop-mark))))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Flight paths!
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun air-path-to-target (start end &optional mesgp)
  "Finds a path to the target sector.
Flies directly to target, remaining over your own land as
long as possible, and avoiding land owned by others.
This is to avoid interceptors."
  (interactive (list (empire-prompt-read-xy)
  		     (empire-prompt-read-xy)
		     t))
  (let ((end-x (car end))
	(end-y (cdr end))
	(path (list start))
	(continue t)
	result)
    (while continue
      (let* ((here-x (car (car path)))
	     (here-y (cdr (car path)))
	     (x-offset (- end-x here-x))
	     (y-offset (- end-y here-y)))
	(cond ((zerop y-offset)			; One way to dest.
	       (setq path (cons (cons
				 (+ here-x (if (minusp x-offset) -2 2))
				 here-y)
				path)))
	      ((= (abs x-offset) (abs y-offset)) ; One way here, too.
	       (setq path (cons (cons
				 (+ here-x (if (minusp x-offset) -1 1))
				 (+ here-y (if (minusp y-offset) -1 1)))
				path)))
	      (t				; Multiple choices.  Take best.
	       (setq path (cons
			   (if (< (abs x-offset) (abs y-offset))
			       (choose-best-path 'gen-y-choices (car path) x-offset y-offset)
			     (choose-best-path 'gen-x-choices (car path) x-offset y-offset))
			   path))))
	(if (equal end (car path))
	    (setq continue nil))))
    ;; sanity for now....
    (unless (= (length path) (1+ (empire-mapdist (car start) (cdr start) end-x end-y)))
      (error (format "**Bogus air-path-to-target path: %s" (r-to-spath (reverse path)))))
    (setq result (r-to-spath (reverse path)))
    (if mesgp (message result))
    result))
(put 'air-path-to-target 'empire t)


;;; path support functions
(defun choose-best-path (fn here x-offset y-offset)
  "Chooses the most direct path keeping on your own sectors, or
avoiding sectors owned by others."
  (let ((valid-sects (funcall fn (car here) (cdr here) x-offset y-offset))
	new-sect)
    (setq new-sect (best-path-member valid-sects (owned-sects-near here)))
    (unless new-sect
      (setq new-sect (best-path-member valid-sects (mtn-sects-near here)))
      (unless new-sect
	(setq new-sect (best-path-member valid-sects (sea-sects-near here)))
	(unless new-sect (setq new-sect (car valid-sects))))) ;if none, choose best path.
    new-sect))


(defun best-path-member (best list)
  (let ((test (car best))
	result)
    (dolist (item list)				;see if best is in set.
      (if (equal test item)
	  (setq result test list nil)))
    (unless result				;see if second is in set.
      (setq test (car (cdr best)))
      (dolist (item list result)		
	(if (equal test item)
	    (setq result test list nil))))))


(defun gen-y-choices (here-x here-y x-offset y-offset)
  (let ((new-y (+ here-y (if (minusp y-offset) -1 1))))
    (list (cons (+ here-x (if (minusp x-offset) -1 1))
		new-y)
	  (cons (+ here-x (if (minusp x-offset) 1 -1))
		new-y))))


(defun gen-x-choices (here-x here-y x-offset y-offset)
  (let ((scaled-y (abs (* 3 y-offset)))
	(scaled-x (if (minusp x-offset) (- x-offset) x-offset))
	(right (cons (+ here-x (if (minusp x-offset) -2 2)) ;should be random...
		     here-y))
	(up (cons (+ here-x (if (minusp x-offset) -1 1))
		  (+ here-y (if (minusp y-offset) -1 1)))))
    (if (> scaled-x scaled-y)
	(list right up)
      (list up right))))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; bombing and paradrop aids
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst plane-buffer-name "*Plane Checks*")
(defvar last-plane-item "mil")
(defvar last-plane-amount 30)
(defvar last-plane-escorts 1)

(defun plane-prompt-for-cargo () ; remembers what we are shipping
  (intern
   (setq last-plane-item
	 (prompt-for-item (append empire-people-commodity-list
				  empire-basic-commodity-list)
			  last-plane-item))))

(defun plane-prompt-for-amount (cargo)
   (setq last-plane-amount
	 (empire-read-number-from-minibuffer
				(format "Try and drop at least how many %s? " cargo)
				(format "%s" last-plane-amount) (format "%s" last-plane-amount) )))

(defun plane-prompt-for-escorts ()
  (setq last-plane-escorts
	(empire-read-number-from-minibuffer
	 "Try and use how many escorts? "
	 (format "%s" last-plane-escorts)
	 (format "%s" last-plane-escorts)
	 )))

(defun prompt-for-plane-type (default)
  (completing-read "Plane type: " empire-plane-cap nil t
		   default))
			  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun print-avail-bombers (x y)
  "Returns a list of bombers that can bomb sector X,Y
execute `plane *' then \\[empire-read-planes] if bomber lists are out-of-date."
  (interactive (let ((sect (empire-prompt-read-xy)))
		 (list (car sect) (cdr sect))))
  (let (empire-strategic-bombers empire-tactical-bombers strategic tactical)
    (dolist (plane empire-planes)
      (let ((type (planedata-recall 'type plane)))
	(if (plane-cap type 'tactical)
	    (setq empire-tactical-bombers (cons plane empire-tactical-bombers))
	  (when (plane-cap type 'bomber)
	    (setq empire-strategic-bombers (cons plane empire-strategic-bombers))))))
    (setq strategic (planes-avail-for-sortie empire-strategic-bombers x y '(shell))
	  tactical (planes-avail-for-sortie empire-tactical-bombers x y '(shell)))
    (if (or strategic tactical)
	(with-output-to-temp-buffer plane-buffer-name
	  (let ((cb (current-buffer)))
	    (set-buffer standard-output)
	    (empire-data-mode t nil)
	    (when strategic
	      (print-banner (format "strategic bombers that can reach %d,%d" x y))
	      (while strategic
		(princ (concat (plane-to-string (car strategic)) "\n"))
		(setq strategic (cdr strategic))))
	    (when tactical
	      (print-banner (format "tactical bombers that can reach %d,%d" x y))
	      (while tactical
		(princ (concat (plane-to-string (car tactical)) "\n"))
		(setq tactical (cdr tactical))))
	  (set-buffer cb)))
      (message "Bummer!  No party since no bombers can reach %d,%d" x y))))
(put 'print-avail-bombers 'empire t)


(defun print-avail-cargo-planes (x y cargo para-p)
  "Returns a list of cargo planes that can be used for missions"
  (interactive
   (let ((sect (empire-prompt-read-xy))
	 (cargo (plane-prompt-for-cargo))
	 paradrop)
     (when (and (eq 'mil cargo)
		(recall (car sect) (cdr sect) (position-of 'own))
		;;(equal "y" (read-input "Is this a paradrop? (y or n) " "n"))
		)
       (setq paradrop t))
     (list (car sect) (cdr sect) cargo paradrop)))
  (let (empire-cargo-planes planes)
    (dolist (plane empire-planes)
      (when (plane-cap (planedata-recall 'type plane) 'cargo)
	(setq empire-cargo-planes (cons plane empire-cargo-planes))))
    (setq planes (planes-avail-for-sortie empire-cargo-planes x y (list cargo) para-p))
    (if planes
	(with-output-to-temp-buffer plane-buffer-name
	  (let ((cb (current-buffer)))
	    (set-buffer standard-output)
	    (empire-data-mode t nil)
	    (print-banner (format "Cargo planes that can deliver %s to %d,%d" cargo x y))
	    (while planes
	      (princ (concat (plane-to-string (car planes)) "\n"))
	      (setq planes (cdr planes)))
	    (set-buffer cb)
	    ))
      (message "No cargo planes can be outfitted.  Try updating the plane database."))))
(put 'print-avail-cargo-planes 'empire t)


(defun drop-paradrop-sector (x y cargo num num-escorts &optional para-p)
  "Drop cargo on sector X,Y with at least NUM cargo.
Will bring along NUM-ESCORTS planes if they can be found.
If optional PARA-P asserted, this is a paradrop mission.

Also see empire-read-flight-conflicts"
  (interactive (let* ((sect (empire-prompt-read-xy))
		      (cargo (plane-prompt-for-cargo))
		      (cnt-mil (plane-prompt-for-amount cargo))
		      (cnt-esc (plane-prompt-for-escorts))
		      paradrop)
		 (when (and (eq 'mil cargo)
			    (not (oursp (car sect) (cdr sect))))
		   (setq paradrop t))
		 (list (car sect) (cdr sect) cargo
		       (if (numberp cnt-mil) cnt-mil 1)
		       (if (numberp cnt-esc) cnt-esc 1)
		       paradrop)))
  (let (empire-cargo-planes plane-list)
    (dolist (plane empire-planes)
      (when (plane-cap (planedata-recall 'type plane) 'cargo)
	(setq empire-cargo-planes (cons plane empire-cargo-planes))))
    (setq plane-list (planes-avail-for-sortie empire-cargo-planes x y (list cargo) para-p))
    (if (null plane-list)
	(message "No go on the drop.  Are your data structures up to date?")
      (setq plane-list (prioritize-planes plane-list))
      (setq plane-list (return-plane-group plane-list num (not para-p)))
      (let ((plane-x (planedata-recall 'x (car plane-list)))
	    (plane-y (planedata-recall 'y (car plane-list)))
	    (print-list plane-list))

	(setq escort-list (escorts-from-xy plane-x plane-y x y num-escorts))

	;; have some sort of verbose flag to turn this off?
; 	(display-before-empire-command
; 	 (insert-banner "Planes being used for this mission:")
; 	 (while print-list
; 	   (insert (plane-to-string (car print-list)) "\n")
; 	   (setq print-list (cdr print-list)))
; 	 (if (not escort-list)
; 	     (insert "---No escorts for this mission!\n")
; 	   (insert "---escorts:\n")
; 	   (setq print-list escort-list)
; 	   (while print-list
; 	     (insert (plane-to-string (car print-list)) "\n")
; 	     (setq print-list (cdr print-list)))))

	(send-empire-command
	 (format "%s %s %s %d,%d %s %s"
		 (if para-p "paradrop" "dropoff")
		 (plane-string plane-list)
		 (plane-string escort-list)
		 plane-x plane-y
		 (air-path-to-target (cons plane-x plane-y) (cons x y))
		 (if para-p "" cargo)
		 ))
	(let ((load (plane-group-capacity plane-list)))
	  (decriment-sector plane-x plane-y (dynamic-position-of cargo) load)
	  (incriment-sector x y (dynamic-position-of cargo) load))
	)))
  nil)
(put 'drop-paradrop-sector 'empire t)


;;; I'd rather use "damage" than "min-shells", but without `exp' it's hard to
;;; guess the amount of damage that the shelling will do.
(defun bomb-sector (x y min-shells &optional num-escorts)
  "Bomb sector X,Y with at least MIN-SHELLS.
See empire-read-flight-conflict"
  (interactive (let ((sect (empire-prompt-read-xy))
		     (cnt (plane-prompt-for-amount 'shells))
		     (cnt-esc (plane-prompt-for-escorts)))
		 (list (car sect) (cdr sect)
		       (if (numberp cnt) cnt 1)
		       (if (numberp cnt) cnt 1))))
  (let* ((target-sect (recall-macro x y 'des))
	 (strategic-p (if target-sect
			  (member target-sect strategic-bomb-sectors)
			nil))			; If we know nothing of sector, strat bomb it!
	 empire-strategic-bombers empire-tactical-bombers plane-list)
    (dolist (plane empire-planes)
      (let ((type (planedata-recall 'type plane)))
	(if (plane-cap type 'tactical)
	    (unless strategic-p
	      (setq empire-tactical-bombers (cons plane empire-tactical-bombers)))
	  (when (and strategic-p (plane-cap type 'bomber))
	    (setq empire-strategic-bombers (cons plane empire-strategic-bombers))))))
    (setq plane-list (planes-avail-for-sortie (if strategic-p empire-strategic-bombers empire-tactical-bombers)
					      x y '(nonuke shell)))
    (if (null plane-list)
	(message "No planes available for bombing raid -- see print-avail-bombers")
      ;; This code limits bombing raids sourced from 1 sector.
      ;; I need to figure out the rendezvous code and add in distances here
      ;; I also need to add fighter escorts here.
      ;; avail-bombers copied plane-list, so we can destructively muck here.
      (setq plane-list (prioritize-planes plane-list))
      (setq plane-list (return-plane-group plane-list min-shells))
      (let ((plane-x (planedata-recall 'x (car plane-list)))
	    (plane-y (planedata-recall 'y (car plane-list)))
	    (print-list plane-list)
	    escort-list)

	(setq escort-list (escorts-from-xy plane-x plane-y x y num-escorts))

	;; have some sort of verbose flag to turn this off?
; 	(display-before-empire-command
; 	 (insert-banner "Planes being used for this mission:")
; 	 (while print-list
; 	   (insert (plane-to-string (car print-list)) "\n")
; 	   (setq print-list (cdr print-list)))
; 	 (if (not escort-list)
; 	     (insert "---No escorts for this mission!\n")
; 	   (insert "---escorts:\n")
; 	   (setq print-list escort-list)
; 	   (while print-list
; 	     (insert (plane-to-string (car print-list)) "\n")
; 	     (setq print-list (cdr print-list)))))

	(send-empire-command
	 (format "bomb %s %s %s %d,%d %s"
		 (plane-string plane-list)
		 (plane-string escort-list)
		 (if strategic-p "strategic" "pinpoint")
		 plane-x plane-y
		 (air-path-to-target (cons plane-x plane-y) (cons x y))))
	(let ((load (plane-group-capacity plane-list)))
	  (decriment-sector plane-x plane-y (dynamic-position-of cargo) load)
	  (incriment-sector x y (dynamic-position-of cargo) load))
	)))
  nil)
(put 'bomb-sector 'empire t)


;; customized macro for empire-read-flight-conflict
(defmacro generate-plane-list (list)
  (` (while (looking-at "\\([0-9]+\\)")	; match flight planes.
       (setq (, list) (cons (read-buffer-num (match-beginning 1) (match-end 1)) (, list)))
       (goto-char (match-end 1))
       (when (looking-at "/")		; is this the ONLY allowed delimiter?
	 (forward-char 1)))))


;; customized macro for empire-read-flight-conflict
(defmacro decrement-plane-val (type)
  (let (body)
    (cond ((eq 'eff type)
	   (setq body
		 '(if (match-beginning 4)
		      (setq empire-planes (delq plane empire-planes))
		    (when plane
		      (setf-plane-fact plane 'eff
				       (- (planedata-recall 'eff plane)
					  (read-buffer-num (match-beginning 2) (match-end 2))))
		      (when (and (null (memq plane-num plane-num-list))
				 (null (memq plane-num escort-num-list)))
			(setf-plane-fact plane 'mob
					 (- (planedata-recall 'mob plane)
					    10)))))))
	  ((eq 'mob type)
	   (setq body
		 '(when plane
		    (setf-plane-fact plane 'mob
				     (- (planedata-recall 'mob plane)
					(if (memq plane-num plane-num-list)
					    20
					  (if (memq plane-num escort-num-list)
					      12
					    0)))))))
	  (t (error "decrement-plane-val passed invalid type: %s" type)))
    (` (let* ((plane-num (read-buffer-num (match-beginning 1) (match-end 1)))
	      (plane (plane-lookup plane-num empire-planes)))
	 (, body)))))


;;; updates plane information without executing server commands!!
;;; If too much has changed, issue a plane * and empire-read-planes.
(defun empire-read-flight-conflict ()
  "Reduces plane efficiency and mobility as reported by flight conflicts.
Searches for flight commands (and plane info in read commands) from the
point to the end of the buffer.

You need to run this between flights to keep the database correct.
This is VERY important for the automated fly commands."
  ;; must distinguish between commands, so we'll search from one prompt to
  ;; the next.
  (interactive)
  (message "Searching for flights...")
  (beginning-of-line)
  (while (re-search-forward shell-prompt-pattern (point-max) t)
    (let (found-flight cmd-start cmd-end plane-num-list escort-num-list)
      (if (looking-at " *read")  ; if you get nuked, re-read planedata!
          (setq found-flight t)
	(when (looking-at " *\\(fly\\)\\|\\(bom\\)\\|\\(dr\\)\\|\\(par\\)\\|\\(rec\\)")
	  (setq found-flight t)
	  (looking-at "\\( *[a-z]+ +\\)")
	  (goto-char (match-end 1))
	  (generate-plane-list plane-num-list)
	  (when (looking-at "\\( *\\)")
	    (goto-char (match-end 1)))
	  (generate-plane-list escort-num-list)))
      (when found-flight
	(forward-line 1)
	(setq cmd-start (point)
	      cmd-end (save-excursion		; search up to next command.
			(unless (re-search-forward shell-prompt-pattern (point-max) t)
			  (goto-char (point-max)))
			(beginning-of-line)
			(point)))
	(while (< (point) cmd-end)
	  (if (looking-at "(#\\([0-9]+\\)) [---/ a-z0-9]+ equipped$")
	      (decrement-plane-val mob)
	    (when (looking-at " +[---/ a-z0-9]+ (#\\([0-9]+\\)) takes \\([0-9]+\\)\\(\\( -- shot down\\)\\|\\( -- aborted\\)\\)?.$")
	      (decrement-plane-val eff)))
	  ;; ignore "nuclear device did \\([0-9]+\\)% damage to [---/ a-z0-9]+ #\\([0-9]+\\) at -?[0-9]+,-?[0,9]+$"
	  (forward-line 1)))))
  (message "Done"))


(defun planes-avail-for-sortie (plane-list x y requirements &optional para-p)
  "Returns a list of planes that can be outfitted for a mission.
PLANE-LIST planes must be able to reach sector X Y, and have
REQUIREMENTS.  Optional PARA-P is t when this is a paradrop.

The requirements is a list that includes any commodity that can
be placed in a plane including nukes.  Some standard examples:

pin or strat bombing mission:  '( shell nonuke )
nuke bombing mission: '( nonuke )
cargo mission '( mil ), '( civ ) or whatever."
  (let (avail)
    (mapcar '(lambda (plane)
	       (let ((plane-x (planedata-recall 'x plane))
		     (plane-y (planedata-recall 'y plane))
		     (type (planedata-recall 'type plane)))
		 (when
		     ;; most of these should be tailored to the flight, like
		     ;; petrol/shells/etc.
		     (and (< 39 (planedata-recall 'eff plane))
			  (>= (or (recall-macro plane-x plane-y 'pet)
				  (ship-recall (planedata-recall 'ship plane)
				       'pet)
				  0)
			      (plane-cap type 'gas))
			  (< 20 (planedata-recall 'mob plane))
			  (> (planedata-recall 'ran plane)
			     (* 2 (empire-mapdist plane-x plane-y x y)))
			  (let ((okay t)
				(req requirements)
				cargo)
			    (while (and okay req)
			      (setq cargo (car req)
				    req (cdr req))
			      (if (eq 'nonuke cargo)
				  (setq okay (null (planedata-recall 'nuke plane)))
				;; will barf if cargo is bogus (in mult that follows)
				(let ((lbs (commodity-info cargo 'lbs))
				      (mult-fact (if (or para-p
							 (null (plane-cap type 'cargo)))
						     1 2)))
				  (when (null lbs) (setq lbs 1))
				  (let ((avail (or (recall plane-x plane-y (dynamic-position-of cargo))
						(dynamic-ship-recall (planedata-recall 'ship plane)
								     cargo)
						0)))
				    (setq okay
					  (if (zerop avail) nil
					    (>= avail
						(/ (* (plane-cap type 'load) mult-fact) lbs))))))))
			    okay))
		   (setq avail (cons plane avail)))))
	    plane-list)
    avail))


;;; This side-effects plane-list due to the destructive sort.
;;; use with EXTREME caution!
;;; use copy-sequence to copy the list
(defun prioritize-planes (plane-list)
  "This function destructively modifies the plane-list argument.
If you want to keep the list around after calling this function,
you MUST reassign its value.

The priority is done as follows:
mobility + (100 - eff) + (6 * bombs) + att + def

This gives priority to planes with more bombs, mobility, and less efficiency.

This is great for bombers, but is risky for cargo planes with low efficiency."
   (sort plane-list
	 '(lambda (x y)
	    (let ((x-val (+ (planedata-recall 'mob x)
			    (* 6 (plane-cap (planedata-recall 'type x) 'load))
			    (- 100 (planedata-recall 'eff x))
			    (planedata-recall 'att x)
			    (planedata-recall 'def x)))
		  (y-val (+ (planedata-recall 'mob y)
			    (* 6 (plane-cap (planedata-recall 'type y) 'load))
			    (- 100 (planedata-recall 'eff y))
			    (planedata-recall 'att y)
			    (planedata-recall 'def y))))
	      (> x-val y-val)))))


;;; could find escorts from further away?
(defun escorts-from-xy (src-x src-y dest-x dest-y num)
  "Return num escorts if possible"
  (let (empire-fighter-planes escort-list)
    (unless (zerop num)
      (dolist (plane empire-planes)
	(let ((type (planedata-recall 'type plane)))
	  (when (and (plane-cap type 'intercept)
		     (not (plane-cap type 'missile)))
	    (setq empire-fighter-planes (cons plane empire-fighter-planes)))))
      (setq escort-list (planes-avail-for-sortie
			 (get-planes-xy src-x src-y empire-fighter-planes)
			 dest-x dest-y nil))
      (when (> (length escort-list) num)
	(setq escort-list (prioritize-planes escort-list))
	(setf (cdr (nthcdr (1- num) escort-list)) nil)))
    escort-list))


(defun return-plane-group (pri-plane-list cnt &optional cargop)
  "Returns plane-list from PRI-PLANE-LIST that collectively carries at least CNT.
If no plane group exceeds this amount return the group that comes closest.
Pri-plane-list should be in the order of priority for the mission.

Groups consist of planes from the same sector only."
  (let (successp light-plane-list)
    (while (null successp)
      (let* 
	  ;; get planes from location of highest priority plane.
	  ((planes-xy (reverse		;preserve order lost in get-planes-xy
		       (get-planes-xy (planedata-recall 'x (car pri-plane-list))
				      (planedata-recall 'y (car pri-plane-list))
				      pri-plane-list)))
	   (plane-sub-list planes-xy)
	   (load 0)
	   good-planes)
	(while (and plane-sub-list (< load cnt))
	  (incf load (* (if cargop 2 1)
			(plane-cap (planedata-recall 'type (car plane-sub-list)) 'load)))
	  (setq good-planes (cons (car plane-sub-list) good-planes)
		plane-sub-list (cdr plane-sub-list)))
	(if (>= load cnt)
	    (setq successp good-planes)
	  (setq light-plane-list (cons (cons load planes-xy) light-plane-list))
	  ;; remove all planes from this sector, check the rest in order.
	  (while planes-xy
	    (setq pri-plane-list (delq (car planes-xy) ; *should* work, as planedata is the same..
				       ;;(plane-lookup (planedata-recall '\# (car planes-xy)) pri-plane-list)
				       pri-plane-list)
		  planes-xy (cdr planes-xy)))))
      (unless pri-plane-list			; if none group carries load, get largest
	(setq successp (cdr (car (sort light-plane-list
				       '(lambda (x y)
					  (> (car x) (car y)))))))))
    successp))

(defun plane-group-capacity (plane-sub-list &optional cargop)
  "Returns the amount that the given planes will carry."
  (let ((load 0))
    (while plane-sub-list
      (incf load (* (if cargop 2 1)
		    (plane-cap (planedata-recall 'type (car plane-sub-list)) 'load)))
      (setq plane-sub-list (cdr plane-sub-list)))
    load))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sat Launch and target designation
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-plane-targets nil
  "List of planes and their targets")
(put 'empire-plane-targets 'empire-system t)

(defvar empire-targeting-plane-type "landsat 2     ")
  
(defun empire-target-plane (x y &optional type)
  "Makes the closest plane of type TYPE have x y as the dest"
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (if current-prefix-arg
		     (setq empire-targeting-plane-type
			   (prompt-for-plane-type
			    empire-targeting-plane-type)))
		 (list (car sect) (cdr sect) empire-targeting-plane-type)
		 ))
  (let (close-plane
	close-range
	range
	(planes empire-planes)
	plane
	range num)
    (while planes
      (setq plane (car planes))
      (setq planes (cdr planes))
      (setq num (planedata-recall '\# plane))
      (if (and (string-equal (planedata-recall 'type plane) empire-targeting-plane-type)
	       (not (assq num empire-plane-targets))
 	       (string-equal (or (recall (planedata-recall 'x plane)
 					 (planedata-recall 'y plane)
					 (position-of 'des))
				 ".")
			     "*")
	       )
	  (progn
	    (setq range (empire-mapdist x y
 					(planedata-recall 'x plane)
 					(planedata-recall 'y plane)))
	    (if (or (not close-plane) (< range close-range))
		(setq close-range range
		      close-plane num))
	    ))
      )
    (if close-plane
	(progn
	  (setq plane (assq close-plane empire-planes))
	  (setq empire-plane-targets (cons (list close-plane (cons x y)) empire-plane-targets))
	  (record-des x y "$" nil)
	  (message "Plane %s is target to %s,%s range %s" close-plane x y close-range))
      (message "No plane can be targeted for %s,%s" x y)
      nil)
    ))

(defun empire-target-grid  (x y &optional type)
  "Targets a type of satellite in a grid around a sector."
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (if current-prefix-arg
		     (setq empire-targeting-plane-type
			   (prompt-for-plane-type
			    empire-targeting-plane-type)))
		 (list (car sect) (cdr sect) empire-targeting-plane-type)
		 ))
  (let ((targeted)
	(desired (list (cons x y)))
	flag)
    ;;(empire-mapdist 0 0 42 14)
    (while (and desired
		(empire-target-plane (caar desired) (cdar desired)))
      (setq targeted (cons (car desired) targeted))
      (setq x (caar desired)
	    y (cdar desired))
      (setq desired (cdr desired))

      ;; target grid around us
      (mapcar
       '(lambda (nsect)
	  (setq flag nil)
	  ;; are we already within 28 of anybody already targeted?
	  (mapcar '(lambda (sect)
		     (if (> 28 (empire-mapdist (car sect) (cdr sect)
					       (car nsect) (cdr nsect)))
			 (setq flag t)))
		  (append targeted desired))
	  ;; if not too close, make a desired
	  (if (not flag)
	      (setq desired (append desired (list nsect)))))
       (list (normalize-sect (cons (+ x 42) (+ y 14)))
	     (normalize-sect (cons (- x 42) (+ y 14)))
	     (normalize-sect (cons (+ x 42) (- y 14)))
	     (normalize-sect (cons (- x 42) (- y 14)))))
      )))


;;(setq empire-plane-targets nil)
;;(empire-target-grid -73 -21)

(defun launchedp (plane)
 (and (plane-cap (planedata-recall 'type plane) 'satellite)
      ;; need a better way to find if sat has been launched.
      ;; probably a new data structure will be required.
      (= (planedata-recall 'eff plane) 100)
      (not
       (and
	(oursp (planedata-recall 'x plane)
	     (planedata-recall 'y plane))
      (string-equal "*" (recall (planedata-recall 'x plane)
				(planedata-recall 'y plane)
				(position-of 'des)))))
      ))

(defun launch (plane x y)
  (if  empire-automatically-execute-commands
      (progn
	(send-empire-command (format "launch %s" plane))
	(sit-for 2)
	(send-empire-command (format "%s,%s" x y))
	(send-empire-command-wait "")
	;; note: does not detect failed launches
	(setf-plane-fact plane 'x x)
	(setf-plane-fact plane 'y y)
	(setf-plane-fact plane 'mob 0)
	t
	)
    (princ (message "Should launch %s to %s,%s"
		    plane
		    (caadr (assq plane empire-plane-targets))
		    (cdadr (assq plane empire-plane-targets))))
    (terpri)
    t
    ))

(defun empire-launch-targeted-sats ()
  "Launches all targeted sats that are ready"
  (interactive)
  (mapcar
   '(lambda (sdata)
      (if (and
	   ;; not launched
	   (not (launchedp (car sdata)))
	   ;; not already positioned
	   (or (not (= (plane-recall (car sdata) 'x)
		       (car (cadr sdata))))
	       (not (= (plane-recall (car sdata) 'y)
		       (cdr (cadr sdata)))))
	   (= (plane-recall (car sdata) 'eff) 100))
	  (launch (car sdata) (car (cadr sdata)) (cdr (cadr sdata)))
	))
   empire-plane-targets))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plane analysis
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-plane-check-hooks nil
  "A list of functions to call when an empire plane check is requested. 

See the fcn doc for register-plane-check-hook to learn how to add your own
hooks. 
")
(put 'empire-plane-check-hooks 'empire-system t)

(defun register-plane-check-hook (symbol label enabledp)
  "Places the function SYMBOL as a possible hook with LABEL. If ENABLEDP
is non nil, the function is in the initial set of dump hooks.

Each function is called just once and must walk the planes or sectors
itself if that is needed. All sector and plane data access must be through
the recall function. 

The hook can do anything, but mainly it is supposed to use prin1 and princ
to write messages. The standard-output is already directed to the *Checks*
buffer and the output of the different hooks are seperated by their LABELS.

Slow hooks quickly become unused hooks -- take care to be efficient.
"
  (put symbol 'empire-plane-check-hook label)
  (if enabledp
      (if (not (memq symbol empire-plane-check-hooks))
	  (setq empire-plane-check-hooks
		(cons symbol empire-plane-check-hooks)))))

(defun empire-edit-plane-check-hooks ()
  "Displays a buffer showing each hook, its description, and whether it is
enabled. Allows toggling of enabledness."
  (interactive)
  (let (hooks)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-plane-check-hook)
			      (setq hooks (cons sym hooks))))))
    ;;(setq hooks (sort hooks 'string-lessp))
    (edit-hooks 'empire-plane-check-hooks hooks
		       "*Hooks for Plane Check Analysis*"
		       "\t\t\tEmpire Plane Check Analysis Options"
)))

(defun check-empire-planes ()
  "Runs all the empire-plane-check-hooks.  These work from the emacs database
only.  Some hooks may issue automated commands.

The planes are sorted before any hooks are run."
  (interactive)

  (setq empire-planes (sort-plane-by-sector empire-planes))

  (with-output-to-temp-buffer plane-buffer-name
    (switch-to-buffer standard-output)
      (empire-data-mode t nil)
      (print-banner (format "Empire Plane Checks as of %s" (current-time-string)))
      (sit-for 0)
      (mapcar '(lambda (test)
		 (end-of-buffer)
		 (terpri)
		 (print-banner (or (get test 'empire-plane-check-hook)
				   (symbol-name test)))
		 (sit-for 0)
		 (message "Plane Checking %s"
			   (or (get test 'empire-plane-check-hook)
				   (symbol-name test)))
		 (condition-case errorcode
		     (funcall test)
		   (error
		    (princ (format "  Hook %s aborted with error %s\n" test errorcode))))
		 (sit-for 0)
		 )
	      empire-plane-check-hooks)
      (message "")
      ))
(put 'check-empire-planes 'empire t)
(register-adjust-hook 'check-empire-planes t)

(defun plane-repair-check ()
  "Checks that all planes of efficiency < 100% are in airports."
  (let ((planes empire-planes)
	plane)
    (while planes
      (setq plane (car planes)
	    planes (cdr planes))
      (when (and (< (planedata-recall 'eff plane) 100)
		 (not (string= "*" (recall-macro (planedata-recall 'x plane)
						 (planedata-recall 'y plane)
						 'des))))
	(princ (concat (plane-to-string plane) "\n"))))))
(register-plane-check-hook 'plane-repair-check
			   "Planes Needing Repair at an Airport" t)

(defun satellite-launch-check ()
  "Check satellites for launch readiness. if
empire-automatically-execute-commands is t, launch them!"  
  (let ((planes empire-planes)
	plane)
    (while planes
      (setq plane (car planes)
	    planes (cdr planes))
      (when (and (plane-cap (planedata-recall 'type plane) 'satellite)
		(not (launchedp plane))
		(= (planedata-recall 'mob plane) 127)
		)
	;; does it have a target
	(if (assq plane empire-plane-targets)
	    ;; it can be launched
	    (launch plane
		    (caadr (assq plane empire-plane-targets))
		    (cdadr (assq plane empire-plane-targets)))
	  (princ (concat
		  "Satellite ready to launch but is not targeted: "
		  (format "#%d %s %d,%d\n"
			  (planedata-recall '\# plane)
			  (planedata-recall 'type plane)
			  (planedata-recall 'x plane)
			  (planedata-recall 'y plane)))))))))
(register-plane-check-hook 'satellite-launch-check "Launchables" t)


(defun plane-sector-resource-check (cargo cap)
  ;; empire-planes must be sorted by sector before calling this function!
  "Checks that appropriate flight resources are in sectors with planes.
   Ignores satelites"
  (let ((rest-planes empire-planes)
	(lbs (commodity-info cargo 'lbs))
	(mult-factor (if (and (eq 'cargo cap) (not (eq 'mil cargo))) 2 1))
	plane-list x y)
    (while rest-planes
      (setq plane-list (cons (car rest-planes) nil)
	    rest-planes (cdr rest-planes))
      (if (plane-cap (planedata-recall 'type (car plane-list)) 'satellite)
	  nil
	;; else
	(setq x (planedata-recall 'x (car plane-list))
	      y (planedata-recall 'y (car plane-list)))
	(while (and rest-planes
		    (= x (planedata-recall 'x (car rest-planes)))
		    (= y (planedata-recall 'y (car rest-planes))))
	  (setq plane-list (cons (car rest-planes) plane-list))
	  (setq rest-planes (cdr rest-planes)))
	;; plane-list now contains all planes at x,y

	(let ((required 0)
	      (sect-des (recall-macro x y 'des))
	      (sect-eff (recall-macro x y 'eff))
	      (planes plane-list)
	      (ship-num (planedata-recall 'ship (car plane-list))))
	  (while plane-list
	    (let* ((plane (car plane-list))
		   (type (planedata-recall 'type plane))
		   (eff (planedata-recall 'eff plane))
		   (mob (planedata-recall 'mob plane))
		   (flights 0))
	      (setq plane-list (cdr plane-list))
	      (when (and (if cap (dynamic-plane-cap type cap) t)
			 (> eff 39) (> mob 20))
		(cond ((or (plane-cap type 'missile)
			   (plane-cap type 'satellite))
		       (when (> eff 59) (setq flights 1)))
		      ((or (plane-cap type 'VTOL)
			   (and (string= "*" sect-des)
				(> sect-eff 59)))
		       (setq flights (if (plane-cap type 'intercept)
					 (/ (- mob 20) 10)
				       (/ (- mob 20) 20))))))
	      (incf required (/ (* flights ; if no cap, checking for fuel.
				   (if cap (plane-cap type 'load) (plane-cap type 'gas))
				   mult-factor)
				lbs))))
	  (when (> required
		   (or (recall x y (dynamic-position-of cargo))
		       (dynamic-ship-recall ship-num cargo)
		       0))
	    (princ (format "  %s only contains %d %s(s), could use up to %d%s\n"
			   (format-xy x y)
			   (recall x y (dynamic-position-of cargo))
			   cargo required			  
			   (if (and (eq 'mil cargo) (eq 'cargo cap))
			       (format " or even %d mils." (* 2 required))
			     ".")))
	    (if empire-verbose-plane-checks
		(progn
		  (setq plane-list planes)
		  (while plane-list
		    (when (or (null cap)
			      (dynamic-plane-cap (planedata-recall 'type (car plane-list)) cap))
		      (princ (concat "    " (plane-to-string (car plane-list)) "\n")))
		    (setq plane-list (cdr plane-list)))
		  (princ "\n")))
	    ))))))

(defun plane-sector-pet-check ()
  "Checks that flyable planes have fuel for full mobilization."
  (plane-sector-resource-check 'pet nil))
(register-plane-check-hook 'plane-sector-pet-check "Petrol" t)

(defun fighter-plane-check ()
  "Checks that all fighters are in a sector they can intercept from."
  (print-banner "Fighter Plane Analysis")
  (let (planes plane headerp)
    (dolist (plane empire-planes)
      (let ((type (planedata-recall 'type plane)))
	(when (and (plane-cap type 'intercept)
		   (not (plane-cap type 'missile)))
	  (setq planes (cons plane planes)))))

    (while planes
      (setq plane (car planes)
	    planes (cdr planes))
      (when (and (not (string= "*" (recall-macro (planedata-recall 'x plane)
						 (planedata-recall 'y plane)
						 'des)))
		 (not (plane-cap (planedata-recall 'type plane) 'VTOL))
		 (< (planedata-recall 'ship plane) 0))
	(princ (concat "Not in airport " (plane-to-string plane) "\n"))))
    (when headerp (princ "\n"))))
(register-plane-check-hook 'fighter-plane-check "Interceptors" t)

(defun plane-sector-bombing-check ()
"Checks that there are enough bombs for full mobility of all bombers in airports"
  (plane-sector-resource-check 'shell 'bomber))
(register-plane-check-hook 'plane-sector-bombing-check "Bombers Readied" t)

(defun plane-sector-paradrop-check ()
"Notifies of any airport without enough mil for full paradrops consuming
all mobility of the cargo planes located there."
  (plane-sector-resource-check 'mil 'cargo))
(register-plane-check-hook 'plane-sector-paradrop-check "Para Assault Readiness" t)

;;; check nuke missles as hardened?

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map for user driving of planes
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-plane-commands-map nil
  "Keymap used from empire shell OR the map with plane manipulation commands.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-plane-commands-map)
  (define-key c-mp "\C-h" 'help-for-empire-plane-commands-map)
  (define-key c-mp help-character 'help-for-empire-plane-commands-map)
  (define-key c-mp "p"		'air-path-to-target)
  (define-key c-mp "b"		'print-avail-bombers)
  (define-key c-mp "\C-b"	'bomb-sector)
  (define-key c-mp "d"		'print-avail-cargo-planes)
  (define-key c-mp "\C-d"		'drop-paradrop-sector)
  (define-key c-mp "f"		'empire-read-flight-conflict)
  (define-key c-mp "t"		'empire-target-plane)
  (define-key c-mp "g"		'empire-target-grid)
  (define-key c-mp "c"		'empire-edit-plane-check-hooks)
  (define-key c-mp "k"		'check-empire-planes)
  (setq empire-plane-commands-map c-mp))

(make-help-screen help-for-empire-plane-commands-map
		  "b C-b d C-d p f t g c k"
		  "You have discovered the empire tool plane commands
   From here, you can use the following options:

b	Print bombers ready at an airport
C-b	  Bomb with those planes
d	Print cargo planes available to do a drop from an airport
C-d	  Paradrop with those planes
p	Calc best flight path from here to there
f	Parse result of a flight
t	Target a sat to be launched at a sector. Select the nearest sat
g	Target sats to cover a grid around a sector. Select nearest sats

k	Run plane check hooks
c	Edit plane check hooks

Use \\[help-for-empire-plane-commands-map] for help on redistribution.
Use \\[help-for-empire-extract-map] for help on data extraction.
Please use \\[describe-key] to find out more about any of the other keys."
		  empire-plane-commands-map)
