/*
 * lwp.c -- lightweight process creation, destruction and manipulation.
 * Copyright (C) 1991-3 Stephen Crane.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * author: Stephen Crane, (jsc@doc.ic.ac.uk), Department of Computing,
 * Imperial College of Science, Technology and Medicine, 180 Queen's
 * Gate, London SW7 2BZ, England.
 */

#include <stdio.h>
#include <signal.h>
#include <malloc.h>
#include "lwp.h"
#include "lwpint.h"

struct lwpQueue	LwpSchedQ[LWP_MAX_PRIO], LwpDeadQ;
struct lwpProc	*LwpCurrent = NULL;
char		**LwpContextPtr;
int		LwpMaxpri=0;		/* maximum priority so far */

static int	oldmask;

#ifdef LWPSTACKCHECK
static void lwpStackCheckInit();
static int lwpStackCheck();
static void lwpStackCheckUsed();

/* Make this a multiple of 1024 */
#define LWPSTACKROOM	1024
#endif

/* check stack direction */
static int growsdown (x)
	void	*x;
{
	int y;

	return x > (void *)&y;
}

/*
 * lwpReschedule -- schedule another process.  we also check for dead
 * processes here and free them.
 */
void lwpReschedule()
{
	extern struct lwpQueue LwpSchedQ[];
	static int lcount = LCOUNT;
	register struct lwpProc *nextp;
	register int i;

#ifdef LWPSTACKCHECK
	if (LwpCurrent) {
		lwpStackCheck(LwpCurrent);
	}
#endif
	if (!--lcount) {
		int p = lwpSetPriority(LWP_MAX_PRIO-1);
		lcount = LCOUNT;
		sigsetmask(sigsetmask(oldmask));
		LwpCurrent->pri = p;
	}
	for (i=LwpMaxpri+1; i--; ) {
		while (nextp = lwpGetFirst(&LwpSchedQ[i])) {
			if (!nextp->dead)
				break;
			/* clean up after dead bodies */
			lwpDestroy(nextp);
			if (nextp == LwpCurrent) {
				LwpCurrent = 0;
				*LwpContextPtr = 0;
			}
			nextp = 0;
		}
		if (nextp)
			break;
	}
	if (LwpCurrent == 0 && nextp == 0) {
		fprintf(stderr, "No processes to run!\n");
		exit(-1);
	}
	/* do context switch */
	if (LwpCurrent != nextp &&
	    !(LwpCurrent && lwpSave(LwpCurrent->context))) {
		/* restore previous context */
		LwpCurrent = nextp;
		*LwpContextPtr = LwpCurrent->ud;
		lwpRestore(LwpCurrent->context);
	}
}

/*
 * lwpEntryPoint -- process entry point.
 */
void lwpEntryPoint()
{
	extern struct lwpProc *LwpCurrent;

	sigsetmask(SIGNALS);
	*LwpContextPtr = LwpCurrent->ud;
	(*LwpCurrent->entry)(LwpCurrent->argc, LwpCurrent->argv,
		LwpCurrent->ud);
	lwpExit();
}

#ifdef notdef
/*
 * lwpNull -- a null process, always ready to run.
 * it (1) sets its priority to maximum to prevent a signal doing a
 * reschedule(2) enables signals, waits for one and handles it
 * and (3) resets its priority causing a reschedule.
 */
static void lwpNull()
{
	for (;;) {
		int p = lwpSetPriority(LWP_MAX_PRIO-1);
		sigpause(oldmask);
		lwpSetPriority(p);
	}
}
#endif

/*
 * lwpCreate -- create a process.
 */
struct lwpProc *lwpCreate(priority, entry, size, argc, argv, ud)
	int	priority;
	void	(*entry)();
	int	size;
	int	argc;
	char	*argv[];
	void	*ud;
{
	extern struct lwpProc *LwpCurrent;
	struct lwpProc *newp;
	int	*s, x;
	void	*sp;

	if (!(newp = (struct lwpProc *)malloc(sizeof(struct lwpProc))))
		return (0);
#ifdef LWPSTACKCHECK
	/* Add a 1K buffer on each side of the stack */
	size += 2 * LWPSTACKROOM;
#endif
	size += sizeof(stkalign_t);
	if (!(s = (int *)malloc(size)))
		return (0);
	newp->entry = entry;
	newp->argc = argc;
	newp->argv = argv;
	newp->ud = ud;
#ifndef LWPSTACKCHECK
	sp = (void *)(growsdown(&x)? (size+(int)s)&-sizeof(stkalign_t): (int)s);
#else
	if (growsdown(&x)) {
		/* AND the address with the inverse of the stack alignment */
		sp = (void *) (((long)(((char *)s) + size - LWPSTACKROOM - 2*sizeof(long)))
					& -sizeof(stkalign_t));
		newp->lowmark = (long *)sp;
		newp->himark = s;
	} else {
		newp->lowmark = s;
		sp = (void *)(((long)s) + LWPSTACKROOM);
		newp->himark = (void *)(((long)s) + size - LWPSTACKROOM);
	}
#endif
	if (LWP_MAX_PRIO <= priority)
		priority = LWP_MAX_PRIO-1;
	if (LwpMaxpri < (newp->pri = priority))
		LwpMaxpri = priority;
	newp->sbtm = s;
	newp->size = size;
	newp->dead = 0;
#ifdef LWPSTACKCHECK
	lwpStackCheckInit(newp);
#endif
	lwpReady(newp);
	lwpReady(LwpCurrent);
	lwpInitContext(newp, sp);  /* architecture-dependent: from arch.c */
	lwpReschedule();
	return (newp);
}

void lwpDestroy(proc)
	struct lwpProc *proc;
{
#ifdef LWPSTACKCHECK
	lwpStackCheckUsed();
#endif
	proc->entry = 0;
	proc->ud = 0;
	proc->argv = 0;
	free(proc->sbtm);
	proc->sbtm = 0;
#ifdef LWPSTACKCHECK
	proc->lowmark = 0;
	proc->himark = 0;
#endif
	free((char *)proc);
}

/*
 * lwpReady -- put process on ready queue.  if null, assume current.
 */
void lwpReady(p)
	struct lwpProc *p;
{
	extern struct lwpProc *LwpCurrent;
	extern struct lwpQueue LwpSchedQ[];

	if (!p)
		p = LwpCurrent;
	lwpAddTail(&LwpSchedQ[p->pri], p);
}

/*
 * return user's data
 */
void *lwpGetUD(p)
	struct lwpProc *p;
{
	if (!p)
		p = LwpCurrent;
	return (p->ud);
}

/*
 * set user's data
 */
void lwpSetUD(p, ud)
	struct lwpProc *p;
	char	*ud;
{
	if (!p)
		p = LwpCurrent;
	p->ud = ud;
}

/*
 * lwpYield -- yield the processor to another thread.
 */
void lwpYield()
{
	lwpReady(LwpCurrent);
	lwpReschedule();
}

/*
 * cause the current process to be scheduled for deletion.
 */
void lwpExit()
{
	LwpCurrent->dead = 1;
	lwpYield();
}

/*
 * mark another process as dead, so it will never be rescheduled.
 * remove any lingering FD action
 */
void lwpTerminate(p)
	struct lwpProc *p;
{
	p->dead = 1;
	if (p->fd >= 0)
		lwpWakeupFd(p);
}

/*
 * set the thread's priority, returning the old.
 * if the new priority is lower than the old, we reschedule.
 */
int lwpSetPriority(new)
	int new;
{
	int old = LwpCurrent->pri;

	if (LWP_MAX_PRIO <= new)
		new = LWP_MAX_PRIO-1;
	if (LwpMaxpri < new)
		LwpMaxpri = new;
	LwpCurrent->pri = new;
	if (new < old)
		lwpYield();
	return (old);
}

/*
 * initialise the coroutine structures
 */
struct lwpProc *lwpInitSystem(pri, ctxptr)
	int	pri;
	char	**ctxptr;
{
	extern struct lwpQueue LwpSchedQ[];
	extern struct lwpProc *LwpCurrent;
	struct lwpQueue *q;
	int i, *stack;
	struct lwpProc *sel;

	LwpContextPtr = ctxptr;
	if (pri < 1)
		pri = 1;
	*LwpContextPtr = 0;
	if (!(LwpCurrent = (struct lwpProc *)malloc(sizeof(struct lwpProc))))
		return (0);
	if (!(stack = (int *)malloc(64)))
		return (0);
	if (LWP_MAX_PRIO <= pri)
		pri = LWP_MAX_PRIO-1;
	if (LwpMaxpri < pri)
		LwpMaxpri = pri;
	LwpCurrent->next = 0;
	LwpCurrent->sbtm = stack;	/* dummy stack for "main" */
	LwpCurrent->pri = pri;
	LwpCurrent->dead = 0;
	for (i=LWP_MAX_PRIO, q=LwpSchedQ; i--; q++)
		q->head = q->tail = 0;
	LwpDeadQ.head = LwpDeadQ.tail = 0;
	/* must be lower in priority than us for this to work right */
	sel = lwpCreate(0, lwpSelect, 16384, 0, 0, 0);
	lwpInitSelect(sel);
	return (LwpCurrent);
}

#ifdef LWPSTACKCHECK

/* This assumes the size of a long.  Not that important, as
 * if we get a stack overwrite, it is not likely to be small...
 */
#define SCHECKMARK 0x5A5A5A5AL

/* lwpStackCheckInit
 *
 * Initialize the entire stack with the stack check mark.
 * Thus, we can get some indication of how much stack was
 * used.
 */
static void lwpStackCheckInit(newp)
	struct lwpProc *newp;
{
	int i;
	long mark;
	long *lp;

	int lim = newp->size/sizeof(long);
	if (!newp || !newp->sbtm) return;
	for (lp=newp->sbtm,i=0; i < lim; i++,lp++) {
		*lp = SCHECKMARK;
	}
}

/* lwpStackCheck
 *
 * Check if the thread has overflowed/underflowed its stack.
 * NOTE:
 *   If an problem occurs, it is not corrected.
 *   The buffer is not cleaned up, nor is the thread terminated.
 *   Cleaning up the buffer would be a mistake, and terminating
 *   the thread, well, could be done.   Should more like take
 *   down the entire process.
 */
static int lwpStackCheck(newp)
	struct lwpProc *newp;
{
	int i, end, amt;
	long *lp;

	if (!newp || !newp->himark || !newp->lowmark) return(1);

	for (lp=newp->himark,i=0; i < LWPSTACKROOM/sizeof(long); i++,lp++) {
		if (*lp != SCHECKMARK) {
			/* Stack overflow. */
			/* Of course we have no way of identifying
			 * which thread it was.
			 */
			if (!growsdown(&i)) {
				end = i;
				while (i < LWPSTACKROOM/sizeof(long)) {
					if (*lp++ != SCHECKMARK)
						end = i;
					i++;
				}
				amt = (end+1) * sizeof(long);
			} else {
				amt = (i+1) * sizeof(long);
			}
			logerror("Thread stack overflowed at least %d bytes!  Stack size %u, pri %d",
					amt,
					newp->size - 2*LWPSTACKROOM - sizeof(stkalign_t),
					newp->pri);
			return(0);
		}
	}
	for (lp=newp->lowmark,i=0; i < LWPSTACKROOM/sizeof(long); i++,lp++) {
		if (*lp != SCHECKMARK) {
			/* Stack underflow. */
			if (!growsdown(&i)) {
				end = i;
				while (i < LWPSTACKROOM/sizeof(long)) {
					if (*lp++ != SCHECKMARK)
						end = i;
					i++;
				}
				amt = (end+1) * sizeof(long);
			} else {
				amt = (LWPSTACKROOM - i+1) * sizeof(long);
			}
			logerror("Thread stack overflowed at least %d bytes!  Stack size %u, pri %d",
			
					amt,
					newp->size - 2*LWPSTACKROOM - sizeof(stkalign_t),
					newp->pri);
			return(0);
		}
	}
	return(1);
}

/* lwpStackCheckUsed
 *
 * Figure out how much stack was used by this thread.
 */
static void lwpStackCheckUsed(newp)
	struct lwpProc *newp;
{
	int i, end, amt;
	long *lp;
	int lim;

	if (!newp || !newp->sbtm) return;
	lim = newp->size/sizeof(long);
	if (growsdown(&i)) {
		/* Start at the bottom and find first non checkmark. */
		for (lp=newp->sbtm,i=0; i < lim; i++,lp++) {
			if (*lp != SCHECKMARK) {
				break;
			}
		}
	} else {
		/* Start at the top and find first non checkmark. */
		lp = newp->sbtm;
		lp += newp->size/sizeof(long);
		lp--;
		for (i=0; i< lim; i++, lp--) {
			if (*lp != SCHECKMARK) {
				break;
			}
		}
	}
	logerror("Thread used %u bytes of stack.  Stack size %u, pri %d",
			
			(i * sizeof(long)) - LWPSTACKROOM,
			newp->size - 2*LWPSTACKROOM - sizeof(stkalign_t),
			newp->pri);
}
#endif
