/*
 * sailio.c
 *
 * written originally for an AF_INET version of sail,
 * this package keeps better track of the connection
 * status, and allows for buffered i/o, a la stdio.h
 *
 * Dave Pare, 1986
 */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#ifdef hpux
#include <time.h>
#else
#include <sys/time.h>
#endif
#include "bit.h"

#define	SAIL	struct sailbuf
#include "sailio.h"


SAIL	*sailbuf;
static	 int nfds;

/*
 * read until end of input, or a newline...whichever comes first.
 */
char *
sgets(s, n, p, delay)
	char	*s;
	int	n;
	register SAIL *p;
	int	delay;
{
	register char *cs;

	*s = 0;
	cs = s;
	while (--n > 0) {
		if (!p->i_cnt || p->i_ptr >= p->i_base + p->i_cnt)
			if (sread(p, delay) == 0)
				break;
		if ((*cs = *p->i_ptr++) == '\n')
			break;
		cs++;
	}
	*cs = '\0';
	if (cs == s)
		return(NULL);
	return(s);
}

static int
sread(p, delay)
	SAIL	*p;
	int	delay;
{
	extern	int errno;
	static	bit_fdmask mask;
	int	i;

	if (mask == 0)
		mask = bit_newfdmask();
	errno = 0;
	/*
	 * do the "select" in the event that people have a zero or
	 * positive delay, and do not have "nodelay" flag set.
	 */
	if (delay >= 0 && (p->flags & _S_NDELAY) == 0) {
		bit_zero(mask);
		BIT_SETB(sio_fileno(p), mask);
		if (wait_on_io(mask, delay) < 0) {
			if (errno != 0) {
				p->errno = errno;
				p->flags |= _S_ERR;
			} else {
				p->flags |= _S_EOF;
				p->flags |= _S_TO;
			}
			return 0;
		}
	}
	i = read(p->fd, p->i_base, p->i_bufsiz);
	if (i < 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
		return 0;
	}
	if (i == 0) {
		if (errno == EWOULDBLOCK && p->flags & _S_NDELAY) {
			p->flags |= _S_EOF;
			errno = 0;
		} else
			p->flags |= _S_CONN;
		return 0;
	}
	p->i_ptr = p->i_base;
	p->i_cnt = i;
	p->i_base[i] = 0;
	return 1;
}

sputs(buf, p)
	char	*buf;
	SAIL	*p;
{
	char	*ptr;
	char	*end;
	char	*bp;
	int	len;
	char	c;

	end = p->o_base + p->o_bufsiz;
	ptr = p->o_ptr;
	bp = buf;
	len = strlen(buf);
	if (len + ptr >= end) {
		sflush(p);
		ptr = p->o_ptr;
	}
	while (c = *bp++)
		*ptr++ = c;
	p->o_ptr += len;
	p->o_cnt += len;
}

sflush(p)
	SAIL	*p;
{
	int	i;

	if (p->o_cnt <= 0)
		return;
	errno = 0;
	i = write(p->fd, p->o_base, p->o_cnt);
	if (i < 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
	}
	/*
	 * the error we would find would be a
	 * broken pipe.  Ignore, and expect
	 * other people to take care of the fact
	 */
	p->o_cnt = 0;
	p->o_ptr = p->o_base;
}

sclose(p)
	SAIL	*p;
{
	(void) close(p->fd);
	p->o_cnt = 0;
	p->o_ptr = NULL;
	p->i_cnt = 0;
	p->i_ptr = NULL;
	(void) free(p->i_base);
	(void) free(p->o_base);
	p->i_base = NULL;
	p->o_base = NULL;
	p->i_bufsiz = 0;
	p->o_bufsiz = 0;
	p->fd = 0;
}

SAIL *
sopen(fd)
	int	fd;
{
	extern	char *calloc();
	extern	char *malloc();
	register SAIL *p;

	if (sailbuf == 0) {
		nfds = getdtablesize();
		sailbuf = (SAIL *) calloc(sizeof(*sailbuf), nfds);
	}
	p = &sailbuf[fd];
	p->fd = fd;
	p->o_cnt = 0;
	p->o_base = (char *) malloc(BUFSIZ);
	p->i_cnt = 0;
	p->i_base = (char *) malloc(BUFSIZ);
	p->i_ptr = p->i_base;
	p->o_ptr = p->o_base;
	p->i_bufsiz = BUFSIZ;
	p->o_bufsiz = BUFSIZ;
	p->flags = 0;
	p->errno = 0;
	return p;
}

/*
 * n: number of milliseconds to wait
 */
siowait(p, n)
	SAIL	*p;
	int	n;
{
	static	bit_fdmask mask;
	int	r;

	if (mask == 0)
		mask = bit_newfdmask();
	bit_zero(mask);
	BIT_SETB(sio_fileno(p), mask);
	r = wait_on_io(mask, n);
	if (r < 0 && errno == 0) {
		p->flags |= _S_TO;
	} else if (errno != 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
	}
}

static
int
wait_on_io(mask, delay)
	bit_fdmask mask;
	int	delay;
{
	extern	int errno;
	static	bit_fdmask savemask;
	struct	timeval tv;
	int	i;

	errno = 0;
	if (savemask == 0)
		savemask = bit_newfdmask();
	bit_copy(mask, savemask);
	if (delay > 0) {
		tv.tv_sec = delay / 1000;
		tv.tv_usec = (delay % 1000) * 1000;
	} else {
		tv.tv_sec = tv.tv_usec = 0;
	}
	do {
		errno = 0;
		bit_copy(savemask, mask);
		/* ick.  Should be some way of finding max fd */
		i = select(nfds, mask, 0, 0, &tv);
		/*
		 * timeout?
		 */
		if (i == 0) {
			errno = 0;
			return -1;
		} else if (i < 0) {
			if (errno == EINTR)
				continue;
			return -1;
		}
	} while ((tv.tv_usec || tv.tv_sec) && bit_fd(mask) < 0);
	return bit_fd(mask);
}
