/*
 * file.c
 *
 * operations on files
 *
 * Dave Pare, 1989
 */

#include <sys/file.h>
#include <signal.h>
#include "misc.h"
#include "xy.h"
#include "nsc.h"
#include "file.h"

int
ef_open(type, mode, how)
	int	type;
	int	mode;
	int	how;
{
	register struct empfile *ep;
	static	int block;
	int	size;

	if (type < 0 || type >= EF_MAX)
		return 0;
	blocksig();
	ep = &empfile[type];
	if ((ep->fd = open(ep->name, mode, 0660)) < 0) {
		logerror("%s: open failed", ep->name);
		goto bad;
	}
	if (block == 0)
		block = blksize(ep->fd);
	ep->baseid = 0;
	ep->cids = 0;
	ep->mode = mode;
	ep->flags |= how;
	ep->fids = fsize(ep->fd) / ep->size;
	if (ep->flags & how)
		ep->csize = ep->fids;
	else
		ep->csize = block / ep->size;
	size = ep->csize * ep->size;
	ep->cache = malloc(size);
	if (ep->cache == 0) {
		logerror("ef_open: %s malloc(%d) failed\n", ep->name, size);
		goto bad;
	}
	if (ep->flags & EFF_MEM) {
		if (read(ep->fd, ep->cache, size) != size) {
			logerror("ef_open: read(%s) failed\n", ep->name);
			goto bad;
		}
	}
	unblocksig();
	return 1;
bad:
	unblocksig();
	return 0;
}

int
ef_close(type)
	int	type;
{
	register struct empfile *ep;
	int	size;

	if (type < 0 || type >= EF_MAX)
		return 0;
	ep = &empfile[type];
	if (ep->cache == 0) {
		/* no cache implies never opened */
		return 0;
	}
	blocksig();
	size = ep->csize * ep->size;
	if (ep->mode > 0 && ep->flags & EFF_MEM) {
		if (lseek(ep->fd, 0L, 0) < 0) {
			logerror("ef_close; lseek (%s) failed", ep->name);
			goto bad;
		}
		if (write(ep->fd, ep->cache, size) != size) {
			logerror("ef_close; write: (%s, %d bytes) failed",
				ep->name, size);
			goto bad;
		}
	}
	ep->flags &= EFF_XY|EFF_COM;
	free(ep->cache);
	ep->cache = 0;
	if (close(ep->fd) < 0)
		logerror("ef_close; close (%s) failed", ep->name);
	unblocksig();
	return 1;
bad:
	unblocksig();
	return 0;
}

char *
ef_ptr(type, id)
	int	type;
	int	id;
{
	register struct empfile *ep;

	if (type < 0 || type >= EF_MAX)
		return 0;
	ep = &empfile[type];
	if (id < 0 || id >= ep->fids)
		return 0;
	if ((ep->flags & EFF_MEM) == 0) {
		logerror("ef_ptr: (%s) only valid for EFF_MEM entries",
			ep->name);
		return 0;
	}
	return (char *) (ep->cache + ep->size * id);
}

/*
 * buffered read.  Tries to read a large number of items.
 * This system won't work is item size is > sizeof buffer area.
 */
int
ef_read(type, id, ptr)
	int	type;
	int	id;
	caddr_t	ptr;
{
	register struct empfile *ep;
	caddr_t	from;

	ep = &empfile[type];
	if (id < 0)
		return 0;
	if (ep->flags & EFF_MEM) {
		if (id >= ep->fids)
			return 0;
		from = ep->cache + (id * ep->size);
	} else {
		if (id >= ep->fids) {
			ep->fids = fsize(ep->fd) / ep->size;
			if (id >= ep->fids)
				return 0;
		}
		if (ep->baseid + ep->cids <= id || ep->baseid > id)
			fillcache(ep, id);
		from = ep->cache + (id - ep->baseid) * ep->size;
	}
	bcopy(from, ptr, ep->size);
	if (ep->postread)
		ep->postread(id, ptr);
	return 1;
}

static
fillcache(ep, start)
	struct	empfile *ep;
	int	start;
{
	int	n;

	ep->baseid = start;
	lseek(ep->fd, start * ep->size, 0);
	n = read(ep->fd, ep->cache, ep->csize * ep->size);
	ep->cids = n / ep->size;
}

/*
 * no-buffered read
 * zaps read cache
 */
int
ef_nbread(type, id, ptr)
	int	type;
	int	id;
	caddr_t	ptr;
{
	register struct empfile *ep;

	ep = &empfile[type];
	if (id < 0)
		return 0;
	if (id > ep->fids) {
		ep->fids = fsize(ep->fd) / ep->size;
		if (id > ep->fids)
			return 0;
	}
	lseek(ep->fd, id * ep->size, 0);
	read(ep->fd, ptr, ep->size);
	ef_zapcache(type);
	if (ep->postread)
		ep->postread(id, ptr);
	return 1;
}

/*
 * no-buffered write
 * zaps read cache
 */
int
ef_nbwrite(type, id, ptr)
	int	type;
	int	id;
	caddr_t	ptr;
{
	register struct empfile *ep;

	if (type < 0 || type >= EF_MAX || id < 0)
		return 0;
	ep = &empfile[type];
	if (lseek(ep->fd, id * ep->size, 0) < 0) {
		logerror("ef_nbwrite: (lseek) id %d, type %d", id, type);
		return 0;
	}
	if (ep->prewrite)
		ep->prewrite(id, ptr);
	if (write(ep->fd, ptr, ep->size) != ep->size) {
		logerror("ef_nbwrite: (write) id %d, type %d", id, type);
		return 0;
	}
	if ((ep->flags & EFF_MEM) == 0)
		ef_zapcache(type);
	if (id > ep->fids)
		ep->fids = id;
	return 1;
}

ef_zapcache(type)
	int	type;
{
	empfile[type].cids = 0;
	empfile[type].baseid = -1;
}

struct castr *
ef_cadef(type)
	int	type;
{
	return empfile[type].cadef;
}

int
ef_nelem(type)
	int	type;
{
	return empfile[type].fids;
}

int
ef_flags(type)
	int	type;
{
	return empfile[type].flags;
}

int
ef_lock(type)
	int	type;
{
	return file_lock(empfile[type].fd);
}

int
ef_unlock(type)
	int	type;
{
	return file_unlock(empfile[type].fd);
}

int
ef_vars(type, sp, nvp, vp, ap)
	int	type;
	register char *sp;
	u_char	**nvp;
	u_char	**vp;
	short	**ap;
{
	register struct empfile *ef;

	ef = &empfile[type];
	if ((ef->flags & EFF_COM) == 0)
		return -1;
	*nvp = (u_char *) (sp + ef->varoffs[0]);
	*vp = (u_char *) (sp + ef->varoffs[1]);
	*ap = (short *) (sp + ef->varoffs[2]);
	return ef->maxvars;
}
