/*
 * sailio.c
 *
 * written originally for an AF_INET version of sail,
 * this package keeps better track of the connection
 * status, and allows for buffered i/o, a la stdio.h
 *
 * Dave Pare, 1986
 */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#ifdef hpux
#include <time.h>
#else
#include <sys/time.h>
#endif
#define	SAIL	struct sailbuf

#include "sailio.h"

SAIL	sailbuf[_NFILE];

/*
 * read until end of input, or a newline...whichever comes first.
 */
char *
sgets(s, n, p, delay)
	char	*s;
	int	n;
	register SAIL *p;
	int	delay;
{
	register char *cs;

	if (p == NULL)
		abort();
	*s = 0;
	cs = s;
	while (--n > 0) {
		if (!p->i_cnt || p->i_ptr >= p->i_base + p->i_cnt)
			if (sread(p, delay) == 0)
				break;
		if ((*cs = *p->i_ptr++) == '\n')
			break;
		cs++;
	}
	*cs = '\0';
	if (cs == s)
		return(NULL);
	return(s);
}

static int
sread(p, delay)
	SAIL	*p;
	int	delay;
{
	extern	int errno;
	int	i;

	errno = 0;
	/*
	 * do the "select" in the event that people have a zero or
	 * positive delay, and do not have "nodelay" flag set.
	 */
	if (delay >= 0 && (p->flags & _S_NDELAY) == 0
			&& wait_on_io(1<<p->fd, delay) < 0) {
		if (errno != 0) {
			p->errno = errno;
			p->flags |= _S_ERR;
		} else {
			p->flags |= _S_EOF;
			p->flags |= _S_TO;
		}
		return 0;
	}
	i = read(p->fd, p->i_base, p->i_bufsiz);
	if (i < 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
		return 0;
	}
	if (i == 0) {
		if (errno == EWOULDBLOCK && p->flags & _S_NDELAY) {
			p->flags |= _S_EOF;
			errno = 0;
		} else
			p->flags |= _S_CONN;
		return 0;
	}
	p->i_ptr = p->i_base;
	p->i_cnt = i;
	p->i_base[i] = 0;
	return 1;
}

void
sputs(buf, p)
	char	*buf;
	SAIL	*p;
{
	void	sflush();
	char	*ptr;
	char	*end;
	char	*bp;
	int	len;
	char	c;

	end = p->o_base + p->o_bufsiz;
	ptr = p->o_ptr;
	bp = buf;
	len = strlen(buf);
	if (len + ptr >= end) {
		sflush(p);
		ptr = p->o_ptr;
	}
	while (c = *bp++)
		*ptr++ = c;
	p->o_ptr += len;
	p->o_cnt += len;
}

void
sflush(p)
	SAIL	*p;
{
	int	i;

	if (p->o_cnt <= 0)
		return;
	errno = 0;
	i = write(p->fd, p->o_base, p->o_cnt);
	if (i < 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
	}
	/*
	 * the error we would find would be a
	 * broken pipe.  Ignore, and expect
	 * other people to take care of the fact
	 */
	p->o_cnt = 0;
	p->o_ptr = p->o_base;
}

void
sclose(p)
	SAIL	*p;
{
	close(p->fd);
	p->o_cnt = 0;
	p->o_ptr = NULL;
	p->i_cnt = 0;
	p->i_ptr = NULL;
	free(p->i_base);
	free(p->o_base);
	p->i_base = NULL;
	p->o_base = NULL;
	p->i_bufsiz = 0;
	p->o_bufsiz = 0;
	p->fd = 0;
}

SAIL *
sopen(fd)
	int	fd;
{
	register SAIL *p;

	p = &sailbuf[fd];
	p->fd = fd;
	p->o_cnt = 0;
	p->o_base = (char *) malloc(BUFSIZ);
	p->i_cnt = 0;
	p->i_base = (char *) malloc(BUFSIZ);
	p->i_ptr = p->i_base;
	p->o_ptr = p->o_base;
	p->i_bufsiz = BUFSIZ;
	p->o_bufsiz = BUFSIZ;
	p->flags = 0;
	p->errno = 0;
	return p;
}

/*
 * n: number of milliseconds to wait
 */
void
siowait(p, n)
	SAIL	*p;
	int	n;
{
	int	r;

	r = wait_on_io(1 << p->fd, n);
	if (r < 0 && errno == 0) {
		p->flags |= _S_TO;
	} else if (errno != 0) {
		p->errno = errno;
		p->flags |= _S_ERR;
	}
}

static int
wait_on_io(mask, delay)
	long	mask;
	int	delay;
{
	extern	int errno;
	struct	timeval tv;
	long	savemask;
	int	i;

	errno = 0;
	if (delay > 0) {
		tv.tv_sec = delay / 1000;
		tv.tv_usec = (delay % 1000) * 1000;
	} else {
		tv.tv_sec = tv.tv_usec = 0;
	}
	do {
		errno = 0;
		savemask = mask;
		i = select(32, &savemask, 0, 0, &tv);
		/*
		 * timeout?
		 */
		if (i == 0) {
			errno = 0;
			return -1;
		} else if (i < 0) {
			if (errno == EINTR)
				continue;
			return -1;
		}
	} while ((tv.tv_usec || tv.tv_sec) && !savemask);
	return ffs(savemask) - 1;
}

#ifdef hpux
/*
 * HP-UX does not have the ffs() function
 */
int
ffs(x)
int x;
{
	int i;

	if (x == 0)
		return(0);
	for (i=0; i < 16; i++)
		if (x & 1<<i)
			return(i);
	return(-1);
}
#endif
