;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-db.el -- Database and extraction fcns for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990,1991 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater (lrs@indetech.com)
;; Created On      : Fri Oct 26 13:50:14 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:17:03 1991
;; Update Count    : 217
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	Holds sector data in generalizable but fast manner. Provides access
;; routines and data extractors. Includes concept of the map buffer but
;; very little about the map mode.
;; 
;; HISTORY
;; 6-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Tue Feb  5 20:09:05 1991 #204 (Lynn Slater x2048)
;;    made empire-read-map do last map column
;; 4-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Mon Feb  4 15:39:40 1991 #199 (Lynn Slater x2048)
;;    brought map-empire up to date with one in emp-db.
;; 4-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Mon Feb  4 15:21:48 1991 #197 (Lynn Slater x2048)
;;    added convert-empire-pos
;;    made describe-sect routines more consistent and easier to code
;; 22-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan 21 20:16:36 1991 #171 (Lynn Slater x2048)
;;    added changes per Ken Stevens
;; 12-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Sat Jan 12 11:21:52 1991 #166 (Lynn Slater x2048)
;;    incorpertated troth's ideas to get invisible remapping
;; 7-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan  7 09:17:59 1991 #160 (Lynn Slater x2048)
;;    added empire-prompt-read-xy from emp-mode
;;    added empire-read-resource
;;    added register-describe-sector-fcn
;; 4-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Thu Jan  3 17:10:38 1991 #147 (Lynn Slater x2048)
;;    Per stevens@hplkss.hpl.hp.com
;;    pposition-of made a macro, plane-pos corrected
;; 30-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sat Dec 29 16:05:27 1990 #137 (Lynn Slater x2048)
;;    added walksects-nontrivial-inactive. Made edit-sector show on map
;; 29-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Wed Dec 26 19:39:49 1990 #136 (Lynn Slater x2048)
;;    multiple radar scan faster, does only 1 map refresh
;; 13-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Mon Dec 10 10:24:18 1990 #129 (Lynn Slater x2048)
;;    added darryl's changes to ownership display.
;; 10-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sun Dec  9 15:09:24 1990 #128 (Lynn Slater x2048)
;;    added load of emp-dev
;; 9-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Fri Dec  7 18:08:33 1990 #126 (Lynn Slater x2048)
;;    made r-to-spath non recursive -- is faster, uses less eval depth
;; 7-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Thu Dec  6 16:14:17 1990 #115 (Lynn Slater x2048)
;;    made emp-read-planes handle 4 digit numbers.
;; 6-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Thu Dec  6 15:12:39 1990 #113 (Lynn Slater x2048)
;;    added empire-max-path-depth
;; 5-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Mon Dec  3 20:00:26 1990 #107 (Lynn Slater x2048)
;;    added refinery and nuc subs to recognized sector types in looks
;;    have ? sectors be marked as own 999 in lue of additional data
;; 3-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Mon Dec  3 11:50:44 1990 #97 (Lynn Slater x2048)
;;    added empire-nation-number
;;    read-looks records ownership
;;    read look can tell bridge head from span
;;    $ will not overwrite sector des as this messes up auto adjust as
;;      mobility code does not know what to do w/o the sector type.
;; 2-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sun Dec  2 12:50:23 1990 #94 (Lynn Slater x2048)
;;    coastwatch is now parsable
;;    point-wysiwyg documented as empire fcn.
;; 27-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Mon Nov 26 16:53:34 1990 #85 (Lynn Slater x2048)
;;    Do not replace harbors with ship des or else mob calc will fail.
;;    read-coastwatch uses short empire sector data for ships
;; 25-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Wed Nov 21 13:25:45 1990 #79 (Lynn Slater)
;;    darryl's changes: sector typesm read-looks fixed, empirw-switch-to-b
;;    spy report scan now handles planes and shot spys.
;; 21-Nov-1990		Lynn Slater	
;;    Last Modified: Sat Nov 17 18:50:42 1990 #78 (Lynn Slater)
;;    better default empire-x-stop
;; 12-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov 12 13:01:45 1990 #71 (Lynn Slater)
;;    added mapsects-active-in-range.
;;    added empire-read-coastwatch
;; 12-Nov-1990		Lynn Slater	
;;    Last Modified: Sun Nov 11 16:07:15 1990 #70 (Lynn Slater)
;;    show-pt allows desc-fcn to be passed
;; 8-Nov-1990		Lynn Slater	
;;    Last Modified: Wed Nov  7 08:46:47 1990 #57 (Lynn Slater)
;;    added pending-map-redesignations and refresh-map and new value option
;;    to empire-map-des-interactively
;; 7-Nov-1990		Lynn Slater	
;;    Last Modified: Tue Nov  6 15:20:15 1990 #56 (Lynn Slater)
;;    made make-header and related inserts conditional
;; 5-Nov-1990		Lynn Slater
;;    Last Modified: Mon Nov  5 08:43:52 1990 #48 (Lynn Slater)
;;    made options mode load only when needed
;;    read-spyreport make plureal
;;    renamed some interactive fcns, documented them better
;;    added describe-empire-interactive-functions
;;    made read-explore set up the short vectors
;; 4-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  2 19:33:05 1990 #43 (Lynn Slater)
;;    modified empire-flash-coordinates for better batch play.
;;    added empire-batch-play
;;    made restore-empire map even if emp-pos differs
;; 2-Nov-1990		Lynn Slater	
;;    Last Modified: Thu Nov  1 18:16:28 1990 #34 (Lynn Slater)
;;    added decriment-sector, incriment-sector, path-to-r, r-to-spath,
;;    last-sector-reference. Changed empire-path
;;    added darryl's fix to read-radar
;; 31-Oct-1990		Lynn Slater	
;;    Last Modified: Wed Oct 31 09:11:41 1990 #28 (Lynn Slater)
;;    added edit-sector.
;; 28-Oct-1990		Lynn Slater	
;;    Last Modified: Sun Oct 28 15:39:59 1990 #12 (Lynn Slater)
;;    all user vars are automatically dumped. This limits the need to edit
;;    this file.
;; TABLE OF CONTENTS
;;   empire-bounds -- Sets empire tools effective scan boundries. Only sectors within these
;;   empire-parms -- Sets empire tool special parms.
;;   empire-new-world -- Reads new version info and starts a new set of sectors, and 
;;   forget-sector -- Erases ALL sector information about sector x y
;;   map-empire -- Shows all that is known about the world map.  The map has special
;;   map-air-umbrella -- Shows the defensive air coverage on a scale of blank, 0-9 for all sectors.
;;   empire-toggle-global-desc-fcn -- Toggle the sector description function among the different alternatives.
;;   point-wysiwyg -- Scrolls the window horozontally to make point visible
;;   window-wysiwyg-point -- Makes point become the visible point
;;   empire-show-next-where -- Grabs the sector designation after or around point, shows it on the map
;;   empire-show-previous-where -- Grabs the sector designation before or around point, shows it on the map
;;   empire-path -- Finds path from the last sector mentioned in the current buffer to a
;;   empire-quick-read-dump -- Reads dump data without applying any of the hooks.
;;   empire-read-dump -- Reads the next dump output. As each sector is read, run
;;   empire-read-census -- Reads an empire census. If given an arg, highlight those sectors on the map.
;;   empire-read-planes -- Reads an empire plane report.  If given an arg, will forget ALL planes
;;   empire-read-version -- Reads empire version info, sets up new world. 
;;   empire-read-production -- Reads next production command output
;;   empire-read-resource -- Reads an empire resource command
;;   empire-read-spyreports -- Reads all spy data to end of buffer into sector db.
;;   empire-read-coastwatch -- Reads sector designations from an empire map into the DB.
;;   empire-read-map -- Reads sector designations from an empire map into the DB.
;;   empire-read-map-bounded -- reads a map presuming that is starts at the empire bounds
;;   empire-read-radars -- Reads all radars to end of buffer into sector DB
;;   empire-read-looks -- Reads into the DB the results of looks from ships
;;   empire-read-navs -- Scans to end of buffer and marks as a '.' all sectors near where a ship
;;   empire-read-deployed-mines -- X
;;   empire-read-exploding-mines -- Places an X as the sector designation of anyplace where a mine
;;   empire-read-mines -- Reads mines laid by us as well as exploding mines we detect.
;;   empire-read-flight -- Reads to end of buffer all designations returned by flights. This
;;   empire-read-pings -- Reads to end of buffer all designations pings.
;;   empire-read-explores -- This reads all explore map up to the end of the buffer. The explored
;;   empire-summarize-resources -- Summarizes all resources found by explore commands.
;;   empire-read-nation-report -- Look for nation reports, and extract the education, happiness,
;;   empire-get-data-from-file -- Scans a normal file for empire data made from previous sessions.
;;   dired-empire -- Reads all files in a directory and extracts empire radar, look, flight,
;;   save-empire -- Stores an empire session.
;;   restore-empire -- Restores a previously saved sessions and its custom values
;;   empire-edit-options -- Display a editable list of Emacs empire user options, with values and
;;   describe-empire-interactive-functions -- Offers function documentation on all interactive functions empire offers
;;   empire-edit-sector -- Allows direct editing of the empire sector database. Good for very
;;   empire-test-world-setup -- Makes diagnostic messages
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'cl)
(require 'emp-const)
(provide 'emp-db)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Autoloads
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(autoload 'empire-display-dev-sectors-if-necessary "emp-dev" "\
Not yet documented."
	  nil)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; special
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst GEET-Version "2dp2a")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User variables -- do not edit here, use empire-edit-options or the
;; listed functions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-width 32
  "The true width of the empire, from 0,0. i.e width 32 means 64x??
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")
(put 'empire-width 'empire-system t)

(defvar empire-height 32
  "The true height of the empire, from 0,0. i.e height 32 means ??x64
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")

(put 'empire-height 'empire-system t)

(defvar empire-x-start (- 0 empire-width)
  "First x coordinate mapped and checked. Set via \\[empire-bounds]")
(defvar empire-x-stop (1- empire-width)
  "Last x coordinate to map and check. Set via \\[empire-bounds]")
(defvar empire-y-start (- 0 empire-height)
  "First y coordinate mapped and checked.. Set via \\[empire-bounds]")
(defvar empire-y-stop empire-height
  "Last y coordinate mapped and checked. Set via \\[empire-bounds]")
(put 'empire-x-start 'empire-system t)
(put 'empire-x-stop 'empire-system t)
(put 'empire-y-start 'empire-system t)
(put 'empire-y-stop 'empire-system t)

;; display control variables
(defvar empire-map-des-interactively nil
  "*If true, sector designations show on the map as they are found. If 0,
they are pended for later refresh. Only automated commands should set this
to 0.")
(defvar empire-attempt-highlighting nil
  "*If t, empire will attempt to highlight the current map point. This works
  on some terminals and not on others.")
(defvar empire-attempt-map-bars t
  "*If t, empire will attempt to bracket the current map point. This works
   on all terminals but may cause too many characters to be sent over slow
  lines. ")

(defvar empire-development-mode nil
  "If non-nil, the map buffer will display the desired designations of
sectors, not what each sector really is. See emp-dev.el")
(make-variable-buffer-local 'empire-development-mode)

(defvar empire-interaction-verbosity t
  "*If `t', GNU Empire will actively display sector X,Y coordinates as
each sector is processed (by whatever).  If `0', only the Y coordinate
will be displayed as the sectors are processed.  If `nil', only the
type of action that is being done will be displayed (no sector
coordinate information of any sort will be displayed).  Setting this
to `nil' or `0' is useful when using this tool via a modem, as the
length of time needed to dynamically display all processed X,Y
coordinates can be longer than the amount of actual processing time.
Note that this variable is *NOT* saved in the empire data file.")

(defvar empire-use-mousep t
   "If t, xun or X11 mouse is enabled.")

(defvar empire-education-level 0
  "*The education level of your country, rounded to the nearest integer.")
(defvar empire-happiness-level 0
  "*The happiness level of your country, rounded to the nearest integer.")
(defvar empire-technology-level 0
  "*The technology level of your country, rounded to the nearest integer.")
(defvar empire-research-level 0
  "*The research level of your country, rounded to the nearest integer.")
(put 'empire-education-level 'empire-system t)
(put 'empire-happiness-level 'empire-system t)
(put 'empire-technology-level 'empire-system t)
(put 'empire-research-level 'empire-system t)

(defvar empire-nation-number 1
  "*The number of your nation")

;; routes
(defvar empire-highway-min-eff 90
 "*Minunum efficiency a highway must have to be
  considered in a user requested path calc or in automatic commodity
  routing commands. Set this low in young countries."  )
(defvar empire-route-past-bridges nil
  "*If true, routes may cross bridges. This affects routs calculated by a
manual command as well as the automatic distribution commands.") 
(defvar empire-max-path-depth 100
  "*Limit of depth to consider making calculated roads. This applies to
both user requested path calcs and to auto distribution paths.
  If too large, you risk overflowign the limits in max-lisp-eval-depth or
max-specpdl-size." )

(defvar empire-display-numeric-unknown-ownership t
  "*If non-nil, enemy sectors for which ownership is unknown will be
displayed as being owned by country \"999\".  If nil, the owner will be
displayed as \"?\".")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Internal state storage
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-save-file-name "empire.data"
  "*Name of file in which empire data is saved")

(defvar empire-shell-buffer "*s1*"
  "Name of buffer running shell running empire.")

(defvar empire-map-buffer nil
  "The buffer in which we map")

(defvar highlighted-map nil
  "Is the highlighted spot on the map")

(defvar empire-current-map-x nil
  "The current X coordinate in the map buffer.  If this variable is nil,
the center of the map is used.")
(put 'empire-current-map-x 'empire-system t)

(defvar empire-current-map-y nil
  "The current Y coordinate in the map buffer.  If this variable is nil,
the center of the map is used.")
(put 'empire-current-map-y 'empire-system t)

(defvar empire-sects nil		; Handled specially in save-empire
  "The storage of sector attributes. It is an array of arrays holding
arrays of sector data. The Y and X offsets access the first two array
levels, and the sector data is stored in offsets specified in the
empire-pos list.
  empire-new-sects sets this initially.")

(defvar empire-row-ranges nil		; Handled specially in save-empire
  "Stores the start and stop of the known sectors in each row. This
information optimizes many sector by sector operations.")



(defconst empire-pos-count		; Handled specially in save-empire
  (let ((max 0))
    (mapcar (function (lambda (pair)
	       (if (> (or (cdr pair) 0) max) (setq max (cdr pair)))))
	    empire-pos)
    (+ max 2))
  "Size of vector to allocate to store the sector data array")

(mapcar 
 (function (lambda (pair)
    ;; scan list and replace nils with a dummy bit-bucket element at the 
    ;; end of each vector. This will make 'record faster at the cost of some
    ;; memory. 
    (if (not (cdr pair)) (setcdr pair (1- empire-pos-count)))
    pair))
 empire-pos
 )

(defmacro empire-pos-var-let (&rest let-body)
  "Introduces a let where each var in empire-pos is bound."
  (cons 'let (cons
	(mapcar '(lambda (pair) (list (car pair)))
	  empire-pos)
	let-body))
  )

(defvar empire-planes nil		; Handled specially in save-empire
  "An alist of plane number and plane data")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Control vars read from version output
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar seconds-per-time-unit 2000 "
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version]")
(put 'seconds-per-time-unit 'empire-system t)

(defvar etu-per-update 8 "
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")
(put 'etu-per-update 'empire-system t)

(defvar upd-per-day 8 "
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")
(put 'upd-per-day 'empire-system t)

(defvar bodies-fead-by-1-food-per-etu 1000 
  "Number of people needed to eat 1 food in an	etu
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")
(put 'bodies-fead-by-1-food-per-etu 'empire-system t)

(defvar edu-divisor 4000
  "Number of people 1 edu unit can educate by 1 level
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version].")
(put 'edu-divisor 'empire-system t)

(defvar happy-divisor 5000
  "Number of people 1 happy unit can make happy by 1 level
  This should almost always be set by \\[empire-new-world] 
                                   or \\[empire-read-version]")
(put 'happy-divisor 'empire-system t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Utility code
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro safe-excursion (body)
  (` (save-excursion
       (condition-case errorcode
       (, body)
       (error (format "Error %s" errorcode))
       (t nil)))))

(defmacro match-string (n)
  (` (buffer-substring (match-beginning (, n)) (match-end (, n)))))

(defmacro read-match-num (n)
  "Reads current buffer identified by match N as a integer number"
  (` (prog1
	 (car (read-from-string
	       (buffer-substring (match-beginning (, n)) (match-end (, n)))))
       (goto-char (match-end (, n))))))

(defun read-buffer-num (start stop)
  "Reads current buffer from START to STOP as a integer number"
  (prog1 (car (read-from-string (buffer-substring start stop)))
    (goto-char stop)))
  

;;
;; amount-in-map-sector:
;; Return the amount of ITEM in the current map sector.  ITEM is returned
;; as an integer.  This is a macro for speed.
;;
(defmacro amount-in-map-sector (item)
  (if (and (listp item) (eq (car item) 'quote))
    (`
     (let (sect x y)
       (setq sect (get-map-sector))
       (setq x (car sect))
       (setq y (cdr sect))
       (recall x y (position-of (, item)))
     )
    )
    (`
     (let (sect x y)
       (setq sect (get-map-sector))
       (setq x (car sect))
       (setq y (cdr sect))
       (recall x y (dynamic-position-of (, item)))
     )
    )
  )
)

(defmacro empire-save-window-excursion (&rest forms)
  "`save-window-excursion' does not keep track of the selected window.
We therefore define a macro which does."
  (`
   (let ( (current-window (selected-window)) )
     (save-window-excursion
       (unwind-protect
	   (progn
	     (,@ forms)
	   )
	 (select-window current-window)
       )
     )
   )
  )
)

(defmacro current-hour-string ()
  (` (substring (current-time-string) 11 19)))

;;
;; empire-flash-coordinates is a macro for speed.
;;
(defvar empire-last-y -9999)
(defvar empire-last-verb "-9999")
(defmacro empire-flash-coordinates (verb x y)
  (`
   (if (integerp empire-interaction-verbosity)
       (if (= empire-interaction-verbosity 0)
	   (if (not (= y empire-last-y))
	       (progn
		 (setq empire-last-y y)
		 (message "%s, y = %s ..." (or (, verb) "Processing") (, y))))
	 (message "%s %s,%s ..." (or (, verb) "Processing") (, x) (, y))
	 )
     (if empire-interaction-verbosity
	 (message "%s %s,%s ..." (or (, verb) "Processing") (, x) (, y))
       (if (not (string-equal  (or (, verb) "Processing") empire-last-verb))
	   (progn
	     (setq empire-last-verb (or (, verb) "Processing"))
	     (if empire-batch-play
		 (message "%s (%s)..." (or (, verb) "Processing") (current-hour-string))
	       (message "%s ..." (or (, verb) "Processing")))))
       )
     )
   )
  )



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Auxilary Data Variable Manipulation
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-bounds (&optional nomap)
  "Sets empire tools effective scan boundries. Only sectors within these
boundries will be checked or considered by most empire tool commands. Thus,
you can have the performance of a small world even if the universe is large
and mostly unknown."
  (interactive)
  (setq empire-x-start
	(max (- 0 empire-width)
	     (car (read-from-string
		   (read-string (format "Effective Starting X (%s -> %s): "
					(- 0 empire-width) empire-width)
				(format "%s" empire-x-start))))))
  (setq empire-x-stop
	(min (1- empire-width)
	     (car (read-from-string
		   (read-string (format "Effective Last X (%s -> %s): "
					(- 0 empire-width)(1-  empire-width))
				(format "%s" empire-x-stop))))))
  (setq empire-y-start
	(max (- 0 empire-height)
	     (car (read-from-string
		   (read-string (format "Effective Starting Y (%s -> %s): "
					(- 0 empire-height) empire-height)
				(format "%s" empire-y-start))))))
  (setq empire-y-stop
	(min (1- empire-height)
	     (car (read-from-string
		   (read-string (format "Effective Last Y (%s -> %s): "
					(- 0 empire-height) (1- empire-height))
				(format "%s" empire-y-stop))))))
  (message "Scan bounds are %s:%s,%s:%s" empire-x-start empire-x-stop
	   empire-y-start empire-y-stop)
  ;; remap so that map moves are not confused
  (if (not nomap)(map-empire))
  )

(defun empire-parms () ;; obsolete and inflexible
  "Sets empire tool special parms."
  (interactive)
  (setq empire-highway-min-eff
	(car (read-from-string
	      (read-string "Minimum eff needed to route on a highway: "
			   (format "%s" empire-highway-min-eff)))))
  (setq empire-route-past-bridges (y-or-n-p "Route goods over bridges? "))
  (setq empire-highway-leave-food
	(car (read-from-string
	      (read-string "Amount of food to leave on a highway: "
			   (format "%s" empire-highway-leave-food)))))
  (setq empire-highway-min-food
	(car (read-from-string
	      (read-string "Min food to make sure each highway has: "
			   (format "%s" empire-highway-min-food)))))
  (setq empire-min-mil-move
	(car (read-from-string (read-string "Min mil to be worth moving: "
					    (format "%s" empire-min-mil-move)))))
  (setq empire-min-mil-move-into-enlistment
	(car (read-from-string
	      (read-string
	       "Min mil to be worth moving into enlistment centers: "
	       (format "%s" empire-min-mil-move-into-enlistment)))))
  (setq empire-min-civ-move
	(car (read-from-string (read-string "Min civ to be worth moving: "
					    (format "%s" empire-min-civ-move)))))
  (setq empire-air-defense-incr
	(car (read-from-string
	      (read-string
	       "Number of attack units to merit 1 level of air defense: "
	       (format "%s" empire-air-defense-incr)))))

  (message "Changes done. Save empire to preserve for the future if desired.")
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Core data manipulation
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro position-of (name);; lookup at compile time
  (if (and (listp name) (eq (car name) 'quote))
      (setq name (car (cdr name))))
  (if (not (symbolp name))
      (error "Compile time position-of used for run time value")
    (let ((offset (cdr (assq name empire-pos))))
      (if (not offset)
	  (error "position-of used for value not in empire-pos: %s" name)
	offset))))

(defmacro dynamic-position-of (name) ;; lookup at run time
  (list 'cdr (list 'assq name 'empire-pos)))

;;(position-of x)
;;(position-of '*)
;;(position-of junk)
;;(position-of 1)
;;(dynamic-position-of 'x)

(defun empire-new-world (file)
"Reads new version info and starts a new set of sectors, and 
   FORGETS ALL DATA ON THE OLD WORLD!
   Set default effective bounds of 64x64 as a new country will likely not
   fall farther away than that. Dietys will want to change right away.
   To change in the future, invoke \\[empire-bounds]"
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)))
  ;; zero out all variables
  (setq empire-planes nil)
  (setq empire-ships nil)
  (setq empire-save-file-name file)
  (setq empire-sects nil)
  (setq empire-row-ranges nil)
  (setq empire-height 2) ;; force error if version read fails
  (setq empire-width 2)
  
  (let ((found-it nil))
    (save-excursion
     (progn
       (end-of-buffer)
       (if (not (or (re-search-backward
		     "^The following parameters have been set for this game:"
		     nil t)
		    (re-search-forward
		     "^The following parameters have been set for this game:"
		     nil t)))
	   (error "Do a version command in the empire shell first")
	 ;; read the parms
	 (if (empire-read-version)
	     (progn
	       (setq found-it t)
	       (empire-new-sects)
	       (message "Data Extracted for New World")
	       (setq empire-x-start (max -32 (- 0 empire-width)))
	       (setq empire-x-stop  (min 32 (1- empire-width)))
	       (setq empire-y-start (max -32 (- 0 empire-height)))
	       (setq empire-y-stop  (min 32 (1- empire-height)))

	       (setq empire-nation-number
		     (car (read-from-string
			   (read-string "Your nation number: " 
					(format "%s" empire-nation-number)))))	       
	       ;;(empire-parms)
	       (empire-bounds t)
	       ))
	 )))
    (if (not found-it)
	(error "Did not find version command output, cannot set up new world")
      (empire-read-data)
      (message "New world setup but not saved"))
    ))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Coordinate system conversion
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro hexx-to-rec (x)
  "Turns empire x offset to internal 0 based array index"
  (`
     (/ (+ (cond
	    ;; handle wrapping
	    ((>= (, x) empire-width) (- (, x) (+ empire-width empire-width)))

	    ((< (, x) (- 0 empire-width))(+ (, x) (+ empire-width empire-width)))
	    (t (, x)))
	   empire-width) 2)))

(defmacro hexy-to-rec (y)
  "Turns empire y offset to internal 0 based array index
   Might have side affect on args."
  ;; Handle wrapping
  (`
   (+ (cond
       ((>= (, y) empire-height) (- (, y) (+ empire-height empire-height)))
       ((< (, y) (- 0 empire-height))(+ (, y) (+ empire-height empire-height)))
       (t (, y)))
      empire-height)))

;;(hexx-to-rec 4)
;;(hexy-to-rec 4)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector Data Storage
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-new-sects ()
  "FORGETS ALL SECTOR DATA ON THE OLD WORLD!"
  (setq empire-sects (make-vector (* empire-height 2) nil))
  (setq empire-row-ranges (make-vector (* empire-height 2) nil))
  )

(defun record (x y fact value)
  "Records about sector X,Y that FACT has VALUE.
   FACT must either be a label in empire-pos or a numeric array offset.
   Value can be any lisp type."
  (let* (
	 (nx (hexx-to-rec x))
	 (ny (hexy-to-rec y))
	 (row (aref empire-sects ny))
	 (data (if row (aref row nx)))
	 ;;(offset (if (numberp fact) fact (dynamic-position-of fact)))
	 (offset fact)
	 cell)
    (if (not row)
	;; define a new row
	(aset empire-sects ny (setq row (make-vector empire-width nil)))
      )
    (if (not data)
	(progn
	  (setq data (make-vector empire-pos-count nil))
	  (aset data 0 x)
	  (aset data 1 y)
	  (aset row nx data))
      (if (< (length data) empire-pos-count)	; make a new and longer vector
	  (let ((newdata (make-vector empire-pos-count nil)))
	    (aset newdata 0 x)
	    (aset newdata 1 y)
	    (aset newdata 2 (aref data 2))	; special case copy of des
	    (setq data newdata)
	    (aset row nx data)
	    )
	  ))
    ;; we have a long data
    (aset data offset value)
    value
    ))

(defun record-max (x y fact max value)
  "Records about sector X,Y that FACT has VALUE.
   FACT must either be a label in empire-pos or a numeric array offset.
   Value can be any lisp type.

   This special variant of record will only allocate room for max entries
   in the sector data vector. This is good if you suspect that more data is
   not forthcoming."
  (let* (
	(nx (hexx-to-rec x))
	(ny (hexy-to-rec y))
	(row (aref empire-sects ny))
	(data (if row (aref row nx)))
	;;(offset (if (numberp fact) fact (dynamic-position-of fact)))
	(offset fact)
	cell)
    (if (not row)
	;; define a new row
	(aset empire-sects ny (setq row (make-vector empire-width nil)))
      )
    (if (not data)
	(progn
	  (setq data (make-vector (1+ max) nil))
	  (aset data 0 x)
	  (aset data 1 y)
	  (aset row nx data)))
      ;; we have a data
      (aset data offset value)
      ;;(list nx ny offset)
      value
      ))

(defun recall (x y fact)
  "Returns what was recorded about sector X,Y attribute FACT"
  (let* (
	 ;;(offset (if (numberp fact) fact (dynamic-position-of fact)))
	 (offset fact)
	 (row (aref empire-sects (hexy-to-rec y)))
	 (data (if row (aref row (hexx-to-rec x))))	
	 )
    (if (or (not data) (>= offset (length data)))
	nil
      ;; we have a data
      (aref data offset))
    ))

(defun forget-sector (x y)
  "Erases ALL sector information about sector x y
   To be used as cleanup code only"
  (interactive "nX: \nnY: ")
  (let (
	(row (aref empire-sects (hexy-to-rec y)))
	)
    (if row
	(aset row (hexx-to-rec x) nil))))

(defun decriment-sector (x y fact value)
  "Reduces sector X,Y numeric attribute FACT by VALUE"
  (let* (
	 (row (aref empire-sects (hexy-to-rec y)))
	 (data (if row (aref row (hexx-to-rec x))))	
	 )
    (if (or (not data) (>= fact (length data)))
	nil
      ;; we have a data
      (aset data fact (- (aref data fact) value))
      )
    ))

(defun incriment-sector (x y fact value)
  "Increases sector X,Y numeric attribute FACT by VALUE.
Fact must be a number; this is the same as is used in record and recall.

This function does not check the validity of the sector, the item, or the
amount.  With funny values, the amount in the sector could be negative;
this is intentional, as negative values should be a flag to the user,
saying that something is wrong."
  (let* (
	 (row (aref empire-sects (hexy-to-rec y)))
	 (data (if row (aref row (hexx-to-rec x))))	
	 )
    (if (or (not data) (>= fact (length data)))
	nil
      ;; we have a data
      (aset data fact (+ (aref data fact) value))
      )
    ))

;;(defmacro record (x y fact VALUE)
;;  ;; this is really just a call to record-fcn but checks the arg
;;  (if (and (listp fact) (eq (car fact) 'quote))
;;      (error "Record called with bad variable value %s" fact))
;;  (list 'record-function x y fact VALUE))
;;
;;(defmacro recall (x y fact)
;;  ;; this is really just a call to recall-fcn but checks the arg
;;  (if (and (listp fact) (eq (car fact) 'quote))
;;      (error "Recall called with bad variable value %s" fact))
;;  (list 'recall-function x y fact))
;;
;;(recall 0 0 des)
;;(recall 0 0 'des)
;;(recall 0 0 (position-of des))

(defmacro recall-macro (x y ffact)
  "Returns what was recorded about sector X,Y attribute FACT.  Just like
`recall', but in macro form."
  (if (not (numberp ffact)) (setq ffact (eval ffact)))
  (if (listp ffact) (setq ffact (nth 1 ffact)))
  (let ((offset (if (numberp ffact) ffact (dynamic-position-of ffact))))
    (`
     (let* (
	    (row (aref empire-sects (hexy-to-rec (, y))))
	    (data (if row (aref row (hexx-to-rec (, x)))))	
	    )
       (if (or (not data) (>= (, offset) (length data)))
	   nil
	 ;; we have a data
	 (aref data (, offset)))
       )
     )
    ))

;(defun test () (let ((sects (list (cons 0 0)))) (recall-macro (car (car sects)) (cdr (car sects)) 'dist_y)))
;(test)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plane data manip
;;  A set of macros to make plane accesses invisible to user.
;;  This allows us to change the data types and the code will still work.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro pposition-of (name);; lookup at compile time
  (if (and (listp name) (eq (car name) 'quote))
      (setq name (car (cdr name))))
  (if (not (symbolp name))
      (error "Compile time pposition-of used for run time value")
    (let ((offset (cdr (assq name empire-plane-pos))))
      (if (not offset)
	  (error "pposition-of used for value not in empire-plane-pos: %s" name)
	offset))))

;;-> (defun plane-recall (number fact)
;;->   (let* ((offset (if (numberp fact) fact (cdr (assq fact empire-plane-pos))))
;;-> 	 (data (assq number empire-planes)))
;;->    (nth offset data)))

(defmacro plane-lookup (number plane-list)
  "Lookup plane NUMBER in list PLANE-LIST.  Returns planedata."
  (list 'assq number plane-list))

(defmacro plane-recall (number fact)
  (` (nth (pposition-of (, fact)) (assq (, number) empire-planes))))

(defmacro planedata-recall (fact plane-data)
  (` (nth (pposition-of (, fact)) (, plane-data))))

(defmacro setf-plane-fact (plane fact val)
   (` (setf (nth (pposition-of (, fact)) (, plane)) (, val))))
  
(defmacro plane-cap (type cap)
  (when (and (listp cap) (eq (car cap) 'quote))
    (setq cap (car (cdr cap))))
  (if (not (symbolp cap))
      (error "Compile time plane-cap used for run time value")
    (let ((offset (cdr (assq cap empire-cap-pos)))
	  flag-p)
      (when (not offset)
	(if (memq cap empire-plane-flags)
	    (setq offset (cdr (assq 'flags empire-cap-pos))
		  flag-p t)
	  (error "plane-cap used invalid capability: %s" cap)))

      (if flag-p
	  (list 'memq (list 'quote cap)
		(list 'nth offset
		      (list 'assoc type
			    'empire-plane-cap)))
	(list 'nth offset
	      (list 'assoc type 'empire-plane-cap))))))

(defun dynamic-plane-cap (type cap)
  "determine the plane capabilities from runtime cap values."
  (let ((planecap-entry (assoc type empire-plane-cap)))
    (if (memq cap empire-plane-flags)
	(memq cap (nth (cdr (assq 'flags empire-cap-pos)) planecap-entry))
      (nth (cdr (assq cap empire-cap-pos)) planecap-entry))))
  

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ship data
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'emp-ship)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector Mapping in the lisp sense
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun mapsects-all-in-bounds (fcn &optional label)
  "Calls FCN on every sector in the effective bounds of the empire,
EVEN IF THE SECTOR HAS NEVER BEEN DESGINATED. Prints LABEL and sector number
on the command line as it goes."
  (let (x
	(x-delta (abs (% empire-x-start 2)))
	(y empire-y-start)
	)
    (setq empire-last-verb "")
    (while (< y empire-y-stop)
	(if (= 0 (% y 2))
	    ;; make x be even
	    (setq x  (+ empire-x-start x-delta))
	  ;; else make x be odd
	  (setq x (+ empire-x-start (- 1 x-delta)))
	  )	 
      ;;(setq x (+ (% y 2) empire-x-start))
      (while (<= x empire-x-stop)
	(empire-flash-coordinates label x y)
	(funcall fcn x y)
	(setq x (+ 2 x)))
      (setq y (1+ y))
      )
    (message "")
    ))

(defun mapsects (fcn &optional label)
  "Calls FCN on every sector in the effective bounds of the empire that has
a designation. Prints LABEL and sector number on the command line as it goes.
See record-des."
  (let (x
	(x-delta (abs (% empire-x-start 2)))
	(y empire-y-start)
	range
	stop-x
	)
    (setq empire-last-verb "")
    (while (< y empire-y-stop)
      (setq range (aref empire-row-ranges (hexy-to-rec y)))
      (if range				; need we process this row?
	  (progn
	    (setq x (max (car range)
			 ;;empire-x-start  ;; why does this break?
			 (logior (logand empire-x-start
						     (lognot 1))
					     (logand (car range) 1))
			 ))
	    (setq stop-x (min (cdr range) empire-x-stop))
	    (while (<= x stop-x)
	      (empire-flash-coordinates label x y)
	      ;;(valid-sector x y)
	      (funcall fcn x y)
	      (setq x (+ 2 x)))
	    ))
      (setq y (1+ y))
      )
    (message "")
    ))


(defmacro walksects (body &optional label)
  "Runs BODY on every sector in the effective bounds of the empire that has
a designation. Prints LABEL and sector number on the command line as it goes.
See record-des."
  (`
   (let (x
	 (y empire-y-start)
	 range
	 stop-x
	 )
     (setq empire-last-verb "")
     (while (< y empire-y-stop)
       (setq range (aref empire-row-ranges (hexy-to-rec y)))
       (if range			; need we process this row?
	   (progn
	     (setq x (max (car range)
			  (logior (logand empire-x-start
					  (lognot 1))
				  (logand (car range) 1))
			  ))
	     (setq stop-x (min (cdr range) empire-x-stop))
	     (while (<= x stop-x)
	       (empire-flash-coordinates (, label) x y)
	       (, body)
	       (setq x (+ 2 x)))
	     ))
       (setq y (1+ y))
       )
     (message "")
     )))

;;(walksects (list "%s,$s" x y) "hi there")
;;(mapsects '(lambda (x y) (list "%s,$s" x y)) "hi there")

(defmacro oursp (x y) ; implies landp
  ;; we only know the mobility of something if we own it
  (list 'recall-macro x y (position-of 'mob)))

(defmacro landp (x y)
  (` (let ((case-fold-search nil))
       (string-match empire-sector-types-regexp
		     (or (recall-macro (, x) (, y) 'des) ".")))))

(defmacro shipp (x y)
  (` (let ((case-fold-search nil))
       (string-match empire-ship-types-regexp
		     (or (recall-macro (, x) (, y) 'des) ".")))))

(defun mapsects-active (fcn &optional label)
  "Calls FCN on every sector in the effective bounds of the empire that has
a designation, is ours, and is land. Prints LABEL and sector number on the
command line as it goes. See record-des."
  (let (x
	(x-delta (abs (% empire-x-start 2)))
	(y empire-y-start)
	range
	stop-x
	)
    (setq empire-last-verb "")
    (while (< y empire-y-stop)
      (setq range (aref empire-row-ranges (hexy-to-rec y)))
      (if range				; need we process this row?
	  (progn
	    (setq x (max (car range)
			 ;;empire-x-start  ;; why does this break?
			 (logior (logand empire-x-start
						     (lognot 1))
					     (logand (car range) 1))
			 ))
	    (setq stop-x (min (cdr range) empire-x-stop))
	    (while (<= x stop-x)
	      (empire-flash-coordinates label x y)
	      ;;(valid-sector x y)
	      (if (oursp x y)
		  (funcall fcn x y))
	      (setq x (+ 2 x)))
	    ))
      (setq y (1+ y))
      )
    (message "")
    ))

(defmacro walksects-active (body &optional label)
  "Runs BODY on every sector in the effective bounds of the empire that has
a designation, is ours, and is land. Prints LABEL and sector number on the
command line as it goes. See record-des."
  (`
   (let (x
	 (y empire-y-start)
	 range
	 stop-x
	 )
     (setq empire-last-verb "")
     (while (< y empire-y-stop)
       (setq range (aref empire-row-ranges (hexy-to-rec y)))
       (if range			; need we process this row?
	   (progn
	     (setq x (max (car range)
			  ;;empire-x-start  ;; why does this break?
			  (logior (logand empire-x-start
					  (lognot 1))
				  (logand (car range) 1))
			  ))
	     (setq stop-x (min (cdr range) empire-x-stop))
	     (while (<= x stop-x)
	       (empire-flash-coordinates (, label) x y)
	       ;;(valid-sector x y)
	       (if (oursp x y)
		   (, body))
	       (setq x (+ 2 x)))
	     ))
       (setq y (1+ y))
       )
     (message "")
     )))

(defmacro walksects-inactive (body &optional label)
  "Runs BODY on every sector in the effective bounds of the empire that has
a designation, and is NOT ours. Prints LABEL and sector number on the
command line as it goes. See record-des."
  (`
   (let (x
	 (y empire-y-start)
	 range
	 stop-x
	 )
     (setq empire-last-verb "")
     (while (< y empire-y-stop)
       (setq range (aref empire-row-ranges (hexy-to-rec y)))
       (if range			; need we process this row?
	   (progn
	     (setq x (max (car range)
			  ;;empire-x-start  ;; why does this break?
			  (logior (logand empire-x-start
					  (lognot 1))
				  (logand (car range) 1))
			  ))
	     (setq stop-x (min (cdr range) empire-x-stop))
	     (while (<= x stop-x)
	       (empire-flash-coordinates (, label) x y)
	       ;;(valid-sector x y)
	       (if (not (oursp x y))
		   (, body))
	       (setq x (+ 2 x)))
	     ))
       (setq y (1+ y))
       )
     (message "")
     )))

(defmacro walksects-nontrivial-inactive (body &optional label)
  "Runs BODY on every sector in the effective bounds of the empire that has
a designation, is NOT ours, and which has more known about it than just
des. Prints LABEL and sector number on the command line as it goes. See
record-des." 
  (`
   (let (x
	 (y empire-y-start)
	 range
	 stop-x
	 )
     (setq empire-last-verb "")
     (while (< y empire-y-stop)
       (setq range (aref empire-row-ranges (hexy-to-rec y)))
       (if range			; need we process this row?
	   (progn
	     (setq x (max (car range)
			  ;;empire-x-start  ;; why does this break?
			  (logior (logand empire-x-start
					  (lognot 1))
				  (logand (car range) 1))
			  ))
	     (setq stop-x (min (cdr range) empire-x-stop))
	     (while (<= x stop-x)
	       (empire-flash-coordinates (, label) x y)
	       (let* (
		      (row (aref empire-sects (hexy-to-rec y)))
		      (data (if row (aref row (hexx-to-rec x))))	
		      )
		 (if (and data
			  (> (length data) 3)
			  ;;(not (oursp x y))
			  (not (aref data (position-of 'mob)))
			  )
		     (, body)))
	       (setq x (+ 2 x)))
	     ))
       (setq y (1+ y))
       )
     (message "")
     )))

(defun mapsects-reverse (fcn &optional label)
  "Calls FCN on every sector in the empire. Prints LABEL and sector number
on the command line as it goes."
  (let (x
	(y empire-y-stop)
	(stop empire-y-start)
	)
    (setq empire-last-verb "")
    (while (> y stop)
      (setq x (+ (% y 2) empire-x-start))
      (while (<= x empire-x-stop)
	(empire-flash-coordinates label x y)
	(funcall fcn x y)
	(setq x (+ 2 x)))
      (setq y (1- y))
      )
    (message "")
    ))

(defun mapsects-active-in-range (x y ran fcn &optional label)
  "Calls FCN on every sector that is ours within rangy of x y.
Prints LABEL and sector number on the command line as it goes."
  (let (yy xx stopx)
    (setq yy (- y ran))
    (while (<= yy (+ y ran))
      (setq xx (+ (- x (* ran 2)) (abs (- y yy))))
      (setq stop-x (- (+ x (* ran 2)) (abs (- y yy))))
      (while (<= xx stop-x)
	(if (oursp xx yy)
	    (progn
	      (if empire-interaction-verbosity
		  (message "%s on %s %s actual range %s from %s,%s"
			   label xx yy (empire-mapdist x y xx yy) x y))
	      (funcall fcn xx yy)))
	(setq xx (+ 2 xx))
	)
      (setq yy (1+ yy))
      ))
  (message ""))

;;(mapsects-active-in-range -2 0 2 '(lambda (x y) (show-pt (cons x y))(sit-for 1)) "testing")

(defun member (x y)
  "Like memq, but uses	equal  for comparison"
  (while (and y (not (equal x (car y))))
    (setq y (cdr y)))
  y)

(defun mapsects-until (dest action acceptance-condition halt-criterion)
  "Calls ACTION on every sector connected to DEST by high efficiency
   roadways provided that (HALT-CRITERION DEST) is false and
   (ACCEPTANCE-CRITERION sector) is true.
     Action is called with the args X, Y, DEST, PATH where X and Y are of
   the section on which the action is being called and PATH is the path
   from the action sector to DEST.

   Is currently not implimented, the code you see is only partially ok.
   If this was implimented, it could serve the basis of commands such as
     'populate sector X,Y taking at most 90 from sectors with >900 civs and
  >100 mob'"
  (let (sects
	sects1
	(here dest)
	(depth 0)
	next
	path)
    (setq empire-last-verb "")
    ;;(princ (format "Am looking at %s -> %s\n" from to))
    ;; first, see if we are it
    (if (funcall halt-criterion here)
	(progn
	  ;;(princ (format "  %s is in it\n" from))
	  )
      ;; next, see if we are one away
      (setq sects1 (setq sects (sects-near from)))
      (while (and sects1 (not next))
	(setq next (car sects1))
	(if (equal next to)
	    (progn
	      ;;(princ (format "  %s is next to it\n" from))
	      (setq next (list from to))
	      (setq sects1 nil))
	  (setq next nil)
	  (setq sects1 (cdr sects1))))
      (if next next
	;;(princ (format "   %s is not next to it\n" from) )
	;; Place ourselves on the examined list
	(setq examined (cons from examined))
	;; Try each unexamined highway neighbor
	(while (and sects (not path))
	  (setq next (car sects))
	  (if (and (not (member next examined))
		   (string-equal (recall (car next) (cdr next) des) "+")
		   (recall (car next) (cdr next) mob)
		   (< 90 (recall (car next) (cdr next) eff))
		   )
	      ;; This might lead to the way
	      (setq depth 0)
	      (setq path (find-route-internal next to))
	    ;;(princ (format "	 %s is already examined or not a road\n" next))
	    )
	  (setq sects (cdr sects))
	  )
	;;(if path
	    ;;(princ (format "Found route through %s  %s\n"  from next))
	    ;;(princ (format "%s is not a route\n" from)))
	(if path (cons from path))
	))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Mapping in the visual sense
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-y-to-map-row (y)
  "Returns cursor line # corresponding to sectors with Y offset."
  (+ y (- 0 empire-y-start) 3))

(defun empire-x-to-map-column (x)
  "Returns cursor column # corresponding to sectors with X offset."
  (+ x (- 0 empire-x-start) tab-width))

(defun map-column-to-empire-x (col)
  (+ (- col tab-width) empire-x-start))

(defmacro map-x ()(list 'map-column-to-empire-x (list 'current-column)))
(defmacro map-y ()
	(list 'map-line-to-empire-y (list 'save-restriction
				   (list 'widen)
				   (list 'save-excursion
				     (list 'beginning-of-line)
      
				     (list '1+ (list 'count-lines 1
						     (list 'point)))))))

(defun empire-prompt-read-xy (&optional x y)
  "Get a simple x,y coordinate from the user, using `X' and `Y' as
default values.  The x,y coordinate is returned as an cons cell (x . y)."
  (if (not (and x y empire-map-buffer))
      (let ((cb (current-buffer)))
	(set-buffer empire-map-buffer)
	(if (not x)
	    (setq x (map-x)))
	(if (not y)
	    (setq y (map-y)))
	(set-buffer cb)))
  (let (str)
    ;;(setq str (read-from-minibuffer "X,Y location? " (format "%s,%s" x y)))
    (setq str (read-string (format "X,Y location (default %s,%s)? " x y)))
    (if (not (string-match "^[ \t]*$" str))
	(progn
	  (if (not (string-match
		    "[ \t]*\\(-?[0-9]+\\)[ \t]*,[ \t]*\\(-?[0-9]+\\)[ \t]*"
		    str))
	      (error "Illegal coordinate.")
	    )
	  (setq
	   x (car (read-from-string
		   (substring str (match-beginning 1) (match-end 1))))
	   y (car (read-from-string
		   (substring str (match-beginning 2) (match-end 2))))
	   )))
    (if (= (logand (logxor x y) 1) 1)
	(error "Illegal coordinate.")
      )
    (cons x y)
    )
  )

(defun empire-prompt-read-xy-list (&optional x y)
  "Just like `empire-prompt-read-xy' except that the coordinates are returned
in a list, instead of a cons cell."
  (let (sect)
    (setq sect (empire-prompt-read-xy x y))
    (list (car sect) (cdr sect))		;;; wasteful of memory
    )
  )


(defmacro empire-switch-to-buffer-not-map (buffer)
  "Switch to and display buffer BUFFER, but NOT in the current window if
the current window is displaying the map buffer."
  (`
   ;;
   ;; Here, we use `get-buffer' to insure that we compare buffer datatypes,
   ;; and not buffer string names.
   ;;
   (if (eq (current-buffer) (get-buffer empire-map-buffer))
       (pop-to-buffer (, buffer))
     (switch-to-buffer (, buffer))
     )
   )
  )
  
(defun map-line-to-empire-y (line)
  (- (+ line empire-y-start) 3))

(defun princ-map-empire-x-line ()
  (let ((x empire-x-start))
    (princ "\t")
    (while (<= x empire-x-stop)
      (princ (format "%s" (abs (% (/ x 10) 10))))
      (setq x (1+ x)))
    (princ "\n\t")
    (setq x empire-x-start)
    (while (<= x empire-x-stop)
      (princ (format "%s" (abs (% x 10))))
      (setq x (1+ x)))
    (princ "\n")))
  
(defun map-empire ()
  "Shows all that is known about the world map.  The map has special
keybindings and can also support mouse clicks.  Do a mode-help from the map
window to learn what the map can do for you."
  (interactive)
  (setq pending-map-redesignations nil)
  (let* (
	 (cb (current-buffer))
	 (x empire-x-start)
	 (y empire-y-start)
	 (bw (if empire-map-buffer (get-buffer-window empire-map-buffer)))
	 (old-start (if bw (window-start bw)))
	 (old-hscroll (if bw (window-hscroll bw) ))
	 row
	 data
	 xo)
    (with-output-to-temp-buffer "*Map*"
      (set-buffer "*Map*")
      (setq empire-map-buffer (current-buffer))
      (setq highlighted-map nil)	; it's not highlighted any more!
      (if overlay-arrow-position
	  (set-marker overlay-arrow-position 0 (current-buffer)))
      (empire-map-mode)

      (princ-map-empire-x-line)
      ;;(sit-for 0)

      (while (< y empire-y-stop)
	(if empire-interaction-verbosity
	    (message "Mapping %s ..." y))
	(if (= (% y 2) 0) 
	    ;; make x be even
	    (setq x  (+ empire-x-start (abs (% empire-x-start 2))))
	  ;; else make x be odd
	  (setq x (+ empire-x-start (- 1 (abs (% empire-x-start 2)))))
	  )
	(insert (if (= x empire-x-start)
		    (format "%s\t" y);; no extra space
		  (format "%s\t " y);; extra space
		  ))
	(if (setq row (aref empire-sects (+ y empire-height)))
	    (progn
	      (setq xo (/ (+ x empire-width) 2))
	      (while (<= x empire-x-stop)
		;;(insert (or (recall-macro x y (position-of 'des)) " ") " ")
		(setq data (aref row xo))
		(insert (if data (aref data (position-of des)) " ") " ")
		(setq x (+ 2 x))
		(setq xo (1+ xo)))))
	(insert "\n")
	(setq y (1+ y))
	;;(sit-for 0)
	)

      (princ-map-empire-x-line)

      (if empire-development-mode	; explain this?
	  (empire-display-dev-sectors-if-necessary))
      )

    ;; If the map is visible somewhere, get back to it
    (if bw
	;; try to restore map view to what it was before
	(progn
	  (set-window-start bw old-start)
	  (set-window-hscroll bw old-hscroll)
	  ))

    ;;
    ;; Move point to the last map location, or to the center of the map,
    ;; if there is no "last map location".  
    ;;
    (setq x empire-current-map-x
	  y empire-current-map-y)
  
    (if (or (not x) (not y))		; handle center of map case
	(progn
	  (setq x (+ (/ (- empire-x-stop empire-x-start) 2) empire-x-start))
	  (setq y (+ (/ (- empire-y-stop empire-y-start) 2) empire-y-start))
	  (if (= 0 (% y 2))
	      (setq x  (+ x (abs (% x 2))));; make x be even
	    (setq x (+ x (- 1 (abs (% x 2)))));; else make x be odd
	    )
	  (move-to-in-empire-map x y)
	  (set-marker overlay-arrow-position (point) (current-buffer))
	  (show-map (point))
	  (point-wysiwyg)
	  )
      )

    ;;If the last map location is not visible, move as close to it as possible.
    (move-to-in-empire-map x y)		; this takes care of "closest" visible
					; sector positioning

    (set-buffer cb)
    )
  (message ""))

(defun map-air-umbrella (&optional arg)
  "Shows the defensive air coverage on a scale of blank, 0-9 for all sectors.
If given a universal arg, calcs the umbrella first (slow).
   Uses variable empire-air-defense-incr."
  (interactive "P")
  (require 'emp-anal)
  (if arg (calc-defensive-air-umbrella))
  (message "Air Defense Mapping ...")
  (with-output-to-temp-buffer "*Air Defense*"
    (let ((cb (current-buffer)))
      (set-buffer "*Air Defense*")
      (empire-map-mode)
      (set-buffer cb))

    (let ((x empire-x-start)
	  (y empire-y-start)
	  extra
	  line)

      (princ-map-empire-x-line)
      (sit-for 0)

      (while (< y empire-y-stop)
	(message "Air Defense Mapping %s" y)
	(if (= 0 (% y 2))
	    ;; make x be even
	    (setq x  (+ empire-x-start (abs (% empire-x-start 2))))
	  ;; else make x be odd
	  (setq x (+ empire-x-start (- 1 (abs (% empire-x-start 2)))))
	  )
	(setq line (if (= x empire-x-start)
		       (format "%s\t" y);; no extra space
		     (format "%s\t " y);; extra space
		     ))
	(while (<= x empire-x-stop)
	  (setq line (concat line (or (recall-macro x y (position-of 'des)) " ")
			     (if (setq intercept (recall-macro x y (position-of 'intercept)))
				   (make-string
				    1
				    (+ 48 ;; "0"
				       (/ (min
					   (- (* 10 empire-air-defense-incr) 1)
					   (car intercept))
					  empire-air-defense-incr)))
			       " ")
			     ))
	  (setq x (+ 2 x)))
	(princ (concat line "\n"))
	(setq y (1+ y))
	(sit-for 0))

      (princ-map-empire-x-line)

      ))
  (message ""))
(put 'map-air-umbrella 'empire t)
	    
(defvar empire-global-desc-fcn 'describe-sect-prod-dist
  "Function normally called from show-pt to describe sector.")
(put 'empire-global-desc-fcn 'empire-system t)

(defun describe-sect-str (x y)
   (funcall empire-global-desc-fcn x y))

(defun describe-sect (x y)
   (message "%s" (funcall empire-global-desc-fcn x y)))

(defvar empire-sector-desc-options nil
  "A list of all the functions which will describe a sector")

(defun register-describe-sector-fcn (name)
  "Makes named fcn be a describe-sector option"
  (if (not empire-global-desc-fcn)
      (setq empire-global-desc-fcn name))
  (if empire-sector-desc-options
      (if (not (memq name empire-sector-desc-options))
	  (setq empire-sector-desc-options
		(nconc  empire-sector-desc-options
			(cons name nil))))
    (setq empire-sector-desc-options (cons name nil))
    )
  empire-sector-desc-options
  )

(defun empire-toggle-global-desc-fcn ()
  "Toggle the sector description function among the different alternatives.
Show the new sector when done toggling."
  (interactive)
  (setq empire-sector-desc-options
	(nconc (cdr empire-sector-desc-options)
	       (cons (car empire-sector-desc-options) nil)))
		
  (setq empire-global-desc-fcn
	(car empire-sector-desc-options))
  (show-pt (cons empire-current-map-x empire-current-map-y)))

(defun sector-contents-string ()
  (let (amt)
	    (mapconcat
	     (function (lambda (item)
		(if (and
		     (setq amt (recall x y (dynamic-position-of (car item))))
		     (> amt 0))
		    (format " %s:%s" (cdr item) amt))))
	     '((pet . p)
	       (oil . o)
	       (iron . i)
	       (lcm . l)
	       (hcm .h)
	       (gun . g)
	       (shell . s)
	       (dust . d)
	       (bar . b)
	       (rad . r)
	       ) "")))

(defun sector-basic-string (x y)
  (format "%s,%s\t%s %s%s%%(%s)%s"
  	  x y
	  (or (recall x y (position-of 'des)) "?")
	  (if (and (recall x y (position-of 'work))
		   (> 100 (recall x y (position-of 'work))))
	      (format "<%s>" (recall x y (position-of 'work)))
	    "")
	  (or (recall x y (position-of 'eff)) "?")
	  (let (owner)
	    (or (recall x y (position-of 'mob))
		(if (and (setq owner (recall x y (position-of 'own)))
			 (or empire-display-numeric-unknown-ownership
			     (not (eq owner 999))
			     )
			 )
		    (format "Own:%s" (recall x y (position-of 'own)))
		  "Own:???")))
	    (if (string-equal (recall x y (position-of '*)) "*") "*" "")
	    ))

(defun describe-sect-prod-dist (x y)
  (valid-sector x y)
  (format "%s\t%s/%s/%s %sf  %s %s Dist: %s,%s%s"
	  (sector-basic-string x y)
	  (or (recall x y (position-of 'civ)) "?")
	  (or (recall x y (position-of 'mil)) "?")
	  (or (recall x y (position-of 'uw)) "?")
	  (or (recall x y (position-of 'food)) "?")
	  (or (recall x y (position-of 'will)) "")
	  (or (recall x y (position-of 'make)) "")
	  (or (recall x y (position-of 'dist_x)) "")
	  (or (recall x y (position-of 'dist_y)) "")
	  (sector-contents-string)
	  ))
(register-describe-sector-fcn 'describe-sect-prod-dist)

(defun describe-sect-contents (x y)
  (valid-sector x y)
  (format "%s\t%s"
	  (sector-basic-string x y)
	  (sector-contents-string)
	  ))
(register-describe-sector-fcn 'describe-sect-contents)

(defun describe-sect-resources (x y)
  "Describe sector with the focus on resources instead of people."
  (valid-sector x y)
  (format "%s\tmin:%s gld:%s frt:%s oct:%s urn:%s %s"
	  (sector-basic-string x y)
	  (or (recall x y (position-of 'min)) "?")
	  (or (recall x y (position-of 'gold)) "?")
	  (or (recall x y (position-of 'fert)) "?")
	  (or (recall x y (position-of 'ocontent)) "?")
	  (or (recall x y (position-of 'uran)) "?")
	  (sector-contents-string)
	  ))
(register-describe-sector-fcn 'describe-sect-resources)

(defun describe-sect-ideals-plague (x y)
  "Describe sector with the focus on ideal levels and plague chance."
  (valid-sector x y)
  (format "%s Plague chance: %s%%"
	  (describe-sect-ideals x y)
	  (empire-calc-plague-chance x y)))
(register-describe-sector-fcn 'describe-sect-ideals-plague)

(defun show-map (point)
  "Moves map's concept of highlight spot, Highlights point (emacs int) on the map"
  (if (and empire-attempt-highlighting
	   (fboundp 'unhighlight-region)) ; some systems cannot take this -- use
					; any highlight code that works for you
      (progn
	(if (and highlighted-map
		 (> (point-max) highlighted-map)
		 (< (point-min) highlighted-map))
	    (unhighlight-region highlighted-map (1+ highlighted-map)))
	(sit-for 0)
	(highlight-region point (1+ point))))
  ;; in either case, set where the map thinks it is

  ;; use the overlay arrow
  (goto-char point)
  (beginning-of-line)
  (set-marker overlay-arrow-position (point) (current-buffer))

  ;; first, undo the bars
  (if highlighted-map
      (progn
	(goto-char highlighted-map)
	(if (eql (preceding-char) ?|)
	    (progn
	      (delete-char -1)
	      (insert " ")
	      ))
	(forward-char 1)
	(if (eql (following-char) ?|)
	    (progn
	      (delete-char 1)
	      (insert " ")
	      ))
	))

  (goto-char point)
  (setq highlighted-map point)
  (if (not (= point (or (mark) 0))) (push-mark highlighted-map t))

  ;; Use "|" to show sector
  (if empire-attempt-map-bars
      (progn
	(if (eql (preceding-char) ? )
	    (progn
	      (delete-char -1)
	      (insert "|")
	      ))
	(forward-char 1)
	(if (eql (following-char) ? )
	    (progn
	      (delete-char 1)
	      (insert "|")
	      ))))
  (goto-char point)
  )

(defun move-to-column-force (column)
  "Move to column COLUMN in current line.
Differs from move-to-column in that it creates or modifies whitespace
if necessary to attain exactly the specified column.

This version (non-standard) insures that the column is visible,
scrolling if needed."
  (move-to-column column)
  (let ((col (current-column)))
    (if (< col column)
	(indent-to column)
      (if (and (not (= col column))
	       (= (preceding-char) ?\t))
	  (let (indent-tabs-mode)
	    (delete-char -1)
	    (indent-to col)
	    (move-to-column column)))))
  (point-wysiwyg)
  )

(defun point-wysiwyg ()
  "Scrolls the window horozontally to make point visible"
  (interactive)
  (let*	 ((min (window-hscroll))
	  (max (- (+ min (window-width)) 2))
	  (here (current-column))
	  (delta (/ (window-width) 2))
	  )
    (if (< here min)
	(scroll-right (max 0 (+ (- min here) delta)))
      (if (>= here  max)
	  (scroll-left (- (- here min) delta))
	))))
(put 'point-wysiwyg 'empire t)
  
(defun window-wysiwyg-point ()
  "Makes point become the visible point
   Moves point, not the scroll."
  (interactive)
  (let*	 ((min (window-hscroll))
	  (max (+ min (window-width)))
	  (here (current-column))
	  (delta (/ (window-width) 2))
	  )
    ;;(message "here=%s min=%s max=%s" here min max)
    ;;(sleep-for-millisecs 4000)
    (if (< here min)
	(move-to-column min)
      (if (>= here  max)
	  (move-to-column (- max 3))
	))))

;;(defun point-wysiwyg-window ()
;;  "Makes point become the visible point
;;   Moves scroll, not the point."
;;  (interactive)
;;  (let ((start-col (max 0 (- (current-column) (/ (window-width) 2))))
;;	(current-start-col (window-hscroll)))
;;    (scroll-left (- current-start-col start-col)))
;;  (recenter))
(defvar empire-batch-play nil "Is t if running in batch mode")

(defun show-pt (pt &optional no-describe desc-fcn)
  "Shows the map, highlights pt (cons X Y), shows sector data in message line"
  (let (
	(current-buf (current-buffer))
	(current-window (selected-window))
	)
    ;; We can't use save-excursion, etc. here because we do not want any
    ;; values of point being saved.  We just want the current buffer/window
    ;; information saved.
    (unwind-protect
	(progn
	  (pop-to-buffer empire-map-buffer)
	  (if (and (not (input-pending-p))
		   (not no-describe)
		   (not empire-batch-play))
	      (if desc-fcn
		  (message "%s" (funcall desc-fcn (car pt) (cdr pt)))
		(describe-sect (car pt) (cdr pt))))
	  (goto-line (empire-y-to-map-row (cdr pt)))
	  (move-to-column-force (empire-x-to-map-column (car pt)))
	  (show-map (point))
	  (set-window-point (get-buffer-window empire-map-buffer) (point))
	  )
      (progn
	(select-window current-window)
	(set-buffer current-buf)
	)
      )
    )
  )

(defun empire-get-next-where ()
  "Return, as a cons cell, the sector coordinates after or around point.
If no coordinates could be found, nil is returned."
  (skip-chars-backward "-0123456789\\,")
  (if (re-search-forward "\\(-*[0-9]+\\),\\(-*[0-9]+\\)")
      (cons
       (read-buffer-num (match-beginning 1) (match-end 1))
       (read-buffer-num (match-beginning 2) (match-end 2))))
  )

(defun empire-show-next-where ()
  "Grabs the sector designation after or around point, shows it on the map
and shows the stored data."
  (interactive)
  (let (there)
    (if (setq there (empire-get-next-where))
      (show-pt there))))

(defun empire-show-previous-where ()
  "Grabs the sector designation before or around point, shows it on the map
and shows the stored data."
  (interactive)
  (save-excursion
    (skip-chars-forward "-0123456789\\,")
    (if (re-search-backward "\\(-*[0-9]+\\)\\,\\(-*[0-9]+\\)")
	(progn
	  (skip-chars-backward "-0123456789")	  
	  (empire-show-next-where))
      )))


(defvar pending-map-redesignations nil
  "List of sectors whose des has changed")

(defun record-des (x y new-des record-seas &optional no-more-to-come)
  "A special variant of record used only for sector designations.
Knows to set empire-row-ranges. Also knows empire logic such as to not
replace a known sector designation with a ?.

A special sector designation, ` (backquote), is supported by this empire
tool to stand for shipping lanes. These are essentially roadways at sea.

  Will show on the map as it happens if empire-map-des-interactively is t."
  (let* ((ny (hexy-to-rec y))
	(range (aref empire-row-ranges ny))
	(old-des (recall-macro x y (position-of 'des))))
    ;; First, have a special side effect on the range
    (if (not no-more-to-come)
	(if (not range)
	    (aset empire-row-ranges ny (cons x x))
	  (if (< x (car range))
	      ;; move lower bound
	      (aset empire-row-ranges ny (cons x (cdr range)))
	    (if (> x (cdr range))
		;; move upper bound
		(aset empire-row-ranges ny (cons (car range) x))
	      ))))

    ;; set the designation
    (if (or (not old-des)
	    (and (not (string-equal old-des new-des)) ; why bother?
		 (not (string-equal new-des "0")) ; a radar site

		 ;; Do not replace a ship with a dot if the . was read by a
		 ;; technique that does not see ships (such as maps)
		 (not (and (not record-seas) (string-equal new-des ".")))

		 ;; do not replace non ? with a ?, unless it is a wilderness
		 (or (not (string-equal new-des "?"))
		     (string-equal old-des "-")
		     (string-equal old-des "."))

		 ;; Do not overwrite an X or a ship lane
		 (not (or (string-equal old-des "X")
			  (and (string-equal old-des "`")
			       ;; except by another X
			       (not (string-equal new-des "X")))
			  ))

		 ;; do not replace harbor with a ship
		 (not (and (string-equal old-des "h")
			   (>= (aref new-des 0) ?A)
			   (<= (aref new-des 0) ?Z)))

		 ;; or non . with a sat ($)
		 (not (and (string-equal new-des "$")
			   (not (string-equal old-des ".")))
		      )
		 ))
	(progn
	  (if no-more-to-come
	      (record-max x y (position-of 'des) (position-of 'des) new-des)
	    (record x y (position-of 'des) new-des)
	    )
	  (if empire-map-des-interactively
	      (if (eql empire-map-des-interactively 0)
		  (setq pending-map-redesignations
			(cons (cons x y) pending-map-redesignations))
	      (map-des x y new-des)))))
    ))

(defun temp-des (x y des)
  "Replaces old des of x y with DES in the map on a 1 time only basis.
   A map redraw will resotre the old visible designations"
  (save-excursion
    (set-buffer empire-map-buffer)
    (goto-line (empire-y-to-map-row y))
    (move-to-column (empire-x-to-map-column x))
    (delete-char 1)
    (insert des)))

(defun refresh-map ()
  "Updates map to show changed data"
  (if (> (length pending-map-redesignations) 50)
      (map-empire)
    (while pending-map-redesignations
      (map-des (car (car pending-map-redesignations))
	       (cdr (car pending-map-redesignations))
	       (recall-macro (car (car pending-map-redesignations))
			     (cdr (car pending-map-redesignations))
			     (position-of 'des)))
      (setq pending-map-redesignations (cdr pending-map-redesignations)))
    (sit-for 0)))

(defun conditionally-refresh-map ()
  ; refreshes if empire-map-des-interactively is set and not 0
  (if (and empire-map-des-interactively
	   (not (eql 0 empire-map-des-interactively)))
      (refresh-map)))

;;(let ((des (position-of 'des)))
;;  (mapsects '(lambda (x y)
;;	       (if (recall x y des)
;;	       (record-des x y (recall x y des))))))
;; (with-output-to-temp-buffer "*Temp*" (mapsects '(lambda (x y) (princ (format "\t%s,%s\n" x y))) "Temp" ))
;;  (setq empire-row-ranges (make-vector (* empire-height 2) nil))
;;empire-row-ranges
	  
(defun map-des (x y des)
  "Shows a new des on the map in inverse-video"
  (if empire-map-buffer
      (progn
	(let ((cb (current-buffer))
	      (inverse-video t)
	      pt)
	  (set-buffer empire-map-buffer)
	  (setq pt (point))
	  (goto-line (empire-y-to-map-row y))
	  (move-to-column (empire-x-to-map-column x))
	  (if t; (pos-visible-in-window-p (point) (selected-window))
	      (progn
		;;(delete-char 1)
		;;(insert " ")
		;;(sit-for 0)
		(delete-char 1)
		(insert des)
		(sit-for 0)))
	  (goto-char pt)
	  (set-buffer cb)))))



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector and direction manip
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun normalize-x (x) 
  (if (>= x empire-width)
      (- x (* 2 empire-width))
    (if (< x (- 0 empire-width))
	(+ x (* 2 empire-width))
      x)))

(defun normalize-y (y) 
  (if (>= y empire-height)
      (- y (* 2 empire-height))
    (if (< y (- 0 empire-height))
	(+ y (* 2 empire-height))
      y)))

(defun normalize-sect (pt)
  "Wraps a sector designation if normal math crossed a border"
  (if (>= (car pt) empire-width)
      (setcar pt (- (car pt) (* 2 empire-width))))
  (if (< (car pt) (- 0 empire-width))
      (setcar pt (+ (car pt) (* 2 empire-width))))
  (if (>= (cdr pt) empire-height)
      (setcdr pt (- (cdr pt) (* 2 empire-height))))
  (if (< (cdr pt) (- 0 empire-height))
      (setcdr pt (+ (cdr pt) (* 2 empire-height))))
  pt
  )

(defun valid-sector (x y)
  (if (not (= 0 (% (+ (% x 2) (% y 2)) 2)))
      (error "%s,%s is not a valid sector number" x y)))

(defun visually-valid-sector (x y)
  (if (or (not (= 0 (% (+ (% x 2) (% y 2)) 2)))
	  (> x empire-x-stop)
	  (> y empire-y-stop)
	  (< x empire-x-start)
	  (< y empire-y-start))
      (error "%s,%s is not a valid displayed sector number" x y)))

(defun visually-normalize-sect (pt)
  "Wraps a sector designation if it is outside a bounds border"
  (if (> (car pt) empire-x-stop) (setcar pt
					 (+ empire-x-start
					    (abs (% (+ (% (cdr pt) 2)
						       (% empire-x-start 2))
						    2)))
					 ))
  (if (< (car pt) empire-x-start) (setcar pt (- empire-x-stop
					    (abs (% (+ (% (cdr pt) 2)
						       (% empire-x-stop 2))
						    2)))
					  ))
  (if (>= (cdr pt) empire-y-stop) (setcdr pt empire-y-start))
  (if (< (cdr pt) empire-y-start) (setcdr pt empire-y-stop))
  pt
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Nearby sectors
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun sects-near (pt)
  (list
   (normalize-sect (cons (+ (car pt) 2) (cdr pt)));; j
   (normalize-sect (cons (+ (car pt) -2) (cdr pt)));; g
   (normalize-sect (cons (+ (car pt) 1) (+ (cdr pt) 1)));; b
   (normalize-sect (cons (+ (car pt) 1) (+ (cdr pt) -1)));; u
   (normalize-sect (cons (+ (car pt) -1) (+ (cdr pt) 1)));; n
   (normalize-sect (cons (+ (car pt) -1)(+ (cdr pt) -1)));; y
   ))

(defun owned-sects-near (pt)
  (delq nil (mapcar (function (lambda (pt)
				(if (oursp (car pt) (cdr pt))
				    pt)))
		    (sects-near pt))))

(defun owned-roads-near (pt);; not bridges
  (delq nil (mapcar (function (lambda (pt)
				(if (and (oursp (car pt) (cdr pt))
					 (string-equal (recall-macro (car pt) (cdr pt) 'des) "+"))
				    pt)))
		    (sects-near pt))))

(defun sea-sects-near (pt)
  "returns all sea-sectors near to the location passed."
  (let (sea-sects)
    (dolist (next (sects-near pt) sea-sects)
      (if (string-equal "." (recall-macro (car next) (cdr next) 'des))
	  (setq sea-sects (cons next sea-sects))))))

(defun mtn-sects-near (pt)
  "returns all sea-sectors near to the location passed."
  (let (mtn-sects)
    (dolist (next (sects-near pt) mtn-sects)
      (if (string-equal "^" (recall-macro (car next) (cdr next) 'des))
	  (setq mtn-sects (cons next mtn-sects))))))

(defun unowned-wilderness-near (x y)
  (let ((sects))
    (mapcar (function (lambda (pair)
			(if (and (recall (car pair) (cdr pair) (position-of 'des))
				 (string-equal "-" (recall (car pair) (cdr pair) (position-of 'des)))
				 (not (oursp (car pair) (cdr pair))))
			    (setq sects (cons pair sects))
			  )))
	    (sects-near  (cons x y)))
    sects
    ))

(defun x-offset-in-dir (dir)
  "returns the delta x for going in direction DIR"
  ;;(nteractive "sdirection: ")
  ( cond
    ( (string-equal dir "y") -1 )
    ( (string-equal dir "u") +1 )
    ( (string-equal dir "g") -2 )
    ( (string-equal dir "j") +2 )
    ( (string-equal dir "b") -1 )
    ( (string-equal dir "n") +1 )
    ( (string-equal dir "h") 0 )
    ( t 0 )
    )
  )

(defun y-offset-in-dir (dir)
  "returns the delta y for going in direction DIR"
  ;;(nteractive "sdirection: ")
  ( cond
    ( (string-equal dir "y") -1 )
    ( (string-equal dir "u") -1 )
    ( (string-equal dir "g") 0 )
    ( (string-equal dir "j") 0 )
    ( (string-equal dir "b") +1 )
    ( (string-equal dir "n") +1 )
    ( (string-equal dir "h") 0 )
    ( t 0 )
    )
  )

(defmacro x-offset-in-char-dir (dir)
  "returns the delta x for going in direction DIR"
  (` ( cond
       ( (= (, dir) ?y) -1 )
       ( (= (, dir) ?u) +1 )
       ( (= (, dir) ?g) -2 )
       ( (= (, dir) ?j) +2 )
       ( (= (, dir) ?b) -1 )
       ( (= (, dir) ?n) +1 )
       ( t 0 )
       )
     ))

(defmacro y-offset-in-char-dir (dir)
  "returns the delta y for going in direction DIR"
  (` ( cond
       ( (= (, dir) ?y) -1 )
       ( (= (, dir) ?u) -1 )
       ( (= (, dir) ?g) 0 )
       ( (= (, dir) ?j) 0 )
       ( (= (, dir) ?b) +1 )
       ( (= (, dir) ?n) +1 )
       ( t 0 )
       )
     ))

(defmacro sector-in-direction (x y dir)
  "returns the sector in direction DIR from X,Y in
   a cons cell (NX . NY).  DIR is a char"
  ;;(nteractive "nX: \nnY: \nsdirection: ")
  (`
   (let ( (nx (, x))
	  (ny (, y))
	  (dir (, dir))
	  (valid t)
	  )
     ( cond
       ( (eq dir ?y)
	 (setq nx (1- nx))
	 (setq ny (1- ny))
	 )
       ( (eq dir ?u)
	 (setq nx (1+ nx))
	 (setq ny (1- ny))
	 )
       ( (eq dir ?g)
	 (setq nx (- nx 2))
	 )
       ( (eq dir ?j)
	 (setq nx (+ nx 2))
	 )
       ( (eq dir ?b)
	 (setq nx (1- nx))
	 (setq ny (1+ ny))
	 )
       ( (eq dir ?n)
	 (setq nx (1+ nx))
	 (setq ny (1+ ny))
	 )
       ( (eq dir ?h)
	 )
       ( t
	 (setq valid nil)
	 )
       )
     (if valid (cons nx ny))
     )
   ))

(defun empire-opposite-dir (dir)
  "accepts a direction in a string and returns the one
 in the opposite direction"
  (cond
   ( (string= "y" dir) "n" )
   ( (string= "u" dir) "b" )
   ( (string= "j" dir) "g" )
   ( (string= "n" dir) "y" )
   ( (string= "b" dir) "u" )
   ( (string= "g" dir) "j" )
   ))

(defun empire-reciprocal-path (path)
  ;;(nteractive "spath: ")
  (let ( (rval "") )
    (while (< 0 (length path))
      (setq rval (concat (empire-opposite-dir (substring path 0 1)) rval))
      (setq path (substring path 1))
      )
    rval
    ))

(defun r-to-path (r)
  "Takes list of sectors and makes empire path with termination"
  (concat (r-to-spath r) "h")
  )

(defun r-to-spath (r)
  "Takes list of sectors and makes empire path with no termination"
  (let ((result "")
	(sect-list r)
	s
	delta-x delta-y)
    (while (and sect-list (cdr sect-list))
      (setq delta-x (- (car (car sect-list)) (car (car (cdr sect-list)))))
      (setq delta-y (- (cdr (car sect-list)) (cdr (car (cdr sect-list)))))

      ;; reverse deltas if they wrapped
      (if (> (abs delta-x) empire-width)
	  (setq delta-x (- 0 delta-x)))
      (if (> (abs delta-y) empire-height)
	  (setq delta-y (- 0 delta-y)))
      (setq s
	    (cond
	     ((= 0 delta-y)
	      (if (< delta-x 0)
		  "j" "g"))
	     ((> delta-y 0)
	      (if (< delta-x 0)
		  ;; did they wrap?
		  "u" "y"))
	     (t
	      (if (< delta-x 0)
		  "n" "b"))
	     ))
      (setq result (concat result s))
      (setq sect-list (cdr sect-list))
      )
    result
    ))

(defun path-to-r (path sect)
  (let ((route (list sect))
	scan
	(index 0)
	(len (length path))
	dir
	)
    (setq scan route)
    (while (and (< index len)
		(not (eq (setq dir (aref path index)) ?h)))
      (setcdr scan (list (setq sect (sector-in-direction (car sect)
							 (cdr sect)
							 dir))))
      (setq scan (cdr scan)
	    index (1+ index))
      )
    route
    ))

;;(setq path "jjjuuunnngggbbbyyy")
;;(setq path "jjjuuunnngggbbbyyyh")
;;(setq path "jjjuuunnnhgggbbbyyyh")
;;(equal (path-to-r path (cons 0 0)) (path-to-r2 path (cons 0 0)))

(defun destination-of (path sect)
  "Given a path and a starting sector, tells which sector the path leads
   to. Does not check validity of the sectors (ocean, occupied, etc." 
  (let ( (index 0)
	 (len (length path))
	 dir
	 )
    (while (and (< index len)
		(not (eq (setq dir (aref path index)) ?h)))
      (setq sect (sector-in-direction (car sect) (cdr sect) dir)
	    index (1+ index))
      )
    sect
    ))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Auto route finding
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun find-route (from to)
  "Returns list of empire sectors that cover FROM to TO using only highways."
  (let ((examined  (list from))
	(des (position-of 'des))
	(mob (position-of 'mob))
	(eff (position-of 'eff))
	(depth 0)
	)
    (find-route-internal from to)))

(defun find-route-internal (from to)
  (let (sects sects1
	      next
	      path)
    (if (> (setq depth (1+ depth)) empire-max-path-depth)
	nil
      ;; first, see if we are it
      (if (equal from to)
	  (progn
	    ;;(princ (format "  %s is in it\n" from))
	    (setq depth (1- depth))
	    to)
	;; next, see if we are one away
	(setq sects1 (setq sects (sects-near from)))
	(while (and sects1 (not next))
	  (setq next (car sects1))
	  (if (equal next to)
	      (progn
		;;(princ (format "  %s is next to it\n" from))
		(setq next (list from to))
		(setq sects1 nil))
	    (setq next nil)
	    (setq sects1 (cdr sects1))))
	(if next
	    (progn
	      (setq depth (1- depth))
	      next)
	  ;;(princ (format "   %s is not next to it\n" from) )
	  ;; Place ourselves on the examined list
	  (setq examined (cons from examined))
	  ;; Try each unexamined highway neighbor
	  (while (and sects (not path))
	    (setq next (car sects))
	    (if (and (not (member next examined))
		     (or (string-equal (recall (car next) (cdr next) des) "+")
			 (and empire-route-past-bridges
			      (string-equal (recall (car next) (cdr next) des)
					    "=")))
		     (recall (car next) (cdr next) mob)
		     (< empire-highway-min-eff (recall (car next) (cdr next) eff))
		     )
		;; This might lead to the way
		(setq path (find-route-internal next to))
	      ;;(princ (format "	 %s is already examined or not a road\n" next))
	      )
	    (setq sects (cdr sects))
	    )
	  (setq depth (1- depth))
	  (if path (cons from path))
	  )))))

(defun last-sector-reference ()
  "Returns a cons cell of the last sector referenced in the current buffer"
  (save-excursion (re-search-backward "\\( -*[0-9]+\\),\\(-*[0-9]+\\)" nil t))
  (cons
	     (car (read-from-string (buffer-substring
				     (match-beginning 1)
				     (match-end 1))))
	     (car (read-from-string (buffer-substring
				     (match-beginning 2)
				     (match-end 2))))))

(defun empire-path (x y)
  "Finds path from the last sector mentioned in the current buffer to a
given X and Y."
  (interactive "nX: \nnY: ")
  (valid-sector x y)
  (let ((pt (last-sector-reference)))
    (message "Looking from %s,%s to %s,%s" (car pt) (cdr pt) x y)
    (insert-space-in-empire-buffer-if-necessary)
    (insert (r-to-path
	     (find-route
	      pt
	      (cons x y))))
    (message "")
    (delete-char -1)))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Distance
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-delta-x (x1 x2)
  "Returns x distance"
  (let ((dis (abs (- x2 x1))))
    (if (> dis empire-width)
	(- (* 2 empire-width) dis)
      dis)))
(defun empire-delta-y (y1 y2)
  "Returns y distance"
  (let ((dis (abs (- y2 y1))))
    (if (> dis empire-height)
	(- (* 2 empire-height) dis)
      dis)))
(defun empire-mapdist (x1 y1 x2 y2)
  (let ((dx (empire-delta-x x1 x2))
	(dy (empire-delta-x y1 y2)))
    (if (> dx dy)
	(+ dy (/ (- dx dy) 2))
      dy)))

(defun within-rangep (x y x2 y2 ran)  
  "Returns T if the first point is within range of the second"
  (>= ran (empire-mapdist x y x2 y2)))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Extractors
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-quick-read-dump ()
  "Reads dump data without applying any of the hooks."
  (interactive)
  (let ((empire-dump-hooks '()))
    (empire-read-dump)))

(defmacro following-following-char ()
  (`
    (let (ch)
      (if (not (setq ch (char-after (1+ (point)))))
	(setq ch 0)
      )
      ch
    )
  )
)

(defmacro following-is-numericp ()
  (` (or (and (eq (following-char) ?-)
	      (and (>= (following-following-char) ?0)
		   (<= (following-following-char) ?9)))
      (and (>= (following-char) ?0)
	   (<= (following-char) ?9)))))

(defun empire-read-dump ()
  "Reads the next dump output. As each sector is read, run
empire-dump-hooks on it. All dump values will be bound at this time, even
if they are not perminately stored in the empire sectors. 

Hook Writters Note: the PREVIOUS values are available via recall.

Things checked:
  Levels and Commodities:
     Levels appropiate to the designation (i.e. no oil in parks)
     commodities appropiate to the designation
    Find efficient oil, gold, or uran sectors with no production -- these
     are out of resources and need redesignation
    Find unused/underused libraries/parks/etc
  Work: unhappy campers are detected.
  Delta: changes in civs, work, food, etc can be detected.
 "
  (interactive)
  (setq empire-last-verb "")
  (if (re-search-forward "^DUMP SECTOR$" nil t)
      (with-output-to-temp-buffer "*Dump*"
	(princ (format "\t\tEmpire Dump Observations as of %s\n"
		       (current-time-string)))
	(princ "These are suggested actions or noted problems the dump scan found using the\n")
	(princ (format "current dump hooks of %s\n\n" empire-dump-hooks))
	(empire-data-mode-buffer t t "*Dump*")
	(let (name
	      positions
	      (standard-input (current-buffer))
	      val
	      vnumericp
	      vnullp
	      pos
	      val-pos-list
	      current-dump-hook
	      (empire-map-des-interactively 0)
	      )
	  (empire-pos-var-let
	   (forward-line 2)
	   (while (following-is-numericp)
	     (sit-for 0)
	     (setq val-pos-list nil
		   positions empire-pos)
	     (while (not (eq (following-char) ?\n))
	       ;; skip blanks
	       (while (= (following-char) ? )
		 (goto-char (1+ (point))))
	       (setq vnumericp (following-is-numericp)
		     vnullp (eq (following-char) ?.) ; seas never dump

		     ;; set all the temp variables
		     val (if vnullp
			     (progn (goto-char (1+ (point))) nil)
			   ;; we have either a number or a letter
			   (if vnumericp
			       (prog1 (read) (goto-char (1- (point)))) 
			     ;;(cdr (car positions))
			     (buffer-substring
			      (point)
			      (progn
				(while (and (not (= (following-char) ? ))
					    (not (= (following-char) ?\n))
					    )
				  (goto-char (1+ (point)))
				  )
				(point)))))
		     )
		     
	       ;; set the temp variable
	       (set (car (car positions)) ; variable name
		    val)

	       ;; do not record yet, but preserve the pair for later recording
	       (setq val-pos-list (cons (cons val
					      (cdr (car positions))) ;pos
					val-pos-list)
		     positions (cdr positions)
		     )
	       )
	     (empire-flash-coordinates "Reading dump" x y)
	     ;; At this point, there is a variable bound for each attribute
	     ;; available from the dump line. No values have been recorded.
	     ;; Call any local evaluation fcns!
	     (condition-case errorcode
		 (run-hooks 'empire-dump-hooks)
	       (error 
		(princ (format "%sEmpire rules had error %s in %s\n"
			       (format-xy x y)
			       errorcode
			       current-dump-hook))
		(backtrace)
		)
	       (void-function
		(princ (format "%sEmpire rules had error %s in %s\n"
			       (format-xy x y) errorcode
			       current-dump-hook))
		(backtrace)
		)
	       (t
		(princ (format "%sEmpire rules had unusual error %s in %s\n"
			       (format-xy x y)
			       errorcode
			       current-dump-hook))
		(backtrace)
		))

	     ;; Now, record the new values
	     ;;   special case -- record the des and get row-ranges ok
	     (record-des x y des t nil)
	     (while val-pos-list
	       (record x y (cdr (car val-pos-list)) (car (car val-pos-list)))
	       (setq val-pos-list (cdr val-pos-list)))
			
	     (forward-line 1))
	   ))
	(conditionally-refresh-map)	; update if needed
	(message "Done")
	)))

;;(defun experimental-empire-read-dump ()
;;  (interactive)
;;  (if (re-search-forward "^Dump Sector$" nil t)
;;      (with-output-to-temp-buffer "*Dump*"
;;	(princ (format "\t\tEmpire Dump Observations as of %s\n"
;;		       (current-time-string)))
;;	(princ "These are suggested actions or noted problems the dump scan found\n\n")
;;	(empire-data-mode-buffer t t "*Dump*")
;;	(let (name
;;	      positions
;;	      (standard-input (current-buffer))
;;	      dvec
;;	      bol
;;	      eol
;;	      val
;;	      vnumericp
;;	      vnullp
;;	      pos
;;	      (standard-input (current-buffer))
;;	      val-pos-list)
;;	  (empire-pos-var-let
;;	   (forward-line 2)
;;	   (while (following-is-numericp)
;;	     (setq bol (point))
;;	     (insert "[")
;;	     (end-of-line)
;;	     (insert "]")
;;	     (setq eol (point))
;;	     (goto-char bol)
;;	     (narrow-to-region bol eol)
;;	     (replace-string "." "nil")
;;	     (goto-char (point-min))
;;	     (setq dvec (read))
;;	     (widen)
;;	     (forward-line 1)
;;	     (sit-for 0)
;;	     (princ (format "Length is %s\n" (length dvec)))
;;	     ))
;;	  ))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Other Sector Data Extractors
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-read-census (&optional argp) 
  "Reads an empire census. If given an arg, highlight those sectors on the map."
  (interactive "P")
  (if (re-search-forward "^CENSUS[ \t]+del dst$" nil t)
      (let (x y type v-eff v-mob v-ep v-civ v-mil v-uw
	      v-food v-work v-avail v-ter
	      (empire-map-des-interactively 0)
	      map-window
	      cw cb
	      )
	(forward-line 2)
	(if argp
	    (progn
	      (setq cw (selected-window))
	      (setq cb (current-buffer))
	      (if (not (setq map-window (get-buffer-window empire-map-buffer)))
		  (progn
		    (empire-switch-to-map nil)
		    (setq map-window (get-buffer-window empire-map-buffer))
		    (select-window cw)
		    ))))
	(while 
	    (looking-at " *\\(-*[0-9]+\\),\\(-*[0-9]+\\) +\\(.\\)[a-z%\\+)\\*\\#]* +\\([0-9]+\\)% +\\([0-9]+\\) .. .. \\(.\\) +")
	  (setq x (read-buffer-num (match-beginning 1) (match-end 1)))
	  (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
	  (setq type (buffer-substring (match-beginning 3) (match-end 3)))
	  (setq v-eff (read-buffer-num (match-beginning 4) (match-end 4)))
	  (setq v-mob (read-buffer-num (match-beginning 5) (match-end 5)))
	  (setq v-ep (buffer-substring (match-beginning 6) (match-end 6)))
	  (goto-char (match-end 0))

	  (if (looking-at " *\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\)% +-*\\([0-9]+\\)")
	      (setq
	       v-civ (read-buffer-num (match-beginning 1) (match-end 1))
	       v-mil (read-buffer-num (match-beginning 2) (match-end 2))
	       v-uw (read-buffer-num (match-beginning 3) (match-end 3))
	       v-food (read-buffer-num (match-beginning 4) (match-end 4))
	       v-work (read-buffer-num (match-beginning 5) (match-end 5))
	       v-avail (read-buffer-num (match-beginning 6) (match-end 6))
	       )
	    (error "not looking")
	    )

	  (record-des x y type t nil)
	  (if argp
	      (progn
		(set-buffer empire-map-buffer)
		(goto-line (empire-y-to-map-row y))
		(move-to-column (empire-x-to-map-column x))
		(if (pos-visible-in-window-p (point) map-window)
		    (progn
		      (select-window map-window)
		      (delete-char 1)
		      (insert " ")
		      (sit-for 0)
		      (let ((inverse-video t))
			(delete-char -1)
			(insert type)
			(sit-for 0)
			)))
		(select-window cw)
		(set-buffer cb)))

	  (record x y (position-of 'eff) v-eff)
	  (record x y  (position-of 'mob) v-mob)
	  (record x y  (position-of '*) v-ep)
	  (record x y  (position-of 'civ) v-civ)
	  (record x y  (position-of 'mil) v-mil)
	  (record x y  (position-of 'uw) v-uw)
	  (record x y  (position-of 'food) v-food)
	  (record x y  (position-of 'work) v-work)
	  (record x y  (position-of 'avail) v-avail)

	  (sit-for 0)
	  (forward-line 1))
	))
  (conditionally-refresh-map)		; update if needed
  )

(defun empire-read-planes (&optional arg) 
  "Reads an empire plane report.  If given an arg, will forget ALL planes
first."
  (interactive "P")
  (if arg
      (setq empire-planes nil))
  (if (re-search-forward "#[ \t]+type[ \t]+x,y[ \t]+w[ \t]+eff[ \t]+mu[ \t]+att[ \t]+def[ \t]+tech[ \t]+ran[ \t]+hard[ \t]+ship[ \t]+nuke" nil t)
      (let (
 	    number type x y wing eff mu att def tech ran hard ship nuke
	    data hit
	    )
	(forward-line 1)
	(while (looking-at " *\\([0-9]+\\) \\(..............\\) +\\(-*[0-9]+\\),\\(-*[0-9]+\\) +\\([A-Z]*\\) +\\([0-9]+\\)% +\\(-*[0-9]+\\) +\\([0-9]+\\)")
	  (setq number (read-buffer-num (match-beginning 1) (match-end 1)))
	  (setq type (buffer-substring (match-beginning 2) (match-end 2)))
	  (setq	 x  (read-buffer-num (match-beginning 3) (match-end 3)))
	  (setq	 y  (read-buffer-num (match-beginning 4) (match-end 4)))
 	  (setq wing (if (= (match-beginning 5) (match-end 5))
 			 " "
 		       (buffer-substring (match-beginning 5) (match-end 5))))
	  (setq	 eff  (read-buffer-num (match-beginning 6) (match-end 6)))
	  (setq	 mu  (read-buffer-num (match-beginning 7) (match-end 7)))
	  (setq	 att  (read-buffer-num (match-beginning 8) (match-end 8)))
	  (goto-char (1+ (match-end 8)))

	  (looking-at " +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\)* +\\([0-9]+\\)*")
	  (setq	 def  (read-match-num 1))
	  (setq	 tech  (read-match-num 2))
	  (setq	 ran  (read-match-num 3))
	  (setq	 hard  (read-match-num 4))
 	  (let ((mb4 (match-beginning 5))
 		(me4 (match-end 5)))
 	    (setq ship (if (and mb4 (not (= mb4 me4))) (read-buffer-num mb4 me4) -1)))
 	  (setq nuke (if (match-beginning 6)
 			 (buffer-substring (match-beginning 6) (match-end 6))
 		       nil))

 	  (setq data (list number type x y wing
 			   eff mu att def tech ran hard ship nuke))

	  (if (setq hit (assoc number empire-planes))
	      (setq empire-planes (delq hit empire-planes)))
	  (setq empire-planes (cons data empire-planes))
	  (sit-for 0)
	  (forward-line 1))
	)))

(defun empire-read-version ()
  "Reads empire version info, sets up new world. 
   Must be executed when positioned somewhere BEFORE the output of a
version command."
  (interactive)
  (if (re-search-forward "^World size is \\([0-9]*\\) by \\([0-9]*\\)" nil t)
      (progn
	(setq
	 empire-width (/ (read-buffer-num (match-beginning 1) (match-end 1)) 2)
	 empire-height (/ (read-buffer-num (match-beginning 2) (match-end 2)) 2))
	(if (re-search-forward "^An Empire time unit is \\([0-9]*\\) seconds long."
			       nil t)
	    (setq seconds-per-time-unit
		  (read-buffer-num (match-beginning 1) (match-end 1)))
	  )
	(if (re-search-forward
	     "^An update consists of \\([0-9]*\\) empire time units"
	     nil t)
	    (setq
	     etu-per-update (read-buffer-num (match-beginning 1) (match-end 1))
	     upd-per-day (/ (/ (* 60 60 24) seconds-per-time-unit) etu-per-update))
	  )
	(if (re-search-forward
	     "^In one time unit, 1000 people eat 1.0 units of food."
	     nil t)
	    (setq bodies-fead-by-1-food-per-etu 1000)
	  (if (re-search-forward
	       "^In one time unit, 1000 people eat \\([0-9]*\\.[0-9]+\\) units of food."
	       nil t)
	      (progn
		(if (not (fboundp 'string-to-float))
		    (progn
		      ;;
		      ;; We try loading our own faster, yet compatible, version of float.el.
		      ;; If this fails, we load the normal one.
		      ;;
		      (condition-case nil
			  (load "emp-float")
			(load "float");; float.el does not provide anything
			)
		      )
		  )
		(let ((food (string-to-float (buffer-substring
					      (match-beginning 1)
					      (match-end 1)))))
		  (setq bodies-fead-by-1-food-per-etu
			(if (f> food _f0)
			    (fint (f/ (f 1000) food ))
			  100000))
		  ))
	    ;; else we cannot handle this -- emacs is int only
	    (message "Cannot parse the bodies-fead-by-1-food-per-etu amount, will take default")
	    (ding)
	    (sit-for 5)
	  
	    ))
	(if (re-search-forward
	     "^Happiness p.e. requires 1 happy stroller per \\([0-9]*\\) civ."
	     nil t)
	    (setq happy-divisor (read-buffer-num (match-beginning 1) (match-end 1)))
	  )
	(if (re-search-forward
	     "^Education p.e. requires 1 class of graduates per \\([0-9]*\\) civ." nil t)
	    (setq edu-divisor (read-buffer-num (match-beginning 1) (match-end 1)))
	  )
	t ;; search was ok
	)
    nil ;; did not find version commands
    ))

(defun empire-read-production ()
  "Reads next production command output"
  (interactive)
  (setq empire-last-verb "")
  (if (re-search-forward "^PRODUCTION" nil t)
      (progn
	(forward-line 2)
	(while 
	    (looking-at " *\\(-*[0-9]+\\),\\(-*[0-9]+\\) +\\([a-z=%]\\) +\\([0-9]+\\)% +\\([0-9]+\\) +\\([0-9\.]+\\) +\\([a-z]+\\) +\\([0-9\\.]+\\) +\\$\\([0-9]+\\) +")
	  (sit-for 0)
	  (let (x y
		  type-val
		  eff-val
		  wkfc-val
		  will-val
		  make-val
		  pe-val
		  cost-val
		  use1-val
		  use2-val
		  use3-val
		  max1-val
		  max2-val
		  max3-val
		  max-val
		  (wkfc (position-of 'wkfc))
		  (will (position-of 'will))
		  (make (position-of 'make))
		  (use1 (position-of 'use1))
		  (use2 (position-of 'use2))
		  (use3 (position-of 'use3))
		  (max1 (position-of 'max1))
		  (max2 (position-of 'max2))
		  (max3 (position-of 'max3))
		  (max (position-of 'max))
		  )

	    (setq x (read-buffer-num (match-beginning 1) (match-end 1)))
	    (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
	    (empire-flash-coordinates "Production" x y)
	    (setq
	     type-val (buffer-substring (match-beginning 3) (match-end 3))
	     eff-val (read-buffer-num (match-beginning 4) (match-end 4))
	     wkfc-val (read-buffer-num (match-beginning 5) (match-end 5))
	     will-val (read-buffer-num (match-beginning 6) (match-end 6))
	     make-val (buffer-substring (match-beginning 7) (match-end 7))
	     pe-val (read-buffer-num (match-beginning 8) (match-end 8))
	     cost-val (read-buffer-num (match-beginning 9) (match-end 9)))
	    (goto-char (match-end 0))

	    ;; convert the will to the proper single char value
	    (setq make-val
		  (cond
		   ((string-equal make-val "food") "f")
		   ((string-equal make-val "iron") "i")
		   ((string-equal make-val "dust") "d")
		   ((string-equal make-val "oil") "o")
		   ((string-equal make-val "lcm") "l")
		   ((string-equal make-val "happy") "~") ; smile
		   ((string-equal make-val "medic") "+") ; red cross
		   ((string-equal make-val "edu") "=") ; deploma
		   ((string-equal make-val "tech") "^") ; tech upgrade
		   ((string-equal make-val "mil") "A")
		   ((string-equal make-val "hcm") "h")
		   ((string-equal make-val "petro") "p")
		   ((string-equal make-val "shell") "s")
		   (t "?")))

	    (beginning-of-line)
	    (move-to-column 43)
	    (setq
	     use1-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))

	    (beginning-of-line)
	    (move-to-column 48)
	    (setq
	     use2-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))

	    (beginning-of-line)
	    (move-to-column 53)
	    (setq
	     use3-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))
	      

	    (beginning-of-line)
	    (move-to-column 58)
	    (setq
	     max1-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))

	    (beginning-of-line)
	    (move-to-column 64)
	    (setq
	     max2-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))

	    (beginning-of-line)
	    (move-to-column 69)
	    (setq
	     max3-val
	     (if (re-search-forward
		  "\\([0-9]+\\)\\([idolhsp]\\)"
		  (+ (point) 6) t)
		 (cons (read-buffer-num (match-beginning 1) (match-end 1))
		       (buffer-substring (match-beginning 2) (match-end 2)))))
	    (beginning-of-line)
	    (move-to-column 74)
	    (if (looking-at " *\\([0-9\\.]+\\)")
		(setq max-val (read-buffer-num (match-beginning 1) (match-end 1))))

	    (record x y wkfc wkfc-val)
	    (record x y will will-val)
	    (record x y make make-val)
	    (record x y use1 use1-val)
	    (record x y use2 use2-val)
	    (record x y use3 use3-val)
	    (record x y max1 max1-val)
	    (record x y max2 max2-val)
	    (record x y max3 max3-val)
	    (record x y max  max-val)
	  
	    ;;(setq sects (cons
	    ;;		   (list (cons x y)
	    ;;			 (cons 'type type)
	    ;;			 (cons 'eff eff)
	    ;;			 (cons 'wkfc wkfc)
	    ;;			 (cons 'will will)
	    ;;			 (cons 'make make)
	    ;;			 (cons 'pe pe)
	    ;;			 (cons 'cost cost)
	    ;;			 (cons 'use1 use1)
	    ;;			 (cons 'use2 use2)
	    ;;			 (cons 'use3 use3)
	    ;;			 (cons 'max1 max1)
	    ;;			 (cons 'max2 max2)
	    ;;			 (cons 'max3 max3)
	    ;;			 (cons 'max max)
	    ;;			 )
	    ;;		   sects))
	      
	    (forward-line 1)))
	    (if (not empire-batch-play) (message "Done"))
	)))

(defun empire-read-resource ()
  "Reads an empire resource command"
  (interactive)
  (if (re-search-forward "^RESOURCE$" nil t)
      (let (x y type v-eff v-mob min gold fert ocontent uran v-ter
	      (empire-map-des-interactively 0)
	      map-window
	      cw cb
	      )
	(forward-line 2)
	(while
	    (looking-at " *\\(-?[0-9]+\\),\\(-?[0-9]+\\) +\\(.\\)[a-z%\\+)\\*]? +\\([0-9]+\\)% +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\)")
	  (setq x (read-buffer-num (match-beginning 1) (match-end 1)))
	  (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
	  (setq type (buffer-substring (match-beginning 3) (match-end 3)))
	  (setq v-eff (read-buffer-num (match-beginning 4) (match-end 4)))
	  (setq min (read-buffer-num (match-beginning 5) (match-end 5)))
	  (setq gold (read-buffer-num (match-beginning 6) (match-end 6)))
	  (setq fert (read-buffer-num (match-beginning 7) (match-end 7)))
	  (setq ocontent (read-buffer-num (match-beginning 8) (match-end 8)))
	  (setq uran (read-buffer-num (match-beginning 9) (match-end 9)))

	  (record-des x y type t nil)

	  (record x y (position-of 'eff) v-eff)
	  (record x y  (position-of 'min) min)
	  (record x y  (position-of 'gold) gold)
	  (record x y  (position-of 'fert) fert)
	  (record x y  (position-of 'ocontent) ocontent)
	  (record x y  (position-of 'uran) uran)

	  (sit-for 0)
	  (forward-line 1))
	))
  (conditionally-refresh-map)		; update if needed
  )


(defun empire-read-spyreports ()
  "Reads all spy data to end of buffer into sector db."
  (interactive)
  (while (re-search-forward "^SPY .*report" nil t)
      (empire-read-spy-data)
    ))

(defun empire-read-spy-data () 
  "Reads spy data from the spy command or from satellites, cursor must already be positioned"
  (if (re-search-forward "^  +sect   type own +eff +civ +mil  shl  gun +iron  pet  food" nil t)
      (let (x y type own eff civ mil shell gun iron pet food)
	(forward-line 1)
	(while
	    (or
	     (looking-at " +\\(-*[0-9]+\\),\\(-*[0-9]+\\) +\\(.\\)[a-z%\\+)\\*]* +\\(-*[0-9]+\\) +\\([0-9]+\\)")
	     (looking-at "BANG"))
	  (if (match-end 1)
	      (progn
		(setq x (read-match-num 1))
		(setq y (read-match-num 2))
		(setq type (match-string 3))
		(setq own (read-match-num 4))
		(setq eff(read-match-num 5))
		(goto-char (match-end 0))

		(if (looking-at " *\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\) +\\([0-9]+\\)")
		    (setq
		     civ (read-match-num 1)
		     mil (read-match-num 2)
		     shell (read-match-num 3)
		     gun (read-match-num 4)
		     iron (read-match-num 5)
		     pet (read-match-num 6)
		     food (read-match-num 7)
		     )
		  (error "not looking")
		  )
		(if (not (= empire-nation-number own))
		    (progn
		      (record-des x y type t nil)
		      (record x y (position-of 'own) own)
		      (record x y (position-of 'eff) eff)

		      (record x y (position-of 'civ) civ)
		      (record x y (position-of 'mil) mil)
		      (record x y (position-of 'shell) shell)
		      (record x y (position-of 'gun) gun)
		      (record x y (position-of 'iron) iron)
		      (record x y (position-of 'pet) pet)
		      (record x y (position-of 'food) food)
		      ))))

	  (sit-for 0)
	  (forward-line 1))
	)))

(defun empire-read-coastwatch (&optional fcn)
  (interactive)
  (if (re-search-forward "^  Country[ \t]+Ship[ \t]+Location" nil t)
      (let (country shipname shipnum shipx shipy)
	(forward-line 1)
	(while (looking-at ".*(#[ \t]+\\([0-9]+\\))[ \t]+\\([a-zA-z]+\\).*#\\([0-9]+\\) @ \\(-*[0-9]+\\),\\(-*[0-9]+\\)")
	  (setq
	   country (read-match-num 1)
	   shipname (match-string 2)
	   shipnum (read-match-num 3)
	   shipx (read-match-num 4)
	   shipy (read-match-num 5))
	  (record-des shipx shipy (empire-code shipname (match-end 2)) t t)
	  (if fcn (funcall fcn country shipname shipnum shipx shipy))
	  (forward-line 1))
	)))

(defun empire-read-map (&optional start-x)
  "Reads sector designations from an empire map into the DB."
  (interactive);; "nX coord of upper left sector: "
  ;; these are hard.
  ;;   You must figure out when to stop reading on the x axis
  (if (re-search-forward
       "^     [0-9\\-][0-9\\-][0-9\\-][0-9\\-][0-9\\-]*$" nil t)
      (let* ((start-x 
	      (let ((coord-string "")
		    (ncoord-string "")
		    first second)
		(beginning-of-line)
		(while (looking-at
			"^     \\([0-9\\-]\\)\\([0-9\\-]\\)[0-9\\-][0-9\\-]")
		  (setq coord-string 
			(concat coord-string 
				(buffer-substring (match-beginning 1)
						  (match-end 1)))
			ncoord-string
			(concat ncoord-string
				(buffer-substring (match-beginning 2)
						  (match-end 2))))
		  (forward-line))
		(setq first  (car (read-from-string coord-string))
		      second (car (read-from-string ncoord-string)))
		(if (> first second)
		    (- first)
		  first)))
	     (stop-x (progn (forward-line -1)
			    (end-of-line)
			    (+ start-x (- (current-column) 6))))
	     x y
	     ;;(des (position-of 'des))
	     (empire-map-des-interactively 0)
	     )
	(forward-line 1)
	;;	(while (looking-at
	;;		"^     [0-9\\-][0-9\\-][0-9\\-][0-9\\-]")
	;;	  (forward-line 1))
	(if (not empire-batch-play) (message "  Scanning map from %s" start-x))
	(while (looking-at "^ *\\(-*[0-9]+\\) ")
	  (sit-for 0)
	  (setq y (read-buffer-num (match-beginning 1) (match-end 1)))
	  (goto-char (1+ (match-end 1)))
	  ;; if y and start x are not both even or odd, move over 1
	  (setq x start-x)
	  (if (not (= 0 (% (+ (% y 2) (% start-x 2)) 2)))
	      (progn (forward-char 1)
		     (setq x (1+ x))
		     ))
	  (while (<= x stop-x)
	    (if (not (= (following-char) 32))
		(record-des (normalize-x x)
			    (normalize-y y)
			    (buffer-substring (point) (1+ (point)))
			    nil
			    (not (= (following-char) ?.))))
	    (if (= (following-char) ??)
		;; force acknowledgement that this is not owned by us
		(if (not (recall-macro x y (position-of 'own)))
		    (record x y (position-of 'own) 999)))
	    (setq x (+ x 2))
	    (forward-char 2))
	  (forward-line 1)
	  )
      
	(if (not empire-batch-play) (message "Done"))
	))
  (conditionally-refresh-map)		; update if needed
  )

(defun empire-read-map-bounded ()
  "reads a map presuming that is starts at the empire bounds"
  (interactive)
  (empire-read-map empire-x-start))

;;(defun empire-read-somebody-elses-map (start-x delta-x delta-y)
;;  (interactive "nX coord of upper left sector: \nnDelta-x: \nnDelta-y: ")
;;  (if (re-search-forward "^	  [0-9\\-][0-9\\-][0-9\\-][0-9\\-][0-9\\-][0-9\\-]" nil t)
;;	(let ((stop-x (progn (end-of-line) (+ start-x (- (current-column) 6))))
;;	    x y
;;	    (des (position-of 'des))
;;	    )
;;	(forward-line 2)
;;	(message "Reading")
;;	(while (looking-at "^ *\\(-*[0-9]+\\) ")
;;	(sit-for 0)
;;	(setq y (read-buffer-num (match-beginning 1) (match-end 1)))
;;	(goto-char (1+ (match-end 1)))
;;	;; if y and start x are not both even or odd, move over 1
;;	(setq x start-x)
;;	(if (not (= 0 (% (+ (% y 2) (% start-x 2)) 2)))
;;	    (progn (forward-char 1)
;;		   (setq x (1+ x))
;;	    ))
;;	(while (< x stop-x)
;;	  (if (not (= (following-char) 32))
;;	      (record-des (normalize-x (+ x delta-x))
;;			  (normalize-y (+ y delta-y))
;;			  (buffer-substring (point) (1+ (point)))))
;;	  (setq x (+ x 2))
;;	  (forward-char 2))
;;	(forward-line 1)
;;	)
;;	
;;	(message "Done")
;;	)))

(defun empire-read-radar (range sx sy)
  (if empire-interaction-verbosity
      (message "Reading %s,%s range %s" sx sy range))
  (let ((x (- sx (* 2 range)))
	(y (- sy range))
	(civ (position-of 'civ))
	new-des
	old-des
	(empire-map-des-interactively 0)
	)
    (while (<= y (+ sy range))
      (while (<= x (+ sx (* range 2)))
	(if (not (eql 32 (following-char)))
	    (progn
	      (setq new-des (make-string 1 (following-char)))
	      (setq old-des (recall x y des))
	      ;;(princ (format "%s,%s is a %s\n" x y new-des))
	      (record-des (normalize-x x)
			  (normalize-y y)
			  (make-string 1 (following-char))
			  t
			  t)
	      )
	  )
	(forward-char 1)
	(setq x (1+ x))
	)
      (sit-for 0)
      (forward-line 1)
      (setq y (1+ y))
      (setq x (- sx (* range 2)))
      ))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message ""))
  )

(defun empire-read-radars ()
  "Reads all radars to end of buffer into sector DB"
  (interactive)
  (message "Reading radars")
  (let ((des (position-of 'des))
	(empire-map-des-interactively nil)
	)
    (if empire-interaction-verbosity (message "Searching for radar ..."))
    (while (re-search-forward "efficiency.* range [0-9]+$" nil t)
      (beginning-of-line)
      (re-search-forward
       "\\(-*[0-9]+\\),\\(-*[0-9]+\\) efficiency.* range \\([0-9]+\\)$" nil t)
      (forward-line 1)
      (let (
	    (range (car (read-from-string (match-string 3))))
	    (x (car (read-from-string (match-string 1))))
	    (y (car (read-from-string (match-string 2))))
	    )
	(if (looking-at "Satellite sector report")
	    (progn
	      (empire-read-spy-data)
	      (if (re-search-forward "Satellite radar report" nil t)
		  (goto-char (match-beginning 0)))
	      ))
	(if (looking-at "Satellite radar report")
	    (forward-line 1))
	(empire-read-radar range x y)
	;; Incorperate the radar site itself
	(goto-char (match-beginning 0))
	(beginning-of-line)
	(if (not (looking-at "[a-z]"))
	    (record-des x y ")" t nil)
	  (cond
	   ((looking-at "destr")
	    (record-des x y  "D" t t))
	   ((looking-at "battleship")
	    (record-des x y  "B" t t))
	   ((looking-at "slave")
	    (record-des x y  "S" t t))
	   ((looking-at "air")
	    (record-des x y  "A" t t))
	   ((looking-at "patrol")
	    (record-des x y  "P" t t))
	   ((looking-at "yacht")
	    (record-des x y  "Y" t t))
	   ((looking-at "submarine")
	    (record-des x y  "U" t t))
	   ((looking-at "minesweep")
	    (record-des x y  "M" t t))
	   ((looking-at "fishing b")
	    (record-des x y  "F" t t))
	   ((looking-at "fishing t")
	    (record-des x y  "W" t t))
	   ((looking-at "guided")
	    (record-des x y  "G" t t))
	   ((looking-at "Satellite")
	    )
	   ((looking-at "landsat")
	    )
	   ((looking-at "cargo")
	    (record-des x y  "C" t t))
	   ((looking-at "ore")
	    (record-des x y  "O" t t))
	   ((looking-at "tanker")
	    (record-des x y  "T" t t))
	   ((looking-at "oil")
	    (record-des x y  "L" t t))
	   (t
	    (message "bad radar device at %s,%s (point %s)" x y (point))
	    (unless empire-batch-play (sit-for 3))
	    (record-des x y "(" t t))
	   ))
	(forward-line 1)
	(if empire-interaction-verbosity (message "Searching for radar ..."))
	))
    (if (not empire-batch-play) (message "Radar search done."))
    )
  (conditionally-refresh-map))

(defun empire-read-looks ()
  "Reads into the DB the results of looks from ships"
  (interactive)
  (let (
	(civ (position-of 'civ))
	(mil (position-of 'mil))
	(des (position-of 'des))
	(eff (position-of 'eff))
	(empire-map-des-interactively 0);; map as we go
	start
	)

    (message "Searching for looks ...")
    (while (re-search-forward "^.* (#\\([0-9]+\\)) \\([^ ]*\\) .* @ *\\(-*[0-9]+\\),\\(-*[0-9]+\\)" nil t)
      (let (
	    (own (read-match-num 1))
	    (cty (car (read-from-string (match-string 2))))
	    (x (car (read-from-string (match-string 3))))
	    (y (car (read-from-string (match-string 4))))
	    (limit (match-end 0))
	    )
	(record-des x y (empire-code cty (match-end 2)) t)
	(record x y (position-of 'own) own)
	(goto-char (setq start (match-beginning 0)))
	(if (re-search-forward "with .* \\([0-9]+\\) mil" limit t)
	    (record x y mil (car (read-from-string (match-string 1)))))
	(goto-char start)		; ?
	(if (re-search-forward "with .* \\([0-9]+\\) civ" limit t)
	    (record x y civ (car (read-from-string (match-string 1)))))
	(goto-char start)		; ?
	(if (re-search-forward "\\([0-9]+\\)% efficient" limit t)
	    (record x y eff (car (read-from-string (match-string 1)))))
	(sit-for 0)
	(goto-char limit)
	(message "Searching for looks ...")
	)))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message "Done"))
  )

(defun empire-read-navs ()
  "Scans to end of buffer and marks as a '.' all sectors near where a ship
paused momentarely. These designations are overridden by read-looks or
other extractors. Acts to make a trail of where ships went and that they
noticed nothing but ocean." 
  (interactive)
  (let (
	(des (position-of 'des))
	(empire-map-des-interactively 0);; map as we go
	x y
	sects
	)

    (message "Searching for navs ...")
    (while (re-search-forward
	    "^<[0-9]+\\.[0-9]+:[0-9]+\\.[0-9]+: \\(-*[0-9]+\\),\\(-*[0-9]+\\)>"
	    nil t)
      (setq x (read-buffer-num (match-beginning 1) (match-end 1)))
      (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
      (sit-for 0)
      ;; presume that we are a sea and all around us is a sea
      
      (mapcar (function (lambda (pt)
			  (if (not (recall (car pt) (cdr pt) des))
			      (record-des (car pt) (cdr pt) "." t t))))
	      (cons (cons x y) (sects-near (cons x y))))
      ))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message ""))
  )

(defun empire-read-deployed-mines ()
  (interactive)
  (let (
	(des (position-of 'des))
	(empire-map-des-interactively 0) ;; map as we go
	)

    (message "Searching for mines ...")
    (while (re-search-forward "^Laying *\\([0-9]+\\) mines" nil t)
      (let (
	    (count (car (read-from-string (match-string 1))))
	    (limit (match-end 0))
	    )
	(re-search-backward " \\(-*[0-9]+\\),\\(-*[0-9]+\\)" nil t)
	(record-des
	 (car (read-from-string (match-string 1)))
	 (car (read-from-string (match-string 2)))
	 "X"
	 t
	 t)
	(goto-char limit)
	(sit-for 0)
	(message "Searching for mines ...")
	)))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message "Done"))
  )

(defun empire-read-exploding-mines ()
  "Places an X as the sector designation of anyplace where a mine
exploded." 
  (interactive)
  (let (
	(des (position-of 'des))
	(empire-map-des-interactively 0) ;; map as we go
	)

    (message "Searching for exploding-mines ...")
    (while (re-search-forward
	    "^Kawhomp! Mine detected .* \\(-*[0-9]+\\),\\(-*[0-9]+\\)"
	    nil t)
      (let (
	    (count (car (read-from-string (match-string 1))))
	    (limit (match-end 0))
	    )
	(record-des
	 (car (read-from-string (match-string 1)))
	 (car (read-from-string (match-string 2)))
	 "X"
	 t
	 t)
	(goto-char limit)
	(sit-for 0)
	(message "Searching for exploding-mines ...")
	)))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message "Done"))
  )

(defun empire-read-mines ()
  "Reads mines laid by us as well as exploding mines we detect."
  (interactive)
  (safe-excursion (empire-read-deployed-mines))
  (safe-excursion (empire-read-exploding-mines))
  )

(defun empire-read-flight ()
  "Reads to end of buffer all designations returned by flights. This
integrates data found out by flying over enemy terretory into your big map."
  (interactive)
  (let (
	(des (position-of 'des))
	(empire-map-des-interactively 0) ;; map as we go
	)

    (message "Searching for fly ...")
    (while (re-search-forward
	    "^flying over +\\([a-z]+\\) .*at *\\(-*[0-9]+\\),\\(-*[0-9]+\\)"
	    nil t)
      (let (
	    (type (match-string 1))
	    (x (car (read-from-string (match-string 2))))
	    (y (car (read-from-string (match-string 3))))
	    (limit (match-end 0))
	    )
	(record-des x y (empire-code type) nil t)
	(goto-char (match-end 0))
	(message "Searching for flight ...")
	)))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message "Done"))
  )

(defun empire-read-pings ()
  "Reads to end of buffer all designations pings."
  (interactive)
  (let (
	(des (position-of 'des))
	(empire-map-des-interactively 0) ;; map as we go
	)

    (message "Searching for pings ...")
    (while (re-search-forward
	    "^Sonar ping from *\\(-*[0-9]+\\),\\(-*[0-9]+\\)"
	    nil t)
      (let (
	    (x (car (read-from-string (match-string 1))))
	    (y (car (read-from-string (match-string 2))))
	    (limit (match-end 0))
	    )
	(record-des x y "~" t t)
	(goto-char (match-end 0))
	(message "Searching for ping ...")
	)))
  (conditionally-refresh-map)		; update if needed
  (if (not empire-batch-play) (message "Done"))
  )


(defun empire-read-explores ()
  "This reads all explore map up to the end of the buffer. The explored
sectors, AND the adjacient ones are integrated into the big map. Thus,
between break and the first update, you can find out about an immediate area of
more than 300 sectors."
  (interactive)
  (let (start
	x
	y
	(des (position-of 'des))
	(empire-map-des-interactively 0)
	)
    (while (re-search-forward "min gold fert  oil uran" nil t)
      (beginning-of-line)
      (setq start (point))
      (re-search-forward "\\(-*[0-9]+\\),\\(-*[0-9]+\\)>" nil t)
      (setq x	(read-buffer-num (match-beginning 1) (match-end 1)))
      (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
      (goto-char (+ start 4))
      (record-des (normalize-x (- x 1)) (normalize-y (- y 1))
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-char 2)
      (record-des (normalize-x (+ x 1)) (normalize-y (- y 1))
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-line 1)
      (forward-char 3)
      (record-des (normalize-x (- x 2)) y
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-char 2)
      (record-des x  y
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-char 2)
      (record-des (normalize-x (+ x 2)) y
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-line 1)
      (forward-char 4)
      (record-des (normalize-x (- x 1)) (normalize-y (+ y 1))
		  (make-string 1 (following-char))
		  t 
		  t)
      (forward-char 2)
      (record-des (normalize-x (+ x 1)) (normalize-y (+ y 1))
		  (make-string 1 (following-char))
		  t 
		  t)
      (goto-char start)
      (forward-line 1)
      (forward-char 15)
      ;; read min
      (forward-line 2)
      ))
  (conditionally-refresh-map)		; update if needed
  )

(defun empire-summarize-resources ()
  "Summarizes all resources found by explore commands.
Resources are most often not saved in the sector info. This scans the
buffer for this info and shows it in another window. This is useful for
within-a-session decision guidance.

Works from point to end of buffer"
  (interactive)
  (let (pat
	x
	y
	(des (position-of 'des))
	list
	)
    (with-output-to-temp-buffer "*Res*"
      (princ "\t\tExplored Resources\nx,y	min gold fert  oil uran\n")
      (empire-data-mode-buffer t t "*Res*")
      (while (re-search-forward "min gold fert	oil uran" nil t)
	(goto-char (match-beginning 0))
	(next-line 1)
	(setq pat (buffer-substring (point) (progn (end-of-line) (point))))
	(re-search-forward "\\(-*[0-9]+\\),\\(-*[0-9]+\\)>" nil t)
	(setq x		(read-buffer-num (match-beginning 1) (match-end 1)))
	(setq y (read-buffer-num (match-beginning 2) (match-end 2)))
	(if (not (member (cons x y) list))
	    (progn
	      (setq list (cons (cons x y) list))
	      (princ (format "%s\t%s\n" (format-xy x y) pat))))
	(forward-line 2)
	(sit-for 0)
	))))

(defun empire-read-nation-report ()
  "Look for nation reports, and extract the education, happiness,
technology, and research levels.  These values are stored as integers,
rounded up or down to the nearest value."
  (interactive)
  (let ()
    (while (re-search-forward
	    "^[ \t]*[A-Za-z0-9]+[ \t]+Nation Report[ \t]+[A-Za-z]+" nil t)
      ;;
      ;; Parse education and happiness
      ;;
      (re-search-forward
       "^Education[.]+[ \t]*\\([0-9]+\\)\\.\\([0-9]\\)[0-9]+[ \t]+Happiness[.]+[ \t]*\\([0-9]+\\)\\.\\([0-9]\\)"
       nil t)
      (setq empire-education-level
	    (read-buffer-num (match-beginning 1) (match-end 1)))
      (if (>= (read-buffer-num (match-beginning 2) (match-end 2)) 5)
	  (setq empire-education-level (1+ empire-education-level)))
      (setq empire-happiness-level
	    (read-buffer-num (match-beginning 3) (match-end 3)))
      (if (>= (read-buffer-num (match-beginning 4) (match-end 4)) 5)
	  (setq empire-happiness-level (1+ empire-happiness-level)))
      ;;
      ;; Parse technology and research
      ;;
      (re-search-forward
       "^Technology[.]+[ \t]*\\([0-9]+\\)\\.\\([0-9]\\)[0-9]+[ \t]+Research[.]+[ \t]*\\([0-9]+\\)\\.\\([0-9]\\)"
       nil t)
      (setq empire-technology-level
	    (read-buffer-num (match-beginning 1) (match-end 1)))
      (if (>= (read-buffer-num (match-beginning 2) (match-end 2)) 5)
	  (setq empire-technology-level (1+ empire-technology-level)))
      (setq empire-research-level
	    (read-buffer-num (match-beginning 3) (match-end 3)))
      (if (>= (read-buffer-num (match-beginning 4) (match-end 4)) 5)
	  (setq empire-research-level (1+ empire-research-level)))
      )
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun empire-drop-spaces (str)
  "Return a copy of STR, with leading and trailing spaces removed."
  (let (loc)
    (if (string-match "^[ \t]+" str)
	(setq str (substring str (match-end 0)))
      )
    (if (string-match "[ \t]+$" str)
	(progn
	  (setq loc (match-beginning 0))
	  (if (> loc 0)
	      (setq loc (1- loc))
	    )
	  (setq str (substring str 0 loc))
	  )
      )
    str
    )
  )

(defun replace-chars-from-list (list)
  "Replace characters with new ones from a LIST.  LIST is of the form:

	((old1 . new1) (old2 . new2) ... )

where `old1', `old2', etc. are the characters to replace, and `new1',
`new2', etc. are the characters which will replace the previous ones.
The replacing will begin from the current location of point, and will
continue to EOF.  The original value of point will be restored when this
routine exits."
  (let ( (here (point)) )
    (dolist (item list)
      (goto-char here)
      (replace-string (car item) (cdr item))
      )
    (goto-char here)
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; State Extraction from previously saved empire sessions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-get-data-from-file (short &optional long)
  "Scans a normal file for empire data made from previous sessions.

You must take care that outdated data does not replace more recient data."
  (interactive "fGEET Data File: ")
  (find-file short)
  (message "am in %s" short)
  (sit-for 0)
  (empire-read-data)
  (kill-buffer (current-buffer)))

(defun dired-empire (file)
  "Reads all files in a directory and extracts empire radar, look, flight,
   and mine info. Save your empire sessions to use this."
  (interactive "FSave File Name: ")
  ;; this next fcn may be a lrs customization -- I have forgotten
  (let ((empire-dump-hooks '()))
    (dired-map-dired-file-lines 'empire-get-data-from-file)
    )
  (map-empire)
  (save-empire file)
  )

;;(setq-default dired-listing-switches "-altr")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; State Storage accross emacs sessions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro dump-variable (name)
    (list 'insert (list 'format (concat "(setq " (symbol-name name) " %s)\n\n")
			(list 'dump-format-value name))))
(defun dump-format-value (value)
  (cond
   ((stringp value)
    (format "\"%s\"" value))
   ((numberp value) (format "%s" value))
   (t (format "'%s" value))))

(defun save-empire (file)
  "Stores an empire session.
   Certain internal variables are stored as well as all empire-user-variables
   and all empire-system-variables (see the function of these names for a
   definition of these terms). Most empire customizers only need to
   introduce new user or system variables to have session to session state
   storage." 
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name
	      (file-name-directory empire-save-file-name))
	  empire-save-file-name)))

  (setq empire-save-file-name
	(expand-file-name file
	 (if empire-save-file-name
	     (file-name-directory empire-save-file-name))))
  (save-excursion
    (find-file file)
    (buffer-flush-undo (current-buffer))
    (delete-region (point-min) (point-max))
    (emacs-lisp-mode)
    (if (fboundp 'make-header)
	(progn
	  (make-header)
	  (insert (format "empire sector information from GEET %s" GEET-Version))
	  (goto-char (point-max))))

    (if empire-interaction-verbosity (message "  Dumping System Variables"))
    (insert (format "(if empire-interaction-verbosity (message \"  Loading empire system variables saved on %s\"))\n"
		    (current-time-string)))
    (mapcar '(lambda (var)
	       (insert (format "(setq %s %s)\n"
			       (symbol-name var)
			       (dump-format-value (eval var)))))
	    (empire-system-variables))

    (if empire-interaction-verbosity (message "  Dumping User Variables"))
    (insert (format "(if empire-interaction-verbosity (message \"  Loading empire user variables saved on %s\"))\n"
		    (current-time-string)))
    (mapcar '(lambda (var)
	       (insert (format "(setq %s %s)\n"
			       (symbol-name var)
			       (dump-format-value (eval var)))))
	    (empire-user-variables))

    (if empire-interaction-verbosity (message "  Dumping planes"))
    (insert (format "(if empire-interaction-verbosity (message \"  Loading empire planes saved on %s\"))\n"
		    (current-time-string)))
    (dump-variable empire-planes)
    

    (dump-variable empire-pos-count)
    (let ((old-empire-pos empire-pos)
	  (old-empire-plane-pos empire-plane-pos))
      ;; we really cannot allow these things to be redefined
      (dump-variable old-empire-pos)
      (dump-variable old-empire-plane-pos)
      )

    (if empire-interaction-verbosity (message "  Dumping sectors"))
    (insert (format "(if empire-interaction-verbosity (message \"  Loading empire sects saved on %s\"))\n"
		    (current-time-string)))
    (dump-variable empire-row-ranges)
    ;; empire-sects is special because of a 1meg limit on format
    ;; dump each row seperately
    ;;(dump-variable empire-sects)
    (insert (format "(setq empire-sects (make-vector (* empire-height 2) nil))\n"))
    (let ((y 0))
      (while (< y (* empire-height 2))
	(if (and empire-interaction-verbosity (not empire-batch-play))
	    (message "  dumping row %s" y))
	(insert (format "  (set-empire-row %s %s)\n"
			y (aref empire-sects y)))
	(setq y (1+ y))))
    (insert "\n(setq empire-load-complete t)\n")
    (insert "(if empire-interaction-verbosity (message \"  Empire Sects Defined as of %s\" (current-hour-string)))\n")
    (let (;;(trim-versions-without-asking t)
	  ;;(make-backup-files nil);; suppress backup files
	  )
      (if empire-batch-play
	  (message "Writing world image file (%s bytes) at %s. . ." (point)
		   (current-hour-string))
	(message "Writing world image file (%s bytes). . ." (point)))
      (save-buffer))
    (if empire-batch-play
	(message "Empire data dumped at %s" (current-hour-string))
	(message "Empire data dumped"))
    (kill-buffer (current-buffer))
    ))

(defun set-empire-row (index val)
  (if (not empire-batch-play)
      (message " Restoring row %s" index))
  (aset empire-sects index val))

(defun restore-empire (file)
  "Restores a previously saved sessions and its custom values"
  (interactive 
   (list (read-file-name
	  (format "Restore from Save File (Default: %s): "
		  empire-save-file-name) 
	  (if empire-save-file-name
	      (file-name-directory empire-save-file-name))
	  empire-save-file-name)))
  (let ((old-empire-pos empire-pos)
	(old-empire-plane-pos empire-plane-pos)
	(empire-pos-count empire-pos-count) ; prevent a change during the load
	(empire-pos empire-pos)
	)
    (load-file file)
    (if (not (equal old-empire-pos empire-pos))
	(progn
	  (message "The value of empire-pos is different in this dump. Converting")
	  (setq empire-sects (convert-empire-pos empire-sects old-empire-pos empire-pos))
	  (message "Sectors converted to new empire-pos offsets")
	  ))
    (if (not empire-batch-play) (map-empire))
    (if (not (equal old-empire-plane-pos empire-plane-pos))
	(message "The value of empire-plane-pos is different in this dump. Recompile empire!"))
    )
  )

(defun convert-empire-pos (data old-pos new-pos)
  "This piece of code takes an empire-sects, an old empire-pos, and a
new empire-pos and returns a new empire-sects."
  (let* ( ( old-len  (apply 'max (mapcar 'cdr old-pos)))
	  ( new-len  (apply 'max (mapcar 'cdr new-pos)))
	  ( height (length data) )
	  value-map
	  ( new-sects (make-vector height nil) )
	  xp
	  (y 0)
	  row row-len 
	  new-row 
	  new-data
	  old-data
	  map
	  )
    (setq value-map
	  (delq nil
		(mapcar
		 '(lambda (pos)
		    ;; for each elem in new-pos, see if there was a
		    ;; registered elem of the same name in old-pos
		    ;; and make a pair
		    ;; (<offset in old> . <offset in new>)
		    ;; discount the offsets in old which are at the
		    ;; last element or the offsets in new whicha re
		    ;; at the last element as the last element in
		    ;; any case is a bit bucket.
		    (if (not (= (cdr pos) new-len)) ; scratch new
					; bit buckets
			(let ((opos (assoc (car pos) old-pos)))
			  (if (and opos
				   (/= old-len (cdr opos)) ; scratch old bitb
				   )
			      (cons  (cdr opos) (cdr pos)
				     )
			    ))))
		 new-pos)))

    (while (< y height )
      (message "Converting row %s" y)
      (setq row (aref data y)
	    row-len (length row)
	    xp 0)
      (if row
	  (progn
	    (setq new-row (make-vector row-len nil))
	    (while (< xp row-len)
	      (setq old-data (aref row xp))
	      (if old-data
		  (progn
		    (if (>= 3 (length old-data))
			(setq new-data old-data)
		      (setq map value-map
			    new-data (make-vector (1+ new-len) nil))
		      (while map
			(aset new-data (cdr (car map))
			      (aref old-data (car (car map))))
			(setq map (cdr map)))
		      )
		    (aset new-row xp new-data)
		    ))
	      (setq xp (1+ xp))
	      )
	    (aset new-sects y new-row)
	    ))
      (setq y (1+ y))
      )
    new-sects
    ))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Help for empire variables and hooks
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'emp-option)
(defun empire-user-variables ()
  "Returns list of all empire user settable variables. These are those
   variables whose doc string begins with '*' and whose name contains the
   string 'empire' or those variables with a 'empire property"
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (and (user-variable-p sym)
				   (or
				    (get sym 'empire)
				    (string-match "empire" (symbol-name sym))))
			      (setq vars (cons sym vars))))))
    (setq vars (sort vars 'string-lessp))
    ;;(nreverse vars)
    ))

(defun empire-interactive-functions ()
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (and (commandp sym)
				   (or
				    (string-match "empire" (symbol-name sym))
				    (get sym 'empire)))
			      (setq vars (cons sym vars))))))
    vars
    (setq vars (sort vars 'string-lessp))
    ))

(defun empire-system-variables ()
  "Returns list of all empire variables not set by the user. These are those
   variables with a 'empire-system property"
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-system)
			      (setq vars (cons sym vars))))))
    ;;(setq vars (sort vars 'string-lessp))
    (nreverse vars)
    ))

(defun empire-edit-options ()
  "Display a editable list of Emacs empire user options, with values and
documentation." 
  (interactive)
  ;; need require of options but that file has no 'provide
  (if (not (fboundp 'edit-options-mode)) (load-library "options"))

  (save-excursion
    (set-buffer (get-buffer-create "*Options for Empire*"))
    (Edit-options-mode))
  (with-output-to-temp-buffer "*Options for Empire*"
    (let ((vars (empire-user-variables)))
      (while vars
	(let ((sym (car vars)))
	  (princ ";; ")
	  (prin1 sym)
	  (princ ":\n\t")
	  (prin1 (symbol-value sym))
	  (terpri)
	  (princ (substitute-command-keys 
		  (documentation-property sym 'variable-documentation)))
	  (princ "\n;;\n"))
	(setq vars (cdr vars))))))

(defun describe-empire-interactive-functions ()
  "Offers function documentation on all interactive functions empire offers
without regard to position in any keymap."
  (interactive)
  (with-output-to-temp-buffer "*Interactive Empire Functions*"
    (mapcar (function (lambda (function)
			(prin1 function)
			(princ ":
")
			(if (documentation function)
			    (princ (documentation function))
			  (princ "not documented"))
			(terpri)
			(terpri)
			))
	    (empire-interactive-functions))))
  


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Novas
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun calc-nova-from-sector (x y fcn)
  "Calcs from x y a path to every land non ^ ? sector, calls FCN with x y
and the path (reversed). Loop stops when there is no more reachable land or
when FCN returns non nil."
  (let ((examined (list (list (cons x y))))
	(being-examined)
	(to-examine (list (list (cons x y))))	
	(stop-loop nil)
	exp-data
	new-exp-data
	des
	(depth 0)
	)
    (while (and (not stop-loop) to-examine)
      (setq being-examined to-examine)
      (setq to-examine nil)
      (setq depth (1+ depth))
      (message "Depth %s with %s land sectors" depth (length being-examined))
      (while (and (not stop-loop) being-examined)
	(setq exp-data (car being-examined))
	(setq being-examined (cdr being-examined))
	;; examining a sector means finding all the adjacient sectors
	;; to which we can go and placing them and their paths on the
	;; to-examine list for the next pass
	(mapcar (function (lambda (pt)
			    (if (and (not (assoc pt examined))
				     (landp (car pt) (cdr pt))
				     (setq des (recall (car pt) (cdr pt)
						       (position-of 'des)))
				     (not (string-equal des "."))
				     (not (string-equal des "^"))
				     (not (string-equal des "?"))
				     ;; enemy known sector?
				     )
				(progn
				  (setq new-exp-data (cons pt exp-data))
				  (setq examined (cons new-exp-data examined))
				  (setq to-examine (cons new-exp-data to-examine))
				  (sit-for 0)
				  (if fcn
				      (setq stop-loop (funcall fcn x y new-exp-data)))
				  )
			      )
			    ))
		(sects-near (car exp-data)))
	))
    stop-loop))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Buffer excursions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Macro to switch to the empire shell buffer if the current buffer is
;; not the empire shell buffer.
;;
(defmacro switch-to-empire-buffer-if-necessary (&rest body)
  (` (if (equal (buffer-name (current-buffer)) empire-shell-buffer)
	 (progn
	   (,@ body)
	   )
       (progn
	 (save-excursion		; <stub> inefficient?
	   (set-buffer empire-shell-buffer)
	   (,@ body)
	   )
	 )
       )
     )
  )

;;
;; Macro to switch to the map buffer if the current buffer is not the
;; map buffer.
;;
(defmacro switch-to-map-buffer-if-necessary (&rest body)
  (` (if (equal (current-buffer) empire-map-buffer)
	 (progn
	   (,@ body)
	   )
       (progn
	 (save-excursion
	   (set-buffer empire-map-buffer)
	   (,@ body)
	   )
	 )
       )
     )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector edit
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-edit-sector (x y)
  "Allows direct editing of the empire sector database. Good for very
special fiddles."
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
	       ))
  (valid-sector x y)
  (let (item label
	(table (mapcar '(lambda (item) (list (symbol-name (car item))))
		       empire-pos))
	(flag t)
	nv
	)
    (while flag
      (setq label (completing-read "Sector Attribute? " table nil t))
      (if  (not (string-match "^[ \t]*$" label))
	  (progn
	    (setq item (intern label))
	    (setq nv (read-string
		      (format "Sector %s,%s has for %s the value %s. New Value: "
			      x y item
			      (recall x y (dynamic-position-of item)))))
	    (if (not (string-match "^[ \t]*$" nv))
		(progn
		  (setq nv (car (read-from-string nv)))
		  (record x y (dynamic-position-of item) nv)
		  (message "nv is %s" nv)
		  )
	      (setq flag nil))
	    )
	(setq flag nil))
      (show-pt (cons x y) nil)
      (map-des x y (recall x y (position-of 'des)))
 )))

  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; formatting
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun format-xy (x y) ;; returns length 10 sector des, always has blank
  (let* ((str (format "%s,%s" x y))
	(len (length str)))
    (if (< len 10)
	(concat str (make-string (- 10 len) ? ))
      str)))

(defun make-banner (str)
  (concat (make-string (/ (- 72 (length str)) 2) ? ) "-*- " str " -*-\n"))
(defun print-banner (str)
  (princ (make-banner str)))
(defun insert-banner (str)
  (insert (make-banner str)))
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Debug
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-test-world-setup ()
  "Makes diagnostic messages"
  (interactive)
  (let ((active-sec-cnt 0)
	(sec-cnt 0)
	)
    (with-output-to-temp-buffer "*World Diagnoses*"
      (princ "\t\t\tEmpire World Diagnoses for version %s\n" GEET-Version)
      (princ "Mail this buffer to gnurus-bugs@indetech.com for help\n\n")
      (sit-for 0)

      (if (not empire-sects)
	  (princ "The sector DB is not set up, do version command and empire-new-world.\n")
	(princ "Sector Db has been set up\n")

	(if (not (and (= (length empire-sects) (* empire-height 2))
		      (= (length empire-row-ranges) (* empire-height 2))))
	    (princ "The sector DB does not match the current height.\n")
	  (princ "Sector DB correctly matches current height.\n"))

	(walksects
	 (setq active-sec-cnt (1+ active-sec-cnt))
	 "Counting Active Sectors ")
	(princ (format "Active Sectors:\t%s\n" active-sec-cnt))
	(mapsects-all-in-bounds
	 '(lambda (x y) (setq sec-cnt (1+ active-sec-cnt)))
	 "Counting All Sectors ")
	(princ (format "All Sectors:\t%s\n" sec-cnt))
	(save-excursion
	  (switch-to-buffer "*World Diagnoses*")
	  (dump-variable empire-width)
	  (dump-variable empire-height)
	  (dump-variable empire-x-start)
	  (dump-variable empire-x-stop)
	  (dump-variable empire-y-start)
	  (dump-variable empire-y-stop)
	  (dump-variable etu-per-update)
	  (dump-variable upd-per-day)
	  (dump-variable bodies-fead-by-1-food-per-etu)
	  (dump-variable seconds-per-time-unit)
	  (dump-variable edu-divisor)
	  (dump-variable happy-divisor)
	  (dump-variable empire-shell-buffer)
	  (princ (format "Process is %s\n" (get-buffer-process empire-shell-buffer)))
	  )
	)
      )))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Reader note
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(if (not empire-batch-play)
  (message "GEET Version %s" GEET-Version))
