;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-anal.el -- Game analysis tools for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater (lrs@indetech.com)
;; Created On      : Fri Oct 26 14:36:25 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:53 1991
;; Update Count    : 241
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	These tools provide analysis of the state of a country based upon
;; the contents of the local sector database or upon dynamically available
;; variables such as the contents of a dump line.
;; 
;; HISTORY
;; 12-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Tue Feb 12 18:51:14 1991 #236 (Lynn Slater x2048)
;;    added empire-find-non-road-dist
;; 1-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Thu Jan 31 10:33:34 1991 #213 (Lynn Slater x2048)
;;    added set-ideal-???-function utilities.
;; 22-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan 21 20:28:52 1991 #202 (Lynn Slater x2048)
;;    cleaned up plane info access per Ken Stevens changes
;; 19-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Tue Jan 15 18:47:46 1991 #196 (Lynn Slater x2048)
;;    made redistribute food do sanity check on GEET levels.
;; 10-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Thu Jan 10 09:17:22 1991 #190 (Lynn Slater x2048)
;;    brought in patches from troth
;; 31-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Sun Dec 30 17:26:51 1990 #168 (Lynn Slater x2048)
;;    added empire-target-plane
;; 26-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Wed Dec 26 19:25:24 1990 #164 (Lynn Slater x2048)
;;    added check-sectors-lost hook
;; 10-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Mon Dec 10 12:06:12 1990 #153 (Lynn Slater x2048)
;;    redistribute mil will not take from dist pts with 4 or more sectors.
;; 7-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Fri Dec  7 10:54:01 1990 #147 (Lynn Slater x2048)
;;    will not suggest redesignating a bridge
;;    redistribute-mil is much better about mob calcs
;; 6-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Wed Dec  5 14:02:37 1990 #141 (Lynn Slater x2048)
;;    added sensitive to empire-max-path-depth
;; 5-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Wed Dec  5 10:08:59 1990 #138 (Lynn Slater x2048)
;;    added ideal-mil-contested
;; 29-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Wed Nov 28 09:19:58 1990 #130 (Lynn Slater x2048)
;;    made bridges be like roads for redistribute civ and mil.
;; 27-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Mon Nov 26 10:41:48 1990 #127 (Lynn Slater x2048)
;;    only thr f on redist civ if there is a dist pt
;; 25-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Wed Nov 21 19:24:18 1990 #118 (Lynn Slater)
;;    added darryl's changes
;;      new variables empire-designations-*, empire-max-civ-redistribute,
;;        empire-use-inefficient-highways 
;;    added empire-never-starve-threshold. Made ideal-civ be settable
;;    ideal-civ always wants 1 civ there no matter what uw and mil count is.
;; 17-Nov-1990		Lynn Slater	
;;    Last Modified: Sat Nov 17 18:50:37 1990 #116 (Lynn Slater)
;;    sector with 4 dependents will not get adjusted.
;; 17-Nov-1990		Lynn Slater	
;;    Last Modified: Sat Nov 17 11:44:16 1990 #115 (Lynn Slater)
;;    idealciv will not starve new roads. Unload dist-node prettier.
;; 15-Nov-1990		Lynn Slater	
;;    Last Modified: Thu Nov 15 09:18:39 1990 #112 (Lynn Slater)
;;    added ideal-mil-enlist
;; 14-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov 12 19:07:25 1990 #110 (Lynn Slater)
;;    changed "mo " to "move" in issued commands
;; 12-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov 12 13:01:06 1990 #106 (Lynn Slater)
;;    had left over sit-for in redistribute-civ. Changed init values.
;; 8-Nov-1990		Lynn Slater	
;;    Last Modified: Wed Nov  7 14:59:43 1990 #87 (Lynn Slater)
;;    fixed bug in redistribute-civ endless loop feeding mountans
;;    find-best-dist only considers efficient roads
;;    minor level adjustments to %lp
;; 7-Nov-1990		Lynn Slater	
;;    Last Modified: Tue Nov  6 16:21:47 1990 #84 (Lynn Slater)
;;    made ideal-mil work was broken.
;;    Provides for user to set the ideal-mil fcn
;; 6-Nov-1990		Lynn Slater	
;;    Last Modified: Tue Nov  6 09:23:21 1990 #79 (Lynn Slater)
;;    lowered proposed gun and shell levels in forts
;;    documented level-should-be better. fixed constipatedp.
;; 4-Nov-1990		Lynn Slater	
;;    Last Modified: Sun Nov  4 13:02:40 1990 #69 (Lynn Slater)
;;    renamed some interactive fcns, documented them better
;;    added *Proposed* buffer, made commands have mesages
;;    split check-dist into two hooks. Fixed plague format problem
;;    made check-empire report errors
;; 2-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  2 09:41:54 1990 #58 (Lynn Slater)
;;    added plague chance hook.
;; 1-Nov-1990		Lynn Slater	
;;    Last Modified: Thu Nov  1 12:39:16 1990 #54 (Lynn Slater)
;;    dist paths will now be suggested that are not over highways
;; 30-Oct-1990		Lynn Slater	
;;    Last Modified: Tue Oct 30 17:50:44 1990 #38 (Lynn Slater)
;;    made provision for automatic issue of suggestions
;; 30-Oct-1990		Lynn Slater	
;;    Last Modified: Mon Oct 29 19:06:42 1990 #29 (Lynn Slater)
;;    made redistribute food pay attention to mobility limits
;; 29-Oct-1990		Lynn Slater	
;;    Last Modified: Mon Oct 29 16:30:17 1990 #26 (Lynn Slater)
;;    added empire-flow for more details on a single sector
;; 28-Oct-1990		Lynn Slater	
;;    Last Modified: Sun Oct 28 16:23:48 1990 #21 (Lynn Slater)
;;    redistrubute-civ was broke. Better level chedks.
;; TABLE OF CONTENTS
;;   empire-edit-adjust-hooks -- Displays a buffer showing each hook, its description, and whether it is
;;   empire-edit-check-hooks -- Displays a buffer showing each hook, its description, and whether it is
;;   check-empire -- Runs all the empire-check-hooks.  These work from the emacs database
;;   empire-edit-dump-hooks -- Displays a buffer showing each hook, its description, and whether it is
;;   redistribute-population -- Moves surplus people from sectors to highways that can hold them.
;;   redistribute-food -- Makes all highways have > empire-highway-min-food food. Only takes from
;;   redistribute-mil -- Makes all sectors have correct mil
;;   redistribute-civ -- Makes all sectors have correct civ by moving on efficient roads.
;;   calc-defensive-air-umbrella -- Calculates and returns the air strength defending a given sector.
;;   calc-air-defenders -- Calculates and returns the air strength defending a given sector.
;;   empire-flows -- Shows the flow into and out of each sector that is the target of a dist.
;;   empire-flow -- Shows the flow into and out of each sector that dists to a given sector
;;   check-demobobilize -- Demobilizes all mil that are more than 10 over the ideal level as
;;   describe-empire-ideal-mil-functions -- Offers function documentation on all interactive functions empire offers
;;   empire-set-ideal-mil-function -- Displays, prompts for, and sets the ideal-mil function
;;   describe-empire-ideal-civ-functions -- Offers function documentation on all interactive functions empire offers
;;   empire-set-ideal-civ-function -- Displays, prompts for, and sets the ideal-civ function
;;   unload-dist-node -- Suggests alternate dist routes for all items distributing to a given
;;   empire-find-non-road-dist -- Reports each dist path going over non roads. Suggests alt dist paths.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-anal)
(require 'emp-db)
(require 'emp-option)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User variables -- do not edit here, use empire-edit-options or the
;; listed functions
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-highway-leave-food 999
  "*Food left on highway after redistribute")
(defvar empire-highway-min-food   900
  "*Food each highway must have less of to be a target of a redistribution")

(defvar empire-use-inefficient-highways t
  "*If `nil', distribution paths will only consider disting to roads that
are more than `empire-highway-min-eff' efficient.  If non-nil, roads of any
efficiency will be considered.  If the undeveloped portions of your
country are designated as roads (and these areas are medium to large),
you should set this variable to `nil'.  If you do not, this tool will
spend a VERY large amount of time trying to find distribution paths for
these undeveloped roads.")

(defvar empire-min-mil-move 3
  "*The smallest number of mil worth moving with redistribute-mil.")
(defvar empire-min-mil-move-into-enlistment 20
  "*The smallest number of mil worth moving into an enlistment center.")
(defvar empire-min-civ-move 10
  "*The smallest number of civs worth moving with redistribute-civ.")

(defvar empire-max-civ-redistribute 200
  "*The maximum number of civilians that will be moved from one sector
to another during a redistribute-civ.  Note that this variable controls
only the maximum that is moved from one sector to another; it does *NOT*
control the *TOTAL* amount moved out.")

(defvar empire-air-defense-incr 25
  "*Number of air attack units to rate 1 increment on the air defense map.")

(defvar empire-dump-hooks nil
  "List of functions called as the dump data for each sector is read.

See the fcn doc for register-dump-hook to learn how to add your own hooks.
This hook can be edited by one of the commands on the user customization map")
(put 'empire-dump-hooks 'empire-system t)

(defvar empire-check-hooks nil
  "A list of functions to call when an empire check is requested. 

See the fcn doc for register-check-hook to learn how to add your own hooks.
This hook can be edited by one of the commands on the user customization map")
(put 'empire-check-hooks 'empire-system t)

(defvar empire-adjust-hooks
  nil
  "An ordered list of all the functions to be called by adjust-empire.
Adjust-empire is callable from the tab map or the extract map and is called
indirectly by refresh-adjust-empire and -adjust or -fire-adjust in batch mode.

You might consider setting this if one of the default hooks works poorly in
your world or takes too much time. For example, the food hooks are needless
in some blitz games.

See the fcn doc for register-adjust-hook to learn how to add your own hooks.
This hook can be edited by one of the commands on the user customization map")
(put 'empire-adjust-hooks 'empire-system t)

(defvar empire-sector-min-efficiency-check 5
  "*Sectors must be at least this efficient before they are checked for distribution.")

(defvar empire-sector-min-people-check 40
  "*Sectors must contain at least this many people (civ+mil+uw) before
they are checked for distribution.")

(defvar empire-sector-civ-threshold 150
  "*Sectors which contain many civilians are candidates for filling other sectors.
  Young worlds will probably want this to be around 150, old worlds will
  want it to be around 800.

  Is used by the ideal-civ function.")

(defvar empire-never-starve-threshold 50
  "*Sectors which contain many bodies are presumed to be able to grow
enough food to keep from starving. You should place a safety factor when
setting this number. Typocal values are 30, 50, and 120.

  Is used by many of the ideal-civ functions.")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; adjustment hooks. See emp-auto for their use
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun register-adjust-hook (symbol enabledp)
  "Places the function named by SYMBOL as a possible adjustment hook. If
ENABLEDP is non nil, the function is in the initial set of activated
analisys hooks. The hook is placed last on the adjust hook list.

Recommended order:
'(empire-flows redistribute-civ redistribute-mil redistribute-population
		 redistribute-food check-empire check-empire-planes)

Each hook is called once for each adjust-empire call.
"
  (put symbol 'empire-adjust-hook t) ;; Make it a hook
  (if enabledp
      (if (not (memq symbol empire-adjust-hooks))
	  (setq empire-adjust-hooks
		(cons symbol empire-adjust-hooks)))))

(defun empire-edit-adjust-hooks ()
  "Displays a buffer showing each hook, its description, and whether it is
enabled. Allows toggling of enabledness."
  (interactive)
  (let (hooks)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-adjust-hook)
			      (setq hooks (cons sym hooks))))))
    (edit-hooks 'empire-adjust-hooks hooks
		       "*Hooks for Empire Adjustment*"
		       "\t\t\tEmpire Adjustment Options"
)))



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; State Checkers that use only the sector DB
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun register-check-hook (symbol label enabledp)
  "Places the function SYMBOL as a possible hook with LABEL. If ENABLEDP
is non nil, the function is in the initial set of dump hooks.

Each function is called just once and must walk the sectors itself if that is
needed. All sector data access must be through the recall function.

The hook can do anything, but mainly it is supposed to use prin1 and princ
to write messages. The standard-output is already directed to the *Checks*
buffer and the output of the different hooks are seperated by their LABELS.

Slow hooks quickly become unused hooks -- take care to be efficient.
"
  (put symbol 'empire-check-hook label)
  (if enabledp
      (if (not (memq symbol empire-check-hooks))
	  (setq empire-check-hooks
		(cons symbol empire-check-hooks)))))

(defun empire-edit-check-hooks ()
  "Displays a buffer showing each hook, its description, and whether it is
enabled. Allows toggling of enabledness."
  (interactive)
  (let (hooks)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-check-hook)
			      (setq hooks (cons sym hooks))))))
    ;;(setq hooks (sort hooks 'string-lessp))
    (edit-hooks 'empire-check-hooks hooks
		       "*Hooks for Check Analysis*"
		       "\t\t\tEmpire Check Analysis Options"
)))

(defun check-empire ()
  "Runs all the empire-check-hooks.  These work from the emacs database
only.  Some hooks may issue automated commands."
  (interactive)
  (with-output-to-temp-buffer "*Checks*"
    (print-banner (format "Empire Checks as of %s" (current-time-string)))
    (let ((cb (current-buffer)))
      (set-buffer "*Checks*")
      (empire-data-mode t t)
      (sit-for 0)
      (mapcar '(lambda (test)
		 (end-of-buffer)
		 (terpri)
		 (print-banner (or (get test 'empire-check-hook)
				   (symbol-name test)))
		 (sit-for 0)
		 (message "Checking %s (%s)"
			   (or (get test 'empire-plane-check-hook)
				   (symbol-name test))
			   (current-hour-string))
		 (condition-case errorcode
		     (funcall test)
		   (error
		    (princ (format "  Hook %s aborted with error %s\n" test errorcode))))
		 (sit-for 0)
		 )
	      empire-check-hooks)
      (set-buffer cb)
      )
    ))

(defun check-spare ()
  "Checks for sectors from which people can be taken and placed in new sectors.
  The sector must be fully ours, efficient, happy, overpopulated, and must
  have mobility." 
  (let (
	(des (position-of 'des))
	(mob (position-of 'mob))
	(work (position-of 'work))
	(civ (position-of 'civ))
	(eff (position-of 'eff))
	)
    (walksects (if (and
		    (recall x y mob);; do we own this?
		    (< 50 (recall x y mob))
		    (< 90 (recall x y work))
		    (< 85 (recall x y eff))
		    (< (funcall empire-ideal-civ-fcn x y) (recall x y civ)))
		   (progn
		     (princ (concat "  "
			     (describe-sect-ideals x y)
			     "\n"))
		     (sit-for 0)
		     ))
	       "Spare")
    ))
(register-check-hook 'check-spare "Surplus Folks" t)

(defun check-security ()
  "Checks that the sector is likely to still be yours after the next update.
  Takes into account residual ownership, mil level, and happyness. Sectors
  with guns and shells are more tightly guarded than less dangerous sectors.
  Planes, missles, ships, and nukes are not considered."
  (let (
	(des (position-of 'des))
	(work (position-of 'work))
	(civ (position-of 'civ))
	(mil (position-of 'mil))
	(mob (position-of 'mob))
	(uw (position-of 'uw))
	(* (position-of '*))
	)
    (walksects-active
	 (if (string-equal "*" (recall x y *))
	     ;; Enemy sector
	     (if (< (recall x y mil)
		    (/ (+ (recall x	 y civ)
			  (recall x y uw)) 10))
		 (princ (format "  %senemy sector insufficiently controlled Mil is %s Need %s%s%s\n"
				(format-xy x y)
				(recall x y mil)
				(- (/ (+ (recall x	y civ)
					 (recall x y uw)) 10)
				   (recall x y mil))
				(if (> 0 (or (recall-macro x y 'gun) 0))
				    " *Guns*" "")
				(if (> 0 (or (recall-macro x y
							   'shell) 0))
				    " *Shells*" "")
				))
	       )
	   (if (> 90 (recall x y work))
	       (princ (format "  %s%s is becoming unhappy (%s) Mil is %s Need %s%s%s\n"
			      (format-xy x y)
			      (recall x y des)
			      (recall x y work)
			      (if (< (recall x y mil)
				     (/ (+ (recall x  y civ)
					   (recall x y uw)) 10))
				  "Insufficient" "OK")
			      (- (/ (+ (recall x  y civ)
				       (recall x y uw)) 10)
				 (recall x y mil))
			      (if (> 0 (or (recall-macro x y 'gun) 0))
				  " *Guns*" "")
			      (if (> 0 (or (recall-macro x y 'shell) 0))
				  " *Shells*" "")
			      )))
	   )
     "Security")
    ))
(register-check-hook 'check-security "Security" t)

(defvar empire-min-acceptable-plague-chance 0
  "*Defines the threshold beyond which the check-plague hook will flag the sector.")
(defun check-plague ()
  "Checks that the sector does not have a high chance of plague. Threshold
is controlled by empire-min-acceptable-plague-chance. Runs somewhat slow
because of the use of floating point numbers in the calcs."
  (require 'emp-sector)
  (let (chance
	)
    (walksects-active
     (if (< empire-min-acceptable-plague-chance
	    (setq chance (empire-calc-plague-chance x y)))
	 (princ (format "  %s%% Plague %s\n"
			chance
			(describe-sect-prod-dist x y))))
     "Plague Checking "
     )))
(register-check-hook 'check-plague "Plague" t)

(defun check-sector-distribution (x y)
  ;; do not doc here, do in check-dist
  (setq dx (recall x y (position-of dist_x)))
  (setq dy (recall x y (position-of dist_y)))
  (if (and dx dy (recall x y (position-of mob));; we own this sector
	   ;; we expect it to have a distribution
	   (not (or
		 ;; not a warehouse
		 (recall x y (position-of dependents))
		 ;;not mtn
		 (string-equal (recall x y (position-of des)) "^")
		 ;; Not road fead by another system
		 (and (string-equal (recall x y (position-of des)) "+")
		      (< 500 (recall x y (position-of food))))

		 ;; Maybe the efficiency and people count is
		 ;; so low that we do not care if dist works
		 (and (< (recall x y (position-of eff))
			 empire-sector-min-efficiency-check)
		      (< (+ (recall x y (position-of civ))
			    (recall x y (position-of mil))
			    (recall x y (position-of uw)))
			 empire-sector-min-people-check))
		 ))
	   ;; OK, so we want dist to work. See if there is a
	   ;; problem.
	   (or
	    ;; distibution is to self
	    (and (= x dx) (= y dy))
	    ;; has a distribution that is not ours
	    (not (and dx dy (recall dx dy (position-of mob)))))
	   )
      (if (and (= x dx) (= y dy))
	  (princ (format "  %s(%s %s%% %s/%s/%s %sf) has no distribution\n"
			 (format-xy x y)
			 (recall x y (position-of des))
			 (recall-macro x y 'eff)
			 (recall-macro x y 'civ)
			 (recall-macro x y 'mil)
			 (recall-macro x y 'uw)
			 (recall-macro x y 'food)
			 ))		
	(princ (format "  %s(%s) has a bad distribution %s,%s\n"
		       (format-xy x y) (recall x y (position-of des)) dx dy))))
  )

(defun check-dist ()
  "Checks the distribution each sector.  Checks to see that the sector has
a dist and that the dist is yours.  Only those sectors that are not w, ^,
or (roads with lots of food) are checked. Only efficient sectors with
adequate people are worth checking.  Note, however, that this can be
changed through the use of the variables
`empire-sector-min-efficiency-check' and 
`empire-sector-min-people-check').  These limits prevent checks in the
new growth areas. 

This hook makes no attempt to suggest dist paths for those sectors without
them.

This hook should not be enabled at the same time as the check-and-fix-dist
hook." 
  (let (
	dx dy
	)
    (mapsects-active 'check-sector-distribution "Distribution")
    ))
(register-check-hook 'check-dist "Distribution" nil)


(defun check-and-fix-sector-distribution (x y)
  ;; do not doc here, do in check-and-fix-dist
  (setq dx (recall x y (position-of dist_x)))
  (setq dy (recall x y (position-of dist_y)))
  (if (and dx dy (recall x y (position-of mob));; we own this sector
	   ;; we expect it to have a distribution
	   (not (or
		 ;; not a warehouse
		 (recall x y (position-of dependents))
		 ;;not mtn
		 (string-equal (recall x y (position-of des)) "^")
		 ;; Not road fead by another system
		 (and (string-equal (recall x y (position-of des)) "+")
		      (< 500 (recall x y (position-of food))))

		 ;; Maybe the efficiency and people count is
		 ;; so low that we do not care if dist works
		 (and (< (recall x y (position-of eff))
			 empire-sector-min-efficiency-check)
		      (< (+ (recall x y (position-of civ))
			    (recall x y (position-of mil))
			    (recall x y (position-of uw)))
			 empire-sector-min-people-check))
		 ))
	   ;; OK, so we want dist to work. See if there is a
	   ;; problem.
	   (or
	    ;; distibution is to self
	    (and (eql x dx) (eql y dy))
	    ;; has a distribution that is not ours
	    (not (and dx dy (recall dx dy (position-of mob)))))
	   )
      (if (and (= x dx) (= y dy))
	  ;; stub -- set up a distribution
	  ;; to warehouse if can find it
	  ;; else to dist node with the most sectors on it within 10 moves.
	  (let* ((examined) to-path
		 (junk
		  (if empire-interaction-verbosity
		      (progn
		      (message "Looking to set up a dist path for %s,%s" x y)
		      (show-pt (cons x y) t))))
		 (paths (find-dist-pts (cons x y) 10))
		 best-path
		 sector
		 new-des
		 ndist-path
		 (best-des "+")
		 (best-cnt 0))
	    (if (not paths)
		;; try to use each neighbor's path
		(mapcar
		 (function (lambda (nsect)
			     (if (and (setq ndist-path
					    (recall (car nsect) (cdr nsect)
						    (position-of dist_path)))
				      (not (string-equal ndist-path "_"))
				      (< (length ndist-path) 10))
				 (setq paths
				       (cons (cons (cons x y)
						   (path-to-r ndist-path nsect))
					     paths))
			       )
			     ))
		 (owned-sects-near (cons x y)))
		)
	    (if paths
		(progn			; choose best path
		  ;; set default best path
		  (setq sector (car (last (car paths))))
		  (setq new-des  (recall (car sector) (cdr sector)
					 (position-of des)))
		  (setq best-des new-des
			best-cnt (or (recall (car sector) (cdr sector)
					     (position-of dependents)) 0)
			best-path (car paths))
		  ;; try to find a better path
		  (setq paths (cdr paths))
		  (while paths
		    (setq sector (car (last (car paths))))
		    (setq new-des  (recall (car sector) (cdr sector)
					   (position-of des)))
		    (cond
		     ;; warehouses are always best
		     ((and (string-equal new-des "w")
			   (string-equal best-des "+"))
		      (setq best-des new-des
			    best-cnt (or (recall (car sector) (cdr sector)
						 (position-of dependents))
					 0)
			    best-path (car paths)))
		     ;; the one with the most dependents is best
		     ;;   (to avoid going to 1 shot sectors)
		     ((and (string-equal new-des best-des) ; w,w or +,+
			   (if (= (or (recall  (car sector) (cdr sector)
					       (position-of dependents)) 0)
				  best-cnt)
			       (< (length (car paths)) (length best-path))
			     (> (or (recall  (car sector) (cdr sector)
					     (position-of dependents)) 0)
				best-cnt)))
		      (setq best-des new-des
			    best-cnt (or (recall (car sector) (cdr sector)
						 (position-of dependents))
					 0)
			    best-path (car paths)))
		     (t			; +,w
		      nil))
		    (setq paths (cdr paths)))
		  (make-automated-command (format "dist %s,%s %s"
						  x y (r-to-path best-path))))
	      ;; else no available path
	      (princ (format "  %s(%s %s%% %s/%s/%s %sf) has no distribution\n"
			     (format-xy x y)
			     (recall x y (position-of des))
			     (recall-macro x y 'eff)
			     (recall-macro x y 'civ)
			     (recall-macro x y 'mil)
			     (recall-macro x y 'uw)
			     (recall-macro x y 'food)
			     ))		
	      )
	    )
	(princ (format "  %s(%s) has a bad distribution %s,%s\n"
		       (format-xy x y) (recall x y (position-of des)) dx dy))))
  )

(defun check-and-fix-dist ()
  "Checks the distribution each sector and suggests one if possible.
Checks to see that the sector has a dist and that the dist is yours.  Only
those sectors that are not w, ^, or (roads with lots of food) are checked.
Only efficient sectors with adequate people are worth checking.  Note,
however, that this can be changed through the use of the variables
`empire-sector-min-efficiency-check' and
`empire-sector-min-people-check').  These limits prevent checks in the new
growth areas.  

Proposed dist paths are done first by looking for a road connection to a
distribution point (found by empire-flows). If many are found, the one used
is the one meeting the following criterion:
  Warehouses are perferred over anything else
  given two warehouses, or two non-warehouses, the one chosen is the one
   already supplying the most sectors (as calculated by empire-flows). (This
   keeps 'one shot' dist nodes from supplying more than was intended.
  Given a tie, the shortest path is proposed.

If no roadway connected distribution is found, all neighbors are checked.
If any has a dist path of length 9 or less, it is considered as a
candidate. If multiple neighbors have dist paths, the one chosen is the one
fitting the evaluation criterion given above. Note that in some cases this
will cause a sector to cross multiple non-roads to get to a more loaded
distribution point when there is a path to another dist pt that crosses
fewer non roads. 

Because of the use of dist point loading as a decision criterion, this tool
works best if empire-flows has been done since the last time dist was
messed with.

The proposal of alternate distribution paths checks can take awhile. If
this is a problem, set the min-eff-check higher or use the check-dist hook
instead of this one.

This hook should not be enabled at the same time as the check-dist hook."
  (let (
	dx dy
	)
    (mapsects-active 'check-and-fix-sector-distribution "Distribution with suggestions")
    ))
(register-check-hook 'check-and-fix-dist "Distribution with suggestions" t)

(defun check-sector-food (x y)
  "Checks that the food currently in the sector will last 1 day, or at
  least one update. Does not account for food that may be growing during the
  update."
  (setq people (+ (recall x y civ)
		  (recall x y mil)
		  (recall x y uw)))
  (setq need (/ (+ (* people etu-per-update upd-per-day)
		   bodies-fead-by-1-food-per-etu)
		bodies-fead-by-1-food-per-etu))
  (setq urgent-need (/ (+ (* people etu-per-update)
			  bodies-fead-by-1-food-per-etu)
		       bodies-fead-by-1-food-per-etu))
  (if (and
       (not (string-equal "^" (recall x y des)))
       (> people 40)
       (> 3 (- (recall x y food) need))
       (< 20 (+ (recall x y civ)
		(recall x y mil)
		(recall x y uw)))
       )
      (progn
	(princ (format "  %s%s%s %s people will eat %s has %s Needs %s/%s\n"
		       (format-xy x y)
		       (recall x y des)
		       (if (> urgent-need (recall x y food))
			   "!" " ")
		       people need (recall x y food)
		       (- need (recall x y food))
		       (- urgent-need (recall x y food))
		       ))
	(sit-for 0))
    )
  )

(defun check-food ()
  "Checks that the food currently in each sector will last 1 day, or at
  least one update. Does not account for food that may be growing during the
  update.

  Sectors with urgent needs are identified with an ! after their designation.
  The first number shown as a need is the amount for 1 day, the second
  number is the amount needed by the next update"
  
  (let (people
	need
	urgent-need
	(des (position-of 'des))
	(food (position-of 'food))
	(civ (position-of 'civ))
	(mil (position-of 'mil))
	(uw (position-of 'uw))
	(mob (position-of 'mob))
	)
    (mapsects-active 'check-sector-food "Food")))
(register-check-hook 'check-food "Food" t)

(defun check-sectors-lost ()
  "Reports all those sectors that should be ours but which have been lost."
  (walksects-nontrivial-inactive
   (if (and				; do we own this?
	(recall-macro x y 'dist_x)	; did we own this
	(recall-macro x y 'civ)		; was it land
	(> (recall-macro x y 'civ) 5)
	(recall-macro x y 'own)   ; somebody else's now
	)
       (progn
	 (princ (message "  %s%s has been %s."
			 (format-xy x y)
			 (recall-macro x y 'des)
			 (if (= 999 (recall-macro x y 'own))
			     "lost" "taken")
			 ))
	 (terpri)
	 ;;(record x y (position-of 'civ) nil)
	 (sit-for 0)
	 )
     )
   "Lost"))
(register-check-hook 'check-sectors-lost "Lost" t)

;;(defun check-btu ()
;;  (mapcar '(lambda (facts)
;;	       (if (string-equal (ffact 'type) "c")
;;		   (princ (format "Sector %s,%s with %s civs will make %s BTUs\n"
;;			      (car (car facts)) (cdr (car facts))
;;			      (ffact 'civ)
;;			      (/ (* (ffact 'civ) etu-per-update upd-per-day) 25)
;;			      ))))
;;	    cfacts)
;;    )
;;
;;(defun check-income ();; no calc of reserve cost
;;  (let (cm mm um people (total 0))
;;    (mapcar '(lambda (facts)
;;	       (setq people (+ (ffact 'civ) (ffact 'mil) (ffact 'uw)))
;;	       (setq cm (* (ffact 'civ) 833 etu-per-update upd-per-day))
;;	       (setq um (* (ffact 'uw) 378 etu-per-update upd-per-day))
;;	       (setq mm (* (ffact 'mil) -8333 etu-per-update upd-per-day))
;;	       (princ (format "Sector %s with %s(%s) people will make %s\n"
;;			      (car facts) people (ffact 'mil)
;;			      (/ (+ mm um cm) 1000)))
;;	       (setq total (+ (/ (+ mm um cm) 1000) total)))
;;	    cfacts)
;;    (princ (format "	 Total: %s\n" total))
;;    ))
  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dump Reading and related hooks
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun register-dump-hook (symbol enabledp)
  "Places the function named by SYMBOL as a possible hook. If ENABLEDP
is non nil, the function is in the initial set of dump hooks.

Each hook is called for EACH sector when a dump is read non-quickly. The
hook has dynamically bound values for EACH variable in the dump, even if
the variable is not stored in the secotr database permanently. i.e. the
hook can refer to eff, mil, civ, f_dist, ... directly and have the correct
values bound to the variables. The preexisting values (those from before
reading the dump line) are available via the recall function.

The hook can do anything, but mainly it is supposed to use prin1 and princ
to write messages. The standard-output is already directed to the *Dump*
buffer before the hook is called.

Note that the hook will be called lots of times and dump reading is the
slowest thing the tool does -- to take care to be efficient.
"
  (put symbol 'empire-dump-hook t) ;; Make it a dump hook
  (if enabledp
      (if (not (memq symbol empire-dump-hooks))
	  (setq empire-dump-hooks
		(cons symbol empire-dump-hooks)))))

(defun empire-edit-dump-hooks ()
  "Displays a buffer showing each hook, its description, and whether it is
enabled. Allows toggling of enabledness."
  (interactive)
  (let (hooks)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-dump-hook)
			      (setq hooks (cons sym hooks))))))
    ;;(setq hooks (sort hooks 'string-lessp))
    (edit-hooks 'empire-dump-hooks hooks
		       "*Hooks for Dump Analysis*"
		       "\t\t\tEmpire Dump Scan Analysis Options"
)))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dump data hooks
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-dump-check-levels ()
  "Checks that the levels are appropiate to the sector type.
    Food is checked relative to the population
    Then a whole series or sector specific checks are performed.

    This is probably a good fcn to customize or replace."
  (if (not (or (string-equal dist_path "_")
	       (< 4 (or (recall x y (position-of dependents)) 0))))
      (progn
	(setq current-dump-hook 'empire-dump-check-levels)
	;; first, check the food
	(if (and (not (or (string-equal des "w")
			  (string-equal des "^")
			  (string-equal des "a")
			  (string-equal des "=")))
		      (< f_dist (/ (+ civ mil uw) 5)))
	    (make-automated-command
	     (format "thr f %s,%s %s"
		     x y
		     (min (if (= 0 f_cut) 999 f_cut)
			  ;; leave room for growth  and disruption
			  (max f_dist (/ (+ civ mil uw) 3)))
		     )))
	;; Check levels appropiate to designation
	(cond
	 ;; pure consumers
	 ((string-equal des "l") ;; library
	  ;; adjustment is optional
	  (level-should-be "i" i_dist i_cut iron 0 200 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 300 120)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1) ;; 10 pet in case
							;; of fighter 1
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  )

	 ((string-equal des "p") ; park
	  ;; adjustment is optional
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 300 120)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((string-equal des "n") ; nuke
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 200 150)
	  (level-should-be "r" r_dist r_cut rad 1 400 200)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 240 200)
	  (level-should-be "h" h_dist h_cut hcm 1 240 200)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )
   

	 ;; pure producers
	 ((string-equal des "m") ; mine
	  (level-should-be "i" i_dist i_cut iron 1 200 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be min 60)

	  ;; check for constipation
	  (constipatedp "i" 'iron iron i_dist i_cut)
	  )

	 ((string-equal des "g") ; gold
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 10 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be gold 1)

	  ;; check for constipation
	  (constipatedp "d" 'dust dust d_dist d_cut)
	  )

	 ((string-equal des "u"); uran
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 1 10 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be uran 1)

	  (constipatedp "r" 'rad rad r_dist r_cut)
	  )

	 ((string-equal des "o")	; oil
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 10 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be ocontent 1)

	  (constipatedp "o" 'oil oil o_dist o_cut)
	  )

	 ;; producers and consumers
	 ((string-equal des "i") ; shell
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 1 100 1)
	  (level-should-be "l" l_dist l_cut lcm 10 300 50)
	  (level-should-be "h" h_dist h_cut hcm 5 200 25)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "s" 'shell shell s_dist s_cut)
	  )
   
	 ((string-equal des "d") ; defense
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 100 30)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 2 200 50)
	  (level-should-be "h" h_dist h_cut hcm 1 200 25)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "d" 'gun gun g_dist g_cut)
	  )
   
	 ((string-equal des "j") ; light
	  (level-should-be "i" i_dist i_cut iron 10 400 200)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 100 1)
	  (level-should-be "h" h_dist h_cut hcm 0 100 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "l" 'lcm lcm l_dist l_cut)
	  )

	 ((string-equal des "k") ; heavy
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "i" i_dist i_cut iron 10 400 200)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 100 1)
	  (level-should-be "h" h_dist h_cut hcm 1 100 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "h" 'hcm hcm h_dist h_cut)
	  )

	 ((string-equal des "%") ; refinary
	  (level-should-be "o" o_dist o_cut oil 1 100 5)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 0 1 1)
	  (level-should-be "p" p_dist p_cut pet 1 1 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "p" 'pet pet p_dist p_cut)
	  )

	 ((string-equal des "b") ; bank
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 10 1)
	  (level-should-be "d" d_dist d_cut dust 20 500 200)
	  (level-should-be "b" b_dist b_cut bar 100 999 999)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ;; Occasional consumers and local surpluses
	 ((or (string-equal des "*")	; airport
	      (string-equal des "h")	; harbor
	      )
	  (level-should-be "i" i_dist i_cut iron 0 999 1)
	  (level-should-be "o" o_dist o_cut oil 0 999 1)
	  (level-should-be "r" r_dist r_cut rad 0 999 1)
	  (level-should-be "d" d_dist d_cut dust 0 999 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 1 999 50)
	  (level-should-be "l" l_dist l_cut lcm 30 999 120)
	  (level-should-be "h" h_dist h_cut hcm 30 999 120)
	  (level-should-be "g" g_dist g_cut gun 0 40 1)
	  (if (string-equal des "*")
	      (level-should-be "p" p_dist p_cut pet 10 150 50)
	    (level-should-be "p" p_dist p_cut pet 0 10 1))
	  )

	 ((string-equal des "f") ; fort
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  ;; guns and shells only if fort is efficient or has mil
	  (if (or  (> eff 50) (> mil 50))
	      (progn
		(level-should-be "s" s_dist s_cut shell 20 200 50)
		(level-should-be "g" g_dist g_cut gun 7 25 10)
		(actual-should-be "s" s_dist s_cut shell 20 500 50)
		(actual-should-be "g" g_dist g_cut gun 7 25 10)
		(actual-should-be "m" 0 0 mil 30 300 50)
		)
	      (progn
		(level-should-be "s" s_dist s_cut shell 0 500 100)
		(level-should-be "g" g_dist g_cut gun 0 25 20)
		)
	    )
	  )
			

	 ((or (string-equal des "+") ; road
	      (string-equal des ")") ; radar
	      (string-equal des "^") ; mountan
	      (string-equal des "-") ; wilderness
	      )
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 0 1 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )


	 ((string-equal des "#") ; bridge
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 1 300 300)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ;; Ag center
	 ;;   food thr < 500
	 ;;   dist on food > 500 to road
	 ;;   lev on this road
	 ((or (string-equal des "a") (string-equal des "="))
	  (level-should-be "f" f_dist f_cut food 20 500 100)

	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (if (string-equal des "a")
	      (resource-should-be fert 70))

	  ;; check for constipation
	  (constipatedp "f" 'food food f_dist f_cut)
	  )

	 ((or (string-equal des "e")
	     (string-equal des "c"))
	  ;; should have nothing
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((or (string-equal des "r")
	      (string-equal des "t")
	      )
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  ;;(level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  ;;(level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  ;;(level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((string-equal des "w")
	  ;; do nothing
	  )
	 
	 (t
	  (princ (format "%s\t%s Cannot handle\n" (format-xy x y) des)))

	 ))
    ;; Else
    ;; this has no distribution
    ))
(register-dump-hook 'empire-dump-check-levels t)

(defun empire-dump-check-levels-no-food ()
  "Checks that the levels are appropiate to the sector type.
    Food is checked relative to the population
    Then a whole series or sector specific checks are performed.

    This is probably a good fcn to customize or replace."
  (if (not (or (string-equal dist_path "_")
	       (< 4 (or (recall x y (position-of dependents)) 0))))
      (progn
	(setq current-dump-hook 'empire-dump-check-levels)
	;; Check levels appropiate to designation
	(cond
	 ;; pure consumers
	 ((string-equal des "l") ;; library
	  ;; adjustment is optional
	  (level-should-be "i" i_dist i_cut iron 0 200 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 300 120)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1) ;; 10 pet in case
							;; of fighter 1
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  )

	 ((string-equal des "p") ; park
	  ;; adjustment is optional
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 240 120)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((string-equal des "n") ; nuke
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 200 100)
	  (level-should-be "r" r_dist r_cut rad 1 400 200)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 200 120)
	  (level-should-be "h" h_dist h_cut hcm 1 200 120)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )
   

	 ;; pure producers
	 ((string-equal des "m") ; mine
	  (level-should-be "i" i_dist i_cut iron 1 200 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be min 60)

	  ;; check for constipation
	  (constipatedp "i" 'iron iron i_dist i_cut)
	  )

	 ((string-equal des "g") ; gold
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 10 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be gold 1)

	  ;; check for constipation
	  (constipatedp "d" 'dust dust d_dist d_cut)
	  )

	 ((string-equal des "u"); uran
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 1 10 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be uran 1)

	  (constipatedp "r" 'rad rad r_dist r_cut)
	  )

	 ((string-equal des "o")	; oil
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 10 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (resource-should-be ocontent 1)

	  (constipatedp "o" 'oil oil o_dist o_cut)
	  )

	 ;; producers and consumers
	 ((string-equal des "i") ; shell
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 1 100 1)
	  (level-should-be "l" l_dist l_cut lcm 10 300 50)
	  (level-should-be "h" h_dist h_cut hcm 5 200 25)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "s" 'shell shell s_dist s_cut)
	  )
   
	 ((string-equal des "d") ; defense
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 1 100 30)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 2 200 50)
	  (level-should-be "h" h_dist h_cut hcm 1 200 25)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "d" 'gun gun g_dist g_cut)
	  )
   
	 ((string-equal des "j") ; light
	  (level-should-be "i" i_dist i_cut iron 10 400 200)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 1 100 1)
	  (level-should-be "h" h_dist h_cut hcm 0 100 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "l" 'lcm lcm l_dist l_cut)
	  )

	 ((string-equal des "k") ; heavy
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "i" i_dist i_cut iron 10 400 200)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 100 1)
	  (level-should-be "h" h_dist h_cut hcm 1 100 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "h" 'hcm hcm h_dist h_cut)
	  )

	 ((string-equal des "%") ; refinary
	  (level-should-be "o" o_dist o_cut oil 1 100 5)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 0 1 1)
	  (level-should-be "p" p_dist p_cut pet 1 1 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)

	  ;; check for constipation
	  (constipatedp "p" 'pet pet p_dist p_cut)
	  )

	 ((string-equal des "b") ; bank
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 10 1)
	  (level-should-be "d" d_dist d_cut dust 20 500 200)
	  (level-should-be "b" b_dist b_cut bar 100 999 999)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ;; Occasional consumers and local surpluses
	 ((or (string-equal des "*")	; airport
	      (string-equal des "h")	; harbor
	      )
	  (level-should-be "i" i_dist i_cut iron 0 999 1)
	  (level-should-be "o" o_dist o_cut oil 0 999 1)
	  (level-should-be "r" r_dist r_cut rad 0 999 1)
	  (level-should-be "d" d_dist d_cut dust 0 999 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 1 999 50)
	  (level-should-be "l" l_dist l_cut lcm 30 999 120)
	  (level-should-be "h" h_dist h_cut hcm 30 999 120)
	  (level-should-be "g" g_dist g_cut gun 0 40 1)
	  (if (string-equal des "*")
	      (level-should-be "p" p_dist p_cut pet 10 150 50)
	    (level-should-be "p" p_dist p_cut pet 0 10 1))
	  )

	 ((string-equal des "f") ; fort
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  ;; guns and shells only if fort is efficient or has mil
	  (if (or  (> eff 50) (> mil 50))
	      (progn
		(level-should-be "s" s_dist s_cut shell 20 200 50)
		(level-should-be "g" g_dist g_cut gun 7 25 10)
		(actual-should-be "s" s_dist s_cut shell 20 500 50)
		(actual-should-be "g" g_dist g_cut gun 7 25 10)
		(actual-should-be "m" 0 0 mil 30 300 50)
		)
	      (progn
		(level-should-be "s" s_dist s_cut shell 0 500 100)
		(level-should-be "g" g_dist g_cut gun 0 25 20)
		)
	    )
	  )
			

	 ((or (string-equal des "+") ; road
	      (string-equal des ")") ; radar
	      (string-equal des "^") ; mountan
	      (string-equal des "-") ; wilderness
	      )
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 0 1 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((string-equal des "#") ; bridge
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 1 1)
	  (level-should-be "h" h_dist h_cut hcm 1 100 300)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )


	 ;; Ag center
	 ;;   food thr < 500
	 ;;   dist on food > 500 to road
	 ;;   lev on this road
	 ((or (string-equal des "a") (string-equal des "="))
	  (level-should-be "f" f_dist f_cut food 20 500 104)

	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  (if (string-equal des "a")
	      (resource-should-be fert 70))

	  ;; check for constipation
	  (constipatedp "f" 'food food f_dist f_cut)
	  )

	 ((or (string-equal des "e")
	     (string-equal des "c"))
	  ;; should have nothing
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  (level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  (level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  (level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((or (string-equal des "r")
	      (string-equal des "t")
	      )
	  (level-should-be "i" i_dist i_cut iron 0 10 1)
	  ;;(level-should-be "o" o_dist o_cut oil 0 1 1)
	  (level-should-be "r" r_dist r_cut rad 0 1 1)
	  ;;(level-should-be "d" d_dist d_cut dust 0 1 1)
	  (level-should-be "b" b_dist b_cut bar 0 1 1)
	  (level-should-be "s" s_dist s_cut shell 0 10 1)
	  ;;(level-should-be "l" l_dist l_cut lcm 0 10 1)
	  (level-should-be "h" h_dist h_cut hcm 0 10 1)
	  (level-should-be "p" p_dist p_cut pet 0 10 1)
	  (level-should-be "g" g_dist g_cut gun 0 1 1)
	  )

	 ((string-equal des "w")
	  ;; do nothing
	  )
	 
	 (t
	  (princ (format "%s\t%s Cannot handle\n" (format-xy x y) des)))

	 ))
    ;; Else
    ;; this has no distribution
    ))
(register-dump-hook 'empire-dump-check-levels-no-food t)

;; need check that cut not < level

(defun level-should-be (item-command-name level cut actual min max ideal)
  "Args are ITEM-COMMAND-NAME LEVEL CUT ACTUAL MIN MAX IDEAL. Issues
commands or suggestions to set the level for ITEM-COMMAND-NAME to IDEAL if
it is not between MIN and MAX (inclusive). ITEM-COMMAND-NAME is the name of
the commodity as known to empire and is used to construct the threshold
command. 

Also Warns if the del amount in CUT is less than the ACTUAL level.

Presumes that x and y are set by the caller.

IDEAL may be the name of a function or a number."

  (if (and (not (= 0 cut)) (> level cut))
      ;; endless flow -- ocassionally useful but often a problem
      (progn
	(princ (format "  %s%s has endless flow of %s level=%s cut=%s\n"
		       (format-xy x y) des item-command-name level cut))
	(sit-for 0)
	)
    ;; else
    (if (or (> level max)
	    (< level min)
	    ;; see if we have leftover of this commodity
	    (and (= 0 level) (> actual max))
	    )
	(progn
	  (make-automated-command
	   (format "thr %s %s,%s %s"
		   item-command-name x y
		   (if (symbolp ideal) (funcall ideal) ideal)))
	  (sit-for 0)
	  )))
  )

(defun actual-should-be (item-command-name level cut actual min max ideal)
  "Args are the same as level-should be (ITEM-COMMAND-NAME LEVEL CUT ACTUAL
MIN MAX IDEAL) but issues warnings if ACTUAL and not within the range MIN
and MAX (inclusive). ITEM-COMMAND-NAME is the name of the commodity as
known to empire. Thus, this function notifies if contents are not up to the
required amounts. 

Presumes that x and y are set by the caller.

(The other args are only for easy conversion between calls to
level-should-be and actual-should-be)."
    (if (or (> actual max)
	    (< actual min)
	    )
	(progn
	  (princ (format "  %s%s needs %s %s has %s\n"
			 (format-xy x y) des ideal item-command-name actual))
	  (sit-for 0)
	  ))
  )
  
(defun resource-should-be (actual min-allowed)
  "Suggests redesignation if ACTUAL is less than MIN. If called from a dump
hook, this can check resource levels.

Presumes that x and y are set by the caller."
  (if  (and (< actual min-allowed) (string-equal sdes "_"))
      (progn
	(princ (format "  %s%s needs redesignation min:%s gold:%s fert:%s oil:%s uran:%s\n"
		       (format-xy x y) des
	       min gold fert ocontent uran))
	(if empire-batch-play
	    (message "  %s%s needs redesignation min:%s gold:%s fert:%s oil:%s uran:%s"
		       (format-xy x y) des
	       min gold fert ocontent uran))
	(sit-for 0)
	))
  )
  
(defun constipatedp (item-command-name emacs-db-symbol actual dist cut)
  "Args are ITEM-COMMAND-NAME EMACS-DB-SYMBOL ACTUAL DIST CUT. 
ACTUAL is typically the actual amount of ITEM-COMMAND-NAME in a sector,
DIST is the dist level for that item, and CUT is the del level for that item.

If ACTUAL is greater than DIST and CUT is 0 (the goods are not being delivered),
this function will attenpt to issue a del command to move the goods to a
road or bridge which has a dist point and whose dist point does not already
have 9k or more of the item (identified by EMACS-DB-SYMBOL). If such a
road cannot be found, a del is set up to any road. Otherwise issues a warning"
  ;; check for constipation
  ;; Problem -- will deliver to a road whose warehouse is full
  (if (or (and (> actual dist);; they are not moving out
	       (= 0 cut);; they have no where else to go
	       )
	  (> actual 920))
      ;; we have constipation, can we do anything about it?
      (if (= 0 cut)
	  ;; look for adjacient road with valid warehouse that is not full
	  (let* ((mob (position-of 'mob))
		 (sects (owned-sects-near (cons x y)))
		 (road-sect nil)
		 warehoused
		 rdx rdy
		 )
	    (while (and sects (not warehoused))
	      ;; try del to +, not =
	      (setq dest-des (recall (car (car sects))
				     (cdr (car sects))
				     (position-of 'des)))
	      (if (string-equal "+" dest-des)
		  (progn
		    (setq road-sect (car sects)
			  rdy nil)
		    (setq warehoused 
			  (not (or (and (= (car road-sect)
				       (setq rdx (recall-macro (car road-sect)
							       (cdr road-sect) 'dist_x)))
				    (= (cdr road-sect)
				       (setq rdy (recall-macro (car road-sect)
							       (cdr road-sect) 'dist_y)))
				    )
				    (> (or (recall rdx (or rdy
							   (recall-macro (car road-sect)
									 (cdr road-sect) 'dist_y))
							   
						   (dynamic-position-of emacs-db-symbol))
					   0) 9000))
			       ))
		    ))
	      (setq sects (cdr sects)))
	    (if road-sect
		(make-automated-command
		 (format "del %s %s,%s (%s) %s"
			 item-command-name x y (max 100 (min (* dist 2) 900))
			 (r-to-path (list (cons x y) road-sect))))
	      (princ (format "  %s%s is constipated not near a road feeding an available dist point\n"
			     (format-xy x y) des))
	      ))
	;; else
	(princ (format
		"  %s%s Is getting constipated and nothing can be done\n"
		(format-xy x y) des))
	)
    ))

(defun empire-dump-check-work ()
  "Detects when a sector's work rating drops since the last read"
  (setq current-dump-hook 'empire-dump-check-work)
  (if (< work (or (recall x y (position-of 'work)) 0))
	    (princ (format "  %s%s(%s%%->%s%%) is dropping in work\n"
			   (format-xy x y)
			   des
			   (recall x y (position-of 'work))
			   work))))
(register-dump-hook 'empire-dump-check-work t) ;; Make it a dump hook

(defun threshold (label num pos-thresh)
  (if (or (< pos-thresh	 num) (> 0 num))
      (progn
	(setq flag t)
	(format "%s:%s "
		label num))
    ""))

(defmacro delta (label pos-thresh)
  (`
   (let ((num
	  (- (or (eval (, label)) 0) (or (recall x y (position-of (, label)))
					 0))))
     (if (or (< (, pos-thresh) num) (> 0 num))
	 (progn
	   (setq flag t)
	   (format "%s:%s "
		   (, label) num))
       ""))))
  
(defun empire-dump-check-some-progress ()
  "Prints deltas of critical variables. Is somewhat slow. Good mainly for
    worlds in danger of revolt or reversion.

   The delta is comparing the dump values to stored values; this is likely
   to be useful mainly just after an update."
  (setq current-dump-hook 'empire-dump-check-some-progress)
  (if (recall x y (position-of 'mob));; we knew this sector
      (let* ((flag nil)
	    (delta-list (list (delta 'eff 5)
			      (delta 'civ 50)
			      (delta 'mil 5)
			      (delta 'uw 30)
			      (delta 'work 5)
			      "\n"
			      ))
	    )
	(if flag
	  (princ (apply 'concat (format "  %s%s Delta "
					(format-xy x y) des
					)
			delta-list
			)))
	  )))
(register-dump-hook 'empire-dump-check-some-progress nil) ;; Make it a dump hook

(defun empire-dump-check-all-progress ()
  "Prints the deltas of all major ratings. Is quite slow. Good mainly to
     check health of young expanding worlds.

   The delta is comparing the dump values to stored values; this is likely
   to be useful mainly just after an update."
  (setq current-dump-hook 'empire-dump-check-all-progress)
  (if (recall x y (position-of 'mob));; we knew this sector
      (let* ((flag nil)
	    (delta-list (list (delta 'eff 2)
			      (delta 'mob 60)
			      (delta 'civ 1)
			      (delta 'mil 2)
			      (delta 'uw 2)
			      (delta 'food 50)
			      (delta 'work 5)
			      (delta 'avail 5)
			      "\n"))
	    )
	(if flag
	  (princ (apply 'concat (format "  %s%s \t%s Delta "
					(format-xy x y) des
					)
			delta-list
			)))
	  )))

(register-dump-hook 'empire-dump-check-all-progress nil) ;; Make it a dump hook
;;(setq empire-dump-hooks nil)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Low cost routes (may take time to calc)
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Define macros to deal with short-path maps.
;;; a shortest path map is an associative array with a
;;; sector (x . y) as a key and a (cost path) as the
;;; value.
;;;
;;; '( (X . Y) cost path )

(defun cost-map-x (m)
  (caar m))

(defun cost-map-y (m)
  (cdar m))

(defun cost-map-cost (m)
  (cadr m))

(defun cost-map-path (m)
  (caddr m))

(defun better-path-p ( a b )
  (or (f< (cost-map-cost a) (cost-map-cost b))
      (and (f= (cost-map-cost a) (cost-map-cost b))
	   (< (length (cost-map-path a))
	      (length (cost-map-path b)))))
  )

(defun least-expensive-sector (sect-list border-list good-coord-func
					 good-path-func)
  "takes SECT-LIST and BORDER-LIST which are shortest path maps
   and returns (cons SECT-LIST BORDER-LIST) after the best of
   the borders has been added on and the border-list updated.

   This function is basically the search part of a breadth-first
search.  all the sectors in sect-list are accepted as having the
least-cost path in their map entry.  All the sectors on border list
are also assumed to have the shortest path in their entry (NOTE! this
is NOT a generic breadth-first.  empire has special characteristics).
We then select the border sector with the least cost (use shortest
path to break ties) and add its neighbors to the border list (not
overriding current border entries ever, another optimization).

   This is the most advanced I've developed yet.  It keeps the
   border list sorted, which can speed things up considerably.

   I need to improve this function with some more predicates.  The
good-path-func can be (and has been) used to restrict path calculation
to sectors reachable from a warehouse."

  (let ( (best-sect nil)
	 best-coord	best-cost	best-path
	 (rest-of-list border-list)
	 current-sector
	 )

    ;; remove the first sector on the border-list and put it on the sect list
    (setq best-sect border-list)
    (setq border-list (cdr border-list))
    (setcdr best-sect sect-list)
    (setq sect-list best-sect)
    (setq best-sect (car best-sect))

    (let ( (directions-to-try '(?y ?u ?g ?j ?b ?n))
	   (x (cost-map-x best-sect))
	   (y (cost-map-y best-sect))
	   (this-cost (cost-map-cost best-sect))
	   (this-path (cost-map-path best-sect))
	   nx	ny
	   new-sector
	   new-cost
	   new-path
	   )
      ;; add the sectors adjacent to the newcomer to the border-list
      (while (setq dir (car directions-to-try))
	(setq directions-to-try (cdr directions-to-try))

	(setq new-sector (sector-in-direction x y dir))
	(if (funcall good-coord-func new-sector)
	    (progn
	      (setq nx (car new-sector)
		    ny (cdr new-sector)
		    current-sector nil
		    new-path (concat this-path (char-to-string dir))
		    delta-cost ( calc-mob-to-sector-float
				 _f1
				 (recall nx ny (position-of 'des))
				 (recall nx ny (position-of 'eff)))
		    new-cost (f+ this-cost delta-cost )
		    new-sector (list new-sector new-cost new-path))
	      
	      ;; if it's a new sector then add it to the border list
	      (if (and (funcall good-path-func new-sector)
		       (f<= _f0 delta-cost)
		       (not (assoc (car new-sector) sect-list))
		       (not (assoc (car new-sector) border-list)
			    ))
		  ;; if there's no border list or our sector is better than
		  ;; the head, then insert it at the front
		  (if (or (not border-list)
			  (not (better-path-p (car border-list) new-sector)) )
		      (setq border-list (cons new-sector border-list))
		    ;; else scan along the border-list for the place it belongs
		    (do ( (scan border-list (cdr scan))
			  )
			( (not (and (cdr scan)
				    (better-path-p (cadr scan) new-sector)))
			  (setcdr scan (cons new-sector (cdr scan)))
			  )
		      )
		    )
		)
	      )
	  )
	)
      )
    )
  (cons sect-list border-list)
  )

(defun least-cost-path (a b)
  "computes the least mob cost path from point A to point B (both cons x y).
Also considers path length important (no more squiggling across
roads spread all over creation just to go next door)."

  (let ( (border-list (list (list a _f0 "")))
	 (sect-list nil)
	 pair
	 terminus
	 )
    (while (and border-list
		(not (setq terminus (assoc b border-list) ) ) )
      (setq pair (least-expensive-sector sect-list border-list
					 (function (lambda (sect)
						     (recall (car sect) (cdr sect)
							     (position-of 'mob))))
					 (function (lambda (sect) t))
					 ))
      (setq sect-list (car pair)
	    border-list (cdr pair))

      ;; (message "%d,%d" (caaar sect-list) (cdaar sect-list))
      )
    (if terminus (caddr terminus)
      "no path (are you from another continent?)" ) ; pitiful error handling
    )
  )


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Redistribution
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun find-open-highway (from)
  (let ((examined  (list from))
	(des (position-of 'des))
	(* (position-of '*))
	(food (position-of 'food))
	(civ (position-of 'civ))
	(eff (position-of 'eff))
	(depth 0)
	)
    (find-open-highway-internal from)
    ))


(defun open-highwayp (pt)
  (and
   (string-equal "+" (recall (car pt) (cdr pt) des))
   (not (string-equal "*" (recall (car pt) (cdr pt) *)))
   (<= 800 (or (recall (car pt) (cdr pt) food) 900))
   (<= 85 (or (recall (car pt) (cdr pt) eff) 0))
   (eql 100 (recall (car pt) (cdr pt) work))
   (> (- 850 delta) (recall (car pt) (cdr pt) civ))))
   
(defun find-open-highway-internal (from)
  (let (sects
	next
	path)
    (if (> (setq depth (1+ depth)) empire-max-path-depth)
	nil
      ;; first, see if this one can take it
      (if (open-highwayp from)
	  (progn
	    (setq depth (1- depth))
	    (list from))
	;; Place ourselves on the examined list
	(setq examined (cons from examined))
	;; Try each unexamined highway neighbor
	(setq sects (owned-sects-near from))
	(while (and sects (not path))
	  (setq next (car sects))
	  (if (and (not (member next examined))
		   (string-equal (recall (car next) (cdr next) des) "+"))
	      ;; This might lead to the way
	      (setq path (find-open-highway-internal next))
	    )
	  (setq sects (cdr sects))
	  )
	(setq depth (1- depth))
	(if path (cons from path))
	))))

(defun redistribute-population-internal (x y)
  (if (and (recall x y des)
	   (eql (recall x y work) 100)
	   (< 0 (setq delta (- (recall x y civ) 900)))
	   (not (string-equal "*" (recall x y *)))
	   (< 95 (recall x y eff))
	   )
      (progn
	(if (setq path (find-open-highway (cons x y)))
	    (progn
	      (sit-for 0)
	      (setq dest (car (last path)))
	      (princ (format "\tBumped %s,%s up by %s from %s people (Moved from %s,%s (%s))\n" (car dest) (cdr dest) delta (recall (car dest) (cdr dest) civ) x y (recall x y civ)))
	      (make-automated-command (format "move c %s,%s %s %s"
					      x y
					      delta
					      (r-to-path path)))
	      (record (car dest) (cdr dest) civ
		      (+ delta (recall (car dest) (cdr dest) civ)))
	      ))))
  )

(defun redistribute-population ()
  "Moves surplus people from sectors to highways that can hold them.
Surplus is any over 900. This is for fast pop growth because people are $
if you have the iron and production to educate them and keep them happy."
  (interactive)
  (with-output-to-temp-buffer "*Population*"
    (princ (format "\t*** Population Redistribution as of %s ***\n\n"
		   (current-time-string)))
    (empire-data-mode-buffer t t "*Population*")
    (let ((des (position-of 'des))
	  (civ (position-of 'civ))
	  (eff (position-of 'eff))
	  (* (position-of '*))
	  (work (position-of 'work))
	  (y (- 0 empire-width))
	  delta
	  dest
	  path)
      (mapsects-active 'redistribute-population-internal "Depopulating maxed out sectors")
      )))
(put 'redistribute-population 'empire t)

(defun efficient-highwayp (pt) 
  (let ((des  (recall (car pt) (cdr pt) des)))
    (and
     (or (string-equal "+" des)
	 (and empire-route-past-bridges
	      (string-equal des "=")))
     (>= (or (recall-macro (car pt) (cdr pt) 'eff) 0)
	 empire-highway-min-eff))
    ))

(defun overfead-highwayp (pt)
  (and
   ;; feed from road only
   ;; (string-equal "+" (recall (car pt) (cdr pt) des)) ; redundent
   ;; that has enough food to still have the leave amount afterwards
   (< (+ empire-highway-leave-food delta) (or (recall (car pt) (cdr pt) food) 0))
   ;; only feed if not enemy or we have full mil control
   (or (not (string-equal "*" (recall (car pt) (cdr pt) *)))
       (> (recall (car pt) (cdr pt) mil) (recall (car pt) (cdr pt) civ)))
   ;; finally, it must have enough mobility to move the desired food level
   ;; to us
   (> (- (recall  (car pt) (cdr pt) (position-of 'mob)) 8)
      (calc-required-mobility 'food delta (car pt) (cdr pt) (r-to-spath to-path)))
   ;; Not a dist pt
   (not (and (recall-macro x y (position-of dependents))
	     (> 4 (recall-macro x y (position-of dependents)))))
   ))

(defun find-fead-highway-internal (from)
  (let (sects
	next
	path)
    (if (> (setq depth (1+ depth)) empire-max-path-depth)
	nil
      (setq to-path (cons from to-path))
      ;; first, see if this one can take it
      (if (overfead-highwayp from)
	  (progn
	    (setq to-path (cdr to-path))
	    (setq depth (1- depth))
	    (list from))
	;; Else place ourselves on the examined list
	(setq examined (cons from examined))
	;; Try each unexamined fead-highway neighbor
	(setq sects (owned-roads-near from))
	(while (and sects (not path))
	  (setq next (car sects))
	  (if (and (not (member next examined))
		   (> (recall (car next) (cdr next) eff) empire-highway-min-eff))
	      ;; This might lead to the way
	      (setq path (find-fead-highway-internal next))
	    )
	  (setq sects (cdr sects))
	  )
	(setq depth (1- depth))
	(setq to-path (cdr to-path))
	(if path (cons from path))
	)
      )))

(defun redistribute-food-internal (x y)
  ;; Are we a hungry highway?
  (if (and (string-equal (recall x y des) "+")
	   (recall x y food)
	   (< (recall x y food) empire-highway-min-food)
	   (< 95 (recall x y eff))
	   ;;(and (setq hdx (recall x y dist_x)) (setq hdy (recall x y dist_y)) (string-equal (recall hdx hdy des) "w"))
	   )
      (progn
	;; look around for food to put in us
	(setq delta (- empire-highway-leave-food (recall x y food)))
	(setq examined (list (cons x y)))
	(setq to-path nil)
	(setq depth 0)
	(if (setq path (find-fead-highway-internal (cons x y)))
	    (progn
	      (setq to-path (reverse path))
	      (setq dest (car to-path))
	      (setq mob-used 
		    (calc-required-mobility 'food delta
					    (car dest) (cdr dest)
					    (r-to-spath to-path)))
	      (sit-for 0)
	      (show-pt (cons x y) nil 'describe-sect-ideals)
	      (princ (format "\tFeeding road %s,%s (%sf) from %s,%s (%sf %smob) using %s mob\n"
			     x y
			     (recall x y food)
			     (car dest) (cdr dest)
			     (recall (car dest) (cdr dest) food)
			     (recall (car dest) (cdr dest) (position-of 'mob))
			     mob-used
			     ))
	      (make-automated-command (format "move f %s,%s %s %s"
					      (car dest) (cdr dest)
					      delta
					      (r-to-path (reverse path))))
	      (incriment-sector x y food delta)
	      (decriment-sector (car dest) (cdr dest) food delta)
	      (decriment-sector (car dest) (cdr dest) mob mob-used)
	      ))))
  )

(defun redistribute-food ()
  "Makes all highways have > empire-highway-min-food food. Only takes from
highways with > empire-highway-leave-food food that have enough mobility to
get the food here. Does not attempt to walk over highways of less than
empire-highway-min-eff" 
  (interactive)
  (if (not (> empire-highway-leave-food empire-highway-min-food))
      (error "empire-highway-leave-food must be > than empire-highway-min-food"))
  (with-output-to-temp-buffer "*Food*"
    (print-banner (format "Food Needs (%s/%s) as of %s"
		   empire-highway-leave-food
		   empire-highway-min-food
		   (current-time-string)))
    (terpri)
    (empire-data-mode-buffer t t "*Food*")
    (let ((food (position-of 'food))
	  (eff (position-of 'eff))
	  (mob (position-of 'mob))
	  (num (position-of 'des))
	  (des (position-of 'des))
	  (mil (position-of 'mil))
	  (civ (position-of 'civ))
	  (dist_x (position-of 'dist_x))
	  (dist_y (position-of 'dist_y))
	  (* (position-of '*))
	  depth
	  hdx hdy
	  delta
	  dest
	  path
	  examined
	  mob-used
	  to-path)
      (mapsects-active 'redistribute-food-internal "Examining for hungry highways")
      )))
(put 'redistribute-food 'empire t)


(defun find-mil-feed-internal (from &optional current-path)
  (let (sects
	next
	roads
	newly-examined
	path
	avail-mil)
    (if (> (setq depth (1+ depth)) empire-max-path-depth)
	nil
      (if (not current-path) 
	  (setq current-path (list from)))
      ;; first, see if this one can supply it
      (if (> (- (recall (car from) (cdr from) mil)
		(funcall empire-ideal-mil-fcn
			 (car from) (cdr from))) delta)
	  (progn
	    (setq depth (1- depth))
	    (list from))
	;; Else place ourselves on the examined list
	(setq examined (cons from examined))
	;; get each unexamined mil-feed neighbor
	(setq sects (owned-sects-near from))
	;; try each one
	(while (and sects (not path))
	  (setq next (car sects))
	  (if (not (member next examined))
	      (progn
		(if (and
		     (or (string-equal (recall (car next) (cdr next) des) "+")
			 (and empire-route-past-bridges
			      (string-equal (recall (car next) (cdr next) des) "=")))
		     (>= (recall-macro (car next) (cdr next) 'eff)
			 empire-highway-min-eff))
		     
		    (setq roads (cons next roads))
		  (setq newly-examined (cons next newly-examined)))
		(if (and
		     ;; it is not undergoing an attitude adjustment
		     (= 100 (recall (car next) (cdr next) work))
		     ;; it has the troops to move
		     (>= (setq avail-mil (- (recall (car next) (cdr next) mil)
					    (funcall empire-ideal-mil-fcn
						     (car next) (cdr next))))
			 empire-min-mil-move)
		     ;; it has enough mobility to move some troops
		     (or dest-is-roadp
			 (>= (min (or (calc-movable-amount 'mil
							   (car next) (cdr next)
							   (r-to-spath
							    (cons next
								  current-path))
							   8)
				      0)
				  avail-mil)
			     empire-min-mil-move))
		     )
		    ;; make this be the source
		    (setq path (list next))
		  )))
	  (setq sects (cdr sects))
	  )
	;; None of these were sources. Make these be "examined"
	(setq examined (append newly-examined examined))
	;; walk the roads
	(setq sects roads)
	(while (and sects (not path))
	  (setq next (car sects))
	  ;; This might lead to the way
	  (setq path (find-mil-feed-internal next (cons next current-path)))
	  (setq sects (cdr sects))
	  )
	(setq depth (1- depth))
	(if path (cons from path))
	))))

(defun redistribute-mil-internal (x y)
  (if (oursp x y)
      (progn
	(setq delta (- (funcall empire-ideal-mil-fcn x y) (recall x y mil)))
	(if (and (> delta empire-min-mil-move)
		 ;; do not fill an enlistment ctr close to ok
		 (not (and (string-equal "e" (setq dest-des (recall x y des)))
			   (< delta empire-min-mil-move-into-enlistment))))
	    (progn			; we should find mil for this guy
	      (setq dest-is-roadp (and (or (string-equal "+" dest-des)
					   (and empire-route-past-bridges
						(string-equal "=" dest-des)))
				       (>= (recall-macro x y 'eff)
					   empire-highway-min-eff)))
	      (let ((examined  (list (cons x y)))
		    directions
		    bodies-to-move
		    source xs ys
		    (depth 0)
		    )
		(if (and (setq path (find-mil-feed-internal (cons x y)))
			 (setq directions (r-to-spath (reverse path)))
			 (setq source (car (last path)))
			 (setq xs (car source))
			 (setq ys (cdr source))
			 ;;
			 ;; Here, we move the minimum of:
			 ;; 1. The needed amount.
			 ;; 2. The maximum amount that is excess.
			 ;; 3. The maximum amount that can be moved.
			 ;;
			 (> (setq bodies-to-move
				  (min delta
				       (- (recall xs ys mil)
					  (funcall empire-ideal-mil-fcn xs ys))
				       (or (calc-movable-amount 'mil
								xs ys
								directions
								8)
					   0))) 0)

			 )
		    (progn
		      (setq mob-needed
			    (calc-required-mobility 'mil bodies-to-move
						    xs ys directions))
		      (show-pt source nil 'describe-sect-ideals)
		      (sit-for 0)
		      (princ (format "\t\tMilitarizing %s,%s (%s %s%% %s%s/%s) with %s from %s,%s (%s/%s)\n"
				     x y
				     dest-des
				     (recall x y (position-of 'work))
				     (if (string-equal (recall x y
							       (position-of '*))
						       "*") "* " "")
				     (recall x y mil)
				     (funcall empire-ideal-mil-fcn x y)
				     bodies-to-move
				     (car source) (cdr source)
				     (recall (car source) (cdr source) mil)
				     (funcall empire-ideal-mil-fcn
					      (car source) (cdr source))
				     ))
		      (make-automated-command (format "move m %s,%s %s %s"
						      (car source) (cdr source)
						      bodies-to-move
						      (r-to-path (reverse path))))
		      ;; decriment source mil and mobility
		      (decriment-sector (car source) (cdr source) mil
					bodies-to-move)
		      (decriment-sector (car source) (cdr source) mob mob-needed)
		      ;; incriment dest mil
		      (incriment-sector x y mil bodies-to-move)
		      )
		  (show-pt (cons x y) nil 'describe-sect-ideals)
		  (princ (format "  %s\t%s(%s%% %swith %s/%s/%s) cannot be militarized. Needs %s more\n"
				 (format-xy x y) dest-des
				 (recall x y (position-of 'work))
				 (if (string-equal (recall x y (position-of '*))
						   "*") "* " "")
				 (recall x y (position-of 'civ))
				 (recall x y (position-of 'mil))
				 (recall x y (position-of 'uw))
				 delta))
		  (sit-for 0)
		  ))))))
  )

(defun redistribute-mil ()
  "Makes all sectors have correct mil
   Uses the function named by empire-ideal-mil-fcn to decide what is correct"
  (interactive)
  (require 'emp-sector)
  (with-output-to-temp-buffer "*Mil*"
    (princ (format "\t\t***Military Needs (%s/%s) as of %s ***\n\n"
		   empire-min-mil-move
		   empire-min-mil-move-into-enlistment
		   (current-time-string)))
    (empire-data-mode-buffer t t "*Mil*")
    (sit-for 0)
    (let ((des (position-of 'des)
	       )
	  (mil (position-of 'mil))
	  (mob (position-of 'mob))
	  (work (position-of 'work))
	  delta
	  source
	  dest-des
	  dest-is-roadp
	  mob-needed
	  ideal
	  path)
      (mapsects-active 'redistribute-mil-internal "Examining for mil needed")
      )))
(put 'redistribute-mil 'empire t)

(defun find-civ-feed-internal (dest)
  ;; returns path along roads to dest from a sector that has some bodies
  ;; and some spare mobility
  (let (sects
	next
	roads
	newly-examined
	path
	available)
    (if (> (setq depth (1+ depth)) empire-max-path-depth)
	nil
      ;; Place ourselves on the examined list
      (setq examined (cons dest examined))

      ;; Try each unexamined neighbor
      (setq sects (owned-sects-near dest))
      (while (and sects (not path))
	(setq next (car sects))
	(setq sects (cdr sects))
	(if (not (member next examined))
	    (progn
	      ;; is this sector a source of bodies?
	      (if (and
		   ;; it is not undergoing an attitude adjustment
		   (= 100 (recall (car next) (cdr next) work))
		   ;; it has civs to move
		   (> (setq available
			    (- (recall-macro (car next) (cdr next) 'civ)
			       (funcall empire-ideal-civ-fcn
					(car next) (cdr next))))
		      empire-min-civ-move)
		   ;; it has some surplus mobility
		   (>= (recall (car next) (cdr next) mob) 16)
		   )
		  ;; make this be the source
		  ;; Note: pleural sources should be allowed
		  (setq path (list next))
		)
	      ;; if this is a road, lets walk some more along this path
	      (if (and (or (string-equal "+" (recall (car next) (cdr next) des))
			   (and empire-route-past-bridges
				(string-equal "=" (recall (car next) (cdr next) des))))
		       (>= (recall-macro (car next) (cdr next) 'eff)
			   empire-highway-min-eff))
		  (setq roads (cons next roads))
		(setq newly-examined (cons next newly-examined)))
	      ))
	)
      ;; None of these were sources. Make these be "examined"
      (setq examined (append newly-examined examined))
      ;; walk the roads
      (setq sects roads)
      (while (and sects (not path))
	(setq next (car sects))
	;; This might lead to the way
	(setq path (find-civ-feed-internal next))
	(setq sects (cdr sects))
	)
      (setq depth (1- depth))
      (if path (cons dest path))
      )))

(defun redistribute-civ-internal (x y)
  ;; tries to populate x y with folks connected by efficient roads
  (let ((ideal-civ (funcall empire-ideal-civ-fcn x y)))
    (if ideal-civ
	(progn
	  ;; Decide on bodies desirable to move.  Limit this to
	  ;; `empire-max-civ-redistribute'.
	  (setq delta (min empire-max-civ-redistribute
			   (- ideal-civ (recall-macro x y 'civ))))
	  (setq dest-des (recall x y des))
	  ;; See if we have enough to attempt to move
	  (if (> delta empire-min-civ-move)
	      (progn
		;; We have need of bodies from somewhere
		(setq dest-is-roadp
		      (and (or (string-equal "+" dest-des)
			       (and empire-route-past-bridges
				    (string-equal "=" dest-des)))
			   (>= (recall-macro x y 'eff)
			       empire-highway-min-eff)))
		(let ((examined  (list (cons x y)))
		      flag
		      stop-loop
		      directions
		      depth
		      )
		  (while (and (setq depth 0)
			      (setq path (find-civ-feed-internal (cons x y)))
			      (> delta empire-min-civ-move)
			      (not stop-loop)
			      )
		    (setq flag t)
		    ;; the path may be from a place with not the full
		    ;; desired number of bodies or else without the
		    ;; mobility to move the desired number. However, any
		    ;; amount is acceptable.
		    (setq source (car (last path)))
		    (setq xs (car source))
		    (setq ys (cdr source))
		    (setq directions (r-to-spath (reverse path)))
		    (show-pt source nil 'describe-sect-ideals)
		    (sit-for 0)
		    (setq bodies-to-move
			  (min delta
			       (- (recall-macro xs ys 'civ)
				  (funcall empire-ideal-civ-fcn xs ys))
			       (or (calc-movable-amount 'civ
							xs ys
							directions
							8)
				   0)))
		    (setq mob-needed
			  (calc-required-mobility 'civ bodies-to-move
						  xs ys
						  directions
						  ))
		    (if (> bodies-to-move 0)
			(progn
			  (princ (format "\t\tTo %s(%s %s%s%%(%s) %s%s->%s %s%s) with %s from\n\t\t   %s(%s %s%s%%(%s) %s->%s %s%s) using %s mob\n"
					 (format-xy x y)
					 dest-des
					 (if (> 100
						(recall x y
							(position-of 'work)))
					     (format "<%s>"
						     (recall x y
							     (position-of
							      'work)))
					   "")
					 (recall x y (position-of 'eff))
					 (recall x y (position-of 'mob))
					 (if (string-equal
					      (recall x y (position-of '*))
					      "*")
					     "* " "")
					 (recall x y (position-of 'civ))
					 ideal-civ
					 (recall x y (position-of 'food))
					 ;; was it food limited?
					 (if (= (+ (recall x y
							   (position-of 'civ))
						   bodies-to-move)
						(allowable-mouths
						 (recall x y
							 (position-of 'food))))
					     "F" "f")
					 bodies-to-move
					 (format-xy xs ys)
					 (recall xs ys
						 (position-of 'des))
					 (if (> 100
						(recall xs ys
							(position-of 'work)))
					     (format "<%s>"
						     (recall xs ys
							     (position-of
							      'work)))
					   "")
					 (recall xs ys (position-of 'eff))
					 (recall xs ys (position-of 'mob))
					 (recall xs ys (position-of 'civ))
					 (funcall empire-ideal-civ-fcn xs ys)
					 (recall xs ys (position-of 'food))
					 (if (= (funcall empire-ideal-civ-fcn xs ys)
						(allowable-mouths
						 (recall xs ys
							 (position-of 'food))))
					     "F" "f")
					 mob-needed
					 ))
			  (make-automated-command (format "move c %s,%s %s %sh"
							  xs ys
							  bodies-to-move
							  directions))
			  ;; decriment source civ and mobility
			  (decriment-sector xs ys (position-of 'civ)
					    bodies-to-move)
			  (decriment-sector xs ys mob mob-needed)
			  (setq delta (- delta bodies-to-move))
			  ;; incriment dest civ
			  (incriment-sector x y (position-of 'civ)
					    bodies-to-move)
			  ;; raise the food level in the dest
			  (if (and (not (or (string-equal dest-des "w")
					    (string-equal dest-des "^")
					    (string-equal dest-des "a")))
				   (not (and
					 (= x (or (recall x y (position-of
							       'dist_x)) x))
					 (= y (or y (recall x y (position-of 'dist_y))))))
				   (< (recall x y (position-of 'food))
				      (/ (+ (recall x y (position-of 'civ))
					    (recall x y (position-of 'mil))
					    (recall x y (position-of 'uw)))
					 5)))
			      (make-automated-command
			       (format "thr f %s,%s %s"
				       x y
				       ;; leave growth factor
				       (/ (+ (recall x y (position-of 'civ))
					     (recall x y (position-of 'mil))
					     (recall x y (position-of 'uw))) 3)
				       )))
			  )
		      ;; calc-movable-amount returned 0
		      (setq flag nil
			    stop-loop t))
		    )
		  (if (not flag)
		      (setq uncivilizable (cons (list x y) uncivilizable)))
		  )));; have enough to move
	  )))
  )

(defun redistribute-civ ()
  "Makes all sectors have correct civ by moving on efficient roads.
     Roads are efficient if their efficiency is more than
     empire-highway-min-eff. 
     Moves are not done unless at least empire-min-civ-move can be moved.
   Uses the function named by empire-ideal-civ-fcn to decide what is correct

Format of output:
		   x,y       (D Eff Mob Bodies_Now->Bodies_ideal  Food)
		To 3,-5      (b 33%(100) 92->113 19f) with 21 from
		   4,-4      (+ 55%(100) 159->136 23F)
mo c 4,-4 21 yh

The following sectors could not yet be civilized
1,-7      	+(100% with 6/0/0 1f) needs 44 more

Notes: If the 'f' for food is capitalized, then either more bodies were not
moved in because the food level could not support them, or bodies are being
moved out because there is not enough food for them.

Acceptable food is defined to be enough for 1 day with a safety factor,
not counting on any growth in the sector itself.
"
  (interactive)
  (require 'emp-sector)
  (with-output-to-temp-buffer "*Civ*"
    (princ (format "\t***Civilan movement Needs as of %s ***\n"
		   (current-time-string)))
    (princ (format "At least %s civs will be moved\n" empire-min-civ-move))
    (princ "See the function doc for redistribute-civ for details of the format.\n\n")

    (princ "\t\t   x,y       (D Eff Mob Bodies_Now->Bodies_ideal  Food)\n")
    (princ "\t\t   ---       -----------------------------------------\n")
    (empire-data-mode-buffer t t "*Civ*")
    (empire-switch-to-buffer-not-map "*Civ*")
    (let ((des (position-of 'des)
	       )
	  (civ (position-of 'civ))
	  (mob (position-of 'mob))
	  (work (position-of 'work))
	  delta
	  source
	  dest-des
	  dest-is-roadp
	  mob-needed
	  ideal
	  path
	  bodies-to-move
	  xs ys
	  (uncivilizable nil))
      (mapsects-active 'redistribute-civ-internal "Examining for civ needed in underdeveloped sectors")
      (terpri)
      (if (not empire-batch-play)
	  (progn
	    (print-banner "Spare civs and mobility unused")
	    (check-spare)
	    (terpri)
	    (print-banner "The following sectors could not yet be civilized")
	    (mapcar '(lambda (arg)
		       (princ (concat "  "
				      (describe-sect-ideals (nth 0 arg) (nth 1 arg))
				      "\n")))
		    (reverse uncivilizable))))
      )))
(put 'redistribute-civ 'empire t)


(defun find-dist-pts (from length)
  ;; finds all paths to any dist point such that the path is less than
  ;; length long
  (if (> length 0)
  (let (sects
	next
	roads
	paths
	new-paths
	)
    ;; Place ourselves on the examined list
    ;;(setq examined (cons from examined))
    (setq to-path (cons from to-path))
    ;; Try each unexamined neighbor
    (setq sects (owned-sects-near from))
    ;;(show-pt from t)
    ;;(sit-for 0)
    (while sects
      (setq next (car sects))
      (setq sects (cdr sects))
      (if (not (member next to-path))
	  (progn
	    ;; is this sector a dist pt?
	    (if (recall (car next) (cdr next) (position-of dependents))
		(progn
		  ;; make this be a source
		  (setq paths (cons (list from next) paths))
		  ))
	      ;; else if this is a road, lets walk some more along this path
	      (if (and (string-equal "+" (recall (car next) (cdr next)
						 (position-of des)))
		       (or empire-use-inefficient-highways
			   (> (recall (car next) (cdr next) (position-of eff))
			      empire-highway-min-eff))
		  )
		  (setq roads (cons next roads)))
	      ))
      )
    ;; walk the roads, but only if there are not too many paths already found
    (if (< (length paths) 20)
	(progn
	  (setq sects roads)
	  (while (and sects)
	    (setq next (car sects))
	    (setq sects (cdr sects))
	    ;; This might lead to another road
	    (setq new-paths (find-dist-pts next (1- length)))
	    (while new-paths
	      (setq paths (cons (cons from (car new-paths)) paths))
	      (setq new-paths (cdr new-paths)))
	    )))
    (setq to-path (cdr to-path))
    paths
    )))

(defun dist-betterp (path1 path2)
  (let (dest1 dest2 des1 des2)
    (setq dest1 (car (last path1)))
    (setq dest2 (car (last path2)))
    (setq des1  (recall (car dest1) (cdr dest1)
			(position-of des)))
    (setq des2  (recall (car dest2) (cdr dest2)
			(position-of des)))

    (if (string-equal des1 des2)
	(< (length path1) (length path2))
      ;; they differ
      ;; take w, +, and then shortest path
      (cond
       ((string-equal des1 "w")
	    t)
       ((string-equal des2 "w")
	    nil)
       ((string-equal des1 "+")
	    t)
       ((string-equal des2 "+")
	    nil)
       ((string-equal des1 "=")
	    t)
       ((string-equal des2 "=")
	    nil)
       (t (< (length path1) (length path2)))))
    ))

(defun find-sorted-dist-pts (from length)
  (sort (find-dist-pts from length) 'dist-betterp))

(defun find-sorted-unique-dist-pts (from length)
  (let ((paths (find-sorted-dist-pts from length))
	new-paths
	p)
    (while paths
      (setq p (nreverse (car paths)))
      (if (not (assoc (car p) new-paths))
	  (setq new-paths (cons p new-paths)))
      (setq paths (cdr paths)))

    (while new-paths
      (setq paths (cons (nreverse (car new-paths)) paths))
      (setq new-paths (cdr new-paths)))
    paths))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plane protection analysis
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun increment-air-protection (x y att def)
  (if (recall x y (position-of 'des))
      (let ((pro (recall x y (position-of 'intercept))))
	(if pro
	    (record x y (position-of 'intercept)
		    (cons (+ att (car pro))
			  (+ def (cdr pro))))
	  (record x y (position-of 'intercept) (cons att def))))))

(defun record-air-protection (plane type x y ran att def)
  "add ATT and DEF to each sector within RAN of X Y"
  (let (yy xx stopx)
    (setq yy (- y ran))
    (while (<= yy (+ y ran))
      (setq xx (+ (- x (* ran 2)) (abs (- y yy))))
      (setq stop-x (- (+ x (* ran 2)) (abs (- y yy))))
      (while (<= xx stop-x)
	(message "For %s %s range %s from %s,%s, incrementing %s,%s by %s and %s"
		 type plane ran x y xx yy att def)
	(increment-air-protection (normalize-x xx)
				  (normalize-y yy)
				  att def)
	(setq xx (+ 2 xx))
	)
      (setq yy (1+ yy))
      )))



(defun calc-defensive-air-umbrella ()
  ;;(interactive)
  (walksects (if (recall x y (position-of 'intercept))
		 (record x y (position-of 'intercept) nil))
	     "Clearing umbrella")
  (let ((planes empire-planes)
	plane)

    (while planes
      (setq plane (car planes))
      (setq planes (cdr planes))
      (if (plane-cap (planedata-recall 'type plane) 'intercept)
	  (progn
	    (message "Plane %s" (planedata-recall '\# plane))
	    (record-air-protection (planedata-recall '\# plane)
				   (planedata-recall 'type plane)
				   (planedata-recall 'x plane)
				   (planedata-recall 'y plane)
				   (/ (planedata-recall 'ran plane) 2)
				   (planedata-recall 'att plane)
				   (planedata-recall 'def plane)))))
    ))

(defun calc-air-defenders (tx ty)
  "Calculates and returns the air strength defending a given sector.
map-air-umbrella is faster for the entire country, this is faster for a
single sector"
  (interactive "nX: \nnY: ")
  (let ((planes empire-planes)
	plane
	defenders)

    (while planes
      (setq plane (car planes))
      (setq planes (cdr planes))
      (if (plane-cap (planedata-recall 'type plane) 'intercept)
	  (progn
	    (message "Plane %s" (planedata-recall '\# plane))
	    (if (within-rangep (planedata-recall 'x plane)
			       (planedata-recall 'y plane) tx ty
			       (/ (planedata-recall 'ran plane) 2))
		(setq defenders (cons (planedata-recall '\# plane) defenders)))
	    )))
    (message "%s,%s is defended by %s" tx ty defenders)
    defenders
    ))
(put 'calc-air-defenders 'empire t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Flow Analysys
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst flow-pos '(
		   (f . 1)
		   (s . 2)
		   (p . 3)
		   (i . 4)
		   (d . 5)
		   (b . 6)
		   (o . 7)
		   (l . 8)
		   (h . 9)
		   (r . 10)
		   (= . 11)
		   (~ . 12)
		   (+ . 13)
		   (^ . 0)
		   (sects . 14)
		   (bodies . 15))
  "Positions in array used by empire-show-flows")

(defun flow-data-incr (farray elem inc printp)
  (let ((pos (cdr (assq elem flow-pos))))
    (if pos
	(progn
	  (aset farray pos (+ (aref farray pos) inc))
	  (if printp
	      (princ (format "%s %s\t%s %s %s \tRunning Total: %s\n"
			     (format-xy x y)
			     (recall-macro x y 'des)
			     (if (> inc 0) "Makes" "Uses ")
			     elem inc
			     (aref farray pos))))
	  ))))

(defun empire-sector-flow-incr (x y sarray warray printp)
  ;; internal to empire-show-flows
  (flow-data-incr sarray 'sects 1 nil)
  (flow-data-incr sarray 'bodies 
		  (+ (recall x y civ) (recall x y mil) (recall x y uw))
		  nil)
  (flow-data-incr warray 'sects 1 nil)
  (flow-data-incr warray 'bodies 
		  (+ (recall x y civ) (recall x y mil) (recall x y uw))
		  nil)

  ;; lookup the production
  (if (and (setq make-str (recall x y make))
	   ;;(not (zerop make-str))	; ?
	   )
      (progn
	(setq symbol (intern make-str))
	(flow-data-incr sarray symbol (recall x y will) printp)
	(flow-data-incr warray symbol (recall x y will) nil)

	(setq use (recall x y use1))
	(if (cdr use)
	    (progn
	      (flow-data-incr sarray (intern (cdr use)) (- 0 (car use)) printp)
	      (flow-data-incr warray (intern (cdr use)) (- 0 (car use)) nil)
	      ))
	    
	(setq use (recall x y use2))
	(if (cdr use)
	    (progn
	      (flow-data-incr sarray (intern (cdr use)) (- 0 (car use)) printp)
	      (flow-data-incr warray (intern (cdr use)) (- 0 (car use)) nil)
	      ))

	(setq use (recall x y use3))
	(if (cdr use)
	    (progn
	      (flow-data-incr sarray (intern (cdr use)) (- 0 (car use)) printp)
	      (flow-data-incr warray (intern (cdr use)) (- 0 (car use)) nil)
	      ))

	)))

(defun empire-print-flow-bodies (set)
  (let* ((fx (car (car set)))
	 (fy (cdr (car set)))
	 (doit (not (equal fx "-")))
	 (f (aref (cdr set) (cdr (assq 'f flow-pos))))
	 (sects (aref (cdr set) (cdr (assq 'sects flow-pos))))
	 (bodies (aref (cdr set) (cdr (assq 'bodies flow-pos))))
	 )
    ;; record count of sects flowing here for later auto dist calc
    (if doit
	(record fx fy (position-of dependents) sects))
    (princ (format "%s%s\t\t%s%s\t\t%s\t\t%s\t     %s\n"
		   (format-xy fx fy)
		   f

		   (if doit
		       (recall fx fy (position-of 'food))
		     "N/A")
		   (if (and doit
			    (> (/ (+ (* bodies etu-per-update
					upd-per-day 2)
				     bodies-fead-by-1-food-per-etu)
				  bodies-fead-by-1-food-per-etu)
			       (recall fx fy (position-of 'food))))
		       "*" "")

		   bodies

		   sects
		   (if doit
		       (/ (* (/ bodies-fead-by-1-food-per-etu
				(* etu-per-update upd-per-day))
			     (recall fx fy (position-of 'food)))
			  bodies)
		     "N/A")
		   ))
    ))

(defun empire-print-flow (set)
  (let* ((fx (car (car set)))
	(fy (cdr (car set)))
	(doit (not (equal fx "-")))
	(f (aref (cdr set) (cdr (assq 'f flow-pos))))
	(s (aref (cdr set) (cdr (assq 's flow-pos))))
	(p (aref (cdr set) (cdr (assq 'p flow-pos))))
	(i (aref (cdr set) (cdr (assq 'i flow-pos))))
	(d (aref (cdr set) (cdr (assq 'd flow-pos))))
	(b (aref (cdr set) (cdr (assq 'b flow-pos))))
	(o (aref (cdr set) (cdr (assq 'o flow-pos))))
	(l (aref (cdr set) (cdr (assq 'l flow-pos))))
	(h (aref (cdr set) (cdr (assq 'h flow-pos))))
	(r (aref (cdr set) (cdr (assq 'r flow-pos))))
	(= (aref (cdr set) (cdr (assq '= flow-pos))))
	(~ (aref (cdr set) (cdr (assq '~ flow-pos))))
	(+ (aref (cdr set) (cdr (assq '+ flow-pos))))
	(^ (aref (cdr set) (cdr (assq '^ flow-pos))))
	(sects (aref (cdr set) (cdr (assq 'sects flow-pos))))
	(bodies (aref (cdr set) (cdr (assq 'bodies flow-pos))))
	)
    (princ (format "%s%s%s\t%s%s\t%s%s\t%s%s\t%s\t%s\t%s%s\t%s\t%s\n"
		   (format-xy fx fy)
		   i (if (and doit
			      (> (* -1 i upd-per-day)
				 (recall fx fy (position-of 'iron))))
			 "*" "")
		   l (if (and doit
			      (> (* -1 l upd-per-day)
				 (recall fx fy (position-of 'lcm))))
			 "*" "")
		   h (if (and doit
			      (> (* -1 h upd-per-day)
				 (recall fx fy (position-of 'hcm))))
			 "*" "")
		   o (if (and doit
			      (> (* -1 o upd-per-day)
				 (recall fx fy (position-of 'oil))))
			 "*" "")
		   s
		   p
		   d (if (and doit
			      (> (* -1 d upd-per-day)
				 (recall fx fy (position-of 'dust))))
			 "*" "")
		   b
		   r
		   ))
    ))

(defun empire-flows ()
  "Shows the flow into and out of each sector that is the target of a dist.
   Also summarizes national flow and undistributed flow.

  Items marked with * have less than a 1 day supply of that commodity or a
  2 day supply for food.

  \\[unload-dist-node] will try to make a node have no dist to it
"
  (interactive)
  (let ((flows (list (cons (cons "-" "-") (make-vector (length flow-pos) 0))))
	(warray (make-vector (length flow-pos) 0))
	;; temp vars for called stuff
	symbol
	use
	fx fy
	flow
	;; empire-pos variables
	(mil (position-of 'mil))
	(uw (position-of 'uw))
	(civ (position-of 'civ))
	(dist-x (position-of 'dist_x))
	(dist-y (position-of 'dist_y))
	(use1 (position-of 'use1))
	(use2 (position-of 'use2))
	(use3 (position-of 'use3))
	(make (position-of 'make))
	(will (position-of 'will))
	;;
	(header-str
	 (format "x,y       %s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n"
		     "iron" "light" "hvy" "oil"
		     "shell" "petrol" "dust" "bar" "rad"))
	(body-header-str
	 (format "x,y       Food Growing\tFood Stored\tBodies\tSectors to feed\tDays on Hand\n"))
	)
    (mapsects-active ;; walk cannot compile -- too deep
     (function (lambda (x y)
		 (setq fx (recall x y dist-x))
		 (setq fy (recall x y dist-y))
		 ;; forget past dependents
		 (record x y (position-of dependents) nil)
		 ;; take care of the null dist
		 (if (or (not fx)
			 (not fy)
			 (not (oursp fx fy))
			 (and (= fx x) (= fy y)))
		     (setq fx "-" fy "-"))
		 ;; get the flow vector
		 (if (not (setq flow (assoc (cons fx fy) flows)))
		     (progn
		       (setq flows
			     (cons (cons (cons fx fy)
					 (make-vector (length flow-pos) 0))
				   flows))
		       (setq flow (car flows))))
		 (empire-sector-flow-incr x y (cdr flow) warray nil)))
     "Flowing")

    ;; now, print the flows
    (with-output-to-temp-buffer "*Flows*"
      (empire-data-mode-buffer t t "*Flows*")
      (print-banner (format "World Flows as of %s" (current-time-string)))
      (terpri)
      (print-banner "Individual Sector Flows")
      (princ header-str)
      (mapcar 'empire-print-flow (cdr (reverse flows)))

      (print-banner "Undistributed Flow")
      (princ header-str)
      (empire-print-flow (car (reverse flows)))
      (terpri)

      (print-banner "World Flow")
      (princ header-str)
      (empire-print-flow (cons (cons "-" "-") warray))

      (let ((= (aref warray (cdr (assq '= flow-pos))))
	    (~ (aref warray (cdr (assq '~ flow-pos))))
	    (+ (aref warray (cdr (assq '+ flow-pos))))
	    (^ (aref warray (cdr (assq '^ flow-pos))))
	    (bodies (aref warray (cdr (assq 'bodies flow-pos))))
	    )
	(princ (format "\nEdu\t%s\tunits yielding level %s\nHappy\t%s\tunits yielding level %s\nTech\t%s\nMed\t%s\n"
		       = (/ = (max 1 (/ bodies edu-divisor)))
		       ~ (/ ~ (max 1 (/ bodies happy-divisor)))
		       ^
		       +
		       )))

      (print-banner "Individual Sector Food")
      (princ body-header-str)
      (mapcar 'empire-print-flow-bodies (cdr (reverse flows)))

      (print-banner "Undistributed Food")
      (princ body-header-str)
      (empire-print-flow-bodies (car (reverse flows)))

      (print-banner "World Food")
      (princ body-header-str)
      (empire-print-flow-bodies (cons (cons "-" "-") warray))
      (let ((f (aref warray (cdr (assq 'f flow-pos)))))
	(princ (format "\nFood growth will support a population of %s\n"
		       (* (/ f etu-per-update) bodies-fead-by-1-food-per-etu))))
      )
    ))

(defun empire-flow (tx ty)
  "Shows the flow into and out of each sector that dists to a given sector"
  (interactive "nX: \nnY: ")
  (valid-sector tx ty)

  (let ((flows (list (cons (cons "-" "-") (make-vector (length flow-pos) 0))))
	(warray (make-vector (length flow-pos) 0))
	;; temp vars for called stuff
	symbol
	use
	fx fy
	flow
	;; empire-pos variables
	(mil (position-of 'mil))
	(uw (position-of 'uw))
	(civ (position-of 'civ))
	(dist-x (position-of 'dist_x))
	(dist-y (position-of 'dist_y))
	(use1 (position-of 'use1))
	(use2 (position-of 'use2))
	(use3 (position-of 'use3))
	(make (position-of 'make))
	(will (position-of 'will))
	;;
	(header-str
	 (format "x,y       %s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n"
		 "iron" "light" "hvy" "oil"
		 "shell" "petrol" "dust" "bar" "rad"))
	(body-header-str
	 (format "x,y       Food Growing\tFood Stored\tBodies\tSectors to feed\tDays on Hand\n"))
	)
    (with-output-to-temp-buffer (format "*Flows_%s,%s*" tx ty)
      (empire-data-mode-buffer t t (format "*Flows_%s,%s*" tx ty))
      (print-banner (format "Sector Flow as of %s" (current-time-string)))

      (mapsects-active			; walk cannot compile -- too deep
       (function (lambda (x y)
		   (setq fx (recall x y dist-x))
		   (setq fy (recall x y dist-y))
		   (if (and (= tx fx) (= ty fy))
		       (progn
       
			 ;; take care of the null dist
			 (if (or (not fx)
				 (not fy)
				 (and (= fx x) (= fy y)))
			     (setq fx "-" fy "-"))
			 ;; get the flow vector
			 (if (not (setq flow (assoc (cons fx fy) flows)))
			     (progn
			       (setq flows
				     (cons (cons (cons fx fy)
						 (make-vector (length flow-pos) 0))
					   flows))
			       (setq flow (car flows))))
			 (empire-sector-flow-incr x y (cdr flow) warray t)))))
       "Flowing")
      (terpri)

      ;; now, print the flows
      (print-banner "Individual Sector Flow")
      (princ header-str)
      (mapcar 'empire-print-flow (cdr (reverse flows)))

      (let ((= (aref warray (cdr (assq '= flow-pos))))
	    (~ (aref warray (cdr (assq '~ flow-pos))))
	    (+ (aref warray (cdr (assq '+ flow-pos))))
	    (^ (aref warray (cdr (assq '^ flow-pos))))
	    (bodies (aref warray (cdr (assq 'bodies flow-pos))))
	    )
	(princ (format "\nEdu\t%s\tunits yielding level %s\nHappy\t%s\tunits yielding level %s\nTech\t%s\nMed\t%s\n"
		       = (/ = (max 1 (/ bodies edu-divisor)))
		       ~ (/ ~ (max 1 (/ bodies happy-divisor)))
		       ^
		       +
		       )))

      (print-banner "Individual Sector Food")
      (princ body-header-str)
      (mapcar 'empire-print-flow-bodies (cdr (reverse flows)))

      )
    ))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun check-demobobilize ()
  "Demobilizes all mil that are more than 10 over the ideal level as
calculated by ideal-mil." 
  (interactive)
  (let (desired delta)
    (walksects-active
     (if (not (string-equal "*" (recall x y (position-of '*))))
	 (progn
	   (setq desired (ideal-mil x y))
	   (if (< 10 (setq delta (- (recall x y (position-of 'mil))) desired)))
	   (make-automated-command (format "demob %s,%s %s y"
					   x y delta))
	   ))
     "Demobilizing")
    ))
(register-check-hook 'check-demobobilize "Demobilizable" nil)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ideal-mil
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-ideal-mil-fcn 'ideal-mil-young
  "*The empire-ideal-mil-fcn variable names the function that will
calculate the ideal number of mil that the tool will attempt to place
everywhere.  Redistribute-mil, empire-demob, and the sector describe
functions use this calc.

The user customization menu will help you set this value
")

(defun empire-ideal-mil-functions ()
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (and (not (commandp sym))
				   (get sym 'empire)
				   (string-match "ideal-mil" (symbol-name sym))
				   )
			      (setq vars (cons sym vars))))))
    vars
    (setq vars (sort vars 'string-lessp))
    ))

(defun describe-empire-ideal-mil-functions ()
  "Offers function documentation on all interactive functions empire offers
without regard to position in any keymap."
  (interactive)
  (with-output-to-temp-buffer "*Ideal Mil Functions*"
    (print-banner "Ideal Mil Functions")
    (terpri)
    (princ 
     (documentation-property 'empire-ideal-mil-fcn 'variable-documentation))
    (terpri)
    (terpri)
    (print-banner "Choices")
    (terpri)
    (mapcar '(lambda (function)
	       (prin1 function)
	       (princ ":")
	       (if (eq function empire-ideal-mil-fcn)
		   (princ " *Active Function*"))
	       (terpri)
	       (if (documentation function)
		   (princ (documentation function))
		 (princ "not documented"))
	       (terpri)
	       (terpri)
	       )
	    (empire-ideal-mil-functions))))

(defun empire-set-ideal-mil-function ()
  "Displays, prompts for, and sets the ideal-mil function"
  (interactive)
  (describe-empire-ideal-mil-functions)
  (setq empire-ideal-mil-fcn
	(intern (completing-read "Ideal Mil Function: "
			 (mapcar '(lambda (n) (list (symbol-name n)))
				 (empire-ideal-mil-functions))
			 nil t
			 (symbol-name empire-ideal-mil-fcn)
			 ))))



(defun ideal-mil-old (x y)
  "Calculates ideal mil amount for the sector.
   This version places a generious amount of mil in each sector."
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	civs
	bodies
	ideal
	(near-enemyp nil))
    (if ours
	(progn
	  (setq happy (= 100 ours))
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (/ (recall x y (position-of 'uw)) 3)))
	  (setq enemy (string-equal "*" (recall x y (position-of '*))))
	  (setq des (recall x y (position-of 'des)))
	  (setq mil (recall x y (position-of 'mil)))

	  (setq ideal
		(cond
		 ((or (string-equal des "w")
		      (string-equal des "b")
		      (string-equal des "n")
		      )
		  (max 75 (1+ (/ bodies (if enemy 7 (if happy 15 10))))))

		 ((or (string-equal des "*")
		      )
		  (max 25 (1+ (/ bodies (if (or (not happy) enemy) 7 25)))))

		 ((string-equal des "c")
		  (max 50
		       (1+ (/ bodies (if (or (not happy) enemy) 5 10)))))

		 ((string-equal des "e")
		  (max 150 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  ;;(min 100 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  )

		 ((string-equal des "f")
		  (max 50 (1+ (/ bodies (if (or (not happy) enemy) 7 20)))))

		 ((string-equal des "+")
		  (max 10 (1+ (/ civs (if enemy
					  10
					(if happy 40 10)))))
		  )

		 ((string-equal des "^")
		  0
		  )

		 (t
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
		  (1+ (/ civs (if enemy (if happy 20 10) (if happy 100 7))))
		  )
		 ))
	  ;;(error "stop")
	  
	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
	  ;;(mapcar '(lambda (pt)
	  ;;	       (if (not (recall x y 'mob))
	  ;;		 (setq near-enemyp t)))
	  ;;	    (sects-near (cons x y)))
	  ;;(if near-enemyp
	  ;;	(/ (* 3 ideal) 2)
	  ;;  ideal)
	  )
      ;; not ours
      (or (recall x y (position-of 'mil)) 0)
      )))

(defun ideal-mil-contested (x y)
  "Calculates ideal mil amount for the sector.
   This version places a generious amount of mil in each sector with extra
mil in sectors near ? or other unknown stuff."
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	civs
	bodies
	ideal
	(near-enemyp nil))
    (if ours
	(progn
	  (setq happy (= 100 ours))
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (/ (recall x y (position-of 'uw)) 3)))
	  (setq enemy (string-equal "*" (recall x y (position-of '*))))
	  (setq des (recall x y (position-of 'des)))
	  (setq mil (recall x y (position-of 'mil)))

	  (setq ideal
		(cond
		 ((or (string-equal des "w")
		      (string-equal des "b")
		      (string-equal des "n")
		      )
		  (max 75 (1+ (/ bodies (if enemy 7 (if happy 15 10))))))

		 ((or (string-equal des "*")
		      )
		  (max 25 (1+ (/ bodies (if (or (not happy) enemy) 7 25)))))

		 ((string-equal des "c")
		  (max 50
		       (1+ (/ bodies (if (or (not happy) enemy) 5 10)))))

		 ((string-equal des "e")
		  (max 150 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  ;;(min 100 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  )

		 ((string-equal des "f")
		  (max 50 (1+ (/ bodies (if (or (not happy) enemy) 7 20)))))

		 ((string-equal des "+")
		  (max 10 (1+ (/ civs (if enemy
					  (if happy 20 10)
					(if happy 40 20)))))
		  )

		 ((string-equal des "^")
		  0
		  )

		 (t
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
		  (1+ (/ civs (if enemy (if happy 20 10) (if happy 100 50))))
		  )
		 ))
	  ;;(error "stop")
	  
	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
	  (mapcar (function (lambda (pt)
			      (if (and (not (recall x y 'mob))
				       (recall x y 'own))
				  (setq near-enemyp t))))
	  	    (sects-near (cons x y)))
	  (if near-enemyp
	  	(/ (* 3 ideal) 2)
	    ideal)
	  )
      ;; not ours
      (or (recall x y (position-of 'mil)) 0)
      )))
(put 'ideal-mil-contested 'empire t)

(defun ideal-mil-young (x y)
  "Calculates ideal mil amount for the sector.
   This version places a small amount of mil in each sector.
   empire-min-mil-move should be small (~3) for this function or else
   nothing will be moved. "
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	civs
	bodies
	ideal
	(near-enemyp nil))
    (if ours
	(if (> 30 (recall x y (position-of 'eff)))
	    0 ; no mil yet
	(progn
	  (setq happy (= 100 ours))
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (/ (recall x y (position-of 'uw)) 3)))
	  (setq enemy (string-equal "*" (recall x y (position-of '*))))
	  (setq des (recall x y (position-of 'des)))
	  (setq mil (recall x y (position-of 'mil)))

	  (setq ideal
		(cond
		 ((or (string-equal des "w")
		      (string-equal des "b")
		      (string-equal des "n")
		      )
		  (max 75 (1+ (/ bodies (if enemy 7 (if happy 20 10))))))

		 ((or (string-equal des "*")
		      )
		  (max 25 (1+ (/ bodies (if (or (not happy) enemy) 7 50)))))

		 ((string-equal des "c")
		  (max 50 (1+ (/ bodies (if (or (not happy) enemy) 5 10)))))

		 ((string-equal des "e")
		  (max 150 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  ;;(min 100 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  )

		 ((string-equal des "f")
		  (max 20 (1+ (/ bodies (if (or (not happy) enemy) 7 40)))))

		 ((string-equal des "+")
		  (max 3 (1+ (/ civs (if enemy
					  (if happy 20 10)
					(if happy 200 20)))))
		  )

		 ((string-equal des "^")
		  0
		  )

		 (t
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
		  (1+ (/ civs (if enemy (if happy 20 10) (if happy 200 50))))
		  )
		 ))
	  ;;(error "stop")
	  
	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
	  ;;(mapcar '(lambda (pt)
	  ;;	       (if (not (recall x y 'mob))
	  ;;		 (setq near-enemyp t)))
	  ;;	    (sects-near (cons x y)))
	  ;;(if near-enemyp
	  ;;	(/ (* 3 ideal) 2)
	  ;;  ideal)
	  ))
      ;; not ours
      (or (recall x y (position-of 'mil)) 0)
      )))
(put 'ideal-mil-young 'empire t)

(defun ideal-mil-protect (x y)
  "Calculates ideal mil amount for the sector.
   This version gets 1 mil everywhere"
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	civs
	bodies
	ideal
	(near-enemyp nil))
    (if ours
	(if (> 30 (recall x y (position-of 'eff)))
	    0 ; no mil yet
	(progn
	  (setq happy (= 100 ours))
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (/ (recall x y (position-of 'uw)) 3)))
	  (setq enemy (string-equal "*" (recall x y (position-of '*))))
	  (setq des (recall x y (position-of 'des)))
	  (setq mil (recall x y (position-of 'mil)))

	  (setq ideal
		(cond
		 ((or (string-equal des "w")
		      (string-equal des "b")
		      (string-equal des "n")
		      )
		  (max 75 (1+ (/ bodies (if enemy 7 (if happy 20 10))))))

		 ((or (string-equal des "*")
		      )
		  (max 25 (1+ (/ bodies (if (or (not happy) enemy) 7 50)))))

		 ((string-equal des "c")
		  (max 50 (1+ (/ bodies (if (or (not happy) enemy) 5 10)))))

		 ((string-equal des "e")
		  (max 150 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  ;;(min 100 (1+ (if enemy (/ bodies 10) (/ civs 3))))
		  )

		 ((string-equal des "f")
		  (max 20 (1+ (/ bodies (if (or (not happy) enemy) 7 40)))))

		 ((string-equal des "+") 1)

		 ((string-equal des "^")
		  0
		  )

		 (t
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
		  1
		  )
		 ))
	  ;;(error "stop")
	  
	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
	  ;;(mapcar '(lambda (pt)
	  ;;	       (if (not (recall x y 'mob))
	  ;;		 (setq near-enemyp t)))
	  ;;	    (sects-near (cons x y)))
	  ;;(if near-enemyp
	  ;;	(/ (* 3 ideal) 2)
	  ;;  ideal)
	  ))
      ;; not ours
      (or (recall x y (position-of 'mil)) 0)
      )))
(put 'ideal-mil-protect 'empire t)

(defun ideal-mil-enlist (x y)
  "Calculates ideal mil amount for the sector
   This version gets 10 mil into all e centers -- is for fast enlistment in
new centers"
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	civs
	bodies
	ideal
	(near-enemyp nil))
    (if ours
	(if (> 5 (recall x y (position-of 'eff)))
	    0 ; no mil yet
	(progn
	  (setq happy (= 100 ours))
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (/ (recall x y (position-of 'uw)) 3)))
	  (setq enemy (string-equal "*" (recall x y (position-of '*))))
	  (setq des (recall x y (position-of 'des)))
	  (setq mil (recall x y (position-of 'mil)))

	  (setq ideal
		(cond
		 ((string-equal des "e") 10)

		 (t
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
		  ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
		  0
		  )
		 ))
	  ;;(error "stop")
	  
	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
	  ;;(mapcar '(lambda (pt)
	  ;;	       (if (not (recall x y 'mob))
	  ;;		 (setq near-enemyp t)))
	  ;;	    (sects-near (cons x y)))
	  ;;(if near-enemyp
	  ;;	(/ (* 3 ideal) 2)
	  ;;  ideal)
	  ))
      ;; not ours
      (or (recall x y (position-of 'mil)) 0)
      )))
(put 'ideal-mil-enlist 'empire t)

;;=> (defun ideal-mil (x y)
;;=>   "Calculates ideal mil amount for the sector.
;;=>	This version is for a RICH paranoid country with high tech and
;;=> insufficient happyness.
;;=>  normal sectors
;;=>	enemy = 1 to 10 if unhappy
;;=>		1 to 5 if happy
;;=>	ours = 1 to 20 if unhappy
;;=>	       10 if happy
;;=>  our forts	 min (50
;;=>		   1 to 7 if unhappy))
;;=>  our airports 1 to 30 if happy
;;=>		   1 to 10 if unhappy (do not lose an airport, it will fight back and scrap your planes when retaken)
;;=>  enemy forts  1 to 7 (do not want to lose fort)
;;=>  enemy banks, capitols  1 to 7
;;=>  our banks, warehouses  1 to 20 if happy, 1 to 10 if unhappy
;;=>  our capitols  1 to 10 if happy
;;=>		    1 to 5 if unhappy
;;=>  sects near rebellion need extra
;;=> "
;;=>   (interactive "nX: \nnY: ")
;;=>   (valid-sector x y)
;;=>   (let ((ours (recall x y 'work))
;;=>	des
;;=>	happy
;;=>	enemy
;;=>	mil
;;=>	civs
;;=>	bodies
;;=>	ideal
;;=>	(near-enemyp nil))
;;=>	 (if ours
;;=>	(progn
;;=>	  (setq happy (= 100 ours))
;;=>	  (setq bodies (+ (setq civs (recall x y 'civ))
;;=>			  (/ (recall x y 'uw) 3)))
;;=>	  (setq enemy (string-equal "*" (recall x y '*)))
;;=>	  (setq des (recall x y 'des))
;;=> 
;;=>	  (setq ideal
;;=>		 (cond
;;=>		  ((or (string-equal des "w")
;;=>		       (string-equal des "b")
;;=>		       (string-equal des "n")
;;=>		       )
;;=>		   (max 75 (1+ (/ bodies (if enemy 7 (if happy 15 10))))))
;;=> 
;;=>		  ((or (string-equal des "*")
;;=>		       )
;;=>		   (max 25 (1+ (/ bodies (if (or (not happy) enemy) 7 25)))))
;;=> 
;;=>		  ((string-equal des "c")
;;=>		   (max 20 (1+ (/ bodies (if (or (not happy) enemy) 5 10)))))
;;=> 
;;=>		  ((string-equal des "e")
;;=>		   (max 150 (1+ (if enemy (/ bodies 10) (/ civs 3))))
;;=>		   ;;(min 100 (1+ (if enemy (/ bodies 10) (/ civs 3))))
;;=>		   )
;;=> 
;;=>		  ((string-equal des "f")
;;=>		   (max 50 (1+ (/ bodies (if (or (not happy) enemy) 7 20)))))
;;=> 
;;=>		  ((string-equal des "+")
;;=>		   (1+ (/ civs (if enemy (if happy 20 10) (if happy 20 10))))
;;=>		   )
;;=> 
;;=>		  ((string-equal des "^")
;;=>		   0
;;=>		   )
;;=> 
;;=>		  (t
;;=>		   ;;(1+ (/ civs (if enemy (if happy 20 10) (if happy 50 10))))
;;=>		   (1+ (/ civs (if enemy (if happy 20 10) (if happy 30 10))))
;;=>		   )
;;=>		  ))
;;=>	  ;; if we are near an enemy we do NOT own, add 50% to normal amount
;;=>	  (mapcar '(lambda (pt)
;;=>		     (if (not (recall x y 'mob))
;;=>			 (setq near-enemyp t)))
;;=>		  (sects-near (cons x y)))
;;=>	  (if near-enemyp
;;=>	      (/ (* 3 ideal) 2)
;;=>	    ideal)
;;=>	  ))))



;;(with-output-to-temp-buffer "*Mil*"
;;  (mapsects '(lambda (x y) (princ (format "%s,%s  \t%s %s\n"
;;					x y des (ideal-mil x y))))))


(defun allowable-mouths (food)
  (/ (/ (* food
	   bodies-fead-by-1-food-per-etu)
	(* etu-per-update upd-per-day))
     3))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Civilian levels
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-ideal-civ-fcn 'ideal-civ-general
  "*The empire-ideal-civ-fcn variable names the function that will
calculate the ideal number of civ that the tool will attempt to place
everywhere. The placement is mostly done by the redistribute-civ command.

The user customization map will help you set this variable.")

(defun empire-ideal-civ-functions ()
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (and (not (commandp sym))
				   (get sym 'empire)
				   (string-match "ideal-civ" (symbol-name sym))
				   )
			      (setq vars (cons sym vars))))))
    vars
    (setq vars (sort vars 'string-lessp))
    ))

(defun describe-empire-ideal-civ-functions ()
  "Offers function documentation on all interactive functions empire offers
without regard to position in any keymap."
  (interactive)
  (with-output-to-temp-buffer "*Ideal Civ Functions*"
    (print-banner "Ideal Civ Functions")
    (terpri)
    (princ 
     (documentation-property 'empire-ideal-civ-fcn 'variable-documentation))
    (terpri)
    (terpri)
    (print-banner "Choices")
    (terpri)
    (mapcar '(lambda (function)
	       (prin1 function)
	       (princ ":")
	       (if (eq function empire-ideal-civ-fcn)
		   (princ " *Active Function*"))
	       (terpri)
	       (if (documentation function)
		   (princ (documentation function))
		 (princ "not documented"))
	       (terpri)
	       (terpri)
	       )
	    (empire-ideal-civ-functions))))

(defun empire-set-ideal-civ-function ()
  "Displays, prompts for, and sets the ideal-civ function"
  (interactive)
  (describe-empire-ideal-civ-functions)
  (setq empire-ideal-civ-fcn
	(intern (completing-read "Ideal Civ Function: "
			 (mapcar '(lambda (n) (list (symbol-name n)))
				 (empire-ideal-civ-functions))
			 nil t
			 (symbol-name empire-ideal-civ-fcn)
			 ))))
  
(defun ideal-civ-general (x y)
  "Calculates ideal civ amount for the sector.
   Basically, as many as food will support with max of
   empire-sector-civ-threshold and a min of 50.

   Efficient roads have a very low ideal civ level.   
   If unhappy or not ours, ideal is same as current"
  ;;(valid-sector x y)
  (let ((ours (recall x y (position-of 'work)))
	des
	happy
	enemy
	mil
	uw
	civs
	bodies
	allowable-mouths
	ideal-bods
	(near-enemyp nil))
    ;;(error "stop")
    (if (and ours (= 100 ours)
	     (not (string-equal "*" (recall x y (position-of '*)))))
	(progn
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (setq uw (recall x y (position-of 'uw)))
			  (setq mil (recall x y (position-of 'mil)))
			  ))
	  (setq des (recall x y (position-of 'des)))
	  (setq allowable-mouths (allowable-mouths (recall x y (position-of 'food))))
	  (if (string-equal des "^")
	      (setq ideal-civs 1)
	    (if (<= 98 (recall x y (position-of 'eff)))
		;; efficient sectors have different ideals
		;; non producers need bare bones staffing
		(cond
		 ((string-match des "+wf)")
		  (setq ideal-civs 55))
		 ;; others need the min level
		 (t (setq ideal-civs (min empire-sector-civ-threshold
					  (max 1
					       (- (max empire-never-starve-threshold
						       allowable-mouths)
						  (+ uw mil)))))))
	      ;; non efficient can take up to three times that amount
	      ;; provided that there is food
	      (setq ideal-civs (min
				950
				(* 3 empire-sector-civ-threshold)
				(max 1
				     (- (max empire-never-starve-threshold
					     allowable-mouths)
					(+ uw mil)))))
	      (if (string-equal des "+")
		  ;; non efficient roads can take twice what anything else can
		  ;; as they start body distribution
		  (setq ideal-civs (min 950
					(- (max empire-never-starve-threshold
						allowable-mouths)
					   (+ uw mil))
					(+ ideal-civs ideal-civs))))
	      ))
	  ideal-civs
	  )
      (recall x y (position-of 'civ))
      )))
(put 'ideal-civ-general 'empire t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Robert Forsman's ideal civ calcs
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun ideal-civ-building (des)
  "Returns how many people should be in a sector of type DES that
is increasing in efficiency.  Edit this function to your tastes."
  ( cond
    ( (string-match des "^-")	5)
    ( (string-match des "+")	200)
    ( (string-match des "w")	800)
    ( t				500)
    )
  )

(defun ideal-civ-demolishing (des)
  "Returns how many people should be tearing down a sector that will
be reconstructed into type DES.  Edit this function to your tastes."
  ( cond
    ( (string-match des "+")	200 )
    ( (string-match des "w")	800)
    ( t				500 )
    )
  )

(defun ideal-civ-buildall (eff)
  "Returns how many people are necessary to bring a sector at efficiency
EFF up to 100% in the next update."
  (/ (* 200 (- 100 eff)) etu-per-update)
  )

(defun ideal-civ-built (des)
  "Returns how many people should be in a sector of type DES that is
at 100% and producing.  Edit this function to your tastes."
  ( cond
    ( (string-match des "+)!")	5 )
    ( (string-match des "a=")	500 )
    ( (string-match des "cpdimfh*jktrnl#b%egou") 600 )
    ( (string-match des "w")	100 )
    ( t				5 )
    )
  )


(defun ideal-civ-max-eff-growth (x y)
  "Calculates ideal civ amount for the sector.  Alternative to
ideal-civ-general.
  This function calculates how many civilians should be in a sector
depending on what state of construction it is in (demolishing /
constructing / constructed ).  I may add another gradiation of
functioning for 60% producing sectors, but I doubt it.

  This function also takes into account exactly how many civs are
required to bring a sector to 100% in an update and not leave extra
civs around if they won't be needed in the completed sector.  This
would be useful for roads (which I decide only need 5 civs) so that
instead of having 200 on a 95% eff road it would drop down to 125 civs
and drop down to 5 after the update.  (this assumes an 8etu/update
game.  etu-per-update IS accounted for in ideal-civ-buildall.)
  In a gold mine, if you have 400 constructing and you are at 75%
efficiency then you only need 625 to finish construction, but you'd
like 800 in the finished sector.  This function will insert the full
800.

  Hopefully this latest bit of artificial brains will convert all the
old guard."

  (let ( (work (recall x y (position-of 'work)))
	 (near-enemyp nil)
	 )
    ;;(error "stop")
    (if (and work
	     (= 100 work)
	     (not (string-equal "*" (recall x y (position-of '*)))))
	(let ( des
	       sdes
	       eff
	       happy	enemy
	       mil	uw	civs
	       bodies	allowable-mouths
	       ideal-civ	ideal-built	ideal-building
	       )
	  (setq bodies (+ (setq civs (recall x y (position-of 'civ)))
			  (setq uw   (recall x y (position-of 'uw)))
			  (setq mil  (recall x y (position-of 'mil)))
			  ))
	  (setq eff  (recall x y (position-of 'eff ))
		des  (recall x y (position-of 'des ))
		sdes (recall x y (position-of 'sdes)))
	  (setq allowable-mouths
		(allowable-mouths (recall x y (position-of 'food))))
	  (setq ideal-civ
		( cond
		  ( (string-equal sdes "_")
		    ;; we're on our way up in efficiency
		    (setq ideal-civ	(ideal-civ-buildall eff)
			  ideal-built	(ideal-civ-built des) )
		    (if (> ideal-built ideal-civ)
			;; we want more in the built sector than it
			;; takes to complete construction.  put them in.
			ideal-built
		      ;; else
		      (setq ideal-building (ideal-civ-building des))
		      (if (> ideal-building ideal-civ)
			  ;; we need less to complete than we'd normally want
			  ;; in a sector under construction.  put the lesser
			  ;; amount in.
			  ideal-civ
			;; else
			;; we need more to complete than we want to allocate
			;; make do with the allocation.
			ideal-building)
		      )
		    )
		  ( t (ideal-civ-demolishing sdes) )
		  )
		)
	  (setq ideal-civ (min allowable-mouths ideal-civ) )
	  )
      (recall x y (position-of 'civ))
      )
    )
  )
(put 'ideal-civ-max-eff-growth 'empire t)

(defun describe-sect-ideals (x y)
  (format "%s  %s/%sc  %s/%sm  %s/%sf"
	  (sector-basic-string x y)
	  (recall x y (position-of civ))
	  (funcall empire-ideal-civ-fcn x y)
	  (recall x y (position-of mil))
	  (funcall empire-ideal-mil-fcn x y)
	  (recall x y (position-of food))
	  (if (and 
	       (recall x y (position-of civ))
	       (recall x y (position-of mil))
	       (recall x y (position-of uw)))
	      (/ (+ (recall x y (position-of civ))
		    (recall x y (position-of mil))
		    (recall x y (position-of uw)))
		 5)
	    "?")))
(register-describe-sector-fcn 'describe-sect-ideals)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun unload-dist-node (dx dy)
  "Suggests alternate dist routes for all items distributing to a given
node. This is useful to reduce the burden on a node or even to eliminate
its use as a dist pt entirely." 
  (interactive "nX: \nnY: ")
  (valid-sector dx dy)
  (with-output-to-temp-buffer (format "*Alt_dist_%s,%s*" dx dy)
    (empire-data-mode-buffer t t (format "*Alt_dist_%s,%s*" dx dy))
    (print-banner "Alternate distribution pts")
    (terpri)
    (empire-switch-to-buffer-not-map (format "*Alt_dist_%s,%s*" dx dy))
    (let ((old-dependents (recall dx dy (position-of dependents)))
	  (old-des (recall dx dy (position-of des)))
	  (send-automated-commands nil) ; never with this command
	  paths
	  dest
	  examined to-path
	  )
      ;; make given pt not seem to be a dist node
      (record dx dy (position-of dependents) nil)
      ;;(record dx dy (position-of des) "+")

      (walksects-active
       (if (and (= dx (or (recall x y (position-of dist_x)) dx))
		(= dy (or (recall x y (position-of dist_y)) dy))
		(not (and (= dx x) (= dy y))))
	   (progn
	     (show-pt (cons x y) t)
	     (setq examined nil to-path nil)
	     (setq paths (find-sorted-unique-dist-pts (cons x y) 10))
	     (if paths
		 (progn
		   (terpri)
		   (princ (format "  Trying to unload %s,%s\n" x y))
		   (while paths
		     (setq dest (car (last (car paths))))
		     (princ (format "    to %s\n"
				    (describe-sect-prod-dist (car dest) (cdr dest))))
		     (princ
		      (format "dist %s,%s %s\n"
			      x y (r-to-path (car paths))))
		     (setq paths (cdr paths))
		     )
		   (sit-for 0)
		   )
	       ;; no paths
	       (princ (format "  %s %s has no other obvious distribution choices\n"
			      (format-xy x y)
			      (recall x y (position-of des))))
	       )))
       "Unloading")
      ;;(record dx dy (position-of dependents) old-dependents)
      ;;(record dx dy (position-of des) old-des)

      )))
(put 'unload-dist-node 'empire t)
		    
(defun check-dist-is-road (x y)
  (let ((path (recall x y (position-of 'dist_path)))
	(sx x)
	(sy y)
	(i 0)
	dir
	des
	paths examined to-path second-pass)
    (if path
	(while (< i (length path))
	  (setq dir (aref path i)
		sx (+ sx (x-offset-in-char-dir dir))
		sy (+ sy (y-offset-in-char-dir dir))
		i (1+ i)
		des (aref (recall-macro sx sy 'des) 0)
		)
	  (if (not (or (= ?+ des)
		       (= ?= des)
		       (= ?w des)))
	      (progn
		(princ (format "  %s %s goes to non-road %s at %s\n"
			       (format-xy x y)
			       (recall-macro x y 'des)
			       (format-xy sx sy)
			       (recall-macro sx sy 'des)))
		(sit-for 0)
	     (show-pt (cons x y) t)
	     (setq i 12)
	     (setq paths (find-sorted-unique-dist-pts (cons x y) 10))
	     (if paths
		 (progn
		   (setq examined nil to-path nil second-path nil)
		   ;;(princ (format "  Trying to redist %s,%s\n" x y))
		   (while paths
		     (setq dest (car (last (car paths))))
		     (princ (format "    to %s\n"
				    (describe-sect-prod-dist (car dest) (cdr dest))))
		     (princ
		      (format "%sdist %s,%s %s\n"
			      (if second-path "      " "")
			      x y (r-to-path (car paths))))
		     (setq second-path t)
		     (setq paths (cdr paths))
		     )
		   (terpri)
		   (sit-for 0)
		   )
	       ;; no paths
	       )
	     ))))))

(defun empire-find-non-road-dist ()
  "Reports each dist path going over non roads. Suggests alt dist paths.
This is useful when sectors have changed designations and what was an
efficient path is no longer efficient.

This command takes a LONG time to execute"
  (interactive)
  (with-output-to-temp-buffer "*Bad Dist Paths*"
    (print-banner "Suspect Dist Paths")
    (empire-data-mode-buffer t t "*Bad Dist Paths*")
    (sit-for 0)
    (walksects-active (check-dist-is-road x y)
		      "Checking dist path")))
    
  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Create a ship check hook
;;   (New gnurus hackers -- note how easy this is)
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'emp-ship)
(defun check-ship-food ()
  "Checks that each ship has enough food for a day or so"
  (mapships-ours-with-vars-bounded
   (function (lambda (num)
	       (if empire-interaction-verbosity
		   (message "Food checking ship %s %s" num type))
	       (if (> (+ 1 (/ (+ civ mil uw 1) 10)) (/ food 3))
		   (progn
		     (princ (format "  Ship %s %s at %s has %s bodies and only %s food\n"
				    num type
				    (format-xy x y)
				    (+ civ mil uw)
				    food))
		     ;; correct the ship, if possible, by finding another ship in
		     ;; the same sector and tending
		     (let ((tended nil))
		       ;; first, see if it is in a harbor
		       (if (and (string-equal (recall x y (position-of 'des)) "h")
				(< 100 (recall x y (position-of 'food))))
			   (progn
			     (setq tended t)
			     (make-automated-command
			      (format "load %s 80 f" num))
			     (decriment-sector x y (position-of 'food) 80)))
		       (mapships-ours
			'(lambda (newnum)
			   (if empire-interaction-verbosity
			       (message "Food source checking ship %s"  newnum))
			   (if (and (not tended)
				    (= (ship-recall newnum 'x) x)
				    (= (ship-recall newnum 'y) y)
				    ;; If I new more about ships, I could calc the best
				    ;; amount fo food to transfer and would know how
				    ;; much could actually make it. As it is, make a
				    ;; safe guess.
				    (> (ship-recall newnum 'food) 40) 
				    )
			       (progn
				 (setq tended t)
				 (make-automated-command
				  (format "tend %s f %s %s"
					  newnum (- (ship-recall newnum 'food) 40) num))
				 ;; I have no idea how much food actually went. Set it
				 ;; for worst case
				 (ship-record newnum 'food 40)
				 )
			     )))
		       (if (and (not tended) (< 0 (+ civ mil uw)))
			   (message "  Ship %s %s at %s has %s bodies and only %s food and no tenders\n"
				    num type
				    (format-xy x y)
				    (+ civ mil uw)
				    food))
		       )

		    
		     ))))))
(register-check-hook 'check-ship-food "Ship Food" t)

;;->                                -*- Ship Food -*-
;;->   Ship 147 destroyer        at -42,6      has 50 bodies and only 10 food
;;->   Ship 76 patrol boat      at 76,42      has 10 bodies and only 3 food
;;->   Ship 68 aircraft carrier at -26,-12    has 99 bodies and only 25 food
;;-> ;;tend 157 f 54 68
;;-> ;;tend 69 f 26 68
;;->   Ship 12 patrol boat      at 5,-19      has 0 bodies and only 0 food
;;->   Ship 11 patrol boat      at -53,71     has 0 bodies and only 0 food


    
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Automatic command execution
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-automatically-execute-commands nil
  "*If t, suggestions found during analysis will automatically be sent to
the client. This is in effect makes you always trust the analysis tools.
Remember that there is no warranty.")

(defun make-automated-command (str &optional no-btu-check)
(if empire-automatically-execute-commands
      (progn
	(wait-for-empire-prompt-if-necessary)
	(if (and (not no-btu-check)
		 (<= empire-btu-left 10)
		 )
	    (if empire-batch-play
		;; print what we would have done
		(message "No-BTU: %s" str)
	      (error "No BTUs left")
	      )
	  ;; else
	  (message "%s: %s" empire-btu-left str)
	  ;; place the line in the buffer as executed
	  (princ ";;")
	  (princ str)
	  (terpri)
	  ;; send it but don't lose our scanning place
	  (save-excursion (send-empire-command str))
	  ;; do not wait
	  ))
    ;; else do not execute it
    (message "No-exec: %s" str)
    (princ str)
    (terpri)
    (princ (concat str "\n")
	   (get-buffer-create "*Proposed*"))
    ))
  
(register-adjust-hook 'empire-flows t)
(register-adjust-hook 'redistribute-civ t)
(register-adjust-hook 'check-empire t)
(register-adjust-hook 'redistribute-mil t)
(register-adjust-hook 'redistribute-population t)
(register-adjust-hook 'redistribute-food t)
