/****************************************************************
*  function:  pathfinder.c
*
*  finds a path from a given sector to a given sector
*                                   or to nearest given sector-type
*
*  assumes that it can access the global data-structure Map
*  returns a pointer to the path if successful;
*  returns NULL if it can't find a path
*
****************************************************************/

#include <stdio.h>
#include "sve.h"
#include <string.h>

#define ALLOWABLE_BADSECTORS 20	/* when to give up! */
#define MAXPATH 64

struct pathinfo {		/* create a doubly linked list */
    struct pathinfo *next;	/* pointer to next structure */
    struct pathinfo *last;	/* pointer to last structure */
    char path[MAXPATH+1];	/* the path itself; allow for null terminator*/
    int end_x;			/* where the x-coord is now */
    int end_y;			/* where the y-coord is now */
    int badsectors;		/* how many "bad sectors" path goes thru */
  };	

extern char Direction[6];
extern struct maptype Map[XWIDTH][YHEIGHT];

struct pathinfo *pathlist;

char *
pathfinder(from_x, from_y, target_x, target_y, target_type, max_pathlength)

int from_x;			/* from x-coord */
int from_y;			/* from y-coord */
int *target_x;			/* to x_coord; if 999 look for target_type */
int *target_y;			/* to y_coord; if 999 look for target_type */
char target_type;		/* look for nearest sector of type
                                   target_type; target_x and target_y
                                   parameters are 999. If target_x and 
                                   target_y are given, target_type is '?'.
                                   If target_type is set, target_x and
                                   target_y addresses are filled with
                                   the coords of the sector that is found. */
int max_pathlength;		/* will not return a path exceeding this */

{ /*main*/
  struct pathinfo *temp;
  struct pathinfo *current;
  struct pathinfo *putbefore;	/* used as a pointer to sort paths */
  int i, j;
  int used[XWIDTH][YHEIGHT];	/* array of flags: 0 if available to
                                   a path, toggle to 1 when used */
  int length;			/* length of current paths */
  char *answer;
  int dir;
  int max_badsectors;

  answer = NULL;

for (max_badsectors=0; max_badsectors < ALLOWABLE_BADSECTORS; max_badsectors++)
  {  /*badsector loop*/

    if (answer != NULL)		/* get out if answer is found */
      break;

    for (i=0; i<XWIDTH; i++)	/* clear array of "used" flags */
      for (j=0; j<YHEIGHT; j++)
	used[i][j] = 0;
    used[from_x][from_y] = 1;	/* flag the starting sector */

    if ((pathlist = (struct pathinfo *) malloc (sizeof(struct pathinfo))) == NULL)
      {
	printf("Memory allocation error at pathlist.\n");
	exit(0);
      }
    pathlist->next = NULL;
    pathlist->last = NULL;	/* starts a new linked list of paths */
    strcpy(pathlist->path, "");
    pathlist->end_x = from_x;
    pathlist->end_y = from_y;
    pathlist->badsectors = 0;

    length = 0;

    while ((length < max_pathlength) && 
	   (answer == NULL) && 
	   (pathlist != NULL))

      { /*reach out one more sector*/

	current = pathlist;
 
	while ((current != NULL) && (answer == NULL))
	  { /*expand each path in list*/
	    
	    for (dir=0; dir < 6; dir++)
	      { /*expand in six directions*/

		if ((temp = (struct pathinfo *) malloc   /* allocate a new */
		     (sizeof(struct pathinfo))) == NULL) /* structure    */
		  {
		    printf("Memory allocation error.\n");
		    exit(0);
		  }

		translocate(dir, current->end_x, current->end_y,
			    &(temp->end_x), &(temp->end_y));
		
		strcpy(temp->path, current->path);
		catachar(temp->path, Direction[dir]);
		temp->badsectors = current->badsectors;
		
		if (used[temp->end_x][temp->end_y])
		  {
		    free(temp);
		    continue;
		  }

		used[temp->end_x][temp->end_y] = 1;

		if (!is_sector(temp->end_x, temp->end_y))
		  {
		    free(temp);
		    continue;
		  }
		
		if (end_path(temp->end_x, temp->end_y, target_x, target_y,
			     target_type))
		  {
		    if ((answer = (char *) malloc (length+4)) == NULL)
		      {
			printf("Memory allocation error at answer.\n");
			exit(0);
		      }

		    strcpy(answer, temp->path);
		    *target_x = temp->end_x;
		    *target_y = temp->end_y;
		    free(temp);
		    break;
		  }

		if ((temp->badsectors += 
		     is_badsector(temp->end_x, temp->end_y)) <= max_badsectors)
		  {
				/* this next section of code sorts */
				/* the pathlist; new path is inserted */
				/* before sectors with higher badsector */
				/* counts. This way, low badsector paths */
				/* will be expanded before high badsector */
				/* paths, preventing screw-ups caused by */
				/* the immediate flagging of "used" even */
				/* if a low badsector path could have used */
				/* the sector successfully... */

		    putbefore = current;
		    while ( (putbefore->last != NULL) && 
			   ((putbefore->last)->badsectors > temp->badsectors))
		      putbefore = putbefore->last;
				/* so: it slides the pointer putbefore */
				/* back (starting at current) until it */
				/* points at an element with equal or less */
				/* badsectors; now insert temp there. */

		    insertpath(&temp, &(putbefore->last), &putbefore);
		  }
		else
		  free(temp);
	      }

	    temp = current->next;
	    deletepath(&current, &(current->last), &(current->next));
	    current = temp;

	  }
    length++;
   }
   
    while (pathlist != NULL)	/* remove the entire pathlist when done */
      {
	temp = pathlist->next;
	free(pathlist);
	pathlist = temp;
      }


  }
		    
  if (answer != NULL)
      return(answer);
  else
    {
      if ((answer = (char *) malloc (5)) == NULL)
	{
	  printf("Memory allocation error leaving pathfinder.\n");
	  exit(0);
	}
      strcpy(answer, "NULL");
      return(answer);
    }
}



/****************************************************************
*  function: insertpath
*
*  insert a path into pathfinder's dynamic linked list
****************************************************************/

insertpath(path, after, before)
struct pathinfo **path;
struct pathinfo **after;
struct pathinfo **before;
{
  extern struct pathinfo *pathlist;
  
  if ((*before)->last == NULL)
    {
      pathlist = *path;
      (*path)->last = NULL;
    }
  else
    {
      (*path)->last = *after;
      (*after)->next = *path;
    }
      
  (*path)->next = *before;
  (*before)->last = *path;
   
}


/****************************************************************
*  function: deletepath
*
*  delete a path from pathfinder's dynamic linked list, and free
*  its pointer
****************************************************************/

deletepath(path, isafter, isbefore)
struct pathinfo **path;
struct pathinfo **isafter;
struct pathinfo **isbefore;

{
  extern struct pathinfo *pathlist;

  if (*isafter != NULL)
    (*isafter)->next = *isbefore;
  else
    pathlist = (*path)->next;

  if (*isbefore != NULL)
    (*isbefore)->last = *isafter;

  free(*path);
}

  
/****************************************************************
*  function: is_sector
*
*  returns 1 if sector x,y is either active or a dummy
*  expects to find global Map[][] structures
****************************************************************/

is_sector(x, y)

int x;				/* sve coords */
int y;
{
  if ((Map[x][y].status == 1) || (Map[x][y].status == 2))
    return(1);
  else
    return(0);
}


/****************************************************************
*  function: is_badsector
*
*  returns an approximate mobility cost
*        +, = return 0
*        ^ return 10
*        everything else returns 1
*
*  expects to find the global structures Map[][]
****************************************************************/

is_badsector(x,y)

int x;				/* sve coords */
int y;
{
  if ((Map[x][y].designation == '+') || (Map[x][y].designation == '='))
    return(0);
  else if (Map[x][y].designation == '^')
    return(10);
  else
    return(1);
}



/****************************************************************
*  function: end_path
*
*  checks for success of pathfinder algorithm
*  returns 1 if success; 0 otherwise
*
*  assumes existence of global structures Map[][]
****************************************************************/

end_path(x, y, target_x, target_y, target_type)

int x;
int y;
int *target_x;
int *target_y;
char target_type;
{
  if ((*target_x == 999) && (Map[x][y].designation == target_type))
    return(1);
  else if ((*target_x == x) && (*target_y == y))
    return(1);
  else
    return(0);
}


/****************************************************************
* function: catachar(s1, c)
* 
* concatenates a char c onto a string f
****************************************************************/

catachar(s1, c)
char *s1;
char c;
{
  int index = 0;

  for (index=0; s1[index] != '\0'; index++)
    ;

  s1[index] = c;
  s1[index+1] = '\0';

}


#ifdef DEBUG
/****************************************************************
*  function: showpaths
*
*  debugging tool: prints out all paths in pathlist
****************************************************************/

showpaths()
extern struct pathinfo *pathlist;
{
  struct pathinfo *current;

  current = pathlist;
  while (current != NULL)
    {
      printf("%s \n", current->path);
      current = current->next;
    }
}
#endif
