/************************************/
/*                                  */
/*            EMI88.c               */
/*                                  */
/*    Execute MARS Instruction      */
/*    ICWS'88 Standard Execution    */
/*                                  */
/* Mark A. Durham                   */
/* Last Update: December 11, 1991   */
/*    Added comments.               */
/*    Fixed errors in SUB and SLT.  */
/*                                  */
/* Scott W. Adkins                  */
/* Last Update: January 2, 1991     */
/*    Added new parameters.         */
/*    Added new fields to the core. */
/*    Modified for loop.            */
/*    Changed to Classic C.         */
/*                                  */
/************************************/

/* This ANSI C function implements a MARS instruction       */
/* interpreter.  It supports two programs, although it can  */
/* be easily changed to support more.  Each program can     */
/* have numerous tasks, the number is passed as a parameter,*/
/* and memory size is also passed as a parameter.           */

/* The comments will help you convert to Classic C if it is */
/* necessary.                                               */

/* Instructions are stored in core in Statement_Registers.  */
/* The Opcode and Modes of the instruction are stored in    */
/* the Control field.  The A-address is stored in the       */
/* AField field and the B-address is stored in the BField   */
/* field.  The field sizes can be changed with little       */
/* change to the program.  For instance, the Control field  */
/* will fit in an unsigned char (in ICWS'88) and making the */
/* address fields unsigned int will double your addressable */
/* core size.                                               */

/* ---- Added the ownership field. (Scott Adkins)           */
/* ---- Added aging field. (Scott Adkins)                   */
struct Statement_Register {         /* Instruction type */
   int   Control;                   /* Opcode and Modes */
   int   AField;
   int   BField;
   int	 Owner;
   int   Aging;
};


/* Pointer counters are stored in CoreNodes.  There is a    */
/* CoreNode for each current task.  A CoreNode has pointers */
/* to both its predecessor (last) and successor (next)      */
/* nodes.  The pointer counter (PC) stores the address of   */
/* the instruction to be executed next by this task.        */

struct CoreNode {
   struct CoreNode *last;           /* Last task to execute */
   int PC;                          /* Pointer counter for task */
   struct CoreNode *next;           /* Next task to execute */
};


/* These are the supported opcodes.  Use #defines if your   */
/* compiler does not support enums.  "#define DAT 0" is an  */
/* example.                                                 */

enum Opcode {
   DAT,
   MOV,
   ADD,
   SUB,
   JMP,
   JMZ,
   JMN,
   DJN,
   CMP,
   SLT,
   SPL
};


/* These are the supported modes.  Use #defines if your     */
/* compiler does not support enums.  "#define IMMEDIATE 0"  */
/* is an example.                                           */

enum Mode {
   IMMEDIATE,
   DIRECT,
   INDIRECT,
   PREDECREMENT
};


/* The function itself does not return any useful infor-    */
/* mation other than TRUE for a successful interpretation   */
/* or FALSE if not.  The assembler and loader should really */
/* catch all of the errors.  No errors should occur during  */
/* instruction execution.                                   */

#define  FALSE 0
#define  TRUE  1


/* For Classic C, you will need to change the parameter       */
/* declarations to                                            */
/*   int EMI88(CLL,CORE,MaxTask,memory,Tasks,Programs,cycles) */
/*       struct CoreNode   *CLL;                              */
/*       etc.                                                 */
/*   {                                                        */

int EMI88(CLL,CORE,MaxTask,memory,Tasks,Programs,cycles)

/* Each program has a Circularly Linked List (CLL) of       */
/* tasks.  It is assumed that each list has been            */
/* initialized to a CoreNode with the starting pointer      */
/* counter set to the program's starting address and the    */
/* last and next pointers both pointing to itself prior to  */
/* first calling EMI88().                                   */

   struct CoreNode            *CLL[];

/* CORE[] is just an array of Statement_Registers.  It is   */
/* assumed that CORE has been preset to DAT 0 0 and that    */
/* the programs have been loaded before first calling       */
/* EMI88().                                                 */

   struct Statement_Register  CORE[];

/* MaxTask is the maximum number of allowable tasks per     */
/* program.                                                 */

   int                        MaxTask;

/* memory is the size of CORE.                              */

   int                        memory;

/* Tasks[] keeps track of the number of viable tasks per    */
/* per program.  It is assumed that Tasks[] is preset to    */
/* one for each program before first calling EMI88().       */

   int                        Tasks[];

/* ---- Added a new parameter to allow more than one battle */
/* ---- program into the arena.  (Scott Adkins)             */

   int                        Programs;

/* ---- Added a new parameter to keep track of how many     */
/* ---- cycles have passed thus far.  This will be used to  */
/* ---- mark the age of the memory location, so when used   */
/* ---- with a curses display, old locations fade from the  */
/* ---- screen.  (Scott Adkins)                             */

   int                        cycles;
{

/* This MARS stores the currently executing instruction in  */
/* the Statement_Register SR.                               */

   struct Statement_Register  SR;

/* This MARS stores the instruction referenced by the       */
/* A-operand in Statement_Register SRA.                     */

   struct Statement_Register  SRA;

/* This MARS stores the instruction referenced by the       */
/* B-operand in Statement_Register SRB.                     */

   struct Statement_Register  SRB;

/* We will often need to change the linkage of tasks for    */
/* executing DAT and SPL.  The pointers lastCoreNode and    */
/* tempCoreNode assist in this.                             */

   struct CoreNode            *lastCoreNode;
   struct CoreNode            *tempCoreNode;

/* index refers to which program we are interpreting.  Each */
/* program will execute one cycle from one task per call to */
/* EMI88().                                                 */

   int                        index;

/* PC is the current Pointer Counter (the address of the    */
/* instruction being interpreted).                          */

   int                        PC;

/* The address of the instruction referred to by the        */
/* A-operand is Pointer Counter A (PCA).                    */

   int                        PCA;

/* The address of the pointer which points to the           */
/* instruction referred to by an indirect A-operand is      */
/* the Pointer Counter A Indirect (PCAI).                   */

   int                        PCAI;

/* The address of the instruction referred to by the        */
/* B-operand is Pointer Counter B (PCB).                    */

   int                        PCB;

/* The address of the pointer which points to the           */
/* instruction referred to by an indirect B-operand is      */
/* the Pointer Counter B Indirect (PCBI).                   */

   int                        PCBI;

/* Note that by careful coding PCAI and PCBI may be removed */
/* from the program.  I think that they serve a useful      */
/* teaching function however, and perhaps they will be      */
/* needed in a future standard.  They were necessary for    */
/* ICWS'86 for instance.                                    */

/* OpCode is the current opcode.  Use "int OpCode" for      */
/* Classic C.                                               */

   enum Opcode                OpCode;

/* AMode is the mode of the A-operand of the current        */
/* instruction.  Use "int AMode" for Classic C.             */

   enum Mode                  AMode;

/* AValue is the value used by MOV, ADD, CMP, etc.          */

   int                        AValue;

/* BMode is the mode of the B-operand of the current        */
/* instruction.  Use "int BMode" for Classic C.             */

   enum Mode                  BMode;

/* BValue is the other value used by MOV, ADD, CMP, etc.    */

   int                        BValue;

/* cmp is used solely within the evaluation for CMP.  It is */
/* either TRUE or FALSE.                                    */

   int                        cmp;


/* Execute one instruction from one task from each program. */
/* ---- Changed loop to take into the fact that there may   */
/* ---- more than one program in the arena.  (Scott Adkins) */
   for (index = 0; index < Programs; index++) {

/* ---- Added a check to make sure there are still tasks to */
/* ---- to be executed for program.  (Scott Adkins)         */

	if (Tasks[index] <= 0) continue;

/* Copy the current task pointer into the pointer counter.  */

      PC = CLL[index]->PC;

/* Copy the current instruction into the Statement Register.*/

      SR = CORE[PC];

/* Extract the A-mode.  I have placed the A-mode in the     */
/* third and fourth bits of the Control field of the        */
/* instruction.                                             */

      AMode = (SR.Control & 0x0C)/4;

/* For IMMEDIATE A-mode instructions the A Pointer Counter  */
/* points to the current instruction, the A Statement       */
/* Register is a copy of the current instruction, and the   */
/* A-value is a copy of the current instruction's A-field.  */

/* It is important to make copies because PREDECREMENT      */
/* modes make changes to CORE.  These changes should not    */
/* affect the evaluation of the operands and the execution  */
/* of instructions (other than those changes PREDECREMENT   */
/* A-mode evaluation makes prior to evaluation of the       */
/* B-operand).                                              */

      if (AMode == IMMEDIATE) {
         PCA = PC;
         SRA = CORE[PCA];
         AValue = SRA.AField;

      } else {

/* For DIRECT A-mode instructions the A Pointer Counter     */
/* points to an instruction relative to the current         */
/* instruction, the A Statement Register is a copy of that  */
/* instruction, and the A-value is a copy of the B-field of */
/* that instruction.                                        */

         PCA = (PC + SR.AField) % memory;
         SRA = CORE[PCA];

         if (AMode != DIRECT) {

/* For both INDIRECT and PREDECREMENT A-mode instructions   */
/* we copy the pointer to the pointer into PCAI.            */

            PCAI = PCA;

/* For PREDECREMENT A-mode instructions the pointer is      */
/* decremented and the memory from which the pointer came   */
/* is decremented.                                          */

            if (AMode == PREDECREMENT) {
               PCA = (PCA + memory - 1) % memory;
               CORE[PCAI].BField = (CORE[PCAI].BField + memory - 1) % memory;
            };

/* For both INDIRECT and PREDECREMENT A-mode instructions   */
/* the offset is added to the A Pointer Counter and the     */
/* target instruction is copied into the A Statement        */
/* register and the A-value is a copy of the B-field of     */
/* instruction.                                             */

            PCA = (PCA + SRA.BField) % memory;
            SRA = CORE[PCA];
         };

/* Here is where DIRECT, INDIRECT, and PREDECREMENT A-mode  */
/* instructions get their A-field.                          */

         AValue = SRA.BField;
      };


/* Extract the B-mode.  I have placed the B-mode in the     */
/* first and second bits of the Control field of the        */
/* instruction.                                             */

      BMode = SR.Control & 0x03;

/* For IMMEDIATE B-mode instructions the B Pointer Counter  */
/* points to the current instruction, the B Statement       */
/* Register is a copy of the current instruction, and the   */
/* B-value is a copy of the current instruction's B-field.  */

      if (BMode == IMMEDIATE) {
         PCB = PC;
         SRB = CORE[PCB];

      } else {

/* For DIRECT B-mode instructions the B Pointer Counter     */
/* points to an instruction relative to the current         */
/* instruction, the B Statement Register is a copy of that  */
/* instruction, and the B-value is a copy of the B-field of */
/* that instruction.                                        */

         PCB = (PC + SR.BField) % memory;
         SRB = CORE[PCB];

         if (BMode != DIRECT) {

/* For both INDIRECT and PREDECREMENT B-mode instructions   */
/* we copy the pointer to the pointer into PCBI.            */

            PCBI = PCB;

/* For PREDECREMENT B-mode instructions the pointer is      */
/* decremented and the memory from which the pointer came   */
/* is decremented.                                          */

            if (BMode == PREDECREMENT) {
               PCB = (PCB + memory - 1) % memory;
               CORE[PCBI].BField = (CORE[PCBI].BField + memory - 1) % memory;
            };

/* For both INDIRECT and PREDECREMENT B-mode instructions   */
/* the offset is added to the B Pointer Counter and the     */
/* target instruction is copied into the B Statement        */
/* register and the B-value is a copy of the B-field of     */
/* instruction.                                             */

            PCB = (PCB + SRB.BField) % memory;
            SRB = CORE[PCB];
         };
      };

/* Here is where instructions get their B-field, no matter  */
/* what their B-mode.                                       */

      BValue = SRB.BField;

/* ---- Added the marking of the current location as being  */
/* ---- owned by program executing it and also added the    */
/* ---- recording of age.  (Scott Adkins)                   */

      SR.Owner = index + 1;
      SR.Aging = cycles;

/* This tasks pointer counter is incremented.               */

      CLL[index]->PC = (CLL[index]->PC + 1) % memory;

/* The program's task pointer is moved to the next task.    */
/* lastCoreNode points to the currently executing task.     */

      lastCoreNode = CLL[index];
      CLL[index] = CLL[index]->next;

/* The opcode is stored in the fifth, sixth, seventh, and   */
/* eighth bits of the Control field.                        */

      OpCode = (SR.Control & 0xF0)/16;


/* Note that the vast majority of the work has already been */
/* done and that we are only now starting to look at the    */
/* opcode.  By evaluating the operands first, we have       */
/* substantially reduced the number of instructions we need */
/* to test in order to verify the proper functioning of the */
/* MARS.                                                    */

      switch (OpCode) {

/* DAT removes the current task from the current program's  */
/* task list.  (Remember the current task is now pointed to */
/* by lastCoreNode).                                        */

      case DAT:
         Tasks[index] -= 1;
         tempCoreNode = lastCoreNode;
         CLL[index]->last = tempCoreNode->last;
         tempCoreNode->last->next = CLL[index];
         free(tempCoreNode);
         break;

/* Note that when a program's last task executes a DAT      */
/* instruction, the CLL becomes a dangling pointer.  It is  */
/* important to check and make sure that Tasks[index] > 0   */
/* before assuming CLL[index] is a valid pointer.           */

/* MOV either copies the A-field to the B-field of the      */
/* instruction pointed to by the B Pointer Counter (for     */
/* IMMEDIATE A-mode instructions) or it copies the entire   */
/* instruction in the A Statement Register into the         */
/* instruction pointed to by the B Pointer Counter.         */

/* ---- Added recording of ownership.  (Scott Adkins)       */
/* ---- Added recording of aging.  (Scott Adkins)           */
      case MOV:
         if (AMode == IMMEDIATE) {
            CORE[PCB].BField = AValue;
            CORE[PCB].Aging = cycles;
         } else {
            CORE[PCB] = SRA;
            CORE[PCB].Owner = index + 1;
            CORE[PCB].Aging = cycles;
         };
         break;

/* ADD adds the A-value and the B-value and stores the      */
/* result in the B-field of the instruction pointed to by   */
/* the B Pointer Counter.  If the A-mode is not IMMEDIATE,  */
/* then the A-field of the instruction in the A Statement   */
/* Register is added to the A-field of the instruction in   */
/* the A-field of the B Statement Register and the result   */
/* is stored in the A-field of the instruction pointed to   */
/* by the B Pointer Counter.                                */

/* ---- Added recording of aging.  (Scott Adkins)           */
      case ADD:
         CORE[PCB].Aging = cycles;
         CORE[PCB].BField = (AValue + BValue) % memory;
         if (AMode != IMMEDIATE) {
            CORE[PCB].AField = (SRA.AField + SRB.AField) % memory;
         };
         break;

/* SUB subtracts the A-value from the B-value and stores    */
/* the result in the B-field of the instruction pointed to  */
/* by the B Pointer Counter.  If the A-mode is not          */
/* IMMEDIATE, then the A-field of the instruction in the A  */
/* Statement Register is subtracted from the A-field of the */
/* instruction in the B Statement Register and the result   */
/* is stored in the A-field of the instruction pointed to   */
/* by the B Pointer Counter.                                */

/* ---- Added recording of aging.  (Scott Adkins)           */
      case SUB:
         CORE[PCB].Aging = cycles;
         CORE[PCB].BField = (BValue + memory - AValue) % memory;
         if (AMode != IMMEDIATE) {
            CORE[PCB].AField = (SRB.AField + memory - SRA.AField) % memory;
         };
         break;

/* JMP simply changes the task's Pointer Counter to the     */
/* address of the instruction pointed to by the A Pointer   */
/* Counter.                                                 */

      case JMP:
         lastCoreNode->PC = PCA;
         break;

/* JMZ changes the task's Pointer Counter to the address of */
/* the instruction pointed to by the A Pointer Counter only */
/* if the B-value is zero.                                  */

      case JMZ:
         if (BValue == 0) {
            lastCoreNode->PC = PCA;
         };
         break;


/* JMN changes the task's Pointer Counter to the address of */
/* the instruction pointed to by the A Pointer Counter only */
/* if the B-value is not zero.                              */

      case JMN:
         if (BValue != 0) {
            lastCoreNode->PC = PCA;
         };
         break;

/* DJN (Decrement Jump if Not zero) decrements the B-field  */
/* of the instruction pointed to by the B Pointer Counter.  */
/* It also decrements the B-value.  If the B0value is not   */
/* zero, DJN changes the task's Pointer Counter to the      */
/* address of the instruction pointed to by the A Pointer   */
/* Counter.                                                 */

/* ---- Added recording of aging.  (Scott Adkins)           */
      case DJN:
         CORE[PCB].Aging = cycles;
         CORE[PCB].BField = (CORE[PCB].BField + memory - 1) % memory;
         if (BValue != 1) {            /* ((BValue-1) != 0) */
            lastCoreNode->PC = PCA;
         };
         break;

/* CMP compares the A-value and the B-value if the A-mode   */
/* is IMMEDIATE, otherwise it compares the instruction in   */
/* the A Statement Register with the instruction in the B   */
/* Statement Register.  If there is no difference in values */
/* (A-mode IMMEDIATE) or instructions (otherwise), then the */
/* next instruction which was to be executed by this task   */
/* is skipped.                                              */

      case CMP:
         cmp = FALSE;
         if (AMode == IMMEDIATE) {
            if (AValue == BValue) {
               cmp = TRUE;
            };
         } else {
            if (SRA.Control == SRB.Control) {
               if (SRA.AField == SRB.AField) {
                  if (SRA.BField == SRB.BField) {
                     cmp = TRUE;
                  };
               };
            };
         };
         if (cmp) {
            lastCoreNode->PC = (PC + 2) % memory;
         };
         break;

/* SPL adds the task pointed to by the A Pointer Counter to */
/* this program's task list.  The new task will execute     */
/* after the next time the current task executes.           */

      case SPL:
         if (Tasks[index] < MaxTask) {
            Tasks[index] += 1;
            tempCoreNode = (struct CoreNode *)malloc(sizeof(struct CoreNode));
            tempCoreNode->last = lastCoreNode;
            tempCoreNode->PC = PCA;
            tempCoreNode->next = CLL[index];
            lastCoreNode->next = tempCoreNode;
            CLL[index]->last = tempCoreNode;
         };
         break;

/* SLT (Skip if A Less Than B) causes the current task to   */
/* skip the next instruction which was to be executed by    */
/* this task only when the A-value is less than the         */
/* B-value.  Note that no value is less than zero as only   */
/* positive values can be represented in Core War.          */

      case SLT :
         if (AValue < BValue) {
            lastCoreNode->PC = (PC + 2) % memory;
         };
         break;

/* Any other opcode is an error.                            */

      default:
         return(FALSE);
      };

   };

/* We are done.  There was no error.                        */

   return(TRUE);

}
