#include "corewar.h"

int Size = DEF_SIZE;			/* Size of the Core */
int Cycles = DEF_CYCLES;		/* Amount of time for game */
int Battles = DEF_BATTLES;		/* Number of battles */
int Display = DEF_DISPLAY;		/* Curses display */
int Fade = DEF_FADE;			/* Fade value on curses display */
int Arena = DEF_ARENA;			/* Number programs in arena */
int Stats = DEF_STATS;			/* Print loading statistics */
int Config = DEF_CONFIG;		/* Display configuration */
int Results = DEF_RESULTS;		/* Display results */
int Totals = DEF_TOTALS;		/* Display totals */
int Fight = DEF_FIGHT;			/* Type of fighting mode */


main(argc, argv)
int argc;
char **argv;
{
	int arena, argf = 1;
	PROGRAM *programs;
	register int i;
	CORE *Core;

	if (argc > 2) {
		if (!strcmp(argv[1],"-f")) {
			if (!ReadConfig(argv[2])) exit(1);
			argf = 3;
		}
	}

	if (Fight == SINGLES) {
		if ((argc-argf) > Arena) {
			printf("No more than %d programs allowed!\n",Arena);
			exit(1);
		}

		if ((argc-argf) < 2) {
			printf("Need at least two programs to play!\n");
			exit(1);
		}

		arena = argc-argf;
	} else if (Fight == DEBUG) {
		if ((argc-argf) != 1) {
			printf("Only one program allowed at a time!\n");
			exit(1);
		}
	} else {
		if ((argc-argf) > MAX_PROG) {
			printf("No more than %d programs allowed!\n",MAX_PROG);
			exit(1);
		}

		if ((argc-argf) < 2) {
			printf("Need at least two programs to play!\n");
			exit(1);
		}
	}

	if (Config) PrintConfig();

	srandom(time(0));

	if (!CreateCore(&Core,Size)) exit(1);

	for (i = argf; i < argc; i++) 
		if (!CreateProgram(&programs,argv[i])) exit(1);

	if (Fight == SINGLES) {
		if (Battles == 1) Singles(Core,programs,arena,Size);
		else Multiples(Core,programs,arena,Size,Battles);
	} else if (Fight == CHALLENGE) {
		Challenge(Core,programs,Size,Battles);
	} else if (Fight == ROUNDS) {
		Rounds(Core,programs,Size,Battles);
	} else if (Fight == DEBUG) {
		Debug(Core,programs,Size);
	}

	free(Core);

	exit(0);
}

int ClearResults(programs)
PROGRAM *programs;
{
	PROGRAM *p;

	p = programs;
	while (p) {
		p->twins = p->wins + p->twins;
		p->tlosses = p->losses + p->tlosses;
		p->tties = p->ties + p->tties;

		p->wins = 0;
		p->losses = 0;
		p->ties = 0;

		p = p->next;
	}

	return (1);
}

int PrintResults(program)
PROGRAM *program;
{
	PROGRAM *p;

	p = program;
	printf("W=%d L=%d T=%d\t",p->wins,p->losses,p->ties);
	printf("%4d\t%s\n",3 * p->wins + p->ties,p->filename);

	return (1);
}

int PrintSingles(programs, cycles, Tasks)
PROGRAM *programs;
int cycles, *Tasks;
{
	PROGRAM *p;

	printf("  W/  L/  T%40s%8s\n","Name","Tasks");
	printf("-----------------------------------------");
	printf("------------------\tCycles: %6d\n",cycles);

	p = programs;
	while (p) {
		printf("%3d/%3d/%3d",p->twins,p->tlosses,p->tties);
		printf("%40s%8d\n",p->filename,Tasks[p->index]);
		p = p->next;
	}

	return (1);
}

int PrintTotals(programs)
PROGRAM *programs;
{
	register int i;
	int max, score;
	PROGRAM *p;

	printf("  W/  L/  T%40s%8s\n","Name","Score");
	printf("-------------------------------");
	printf("----------------------------\n");

	p = programs;
	while (p) {
		score = 3 * p->twins + p->ties;
		if (score > max) max = score;
		p = p->next;
	}

	for (i = max; i >= 0; i--) {
		p = programs;
		while (p) {
			score = 3 * p->twins + p->ties;
			if (i != score) {
				p = p->next;
				continue;
			}

			printf("%3d/%3d/%3d",p->twins,p->tlosses,p->tties);
			printf("%40s%8d\n",p->filename,score);
			p = p->next;
		}
	}

	return (1);
}

int Singles(Core, programs, arena, size)
CORE *Core;
PROGRAM *programs;
int arena, size;
{
	int play, Tasks[MAX_PROG];
	TASKS *CLL[MAX_PROG];
	char output[81];
	register int i;
	PROGRAM *p;

	InitCore(Core,size);

	p = programs;
	while (p) {
		if (!LoadProgram(Core,p,size))
			return (0);

		if (Stats) {
			printf("Loading program %s into Core!\n",p->filename);
			PrintCore(Core,p->address,p->length,size);
		}

		if (!InitProgram(p,&CLL[p->index],&Tasks[p->index]))
			return (0);

		p = p->next;
	}

	if (Display) InitDisplay();

	for (i=0, play=arena; (i < Cycles) && (play-1 > 0); i++) {
		if (Display) DisplayCore(Core,size,i,Display,Fade,arena,Tasks);
		EMI88(CLL,Core,64,size,Tasks,arena,i);

		play = 0;
		p = programs;
		while (p) {
			if (Tasks[p->index] > 0) play++;
			p = p->next;
		}
	}

	if (Display) DisplayCore(Core,size,i,0,Fade,arena,Tasks);

	p = programs;
	while (p) {
		if ((Tasks[p->index] > 0) && (play-1 > 0)) {
			if (Display) DisplayString("a tie is declared");
			p->ties++;
		} else if (Tasks[p->index] > 0) {
			sprintf(output," %s is winner",p->filename);
			if (Display) DisplayString(output);
			p->wins++;
		} else if ((play-1 < 0) && (!p->next)) {
			sprintf(output," %s is winner",p->filename);
			if (Display) DisplayString(output);
			p->wins++;
		} else p->losses++;
		p = p->next;
	}

	ClearResults(programs);

	if (Display) EndDisplay();
	else if (Totals) PrintSingles(programs,i,Tasks);

	return (1);
}

int Multiples(Core, programs, arena, size, counter)
CORE *Core;
PROGRAM *programs;
int arena, size, counter;
{
	int play, Tasks[MAX_PROG];
	TASKS *CLL[MAX_PROG];
	register int i;
	PROGRAM *p;

	while (counter-- > 0) {
		InitCore(Core,size);

		p = programs;
		while (p) {
			if (!LoadProgram(Core,p,size))
				return (0);

			if (!InitProgram(p,&CLL[p->index],&Tasks[p->index]))
				return (0);

			p = p->next;
		}

		for (i=0, play=arena; (i < Cycles) && (play-1 > 0); i++) {
			EMI88(CLL,Core,64,size,Tasks,arena,i);

			play = 0;
			p = programs;
			while (p) {
				if (Tasks[p->index] > 0) play++;
				p = p->next;
			}
		}

		p = programs;
		while (p) {
			if ((Tasks[p->index] > 0) && (play-1 > 0)) p->ties++;
			else if (Tasks[p->index] > 0) p->wins++;
			else if ((play-1 < 0) && (!p->next)) p->wins++;
			else p->losses++;
			p = p->next;
		}
	}

	ClearResults(programs);

	if (Totals) PrintTotals(programs);

	return (1);
}

int Challenge(Core, programs, size, counter)
CORE *Core;
PROGRAM *programs;
int size, counter;
{
	int play, Tasks[MAX_PROG];
	TASKS *CLL[MAX_PROG];
	register int c, i;
	PROGRAM *p, *q;

	p = q = programs;
	while (q) {
		c = counter;
		if (p == q) c = counter/2 + counter%2;

		while (c-- > 0) {
			InitCore(Core,size);

			if (!LoadProgram(Core,p,size))
				return (0);

			if (!InitProgram(p,&CLL[0],&Tasks[0]))
				return (0);

			if (!LoadProgram(Core,q,size))
				return (0);

			if (!InitProgram(q,&CLL[1],&Tasks[1]))
				return (0);

			for (i=0, play=1; (i < Cycles) && (play); i++) {
				EMI88(CLL,Core,64,size,Tasks,2,i);
				if (Tasks[0] <= 0) {
					p->losses++;
					play = 0;
				} else if (Tasks[1] <= 0) {
					q->losses++;
					play = 0;
				}
			}

			if (play) p->ties++;
			else if (Tasks[0] > 0) p->wins++;

			if (play) q->ties++;
			else if (Tasks[1] > 0) q->wins++;
		}

		if (Results) {
			if (p == q) {
				printf("Program \"%s\" ",p->filename);
				printf("fights against itself:\n");
			}

			PrintResults(p);
			if (p != q) PrintResults(q);
			printf("\n");
		}

		ClearResults(programs);

		q = q->next;
	}

	if (!Totals) return (1);

	printf("  W/  T/  L%40s%8s\n","Name","Score");
	printf("-----------------------------------------------------------\n");
	printf("%3d/%3d/%3d",p->twins,p->tlosses,p->tties);
	printf("%40s%8d\n",p->filename,3 * p->twins + p->tties);

	return (1);
}

int Rounds(Core, programs, size, counter)
CORE *Core;
PROGRAM *programs;
int size, counter;
{
	int play, Tasks[MAX_PROG];
	TASKS *CLL[MAX_PROG];
	register int c, i;
	PROGRAM *p, *q;

	p = programs;
	while (p) {
		q = p;
		while (q) {
			c = counter;
			if (p == q) c = counter/2 + counter%2;

			while (c-- > 0) {
				InitCore(Core,size);

				if (!LoadProgram(Core,p,size))
					return (0);

				if (!InitProgram(p,&CLL[0],&Tasks[0]))
					return (0);

				if (!LoadProgram(Core,q,size))
					return (0);

				if (!InitProgram(q,&CLL[1],&Tasks[1]))
					return (0);

				for (i=0, play=1; (i < Cycles) && (play); i++) {
					EMI88(CLL,Core,64,size,Tasks,2,i);
					if (Tasks[0] <= 0) {
						p->losses++;
						play = 0;
					} else if (Tasks[1] <= 0) {
						q->losses++;
						play = 0;
					}
				}

				if (play) p->ties++;
				else if (Tasks[0] > 0) p->wins++;

				if (play) q->ties++;
				else if (Tasks[1] > 0) q->wins++;
			}

			if (Results) {
				if (p == q) {
					printf("Program \"%s\" ",p->filename);
					printf("fights against itself:\n");
				}

				PrintResults(p);
				if (p != q) PrintResults(q);
				printf("\n");
			}

			ClearResults(programs);

			q = q->next;
		}

		p = p->next;
	}

	if (Totals) {
		if (Results) printf("\n");
		PrintTotals(programs);
	}

	return (1);
}

int Debug(Core, programs, size)
CORE *Core;
PROGRAM *programs;
int size;
{
	int Tasks[MAX_PROG];
	TASKS *CLL[MAX_PROG];
	register int i;
	PROGRAM *p;

	InitCore(Core,size);

	p = programs;
	if (!LoadProgram(Core,p,size)) return (0);

	if (Stats) {
		printf("Loading program %s into Core!\n",p->filename);
		PrintCore(Core,p->address,p->length,size);
	}

	if (!InitProgram(p,&CLL[0],&Tasks[0]))
		return (0);


	for (i=0; (i < Cycles) && (Tasks[0] > 0); i++) {
		if (i) {
			printf("Press <return> to continue.\n");
			getchar();
		}

		EMI88(CLL,Core,64,size,Tasks,1,i);
		printf("Cycles = %d\t",i+1);
		printf("Tasks = %d\t",Tasks[0]);
		printf("PC = %d\n",CLL[0]->PC);
		PrintCore(Core,p->address-2,p->length+4,size);
	}

	printf("Total Cycles = %d\n",i);

	return (1);
}
