/*
 * ntp_util.c - stuff I didn't have any other place for
 */
#include <stdio.h>
#include <errno.h>
#include <strings.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/time.h>
#ifdef convexos9
#include "/sys/sync/queue.h"
#include "/sys/sync/sema.h"
#endif
#include <net/if.h>
#include <netinet/in.h>

#include "ntp_syslog.h"
#include "ntp_fp.h"
#include "ntp.h"

#ifdef	DOSYNCTODR_SUCKS
#include <sys/resource.h>
#endif

/*
 * This contains odds and ends.  Right now the only thing you'll find
 * in here is the hourly stats printer and some code to support rereading
 * the keys file, but I may eventually put other things in here such as
 * code to do something with the leap bits.
 */

/*
 * Name of the keys file
 */
char *key_file_name;

/*
 * The name of the drift_comp file and the temporary.
 */
char *stats_drift_file;
char *stats_temp_file;

/*
 * Statistics file stuff
 */
FILE *stats_stats_fp;		/* stats file pointer */
char *stats_stats_file;		/* stats file base name */
int stats_stats_age;		/* current age of stats file (hours) */
int stats_stats_gen;		/* current generation number of stats */

FILE *stats_loop_fp;		/* loop stats file pointer */
char *stats_loop_file;		/* loop stats file base name */
int stats_loop_age;		/* current age of loop stats file (hours) */
int stats_loop_gen;		/* current generation number of loop stats */
/*
 * We query the errno to see what kind of error occured
 * when opening the drift file.
 */
extern int errno;

#ifdef DEBUG
extern int debug;
#endif

/*
 * init_util - initialize the utilities
 */
void
init_util()
{
	stats_drift_file = 0;
	stats_temp_file = 0;
	key_file_name = 0;
	stats_stats_fp = NULL;
	stats_stats_file = 0;
	stats_stats_age = 0;
	stats_stats_gen = 0;
}


/*
 * hourly_stats - print some interesting stats
 */
void
hourly_stats()
{
	int fd;
	char *val;
	int vallen;
	extern l_fp last_offset;
	extern s_fp drift_comp;
	extern int time_constant;
	extern char *fptoa();
	extern char *lfptoa();
	extern char *mfptoa();
	char filename[128];
	FILE *fp;

#ifdef DOSYNCTODR_SUCKS
	struct timeval tv;
	int o_prio;

/*
 * Sometimes having a Sun can be a drag.
 *
 * The kernel variable dosynctodr controls whether the system's
 * soft clock is kept in sync with the battery clock. If it
 * is zero, then the soft clock is not synced, and the battery
 * clock is simply left to rot. That means that when the system
 * reboots, the battery clock (which has probably gone wacky)
 * sets the soft clock. That means xntpd starts off with a very
 * confused idea of what time it is. It then takes a large
 * amount of time to figure out just how wacky the battery clock
 * has made things drift, etc, etc. The solution is to make the
 * battery clock sync up to system time. The way to do THAT is
 * to simply set the time of day to the current time of day, but
 * as quickly as possible. This may, or may not be a sensible
 * thing to do.
 */

	o_prio=getpriority(PRIO_PROCESS,0); /* Save setting */
	if (setpriority(PRIO_PROCESS,0,-20) != 0) /* overdrive */
	{
		syslog(LOG_ERR, "can't elevate priority: %m");
		goto skip;
	}
	gettimeofday(&tv,(struct timezone *)NULL);
	if (settimeofday(&tv,(struct timezone *)NULL) != 0)
	{
		syslog(LOG_ERR, "can't sync battery time: %m");
	}
	setpriority(PRIO_PROCESS,0,o_prio); /* downshift */

skip:
#endif

	syslog(LOG_INFO, "offset %s freq %s comp %d",
		lfptoa(&last_offset, 6), fptoa(drift_comp, 5), time_constant);
	
	if (stats_drift_file != 0) {
		fd = open(stats_temp_file, O_WRONLY|O_TRUNC|O_CREAT, 0644);
		if (fd == -1) {
			syslog(LOG_ERR, "can't open %s: %m", stats_temp_file);
			return;
		}

		val = fptoa(drift_comp, 5);
		vallen = strlen(val);
		/*
		 * Hack here.  Turn the trailing \0 into a \n and write it.
		 */
		val[vallen] = '\n';
		if (write(fd, val, vallen+1) == -1) {
			syslog(LOG_ERR, "write to %s failed: %m",
			    stats_temp_file);
			(void) close(fd);
			(void) unlink(stats_temp_file);
		} else {
			(void) close(fd);
			/* atomic */
			(void) rename(stats_temp_file, stats_drift_file);
		}
	}
	if (stats_stats_fp != NULL) {
		(void)fclose(stats_stats_fp);
		stats_stats_age++;
		if (stats_stats_age >= STATS_MAXAGE) {
			stats_stats_age = 0;
			while (1) {
				stats_stats_gen++;
				sprintf(filename, "%s.%d",
					stats_stats_file, stats_stats_gen);
				if ((fp = fopen(filename, "r")) == NULL) break;
				(void)fclose(fp);
			}
			rename(stats_stats_file, filename);
			syslog(LOG_INFO, "NTP statistics file: %s",
				filename);
		}
		if ((stats_stats_fp = fopen(stats_stats_file, "a")) == NULL) {
                        if (errno != ENOENT)
                                syslog(LOG_ERR, "can't open %s: %m",
                                        stats_stats_file);
		}
	}
}


/*
 * stats_config - configure the stats operation
 */
void
stats_config(item, value)
	int item;
	char *value;	/* only one type so far */
{
	register char *cp;
	FILE *fp;
	int len;
	char buf[128];
	l_fp old_drift;
	extern char *emalloc();
	extern void loop_config();
	extern char *lfptoa();

	switch(item) {
	case STATS_FREQ_FILE:
		if (stats_drift_file != 0) {
			(void) free(stats_drift_file);
			(void) free(stats_temp_file);
			stats_drift_file = 0;
			stats_temp_file = 0;
		}

		if (value == 0 || (len = strlen(value)) == 0)
			break;

		stats_drift_file = emalloc((u_int)(len + 1));
		stats_temp_file = emalloc((u_int)(len + sizeof(".TEMP")));
		bcopy(value, stats_drift_file, len+1);
		bcopy(value, stats_temp_file, len);
		bcopy(".TEMP", stats_temp_file + len, sizeof(".TEMP"));
#ifdef DEBUG
		if (debug > 1) {
			printf("stats drift file %s\n", stats_drift_file);
			printf("stats temp file %s\n", stats_temp_file);
		}
#endif

		if ((fp = fopen(stats_drift_file, "r")) == NULL) {
			if (errno != ENOENT)
				syslog(LOG_ERR, "can't open %s: %m",
				    stats_drift_file);
			break;
		}

		if (fgets(buf, sizeof buf, fp) == NULL) {
			syslog(LOG_ERR, "can't read %s: %m",
			    stats_drift_file);
			(void) fclose(fp);
			break;
		}

		(void) fclose(fp);

		/*
		 * We allow leading spaces, then the number.  Terminate
		 * at any trailing space or string terminator.
		 */
		cp = buf;
		while (isspace(*cp))
			cp++;
		while (*cp != '\0' && !isspace(*cp))
			cp++;
		*cp = '\0';

		if (!atolfp(buf, &old_drift)) {
			syslog(LOG_ERR, "drift value %s invalid", buf);
			break;
		}

		/*
		 * Finally!  Give value to the loop filter.
		 */
#ifdef DEBUG
		if (debug > 1) {
			printf("loop_config finds old drift of %s\n",
			    lfptoa(&old_drift, 9));
		}
#endif
		loop_config(LOOP_DRIFTCOMP, &old_drift);
		break;
	
	case STATS_STAT_FILE:
		stats_stats_age = 0;
		if (stats_stats_fp != NULL) {
			(void) fclose(stats_stats_fp);
			stats_stats_fp = NULL;
		}
		if (stats_stats_file != 0) {
			(void) free(stats_stats_file);
                        stats_stats_file = 0;
		}
		if (value == 0 || (len = strlen(value)) == 0)
			break;
		stats_stats_file = emalloc((u_int)(len + 1));
		bcopy(value, stats_stats_file, len + 1);
#ifdef DEBUG
		if (debug > 1) {
			printf("statistics file %s\n", stats_stats_file);
		}
#endif
		if ((stats_stats_fp = fopen(stats_stats_file, "a")) == NULL) {
			if (errno != ENOENT)
				syslog(LOG_ERR, "can't open %s: %m",
					stats_stats_file);
		}
		break;

	  case STATS_LOOP_FILE:
		stats_loop_age = 0;
		if (stats_loop_fp != NULL) {
			(void) fclose(stats_loop_fp);
			stats_loop_fp = NULL;
		}
		if (stats_loop_file != 0) {
			(void) free(stats_loop_file);
                        stats_loop_file = 0;
		}
		if (value == 0 || (len = strlen(value)) == 0)
			break;
		stats_loop_file = emalloc((u_int)(len + 1));
		bcopy(value, stats_loop_file, len + 1);
#ifdef DEBUG
		if (debug > 1) {
			printf("loop statistics file %s\n", stats_loop_file);
		}
#endif
		if ((stats_loop_fp = fopen(stats_loop_file, "a")) == NULL) {
			if (errno != ENOENT)
				syslog(LOG_ERR, "can't open %s: %m",
					stats_loop_file);
		}
		break;

	case STATS_PID_FILE:
		if ((fp = fopen(value, "w")) == NULL) {
		    syslog(LOG_ERR, "Can't open %s: %m", value);
		    break;
		}
		fprintf(fp, "%d", getpid());
		fclose(fp);;
		break;

	default:
		/* oh well */
		break;
	}
}

/*
 * record_stats - write statistics to statistics file
 *
 * file format:
 * day (mjd)
 * time (s past midnight)
 * peer (ip address)
 * offset (s)
 * delay (s)
 * dispersion (s)
 */
void
record_stats(addr, status, offset, delay, dispersion)
	struct sockaddr_in *addr;
	u_short status;
	l_fp *offset;
	s_fp delay;
	u_fp dispersion;
{
	struct timeval tv;
	u_long day, sec, msec;
	char *ntoa();
	char *lfptoa();
	char *fptoa();
	char *ufptoa();

	gettimeofday(&tv, (struct timezone *)NULL);
	day = (u_long)tv.tv_sec / 86400 + MJD_1970;
	sec = (u_long)tv.tv_sec % 86400;
	msec = (u_long)tv.tv_usec / 1000;

	if (stats_stats_fp != NULL) {
		fprintf(stats_stats_fp, "%lu %lu.%03lu %s %x %s %s %s\n",
			day, sec, msec, ntoa(addr), status, lfptoa(offset, 6),
			fptoa(delay, 5), ufptoa(dispersion, 5));
		fflush(stats_stats_fp);
	}
}
/*
 * record_loop_stats - write loop filter statistics to file
 *
 * file format:
 * day (mjd)
 * time (s past midnight)
 * offset (s)
 * drift_comp
 * time_constant
 */
void
record_loop_stats(offset, drift_comp, time_constant)
	l_fp *offset;
        s_fp *drift_comp;
        int  time_constant;
{
	struct timeval tv;
	u_long day, sec, msec;
	char *lfptoa();
	char *fptoa();
	char *ufptoa();

	gettimeofday(&tv, (struct timezone *)NULL);
	day = (u_long)tv.tv_sec / 86400 + MJD_1970;
	sec = (u_long)tv.tv_sec % 86400;
	msec = (u_long)tv.tv_usec / 1000;

	if (stats_loop_fp != NULL) {
		fprintf(stats_loop_fp, "%lu %lu.%03lu %s %s %d\n",
			day, sec, msec, lfptoa(offset, 8),
			fptoa(*drift_comp, 7), time_constant);
		fflush(stats_loop_fp);
	}
}

/*
 * getauthkeys - read the authentication keys from the specified file
 */
void
getauthkeys(keyfile)
	char *keyfile;
{
	int len;
	extern char *emalloc();

	len = strlen(keyfile);
	if (len == 0)
		return;
	
	if (key_file_name != 0) {
		if (len > strlen(key_file_name)) {
			(void) free(key_file_name);
			key_file_name = 0;
		}
	}

	if (key_file_name == 0)
		key_file_name = emalloc((u_int)(len + 1));
	
	bcopy(keyfile, key_file_name, len+1);

	authreadkeys(key_file_name);
}


/*
 * rereadkeys - read the authentication key file over again.
 */
void
rereadkeys()
{
	if (key_file_name != 0)
		authreadkeys(key_file_name);
}
