/*
 * baseClasses.c
 * Handle base classes.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#define	DBG(s)

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "gtypes.h"
#include "classMethod.h"
#include "baseClasses.h"
#include "lookup.h"
#include "md.h"

classes* StringClass;
classes* ClassClass;
classes* ObjectClass;
strpair* initpair;
strpair* finalpair;
classes* classInitHead;
int classInitLevel = 2;

#define	INIT		"<clinit>"
#define	INITSIG		"()V"
#define	FINAL		"finalize"
#define	FINALSIG	"()V"

/*
 * Initialise the machine.
 */
void
initialise(void)
{
	/* Setup CLASSPATH */
	initClasspath();

	/* Create the initialise launch pair */
	initpair = addStringPair(INIT, INITSIG);

	/* Create the finalize launch pair */
	finalpair = addStringPair(FINAL, FINALSIG);

	/* Read in base classes */
	initBaseClasses();

	/* Setup exceptions */
	initExceptions();

	/* Init native support */
	initNative();

	/* Init thread support */
	initThreads();
}

/*
 * We need to use certain classes in the internal machine so we better
 * get them in now in a known way so we can refer back to them.
 * Currently we need java/lang/Object, java/lang/Class and java/lang/String.
 */
void
initBaseClasses(void)
{
	/* Start with the simple types. */
	initTypes();

	/* Read in object */
	ObjectClass = lookupClass(addString(OBJECTCLASS));
	if (ObjectClass == 0) {
		fprintf(stderr, "Failed to find class %s ... aborting.\n", OBJECTCLASS);
		exit(1);
	}

	/* Read in class */
	ClassClass = lookupClass(addString(CLASSCLASS));
	if (ClassClass == 0) {
		fprintf(stderr, "Failed to find class %s ... aborting.\n", CLASSCLASS);
		exit(1);
	}

	/* Fixup mtable because it couldn't be made for the first classes */
	ClassClass->head.mtable = ClassClass->mtable;
	ObjectClass->head.mtable = ClassClass->mtable;

	/* Read in strings */
	StringClass = lookupClass(addString(STRINGCLASS));
	if (StringClass == 0) {
		fprintf(stderr, "Failed to find class %s ... aborting.\n", STRINGCLASS);
		exit(1);
	}

	initClasses();
}

/*
 * Creata a string in the string cache.
 */
char*
addString(char* s)
{
	strconst* m;

	m = malloc(sizeof(strconst) + strlen(s) + 1);
	strcpy(m->data, s);
	return (addStringConstant(m));
}

/*
 * Create a string pair in the string pair cache.
 */
strpair*
addStringPair(char* s1, char* s2)
{
	strpair* pair;

	pair = addStringConstantPair(addString(s1), addString(s2));
	assert(pair != 0);

	return (pair);
}

/*
 * We translate a CONSTANT_Chararray to a CONSTANT_String on demand.
 * The strings in the .class file are not held as real objects, so
 * we convert them here.
 */
void
makeStringObject(int idx, constants* pool)
{
	stringClass* obj;

	assert(pool->tags[idx] == CONSTANT_Chararray);

	obj = getString(STRING_DATA2BASE(pool->data[STRING_NAME(idx, pool)]));

	/* Install new object */
	pool->tags[idx] = CONSTANT_String;
	pool->data[idx] = (int)obj;
}

/*
 * Convert a strconst to a String object.
 */
stringClass*
getString(strconst* str)
{
	stringClass* obj;

	if (str->string == 0) {
		obj = (stringClass*)alloc_object(StringClass, true);
		assert(obj != 0);

		obj->value = &str->obj;
		obj->offset = 0;				/* ??? */
		obj->count = obj->value->size;			/* ??? */
		str->string = obj;
	}
	return (str->string);
}

/*
 * Initialise classes.
 */
void
initClasses(void)
{
	classes* class;
	classes* nclass;
	void* func;

	classInitLevel++;

	class = classInitHead;
	classInitHead = 0;
	while (class != 0) {
		class->state = CSTATE_OK;

		func = findMethod(class, initpair);
		assert(func != 0);
DBG(		printf("Initialising %s static %d\n", class->name, class->sfsize); fflush(stdout); )
		CALL_KAFFE_FUNCTION_VARARGS(func, 0, 0, 0);
		nclass = class->nextInit;
		class->nextInit = 0;
		class->prevInit = 0;
		class = nclass;
	}

	classInitLevel--;
	/* We should never drop below our initial level of two */
	assert(classInitLevel >= 2);
}
