/*********************************************************************/
/*  bibView: Administration of BibTeX-Databases                      */
/*           (Verwaltung von BibTeX-Literaturdatenbanken)            */
/*                                                                   */
/*  Module:  bibview.c                                               */
/*                                                                   */
/*             - Main Program                                        */
/*             - Signal Handling                                     */
/*                                                                   */
/*  Author:  Holger Martin,  martinh@informatik.tu-muenchen.de       */
/*           Peter M. Urban, urban@informatik.tu-muenchen.de         */
/*                                                                   */
/*  History:                                                         */
/*    11.22.91  PMU  created                                         */
/*    05.26.92       Version 1.0 released                            */
/*                                                                   */
/*  Copyright 1992 TU MUENCHEN				 	     */
/*    See ./Copyright for complete rights and liability information. */
/*                                                                   */
/*********************************************************************/


#include <stdio.h>
#include <unistd.h>
#include <sys/param.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xos.h>
#include "bibview.h"
#include "bitmaps/bv_ico64"
#include "bitmaps/bv_lst64"
#include "bitmaps/bv_mac64"
#include "bitmaps/bv_crd64"
#include "bitmaps/bv_hlp64"
#include "bitmaps/bv_ann64"
#include "bitmaps/chkmrk16"
#include "bitmaps/error32"
#include "bitmaps/warn32"
#include "bitmaps/notice32"
#include "bitmaps/quest32"
#ifdef GERMAN
#include "bitmaps/proggerm.xbm"
#else
#include "bitmaps/proginfo.xbm"
#endif


/* external global variables */
/* ------------------------- */
extern Widget desktop;
extern char *latexHeader, *latexFooter;
extern int latexHeaderLen, latexFooterLen;
extern char *style_file;
extern int cont_indent, fld_indent, newline_indent, max_linelen;
extern char *actual_path;
extern UserDefFld userDefFlds[];
extern PredefLists predefLst;
extern listEntry *list_layout;



/* exported global variables */
/* ------------------------- */
XtAppContext app_context;
Widget topLevel;
Pixmap bvIconPixmap;
String bvIconName;
Pixmap lstIconPixmap;
Pixmap macIconPixmap;
Pixmap crdIconPixmap;
Pixmap helpIconPixmap;
Pixmap annoteIconPixmap;
Pixmap chkmarkPixmap;
Pixmap errorPixmap;
Pixmap warnPixmap;
Pixmap noticePixmap;
Pixmap questPixmap;
Pixmap progInfoPixmap;

int sortedby;


/* names for card types , the order must be the same as */
/* for the type CardType in bibview.h                   */
String cardNames[MAX_BIBTEX_TYPES+1] = {
   "article",
   "book",
   "booklet",
   "conference",
   "inbook",
   "incollection",
   "inproceedings",
   "manual",
   "mastersthesis",
   "misc",
   "phdthesis",
   "proceedings",
   "techreport",
   "unpublished",
   NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   "userdefined"
};

/* names for fields , the order must be the same as for  */
/* the type FieldName in bibview.h                       */
String fieldNames[MAX_FIELDS] = {
   "crossref",
   "key",
   "author",
   "title",
   "booktitle",
   "institution",
   "journal",
   "organization",
   "chapter",
   "year",
   "month",
   "editor",
   "publisher",
   "address",
   "school",
   "edition",
   "howpublished",
   "volume",
   "number",
   "pages",
   "series",
   "type",
   "note",
   NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL
};

char sort_field[MAX_FIELDS];
char is_search_field[MAX_FIELDS];

char standardfields [MAX_BIBTEX_TYPES+1][MAX_FIELDS];
char requiredfields [MAX_BIBTEX_TYPES+1][MAX_FIELDS];

int max_fields;
int max_bibtex_types;



/* local global variables */
/* ---------------------- */

/* fallback resources */
static char *fallback_resources[] = {
#ifdef GERMAN
   #include "fb_ger.h"
#else
   #include "fb_res.h"
#endif
   NULL
};


static char bibfile[MAXPATHLEN+1];

OptionsRec options;

#define Offset(field)  XtOffsetOf(OptionsRec, field)
XtResource resources[] = {
   { "quiet", "Quiet", XtRBoolean, sizeof(Boolean),
        Offset(quiet), XtRString, "False" },
   { "iconsOnDesk", "IconsOnDesk", XtRBoolean, sizeof(Boolean),
	Offset(iconsOnDesk), XtRString, (XtPointer)"False" },
   { "makeBackup", "MakeBackup", XtRBoolean, sizeof(Boolean),
	Offset(makeBackup), XtRString, (XtPointer)"True" },
   { "autoCheck", "AutoCheck", XtRBoolean, sizeof(Boolean),
	Offset(autoCheck), XtRString, (XtPointer)"True" },
   { "reqFields", "ReqFields", XtRBoolean, sizeof(Boolean),
	Offset(reqFields), XtRString, (XtPointer)"False" },
   { "ignoreCase", "IgnoreCase", XtRBoolean, sizeof(Boolean),
	Offset(ignoreCase), XtRString, (XtPointer)"True" },
   { "printAsBib", "PrintAsBib", XtRBoolean, sizeof(Boolean),
	Offset(printAsBib), XtRString, (XtPointer)"True" },
   { "displayErrWin", "DisplayErrWin", XtRBoolean, sizeof(Boolean),
	Offset(displayErrWin), XtRString, (XtPointer)"True" },
   { "focus", "Focus", XtRBoolean, sizeof(Boolean),
	Offset(focus), XtRString, (XtPointer)"True" },
   { "bibfile", "Bibfile", XtRString, sizeof(String),
	Offset(bibfile), XtRString, (XtPointer)NULL },
   { "rcfile", "Rcfile", XtRString, sizeof(String),
	Offset(rcfile), XtRString, (XtPointer)NULL },
   { "bibdir", "Bibdir", XtRString, sizeof(String),
	Offset(bibdir), XtRString, (XtPointer)NULL },
   { "focusColor", "FocusColor", XtRPixel, sizeof(Pixel),
	Offset(focusColor), XtRString, XtDefaultBackground },
};
#undef Offset

static XrmOptionDescRec optionDesc[] = {
   { "-quiet",       "*quiet",       XrmoptionNoArg, (XtPointer)"on" },
   { "+quiet",       "*quiet",       XrmoptionNoArg, (XtPointer)"off" },
   { "-iconsOnDesk", "*iconsOnDesk", XrmoptionNoArg, (XtPointer)"on" },
   { "+iconsOnDesk", "*iconsOnDesk", XrmoptionNoArg, (XtPointer)"off" },
   { "-makeBackup",  "*makeBackup",  XrmoptionNoArg, (XtPointer)"on" },
   { "+makeBackup",  "*makeBackup",  XrmoptionNoArg, (XtPointer)"off" },
   { "-autoCheck",   "*autoCheck",   XrmoptionNoArg, (XtPointer)"on" },
   { "+autoCheck",   "*autoCheck",   XrmoptionNoArg, (XtPointer)"off" },
   { "-reqFields",   "*reqFields",   XrmoptionNoArg, (XtPointer)"on" },
   { "+reqFields",   "*reqFields",   XrmoptionNoArg, (XtPointer)"off" },
   { "-ignoreCase",  "*ignoreCase",  XrmoptionNoArg, (XtPointer)"on" },
   { "+ignoreCase",  "*ignoreCase",  XrmoptionNoArg, (XtPointer)"off" },
   { "-printAsBib",  "*printAsBib",  XrmoptionNoArg, (XtPointer)"on" },
   { "+printAsBib",  "*printAsBib",  XrmoptionNoArg, (XtPointer)"off" },
   { "-displayErrWin","*displayErrWin",  XrmoptionNoArg, (XtPointer)"on" },
   { "+displayErrWin","*displayErrWin",  XrmoptionNoArg, (XtPointer)"off" },
   { "-focus",        "*focus",      XrmoptionNoArg, (XtPointer)"on" },
   { "+focus",        "*focus",      XrmoptionNoArg, (XtPointer)"off" },
   { "-rcfile",      "*rcfile",      XrmoptionSepArg, (XtPointer)NULL },
   { "-bibdir",      "*bibdir",      XrmoptionSepArg, (XtPointer)NULL },
   { "-file",        "*bibfile",     XrmoptionSepArg, (XtPointer)NULL } 
};


/* local function prototypes */
static void printHelp (int argc, char **argv);
static void processCommandlineOptions (void);
static Boolean readBibfile (XtPointer clientData);
static void makeIconPixmaps (void);
static void installErrorHandlers (void);
static XtErrorHandler Xt_warningHandler (String warn);
static XtErrorHandler Xt_errorHandler (String error);
static int X_errorHandler (Display *dsp, XErrorEvent *event);
static int IO_errorHandler (Display *dsp);
static void signalHandler (int signal, int code);
static void initializeFields (void);



/*********************************************************************/
/* MAIN:                                                             */
/*********************************************************************/
int
main(int argc, char *argv[])
{
static char iconname[MAX_ICONNAMELEN+1];
static char *p, path[MAX_FILEPATHLEN+1];
int i;

   topLevel = XtVaAppInitialize(&app_context, /* Appl context */
                 "BibView",                   /* Appl class */
                 optionDesc,                  /* cmdline options */ 
		 XtNumber(optionDesc),
                 &argc, argv,     /* command line args */
                 (String *)fallback_resources,/* fallback resources */
		 NULL);                       /* end varargs list */
   
   /* process commandline options */
   XtGetApplicationResources(topLevel,
                             (XtPointer) &options,
                             resources, XtNumber(resources),
                             (Arg *)NULL, 0);
   if (argc > 1)
      printHelp(argc, argv);

   style_file = XtMalloc(MAXPATHLEN+1); 
   strcpy(style_file, "alpha");
   
   sortedby = SORT_UNSRT;
   cont_indent = CONTINDENT;
   fld_indent = FLDINDENT;
   newline_indent = NEWLINEINDENT;
   max_linelen = MAX_LINELEN;
   actual_path = XtMalloc(MAXPATHLEN+1); 

   for (i=0; i<MAX_BIBTEX_TYPES+1; i++)
      userDefFlds[i] = NULL;

   predefLst.allfields = NULL;
   predefLst.bibtype = NULL;
   predefLst.mainkey = NULL;
   for (i=0; i<MAX_FIELDS; i++)
      predefLst.field[i] = NULL;

   initializeFields();

#ifdef SYSV
      (void) getcwd(actual_path, MAX_FILEPATHLEN);
#else
      (void) getwd(actual_path);
#endif
   
   
   fieldNames[nannote] = XtCalloc(7, sizeof(char));
   strcpy(fieldNames[nannote], "annote");


   /* read rc options file */
   if (options.rcfile != NULL)
      rcfReadCfgFile(options.rcfile);
   else if (access(DEFAULT_RCFILE, R_OK) != -1) {
      /* read file in current directory */
      rcfReadCfgFile(DEFAULT_RCFILE);
   }
   else {
      /* look for file in $HOME-Directory */
      if ((p = (char *)getenv("HOME")) != NULL) {
         strcpy(path, p);
	 strcat(path, "/");
	 strcat(path, DEFAULT_RCFILE);
	 if (access(path, R_OK) != -1)
	    rcfReadCfgFile(path);
      }
   }

   processCommandlineOptions();

   if (options.bibdir != NULL) 
      procDefaultDirLine(options.bibdir);

   /* create icon pixmaps, initialize main icon for bibview */
   makeIconPixmaps();
   sprintf(iconname, "%s %d.%d", PROGNAME, VERSION, PATCHLEVEL);
   bvIconName = iconname;
   XtVaSetValues(topLevel, 
		 XtNiconPixmap, bvIconPixmap,
		 XtNtitle, bvIconName,
		 XtNiconName, bvIconName, NULL);

   /* instance main window */
   gwmMainWin();

   XtRealizeWidget(topLevel);
   
   /* load a file from command line */ 
   if (options.bibfile != NULL)
     {CollapsePath(options.bibfile, bibfile);
      XtAppAddWorkProc(app_context, readBibfile, bibfile);
      }

   rcfSetPrintMode(FALSE);
   rcfReadOptions(FALSE);
   XtAppMainLoop(app_context); 
}



/*********************************************************************/
/* LOCAL FUNCTIONS                                                   */
/*********************************************************************/


/*********************************************************************/
/* printHelp:                                                        */
/*   Process excess options and show help                            */
/*********************************************************************/
static void 
printHelp (int argc, char **argv)
{
static char *help[] = {
 "\n",
 "bibview: Usage\n",
 "         bibview understands all standard Xt command-line options.\n",
 "\n",
 "Additional options are as follows:\n",
 "   Option        Valid Range     Function\n",
 "   -quiet        none            Turn off beep on error\n",
 "   -iconsOnDesk  none            Keep icons on bibview desktop\n",
 "   -makeBackup   none            Save files to backup on save\n",
 "   -autoCheck    none            Turn on/off consistency checking\n",
 "   -reqFields    none            Disallow empty required fields\n",
 "   -ignoreCase   none            Ignore case in search\n",
 "   -focus        none            Use <tab> and <esc> key in card window\n",
 "   +quiet        none            Turn on/off beep on error\n",
 "   +iconsOnDesk  none            Don't keep icons on bibview desktop\n",
 "   +makeBackup   none            Don't save files to backup on save\n",
 "   +autoCheck    none            Turn off consistency checking\n",
 "   +reqFields    none            Allow empty required fields\n",
 "   +ignoreCase   none            Use case in search\n",
 "   +focus        none            Use mouse in card window\n",
 "   -file         BibTeX File     Load BibTeX-File on startup\n",
 "   -rcfile       bibview rcFile  Use specific configuration file\n",
 "   -help         none            Show this list\n",
 "\n",
 NULL
};
char **p;
int  i;

   /* print help message */
   p = help;
   while (*p) {
      fprintf(stderr, *p);
      p++;
   }

   /* show illegal arguments */
   if (strcmp(argv[1], "-help")) {
      fprintf(stderr, "bibview: command line option unknown:\n");
      for (i=1; i < argc; i++)
         fprintf(stderr, " option: %s\n", argv[i]);
   }
   else {
      /* just show help and exit */
      exit(0);
   }
}


/*********************************************************************/
/* processCommandlineOptions:                                        */
/*   Process options not controlling widgets                         */
/*********************************************************************/
static void 
processCommandlineOptions (void)
{
   optionsStatus[OPT_BEEP_ON_ERROR]   = !options.quiet;
   optionsStatus[OPT_BACKUP_ON_SAVE]  = options.makeBackup;
   optionsStatus[OPT_ICON_ON_DESKTOP] = options.iconsOnDesk;
   optionsStatus[OPT_AUTO_CHECK_BIB]  = options.autoCheck;
   optionsStatus[OPT_REQUIRED_FIELDS] = options.reqFields;
   optionsStatus[OPT_IGNORE_CASE]     = options.ignoreCase;
   optionsStatus[OPT_PRINT_AS_BIB]    = options.printAsBib;
   optionsStatus[OPT_DISPLAY_ERRWIN]  = options.displayErrWin;
   optionsStatus[OPT_FOCUS]           = options.focus;
}


/*********************************************************************/
/* readBibfile:                                                      */
/*   Read a bibfile with name given on command line                  */
/*********************************************************************/
static Boolean
readBibfile (XtPointer clientData)
{
String filename = (String)clientData;
BibPtr bp;
Errcode status;

   /* check whether desktop positioned already */
   if (XtWindow(desktop) == (Window) NULL)
      return (False);

   /* desktop is there, open file and window */
   if ((status = glbNewBibListEl(&bp)) != OK) {
      guwError(status);
      return (True);
   }
   strcpy(bp->filename, filename);
   if ((filename[0]!='/') && (filename[0]!='~')){
#ifdef SYSV
      (void) getcwd(bp->filepath, MAX_FILEPATHLEN);
#else
      (void) getwd(bp->filepath);
#endif
      strcat(bp->filepath, "/");
      strcat(bp->filepath, filename);
      CollapsePath(bp->filepath, filename);
      strcpy(bp->filepath, filename);
      }
   else
      CollapsePath(filename, bp->filepath);
   bp->mw=NULL;
   bp->ew=NULL;
   bp->lw=NULL;
   bp->sortedby=sortedby;
   glbReadFileOpenBib(bp);
   return(True);

}


/*********************************************************************/
/* initializeFields:                                                 */
/*   initialize standardfields and requiredfields                    */
/*********************************************************************/
static void 
initializeFields (void)
{
int i, j;
   
   
   max_fields = 24;
   max_bibtex_types = 14;
   
   list_layout = (listEntry *) XtCalloc(1, sizeof(listEntry));
   list_layout->number = 0;
   list_layout->field[list_layout->number] = nauthor;
   list_layout->width[(list_layout->number)++] = 30;
   list_layout->field[list_layout->number] = ntitle; 
   list_layout->width[(list_layout->number)++] = 50;
   list_layout->field[list_layout->number] = nyear; 
   list_layout->width[(list_layout->number)++] = 5;
   list_layout->field[list_layout->number] = -2;  /* BibTeX type */
   list_layout->width[(list_layout->number)++] = 10;
   list_layout->field[list_layout->number] = -1;  /* mainkey */
   list_layout->width[(list_layout->number)++] = 10;

   for (i=0; i< MAX_BIBTEX_TYPES+1; i++){
      for (j=0; j< MAX_FIELDS; j++){
	 standardfields[i][j] = '0';
	 requiredfields[i][j] = '0';
      }
   }
   for (i=0; i< max_bibtex_types; i++){
      for (j=0; j<max_fields; j++){
         standardfields[i][nmonth] = '1';
         standardfields[i][nyear] = '1';
         standardfields[i][nnote] = '1';
         standardfields[i][nkey] = '1';
         standardfields[i][nannote] = '1';
         standardfields[i][ncrossref] = '1';
      }
   }

   for (i=0; i< MAX_FIELDS; i++){
      sort_field[i] = '0';
      is_search_field[i] = '0';
      }
   
   for (i=0; i< max_fields; i++){
      sort_field[i] = '1';
      is_search_field[i] = '1';
      }
   
   sort_field[nauthor] = 'n';
   sort_field[neditor] = 'n';

/* article */
   standardfields[article][nauthor] =
   requiredfields[article][nauthor] = '1';
   standardfields[article][ntitle] =
   requiredfields[article][ntitle] = '1';
   standardfields[article][njournal] =
   requiredfields[article][njournal] = '1';
   requiredfields[article][nyear] = '1';

   standardfields[article][nvolume] = '1';
   standardfields[article][nnumber] = '1';
   standardfields[article][npages] = '1';

/* book */
   standardfields[book][nauthor] =
   requiredfields[book][nauthor] = '1';
   standardfields[book][neditor] =
   requiredfields[book][neditor] = '1';
   standardfields[book][ntitle] =
   requiredfields[book][ntitle] = '1';
   standardfields[book][npublisher] =
   requiredfields[book][npublisher] = '1';
   requiredfields[book][nyear] = '1';

   standardfields[book][nvolume] = '1';
   standardfields[book][nseries] = '1';
   standardfields[book][naddress] = '1';
   standardfields[book][nedition] = '1';

/* booklet */
   standardfields[booklet][ntitle] =
   requiredfields[booklet][ntitle] = '1';
   standardfields[booklet][nauthor] = '1';
   standardfields[booklet][nhowpublished] = '1';
   standardfields[booklet][naddress] = '1';

/* conference */
   standardfields[conference][nauthor] =
   requiredfields[conference][nauthor] = '1';
   standardfields[conference][ntitle] =
   requiredfields[conference][ntitle] = '1';
   standardfields[conference][nbooktitle] =
   requiredfields[conference][nbooktitle] = '1';
   requiredfields[conference][nyear] = '1';

   standardfields[conference][neditor] = '1';
   standardfields[conference][npages] = '1';
   standardfields[conference][norganization] = '1';
   standardfields[conference][npublisher] = '1';
   standardfields[conference][naddress] = '1';

/* inbook */
   standardfields[inbook][nauthor] =
   requiredfields[inbook][nauthor] = '1';
   standardfields[inbook][neditor] =
   requiredfields[inbook][neditor] = '1';
   standardfields[inbook][ntitle] =
   requiredfields[inbook][ntitle] = '1';
   standardfields[inbook][nchapter] =
   requiredfields[inbook][nchapter] = '1';
   standardfields[inbook][npages] =
   requiredfields[inbook][npages] = '1';
   standardfields[inbook][npublisher] =
   requiredfields[inbook][npublisher] = '1';
   requiredfields[inbook][nyear] = '1';

   standardfields[inbook][nvolume] = '1';
   standardfields[inbook][nseries] = '1';
   standardfields[inbook][naddress] = '1';
   standardfields[inbook][nedition] = '1';

/* incollection */
   standardfields[incollection][nauthor] =
   requiredfields[incollection][nauthor] = '1';
   standardfields[incollection][ntitle] =
   requiredfields[incollection][ntitle] = '1';
   standardfields[incollection][nbooktitle] =
   requiredfields[incollection][nbooktitle] = '1';
   standardfields[incollection][npublisher] =
   requiredfields[incollection][npublisher] = '1';
   requiredfields[incollection][nyear] = '1';

   standardfields[incollection][neditor] = '1';
   standardfields[incollection][nchapter] = '1';
   standardfields[incollection][npages] = '1';
   standardfields[incollection][naddress] = '1';

/* inproceedings */
   standardfields[inproceedings][nauthor] =
   requiredfields[inproceedings][nauthor] = '1';
   standardfields[inproceedings][ntitle] =
   requiredfields[inproceedings][ntitle] = '1';
   standardfields[inproceedings][nbooktitle] =
   requiredfields[inproceedings][nbooktitle] = '1';
   requiredfields[inproceedings][nyear] = '1';

   standardfields[inproceedings][neditor] = '1';
   standardfields[inproceedings][npages] = '1';
   standardfields[inproceedings][norganization] = '1';
   standardfields[inproceedings][npublisher] = '1';
   standardfields[inproceedings][naddress] = '1';

/* manual */
   standardfields[manual][ntitle] =
   requiredfields[manual][ntitle] = '1';

   standardfields[manual][nauthor] = '1';
   standardfields[manual][norganization] = '1';
   standardfields[manual][naddress] = '1';
   standardfields[manual][nedition] = '1';

/* mastersthesis */
   standardfields[mastersthesis][nauthor] =
   requiredfields[mastersthesis][nauthor] = '1';
   standardfields[mastersthesis][ntitle] =
   requiredfields[mastersthesis][ntitle] = '1';
   standardfields[mastersthesis][nschool] =
   requiredfields[mastersthesis][nschool] = '1';
   requiredfields[mastersthesis][nyear] = '1';

   standardfields[mastersthesis][naddress] = '1';

/* misc */
   standardfields[misc][nauthor] = '1';
   standardfields[misc][ntitle] = '1';
   standardfields[misc][nhowpublished] = '1';
      
/* phdthesis */
   standardfields[phdthesis][nauthor] =
   requiredfields[phdthesis][nauthor] = '1';
   standardfields[phdthesis][ntitle] =
   requiredfields[phdthesis][ntitle] = '1';
   standardfields[phdthesis][nschool] =
   requiredfields[phdthesis][nschool] = '1';
   requiredfields[phdthesis][nyear] = '1';

   standardfields[phdthesis][naddress] = '1';
      
/* proceedings */
   standardfields[proceedings][ntitle] =
   requiredfields[proceedings][ntitle] = '1';
   requiredfields[proceedings][nyear] = '1';

   standardfields[proceedings][neditor] = '1';
   standardfields[proceedings][npublisher] = '1';
   standardfields[proceedings][norganization] = '1';
   standardfields[proceedings][naddress] = '1';
      
/* techreport */
   standardfields[techreport][nauthor] =
   requiredfields[techreport][nauthor] = '1';
   standardfields[techreport][ntitle] =
   requiredfields[techreport][ntitle] = '1';
   standardfields[techreport][ninstitution] =
   requiredfields[techreport][ninstitution] = '1';
   requiredfields[techreport][nyear] = '1';

   standardfields[techreport][ntype] = '1';
   standardfields[techreport][nnumber] = '1';
   standardfields[techreport][naddress] = '1';
      
/* unpublished */
   standardfields[unpublished][nauthor] =
   requiredfields[unpublished][nauthor] = '1';
   standardfields[unpublished][ntitle] =
   requiredfields[unpublished][ntitle] = '1';
   standardfields[unpublished][nnote] =
   requiredfields[unpublished][nnote] = '1';
      
   standardfields[MAX_BIBTEX_TYPES][nannote] = '1';
   standardfields[MAX_BIBTEX_TYPES][nkey] = '1';
   standardfields[MAX_BIBTEX_TYPES][nnote] = '1';
   standardfields[MAX_BIBTEX_TYPES][ncrossref] = '1';

}


/*********************************************************************/
/* makeIconPixmaps:                                                  */
/*   Create all pixmaps for application                              */
/*********************************************************************/
static void 
makeIconPixmaps (void)
{
   bvIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_ico64_bits,
	            bv_ico64_width, bv_ico64_height);
   lstIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_lst64_bits,
	            bv_lst64_width, bv_lst64_height);
   macIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_mac64_bits,
	            bv_mac64_width, bv_mac64_height);
   helpIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_help64_bits,
	            bv_help64_width, bv_help64_height);
   annoteIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_annote64_bits,
	            bv_annote64_width, bv_annote64_height);
   crdIconPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)bv_crd64_bits,
	            bv_crd64_width, bv_crd64_height);
   chkmarkPixmap = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)chkmrk16_bits,
	            chkmrk16_width, chkmrk16_height);
   errorPixmap   = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)error32_bits,
	            error32_width, error32_height);
   warnPixmap    = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)warn32_bits,
	            warn32_width, warn32_height);
   noticePixmap  = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)notice32_bits,
	            notice32_width, notice32_height);
   questPixmap   = XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)quest32_bits,
	            quest32_width, quest32_height);
   progInfoPixmap =XCreateBitmapFromData(XtDisplay(topLevel),
	            RootWindowOfScreen(XtScreen(topLevel)),
	            (char *)proginfo_bits,
	            proginfo_width, proginfo_height);
}


/*********************************************************************/
/* ERROR HANDLING FUNCTIONS                                          */
/*********************************************************************/

/*********************************************************************/
/* installErrorHandlers:                                             */
/*   Register signal, warning and error handlers                     */
/*********************************************************************/
static void
installErrorHandlers (void)
{
int i;

   /* install signal handler */
   for (i = 1; i<= 15; i++)
      signal(i, signalHandler); 

   /* install Xtoolkit handlers */
   XtSetWarningHandler((XtErrorHandler)Xt_warningHandler);
   XtSetErrorHandler((XtErrorHandler)Xt_errorHandler);

   /* install X error handler */
   XSetErrorHandler(X_errorHandler);
   XSetIOErrorHandler(IO_errorHandler);
}


/*********************************************************************/
/* Xt_warningHandler:                                                */
/*   Xt warning handler                                              */
/*********************************************************************/
static XtErrorHandler
Xt_warningHandler (String warn)
{
}


/*********************************************************************/
/* Xt_errorHandler:                                                  */
/*   Xt warning handler                                              */
/*********************************************************************/
static XtErrorHandler
Xt_errorHandler (String error)
{
}


/*********************************************************************/
/* X_errorHandler:                                                   */
/*   Service X protocol errors                                       */
/*********************************************************************/
static int
X_errorHandler (Display *dsp, XErrorEvent *event)
{
  char msg[80];
  XGetErrorText(dsp, event->error_code, msg, 80);
  fprintf(stderr, "Error code %s\n", msg);
  return(0);
}


/*********************************************************************/
/* IO_errorHandler:                                                  */
/*   Service I/O errors from X                                       */
/*********************************************************************/
static int
IO_errorHandler (Display *dsp)
{ 
  fprintf(stderr, "Fatal Error on X Display %s\n",
		  (char *)XDisplayName(NULL));
  exit(1);

}


/*********************************************************************/
/* signalHandler:                                                    */
/*   Service signals                                                 */
/*********************************************************************/
static void
signalHandler (int s, int code)
{  
  signal(s, signalHandler); 
}


