%{
/*
 * Copyright 1991 Klaus Zitzmann, 1993-1996 Johannes Sixt
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation. The authors
 * make no representations about the suitability of this software for
 * any purpose. It is provided "as is" without express or implied warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Authors: Klaus Zitzmann <zitzmann@infko.uni-koblenz.de>
 *          Johannes Sixt <Johannes.Sixt@telecom.at>
 */

#include <stdio.h>		/* need FILE */
#ifndef SYSV
#include <strings.h>
#else
#include <string.h>
#endif
#include <ctype.h>

#include <stdlib.h>		/* atof */
#include <X11/Intrinsic.h>	/* needed for X stuff in objects.h */
#include "objects.h"
#include "yyscan.h"

#define yywrap() 1

float yyfloatval;		/* value of a TOK_FLOAT */
char yystrval[MAX_TEXT_LEN+1];	/* text of a text argument or align argument */
Unit yyunit;			/* value of a TOK_UNIT */

/* these are prototypes of functions defined by lex */
int yylook(void);
int yyback(int *p, int m);

/* we do the line counting ourselves */
int lineno = 1;

static enum {gtNo,gtNext,gtForceNext,gtForceNext2} getText;
	/* specifies if text argument is possible or not:
		gtNo		not possible
		gtNext		possible after next {
		gtForceNext	must come after next {
		gtForceNext2	must come after second next {
	*/

#define TRUE  1
#define FALSE 0

static int getUnitLen;		/* TRUE if \unitlength is read */
static int getUnit;		/* TRUE if unit of \unitlength is read */

static char *readText(unsigned numBr);

%}


%s  inPictEnv  inPut  lookOut


white		[ \t]
digit		[0-9]
letter		[A-Za-z]
sign		[+-]


%%


{white}+		{ /* ignore */ }
"%".*\n			{ /* ignore LaTeX comments */ ++lineno; }
\n			{ ++lineno; }

<INITIAL>\\unitlength=			|
<INITIAL>\\setlength\{\\unitlength\}	{ getUnitLen = TRUE; }
<INITIAL>{sign}?{digit}+		|
<INITIAL>{sign}?{digit}*"."{digit}*	{ if(getUnitLen)
					  {
					    yyfloatval = (float) atof(yytext);
					    getUnitLen = FALSE;
					    getUnit = TRUE;
					    return TOK_UNITLENGTH;
					  }
					}
<INITIAL>[Pp][Tt]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitPT; return TOK_UNIT; } }
<INITIAL>[Cc][Mm]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitCM; return TOK_UNIT; } }
<INITIAL>[Mm][Mm]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitMM; return TOK_UNIT; } }
<INITIAL>[Pp][Cc]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitPC; return TOK_UNIT; } }
<INITIAL>[Ii][Nn]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitIN; return TOK_UNIT; } }
<INITIAL>[Bb][Pp]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitBP; return TOK_UNIT; } }
<INITIAL>[Dd][Dd]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitDD; return TOK_UNIT; } }
<INITIAL>[Cc][Cc]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitCC; return TOK_UNIT; } }
<INITIAL>[Ss][Pp]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitSP; return TOK_UNIT; } }
<INITIAL>[Ee][Mm]	{if(getUnit)
			{ getUnit = FALSE; yyunit = UnitEM; return TOK_UNIT; } }

<INITIAL>\\begin\{picture\}	{BEGIN inPictEnv; return TOK_BEGIN;}

<INITIAL>.			{if (yytext[0] == '\n') ++lineno; }


<inPictEnv>\\bezier		|
<inPictEnv>\\qbezier{white}*\[	{return TOK_BEZIER;}
<inPictEnv>\\qbezier		{return TOK_QBEZIER;}
<inPictEnv>\\put                {BEGIN inPut; return TOK_PUT;}

<lookOut>\\framebox             {getText = gtForceNext;  BEGIN inPictEnv;
				 return TOK_FRAMEBOX;}
<lookOut>\\makebox              {getText = gtForceNext;  BEGIN inPictEnv;
				 return TOK_MAKEBOX;}
<lookOut>\\dashbox              {getText = gtForceNext2; BEGIN inPictEnv;
				 return TOK_DASHBOX;}
<lookOut>\\line                 {BEGIN inPictEnv; return TOK_LINE;}
<lookOut>\\vector               {BEGIN inPictEnv; return TOK_VECTOR;}
<lookOut>\\circle               {BEGIN inPictEnv; return TOK_CIRCLE;}
<lookOut>\\circle\*             {BEGIN inPictEnv; return TOK_CIRCLE_AST;}
<lookOut>\\oval                 {BEGIN inPictEnv; return TOK_OVAL;}
<lookOut>\\rule                 {BEGIN inPictEnv; return TOK_RULE;}

<lookOut>\}			{/* empty text */
				 yystrval[0] = '\0';
				 BEGIN inPictEnv;
				 return TOK_TEXT;}
<lookOut>.			{/* non-empty text */
				 yystrval[0] = yytext[0];
				 if (yytext[0] == '\n') ++lineno;
				 strcpy(&yystrval[1],
					readText(yytext[0] == '{'));
				 BEGIN inPictEnv;
				 return TOK_TEXT;}

<inPictEnv>\\unitlength		{ return TOK_DIM_UNITLENGTH; }
<inPictEnv>[Pp][Tt]		{ yyunit = UnitPT; return TOK_UNIT; }
<inPictEnv>[Cc][Mm]		{ yyunit = UnitCM; return TOK_UNIT; }
<inPictEnv>[Mm][Mm]		{ yyunit = UnitMM; return TOK_UNIT; }
<inPictEnv>[Pp][Cc]		{ yyunit = UnitPC; return TOK_UNIT; }
<inPictEnv>[Ii][Nn]		{ yyunit = UnitIN; return TOK_UNIT; }
<inPictEnv>[Bb][Pp]		{ yyunit = UnitBP; return TOK_UNIT; }
<inPictEnv>[Dd][Dd]		{ yyunit = UnitDD; return TOK_UNIT; }
<inPictEnv>[Cc][Cc]		{ yyunit = UnitCC; return TOK_UNIT; }
<inPictEnv>[Ss][Pp]		{ yyunit = UnitSP; return TOK_UNIT; }
<inPictEnv>[Ee][Mm]		{ yyunit = UnitEM; return TOK_UNIT; }

<inPictEnv,inPut>{sign}?{digit}+		|
<inPictEnv,inPut>{sign}?{digit}*"."{digit}*  {yyfloatval = (float) atof(yytext);
				 	        return TOK_FLOAT;}

<inPictEnv>\{		{ if(getText == gtForceNext)
			    {
			      (void) strcpy(yystrval,readText(0));
			      getText = gtNo;
			      return TOK_TEXT;
			    }
			  else if(getText == gtForceNext2)
			    getText = gtForceNext;}

<inPut>\{			BEGIN lookOut;
<inPictEnv>\}			{/* ignore */}
<inPictEnv,inPut>[(),\[\]]	{/* ignore */}

<inPictEnv>\\end\{picture\}	{return TOK_END;}

<inPictEnv>\[{letter}+\]	{  strcpy(yystrval,&yytext[1]);
				   yystrval[yyleng - 1] = '\0';
				   return TOK_LETTERS_OPT;}

<inPictEnv>\\thicklines		{return TOK_THICKLINE;}
<inPictEnv>\\thinlines		{return TOK_THINLINE;}


<inPictEnv>.		{fprintf(stderr,
			 "yylex: unrecognized char '%c' (%#X) in line %d\n",
				yytext[0], yytext[0], lineno);
			 if (yytext[0] == '\n') ++lineno;
			}

%%


/*
 * Initializes the input file. The file must already be open!
 */
void yyinitscan(FILE *fp)
{
	yyin = fp;
	lineno = 1;

	BEGIN INITIAL;

	getUnitLen = FALSE;
	getUnit = FALSE;
} /* yyinitscan */


/*
 * Reads in text which comes as argument to a command up to the closing
 * brace. Copes with nested { and }. The returned value does NOT include
 * the delimiting braces.
 * THE RESULT IS ONLY VALID UNTIL THE NEXT CALL!!
 */
char *readText(unsigned numBr)
{
    static char text[MAX_TEXT_LEN+1];
    unsigned    i = 0;
    char        ch;
    unsigned    numBraces = numBr + 1;
    int		start_line = lineno;

    /* pre: (numBr + 1) '{' is already read */
    while (numBraces > 0) {
	text[i++] = ch = input();
	if (ch == '\0') {
		/* end of file */
		(void) fprintf(stderr,
			"end of file in text on line %d\n",
			start_line);
		return text;
	}
	switch (ch) {
	case '{':	numBraces++;	break;
	case '}':	numBraces--;	break;
	case '\n':	++lineno;	break;
	}
	if (i >= MAX_TEXT_LEN) {
		(void) fprintf(stderr,
			"text argument on line %d too long\n",
			lineno);
		break;
	}
    }

    /* if numBraces > 0, then text arg was too long; skip to end of text arg */
    while (numBraces > 0 && (ch = input()) != '\0') {
	switch (ch) {
	case '{':	numBraces++;	break;
	case '}':	numBraces--;	break;
	case '\n':	++lineno;	break;
	}
    }
    text[--i] = '\0';

    return text;
}
