# -*- coding: utf-8 -*- #
# Copyright 2022 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Implementation of insights inventory."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

from googlecloudsdk.api_lib.storage import insights_api
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.storage.insights.inventory_reports import resource_args


class List(base.ListCommand):
  """List inventory report details."""

  detailed_help = {
      'DESCRIPTION':
          """
      List all inventory report details generated by a given inventory report
      config.
      """,
      'EXAMPLES':
          """

      To list all inventory report details for report config ID=1234,
      location=us-central1, and project=foo:

        $ {command} 1234 --location=us-central1 --project=foo

      To list all the same inventory report details with fully specified name
      of the report config:

        $ {command} /projects/foo/locations/us-central1/reportConfigs/1234

      To list all inventory reports, only returning the "status" key:

        $ {command} projects/a/locations/b/reportConfigs/some-id --format="json(status)"
      """,
  }

  @staticmethod
  def Args(parser):
    resource_args.add_report_config_resource_arg(
        parser, 'for which the report details should be listed')
    parser.display_info.AddFormat(
        """
        table(
            format('{}',name.basename()):label=REPORT_DETAIL_ID,
            snapshotTime,
            status.message:label=STATUS
        )
        """
    )

  def Run(self, args):
    report_config_name = args.CONCEPTS.report_config.Parse().RelativeName()
    return insights_api.InsightsApi().list_report_details(
        report_config_name, args.page_size)
