/* $Id: output.c,v 30000.23 1993/05/25 01:04:54 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/*****************************************************************
 * Fugue output handling                                         *
 *                                                               *
 * Screen handling written by Greg Hudson (Explorer_Bob) and     *
 * Ken Keys (Hawkeye).  Output queueing written by Ken Keys.     *
 * Handles all screen-related phenomena.                         *
 *****************************************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "world.h"
#include "socket.h"
#include "output.h"
#include "macro.h"
#include "search.h"
#include "signal.h"
#include "tfio.h"
#include "history.h"
#include "keyboard.h"

#ifdef HARDCODE
   smallstr BUF;
#  define clear_line()     ewrite("\r\033[K")
#  define clear_to_end()   ewrite("\033[K")
#  define attributes_off() ewrite("\033[m")
#  define enable_cm()      /* not used by vt100 */
#  define disable_cm()     /* not used by vt100 */
#  define xy(x,y)     do { \
                          sprintf(BUF,"\033[%d;%dH",(y),(x)); ewrite(BUF); \
                      } while (0)
#  define scroll(Y,y) do { \
                          sprintf(BUF,"\033[%d;%dr",(Y),(y)); ewrite(BUF); \
                      } while (0)

#else
#  ifdef TERMCAP
     static int   FDECL(getcap,(char *cap, Stringp str));
     static void  FDECL(tp,(char *s));
#    define clear_to_end()   tp(clear_to_eol->s)
#    define attributes_off() tp(attr_off->s)
#    define enable_cm()      tp(ti->s)
#    define disable_cm()     tp(te->s)
#    define xy(x,y)          tp(tgoto(cm->s, (x) - 1, (y) - 1));
#    define scroll(Y,y)      tp(tgoto(cs->s, (y) - 1, (Y) - 1));

#  else
#    define clear_to_end()   /* not supported w/o TERMCAP or HARDCODE */
#    define attributes_off() /* not supported w/o TERMCAP or HARDCODE */
#    define xy(x,y)          /* not supported w/o TERMCAP or HARDCODE */
#    define scroll(Y,y)      /* not supported w/o TERMCAP or HARDCODE */
#  endif
#endif

#ifndef HARDCODE
static void  NDECL(clear_line);
#endif
static void  NDECL(status_bar);
static void  FDECL(clear_lines,(int start, int end));
static void  NDECL(clear_input_window);
static void  NDECL(scroll_input);
static void  FDECL(ioutput,(int c));
static void  FDECL(ioutputs,(char *str, int len));
static void  FDECL(lwrite,(char *str, int len));
static void  FDECL(hwrite,(char *s, unsigned int len, int attrs));
static void  FDECL(attributes_on,(int attr));
static void  NDECL(discard_screen_queue);
static int   NDECL(check_more);
static void  NDECL(clear_more);
static String *FDECL(wrapline,(short *attrp));
static void  NDECL(output_novisual);
#ifdef SCREEN
static void  NDECL(output_noscroll);
static void  NDECL(output_scroll);
#else
# define output_noscroll()      /* not supported without TERMCAP or HARDCODE */
# define output_scroll()        /* not supported without TERMCAP or HARDCODE */
#endif
#define ewrite(str) lwrite(str, strlen(str))
static void  FDECL(putnch,(int c, int n));

void NDECL(ipos);
void NDECL(clr);
int  FDECL(putch,(int c));
void NDECL(init_term);
void NDECL(setup_screen);
void NDECL(oflush);
TOGGLER(tog_more);
TOGGLER(tog_visual);
TOGGLER(tog_insert);
TOGGLER(change_isize);
#ifdef SCREEN
void FDECL(put_world,(char *name));
void FDECL(put_mail,(int flag));
void FDECL(put_logging,(int flag));
void FDECL(put_active,(int count));
#endif
void NDECL(fix_screen);
void NDECL(clear_input_line);
void FDECL(iputs,(char *s));
void NDECL(inewline);
void NDECL(ibs);
void FDECL(ibackword,(int place));
void FDECL(newpos,(int place));
void NDECL(dEOL);
void NDECL(do_line_refresh);
void NDECL(do_replace);
void NDECL(reset_outcount);
int  NDECL(getwrap);
void FDECL(setprompt,(char *s));
String *FDECL(wrap,(char **startp, int *firstp));

void NDECL(do_refresh);
void NDECL(do_page);
void NDECL(do_hpage);
void NDECL(do_line);
void NDECL(do_flush);

#define Wrap (wrapflag ? wrapsize : columns)
#define keyboard_end (keybuf->len)

#define DEFAULT_LINES 24
#define DEFAULT_COLUMNS 80

/* Flags */

extern int log_on;

/* Termcap strings and flags */

#ifdef TERMCAP
static Stringp write_buffer;
static Stringp ti;             /* Cursor motion mode */
static Stringp te;             /* Cursor motion mode */
static Stringp cl;             /* Clear screen       */
static Stringp cm;             /* Move cursor        */
static Stringp cs;             /* Set scroll area    */
static Stringp underline;      /* termcap us         */
static Stringp reverse;        /* termcap mr         */
static Stringp blink;          /* termcap mb         */
static Stringp dim;            /* termcap mh         */
static Stringp bold;           /* termcap md         */
static Stringp attr_off;       /* termcap me         */
static Stringp start_of_line;  /* termcap cr or '\r' */
static Stringp clear_to_eol;   /* termcap ce         */
#endif
static short have_attr;
static int have_clear, have_scroll;

int input_cursor = TRUE;            /* is cursor in input window position? */
int can_have_visual = FALSE;        /* cm, cl and ce caps necessary */

/* Others */

extern int keyboard_pos;   /* position of logical cursor in keybuf */
extern int mail_flag;               /* should mail flag on status bar be lit? */
extern int active_count;            /* number of active sockets */
extern Stringp keybuf;              /* input buffer */

static int ox, oy;                  /* Current output cursor */
static int ix, iy;                  /* Current input cursor */
static int ystatus;                 /* line # of status line */
static int istarty, iendy, iendx;   /* start/end of current input line */
static Stringp moreprompt;          /* pager prompt */
static String *prompt;              /* current prompt */
static int outcount;                /* lines remaining until more prompt */
static int paused = FALSE;          /* output paused? */
static short more_attrs;            /* attributes for more prompt */
static Aline *currentline = NULL;   /* current logical line for printing */
static char *physline = NULL;       /* start of next physical line */

Stringp lpprompt;                   /* LP prompt */
int lines, columns;                 /* Take a wild guess */
int screen_setup = FALSE;           /* is *screen* in visual mode? */
TFILE *tfscreen;                    /* text waiting to be displayed */
TFILE *tfout;                       /* pointer to current output queue */
TFILE *tferr;                       /* pointer to current error queue */

#ifdef TERMCAP
extern int   FDECL(tgetent,(char *buf, char *name));
extern int   FDECL(tgetnum,(char *id));
extern char *FDECL(tgetstr,(char *id, char **area));
extern char *FDECL(tgoto,(char *code, int destcol, int destline));
extern int   FDECL(tputs,(char *cp, int affcnt, int (*outc)(int)));
#endif

/********************************
 * INTERMEDIARY OUTPUT ROUTINES *
 ********************************/

int putch(c)
    char c;
{
    lwrite(&c, 1);
    return 1;
}

static void putnch(c, n)
    int c;
    int n;
{
    STATIC_BUFFER(buf)

    Stringnadd(Stringterm(buf, 0), c, n);
    lwrite(buf->s, n);
}

static void lwrite(str, len)
    char *str;
    int len;
{
    int numwritten;

    while (len > 0) {
        if ((numwritten = write(1, str, len)) < 0) numwritten = 0;
        len -= numwritten;
        str += numwritten;
        if (len > 0) sleep(1);
    }
}

/********************************************************
 *                                                      *
 *                   TERMCAP ROUTINES                   *
 *                                                      *
 ********************************************************/

#ifdef TERMCAP
static int getcap(cap, str)
    char *cap;
    Stringp str;
{
    char tempstr[1024];    /* max length of termcap string is 1024 */
    char *temp;

    temp = tempstr;
    if (tgetstr(cap, &temp) == NULL) return 0;
    else Stringcpy(str, tempstr);
    return 1;
}
#endif

void init_term()
{
    char termcap_entry[1024];
    char *termtype;

    ix = 1;

    tfout = tferr = tfscreen = tfopen(NULL, "q");
    Stringinit(lpprompt);
    Stringcpy(Stringinit(moreprompt), "--More--");
    prompt = lpprompt;
    lines = DEFAULT_LINES;
    columns = DEFAULT_COLUMNS;
#ifdef HARDCODE
    have_attr = F_UNDERLINE | F_REVERSE | F_FLASH | F_BOLD | F_BELL;
    have_scroll = TRUE;
    can_have_visual = TRUE;
    have_clear = TRUE;
#else
# ifdef TERMCAP
    Stringinit(write_buffer);
    Stringinit(start_of_line);
    Stringinit(clear_to_eol);
    Stringinit(ti);
    Stringinit(te);
    Stringinit(cl);
    Stringinit(cm);
    Stringinit(cs);
    Stringinit(underline);
    Stringinit(reverse);
    Stringinit(blink);
    Stringinit(dim);
    Stringinit(bold);
    Stringinit(attr_off);
    outcount = lines;
    if ((termtype = getvar("TERM")) == NULL) {
        setivar("wrapsize", columns - 1, FALSE);
        tfputs("% Warning: terminal type undefined.", tferr);
        return;
    }
    if (tgetent(termcap_entry, termtype) != 1) {
        setivar("wrapsize", columns - 1, FALSE);
        tfprintf(tferr, "%% Warning: \"%s\" terminal unsupported.", termtype);
        return;
    }
    have_attr = 0;
    have_clear = have_scroll = can_have_visual = TRUE;

    if ((columns = tgetnum("co")) < 0) columns = DEFAULT_COLUMNS;
    setivar("wrapsize", columns - 1, FALSE);
    if ((lines = tgetnum("li")) < 0) lines = DEFAULT_LINES;
    if (!getcap("cr", start_of_line)) Stringcpy(start_of_line, "\r");
    if (!getcap("ce", clear_to_eol)) have_clear = FALSE;
    if (!getcap("ti", ti)) Stringcpy(ti, "");
    if (!getcap("te", ti)) Stringcpy(te, "");
    if (!getcap("cl", cl)) can_have_visual = FALSE;
    if (!getcap("cm", cm)) can_have_visual = FALSE;
    if (!getcap("cs", cs)) have_scroll = FALSE;
    if (getcap("us", underline))  have_attr |= F_UNDERLINE;
    if (getcap("mr", reverse))    have_attr |= F_REVERSE;
    if (getcap("mb", blink))      have_attr |= F_FLASH;
    if (getcap("mh", dim))        have_attr |= F_DIM;
    if (getcap("md", bold))       have_attr |= F_BOLD;
    else if (getcap("so", bold))  have_attr |= F_BOLD;
    if (!getcap("me", attr_off)) have_attr = 0;
    have_attr |= F_BELL;
# else
    have_attr = F_BELL;
    have_clear = have_scroll = can_have_visual = FALSE;
# endif
#endif
    outcount = lines;
}

void ipos()
{
    input_cursor = TRUE;
    xy(ix, iy);
}

void clr()
{
#ifdef HARDCODE
    ewrite("\033[2J\033[H");
#else
# ifdef TERMCAP
    tp(cl->s);
# endif
#endif
}

#ifndef HARDCODE
static void clear_line()
{
    STATIC_BUFFER(buf)

#ifdef TERMCAP
    if (have_clear) {
        tp(start_of_line->s);
        tp(clear_to_eol->s);
        return;
    }
#endif
    Stringadd(Stringnadd(Stringcpy(buf, "\r"), ' ', columns - 1), '\r');
    lwrite(buf->s, buf->len);
}
#endif

static void attributes_on(attr)
    int attr;
{
    if ((attr & F_BELL) && beeping) putch('\007');
    if (!hilite) return;
    if (attr & F_HILITE) attr |= hiliteattr;
#ifdef HARDCODE
    if (attr & F_UNDERLINE) ewrite("\033[4m");
    if (attr & F_REVERSE)   ewrite("\033[7m");
    if (attr & F_FLASH)     ewrite("\033[5m");
    if (attr & F_BOLD)      ewrite("\033[1m");
#else
# ifdef TERMCAP
    if (have_attr & attr & F_UNDERLINE) tp(underline->s);
    if (have_attr & attr & F_REVERSE)   tp(reverse->s);
    if (have_attr & attr & F_FLASH)     tp(blink->s);
    if (have_attr & attr & F_DIM)       tp(dim->s);
    if (have_attr & attr & F_BOLD)      tp(bold->s);
# endif
#endif
}

#ifdef TERMCAP
static void tp(s)
    char *s;
{
    tputs(s, 1, putch);
    fflush(stdout);
}
#endif

/*****************************************************
 *                                                   *
 *                  WINDOW HANDLING                  *
 *                                                   *
 *****************************************************/

static void status_bar()
{
    int i;
    char *line;
    World *world;

    line = (char *)MALLOC(columns + 1);
    strcpy(line, "_________");
    if ((world = fworld())) strncat(line, world->name, columns - 40 - 10 - 1);
    for (i = strlen(line); i < columns - 40 - 1; i++) line[i] = '_';
    if (active_count) sprintf(line + i, "(Active:%2d)_", active_count);
    else strcpy(line + i, "____________");
    strcat(line, log_on ? "(Logging)_" : "__________");
    strcat(line, mail_flag ? "(Mail)__Insert: " : "________Insert: ");
    strcat(line, (insert) ? "On_" : "Off");
    xy(1, ystatus);
    hwrite(line, columns, barattr);
    FREE(line);
    if (paused) {
        xy(1, ystatus);
        hwrite(moreprompt->s, moreprompt->len, more_attrs);
    }
}

void setup_screen()
{
    Stringp scr;

    attributes_off();
    if (!visual) {
        if (paused) prompt = moreprompt;
        return;
    }
#ifdef SCREEN
    prompt = lpprompt;
    if (isize <= 0) isize = 3;
    if (isize > lines - 3) isize = lines - 3;
    ystatus = lines - isize;
    outcount = ystatus - 1;
    enable_cm();

    if (have_scroll) {
        Stringnadd(Stringinit(scr), '\n', isize);
        scroll(1, lines);
        xy(1, lines);
        lwrite(scr->s, scr->len);
        Stringfree(scr);
    } else {
        clr();
        oy = 1;
    }
    status_bar();
    if (have_scroll) scroll(1, ystatus - 1);
    ix = iendx = oy = ox = 1;
    iy = iendy = istarty = ystatus + 1;
    ipos();
    screen_setup = TRUE;
#endif
}

#ifdef SCREEN
void put_world(name)
    char *name;
{
    char *line;
    int i;
    int len;

    if (!visual) return;
    len = columns - 40 - 10 - 1;
    line = (char *)MALLOC(len);
    line[0] = '\0';
    if (name) strncat(line, name, len);
    for (i = strlen(line); i < len; i++) line[i] = '_';
    line[i] = '\0';
    xy(10, ystatus);
    hwrite(line, len, barattr);
    FREE(line);
    ipos();
}

void put_mail(flag)
    int flag;
{
    if (screen_setup) {
        xy(columns - 18, ystatus);
        hwrite(flag ? "(Mail)" : "______", 6, barattr);
        ipos();
    }
}

void put_logging(flag)
    int flag;
{
    if (screen_setup) {
        xy(columns - 28, ystatus);
        hwrite(flag ? "(Logging)" : "_________", 9, barattr);
        ipos();
    }
}

void put_active(count)
    int count;
{
    smallstr buf;
    if (screen_setup) {
        xy(columns - 40, ystatus);
        if (count) {
            sprintf(buf, "(Active:%2d)", count);
            hwrite(buf, 11, barattr);
        } else hwrite("___________", 11, barattr);
        ipos();
    }
}
#endif

void change_isize()
{
    if (visual) {
        fix_screen();
        setup_screen();
    }
}

void tog_insert()
{
    if (screen_setup) {
        xy(columns - 2, ystatus);
        hwrite(insert ? "On_" : "Off", 3, barattr);
        ipos();
    }
}

void tog_visual()
{
    if (!can_have_visual) {
        setvar("visual", "0", FALSE);
        tfputs("% Visual mode not supported.", tferr);
        return;
    }
    if (!visual) fix_screen();
    setup_screen();
}

void fix_screen()
{
    clear_input_window();
    outcount = lines - 1;
#ifdef SCREEN
    if (have_scroll) scroll(1, lines);
    xy(1, ystatus);
    input_cursor = TRUE;
    clear_line();
    disable_cm();
    screen_setup = 0;
#endif
}

static void clear_lines(start, end)
    int start, end;
{
    while (start <= end) {
        xy(1, start++);
        clear_line();
    }
}

static void clear_input_window()
{
    /* only called if screen_setup */
    clear_lines(ystatus + 1, lines);
    ix = iendx = 1;
    iy = iendy = istarty = ystatus + 1;
    ipos();
}

void clear_input_line()
{
    if (!visual) {
        clear_line();
        iendx = ix = 1;
    } else {
        clear_lines(istarty, iendy);
        ix = iendx = 1;
        iy = iendy = istarty;
        ipos();
    }
}

static void scroll_input()
{
    scroll(ystatus + 1, lines);
    xy(1, lines);
    putch('\n');
    scroll(1, ystatus - 1);
    xy(iendx = 1, iy = lines);
}

/***********************************************************************
 *                                                                     *
 *                        INPUT WINDOW HANDLING                        *
 *                                                                     *
 ***********************************************************************/

static void ioutput(c)
    char c;
{
    ioutputs(&c, 1);
}

static void ioutputs(str, len)
    char *str;
    int len;
{
    int size;

    if (len <= 0) return;
    if (visual) {
        while (len) {
            if (iendy == lines && iendx > Wrap) break;
            size = Wrap - iendx + 1;
            if (size > len) size = len;
            lwrite(str, size);
            len -= size;
            str += size;
            iendx += size;
            if (iendx > Wrap && iendy != lines) xy(iendx = 1, ++iendy);
        }
    } else {
        size = Wrap - iendx + 1;
        if (size <= 0) return;
        if (size > len) size = len;
        iendx += size;
        lwrite(str, size);
    }
}

void iputs(s)
    char *s;
{
    int i, j, end;

    if (!s[0]) return;
    if (!visual) {
        for (i = j = 0; s[j]; j++) {
            if (++iendx > Wrap) iendx = Wrap;
            if (++ix > Wrap) {
                lwrite(s + i, j - i + 1);
                putch('\n');
                iendx = ix = 1;
                i = j + 1;
            }
        }
        if (i < j) lwrite(s + i, j - i);

        if (insert || ix == 1) {
            iendx = ix;
            ioutputs(keybuf->s + keyboard_pos, keyboard_end - keyboard_pos);
            putnch('\010', iendx - ix);
        }
        return;
    }

    for (j = 0; s[j]; j++) {
        iendx = ix;
        iendy = iy;
        if (ix == Wrap && iy == lines) {
            if (have_scroll && isize > 1 && !clearfull) {
                putch(s[j]);
                scroll_input();
                ix = 1;
                ipos();
                if (--istarty < ystatus + 1) istarty = ystatus + 1;
            } else {
                putch(s[j]);
                clear_input_window();
            }
        } else ioutput(s[j]);
        ix = iendx;
        iy = iendy;
    }
    if (insert) {
        ioutputs(keybuf->s + keyboard_pos, keyboard_end - keyboard_pos);
        if (keyboard_pos != keyboard_end) ipos();
    } else {
        end = ix + keyboard_end - keyboard_pos - 1;
        if ((iendy = iy + end / Wrap) > lines) {
            iendy = lines;
            iendx = Wrap;
        } else {
            iendx = end % Wrap + 1;
        }
    }
}

void inewline()
{
    Stringterm(lpprompt, 0);
    ix = iendx = 1;
    if (!visual) {
        putch('\n');
        if (prompt->len > 0) set_refresh_pending();
        return;
    }

    if (cleardone) {
        clear_lines(ystatus + 1, iendy);
        iy = iendy = istarty = ystatus + 1;
    } else {
        iy = iendy + 1;
        if (iy > lines) {
            if (have_scroll && isize > 1 && !clearfull) {
                scroll_input();
            } else {
                clear_input_window();
            }
        }
        istarty = iendy = iy;
    }
    ipos();
}

void ibs()
{
    int kpos, ppos;

    if (!visual) {
        ix--;
        if (ix) {
            putch('\010');
            iendx = ix;
            ioutputs(keybuf->s + keyboard_pos, keyboard_end - keyboard_pos);
            ioutput(' ');
            putnch('\010', iendx - ix);
        } else {
            iendx = ix = Wrap;
            do_line_refresh();
        }
        return;
    }

    if (!input_cursor) ipos();
    if (ix == 1 && iy == ystatus + 1) {              /* Move back a screen. */
        clear_input_window();
        kpos = keyboard_pos - (Wrap * isize) + 1;
        if (kpos < 0) {
            ppos = prompt->len + kpos;
            if (ppos < 0) ppos = 0;
            kpos = 0;
        } else ppos = prompt->len;
        ioutputs(prompt->s + ppos, prompt->len - ppos);
        ioutputs(keybuf->s + kpos, keyboard_pos - kpos);
        ix = iendx;
        iy = iendy;
        ioutputs(keybuf->s + keyboard_pos, keyboard_end - keyboard_pos);
        ipos();
        return;
    }
    if (ix == 1) {
        ix = Wrap;
        iy--;
        ipos();
    } else {
        ix--;
        putch('\010');
    }
    iendx = ix;
    iendy = iy;
    ioutputs(keybuf->s + keyboard_pos, keyboard_end - keyboard_pos);
    putch(' ');
  
    if ((keyboard_pos == keyboard_end) && (ix != Wrap)) putch('\010');
    else ipos();
}

void ibackword(place)
    int place;
{
    int kpos, ppos, i, oiex, oiey;

    if (!visual) {
        if (place == keyboard_end && ix - keyboard_pos + place > 0)
            for (i = place; i < keyboard_pos; i++) ibs();
        else {
            keyboard_pos = place;
            iendx = ix = Wrap;
            do_line_refresh();
        }
        return;
    }

    if (!input_cursor) ipos();
    ix -= (keyboard_pos - place);
    if (ix < 1) {
        ix += Wrap;
        iy--;
        if (iy < ystatus + 1) {
            clear_input_window();
            kpos = place - (Wrap * isize) + (Wrap - ix) + 1;
            if (kpos < 0) {
                ppos = prompt->len + kpos;
                if (ppos < 0) ppos = 0;
                kpos = 0;
            } else ppos = prompt->len;
            ioutputs(prompt->s + ppos, prompt->len - ppos);
            ioutputs(keybuf->s + kpos, place - kpos);
            ix = iendx;
            iy = iendy;
            ioutputs(keybuf->s + place, keyboard_end - place);
            if (keyboard_pos != keyboard_end) ipos();
            return;
        }
    }
    ipos();
    iendx = ix;
    iendy = iy;
    ioutputs(keybuf->s + place, keyboard_end - place);

    oiex = iendx;
    oiey = iendy;
    for (kpos = place; kpos < keyboard_pos; kpos++) ioutput(' ');
    iendx = oiex;
    iendy = oiey;
    ipos();
}

void newpos(place)
    int place;
{
    int diff, nix, niy, i, kpos, ppos;

    if (place < 0) place = 0;
    if (place > keyboard_end) place = keyboard_end;
    if (place == keyboard_pos) return;
    if (!visual) {
        nix = ix + place - keyboard_pos;
        if (nix == ix) return;
        if (nix < 1 || nix > Wrap) {
            keyboard_pos = place;
            do_line_refresh();
        } else {
            if (nix < ix) putnch('\010', keyboard_pos - place);
            else for (i = keyboard_pos; i < place; i++) putch(keybuf->s[i]);
            ix = nix;
        }
        keyboard_pos = place;
        return;
    }

    diff = place - keyboard_pos;
    if (diff == 0) return;
    niy = iy + diff / Wrap;
    nix = ix + diff % Wrap;

    if (nix > Wrap) {
        niy++;
        nix -= Wrap;
    }
    while (niy > lines) {
        kpos = keyboard_pos + (lines - iy + 1) * Wrap - ix + 1;
        scroll_input();
        ioutputs(keybuf->s + kpos, keyboard_end - kpos);
        keyboard_pos += (lines - ystatus) * Wrap;
        niy--;
    }

    if (nix < 1) {
        niy--;
        nix += Wrap;
    }
    if (niy < ystatus + 1) {
        kpos = keyboard_pos - (iy - niy) * Wrap - ix + 1;
        if (kpos < 0) {
            ppos = prompt->len + kpos;
            if (ppos < 0) ppos = 0;
            kpos = 0;
        } else ppos = prompt->len;
        clear_input_window();
        ioutputs(prompt->s + ppos, prompt->len - ppos);
        ioutputs(keybuf->s + kpos, keybuf->len - kpos);
        niy = ystatus + 1;
    }

    ix = nix;
    iy = niy;

    ipos();
    keyboard_pos = place;
}

void dEOL()
{
    STATIC_BUFFER(buf)

    if (have_clear) clear_to_end();
    else {
        Stringterm(buf, 0);
        Stringnadd(buf, ' ', iendx - ix + 1);
        Stringnadd(buf, '\010', iendx - ix + 1);
        lwrite(buf->s, buf->len);
    }
    if (visual && iy < iendy) {
        clear_lines(iy + 1, iendy);
        ipos();
    }
    iendx = ix;
    iendy = iy;
}

void do_refresh()
{
    int oix, oiy, kpos, ppos;

    if (!visual) {
        do_replace();
        return;
    }

    oix = ix;
    oiy = iy;

    kpos = keyboard_pos - (iy - istarty) * Wrap - ix + 1;
    if (kpos < 0) {
        ppos = prompt->len + kpos;
        kpos = 0;
    } else ppos = prompt->len;

    clear_input_line();
    ioutputs(prompt->s + ppos, prompt->len - ppos);
    ioutputs(keybuf->s + kpos, keybuf->len - kpos);
    ix = oix;
    iy = oiy;
    if (keyboard_pos != keyboard_end) ipos();
}

void do_line_refresh()
{
    int curcol, kpos, ppos;

    if (!visual && paused) return;
    clear_refresh_pending();
    if (visual) {
        ipos();
        return;
    }

    curcol = (prompt->len + keyboard_pos) % Wrap;
    kpos = keyboard_pos - curcol;
    if (kpos < 0) {
        ppos = prompt->len + kpos;
        kpos = 0;
    } else ppos = prompt->len;

    clear_input_line();
    ioutputs(prompt->s + ppos, prompt->len - ppos);
    ioutputs(keybuf->s + kpos, keybuf->len - kpos);
    ix = curcol + 1;
    putnch('\010', iendx - ix);
}

void do_replace()
{
    int okpos;
    extern Sock *fsock;

    clear_input_line();
    okpos = keyboard_pos;
    keyboard_pos = keyboard_end;
    iputs(prompt->s);
    if (!fsock || fsock->flags & SOCKECHO) iputs(keybuf->s);
    if (visual) newpos(okpos);
    keyboard_pos = okpos;
    if (!visual) {
        if (ix - (keyboard_end - keyboard_pos) >= 1)
            putnch('\010', keyboard_end - keyboard_pos);
        else {
            putch('\n');
            do_line_refresh();
        }
    }
    clear_refresh_pending();
}


/*****************************************************
 *                                                   *
 *                  OUTPUT HANDLING                  *
 *                                                   *
 *****************************************************/

/*************
 * Utilities *
 *************/

static void hwrite(s, len, attrs)
    char *s;
    unsigned int len;
    int attrs;
{
    STATIC_BUFFER(spaces)

    if (attrs & F_INDENT) {
        if (spaces->len != wrapspace) {
            Stringterm(spaces, 0);
            Stringnadd(spaces, ' ', wrapspace);
        }
        lwrite(spaces->s, spaces->len);
    }
    attrs &= ~(F_NEWLINE | F_INDENT);
    if (attrs) attributes_on(attrs);
    lwrite(s, len);
    if (attrs) attributes_off();
}

void reset_outcount()
{
    outcount = visual ? (have_scroll ? (ystatus - 1) : outcount) : lines - 1;
}

/* return TRUE if okay to print */
static int check_more()
{
    if (paused) return FALSE;
    if (!more || ox != 1) return TRUE;
    if (outcount-- > 0) return TRUE;

    paused = TRUE;
    if ((more_attrs = do_hook(H_MORE, NULL, "")) == 0)
        more_attrs = F_BOLD | F_REVERSE;
    if (visual) {
        xy(1, ystatus);
        hwrite(moreprompt->s, moreprompt->len, more_attrs);
        ipos();
    } else {
        prompt = moreprompt;
        do_replace();
    }
    return FALSE;
}

static void clear_more()
{
    if (!paused) return;
    paused = FALSE;
    if (visual) {
        xy(1, ystatus);
        hwrite("________", 8, barattr);
        ipos();
        if (!have_scroll) {
            outcount = ystatus - 1;
            do_redraw();
        }
    } else {
        prompt = lpprompt;
        clear_input_line();
        set_refresh_pending();
    }
    oflush();
}

void tog_more()
{
    if (!more) {
        reset_outcount();
        clear_more();
    } else if (!have_scroll) {
        setup_screen();
    }
}

void do_page()
{
    if (!paused) return;
    outcount = visual ? ystatus - 1 : lines - 1;
    clear_more();
}

void do_hpage() 
{
    if (!paused) return;
    outcount = (visual ? ystatus - 1 : lines - 1) / 2;
    clear_more();
}

void do_line()
{
    if (!paused) return;
    outcount = 1;
    clear_more();
}

void do_flush()
{
    if (!paused) return;
    discard_screen_queue();
}

static void discard_screen_queue()
{
    outcount = visual ? ystatus - 1 : lines - 1;
    free_queue(tfscreen->u.queue);
    if (currentline) {
        free_aline(currentline);
        currentline = NULL;
        physline = NULL;
    }
    clear_more();
    screenout(new_aline("--- Output discarded ---", F_NEWLINE));
}

/* return the next physical line to be printed */
static String *wrapline(attrp)
    short *attrp;
{
    static int firstline = TRUE;
    String *result;

    while (!currentline || ((currentline->attrs & F_GAG) && gag)) {
        if (currentline) free_aline(currentline);
        if (!(currentline = dequeue(tfscreen->u.queue))) return NULL;
    }

    if (!check_more()) return NULL;
    if (!physline) {
        physline = currentline->str;
        firstline = TRUE;
    }
    *attrp = currentline->attrs;
    if (!firstline) *attrp &= ~F_BELL;
    if (!(*attrp & F_NEWLINE))                           /* debug */
        printf("BUG in wrapline:  partial line\n");    /* debug */
    if (!firstline && wrapflag) *attrp |= F_INDENT;
    result = wrap(&physline, &firstline);
    if (!*physline) {
        physline = NULL;
        free_aline(currentline);
        currentline = NULL;
    }
    return result;
}

/* Note:  wrap() returns a pointer to an internal static area! */
String *wrap(startp, firstp)
    char **startp;
    int *firstp;
{
    char *place, *max;
    STATIC_BUFFER(dest)

    Stringterm(dest, 0);
    max = *startp + Wrap - 1;
    if (!*firstp) {
        if (wrapflag) {
            max -= wrapspace;
        }
    }

    for (place = *startp; *place && place < max && *place != '\n'; place++);
    if (!*place) {
        Stringcat(dest, *startp);
        *firstp = TRUE;
        *startp = place;
    } else if (*place == '\n') {
        Stringncat(dest, *startp, place - *startp);
        *firstp = TRUE;
        *startp = ++place;
    } else {
        if (wrapflag)
            while (place != *startp && !isspace(*place)) place--;
        if (place == *startp) {
            Stringncat(dest, *startp, max - *startp);
            *startp = max;
        } else {
            Stringncat(dest, *startp, ++place - *startp);
            *startp = place;
        }
        *firstp = FALSE;
    }
    return dest;
}


/****************
 * Main drivers *
 ****************/

void globalout(aline)
    Aline *aline;
{
    record_global(aline);
    screenout(aline);
}

/* write to tfscreen (no history) */
void screenout(aline)
    Aline *aline;
{
    if (!currentline && !tfscreen->u.queue->head) {
        /* shortcut if screen queue is empty */
        (currentline = aline)->links++;
    } else {
        aline->links++;
        enqueue(tfscreen->u.queue, aline);
    }
    if (!paused && !interrupted()) oflush();
}

void oflush()
{
    if (!visual || !screen_setup) output_novisual();
    else if (!have_scroll) output_noscroll();
    else output_scroll();
}

static void output_novisual()
{
    String *line;
    short attrs;
    int first = 1;

    while ((line = wrapline(&attrs)) != NULL) {
        if (first && ix != 1) {
            if (have_clear)
                clear_input_line();
            else /* output optimization */
                Stringadd(Stringnadd(line, ' ', columns - 1 - line->len), '\r');
            set_refresh_pending();
        }
        first = 0;
        hwrite(line->s, line->len, attrs);
        if (attrs & F_NEWLINE) {
            putch('\n');
            ox = 1;
        } else ox = line->len + 1;
    }
}

#ifdef SCREEN
static void output_noscroll()
{
    String *line;
    short attrs;

    while ((line = wrapline(&attrs)) != NULL) {
        if (ox == 1 && !more) {
            xy(1, (oy + 1) % (ystatus - 1) + 1);
            clear_line();
        }
        xy(ox, oy);
        input_cursor = FALSE;
        hwrite(line->s, line->len, attrs);
        if (attrs & F_NEWLINE) {
            ox = 1;
            if ((oy = oy % (ystatus - 1) + 1) == 1) xy(ox, 1);
        } else ox = line->len + 1;
    }
}

static void output_scroll()
{
    String *line;
    short attrs;
    static short newline_held = TRUE;

    while ((line = wrapline(&attrs)) != NULL) {
        if (input_cursor) {
            xy(ox, (ystatus - 1));
            input_cursor = FALSE;
        }
        if (newline_held) putch('\n');
        if ((newline_held = attrs & F_NEWLINE)) ox = 1;
        else ox = line->len + 1;
        hwrite(line->s, line->len, attrs);
    }
}
#endif

/***********************************
 * Interfaces with rest of program *
 ***********************************/

int getwrap()
{
    return (Wrap);
}

void setprompt(s)
    char *s;
{
    if (strcmp(lpprompt->s, s)) {
        Stringcpy(lpprompt, s);
        do_replace();
    }
}

void change_bar()
{
    char *value;

    if (!(value = getvar("barattr"))) return;
    if ((barattr = parse_attrs(value)) < 0) barattr = 0;
    if (visual) status_bar();
    ipos();
}


#ifdef DMALLOC
void free_term()
{
#ifdef TERMCAP
    Stringfree(write_buffer);
    Stringfree(start_of_line);
    Stringfree(clear_to_eol);
    Stringfree(ti);
    Stringfree(te);
    Stringfree(cl);
    Stringfree(cm);
    Stringfree(cs);
    Stringfree(underline);
    Stringfree(reverse);
    Stringfree(blink);
    Stringfree(dim);
    Stringfree(bold);
    Stringfree(attr_off);
#endif
    Stringfree(lpprompt);
    Stringfree(moreprompt);
    tfclose(tfscreen);
}
#endif
