#include <stdio.h>
#include <string.h>       
#include <ctype.h>
#include <stdlib.h>
#include <fcntl.h>
#include <pwd.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>

#include "contest.h"

#define lower(c) (isupper(c) ? tolower(c) : (c))

#define USAGE "Usage: %s problem_letter\n"

static char *read_file();
static void write_file();
static void panic();
static void out_file_name();

static char *progname;
static char in_file[MAXPATHLEN];   /* problemX.Y */
static char out_file[MAXPATHLEN] = ".";
static int real_uid;
static char prob;
static char extn[MAXPATHLEN];

        int
main(argc, argv)
        int argc;
        char *argv[];
{
	int file_size;
	char *file_contents;

	progname = argv[0];

	if (argc != 2 || strlen(argv[1]) != 1) {
		(void) fprintf(stderr, USAGE, progname);
		panic();
	}

	prob = lower(argv[1][0]);

	file_contents = read_file(prob, &file_size);
	write_file(file_contents, file_size);
	return 0;
}

        static
char *read_file(prob, size)
        char prob;
        int *size;
{
	char *cp;
	int in_fd;
	int save_uid;
	struct stat stat;
	char *ret_val;
	char *dot_pos;

	/*
	 * Switch to the real uid while reading the file
	 */
	save_uid = geteuid();
	real_uid = getuid();
	setreuid(-1, real_uid);

	(void) sprintf(in_file, "%s%c", PREFIX, prob);
	if ((in_fd = prob_search(in_file, O_RDONLY)) < 0) {
		(void) fprintf(stderr,
			       "%s: couldn't open input file for problem %c\n",
					       progname, prob);
		panic();
	}
        dot_pos = strrchr(in_file, '.');
	assert(dot_pos != 0);
	strcpy(extn, dot_pos+1);

	if (fstat(in_fd, &stat) == -1) {
		(void) fprintf(stderr, "%s: couldn't stat %s\n", progname,
			       in_file);
		panic();
	}
	*size = stat.st_size;
	ret_val = emalloc(*size);
	if (read(in_fd, ret_val, *size) != *size) {
		(void) fprintf(stderr, "%s: read failed for file %s\n",
			       progname, in_file);
		panic();
	}
	(void) close(in_fd);

	/*
	 * revert to the initial setuid value
	 */
	setreuid(-1, save_uid);
	return ret_val;
}
			       

        static void
write_file(contents, size)
        char *contents;
        int size;
{
	int fd;
	int version;
	struct passwd *pw;
	char out_file[MAXPATHLEN];
	FILE *mail;

	if ((pw = getpwuid(real_uid)) == 0) {
		(void) fprintf(stderr, "%s: failed to get passwd entry\n",
			       progname);
		panic();
	}

	version = get_version(pw->pw_name, prob);
	(void) sprintf(out_file, "%s/%s%c%d.%s", get_teamdir(pw->pw_name),
		       PREFIX, prob, version, extn);

	if ((fd = open(out_file, O_WRONLY | O_CREAT | O_EXCL, 0400)) == -1) {
		perror(progname);
		(void) fprintf(stderr, "%s: error opening output file %s\n",
			       progname, out_file);
		panic();
	}

	if (write(fd, contents, size) != size) {
		(void) fprintf(stderr, "%s: error writing output file\n",
			       progname);
		panic();
	}
	close(fd);

	{
#define MAIL_COMMAND "/bin/mail %s >/dev/null"
		char *judge = check_env_var(JUDGEVAR, JUDGE);
		char *mail_command = emalloc(strlen(MAIL_COMMAND) +
					     strlen(judge));

		sprintf(mail_command, MAIL_COMMAND, judge);
		if ((mail = popen(mail_command, "w")) == 0) {
			(void) fprintf(stderr, "%s: mail to marker failed\n",
				       progname);
			panic();
		}
		free(mail_command);
	}

	(void) fprintf(mail, "Submission %d for problem %c, set %s\n", version,
		       toupper(prob), get_contest_name());
	(void) fprintf(mail, "\tin file %s\n", out_file);
	(void) fclose(mail);

	(void) printf(
	           "Submission %d for problem %c of problem set %s recorded\n",
		      version, toupper(prob), get_contest_name());
	
}


        static void
panic()
{
	if (strcmp(out_file, ".") != 0) {
		(void) unlink(out_file);
	}
	(void) fprintf(stderr, "%s: submission not recorded\n", progname);
	exit(1);
}


        static int
get_version(team, prob)
        char *team;
        char prob;
{
	int version;
	int fd;
	char path[MAXPATHLEN];

	for (version = 1 ; ; version++) {
		sprintf(path, "%s/%s%c%d", get_teamdir(team),
			PREFIX, prob, version);
		switch(fd = prob_search(path)) {
		      case -2:
			(void) fprintf(stderr,
				       "%s: error determining version\n",
				       progname);
			panic();
		      case -1:
			return version;

		      default:
			close(fd);
			break;
		}
	}
	/*NOTREACHED*/
}			
