/* --------------------------------- grmswin.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Graphics driver for Microsoft Windows.
 * Windows support by Michael Taylor miket@canb.auug.org.au
*/

#define STRICT
#include <windows.h>

#include "fly.h"
#include "common.h"


#define INITED		0x1000

static HDC      memhdc;
static HBITMAP	mybm1, oldbm=0;
static RECT	rect;
static HPEN 	pens[256];
static COLORREF	colors[256];
static HPALETTE hpal;
static HPALETTE oldhpal;
static LOGPALETTE * plgpl;
static PALETTEENTRY ape[256];
static int 	currx = 0, curry = 0, plotpt = 0;

static void FAR
GrmChangeSize ()
{
        RECT rect;

	GetClientRect (ghWndMain, &rect);
        
  	CS->sizex = rect.right;
	CS->sizey = rect.bottom;
  	set_screen (rect.right, rect.bottom);
	if (oldbm) {		/* resize the bitmap as well */
		SelectObject (memhdc, oldbm);
		DeleteObject (mybm1);
		mybm1 = CreateCompatibleBitmap (hdc, rect.right, rect.bottom);
		oldbm = SelectObject (memhdc, mybm1);
	}
	FillRect (memhdc, &rect, GetStockObject (BLACK_BRUSH));
}

extern void FAR
MSResetPalette (void)
{
        if (hpal) {
		SelectPalette (memhdc, hpal, FALSE);
		RealizePalette (memhdc);  /* remap colours as we lost the focus */
		if (memhdc != hdc) {
			SelectPalette (hdc, hpal, FALSE);
			RealizePalette (hdc);  /* remap colours as we lost the focus */
		}
	}
}

static int FAR
GrmSetActive (int page)
{
	page = page;
	return (0);
}

static int FAR
GrmSetVisual (int page)
{
	page = page;
	return (0);
}

static int FAR
GrmSetWriteMode (int mode)
{
	switch (mode) {
	default:
	case T_MSET:
		mode = R2_COPYPEN;
		break;
	case T_MOR:
		mode = R2_MERGEPEN;
		break;
	case T_MXOR:
		mode = R2_XORPEN;
		break;
	}
	SetROP2 (memhdc, mode);

	return (0);
}

static int FAR
GrmSetPalette (int n, long c)
{
	Uint	r, g, b;

	n = st.colors[n];

	r = C_RGB_R (c);
	g = C_RGB_G (c);
	b = C_RGB_B (c);

	c = RGB (r, g, b);
	
     	ape[n].peRed = 	 LOBYTE(r);
     	ape[n].peGreen = LOBYTE(g);
    	ape[n].peBlue =  LOBYTE(b);
    	ape[n].peFlags = 0;

	SetPaletteEntries (hpal, 0, 256, ape);
	RealizePalette (memhdc);
	if (memhdc != hdc) {
		SelectPalette (hdc, hpal, FALSE);
		RealizePalette (hdc);
	}

	if (pens[n])
		DeleteObject (pens[n]);

	pens[n] = CreatePen (PS_SOLID, 0, 
			PALETTEINDEX(GetNearestPaletteIndex (hpal, c)));
	colors[n] = c;
	return (0);
}

static Uint    last_color = 0xffff;

static void
GrmMoveTo (Uint x, Uint y)
{
	if (plotpt) {
		SetPixel (memhdc, currx, curry, colors[last_color]);
		plotpt = 0;
	}
	MoveTo (memhdc, x, y);
	currx = x;
	curry = y;
}

static void
GrmDrawTo (Uint x, Uint y, Uint c)
{
  	if (last_color != c) {
    		if (!pens[c])
			MessageBox((HWND)0, 
				   (LPCSTR)"No pen created for that colour",
				   (LPCSTR)"Fly8 Error", MB_OK);
		else
			SelectObject (memhdc, pens[c]);
		last_color = c;
  	}

  	LineTo (memhdc, x, y);
  	currx = x;
	curry = y;
	plotpt = 1;
}

#if USE_CLEAR
static void FAR
GrmClearWin (SCREEN *scr)
{
	scr = scr;
  	st.flags |= SF_CLEARED;
	FillRect (memhdc, &rect, GetStockObject (BLACK_BRUSH));
}
#endif

static int FAR
GrmInit (DEVICE *dev, char * options, int blt)
{
	int i, r, g, b;
	
	if (dev->sizex == 0 || dev->sizey == 0)
		return (1);

	for (i = 0; i < 256; ++i) {
		colors[i] = 0;
		pens[i] = 0;
	}
	
	ghWndMain = CreateWindow(
			(LPSTR)Fly8AppName,
			(LPSTR)Fly8Message,
			WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
			0,			/*  x  */
			0,			/*  y  */
			dev->sizex,		/* cx  */
			dev->sizey,		/* cy  */
			(HWND)0,		/* no parent */
			(HMENU)0,		/* use class menu */
			(HANDLE)Fly8Instance,	/* handle to window instance */
			(LPSTR)0		/* no params to pass on */
			);

/* Make window visible
*/
	ShowWindow (ghWndMain, SW_SHOW );
	UpdateWindow (ghWndMain);

	memhdc = hdc = GetDC (ghWndMain);
	GetClientRect (ghWndMain, &rect);
  	CS->sizex = rect.right;
  	CS->sizey = rect.bottom;
  	set_screen (rect.right, rect.bottom);
	
	/* decide if double buffering using bit-blitting is possible */
	if ( blt &&
	     (GetDeviceCaps (hdc, RASTERCAPS) & RC_BITBLT) && 
	    T(memhdc = CreateCompatibleDC (hdc)) && 
	    T(mybm1  = CreateCompatibleBitmap (hdc, CS->sizex, CS->sizey))) {
		oldbm = SelectObject (memhdc, mybm1);
		FillRect (memhdc, &rect, GetStockObject (BLACK_BRUSH));
		SelectObject (memhdc, mybm1);
		SetPaletteEntries (hpal, 0, 256, ape);
		RealizePalette (memhdc);
		dev->npages = 1; /* double-bufferred */
	} else {
		if (mybm1) DeleteObject (mybm1);
		if (blt)
			MessageBox((HWND)0, (LPCSTR)"Cannot use BitBlt", (
					     LPCSTR)"Fly8 Message", MB_OK);
		memhdc = hdc;
		oldbm = 0L;
		dev->npages = 1;
	}

	SetROP2 (memhdc, R2_COPYPEN);

	plgpl = (LOGPALETTE *) 
	        malloc (sizeof(LOGPALETTE) + 256 * sizeof(PALETTEENTRY));

	plgpl->palNumEntries = 256;
	plgpl->palVersion = 0x300;

	for (i = 0, r = 0, g = 127, b = 127; i < 256;
		i++, r += 1, g += 1, b += 1) {
    		ape[i].peRed   = plgpl->palPalEntry[i].peRed = LOBYTE(r);
    		ape[i].peGreen = plgpl->palPalEntry[i].peGreen = LOBYTE(g);
    		ape[i].peBlue  = plgpl->palPalEntry[i].peBlue = LOBYTE(b);
    		ape[i].peFlags = plgpl->palPalEntry[i].peFlags = 0;
	}
	hpal = CreatePalette(plgpl);
	free(plgpl);
	oldhpal = SelectPalette (memhdc, hpal, FALSE);
	SetPaletteEntries (hpal, 0, 256, ape);
	RealizePalette (memhdc);

	dev->colors = GetDeviceCaps (memhdc, NUMCOLORS);
	if (dev->colors < 16) {
		for (i = 0; i < rangeof (st.colors); ++i)
			st.colors[i] = 1;
		st.colors[CC_BLACK] = 0;
		GrmSetPalette (CC_BLACK, C_BLACK);
		GrmSetPalette (CC_WHITE, C_WHITE);
	}

	usingWinG = 0;

	Gr->flags |= INITED;

	return (0);
}

static int FAR
GrmInitS (DEVICE *dev, char * options)
{
        return (GrmInit (dev, options, 0));
}
        
static int FAR
GrmInitB (DEVICE *dev, char * options)
{
        return (GrmInit (dev, options, 1));
}
        
static void FAR
GrmTerm (DEVICE *dev)
{
	int i;
	
	if (!(Gr->flags & INITED))
		return;
	Gr->flags &= ~INITED;

	for (i = 0; i < 256; ++i)
		if (pens[i])
			DeleteObject (pens[i]);	/* delete Pens */
	
	SelectPalette (hdc, oldhpal, FALSE);
	DeleteObject (hpal);
	
	if (oldbm) {
		SelectObject (memhdc, oldbm);
		DeleteObject (mybm1);
		DeleteDC (memhdc);
	}
	
	if (ghWndMain)
		ReleaseDC (ghWndMain, hdc);

	hpal=0;
	ghWndMain=0;
	oldbm=0;
	memhdc=0;
	mybm1=0;
	hdc=0;

	dev = dev;
}

static void
GrmFlush (void)
{
	if (resetSSize) {
	        GrmChangeSize ();
	        resetSSize = 0;
	} else
		BitBlt (hdc, 0, 0, CS->sizex, CS->sizey, memhdc, 0, 0, SRCCOPY); 
}

static void
GrmFlushS (void)
{
	if (resetSSize) {
	        GrmChangeSize ();
	        resetSSize = 0;
	}
}

struct GrDriver NEAR GrMSWinB = {
	"GrBitBlt",
	0,
	NULL,	/* extra */
	0,
	GrmInitB,
	GrmTerm,
	GrmMoveTo,
	GrmDrawTo,
	GrmSetVisual,
	GrmSetActive,
#if USE_CLEAR
	GrmClearWin,
#else
	0,	
#endif
	GrmSetWriteMode,
	GrmSetPalette,
	0,	/* ellipse */
	GrmFlush,
	0	/* Shutters */
};

struct GrDriver NEAR GrMSWinS = {
	"GrGDI",
	0,
	NULL,	/* extra */
	0,
	GrmInitS,
	GrmTerm,
	GrmMoveTo,
	GrmDrawTo,
	GrmSetVisual,
	GrmSetActive,
#if USE_CLEAR
	GrmClearWin,
#else
	0,	
#endif
	GrmSetWriteMode,
	GrmSetPalette,
	0,	/* ellipse */
	GrmFlushS,
	0	/* Shutters */
};
#undef INITED
