/*
 * VGA 16 colour mode support routines
 *
 * Copyright (C) Evan Harris, 1993, 1994
 *
 * Permission is granted to freely redistribute and modify this code,
 * providing the author(s) get credit for having written it.
 */

#include <stdlib.h>
#include <vga.h>
#include "vga16.h"
#include "font.h"


static unsigned char *vga16mem = NULL;
static int vga16width, vga16height;


/*
 * To be called after vga_setmode()
 */

int
vga16_init()
{
    vga_modeinfo *mode;

    mode = vga_getmodeinfo(vga_getcurrentmode());
    if (mode->colors != 16) {
	return -1;
    }
    if (vga16mem != NULL) {
	free(vga16mem);
    }
    vga16mem = calloc(mode->width * mode->height / 2, sizeof(unsigned char));
    if (vga16mem == NULL) {
	return -1;
    }
    vga16width = mode->width;
    vga16height = mode->height;
    return 0;
}


void
vga16_drawscansegment(unsigned char *colors, int x, int y, int length)
{
    unsigned char *p, *c;
    int l, changed;
    
    p = &vga16mem[(y * vga16width + x) >> 1];
    c = colors;
    l = length;
    changed = 0;
    while (l > 0) {
	if (*p != (*c << 4 | *(c + 1))) {
	    *p = *c << 4 | *(c + 1);
	    changed = 1;
	}
	p++;
	c += 2;
	l -= 2;
    }

    if (changed) {
	vga_drawscansegment(colors, x, y, length);
    }
}


void
vga16_drawscanline(int y, unsigned char *colors)
{
    vga16_drawscansegment(colors, 0, y, vga16width);
}


static void
vga16_putchar(int x, int y, char c, int fg, int bg)
{
    int i, j, mask, vp, colour;

    y -= FONT_LINES - 1;
    for (i = FONT_LINES * c; i < FONT_LINES * (c + 1); i++) {
	mask = 0x80;
	for (j = x; j < x + 8; j++) {
	    if (font_bits[i] & mask) {
		colour = fg;
	    } else {
		colour = bg;
	    }
	    
	    vp = (y * vga16width + j) / 2;

	    if (j & 1) {
		if ((vga16mem[vp] & 0x0f) != colour) {
		    vga16mem[vp] = (vga16mem[vp] & 0xf0) | colour;
		    vga_setcolor(colour);
		    vga_drawpixel(j, y);
		}
	    } else {
		if ((vga16mem[vp] >> 4) != colour) {
		    vga16mem[vp] = (colour << 4) | (vga16mem[vp] & 0x0f);
		    vga_setcolor(colour);
		    vga_drawpixel(j, y);
		}
	    }

	    mask = mask >> 1;
	}
	y++;
    }
}


void
vga16_text(int x, int y, char *s, int fg, int bg)
{
    int startx = x;

    while (*s != '\0') {
	switch (*s) {
	  case '\n':
	    y += FONT_LINES;
	    x = startx;
	    break;
	  case '\b':
	    x -= FONT_WIDTH;
	    break;
	  default:
	    vga16_putchar(x, y, *s, fg, bg);
	    x += FONT_WIDTH;
	    break;
	}
	s++;
    }
}


void
vga16_setpixel(int color, int x, int y)
{
    unsigned char *vp = &vga16mem[(y * vga16width + x) >> 1];

    if (x & 1) {
	if ((*vp & 0x0f) == color) {
	    return;
	}
	*vp = (*vp & 0xf0) | color;
    } else {
	if ((*vp >> 4) == color) {
	    return;
	}
	*vp = (color << 4) | (*vp & 0x0f);
    }

    vga_setcolor(color);
    vga_drawpixel(x, y);
}


int
vga16_getpixel(int x, int y)
{
    unsigned char *vp = &vga16mem[(y * vga16width + x) >> 1];

    if (x & 1) {
	return *vp & 0x0f;
    } else {
	return *vp >> 4;
    }
}
