/*
 *
 * Simple VFS definitions for fileio.
 *
 * Authors: Marco van Wieringen <v892273@si.hhs.nl> <mvw@mcs.nl.mugnet.org>
 *          Edvard Tuinder <v892231@si.hhs.nl> <ed@delirium.nl.mugnet.org>
 *
 * Version: $Id: fileio.c,v 1.2 1994/03/12 16:30:06 mvw Exp mvw $
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/fcntl.h>

#ifdef CONFIG_QUOTA
int lookup(struct inode *, const char *, int, struct inode **);

int vfs_write(struct inode *ino, struct file *filp, char *addr, size_t bytes)
{
   size_t written;
   u_long cur_blocks, wanted_blocks = 0, avail_blocks = 0;

   if (S_ISREG(ino->i_mode)) {
      cur_blocks = isize_to_blocks(ino->i_size, ino->i_blksize);
      if ((filp->f_pos + bytes) > ino->i_size) {
         wanted_blocks = isize_to_blocks(filp->f_pos + bytes,
                         ino->i_blksize) - cur_blocks;
         if (wanted_blocks && quota_alloc(ino, 0, wanted_blocks,
                                          &avail_blocks) == NO_QUOTA)
            return -EDQUOT;
         if (wanted_blocks && (avail_blocks < wanted_blocks))
            bytes = blocks_to_isize((cur_blocks + avail_blocks),
                    ino->i_blksize) - filp->f_pos;
      }
      if ((written = filp->f_op->write(ino, filp, addr, bytes)) != bytes) {
         quota_remove(ino, 0, avail_blocks -
                     (isize_to_blocks(ino->i_size, ino->i_blksize) -
                      isize_to_blocks((ino->i_size - written),
                      ino->i_blksize)));
      }
      if (wanted_blocks && (avail_blocks < wanted_blocks))
         return -EDQUOT;

      return written;
   } else
      return filp->f_op->write(ino, filp, (char *)addr, bytes);
}

int vfs_create(struct inode *dir, const char *basename,
               int namelen, int mode, struct inode **res_ino)
{
   int error;
   struct inode dummy_ino;

   memset(&dummy_ino, 0, sizeof(struct inode));
   dummy_ino.i_dev = dir->i_dev;
   dummy_ino.i_uid = current->euid;
   dummy_ino.i_gid = current->egid;
   getinoquota(&dummy_ino, -1);

   if (quota_alloc(&dummy_ino, 1, 0, (u_long *)0) == NO_QUOTA) {
      putinoquota(&dummy_ino);
      return -EDQUOT;
   }
   error = dir->i_op->create(dir, basename, namelen, mode, res_ino);
   if (error)
      quota_remove(&dummy_ino, 1, 0);
   putinoquota(&dummy_ino);

   return error;
}

int vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;
   size_t old_isize;

   old_isize = ino->i_size;
   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   ino->i_ctime = ino->i_mtime = CURRENT_TIME;
   ino->i_dirt = 1;
   if ((error = notify_change(NOTIFY_SIZE, ino)))
      return error;
   getinoquota(ino, -1);
   quota_remove(ino, 0, isize_to_blocks(old_isize, ino->i_blksize));
   putinoquota(ino);

   return error;
}

int vfs_mknod(struct inode *dir, const char *basename,
              int namelen, int mode, dev_t dev)
{
   int error;
   struct inode dummy_ino;

   memset(&dummy_ino, 0, sizeof(struct inode));
   dummy_ino.i_dev = dir->i_dev;
   dummy_ino.i_uid = current->euid;
   dummy_ino.i_gid = current->egid;
   getinoquota(&dummy_ino, -1);

   if (quota_alloc(&dummy_ino, 1, 0, (u_long *)0) == NO_QUOTA) {
      putinoquota(&dummy_ino);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   error = dir->i_op->mknod(dir, basename, namelen, mode, dev);
   if (error)
      quota_remove(&dummy_ino, 1, 0);
   putinoquota(&dummy_ino);
   iput(dir);

   return error;
}

int vfs_mkdir(struct inode *dir, const char *basename, int namelen, int mode)
{
   int error;
   struct inode dummy_ino;

   memset(&dummy_ino, 0, sizeof(struct inode));
   dummy_ino.i_dev = dir->i_dev;
   dummy_ino.i_uid = current->euid;
   dummy_ino.i_gid = current->egid;
   getinoquota(&dummy_ino, -1);

   if (quota_alloc(&dummy_ino, 1, 1, (u_long *)0) == NO_QUOTA) {
      putinoquota(&dummy_ino);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   error = dir->i_op->mkdir(dir, basename, namelen, mode);
   if (error)
      quota_remove(&dummy_ino, 1, 1);
   putinoquota(&dummy_ino);
   iput(dir);

   return error;
}

int vfs_rmdir(struct inode *dir, const char *basename, int namelen)
{
   int error;
   struct inode dummy_ino, *ino;

   /*
    * Need inode entry of directory for quota operations
    */
   dir->i_count++;
   if ((error = lookup(dir, basename, namelen, &ino))) {
      iput(dir);
      return error;
   }
   dummy_ino = *ino;
   iput(ino);
   getinoquota(&dummy_ino, -1);
   if (!(error = dir->i_op->rmdir(dir, basename, namelen)))
      quota_remove(&dummy_ino, 1, 1);
   putinoquota(&dummy_ino);

   return error;
}

int vfs_unlink(struct inode *dir, const char *basename, int namelen)
{
   int error;
   struct inode dummy_ino, *ino;

   /*
    * Need inode info of to remove file for quota operations.
    */
   dir->i_count++;
   if ((error = lookup(dir, basename, namelen, &ino))) {
      iput(dir);
      return error;
   }
   dummy_ino = *ino;
   iput(ino);
   getinoquota(&dummy_ino, -1);
   error = dir->i_op->unlink(dir, basename, namelen);
   /*
    * Remove blocks and inode. Only if link-count was 1 !
    */
   if (!error && dummy_ino.i_nlink == 1)
      quota_remove(&dummy_ino, 1, isize_to_blocks(dummy_ino.i_size,
                   dummy_ino.i_blksize));
   putinoquota(&dummy_ino);

   return error;
}

int vfs_symlink(struct inode *dir, const char *basename, int namelen,
                const char *oldname)
{
   int error;
   struct inode dummy_ino;

   memset(&dummy_ino, 0, sizeof(struct inode));
   dummy_ino.i_dev = dir->i_dev;
   dummy_ino.i_uid = current->euid;
   dummy_ino.i_gid = current->egid;
   getinoquota(&dummy_ino, -1);

   if (quota_alloc(&dummy_ino, 1, 1, (u_long *)0) == NO_QUOTA) {
      putinoquota(&dummy_ino);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   if (!(error = dir->i_op->symlink(dir, basename, namelen, oldname)))
      quota_remove(&dummy_ino, 1, 1);
   putinoquota(&dummy_ino);
   iput(dir);

   return error;
}

int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   int error;
   uid_t olduid, oldgid;

   olduid = ino->i_uid;
   oldgid = ino->i_gid;
   getinoquota(ino, -1);
   if (quota_transfer(ino, uid, gid, 1, isize_to_blocks(ino->i_size,
                      ino->i_blksize)) == NO_QUOTA) {
      putinoquota(ino);
      return -EDQUOT;
   }
   ino->i_uid = uid;
   ino->i_gid = gid;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   if ((error = notify_change(NOTIFY_UIDGID, ino)) != 0)
      quota_transfer(ino, olduid, oldgid, 1,
                     isize_to_blocks(ino->i_size, ino->i_blksize));
   putinoquota(ino);
   return error;
}

int vfs_rename(struct inode *old_dir, const char *old_base, int old_len,
               struct inode *new_dir, const char *new_base, int new_len)
{
   int error;
   struct inode *ino, dummy_ino;

   /*
    * Check if target file already exists, drop quota of file if
    * it already exists and is overwritten. Extra check needed for
    * renames of file to the same name.
    */
   new_dir->i_count++;
   if (!lookup(new_dir, new_base, new_len, &ino)) {
      dummy_ino = *ino;
      iput(ino);
      error = old_dir->i_op->rename(old_dir, old_base, old_len, 
                                    new_dir, new_base, new_len);
      if (!error && (old_dir != new_dir || strcmp(old_base, new_base))) {
         getinoquota(&dummy_ino, -1);
         quota_remove(&dummy_ino, 1,
                      isize_to_blocks(dummy_ino.i_size, dummy_ino.i_blksize));
         putinoquota(&dummy_ino);
      }
   }
   else
      error = old_dir->i_op->rename(old_dir, old_base, old_len, 
                                    new_dir, new_base, new_len);
   return error;
}

#else /* CONFIG_QUOTA */

int vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;

   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   if ((error = notify_change(NOTIFY_SIZE, ino))) {
      return error;
   }
   ino->i_dirt = 1;

   return error;
}

int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   ino->i_uid = uid;
   ino->i_gid = gid;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   return notify_change(NOTIFY_UIDGID, ino);
}

#endif /* CONFIG_QUOTA */

