/*
 *  linux/fs/file.c
 *
 *  Copyright (C) 1991, 1992, 1993, 1994  Linus Torvalds
 *
 *  Changed a bit by M. v Wieringen to not export the filetable variables, but
 *  put all related functions in this file. No one needs to have access directly
 *  to our datatypes. Just a little bit of an ADT as far as we can speak of one
 *  here.
 */

#include <linux/fs.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/config.h>

static struct file *first_file;
static int nr_files = 0;

static void insert_file_free(struct file *filp)
{
   filp->f_next = first_file;
   filp->f_prev = first_file->f_prev;
   filp->f_next->f_prev = filp;
   filp->f_prev->f_next = filp;
   first_file = filp;
}

static void remove_file_free(struct file *filp)
{
   if (first_file == filp)
      first_file = first_file->f_next;
   if (filp->f_next)
      filp->f_next->f_prev = filp->f_prev;
   if (filp->f_prev)
      filp->f_prev->f_next = filp->f_next;
   filp->f_next = filp->f_prev = NULL;
}

static void put_last_free(struct file *filp)
{
   remove_file_free(filp);
   filp->f_prev = first_file->f_prev;
   filp->f_prev->f_next = filp;
   filp->f_next = first_file;
   filp->f_next->f_prev = filp;
}

static void grow_files(void)
{
   struct file *filp;
   int i;

   filp = (struct file *) get_free_page(GFP_KERNEL);
   if (!filp)
      return;
   nr_files+=i= PAGE_SIZE/sizeof(struct file);
   if (!first_file)
      filp->f_next = filp->f_prev = first_file = filp++, i--;
   for (; i ; i--)
      insert_file_free(filp++);
}

/*
 * External references.
 */
unsigned long file_table_init(unsigned long start, unsigned long end)
{
   first_file = (struct file *)NULL;
   return start;
}

struct file *get_empty_filp(void)
{
   int i;
   struct file *filp;

   if (!first_file)
      grow_files();
repeat:
   for (filp = first_file, i = 0; i < nr_files; i++, filp = filp->f_next)
      if (!filp->f_count) {
         remove_file_free(filp);
         memset(filp, 0, sizeof(*filp));
         put_last_free(filp);
         filp->f_count = 1;
         return filp;
      }
   if (nr_files < NR_FILE) {
      grow_files();
      goto repeat;
   }
   return NULL;
}

int fs_may_remount_ro(dev_t dev)
{
   struct file *filp;
   int i;

   /* Check that no files are currently opened for writing. */
   for (filp = first_file, i = 0; i < nr_files; i++, filp = filp->f_next) {
      if (!filp->f_count || !filp->f_inode || filp->f_inode->i_dev != dev)
         continue;
      if (S_ISREG(filp->f_inode->i_mode) && (filp->f_mode & 2))
         return 0;
   }
   return 1;
}

/*
 * Functions called from tty_io.c in the tty driver. Placed here
 * to avoid a global file_table. This should be safer then using
 * a global file_table.
 */
void set_tty_fops(dev_t dev, dev_t console, struct file_operations *tty_fops,
                                            struct file_operations *fops)
{
   struct file *filp;
   int i;

   for (filp = first_file, i = 0; i < nr_files; i++, filp = filp->f_next) {
      if (!filp->f_count || filp->f_rdev != dev)
         continue;
      if (filp->f_inode && filp->f_inode->i_rdev == console)
         continue;
      if (filp->f_op != tty_fops)
         continue;
      filp->f_op = fops;
   }
}

#ifdef CONFIG_QUOTA
void add_dquot_ref(dev_t dev, short type)
{
   struct file *filp;
   int i;

   /* Check files that are currently opened for writing. */
   for (filp = first_file, i = 0; i < nr_files; i++, filp = filp->f_next) {
      if (!filp->f_count || !filp->f_inode || filp->f_inode->i_dev != dev)
         continue;
      if (filp->f_inode->i_writecount > 0) {
         getinoquota(filp->f_inode, type);
         if (filp->f_inode->i_dquot[type] != NODQUOT)
            filp->f_inode->i_dquot[type]->dq_count +=
            (filp->f_inode->i_writecount - 1);
      }
   }
}

void reset_dquot_ptrs(dev_t dev, short type)
{
   struct file *filp;
   int i;

   for (filp = first_file, i = 0; i < nr_files; i++, filp = filp->f_next)
      if (filp->f_count && filp->f_inode &&
          filp->f_inode->i_writecount && filp->f_inode->i_dev == dev)
         filp->f_inode->i_dquot[type] = NODQUOT;
}
#endif

