#ifndef _MISC_H
#define _MISC_H

#include <math.h>

/* ---------------------------------------------------------------------- */

/*
 * Hamming weight (number of bits that are ones).
 */
extern inline unsigned int hweight32(unsigned int w) 
{
	w = (w & 0x55555555) + ((w >>  1) & 0x55555555);
	w = (w & 0x33333333) + ((w >>  2) & 0x33333333);
	w = (w & 0x0F0F0F0F) + ((w >>  4) & 0x0F0F0F0F);
	w = (w & 0x00FF00FF) + ((w >>  8) & 0x00FF00FF);
	w = (w & 0x0000FFFF) + ((w >> 16) & 0x0000FFFF);
	return w;
}

extern inline unsigned int hweight16(unsigned short w)
{
	w = (w & 0x5555) + ((w >> 1) & 0x5555);
	w = (w & 0x3333) + ((w >> 2) & 0x3333);
	w = (w & 0x0F0F) + ((w >> 4) & 0x0F0F);
	w = (w & 0x00FF) + ((w >> 8) & 0x00FF);
	return w;
}

extern inline unsigned int hweight8(unsigned char w)
{
	w = (w & 0x55) + ((w >> 1) & 0x55);
	w = (w & 0x33) + ((w >> 2) & 0x33);
	w = (w & 0x0F) + ((w >> 4) & 0x0F);
	return w;
}

/* ---------------------------------------------------------------------- */

/*
 * Reverse order of bits.
 */
extern inline unsigned int rbits32(unsigned int w)
{
	w = ((w >>  1) & 0x55555555) | ((w <<  1) & 0xAAAAAAAA);
	w = ((w >>  2) & 0x33333333) | ((w <<  2) & 0xCCCCCCCC);
	w = ((w >>  4) & 0x0F0F0F0F) | ((w <<  4) & 0xF0F0F0F0);
	w = ((w >>  8) & 0x00FF00FF) | ((w <<  8) & 0xFF00FF00);
	w = ((w >> 16) & 0x0000FFFF) | ((w << 16) & 0xFFFF0000);
	return w;
}

extern inline unsigned short rbits16(unsigned short w)
{
	w = ((w >> 1) & 0x5555) | ((w << 1) & 0xAAAA);
	w = ((w >> 2) & 0x3333) | ((w << 2) & 0xCCCC);
	w = ((w >> 4) & 0x0F0F) | ((w << 4) & 0xF0F0);
	w = ((w >> 8) & 0x00FF) | ((w << 8) & 0xFF00);
	return w;
}

extern inline unsigned char rbits8(unsigned char w)
{
	w = ((w >> 1) & 0x55) | ((w << 1) & 0xFF);
	w = ((w >> 2) & 0x33) | ((w << 2) & 0xCC);
	w = ((w >> 4) & 0x0F) | ((w << 4) & 0xF0);
	return w;
}

/* ---------------------------------------------------------------------- */

/*
 * Integer base-2 logarithm
 */
extern inline int log2(unsigned int x)
{
	int y = 0;

	x >>= 1;

	while (x) {
		x >>= 1;
		y++;
	}

	return y;
}

/* ---------------------------------------------------------------------- */

/*
 * Gray encoding and decoding (8 bit)
 */
extern inline unsigned char grayencode(unsigned char data)
{
	unsigned char bits = data;

	bits ^= data >> 1;
	bits ^= data >> 2;
	bits ^= data >> 3;
	bits ^= data >> 4;
	bits ^= data >> 5;
	bits ^= data >> 6;
	bits ^= data >> 7;

	return bits;
}

extern inline unsigned char graydecode(unsigned char data)
{
	return data ^ (data >> 1);
}

/* ---------------------------------------------------------------------- */

/*
 * Hamming window function
 */
extern inline double hamming(double x)
{
        return 0.54 - 0.46 * cos(2 * M_PI * x);
}

/* ---------------------------------------------------------------------- */

/*
 * Sinc etc...
 */
extern inline double sinc(double x)
{
	return (fabs(x) < 1e-10) ? 1.0 : (sin(M_PI * x) / (M_PI * x));
}

extern inline double cosc(double x)
{
	return (fabs(x) < 1e-10) ? 0.0 : ((1.0 - cos(M_PI * x)) / (M_PI * x));
}

/* ---------------------------------------------------------------------- */

extern inline float clamp(float x, float min, float max)
{
	return (x < min) ? min : ((x > max) ? max : x);
}

/* ---------------------------------------------------------------------- */

#endif				/* _MISC_H */
