#define XGRABSC_C
/*========================================================================
 *
 * Name - xgrabsc.c
 *
 * ccs version:	1.40
 *
 * ccsid:	@(#)xgrabsc.c	1.40 - 08/16/95 08:16:44
 * from: 	ccs/s.xgrabsc.c
 * date: 	11/07/95 11:31:40
 *
 * Copyright (c) 1990-95 Bruce Schuchardt.
 * Read the file cpyright.h for full copyright information.
 *
 * XmuClientWindow and TryChildren are Copyright 1989 by the Massachusetts
 * Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this [XmuClientWindow]
 * software and its
 * documentation for any purpose and without fee is hereby granted, provided 
 * that the above copyright notice appear in all copies and that both that 
 * copyright notice and this permission notice appear in supporting 
 * documentation, and that the name of M.I.T. not be used in advertising
 * or publicity pertaining to distribution of the software without specific, 
 * written prior permission. M.I.T. makes no representations about the 
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * Description:
 *
 * xgrabsc - grab screen images and store in files
 *
 *========================================================================
 */

#include "xgrabsc.h"



static Window xgTryChildren();

/* versions of XmuGetClientWindow and screenNumberOfScreen.
 * GetClientWindow is used so Xmu need not be linked (causes problems on
 *  some platforms and configurations if Xt is not used)
 * ScreenNumberOfScreen is used because some X implementations do not
 *  include this function (HP in particular).
 *
 * The following three functions are from the X11R4 MIT distribution and
 * are copyrighted by MIT (see the file header)
 */

static Window GetClientWindow (win)
    Window win;
{
    Atom WM_STATE;
    Atom type = None;
    int format;
    unsigned long nitems, after;
    unsigned char *data;
    Window inf;

    WM_STATE = XInternAtom(hDisplay, "WM_STATE", True);
    if (!WM_STATE)
	return win;
    XGetWindowProperty(hDisplay, win, WM_STATE, 0, 0, False, AnyPropertyType,
		       &type, &format, &nitems, &after, &data);
    if (type)
	return win;
    inf = xgTryChildren(win, WM_STATE);
    if (!inf)
	inf = win;
    return inf;
}

static Window xgTryChildren (win, WM_STATE)
    Window win;
    Atom WM_STATE;
{
    Window root, parent;
    Window *children;
    unsigned int nchildren;
    unsigned int i;
    Atom type = None;
    int format;
    unsigned long nitems, after;
    unsigned char *data;
    Window inf = 0;

    if (!XQueryTree(hDisplay, win, &root, &parent, &children, &nchildren))
	return 0;
    for (i = 0; !inf && (i < nchildren); i++) {
	XGetWindowProperty(hDisplay, children[i], WM_STATE, 0, 0, False,
			   AnyPropertyType, &type, &format, &nitems,
			   &after, &data);
	if (type)
	    inf = children[i];
    }
    for (i = 0; !inf && (i < nchildren); i++)
	inf = xgTryChildren(children[i], WM_STATE);
    if (children) XFree((char *)children);
    return inf;
}



int ScreenNumberOfScreen (scr)
    register Screen *scr;
{
    register Display *dpy = scr->display;
    register int i;

    for (i = 0; i < ScreenCount(dpy); i++) {
	if (scr == ScreenOfDisplay(dpy,i)) return i;
    }
    return -1;
}


Window windowNamed(startingWindow, name)
  Window startingWindow;
  char *name;
{
  int i;
  unsigned int nchildren;
  Window root, parent, *children, found;
  char *windowName;
  
  if (!XQueryTree(hDisplay, startingWindow, &root, &parent, &children, &nchildren)) {
    fprintf (stderr, "%s: unable to find top level windows\n", programName);
    exit (3);
  }
  else {
    found = (Window)NULL;
    for (i=0; i<nchildren; i++) {
      XFetchName(hDisplay, children[i], &windowName);
      if (windowName) {
	if (strcmp(name, windowName) == 0) {
	  found = children[i];
	  break;
	}
      }
      else {
	if (found = windowNamed(children[i], name))
	  break;
      }
    }
    if (children)
      XFree((char *)children);
  }

  return found;
}




xgrabsc(argc, argv, env)
  int argc;
  char *argv[];  /* must have at least argv[0] == programName */
  char *env;
{
  char        *args[100], *arg;
  extern char *optarg;
  FILE        *outfile;
  XRectangle  xrect, xoffset;
  XWindowAttributes xwa;
  imageInfo   image;
  int         argn, argi, polarity;
  int	      cmdi, cmdargi;
  int         doAnd;
  int         doOr;
  int         doReverse;
  int         noBell;
  int         psColor;
  int         brighten;
  int         grabServer;
  int         compress;
  int         sleepSeconds;
  int         postSelectSleepSeconds;
  int         andBits;
  int         orBits;
  int         encapsulate;
  int         subrectSource;
  Window      sourceWindow, childWindow, clientWindow, wmWindow, ignored;
  Window      colorWin, dumyWin, imageWin;
  char       *ptr;
  char       *display;
  int         i, x, y;
  int         brightenFactor;
  sourceType  source;
  formatType  outputFormat;
  int         xpmFormat;
  int         landscape;
  int         binary;
  int	      borders;
  int         frame;
  int	      checkLimits;
  int	      preview;
  int         onlyEpsi;
  int         xwdxy;
  int         absx, absy;
  int         dwidth, dheight;


  outfile     = stdout;
  outfileName = NULL;
  display     = NULL;
  programName = argv[0];

  noBell       = FALSE;
  brighten     = FALSE;
  compress     = TRUE;
  DitherKind   = NO_DITHER;
  doAnd        = FALSE;
  doOr         = FALSE;
  doReverse    = FALSE;
  encapsulate  = FALSE;
  ForceBitmap  = FALSE;
  grabServer   = TRUE;
  Halftone     = FALSE;
  landscape    = FALSE;
  binary       = FALSE;
  subrectSource = FALSE;
  sleepSeconds = 0;
  postSelectSleepSeconds = 0;
  verbose      = FALSE;
  borders      = TRUE;
  frame        = TRUE;
#ifdef NO_PRINTER_MEMORY_CHECKS
  checkLimits  = FALSE;
#else
  checkLimits  = TRUE;
#endif
  preview      = FALSE;
  needColorImageProc = DEFAULT_NEED_COLORIMAGE_PROC;

  source       = sourceRect;
  outputFormat = psFormat;
  psColor      = FALSE;
  onlyEpsi     = FALSE;
  xwdxy	       = FALSE;
  colorWin     = None;

  /* merge environment options and command line options */
  args[0] = programName;
  if (env != NULL) {
    args[1] = env;
    for (argn=2; argn<100  &&
                (args[argn]=(char *)strchr(args[argn-1], ' ')) != NULL;
	        argn++) {
      /* remove leading white space */
      while (*args[argn] == ' ' || *args[argn] == 9) {
        *(args[argn]) = '\0';
        args[argn]++;
      }
      if (*args[argn] == '|' || *args[argn] == '>') /* dbx leaves these in the cmd line */
        break;
    }
  }
  else
    argn = 1;

  for (i=1; i<argc && argn<100; argn++, i++)
    args[argn] = argv[i];

  
  for (argi=1; argi<argn; argi++) {
    arg = args[argi];

    polarity = 1;
    if (arg[0] == '-') {
      arg++;
      if (arg[0] == '-') {
        arg++;
        polarity = 0;
      }
      else if (arg[0] == 'n' && arg[1] == 'o') {
        arg += 2;
	polarity = 0;
      }
    }

    for (cmdi=0; cmdi<numCmds; cmdi++) {
      if (strcmp(arg, commands[cmdi].userstr) == 0)
        break;
    }
    if (cmdi >= numCmds) {
      fprintf(stderr, "%s: unknown option '%s'\n", programName, arg);
      exit(3);
    }

    cmdargi = argi+1;
    argi += commands[cmdi].numargs;
    if (argi >= argn) {
      fprintf(stderr, "%s: not enough arguments for '%s'\n", programName, arg);
      exit(3);
    }

    switch (commands[cmdi].command) {
      case CMD_DISPLAY:
        display = args[cmdargi];
        break;
      case CMD_BELL:
        noBell = !polarity;
        break;
      case CMD_GRABSERVER:
        grabServer = polarity;
        break;
      case CMD_OUTPUT:
        outfileName = args[cmdargi];
        break;
      case CMD_PRESLEEP:
        sleepSeconds = atoi(args[cmdargi]);
        if (sleepSeconds < 0) sleepSeconds = 0;
        break;
      case CMD_POSTSLEEP:
        postSelectSleepSeconds = atoi(args[cmdargi]);
        if (postSelectSleepSeconds < 0) postSelectSleepSeconds = 0;
        break;
      case CMD_VERBOSE:
        verbose = polarity;
        break;
      case CMD_BORDERS:
        borders = polarity;
	break;
      case CMD_FRAME:
        frame = polarity;
	break;


      case CMD_SOURCE_KEY:
        source = sourceKey;
        grabServer = FALSE;
	break;
      case CMD_SOURCE_COORDS:
      { int x,y,parseResult;
        unsigned int w,h;
	source = sourceCoords;
	x = y = w = h = 0;
	parseResult = XParseGeometry(args[cmdargi], &x, &y, &w, &h);
	if ((parseResult & (WidthValue | HeightValue)) != (WidthValue | HeightValue)) {
	  fprintf(stderr, "%s: coordinate rectangle underspecified\n", programName);
	  exit(3);
	}
	xrect.width = w; 
	xrect.height = h;
	xrect.x = x;  
	xrect.y = y;
	break;
      }

      case CMD_SOURCE_ID:
        break;
      case CMD_SOURCE_RECT:
	source = sourceRect;
	break;
      case CMD_SOURCE_ROOT:
        source = sourceRoot;
        break;
      case CMD_SOURCE_WD:
        source = sourceWd;
        break;

      case CMD_OFFSET_COORDS:
      { int x,y,parseResult;
        unsigned int w,h;
	subrectSource = TRUE;
	x = y = w = h = 0;
	parseResult = XParseGeometry(args[cmdargi], &x, &y, &w, &h);
	xoffset.width = w; 
	xoffset.height = h;
	xoffset.x = x;  
	xoffset.y = y;
	break;
      }

      case CMD_COLORWIN:
        break;

      case CMD_AND:
        doAnd = polarity;
	if (doAnd)
          andBits = atoi(args[cmdargi]);
        break;
      case CMD_OR:
        doOr = polarity;
	if (doOr)
          orBits = atoi(args[cmdargi]);
        break;
      case CMD_NOT:
        doReverse = polarity;
	break;
      case CMD_BRIGHTEN:
        brighten = polarity;
	if (brighten) {
          brightenFactor = atoi(args[cmdargi]);
          if (brightenFactor <= 0) {
            fprintf(stderr, "%s: brightening factor must be a positive number\n",
              programName);
            exit(3);
          }
	  if (brightenFactor != 100)
            brighten = TRUE;
	  else
	    brighten = FALSE;
	}
        break;

      case CMD_DITHER_MAP:
        ForceBitmap = TRUE;
        Halftone = FALSE;
        break;
      case CMD_DITHER_MATRIX:
        DitherKind = MATRIX_DITHER;
        Halftone = TRUE;
        ForceBitmap = FALSE;
        break;
      case CMD_DITHER_FS:
        if (!polarity) {
          DitherKind = NO_DITHER;
	  Halftone = ForceBitmap = FALSE;
        }
        else {
          DitherKind = FS_DITHER;
          Halftone = TRUE;
          ForceBitmap = FALSE;
        }
        break;
      case CMD_DITHER_HALFTONE:
        DitherKind = MATRIX_HALFTONE;
        Halftone = TRUE;
        ForceBitmap = FALSE;
        break;



      case CMD_OUTPUT_PS:
        psColor = FALSE;
        outputFormat = psFormat;
        break;
      case CMD_OUTPUT_CPS:
        psColor = polarity;
        outputFormat = psFormat;
	break;
      case CMD_OUTPUT_SIMPLE:
        outputFormat = simpleFormat;
        break;
      case CMD_OUTPUT_XWD:
        outputFormat = xwdFormat;
	xwdxy = FALSE;
        break;
      case CMD_OUTPUT_XWDXY:
        outputFormat = xwdFormat;
	xwdxy = TRUE;
        break;
      case CMD_OUTPUT_PIXMAP:
        xpmFormat = 1;
        outputFormat = pixmapFormat;
        break;
      case CMD_OUTPUT_XPM2:
        if (polarity) xpmFormat = 2;
	else if (xpmFormat == 2) xpmFormat = 1;
	outputFormat = pixmapFormat;
	break;
      case CMD_OUTPUT_XPM3:
        if (polarity) xpmFormat = 3;
	else if (xpmFormat == 3) xpmFormat = 1;
	outputFormat = pixmapFormat;
	break;

      case CMD_OUTPUT_PUZZLE:
        outputFormat = puzzleFormat;
        break;

      case CMD_OUTPUT_PPM:
        outputFormat = ppmFormat;
        break;

      case CMD_LANDSCAPE:
        landscape = polarity;
	outputFormat = psFormat;
        break;
      case CMD_BIN:
        binary = polarity;
        outputFormat = psFormat;
	break;
      case CMD_COMPRESS:
        compress = polarity;
        outputFormat = psFormat;
        break;
      case CMD_EPS:
        encapsulate = polarity;
        outputFormat = psFormat;
	break;
      case CMD_PREVIEWONLY:
        onlyEpsi = polarity;
	preview = polarity;
        outputFormat = psFormat;
	break;
      case CMD_LIMIT:
        checkLimits = polarity;
        outputFormat = psFormat;
	break;
      case CMD_PREVIEW:
        preview = polarity;
        outputFormat = psFormat;
	break;  
      case CMD_PAGE:
        outputFormat = psFormat;
        sscanf(args[cmdargi], "%lfx%lf-%lf-%lf",
	  &pageWidth, &pageHeight, &horizMargin, &vertMargin);
	horizInset = pageWidth - horizMargin - horizMargin;
	vertInset  = pageHeight - vertMargin - vertMargin;
        break;
      case CMD_COLORPROC:
        outputFormat = psFormat;
	needColorImageProc = polarity;
	break;
    }
  }


  if (!display) display = (char *)getenv("DISPLAY");
  hDisplay = XOpenDisplay(display);
  if (!hDisplay) {
    fprintf(stderr, "%s: could not open X display\n", programName);
    exit(3);
  }
  hScreen  = DefaultScreen(hDisplay);
  hRoot    = DefaultRootWindow(hDisplay);
#ifndef NO_VIRTUAL_WINDOW
  vRoot    = VirtualRootWindow(hDisplay, hScreen);
#else
  vRoot    = hRoot;
#endif

  /* process command line options that may require an X connection */
  for (argi=1; argi<argn; argi++) {
    arg = args[argi];

    polarity = 1;
    if (arg[0] == '-') {
      arg++;
      if (arg[0] == '-') {
        arg++;
        polarity = 0;
      }
      else if (arg[0] == 'n' && arg[1] == 'o') {
        arg += 2;
	polarity = 0;
      }
    }

    for (cmdi=0; cmdi<numCmds; cmdi++) {
      if (strcmp(arg, commands[cmdi].userstr) == 0)
        break;
    }

    cmdargi = argi+1;
    argi += commands[cmdi].numargs;

    switch (commands[cmdi].command) {
      case CMD_SOURCE_ID:
        source = sourceId;
        sourceWindow = 0;
	if ( '0'<=(args[cmdargi][0])  && (args[cmdargi][0])<='9') {
	  if (!sscanf(args[cmdargi], "%i", &sourceWindow)) {
            fprintf(stderr, "%s: invalid window id '%s'\n", programName, args[cmdargi]);
	    exit(3);
	  }
	}
	else {
	  sourceWindow = windowNamed(hRoot, args[cmdargi]);
	  if (!sourceWindow) {
	    fprintf(stderr, "%s: invalid window name '%s'\n", programName, args[cmdargi]);
	    exit(3);
	  }
        }
      case CMD_COLORWIN:
        colorWin = (Window)NULL;
	if ( '0'<=(args[cmdargi][0])  && (args[cmdargi][0])<='9') {
	  if (!sscanf(args[cmdargi], "%i", &colorWin)) {
            fprintf(stderr, "%s: invalid window id '%s'\n", programName, args[cmdargi]);
	    exit(3);
	  }
	}
	else {
	  colorWin = windowNamed(hRoot, args[cmdargi]);
	  if (!colorWin) {
	    fprintf(stderr, "%s: invalid window name '%s'\n", programName, args[cmdargi]);
	    exit(3);
	  }
        }
	break;
    }
  }

  if (verbose) {
    fprintf(stderr, "%s: xgrabsc version %s\n", programName, version);
    fprintf(stderr, "%s:         patchlevel %d\n", programName, patchLevel);
    fprintf(stderr, "%s:         %s\n\n", programName, Copyright);
  }


  /* sleep if asked to do so */
  if (sleepSeconds)
    sleep(sleepSeconds);

  if (verbose && grabServer) {
    fprintf(stderr, "%s: disabling server grabs for verbose mode\n", programName);
    grabServer = FALSE;
  }

  /* grab the screen if asked to do so */
  if (grabServer)
    XGrabServer(hDisplay);

  if (source == sourceId)
    childWindow = sourceWindow;
  else
    childWindow = (Window)NULL;

  sourceWindow = hRoot;

  switch (source) {
    case sourceKey:
      childWindow =
#ifdef SELECTION_MASK
          getWindowWhenKeyIsPressed(SELECTION_MASK)
#else
          getWindowWhenKeyIsPressed(ControlMask);
#endif
      if (!childWindow) {
        fprintf(stderr, "%s: unable to find source window\n", programName);
        XCloseDisplay(hDisplay);
        exit(3);
      }
      break;
    case sourceCoords:
      /* already got the xrect in the command line processing section */
      break;
    case sourceId:
      break;
    case sourceWd:
      /* grab the image from the root window so menus will show up on top
       * of the window */
      childWindow=getWindow();
      if (!childWindow) {
        fprintf(stderr, "%s: unable to find source window\n", programName);
        XCloseDisplay(hDisplay);
        exit(3);
      }
      break;
    case sourceRoot:
      xrect.x = xrect.y = 0;
      xrect.width  = DisplayWidth(hDisplay, hScreen);
      xrect.height = DisplayHeight(hDisplay, hScreen);
      break;
    case sourceRect:
    default:
      if (!getRectangle(&xrect)) {
        XCloseDisplay(hDisplay);
        exit(3);
      }
      break;
  }



  /* if frame removal has been requested, look for a "client" child window */
  clientWindow = (Window)NULL;
  if (!frame  &&
      (source == sourceKey || source == sourceWd || source == sourceId)  &&
      childWindow != hRoot) {
    /* look for a different client window */
    clientWindow = GetClientWindow(childWindow);
    if (clientWindow && clientWindow != childWindow) {
      if (verbose)
	fprintf(stderr, "%s: found subwindow 0x%x\n", programName, clientWindow);
      wmWindow = childWindow;
      childWindow = clientWindow;
    }
  }

  /* Get the root's attributes, including colormap, depth and visual.
   * This is overwritten if a child window is processed
   */
  if (!XGetWindowAttributes(hDisplay, hRoot, &xwa)) {
    fprintf(stderr, "%s: unable to get root window attributes\n", programName);
    XCloseDisplay(hDisplay);
    exit(3);
  }

  if ((source == sourceKey) ||
      source == sourceId ||
      source == sourceWd) {

    if (childWindow == hRoot) {
      xrect.x      = 0;
      xrect.y      = 0;
      xrect.width  = DisplayWidth(hDisplay, hScreen);
      xrect.height = DisplayHeight(hDisplay, hScreen);
    }
    else {

      /* take the child window (and optional client/wm window info) and
       * determine the portion of the root window that should be grabbed */

      if (!XGetWindowAttributes(hDisplay, childWindow, &xwa)) {
        fprintf(stderr, "%s: unable to get window coordinates\n", programName);
        XCloseDisplay(hDisplay);
        exit(3);
      }

      /* get the correct screen and root window for the selected window,
       * and if it isn't the default, muck with the global state of the
       * program a bit
       */
      i = XScreenNumberOfScreen(xwa.screen);
      if (i != hScreen) {
        hRoot   = xwa.root;
        hScreen = i;
#ifndef NO_VIRTUAL_WINDOW
        vRoot   = VirtualRootWindow(hDisplay, hScreen);
#else
        vRoot   = hRoot;
#endif
      }

      if (!XTranslateCoordinates (hDisplay, childWindow, vRoot, 0, 0,
              &absx, &absy, &ignored)) {
        fprintf(stderr,
                "%s:  unable to translate window coordinates (%d,%d)\n",
                        programName, absx, absy);
        XCloseDisplay(hDisplay);
        exit  (3);
      }

      xrect.width = xwa.width;
      xrect.height = xwa.height;

      if (borders) {
        absx         -= xwa.border_width;
        absy         -= xwa.border_width;
        xrect.width  += (2 * xwa.border_width);
        xrect.height += (2 * xwa.border_width);
      }

      if (hRoot == vRoot) {
        /* do some clipping here */
        if (absx < 0) {
          xrect.width += absx;
          absx = 0;
        }
        if (absy < 0) {
          xrect.height += absy;
          absy = 0;
        }
        dwidth = DisplayWidth(hDisplay, hScreen);
        dheight = DisplayHeight(hDisplay, hScreen);
        if (absx + xrect.width > dwidth)
          xrect.width = dwidth - absx;
        if (absy + xrect.height > dheight)
          xrect.height = dheight - absy;
      }

      xrect.x = absx;
      xrect.y = absy;
    }
  }


  if (subrectSource) {
    /* note that constraint checks here were left out intentionally so
     * that you can grab an area outside of the selected window.  You
     * should be able to grab the area to the left of the window, for
     * instance */
    xrect.x      += xoffset.x;
    xrect.y      += xoffset.y;
    xrect.width  = xoffset.width;
    xrect.height = xoffset.height;
    }

  dwidth = DisplayWidth(hDisplay, hScreen);
  dheight = DisplayHeight(hDisplay, hScreen);
  if (xrect.x + xrect.width > dwidth)
    xrect.width = dwidth - xrect.x;
  if (xrect.y + xrect.height > dheight)
    xrect.height = dheight - xrect.y;
  
  if (verbose)
    fprintf(stderr, "%s: bounding box is [x=%d y=%d w=%d h=%d]\n", programName, xrect.x, xrect.y,
                    xrect.width, xrect.height);

  /* get the depth, visual and colormap */
  if (colorWin == None)
    colorWin = childWindow? childWindow : sourceWindow;

  if (!XGetWindowAttributes(hDisplay, colorWin, &xwa)) {
    fprintf(stderr, "%s: unable to get window coordinates\n", programName);
    XCloseDisplay(hDisplay);
    exit(3);
  }

  image.depth    = xwa.depth;
  image.visual   = xwa.visual;
  image.colormap = xwa.colormap;
  if (verbose)
    fprintf(stderr, "%s: visual=%x depth=%d colormap=%x\n", programName,
                    image.visual, image.depth, image.colormap);


  /* sleep if asked to do so */
  if (postSelectSleepSeconds) {
    if (grabServer) {
      XUngrabServer(hDisplay);
      XSync(hDisplay, FALSE);
    }
    sleep(postSelectSleepSeconds);
    if (grabServer)
      XGrabServer(hDisplay);
  }


  /* get the image bounded by the rectangle from the source window */
  if (!noBell)
    XBell(hDisplay, 50);

  if (!getImage(&xrect, &image, sourceWindow)) {
    XCloseDisplay(hDisplay);
    exit(3);
  }
  if (grabServer)
    XUngrabServer(hDisplay);

  if (!noBell) {
    XBell(hDisplay, 20);
    XBell(hDisplay, 30);
  }

  XFlush(hDisplay);



  /* do color image processing/conversions */

  if (image.depth >= 2) {
    if (brighten)
      brightenColors(&image, brightenFactor);
    if (doAnd)
      alterPlanes(&image, TRUE, andBits);
    if (doOr)
      alterPlanes(&image, FALSE, orBits);
  }
  
  if (doReverse)
    reverseColors(&image);

  if (image.depth >= 2) {
    if (! (preview && psColor) ) {
      if (ForceBitmap) {
        pixmap2bitmap(&image);
        image.depth = 1;
      }
      else if (Halftone)
        pixmap2halftone(&image, DitherKind);
      else
        compressColormap(&image);
    }
    else
      compressColormap(&image);
  }


  /* open the output stream */
  if (outfileName) {
    outfile = fopen(outfileName, "w");
    if (!outfile) {
      fprintf(stderr, "%s: ", programName);
      perror(outfileName);
      exit(3);
    }
    /* form an image name based on the file name */
    {
    char *img = rindex(outfileName, '/');
    if (img) strcpy(imageName, img + 1);
    else strcpy(imageName, outfileName);
    ptr = rindex(imageName, '.');  /* Drop the final extension */
    if (ptr) *ptr = '\0';
    }
  }
  else
    strcpy(imageName,"unnamed");


  /* write to the output stream in the requested format */
  switch (outputFormat) {
    case xwdFormat:
      writeXWD(&image, outfile, xwdxy);
      break;
    case simpleFormat:
      writeSimple(&image, outfile);
      break;
    case puzzleFormat:
      writePuzzle(&image, outfile);
      break;
    case pixmapFormat:
      if (image.ximage->depth <= 1)
        writeXYPixmap(&image, outfile);
      else
        writeZPixmap(xpmFormat, &image, outfile);
      break;
    case ppmFormat:
      writePPM(&image, outfile);
      break;
    case psFormat:
    default:
      if (onlyEpsi)
        writeOnlyPreview(&image, outfile);
      else if (psColor && image.ximage->depth >= 2)
        writeColorPS(&image, outfile, compress, encapsulate,
	                preview, landscape, binary, checkLimits);
      else
        writePostscript(&image, outfile, compress, encapsulate,
	                preview, landscape, binary, checkLimits);
      break;
  }


  XDestroyImage(image.ximage);
  XCloseDisplay(hDisplay);
  if (outfileName)
    fclose(outfile);

  exit(0);
}





#ifndef NO_MAIN
main(argc, argv)
  int argc;
  char *argv[];
{
  char *env;
  env = (char *)getenv("XGRABSC");
  xgrabsc(argc, argv, env);
}
#endif
