/*
 This file is part of GNU Taler
 (C) 2019 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Helpers to create headless wallets.
 * @author Florian Dold <dold@taler.net>
 */

/**
 * Imports.
 */
import type {
  ResultRow,
  Sqlite3Interface,
  Sqlite3Statement,
} from "@gnu-taler/idb-bridge";
// eslint-disable-next-line no-duplicate-imports
import {
  BridgeIDBFactory,
  createSqliteBackend,
  MemoryBackend,
  shimIndexedDB,
} from "@gnu-taler/idb-bridge";
import {
  j2s,
  Logger,
  SetTimeoutTimerAPI,
  WalletRunConfig,
} from "@gnu-taler/taler-util";
import { createPlatformHttpLib } from "@gnu-taler/taler-util/http";
import { qjsOs, qjsStd } from "@gnu-taler/taler-util/qtart";
import { SynchronousCryptoWorkerFactoryPlain } from "./crypto/workers/synchronousWorkerFactoryPlain.js";
import {
  DefaultNodeWalletArgs,
  getSqlite3FilenameFromStoragePath,
  makeTempfileId,
} from "./host-common.js";
import { exportDb } from "./index.js";
import { Wallet, WalletDatabaseImplementation } from "./wallet.js";

const logger = new Logger("host-impl.qtart.ts");

let numStmt = 0;

export async function createQtartSqlite3Impl(): Promise<Sqlite3Interface> {
  const tart: any = (globalThis as any)._tart;
  if (!tart) {
    throw Error("globalThis._qtart not defined");
  }
  return {
    async open(filename: string) {
      const internalDbHandle = tart.sqlite3Open(filename);
      return {
        internalDbHandle,
        async close() {
          tart.sqlite3Close(internalDbHandle);
        },
        async prepare(stmtStr): Promise<Sqlite3Statement> {
          const stmtHandle = tart.sqlite3Prepare(internalDbHandle, stmtStr);
          return {
            internalStatement: stmtHandle,
            async getAll(params): Promise<ResultRow[]> {
              numStmt++;
              return tart.sqlite3StmtGetAll(stmtHandle, params);
            },
            async getFirst(params): Promise<ResultRow | undefined> {
              numStmt++;
              return tart.sqlite3StmtGetFirst(stmtHandle, params);
            },
            async run(params) {
              numStmt++;
              return tart.sqlite3StmtRun(stmtHandle, params);
            },
          };
        },
        async exec(sqlStr): Promise<void> {
          numStmt++;
          tart.sqlite3Exec(internalDbHandle, sqlStr);
        },
      };
    },
  };
}

async function makeSqliteDb(
  args: DefaultNodeWalletArgs,
): Promise<WalletDatabaseImplementation> {
  BridgeIDBFactory.enableTracing = false;
  const filename = getSqlite3FilenameFromStoragePath(
    args.persistentStoragePath,
  );
  logger.info(`opening sqlite3 database ${j2s(filename)}`);
  const imp = await createQtartSqlite3Impl();
  const myBackend = await createSqliteBackend(imp, {
    filename,
  });
  myBackend.trackStats = true;
  myBackend.enableTracing = false;
  const myBridgeIdbFactory = new BridgeIDBFactory(myBackend);
  return {
    getStats() {
      return {
        ...myBackend.accessStats,
        primitiveStatements: numStmt,
      };
    },
    async exportToFile(directory, stem, forceFormat) {
      if (forceFormat === "json") {
        const path = `${directory}/${stem}.json`;
        const dbDump = await exportDb(myBridgeIdbFactory);
        const errObj = { errno: undefined };
        const file = qjsStd.open(path, "w+", errObj);
        if (!file) {
          throw Error(`could not create file (errno=${errObj.errno})`);
        }
        file.puts(JSON.stringify(dbDump));
        file.close();
        return {
          path,
        };
      } else if (forceFormat === "sqlite3" || forceFormat == null) {
        const path = `${directory}/${stem}.sqlite3`;
        await myBackend.backupToFile(path);
        return {
          path,
        };
      } else {
        throw Error(`forcing format ${forceFormat} not supported`);
      }
    },
    async readBackupJson(path: string): Promise<any> {
      const errObj = { errno: undefined };
      const file = qjsStd.open(path, "r", errObj);
      if (!path.endsWith(".json")) {
        throw Error("DB file import only supports .json files at the moment");
      }
      if (!file) {
        throw Error(`could not open file (errno=${errObj.errno})`);
      }
      const dumpStr = file.readAsString();
      file.close();
      const dump = JSON.parse(dumpStr);
      return dump;
    },
    idbFactory: myBridgeIdbFactory,
  };
}

async function makeFileDb(
  args: DefaultNodeWalletArgs = {},
): Promise<WalletDatabaseImplementation> {
  BridgeIDBFactory.enableTracing = false;
  const myBackend = new MemoryBackend();
  myBackend.enableTracing = false;

  const storagePath = args.persistentStoragePath;
  if (storagePath) {
    const dbContentStr = qjsStd.loadFile(storagePath);
    if (dbContentStr != null) {
      const dbContent = JSON.parse(dbContentStr);
      myBackend.importDump(dbContent);
    }

    myBackend.afterCommitCallback = async () => {
      logger.trace("committing database");
      // Allow caller to stop persisting the wallet.
      if (args.persistentStoragePath === undefined) {
        return;
      }
      const tmpPath = `${args.persistentStoragePath}-${makeTempfileId(5)}.tmp`;
      const dbContent = myBackend.exportDump();
      logger.trace("exported DB dump");
      qjsStd.writeFile(tmpPath, JSON.stringify(dbContent, undefined, 2));
      // Atomically move the temporary file onto the DB path.
      const res = qjsOs.rename(tmpPath, args.persistentStoragePath);
      if (res != 0) {
        throw Error("db commit failed at rename");
      }
      logger.trace("committing database done");
    };
  }

  const myBridgeIdbFactory = new BridgeIDBFactory(myBackend);
  return {
    idbFactory: myBridgeIdbFactory,
    getStats: () => myBackend.accessStats,
    exportToFile(directory, stem) {
      throw Error("not supported");
    },
    async readBackupJson(path: string): Promise<any> {
      throw Error("not supported");
    },
  };
}

export async function createNativeWalletHost2(
  args: DefaultNodeWalletArgs = {},
): Promise<{
  wallet: Wallet;
}> {
  BridgeIDBFactory.enableTracing = false;

  let dbResp: WalletDatabaseImplementation;

  if (
    args.persistentStoragePath &&
    args.persistentStoragePath.endsWith(".json")
  ) {
    logger.info("using JSON file DB backend (slow, only use for testing)");
    dbResp = await makeFileDb(args);
  } else {
    //    logger.info("using sqlite3 DB backend");
    dbResp = await makeSqliteDb(args);
  }

  shimIndexedDB(dbResp.idbFactory);

  const myHttpFactory = (config: WalletRunConfig) => {
    let myHttpLib;
    if (args.httpLib) {
      myHttpLib = args.httpLib;
    } else {
      myHttpLib = createPlatformHttpLib({
        enableThrottling: true,
        requireTls: !config.features.allowHttp,
      });
    }
    return myHttpLib;
  };

  let workerFactory;
  workerFactory = new SynchronousCryptoWorkerFactoryPlain();

  const timer = new SetTimeoutTimerAPI();

  const w = await Wallet.create(dbResp, myHttpFactory, timer, workerFactory);

  if (args.notifyHandler) {
    w.addNotificationListener(args.notifyHandler);
  }
  return {
    wallet: w,
  };
}
