// 2007-02-04  Edward Smith-Rowland <3dw4rd@verizon.net>
//
// Copyright (C) 2007-2013 Free Software Foundation, Inc.
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING3.  If not see
// <http://www.gnu.org/licenses/>.

//  assoc_legendre


//  Compare against values generated by the GNU Scientific Library.
//  The GSL can be found on the web: http://www.gnu.org/software/gsl/

#include <tr1/cmath>
#if defined(__TEST_DEBUG)
#include <iostream>
#define VERIFY(A) \
if (!(A)) \
  { \
    std::cout << "line " << __LINE__ \
      << "  max_abs_frac = " << max_abs_frac \
      << std::endl; \
  }
#else
#include <testsuite_hooks.h>
#endif
#include "../testcase.h"


// Test data for l=0, m=0.
testcase_assoc_legendre<double> data001[] = {
  { 1.0000000000000000, 0, 0, 
          -1.0000000000000000 },
  { 1.0000000000000000, 0, 0, 
          -0.90000000000000002 },
  { 1.0000000000000000, 0, 0, 
          -0.80000000000000004 },
  { 1.0000000000000000, 0, 0, 
          -0.69999999999999996 },
  { 1.0000000000000000, 0, 0, 
          -0.59999999999999998 },
  { 1.0000000000000000, 0, 0, 
          -0.50000000000000000 },
  { 1.0000000000000000, 0, 0, 
          -0.40000000000000002 },
  { 1.0000000000000000, 0, 0, 
          -0.30000000000000004 },
  { 1.0000000000000000, 0, 0, 
          -0.19999999999999996 },
  { 1.0000000000000000, 0, 0, 
          -0.099999999999999978 },
  { 1.0000000000000000, 0, 0, 
          0.0000000000000000 },
  { 1.0000000000000000, 0, 0, 
          0.10000000000000009 },
  { 1.0000000000000000, 0, 0, 
          0.19999999999999996 },
  { 1.0000000000000000, 0, 0, 
          0.30000000000000004 },
  { 1.0000000000000000, 0, 0, 
          0.39999999999999991 },
  { 1.0000000000000000, 0, 0, 
          0.50000000000000000 },
  { 1.0000000000000000, 0, 0, 
          0.60000000000000009 },
  { 1.0000000000000000, 0, 0, 
          0.69999999999999996 },
  { 1.0000000000000000, 0, 0, 
          0.80000000000000004 },
  { 1.0000000000000000, 0, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 0, 0, 
          1.0000000000000000 },
};

// Test function for l=0, m=0.
template <typename Tp>
void test001()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data001)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data001[i].l), Tp(data001[i].m),
                   Tp(data001[i].x));
      const Tp f0 = data001[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=1, m=0.
testcase_assoc_legendre<double> data002[] = {
  { -1.0000000000000000, 1, 0, 
          -1.0000000000000000 },
  { -0.90000000000000002, 1, 0, 
          -0.90000000000000002 },
  { -0.80000000000000004, 1, 0, 
          -0.80000000000000004 },
  { -0.69999999999999996, 1, 0, 
          -0.69999999999999996 },
  { -0.59999999999999998, 1, 0, 
          -0.59999999999999998 },
  { -0.50000000000000000, 1, 0, 
          -0.50000000000000000 },
  { -0.40000000000000002, 1, 0, 
          -0.40000000000000002 },
  { -0.30000000000000004, 1, 0, 
          -0.30000000000000004 },
  { -0.19999999999999996, 1, 0, 
          -0.19999999999999996 },
  { -0.099999999999999978, 1, 0, 
          -0.099999999999999978 },
  { 0.0000000000000000, 1, 0, 
          0.0000000000000000 },
  { 0.10000000000000009, 1, 0, 
          0.10000000000000009 },
  { 0.19999999999999996, 1, 0, 
          0.19999999999999996 },
  { 0.30000000000000004, 1, 0, 
          0.30000000000000004 },
  { 0.39999999999999991, 1, 0, 
          0.39999999999999991 },
  { 0.50000000000000000, 1, 0, 
          0.50000000000000000 },
  { 0.60000000000000009, 1, 0, 
          0.60000000000000009 },
  { 0.69999999999999996, 1, 0, 
          0.69999999999999996 },
  { 0.80000000000000004, 1, 0, 
          0.80000000000000004 },
  { 0.89999999999999991, 1, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 1, 0, 
          1.0000000000000000 },
};

// Test function for l=1, m=0.
template <typename Tp>
void test002()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data002)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data002[i].l), Tp(data002[i].m),
                   Tp(data002[i].x));
      const Tp f0 = data002[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=1, m=1.
testcase_assoc_legendre<double> data003[] = {
  { -0.0000000000000000, 1, 1, 
          -1.0000000000000000 },
  { -0.43588989435406728, 1, 1, 
          -0.90000000000000002 },
  { -0.59999999999999987, 1, 1, 
          -0.80000000000000004 },
  { -0.71414284285428509, 1, 1, 
          -0.69999999999999996 },
  { -0.80000000000000004, 1, 1, 
          -0.59999999999999998 },
  { -0.86602540378443860, 1, 1, 
          -0.50000000000000000 },
  { -0.91651513899116799, 1, 1, 
          -0.40000000000000002 },
  { -0.95393920141694577, 1, 1, 
          -0.30000000000000004 },
  { -0.97979589711327120, 1, 1, 
          -0.19999999999999996 },
  { -0.99498743710661997, 1, 1, 
          -0.099999999999999978 },
  { -1.0000000000000000, 1, 1, 
          0.0000000000000000 },
  { -0.99498743710661997, 1, 1, 
          0.10000000000000009 },
  { -0.97979589711327120, 1, 1, 
          0.19999999999999996 },
  { -0.95393920141694577, 1, 1, 
          0.30000000000000004 },
  { -0.91651513899116799, 1, 1, 
          0.39999999999999991 },
  { -0.86602540378443860, 1, 1, 
          0.50000000000000000 },
  { -0.79999999999999993, 1, 1, 
          0.60000000000000009 },
  { -0.71414284285428509, 1, 1, 
          0.69999999999999996 },
  { -0.59999999999999987, 1, 1, 
          0.80000000000000004 },
  { -0.43588989435406750, 1, 1, 
          0.89999999999999991 },
  { -0.0000000000000000, 1, 1, 
          1.0000000000000000 },
};

// Test function for l=1, m=1.
template <typename Tp>
void test003()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data003)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data003[i].l), Tp(data003[i].m),
                   Tp(data003[i].x));
      const Tp f0 = data003[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=0.
testcase_assoc_legendre<double> data004[] = {
  { 1.0000000000000000, 2, 0, 
          -1.0000000000000000 },
  { 0.71500000000000008, 2, 0, 
          -0.90000000000000002 },
  { 0.46000000000000019, 2, 0, 
          -0.80000000000000004 },
  { 0.23499999999999982, 2, 0, 
          -0.69999999999999996 },
  { 0.039999999999999925, 2, 0, 
          -0.59999999999999998 },
  { -0.12500000000000000, 2, 0, 
          -0.50000000000000000 },
  { -0.25999999999999995, 2, 0, 
          -0.40000000000000002 },
  { -0.36499999999999994, 2, 0, 
          -0.30000000000000004 },
  { -0.44000000000000000, 2, 0, 
          -0.19999999999999996 },
  { -0.48499999999999999, 2, 0, 
          -0.099999999999999978 },
  { -0.50000000000000000, 2, 0, 
          0.0000000000000000 },
  { -0.48499999999999999, 2, 0, 
          0.10000000000000009 },
  { -0.44000000000000000, 2, 0, 
          0.19999999999999996 },
  { -0.36499999999999994, 2, 0, 
          0.30000000000000004 },
  { -0.26000000000000012, 2, 0, 
          0.39999999999999991 },
  { -0.12500000000000000, 2, 0, 
          0.50000000000000000 },
  { 0.040000000000000160, 2, 0, 
          0.60000000000000009 },
  { 0.23499999999999982, 2, 0, 
          0.69999999999999996 },
  { 0.46000000000000019, 2, 0, 
          0.80000000000000004 },
  { 0.71499999999999975, 2, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 2, 0, 
          1.0000000000000000 },
};

// Test function for l=2, m=0.
template <typename Tp>
void test004()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data004)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data004[i].l), Tp(data004[i].m),
                   Tp(data004[i].x));
      const Tp f0 = data004[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=1.
testcase_assoc_legendre<double> data005[] = {
  { 0.0000000000000000, 2, 1, 
          -1.0000000000000000 },
  { 1.1769027147559816, 2, 1, 
          -0.90000000000000002 },
  { 1.4399999999999999, 2, 1, 
          -0.80000000000000004 },
  { 1.4996999699939983, 2, 1, 
          -0.69999999999999996 },
  { 1.4399999999999999, 2, 1, 
          -0.59999999999999998 },
  { 1.2990381056766580, 2, 1, 
          -0.50000000000000000 },
  { 1.0998181667894018, 2, 1, 
          -0.40000000000000002 },
  { 0.85854528127525132, 2, 1, 
          -0.30000000000000004 },
  { 0.58787753826796263, 2, 1, 
          -0.19999999999999996 },
  { 0.29849623113198592, 2, 1, 
          -0.099999999999999978 },
  { -0.0000000000000000, 2, 1, 
          0.0000000000000000 },
  { -0.29849623113198626, 2, 1, 
          0.10000000000000009 },
  { -0.58787753826796263, 2, 1, 
          0.19999999999999996 },
  { -0.85854528127525132, 2, 1, 
          0.30000000000000004 },
  { -1.0998181667894014, 2, 1, 
          0.39999999999999991 },
  { -1.2990381056766580, 2, 1, 
          0.50000000000000000 },
  { -1.4400000000000002, 2, 1, 
          0.60000000000000009 },
  { -1.4996999699939983, 2, 1, 
          0.69999999999999996 },
  { -1.4399999999999999, 2, 1, 
          0.80000000000000004 },
  { -1.1769027147559821, 2, 1, 
          0.89999999999999991 },
  { -0.0000000000000000, 2, 1, 
          1.0000000000000000 },
};

// Test function for l=2, m=1.
template <typename Tp>
void test005()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data005)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data005[i].l), Tp(data005[i].m),
                   Tp(data005[i].x));
      const Tp f0 = data005[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=2.
testcase_assoc_legendre<double> data006[] = {
  { 0.0000000000000000, 2, 2, 
          -1.0000000000000000 },
  { 0.56999999999999984, 2, 2, 
          -0.90000000000000002 },
  { 1.0799999999999996, 2, 2, 
          -0.80000000000000004 },
  { 1.5300000000000005, 2, 2, 
          -0.69999999999999996 },
  { 1.9200000000000004, 2, 2, 
          -0.59999999999999998 },
  { 2.2500000000000000, 2, 2, 
          -0.50000000000000000 },
  { 2.5200000000000000, 2, 2, 
          -0.40000000000000002 },
  { 2.7300000000000004, 2, 2, 
          -0.30000000000000004 },
  { 2.8799999999999999, 2, 2, 
          -0.19999999999999996 },
  { 2.9700000000000002, 2, 2, 
          -0.099999999999999978 },
  { 3.0000000000000000, 2, 2, 
          0.0000000000000000 },
  { 2.9700000000000002, 2, 2, 
          0.10000000000000009 },
  { 2.8799999999999999, 2, 2, 
          0.19999999999999996 },
  { 2.7300000000000004, 2, 2, 
          0.30000000000000004 },
  { 2.5200000000000000, 2, 2, 
          0.39999999999999991 },
  { 2.2500000000000000, 2, 2, 
          0.50000000000000000 },
  { 1.9199999999999997, 2, 2, 
          0.60000000000000009 },
  { 1.5300000000000005, 2, 2, 
          0.69999999999999996 },
  { 1.0799999999999996, 2, 2, 
          0.80000000000000004 },
  { 0.57000000000000040, 2, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 2, 2, 
          1.0000000000000000 },
};

// Test function for l=2, m=2.
template <typename Tp>
void test006()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data006)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data006[i].l), Tp(data006[i].m),
                   Tp(data006[i].x));
      const Tp f0 = data006[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=0.
testcase_assoc_legendre<double> data007[] = {
  { -1.0000000000000000, 5, 0, 
          -1.0000000000000000 },
  { 0.041141249999999997, 5, 0, 
          -0.90000000000000002 },
  { 0.39951999999999993, 5, 0, 
          -0.80000000000000004 },
  { 0.36519874999999991, 5, 0, 
          -0.69999999999999996 },
  { 0.15263999999999994, 5, 0, 
          -0.59999999999999998 },
  { -0.089843750000000000, 5, 0, 
          -0.50000000000000000 },
  { -0.27063999999999988, 5, 0, 
          -0.40000000000000002 },
  { -0.34538624999999995, 5, 0, 
          -0.30000000000000004 },
  { -0.30751999999999996, 5, 0, 
          -0.19999999999999996 },
  { -0.17882874999999995, 5, 0, 
          -0.099999999999999978 },
  { 0.0000000000000000, 5, 0, 
          0.0000000000000000 },
  { 0.17882875000000015, 5, 0, 
          0.10000000000000009 },
  { 0.30751999999999996, 5, 0, 
          0.19999999999999996 },
  { 0.34538624999999995, 5, 0, 
          0.30000000000000004 },
  { 0.27064000000000010, 5, 0, 
          0.39999999999999991 },
  { 0.089843750000000000, 5, 0, 
          0.50000000000000000 },
  { -0.15264000000000022, 5, 0, 
          0.60000000000000009 },
  { -0.36519874999999991, 5, 0, 
          0.69999999999999996 },
  { -0.39951999999999993, 5, 0, 
          0.80000000000000004 },
  { -0.041141250000000407, 5, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 5, 0, 
          1.0000000000000000 },
};

// Test function for l=5, m=0.
template <typename Tp>
void test007()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data007)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data007[i].l), Tp(data007[i].m),
                   Tp(data007[i].x));
      const Tp f0 = data007[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=1.
testcase_assoc_legendre<double> data008[] = {
  { 0.0000000000000000, 5, 1, 
          -1.0000000000000000 },
  { -2.8099369608350973, 5, 1, 
          -0.90000000000000002 },
  { -0.72180000000000089, 5, 1, 
          -0.80000000000000004 },
  { 1.0951826834447254, 5, 1, 
          -0.69999999999999996 },
  { 1.9775999999999998, 5, 1, 
          -0.59999999999999998 },
  { 1.9282596881137892, 5, 1, 
          -0.50000000000000000 },
  { 1.2070504380513685, 5, 1, 
          -0.40000000000000002 },
  { 0.16079837663884430, 5, 1, 
          -0.30000000000000004 },
  { -0.87005875663658538, 5, 1, 
          -0.19999999999999996 },
  { -1.6083350053680323, 5, 1, 
          -0.099999999999999978 },
  { -1.8750000000000000, 5, 1, 
          0.0000000000000000 },
  { -1.6083350053680314, 5, 1, 
          0.10000000000000009 },
  { -0.87005875663658538, 5, 1, 
          0.19999999999999996 },
  { 0.16079837663884430, 5, 1, 
          0.30000000000000004 },
  { 1.2070504380513674, 5, 1, 
          0.39999999999999991 },
  { 1.9282596881137892, 5, 1, 
          0.50000000000000000 },
  { 1.9775999999999996, 5, 1, 
          0.60000000000000009 },
  { 1.0951826834447254, 5, 1, 
          0.69999999999999996 },
  { -0.72180000000000089, 5, 1, 
          0.80000000000000004 },
  { -2.8099369608350959, 5, 1, 
          0.89999999999999991 },
  { 0.0000000000000000, 5, 1, 
          1.0000000000000000 },
};

// Test function for l=5, m=1.
template <typename Tp>
void test008()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data008)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data008[i].l), Tp(data008[i].m),
                   Tp(data008[i].x));
      const Tp f0 = data008[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=2.
testcase_assoc_legendre<double> data009[] = {
  { 0.0000000000000000, 5, 2, 
          -1.0000000000000000 },
  { -12.837824999999995, 5, 2, 
          -0.90000000000000002 },
  { -13.910399999999997, 5, 2, 
          -0.80000000000000004 },
  { -8.8089749999999984, 5, 2, 
          -0.69999999999999996 },
  { -1.6128000000000009, 5, 2, 
          -0.59999999999999998 },
  { 4.9218750000000000, 5, 2, 
          -0.50000000000000000 },
  { 9.1728000000000005, 5, 2, 
          -0.40000000000000002 },
  { 10.462725000000001, 5, 2, 
          -0.30000000000000004 },
  { 8.8703999999999983, 5, 2, 
          -0.19999999999999996 },
  { 5.0415749999999999, 5, 2, 
          -0.099999999999999978 },
  { -0.0000000000000000, 5, 2, 
          0.0000000000000000 },
  { -5.0415750000000044, 5, 2, 
          0.10000000000000009 },
  { -8.8703999999999983, 5, 2, 
          0.19999999999999996 },
  { -10.462725000000001, 5, 2, 
          0.30000000000000004 },
  { -9.1728000000000005, 5, 2, 
          0.39999999999999991 },
  { -4.9218750000000000, 5, 2, 
          0.50000000000000000 },
  { 1.6128000000000071, 5, 2, 
          0.60000000000000009 },
  { 8.8089749999999984, 5, 2, 
          0.69999999999999996 },
  { 13.910399999999997, 5, 2, 
          0.80000000000000004 },
  { 12.837825000000004, 5, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 5, 2, 
          1.0000000000000000 },
};

// Test function for l=5, m=2.
template <typename Tp>
void test009()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data009)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data009[i].l), Tp(data009[i].m),
                   Tp(data009[i].x));
      const Tp f0 = data009[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=5.
testcase_assoc_legendre<double> data010[] = {
  { -0.0000000000000000, 5, 5, 
          -1.0000000000000000 },
  { -14.870165800941818, 5, 5, 
          -0.90000000000000002 },
  { -73.483199999999925, 5, 5, 
          -0.80000000000000004 },
  { -175.53238298794764, 5, 5, 
          -0.69999999999999996 },
  { -309.65760000000006, 5, 5, 
          -0.59999999999999998 },
  { -460.34662869916559, 5, 5, 
          -0.50000000000000000 },
  { -611.12496255819883, 5, 5, 
          -0.40000000000000002 },
  { -746.50941479523760, 5, 5, 
          -0.30000000000000004 },
  { -853.31600434671316, 5, 5, 
          -0.19999999999999996 },
  { -921.55189181724734, 5, 5, 
          -0.099999999999999978 },
  { -945.00000000000000, 5, 5, 
          0.0000000000000000 },
  { -921.55189181724734, 5, 5, 
          0.10000000000000009 },
  { -853.31600434671316, 5, 5, 
          0.19999999999999996 },
  { -746.50941479523760, 5, 5, 
          0.30000000000000004 },
  { -611.12496255819883, 5, 5, 
          0.39999999999999991 },
  { -460.34662869916559, 5, 5, 
          0.50000000000000000 },
  { -309.65759999999989, 5, 5, 
          0.60000000000000009 },
  { -175.53238298794764, 5, 5, 
          0.69999999999999996 },
  { -73.483199999999925, 5, 5, 
          0.80000000000000004 },
  { -14.870165800941855, 5, 5, 
          0.89999999999999991 },
  { -0.0000000000000000, 5, 5, 
          1.0000000000000000 },
};

// Test function for l=5, m=5.
template <typename Tp>
void test010()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data010)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data010[i].l), Tp(data010[i].m),
                   Tp(data010[i].x));
      const Tp f0 = data010[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=0.
testcase_assoc_legendre<double> data011[] = {
  { 1.0000000000000000, 10, 0, 
          -1.0000000000000000 },
  { -0.26314561785585977, 10, 0, 
          -0.90000000000000002 },
  { 0.30052979559999998, 10, 0, 
          -0.80000000000000004 },
  { 0.085805795531640333, 10, 0, 
          -0.69999999999999996 },
  { -0.24366274560000006, 10, 0, 
          -0.59999999999999998 },
  { -0.18822860717773438, 10, 0, 
          -0.50000000000000000 },
  { 0.096839064399999869, 10, 0, 
          -0.40000000000000002 },
  { 0.25147634951601561, 10, 0, 
          -0.30000000000000004 },
  { 0.12907202559999989, 10, 0, 
          -0.19999999999999996 },
  { -0.12212499738710947, 10, 0, 
          -0.099999999999999978 },
  { -0.24609375000000000, 10, 0, 
          0.0000000000000000 },
  { -0.12212499738710922, 10, 0, 
          0.10000000000000009 },
  { 0.12907202559999989, 10, 0, 
          0.19999999999999996 },
  { 0.25147634951601561, 10, 0, 
          0.30000000000000004 },
  { 0.096839064400000258, 10, 0, 
          0.39999999999999991 },
  { -0.18822860717773438, 10, 0, 
          0.50000000000000000 },
  { -0.24366274559999987, 10, 0, 
          0.60000000000000009 },
  { 0.085805795531640333, 10, 0, 
          0.69999999999999996 },
  { 0.30052979559999998, 10, 0, 
          0.80000000000000004 },
  { -0.26314561785585888, 10, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 10, 0, 
          1.0000000000000000 },
};

// Test function for l=10, m=0.
template <typename Tp>
void test011()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data011)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data011[i].l), Tp(data011[i].m),
                   Tp(data011[i].x));
      const Tp f0 = data011[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=1.
testcase_assoc_legendre<double> data012[] = {
  { -0.0000000000000000, 10, 1, 
          -1.0000000000000000 },
  { -3.0438748781479061, 10, 1, 
          -0.90000000000000002 },
  { -0.87614260800000199, 10, 1, 
          -0.80000000000000004 },
  { 2.9685359952934522, 10, 1, 
          -0.69999999999999996 },
  { 1.2511825919999990, 10, 1, 
          -0.59999999999999998 },
  { -2.0066877394361260, 10, 1, 
          -0.50000000000000000 },
  { -2.4822196173476661, 10, 1, 
          -0.40000000000000002 },
  { -0.12309508907433941, 10, 1, 
          -0.30000000000000004 },
  { 2.2468221751958408, 10, 1, 
          -0.19999999999999996 },
  { 2.2472659777983512, 10, 1, 
          -0.099999999999999978 },
  { -0.0000000000000000, 10, 1, 
          0.0000000000000000 },
  { -2.2472659777983530, 10, 1, 
          0.10000000000000009 },
  { -2.2468221751958408, 10, 1, 
          0.19999999999999996 },
  { 0.12309508907433941, 10, 1, 
          0.30000000000000004 },
  { 2.4822196173476643, 10, 1, 
          0.39999999999999991 },
  { 2.0066877394361260, 10, 1, 
          0.50000000000000000 },
  { -1.2511825920000037, 10, 1, 
          0.60000000000000009 },
  { -2.9685359952934522, 10, 1, 
          0.69999999999999996 },
  { 0.87614260800000199, 10, 1, 
          0.80000000000000004 },
  { 3.0438748781479110, 10, 1, 
          0.89999999999999991 },
  { 0.0000000000000000, 10, 1, 
          1.0000000000000000 },
};

// Test function for l=10, m=1.
template <typename Tp>
void test012()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data012)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data012[i].l), Tp(data012[i].m),
                   Tp(data012[i].x));
      const Tp f0 = data012[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=2.
testcase_assoc_legendre<double> data013[] = {
  { 0.0000000000000000, 10, 2, 
          -1.0000000000000000 },
  { 16.376387762496122, 10, 2, 
          -0.90000000000000002 },
  { -35.394657803999991, 10, 2, 
          -0.80000000000000004 },
  { -3.6191429423788892, 10, 2, 
          -0.69999999999999996 },
  { 28.679675904000010, 10, 2, 
          -0.59999999999999998 },
  { 18.388023376464844, 10, 2, 
          -0.50000000000000000 },
  { -12.818955995999996, 10, 2, 
          -0.40000000000000002 },
  { -27.739821675972660, 10, 2, 
          -0.30000000000000004 },
  { -13.280661503999987, 10, 2, 
          -0.19999999999999996 },
  { 13.885467170308596, 10, 2, 
          -0.099999999999999978 },
  { 27.070312500000000, 10, 2, 
          0.0000000000000000 },
  { 13.885467170308573, 10, 2, 
          0.10000000000000009 },
  { -13.280661503999987, 10, 2, 
          0.19999999999999996 },
  { -27.739821675972660, 10, 2, 
          0.30000000000000004 },
  { -12.818955996000019, 10, 2, 
          0.39999999999999991 },
  { 18.388023376464844, 10, 2, 
          0.50000000000000000 },
  { 28.679675903999982, 10, 2, 
          0.60000000000000009 },
  { -3.6191429423788892, 10, 2, 
          0.69999999999999996 },
  { -35.394657803999991, 10, 2, 
          0.80000000000000004 },
  { 16.376387762495984, 10, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 10, 2, 
          1.0000000000000000 },
};

// Test function for l=10, m=2.
template <typename Tp>
void test013()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data013)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data013[i].l), Tp(data013[i].m),
                   Tp(data013[i].x));
      const Tp f0 = data013[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=10, m=5.
testcase_assoc_legendre<double> data014[] = {
  { 0.0000000000000000, 10, 5, 
          -1.0000000000000000 },
  { 21343.618518164669, 10, 5, 
          -0.90000000000000002 },
  { 40457.016407807983, 10, 5, 
          -0.80000000000000004 },
  { 20321.279317331318, 10, 5, 
          -0.69999999999999996 },
  { -14410.820616192013, 10, 5, 
          -0.59999999999999998 },
  { -30086.169706116176, 10, 5, 
          -0.50000000000000000 },
  { -17177.549337582859, 10, 5, 
          -0.40000000000000002 },
  { 9272.5119495412346, 10, 5, 
          -0.30000000000000004 },
  { 26591.511184414714, 10, 5, 
          -0.19999999999999996 },
  { 21961.951238504211, 10, 5, 
          -0.099999999999999978 },
  { -0.0000000000000000, 10, 5, 
          0.0000000000000000 },
  { -21961.951238504229, 10, 5, 
          0.10000000000000009 },
  { -26591.511184414714, 10, 5, 
          0.19999999999999996 },
  { -9272.5119495412346, 10, 5, 
          0.30000000000000004 },
  { 17177.549337582830, 10, 5, 
          0.39999999999999991 },
  { 30086.169706116176, 10, 5, 
          0.50000000000000000 },
  { 14410.820616191975, 10, 5, 
          0.60000000000000009 },
  { -20321.279317331318, 10, 5, 
          0.69999999999999996 },
  { -40457.016407807983, 10, 5, 
          0.80000000000000004 },
  { -21343.618518164698, 10, 5, 
          0.89999999999999991 },
  { 0.0000000000000000, 10, 5, 
          1.0000000000000000 },
};

// Test function for l=10, m=5.
template <typename Tp>
void test014()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data014)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data014[i].l), Tp(data014[i].m),
                   Tp(data014[i].x));
      const Tp f0 = data014[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=10.
testcase_assoc_legendre<double> data015[] = {
  { 0.0000000000000000, 10, 10, 
          -1.0000000000000000 },
  { 162117.40078784220, 10, 10, 
          -0.90000000000000002 },
  { 3958896.3481267113, 10, 10, 
          -0.80000000000000004 },
  { 22589806.343887307, 10, 10, 
          -0.69999999999999996 },
  { 70300999.121633321, 10, 10, 
          -0.59999999999999998 },
  { 155370278.54003900, 10, 10, 
          -0.50000000000000000 },
  { 273815518.20150518, 10, 10, 
          -0.40000000000000002 },
  { 408571989.13158917, 10, 10, 
          -0.30000000000000004 },
  { 533848212.07990247, 10, 10, 
          -0.19999999999999996 },
  { 622640835.70523083, 10, 10, 
          -0.099999999999999978 },
  { 654729075.00000000, 10, 10, 
          0.0000000000000000 },
  { 622640835.70523083, 10, 10, 
          0.10000000000000009 },
  { 533848212.07990247, 10, 10, 
          0.19999999999999996 },
  { 408571989.13158917, 10, 10, 
          0.30000000000000004 },
  { 273815518.20150518, 10, 10, 
          0.39999999999999991 },
  { 155370278.54003900, 10, 10, 
          0.50000000000000000 },
  { 70300999.121633217, 10, 10, 
          0.60000000000000009 },
  { 22589806.343887307, 10, 10, 
          0.69999999999999996 },
  { 3958896.3481267113, 10, 10, 
          0.80000000000000004 },
  { 162117.40078784304, 10, 10, 
          0.89999999999999991 },
  { 0.0000000000000000, 10, 10, 
          1.0000000000000000 },
};

// Test function for l=10, m=10.
template <typename Tp>
void test015()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data015)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data015[i].l), Tp(data015[i].m),
                   Tp(data015[i].x));
      const Tp f0 = data015[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=0.
testcase_assoc_legendre<double> data016[] = {
  { 1.0000000000000000, 20, 0, 
          -1.0000000000000000 },
  { -0.14930823530984821, 20, 0, 
          -0.90000000000000002 },
  { 0.22420460541741344, 20, 0, 
          -0.80000000000000004 },
  { -0.20457394463834172, 20, 0, 
          -0.69999999999999996 },
  { 0.15916752910098114, 20, 0, 
          -0.59999999999999998 },
  { -0.048358381067373557, 20, 0, 
          -0.50000000000000000 },
  { -0.10159261558628156, 20, 0, 
          -0.40000000000000002 },
  { 0.18028715947998047, 20, 0, 
          -0.30000000000000004 },
  { -0.098042194344594741, 20, 0, 
          -0.19999999999999996 },
  { -0.082077130944527649, 20, 0, 
          -0.099999999999999978 },
  { 0.17619705200195312, 20, 0, 
          0.0000000000000000 },
  { -0.082077130944528037, 20, 0, 
          0.10000000000000009 },
  { -0.098042194344594741, 20, 0, 
          0.19999999999999996 },
  { 0.18028715947998047, 20, 0, 
          0.30000000000000004 },
  { -0.10159261558628112, 20, 0, 
          0.39999999999999991 },
  { -0.048358381067373557, 20, 0, 
          0.50000000000000000 },
  { 0.15916752910098084, 20, 0, 
          0.60000000000000009 },
  { -0.20457394463834172, 20, 0, 
          0.69999999999999996 },
  { 0.22420460541741344, 20, 0, 
          0.80000000000000004 },
  { -0.14930823530984949, 20, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 20, 0, 
          1.0000000000000000 },
};

// Test function for l=20, m=0.
template <typename Tp>
void test016()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data016)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data016[i].l), Tp(data016[i].m),
                   Tp(data016[i].x));
      const Tp f0 = data016[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=1.
testcase_assoc_legendre<double> data017[] = {
  { 0.0000000000000000, 20, 1, 
          -1.0000000000000000 },
  { 4.3838334818220490, 20, 1, 
          -0.90000000000000002 },
  { -0.63138296146340556, 20, 1, 
          -0.80000000000000004 },
  { 0.72274871413391228, 20, 1, 
          -0.69999999999999996 },
  { -2.3203528743824893, 20, 1, 
          -0.59999999999999998 },
  { 3.7399919228791405, 20, 1, 
          -0.50000000000000000 },
  { -3.1692202279270028, 20, 1, 
          -0.40000000000000002 },
  { 0.15804468835344049, 20, 1, 
          -0.30000000000000004 },
  { 3.0366182393271162, 20, 1, 
          -0.19999999999999996 },
  { -3.2115523815580209, 20, 1, 
          -0.099999999999999978 },
  { 0.0000000000000000, 20, 1, 
          0.0000000000000000 },
  { 3.2115523815580160, 20, 1, 
          0.10000000000000009 },
  { -3.0366182393271162, 20, 1, 
          0.19999999999999996 },
  { -0.15804468835344049, 20, 1, 
          0.30000000000000004 },
  { 3.1692202279270076, 20, 1, 
          0.39999999999999991 },
  { -3.7399919228791405, 20, 1, 
          0.50000000000000000 },
  { 2.3203528743825008, 20, 1, 
          0.60000000000000009 },
  { -0.72274871413391228, 20, 1, 
          0.69999999999999996 },
  { 0.63138296146340556, 20, 1, 
          0.80000000000000004 },
  { -4.3838334818220304, 20, 1, 
          0.89999999999999991 },
  { 0.0000000000000000, 20, 1, 
          1.0000000000000000 },
};

// Test function for l=20, m=1.
template <typename Tp>
void test017()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data017)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data017[i].l), Tp(data017[i].m),
                   Tp(data017[i].x));
      const Tp f0 = data017[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=20, m=2.
testcase_assoc_legendre<double> data018[] = {
  { 0.0000000000000000, 20, 2, 
          -1.0000000000000000 },
  { 80.812425587310116, 20, 2, 
          -0.90000000000000002 },
  { -95.849622172549317, 20, 2, 
          -0.80000000000000004 },
  { 87.337927630325510, 20, 2, 
          -0.69999999999999996 },
  { -70.330891533985834, 20, 2, 
          -0.59999999999999998 },
  { 24.629090735179489, 20, 2, 
          -0.50000000000000000 },
  { 39.902576338912418, 20, 2, 
          -0.40000000000000002 },
  { -75.621201471396589, 20, 2, 
          -0.30000000000000004 },
  { 42.417415829726487, 20, 2, 
          -0.19999999999999996 },
  { 33.826848678871301, 20, 2, 
          -0.099999999999999978 },
  { -74.002761840820312, 20, 2, 
          0.0000000000000000 },
  { 33.826848678871457, 20, 2, 
          0.10000000000000009 },
  { 42.417415829726487, 20, 2, 
          0.19999999999999996 },
  { -75.621201471396589, 20, 2, 
          0.30000000000000004 },
  { 39.902576338912240, 20, 2, 
          0.39999999999999991 },
  { 24.629090735179489, 20, 2, 
          0.50000000000000000 },
  { -70.330891533985664, 20, 2, 
          0.60000000000000009 },
  { 87.337927630325510, 20, 2, 
          0.69999999999999996 },
  { -95.849622172549317, 20, 2, 
          0.80000000000000004 },
  { 80.812425587310585, 20, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 20, 2, 
          1.0000000000000000 },
};

// Test function for l=20, m=2.
template <typename Tp>
void test018()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data018)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data018[i].l), Tp(data018[i].m),
                   Tp(data018[i].x));
      const Tp f0 = data018[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=5.
testcase_assoc_legendre<double> data019[] = {
  { -0.0000000000000000, 20, 5, 
          -1.0000000000000000 },
  { -315702.32715134107, 20, 5, 
          -0.90000000000000002 },
  { 503060.91484852595, 20, 5, 
          -0.80000000000000004 },
  { -298127.28360361949, 20, 5, 
          -0.69999999999999996 },
  { -114444.61447464002, 20, 5, 
          -0.59999999999999998 },
  { 543428.40914592857, 20, 5, 
          -0.50000000000000000 },
  { -613842.07728185481, 20, 5, 
          -0.40000000000000002 },
  { 143765.42411270953, 20, 5, 
          -0.30000000000000004 },
  { 472600.45321372297, 20, 5, 
          -0.19999999999999996 },
  { -563861.76771496492, 20, 5, 
          -0.099999999999999978 },
  { 0.0000000000000000, 20, 5, 
          0.0000000000000000 },
  { 563861.76771496458, 20, 5, 
          0.10000000000000009 },
  { -472600.45321372297, 20, 5, 
          0.19999999999999996 },
  { -143765.42411270953, 20, 5, 
          0.30000000000000004 },
  { 613842.07728185505, 20, 5, 
          0.39999999999999991 },
  { -543428.40914592857, 20, 5, 
          0.50000000000000000 },
  { 114444.61447464184, 20, 5, 
          0.60000000000000009 },
  { 298127.28360361949, 20, 5, 
          0.69999999999999996 },
  { -503060.91484852595, 20, 5, 
          0.80000000000000004 },
  { 315702.32715134590, 20, 5, 
          0.89999999999999991 },
  { 0.0000000000000000, 20, 5, 
          1.0000000000000000 },
};

// Test function for l=20, m=5.
template <typename Tp>
void test019()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data019)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data019[i].l), Tp(data019[i].m),
                   Tp(data019[i].x));
      const Tp f0 = data019[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=10.
testcase_assoc_legendre<double> data020[] = {
  { -0.0000000000000000, 20, 10, 
          -1.0000000000000000 },
  { 990017476694.99084, 20, 10, 
          -0.90000000000000002 },
  { 2392757933281.0503, 20, 10, 
          -0.80000000000000004 },
  { -1548364524949.5808, 20, 10, 
          -0.69999999999999996 },
  { -424471915195.05609, 20, 10, 
          -0.59999999999999998 },
  { 1744502295946.2065, 20, 10, 
          -0.50000000000000000 },
  { -899973487310.55212, 20, 10, 
          -0.40000000000000002 },
  { -1092420454297.7164, 20, 10, 
          -0.30000000000000004 },
  { 1466609267659.8816, 20, 10, 
          -0.19999999999999996 },
  { 356041756390.71661, 20, 10, 
          -0.099999999999999978 },
  { -1612052956674.3164, 20, 10, 
          0.0000000000000000 },
  { 356041756390.71985, 20, 10, 
          0.10000000000000009 },
  { 1466609267659.8816, 20, 10, 
          0.19999999999999996 },
  { -1092420454297.7164, 20, 10, 
          0.30000000000000004 },
  { -899973487310.55469, 20, 10, 
          0.39999999999999991 },
  { 1744502295946.2065, 20, 10, 
          0.50000000000000000 },
  { -424471915195.05914, 20, 10, 
          0.60000000000000009 },
  { -1548364524949.5808, 20, 10, 
          0.69999999999999996 },
  { 2392757933281.0503, 20, 10, 
          0.80000000000000004 },
  { 990017476694.99353, 20, 10, 
          0.89999999999999991 },
  { 0.0000000000000000, 20, 10, 
          1.0000000000000000 },
};

// Test function for l=20, m=10.
template <typename Tp>
void test020()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data020)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data020[i].l), Tp(data020[i].m),
                   Tp(data020[i].x));
      const Tp f0 = data020[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=20.
testcase_assoc_legendre<double> data021[] = {
  { 0.0000000000000000, 20, 20, 
          -1.0000000000000000 },
  { 19609049712023808., 20, 20, 
          -0.90000000000000002 },
  { 1.1693527616833221e+19, 20, 20, 
          -0.80000000000000004 },
  { 3.8073455880620691e+20, 20, 20, 
          -0.69999999999999996 },
  { 3.6874002249007927e+21, 20, 20, 
          -0.59999999999999998 },
  { 1.8010806978179592e+22, 20, 20, 
          -0.50000000000000000 },
  { 5.5938832584012466e+22, 20, 20, 
          -0.40000000000000002 },
  { 1.2454734132297811e+23, 20, 20, 
          -0.30000000000000004 },
  { 2.1263407800797497e+23, 20, 20, 
          -0.19999999999999996 },
  { 2.8924941146976873e+23, 20, 20, 
          -0.099999999999999978 },
  { 3.1983098677287775e+23, 20, 20, 
          0.0000000000000000 },
  { 2.8924941146976873e+23, 20, 20, 
          0.10000000000000009 },
  { 2.1263407800797497e+23, 20, 20, 
          0.19999999999999996 },
  { 1.2454734132297811e+23, 20, 20, 
          0.30000000000000004 },
  { 5.5938832584012466e+22, 20, 20, 
          0.39999999999999991 },
  { 1.8010806978179592e+22, 20, 20, 
          0.50000000000000000 },
  { 3.6874002249007807e+21, 20, 20, 
          0.60000000000000009 },
  { 3.8073455880620691e+20, 20, 20, 
          0.69999999999999996 },
  { 1.1693527616833221e+19, 20, 20, 
          0.80000000000000004 },
  { 19609049712024020., 20, 20, 
          0.89999999999999991 },
  { 0.0000000000000000, 20, 20, 
          1.0000000000000000 },
};

// Test function for l=20, m=20.
template <typename Tp>
void test021()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data021)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data021[i].l), Tp(data021[i].m),
                   Tp(data021[i].x));
      const Tp f0 = data021[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=0.
testcase_assoc_legendre<double> data022[] = {
  { 1.0000000000000000, 50, 0, 
          -1.0000000000000000 },
  { -0.17003765994383685, 50, 0, 
          -0.90000000000000002 },
  { 0.13879737345093118, 50, 0, 
          -0.80000000000000004 },
  { -0.014572731645892805, 50, 0, 
          -0.69999999999999996 },
  { -0.058860798844002173, 50, 0, 
          -0.59999999999999998 },
  { -0.031059099239609828, 50, 0, 
          -0.50000000000000000 },
  { 0.041569033381825368, 50, 0, 
          -0.40000000000000002 },
  { 0.10911051574714808, 50, 0, 
          -0.30000000000000004 },
  { 0.083432272204197466, 50, 0, 
          -0.19999999999999996 },
  { -0.038205812661313579, 50, 0, 
          -0.099999999999999978 },
  { -0.11227517265921705, 50, 0, 
          0.0000000000000000 },
  { -0.038205812661314169, 50, 0, 
          0.10000000000000009 },
  { 0.083432272204197466, 50, 0, 
          0.19999999999999996 },
  { 0.10911051574714808, 50, 0, 
          0.30000000000000004 },
  { 0.041569033381824647, 50, 0, 
          0.39999999999999991 },
  { -0.031059099239609828, 50, 0, 
          0.50000000000000000 },
  { -0.058860798844001430, 50, 0, 
          0.60000000000000009 },
  { -0.014572731645892805, 50, 0, 
          0.69999999999999996 },
  { 0.13879737345093118, 50, 0, 
          0.80000000000000004 },
  { -0.17003765994383663, 50, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 50, 0, 
          1.0000000000000000 },
};

// Test function for l=50, m=0.
template <typename Tp>
void test022()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data022)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data022[i].l), Tp(data022[i].m),
                   Tp(data022[i].x));
      const Tp f0 = data022[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=1.
testcase_assoc_legendre<double> data023[] = {
  { 0.0000000000000000, 50, 1, 
          -1.0000000000000000 },
  { -0.13424149984450387, 50, 1, 
          -0.90000000000000002 },
  { 2.2011219672413085, 50, 1, 
          -0.80000000000000004 },
  { 6.6622414993232013, 50, 1, 
          -0.69999999999999996 },
  { 5.5772846936919231, 50, 1, 
          -0.59999999999999998 },
  { 5.8787148815607617, 50, 1, 
          -0.50000000000000000 },
  { 5.5473459458634000, 50, 1, 
          -0.40000000000000002 },
  { 1.8444956647619912, 50, 1, 
          -0.30000000000000004 },
  { -3.8722014306642150, 50, 1, 
          -0.19999999999999996 },
  { -5.3488751322285593, 50, 1, 
          -0.099999999999999978 },
  { -0.0000000000000000, 50, 1, 
          0.0000000000000000 },
  { 5.3488751322285459, 50, 1, 
          0.10000000000000009 },
  { 3.8722014306642150, 50, 1, 
          0.19999999999999996 },
  { -1.8444956647619912, 50, 1, 
          0.30000000000000004 },
  { -5.5473459458634098, 50, 1, 
          0.39999999999999991 },
  { -5.8787148815607617, 50, 1, 
          0.50000000000000000 },
  { -5.5772846936919489, 50, 1, 
          0.60000000000000009 },
  { -6.6622414993232013, 50, 1, 
          0.69999999999999996 },
  { -2.2011219672413085, 50, 1, 
          0.80000000000000004 },
  { 0.13424149984460862, 50, 1, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 1, 
          1.0000000000000000 },
};

// Test function for l=50, m=1.
template <typename Tp>
void test023()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data023)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data023[i].l), Tp(data023[i].m),
                   Tp(data023[i].x));
      const Tp f0 = data023[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000029e-12));
}

// Test data for l=50, m=2.
testcase_assoc_legendre<double> data024[] = {
  { 0.0000000000000000, 50, 2, 
          -1.0000000000000000 },
  { 433.04168483713448, 50, 2, 
          -0.90000000000000002 },
  { -348.06364372056390, 50, 2, 
          -0.80000000000000004 },
  { 50.221071418108465, 50, 2, 
          -0.69999999999999996 },
  { 158.46096409274341, 50, 2, 
          -0.59999999999999998 },
  { 85.988858299721414, 50, 2, 
          -0.50000000000000000 },
  { -101.15891460879277, 50, 2, 
          -0.40000000000000002 },
  { -277.07168105316612, 50, 2, 
          -0.30000000000000004 },
  { -214.33311373510401, 50, 2, 
          -0.19999999999999996 },
  { 96.349657930951722, 50, 2, 
          -0.099999999999999978 },
  { 286.30169028100346, 50, 2, 
          0.0000000000000000 },
  { 96.349657930953285, 50, 2, 
          0.10000000000000009 },
  { -214.33311373510401, 50, 2, 
          0.19999999999999996 },
  { -277.07168105316612, 50, 2, 
          0.30000000000000004 },
  { -101.15891460879106, 50, 2, 
          0.39999999999999991 },
  { 85.988858299721414, 50, 2, 
          0.50000000000000000 },
  { 158.46096409274131, 50, 2, 
          0.60000000000000009 },
  { 50.221071418108465, 50, 2, 
          0.69999999999999996 },
  { -348.06364372056390, 50, 2, 
          0.80000000000000004 },
  { 433.04168483713414, 50, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 2, 
          1.0000000000000000 },
};

// Test function for l=50, m=2.
template <typename Tp>
void test024()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data024)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data024[i].l), Tp(data024[i].m),
                   Tp(data024[i].x));
      const Tp f0 = data024[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=5.
testcase_assoc_legendre<double> data025[] = {
  { -0.0000000000000000, 50, 5, 
          -1.0000000000000000 },
  { -27340473.952132788, 50, 5, 
          -0.90000000000000002 },
  { 27753716.768532373, 50, 5, 
          -0.80000000000000004 },
  { 40808153.913493626, 50, 5, 
          -0.69999999999999996 },
  { 32071189.035790101, 50, 5, 
          -0.59999999999999998 },
  { 36265736.218529105, 50, 5, 
          -0.50000000000000000 },
  { 37089596.700204894, 50, 5, 
          -0.40000000000000002 },
  { 14562029.629244510, 50, 5, 
          -0.30000000000000004 },
  { -23686895.217517227, 50, 5, 
          -0.19999999999999996 },
  { -34878992.965676002, 50, 5, 
          -0.099999999999999978 },
  { -0.0000000000000000, 50, 5, 
          0.0000000000000000 },
  { 34878992.965675950, 50, 5, 
          0.10000000000000009 },
  { 23686895.217517227, 50, 5, 
          0.19999999999999996 },
  { -14562029.629244510, 50, 5, 
          0.30000000000000004 },
  { -37089596.700204931, 50, 5, 
          0.39999999999999991 },
  { -36265736.218529105, 50, 5, 
          0.50000000000000000 },
  { -32071189.035790265, 50, 5, 
          0.60000000000000009 },
  { -40808153.913493626, 50, 5, 
          0.69999999999999996 },
  { -27753716.768532373, 50, 5, 
          0.80000000000000004 },
  { 27340473.952133406, 50, 5, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 5, 
          1.0000000000000000 },
};

// Test function for l=50, m=5.
template <typename Tp>
void test025()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data025)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data025[i].l), Tp(data025[i].m),
                   Tp(data025[i].x));
      const Tp f0 = data025[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=10.
testcase_assoc_legendre<double> data026[] = {
  { -0.0000000000000000, 50, 10, 
          -1.0000000000000000 },
  { -8994661710093158.0, 50, 10, 
          -0.90000000000000002 },
  { 932311375306557.00, 50, 10, 
          -0.80000000000000004 },
  { 12153535011507012., 50, 10, 
          -0.69999999999999996 },
  { 12176690755542244., 50, 10, 
          -0.59999999999999998 },
  { 9180035388465752.0, 50, 10, 
          -0.50000000000000000 },
  { 889201701866911.12, 50, 10, 
          -0.40000000000000002 },
  { -9451384032851600.0, 50, 10, 
          -0.30000000000000004 },
  { -9926439446673564.0, 50, 10, 
          -0.19999999999999996 },
  { 2794368162749970.5, 50, 10, 
          -0.099999999999999978 },
  { 11452238249246346., 50, 10, 
          0.0000000000000000 },
  { 2794368162750031.0, 50, 10, 
          0.10000000000000009 },
  { -9926439446673564.0, 50, 10, 
          0.19999999999999996 },
  { -9451384032851600.0, 50, 10, 
          0.30000000000000004 },
  { 889201701866976.25, 50, 10, 
          0.39999999999999991 },
  { 9180035388465752.0, 50, 10, 
          0.50000000000000000 },
  { 12176690755542216., 50, 10, 
          0.60000000000000009 },
  { 12153535011507012., 50, 10, 
          0.69999999999999996 },
  { 932311375306557.00, 50, 10, 
          0.80000000000000004 },
  { -8994661710093322.0, 50, 10, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 10, 
          1.0000000000000000 },
};

// Test function for l=50, m=10.
template <typename Tp>
void test026()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data026)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data026[i].l), Tp(data026[i].m),
                   Tp(data026[i].x));
      const Tp f0 = data026[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=50, m=20.
testcase_assoc_legendre<double> data027[] = {
  { 0.0000000000000000, 50, 20, 
          -1.0000000000000000 },
  { 1.6630925158645460e+33, 50, 20, 
          -0.90000000000000002 },
  { 1.0622676657892043e+33, 50, 20, 
          -0.80000000000000004 },
  { 8.6022521164717170e+32, 50, 20, 
          -0.69999999999999996 },
  { 4.0860128756808430e+32, 50, 20, 
          -0.59999999999999998 },
  { -4.0169860814274409e+32, 50, 20, 
          -0.50000000000000000 },
  { -8.2324325279774008e+32, 50, 20, 
          -0.40000000000000002 },
  { -4.0054067236243299e+31, 50, 20, 
          -0.30000000000000004 },
  { 7.9309266056434309e+32, 50, 20, 
          -0.19999999999999996 },
  { 5.4151358290899148e+31, 50, 20, 
          -0.099999999999999978 },
  { -7.8735935697332210e+32, 50, 20, 
          0.0000000000000000 },
  { 5.4151358290895032e+31, 50, 20, 
          0.10000000000000009 },
  { 7.9309266056434309e+32, 50, 20, 
          0.19999999999999996 },
  { -4.0054067236243299e+31, 50, 20, 
          0.30000000000000004 },
  { -8.2324325279773965e+32, 50, 20, 
          0.39999999999999991 },
  { -4.0169860814274409e+32, 50, 20, 
          0.50000000000000000 },
  { 4.0860128756807882e+32, 50, 20, 
          0.60000000000000009 },
  { 8.6022521164717170e+32, 50, 20, 
          0.69999999999999996 },
  { 1.0622676657892043e+33, 50, 20, 
          0.80000000000000004 },
  { 1.6630925158645483e+33, 50, 20, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 20, 
          1.0000000000000000 },
};

// Test function for l=50, m=20.
template <typename Tp>
void test027()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data027)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data027[i].l), Tp(data027[i].m),
                   Tp(data027[i].x));
      const Tp f0 = data027[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=50, m=50.
testcase_assoc_legendre<double> data028[] = {
  { 0.0000000000000000, 50, 50, 
          -1.0000000000000000 },
  { 2.5366994974431341e+60, 50, 50, 
          -0.90000000000000002 },
  { 2.2028834403101213e+67, 50, 50, 
          -0.80000000000000004 },
  { 1.3325496559566651e+71, 50, 50, 
          -0.69999999999999996 },
  { 3.8898096431781969e+73, 50, 50, 
          -0.59999999999999998 },
  { 2.0509760257037188e+75, 50, 50, 
          -0.50000000000000000 },
  { 3.4866724533443283e+76, 50, 50, 
          -0.40000000000000002 },
  { 2.5790740224150207e+77, 50, 50, 
          -0.30000000000000004 },
  { 9.8222237931680989e+77, 50, 50, 
          -0.19999999999999996 },
  { 2.1198682190366617e+78, 50, 50, 
          -0.099999999999999978 },
  { 2.7253921397507295e+78, 50, 50, 
          0.0000000000000000 },
  { 2.1198682190366617e+78, 50, 50, 
          0.10000000000000009 },
  { 9.8222237931680989e+77, 50, 50, 
          0.19999999999999996 },
  { 2.5790740224150207e+77, 50, 50, 
          0.30000000000000004 },
  { 3.4866724533443283e+76, 50, 50, 
          0.39999999999999991 },
  { 2.0509760257037188e+75, 50, 50, 
          0.50000000000000000 },
  { 3.8898096431781724e+73, 50, 50, 
          0.60000000000000009 },
  { 1.3325496559566651e+71, 50, 50, 
          0.69999999999999996 },
  { 2.2028834403101213e+67, 50, 50, 
          0.80000000000000004 },
  { 2.5366994974431990e+60, 50, 50, 
          0.89999999999999991 },
  { 0.0000000000000000, 50, 50, 
          1.0000000000000000 },
};

// Test function for l=50, m=50.
template <typename Tp>
void test028()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data028)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data028[i].l), Tp(data028[i].m),
                   Tp(data028[i].x));
      const Tp f0 = data028[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=0.
testcase_assoc_legendre<double> data029[] = {
  { 1.0000000000000000, 100, 0, 
          -1.0000000000000000 },
  { 0.10226582055871908, 100, 0, 
          -0.90000000000000002 },
  { 0.050861167913584124, 100, 0, 
          -0.80000000000000004 },
  { -0.077132507199778780, 100, 0, 
          -0.69999999999999996 },
  { -0.023747023905133110, 100, 0, 
          -0.59999999999999998 },
  { -0.060518025961861198, 100, 0, 
          -0.50000000000000000 },
  { -0.072258202125684429, 100, 0, 
          -0.40000000000000002 },
  { 0.057127392202801719, 100, 0, 
          -0.30000000000000004 },
  { 0.014681835355659636, 100, 0, 
          -0.19999999999999996 },
  { -0.063895098434750303, 100, 0, 
          -0.099999999999999978 },
  { 0.079589237387178727, 100, 0, 
          0.0000000000000000 },
  { -0.063895098434749775, 100, 0, 
          0.10000000000000009 },
  { 0.014681835355659636, 100, 0, 
          0.19999999999999996 },
  { 0.057127392202801719, 100, 0, 
          0.30000000000000004 },
  { -0.072258202125685012, 100, 0, 
          0.39999999999999991 },
  { -0.060518025961861198, 100, 0, 
          0.50000000000000000 },
  { -0.023747023905134217, 100, 0, 
          0.60000000000000009 },
  { -0.077132507199778780, 100, 0, 
          0.69999999999999996 },
  { 0.050861167913584124, 100, 0, 
          0.80000000000000004 },
  { 0.10226582055871723, 100, 0, 
          0.89999999999999991 },
  { 1.0000000000000000, 100, 0, 
          1.0000000000000000 },
};

// Test function for l=100, m=0.
template <typename Tp>
void test029()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data029)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data029[i].l), Tp(data029[i].m),
                   Tp(data029[i].x));
      const Tp f0 = data029[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=1.
testcase_assoc_legendre<double> data030[] = {
  { -0.0000000000000000, 100, 1, 
          -1.0000000000000000 },
  { 6.5200167187780309, 100, 1, 
          -0.90000000000000002 },
  { 9.0065170007027540, 100, 1, 
          -0.80000000000000004 },
  { -5.4690908541181056, 100, 1, 
          -0.69999999999999996 },
  { -8.6275439170430861, 100, 1, 
          -0.59999999999999998 },
  { -6.0909031663448499, 100, 1, 
          -0.50000000000000000 },
  { 4.1160338699561265, 100, 1, 
          -0.40000000000000002 },
  { 5.8491043010758004, 100, 1, 
          -0.30000000000000004 },
  { -7.9435138723089826, 100, 1, 
          -0.19999999999999996 },
  { 4.7996285823989346, 100, 1, 
          -0.099999999999999978 },
  { 0.0000000000000000, 100, 1, 
          0.0000000000000000 },
  { -4.7996285823990057, 100, 1, 
          0.10000000000000009 },
  { 7.9435138723089826, 100, 1, 
          0.19999999999999996 },
  { -5.8491043010758004, 100, 1, 
          0.30000000000000004 },
  { -4.1160338699560359, 100, 1, 
          0.39999999999999991 },
  { 6.0909031663448499, 100, 1, 
          0.50000000000000000 },
  { 8.6275439170430559, 100, 1, 
          0.60000000000000009 },
  { 5.4690908541181056, 100, 1, 
          0.69999999999999996 },
  { -9.0065170007027540, 100, 1, 
          0.80000000000000004 },
  { -6.5200167187783133, 100, 1, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 1, 
          1.0000000000000000 },
};

// Test function for l=100, m=1.
template <typename Tp>
void test030()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data030)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data030[i].l), Tp(data030[i].m),
                   Tp(data030[i].x));
      const Tp f0 = data030[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=100, m=2.
testcase_assoc_legendre<double> data031[] = {
  { 0.0000000000000000, 100, 2, 
          -1.0000000000000000 },
  { -1005.9604880761007, 100, 2, 
          -0.90000000000000002 },
  { -489.68041725865896, 100, 2, 
          -0.80000000000000004 },
  { 768.31676011669970, 100, 2, 
          -0.69999999999999996 },
  { 226.90362556628003, 100, 2, 
          -0.59999999999999998 },
  { 604.19889304940330, 100, 2, 
          -0.50000000000000000 },
  { 733.40061037838029, 100, 2, 
          -0.40000000000000002 },
  { -573.30774483996402, 100, 2, 
          -0.30000000000000004 },
  { -151.52946305080897, 100, 2, 
          -0.19999999999999996 },
  { 646.30525583588019, 100, 2, 
          -0.099999999999999978 },
  { -803.85129761050518, 100, 2, 
          0.0000000000000000 },
  { 646.30525583587439, 100, 2, 
          0.10000000000000009 },
  { -151.52946305080897, 100, 2, 
          0.19999999999999996 },
  { -573.30774483996402, 100, 2, 
          0.30000000000000004 },
  { 733.40061037838507, 100, 2, 
          0.39999999999999991 },
  { 604.19889304940330, 100, 2, 
          0.50000000000000000 },
  { 226.90362556629168, 100, 2, 
          0.60000000000000009 },
  { 768.31676011669970, 100, 2, 
          0.69999999999999996 },
  { -489.68041725865896, 100, 2, 
          0.80000000000000004 },
  { -1005.9604880760811, 100, 2, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 2, 
          1.0000000000000000 },
};

// Test function for l=100, m=2.
template <typename Tp>
void test031()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data031)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data031[i].l), Tp(data031[i].m),
                   Tp(data031[i].x));
      const Tp f0 = data031[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=5.
testcase_assoc_legendre<double> data032[] = {
  { 0.0000000000000000, 100, 5, 
          -1.0000000000000000 },
  { 900551126.09653592, 100, 5, 
          -0.90000000000000002 },
  { 988567431.55756044, 100, 5, 
          -0.80000000000000004 },
  { -645646451.90344620, 100, 5, 
          -0.69999999999999996 },
  { -897114585.29920745, 100, 5, 
          -0.59999999999999998 },
  { -661710744.42483830, 100, 5, 
          -0.50000000000000000 },
  { 380163158.51425594, 100, 5, 
          -0.40000000000000002 },
  { 617391071.36632609, 100, 5, 
          -0.30000000000000004 },
  { -805288801.85509181, 100, 5, 
          -0.19999999999999996 },
  { 481041740.16728652, 100, 5, 
          -0.099999999999999978 },
  { 0.0000000000000000, 100, 5, 
          0.0000000000000000 },
  { -481041740.16729414, 100, 5, 
          0.10000000000000009 },
  { 805288801.85509181, 100, 5, 
          0.19999999999999996 },
  { -617391071.36632609, 100, 5, 
          0.30000000000000004 },
  { -380163158.51424748, 100, 5, 
          0.39999999999999991 },
  { 661710744.42483830, 100, 5, 
          0.50000000000000000 },
  { 897114585.29920506, 100, 5, 
          0.60000000000000009 },
  { 645646451.90344620, 100, 5, 
          0.69999999999999996 },
  { -988567431.55756044, 100, 5, 
          0.80000000000000004 },
  { -900551126.09655809, 100, 5, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 5, 
          1.0000000000000000 },
};

// Test function for l=100, m=5.
template <typename Tp>
void test032()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data032)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data032[i].l), Tp(data032[i].m),
                   Tp(data032[i].x));
      const Tp f0 = data032[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=10.
testcase_assoc_legendre<double> data033[] = {
  { 0.0000000000000000, 100, 10, 
          -1.0000000000000000 },
  { 2.5643395957658915e+17, 100, 10, 
          -0.90000000000000002 },
  { 1.5778673545673316e+18, 100, 10, 
          -0.80000000000000004 },
  { 4.4355048487496776e+18, 100, 10, 
          -0.69999999999999996 },
  { -9.5936111659124493e+17, 100, 10, 
          -0.59999999999999998 },
  { 4.2387123021963372e+18, 100, 10, 
          -0.50000000000000000 },
  { 8.2370834618426563e+18, 100, 10, 
          -0.40000000000000002 },
  { -4.9089358388052890e+18, 100, 10, 
          -0.30000000000000004 },
  { -2.3468810358091279e+18, 100, 10, 
          -0.19999999999999996 },
  { 6.8627855225034568e+18, 100, 10, 
          -0.099999999999999978 },
  { -8.2494597181670380e+18, 100, 10, 
          0.0000000000000000 },
  { 6.8627855225034056e+18, 100, 10, 
          0.10000000000000009 },
  { -2.3468810358091279e+18, 100, 10, 
          0.19999999999999996 },
  { -4.9089358388052890e+18, 100, 10, 
          0.30000000000000004 },
  { 8.2370834618426747e+18, 100, 10, 
          0.39999999999999991 },
  { 4.2387123021963372e+18, 100, 10, 
          0.50000000000000000 },
  { -9.5936111659112115e+17, 100, 10, 
          0.60000000000000009 },
  { 4.4355048487496776e+18, 100, 10, 
          0.69999999999999996 },
  { 1.5778673545673316e+18, 100, 10, 
          0.80000000000000004 },
  { 2.5643395957690282e+17, 100, 10, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 10, 
          1.0000000000000000 },
};

// Test function for l=100, m=10.
template <typename Tp>
void test033()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data033)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data033[i].l), Tp(data033[i].m),
                   Tp(data033[i].x));
      const Tp f0 = data033[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000014e-11));
}

// Test data for l=100, m=20.
testcase_assoc_legendre<double> data034[] = {
  { 0.0000000000000000, 100, 20, 
          -1.0000000000000000 },
  { 7.1604344878778576e+37, 100, 20, 
          -0.90000000000000002 },
  { -8.3963895116962035e+38, 100, 20, 
          -0.80000000000000004 },
  { 7.9022236853110024e+38, 100, 20, 
          -0.69999999999999996 },
  { 8.2680005574120908e+38, 100, 20, 
          -0.59999999999999998 },
  { 3.0750497039999469e+38, 100, 20, 
          -0.50000000000000000 },
  { -7.6120586043843874e+38, 100, 20, 
          -0.40000000000000002 },
  { 1.1474496891901873e+38, 100, 20, 
          -0.30000000000000004 },
  { 4.3966251307444218e+38, 100, 20, 
          -0.19999999999999996 },
  { -7.0503266451702621e+38, 100, 20, 
          -0.099999999999999978 },
  { 7.7727439836159581e+38, 100, 20, 
          0.0000000000000000 },
  { -7.0503266451702213e+38, 100, 20, 
          0.10000000000000009 },
  { 4.3966251307444218e+38, 100, 20, 
          0.19999999999999996 },
  { 1.1474496891901873e+38, 100, 20, 
          0.30000000000000004 },
  { -7.6120586043843511e+38, 100, 20, 
          0.39999999999999991 },
  { 3.0750497039999469e+38, 100, 20, 
          0.50000000000000000 },
  { 8.2680005574120424e+38, 100, 20, 
          0.60000000000000009 },
  { 7.9022236853110024e+38, 100, 20, 
          0.69999999999999996 },
  { -8.3963895116962035e+38, 100, 20, 
          0.80000000000000004 },
  { 7.1604344878809460e+37, 100, 20, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 20, 
          1.0000000000000000 },
};

// Test function for l=100, m=20.
template <typename Tp>
void test034()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data034)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data034[i].l), Tp(data034[i].m),
                   Tp(data034[i].x));
      const Tp f0 = data034[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for l=100, m=50.
testcase_assoc_legendre<double> data035[] = {
  { 0.0000000000000000, 100, 50, 
          -1.0000000000000000 },
  { 9.3231278516893773e+96, 100, 50, 
          -0.90000000000000002 },
  { -1.1029797977454331e+98, 100, 50, 
          -0.80000000000000004 },
  { 1.8089333903465883e+97, 100, 50, 
          -0.69999999999999996 },
  { 5.9364045925669389e+97, 100, 50, 
          -0.59999999999999998 },
  { -8.2252620339727163e+97, 100, 50, 
          -0.50000000000000000 },
  { 7.1431385093739848e+97, 100, 50, 
          -0.40000000000000002 },
  { -3.3520602067479374e+97, 100, 50, 
          -0.30000000000000004 },
  { -2.7791149588121386e+97, 100, 50, 
          -0.19999999999999996 },
  { 9.0119338550180433e+97, 100, 50, 
          -0.099999999999999978 },
  { -1.1712145031578381e+98, 100, 50, 
          0.0000000000000000 },
  { 9.0119338550181176e+97, 100, 50, 
          0.10000000000000009 },
  { -2.7791149588121386e+97, 100, 50, 
          0.19999999999999996 },
  { -3.3520602067479374e+97, 100, 50, 
          0.30000000000000004 },
  { 7.1431385093740728e+97, 100, 50, 
          0.39999999999999991 },
  { -8.2252620339727163e+97, 100, 50, 
          0.50000000000000000 },
  { 5.9364045925668061e+97, 100, 50, 
          0.60000000000000009 },
  { 1.8089333903465883e+97, 100, 50, 
          0.69999999999999996 },
  { -1.1029797977454331e+98, 100, 50, 
          0.80000000000000004 },
  { 9.3231278516894892e+96, 100, 50, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 50, 
          1.0000000000000000 },
};

// Test function for l=100, m=50.
template <typename Tp>
void test035()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data035)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data035[i].l), Tp(data035[i].m),
                   Tp(data035[i].x));
      const Tp f0 = data035[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=100, m=100.
testcase_assoc_legendre<double> data036[] = {
  { 0.0000000000000000, 100, 100, 
          -1.0000000000000000 },
  { 5.7751792255758316e+150, 100, 100, 
          -0.90000000000000002 },
  { 4.3552236041585515e+164, 100, 100, 
          -0.80000000000000004 },
  { 1.5936546850595123e+172, 100, 100, 
          -0.69999999999999996 },
  { 1.3579510590289176e+177, 100, 100, 
          -0.59999999999999998 },
  { 3.7752749682889513e+180, 100, 100, 
          -0.50000000000000000 },
  { 1.0910627330458913e+183, 100, 100, 
          -0.40000000000000002 },
  { 5.9697347526822483e+184, 100, 100, 
          -0.30000000000000004 },
  { 8.6585879147526714e+185, 100, 100, 
          -0.19999999999999996 },
  { 4.0331571908057011e+186, 100, 100, 
          -0.099999999999999978 },
  { 6.6663086700729543e+186, 100, 100, 
          0.0000000000000000 },
  { 4.0331571908057011e+186, 100, 100, 
          0.10000000000000009 },
  { 8.6585879147526714e+185, 100, 100, 
          0.19999999999999996 },
  { 5.9697347526822483e+184, 100, 100, 
          0.30000000000000004 },
  { 1.0910627330458913e+183, 100, 100, 
          0.39999999999999991 },
  { 3.7752749682889513e+180, 100, 100, 
          0.50000000000000000 },
  { 1.3579510590289000e+177, 100, 100, 
          0.60000000000000009 },
  { 1.5936546850595123e+172, 100, 100, 
          0.69999999999999996 },
  { 4.3552236041585515e+164, 100, 100, 
          0.80000000000000004 },
  { 5.7751792255761289e+150, 100, 100, 
          0.89999999999999991 },
  { 0.0000000000000000, 100, 100, 
          1.0000000000000000 },
};

// Test function for l=100, m=100.
template <typename Tp>
void test036()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data036)
                         / sizeof(testcase_assoc_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::assoc_legendre(Tp(data036[i].l), Tp(data036[i].m),
                   Tp(data036[i].x));
      const Tp f0 = data036[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

int main(int, char**)
{
  test001<double>();
  test002<double>();
  test003<double>();
  test004<double>();
  test005<double>();
  test006<double>();
  test007<double>();
  test008<double>();
  test009<double>();
  test010<double>();
  test011<double>();
  test012<double>();
  test013<double>();
  test014<double>();
  test015<double>();
  test016<double>();
  test017<double>();
  test018<double>();
  test019<double>();
  test020<double>();
  test021<double>();
  test022<double>();
  test023<double>();
  test024<double>();
  test025<double>();
  test026<double>();
  test027<double>();
  test028<double>();
  test029<double>();
  test030<double>();
  test031<double>();
  test032<double>();
  test033<double>();
  test034<double>();
  test035<double>();
  test036<double>();
  return 0;
}
