/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  j2s,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
  createWalletDaemonWithClient,
  postAmlDecisionNoRules,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState, WalletClient } from "../harness/harness.js";

export async function runKycPeerPushTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, amlKeypair } =
    await createKycTestkudosEnvironment(t, {
      adjustExchangeConfig(config) {
        configureCommonKyc(config);

        config.setString("KYC-RULE-R1", "operation_type", "merge");
        config.setString("KYC-RULE-R1", "enabled", "yes");
        config.setString("KYC-RULE-R1", "exposed", "yes");
        config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
        config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
        config.setString("KYC-RULE-R1", "timeframe", "1d");
        config.setString("KYC-RULE-R1", "next_measures", "M1");

        config.setString("KYC-MEASURE-M1", "check_name", "C1");
        config.setString("KYC-MEASURE-M1", "context", "{}");
        config.setString("KYC-MEASURE-M1", "program", "NONE");

        config.setString("KYC-CHECK-C1", "type", "INFO");
        config.setString("KYC-CHECK-C1", "description", "my check!");
        config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
      },
    });

  // Origin wallet for the p2p transaction.
  const w0 = await createWalletDaemonWithClient(t, {
    name: "w0",
  });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    bankClient,
    amount: "TESTKUDOS:20",
    exchange: exchange,
    walletClient: w0.walletClient,
  });

  await wres.withdrawalFinishedCond;

  const pushDebitRes = await doPeerPushDebit(t, {
    walletClient: w0.walletClient,
    amount: "TESTKUDOS:10",
    summary: "Test1",
  });

  const prepRes = await walletClient.call(
    WalletApiOperation.PreparePeerPushCredit,
    {
      talerUri: pushDebitRes.talerUri,
    },
  );

  console.log("prepRes", j2s(prepRes));

  await walletClient.call(WalletApiOperation.ConfirmPeerPushCredit, {
    transactionId: prepRes.transactionId,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: prepRes.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.MergeKycRequired,
    },
  });

  const txDet = await walletClient.call(WalletApiOperation.GetTransactionById, {
    transactionId: prepRes.transactionId,
  });

  console.log("tx details", j2s(txDet));

  const kycPaytoHash = txDet.kycPaytoHash;

  t.assertTrue(!!kycPaytoHash);

  t.assertTrue(!!txDet.kycAccessToken);
  t.assertTrue(!!txDet.kycUrl);

  await postAmlDecisionNoRules(t, {
    amlPriv: amlKeypair.priv,
    amlPub: amlKeypair.pub,
    exchangeBaseUrl: exchange.baseUrl,
    paytoHash: kycPaytoHash,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: prepRes.transactionId,
    txState: {
      major: TransactionMajorState.Done,
    },
  });
}

/**
 * Initiate a push debit transaction, wait until the transaction
 * is ready.
 */
async function doPeerPushDebit(
  t: GlobalTestState,
  args: {
    walletClient: WalletClient;
    amount: AmountString;
    summary?: string;
  },
): Promise<{
  transactionId: string;
  talerUri: string;
}> {
  const purse_expiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );
  const initRet = await args.walletClient.call(
    WalletApiOperation.InitiatePeerPushDebit,
    {
      partialContractTerms: {
        amount: args.amount,
        summary: args.summary ?? "Test P2P Payment",
        purse_expiration,
      },
    },
  );

  await args.walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: initRet.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.Ready,
    },
  });

  const txDet = await args.walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: initRet.transactionId,
    },
  );

  t.assertTrue(txDet.type === TransactionType.PeerPushDebit);
  const talerUri = txDet.talerUri;
  t.assertTrue(!!talerUri);

  return {
    transactionId: initRet.transactionId,
    talerUri,
  };
}

runKycPeerPushTest.suites = ["wallet"];
