/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Logger,
  ScopeType,
  TalerWireGatewayHttpClient,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { createSimpleTestkudosEnvironmentV3 } from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const logger = new Logger("test-withdrawal-manual.ts");

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runWithdrawalManualTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, exchangeBankAccount, bank } =
    await createSimpleTestkudosEnvironmentV3(t);

  {
    const accounts = await walletClient.call(
      WalletApiOperation.ListBankAccounts,
      {},
    );

    // Wallet doesn't have known bank accounts initially
    t.assertTrue(accounts.accounts.length == 0);
  }

  // Create a withdrawal operation

  const user = await bankClient.createRandomBankUser();

  await walletClient.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  const tStart = AbsoluteTime.now();

  // Test some getWithdrawalDetailsForAmount requests
  {
    // Zero should be acceptable
    const det = await walletClient.call(
      WalletApiOperation.GetWithdrawalDetailsForAmount,
      {
        amount: "TESTKUDOS:0",
        restrictScope: {
          type: ScopeType.Exchange,
          currency: "TESTKUDOS",
          url: exchange.baseUrl,
        },
      },
    );
    t.assertAmountEquals(det.amountEffective, "TESTKUDOS:0");
    t.assertAmountEquals(det.amountRaw, "TESTKUDOS:0");
    t.assertDeepEqual(det.numCoins, 0);
  }

  logger.info("starting AcceptManualWithdrawal request");
  // We expect this to return immediately.

  const wres = await walletClient.call(
    WalletApiOperation.AcceptManualWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      amount: "TESTKUDOS:10" as AmountString,
    },
  );

  logger.info("AcceptManualWithdrawal finished");
  logger.info(`result: ${j2s(wres)}`);

  // Check that the request did not go into long-polling.
  const duration = AbsoluteTime.difference(tStart, AbsoluteTime.now());
  if (typeof duration.d_ms !== "number" || duration.d_ms > 5 * 1000) {
    throw Error("withdrawal took too long (longpolling issue)");
  }

  const reservePub: string = wres.reservePub;

  const wireGatewayApiClient = new TalerWireGatewayHttpClient(
    exchangeBankAccount.wireGatewayApiBaseUrl,
  );

  await wireGatewayApiClient.addIncoming({
    auth: bank.getAdminAuth(),
    body: {
      amount: "TESTKUDOS:10",
      debit_account: user.accountPaytoUri,
      reserve_pub: reservePub,
    },
  });

  await exchange.runWirewatchOnce();

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Check balance

  const balResp = await walletClient.call(WalletApiOperation.GetBalances, {});
  t.assertAmountEquals("TESTKUDOS:9.72", balResp.balances[0].available);

  {
    const accounts = await walletClient.call(
      WalletApiOperation.ListBankAccounts,
      {},
    );

    console.log(j2s(accounts));

    // Account from withdrawal is now known to the wallet,
    // as the exchange returns it in the reserve status response
    t.assertTrue(accounts.accounts.length == 1);
  }
}

runWithdrawalManualTest.suites = ["wallet"];
