/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once


/////////////////////// Stdlib includes
#include <vector>


/////////////////////// Qt includes
#include <QDomElement>


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/Formula.hpp"
#include "MsXpS/libXpertMassCore/Modif.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


typedef CrossLinker *CrossLinkerRPtr;
typedef const CrossLinker *CrossLinkerCstRPtr;
typedef std::unique_ptr<CrossLinker> CrossLinkerUPtr;
typedef std::unique_ptr<const CrossLinker> CrossLinkerCstUPtr;
typedef std::shared_ptr<CrossLinker> CrossLinkerSPtr;
typedef std::shared_ptr<const CrossLinker> CrossLinkerCstSPtr;
typedef std::weak_ptr<CrossLinker> CrossLinkerWPtr;
typedef std::weak_ptr<const CrossLinker> CrossLinkerCstWPtr;

class DECLSPEC CrossLinker
{
  public:
  CrossLinker(PolChemDefCstSPtr pol_chem_def_csp);

  CrossLinker(PolChemDefCstSPtr pol_chem_def_csp,
              const QDomElement &element,
              int version = 1);
  CrossLinker(PolChemDefCstSPtr pol_chem_def_csp = nullptr,
              const QString &name                = QString(),
              const QString &formula             = QString(),
              double mono                        = 0.0,
              double avg                         = 0.0);
  CrossLinker(const CrossLinker &other);
  virtual ~CrossLinker();

  bool initialize(const QDomElement &element, int version = 1);

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolChemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  const PolChemDefCstSPtr &getPolChemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  QString getName() const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const QString &formula_string);
  const QString &getFormula() const;

  //////////////// THE MODIFICATIONS /////////////////////
  const std::vector<ModifCstSPtr> &getModifsCstRef() const;
  std::vector<ModifCstSPtr> &getModifsRef();
  bool insertModifAt(ModifCstSPtr modif_csp, std::size_t);
  bool appendModif(ModifCstSPtr modif_csp);
  ModifCstSPtr getModifAt(std::size_t index) const;
  bool removeModifAt(std::size_t index);
  bool hasModif(const QString &name);
  int modifIndex(const QString &name);

  //////////////// OPERATORS /////////////////////
  CrossLinker &operator=(const CrossLinker &other);
  bool operator==(const CrossLinker &) const;
  bool operator!=(const CrossLinker &) const;

  //////////////// VALIDATIONS /////////////////////
  const CrossLinkerCstSPtr getFromPolChemDefByName() const;
  Enums::PolChemDefEntityStatus isKnownByNameInPolChemDef() const;

  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// MASS OPERATIONS /////////////////////
  bool calculateMasses(const IsotopicDataCstSPtr &isotopic_data_csp,
                       double &mono,
                       double &avg) const;
  bool calculateMasses(const IsotopicDataCstSPtr &isotopic_data_csp);
  const CrossLinker &accountMasses(double *mono_p = nullptr,
                                   double *avg_p  = nullptr,
                                   int times      = 1) const;
  const CrossLinker &
  accountMasses(double &mono, double &avg, int times = 1) const;
  double getMass(Enums::MassType mass_type) const;

  //////////////// XML /////////////////////
  bool renderXmlClkElement(const QDomElement &element, int version);
  QString
  formatXmlClkElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken) const;

  //////////////// UTILS /////////////////////
  QString toString() const;

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name                    = "";
  QString m_formula                 = "";

  double m_mono = 0.0;
  double m_avg  = 0.0;

  mutable bool m_isValid = false;

  // Modif instances might belong to the PolChemDef, but not necessarily.
  std::vector<ModifCstSPtr> m_modifs;
};


} // namespace libXpertMassCore
} // namespace MsXpS
